"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

class Task {
  constructor(fn, parallel = true) {
    this.fn = fn;
    this.parallel = parallel;
    this.promise = new Promise((resolve, reject) => {
      this.resolve = resolve;
      this.reject = reject;
    });
  }

  async execute() {
    try {
      const value = await this.fn.call(undefined);
      this.resolve(value);
    } catch (err) {
      this.reject(err);
    }
  }

  runsInParallel() {
    return this.parallel;
  }

  runsInSerial() {
    return !this.parallel;
  }

  getPromise() {
    return this.promise;
  }

}

class AsyncQueue {
  constructor(options = {}) {
    this.parallelism = options.parallelism || 1;
    this.nonParallelizableOperation = false;
    this.tasksInProgress = 0;
    this.queue = [];
  }

  push(fn, {
    parallel
  } = {
    parallel: true
  }) {
    const task = new Task(fn, parallel);
    this.queue.push(task);
    this.processQueue();
    return task.getPromise();
  }

  processQueue() {
    if (!this.queue.length || this.nonParallelizableOperation || this.disposed) {
      return;
    }

    const task = this.queue[0];
    const canRunParallelOp = task.runsInParallel() && this.tasksInProgress < this.parallelism;
    const canRunSerialOp = task.runsInSerial() && this.tasksInProgress === 0;

    if (canRunSerialOp || canRunParallelOp) {
      this.processTask(task, task.runsInParallel());
      this.queue.shift();
      this.processQueue();
    }
  }

  async processTask(task, runsInParallel) {
    if (this.disposed) {
      return;
    }

    this.tasksInProgress++;

    if (!runsInParallel) {
      this.nonParallelizableOperation = true;
    }

    try {
      await task.execute();
    } finally {
      this.tasksInProgress--;
      this.nonParallelizableOperation = false;
      this.processQueue();
    }
  }

  dispose() {
    this.queue = [];
    this.disposed = true;
  }

}

exports.default = AsyncQueue;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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