"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

var _react = _interopRequireWildcard(require("react"));

var _reactDom = _interopRequireDefault(require("react-dom"));

var _propTypes = _interopRequireDefault(require("prop-types"));

var _eventKit = require("event-kit");

var _uriPattern = _interopRequireWildcard(require("./uri-pattern"));

var _refHolder = _interopRequireDefault(require("../models/ref-holder"));

var _stubItem = _interopRequireDefault(require("../items/stub-item"));

var _helpers = require("../helpers");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _interopRequireWildcard(obj) { if (obj && obj.__esModule) { return obj; } else { var newObj = {}; if (obj != null) { for (var key in obj) { if (Object.prototype.hasOwnProperty.call(obj, key)) { var desc = Object.defineProperty && Object.getOwnPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : {}; if (desc.get || desc.set) { Object.defineProperty(newObj, key, desc); } else { newObj[key] = obj[key]; } } } } newObj.default = obj; return newObj; } }

function _objectSpread2(target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i] != null ? arguments[i] : {}; var ownKeys = Object.keys(source); if (typeof Object.getOwnPropertySymbols === 'function') { ownKeys = ownKeys.concat(Object.getOwnPropertySymbols(source).filter(function (sym) { return Object.getOwnPropertyDescriptor(source, sym).enumerable; })); } ownKeys.forEach(function (key) { _defineProperty(target, key, source[key]); }); } return target; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

/**
 * PaneItem registers an opener with the current Atom workspace as long as this component is mounted. The opener will
 * trigger on URIs that match a specified pattern and render a subtree returned by a render prop.
 *
 * The render prop can receive three arguments:
 *
 * * itemHolder: A RefHolder. If used as the target for a ref, the referenced component will be used as the "item" of
 *   the pane item - its `getTitle()`, `getIconName()`, and other methods will be used by the pane.
 * * params: An object containing the named parameters captured by the URI pattern.
 * * uri: The exact, matched URI used to launch this item.
 *
 * render() {
 *   return (
 *     <PaneItem workspace={this.props.workspace} uriPattern='atom-github://host/{id}'>
 *       {({itemHolder, params}) => (
 *         <ItemComponent ref={itemHolder.setter} id={params.id} />
 *       )}
 *     </PaneItem>
 *   );
 * }
 */
class PaneItem extends _react.default.Component {
  constructor(props) {
    super(props);
    (0, _helpers.autobind)(this, 'opener');
    const uriPattern = new _uriPattern.default(this.props.uriPattern);
    const currentlyOpen = this.props.workspace.getPaneItems().reduce((arr, item) => {
      const element = item.getElement ? item.getElement() : null;
      const match = item.getURI ? uriPattern.matches(item.getURI()) : _uriPattern.nonURIMatch;
      const stub = item.setRealItem ? item : null;

      if (element && match.ok()) {
        const openItem = new OpenItem(match, element, stub);
        arr.push(openItem);
      }

      return arr;
    }, []);
    this.subs = new _eventKit.CompositeDisposable();
    this.state = {
      uriPattern,
      currentlyOpen
    };
  }

  static getDerivedStateFromProps(nextProps, prevState) {
    if (prevState.uriPattern.getOriginal() === nextProps.uriPattern) {
      return null;
    }

    return {
      uriPattern: new _uriPattern.default(nextProps.uriPattern)
    };
  }

  componentDidMount() {
    // Listen for and adopt StubItems that are added after this component has
    // already been mounted.
    this.subs.add(this.props.workspace.onDidAddPaneItem(({
      item
    }) => {
      if (!item._getStub) {
        return;
      }

      const stub = item._getStub();

      if (stub.getRealItem() !== null) {
        return;
      }

      const match = this.state.uriPattern.matches(item.getURI());

      if (!match.ok()) {
        return;
      }

      const openItem = new OpenItem(match, stub.getElement(), stub);
      openItem.hydrateStub({
        copy: () => this.copyOpenItem(openItem)
      });

      if (this.props.className) {
        openItem.addClassName(this.props.className);
      }

      this.registerCloseListener(item, openItem);
      this.setState(prevState => ({
        currentlyOpen: [...prevState.currentlyOpen, openItem]
      }));
    }));

    for (const openItem of this.state.currentlyOpen) {
      this.registerCloseListener(openItem.stubItem, openItem);
      openItem.hydrateStub({
        copy: () => this.copyOpenItem(openItem)
      });

      if (this.props.className) {
        openItem.addClassName(this.props.className);
      }
    }

    this.subs.add(this.props.workspace.addOpener(this.opener));
  }

  render() {
    return this.state.currentlyOpen.map(item => {
      return _react.default.createElement(_react.Fragment, {
        key: item.getKey()
      }, item.renderPortal(this.props.children));
    });
  }

  componentWillUnmount() {
    this.subs.dispose();
  }

  opener(uri) {
    const m = this.state.uriPattern.matches(uri);

    if (!m.ok()) {
      return undefined;
    }

    const openItem = new OpenItem(m);

    if (this.props.className) {
      openItem.addClassName(this.props.className);
    }

    return new Promise(resolve => {
      this.setState(prevState => ({
        currentlyOpen: [...prevState.currentlyOpen, openItem]
      }), () => {
        const paneItem = openItem.create({
          copy: () => this.copyOpenItem(openItem)
        });
        this.registerCloseListener(paneItem, openItem);
        resolve(paneItem);
      });
    });
  }

  copyOpenItem(openItem) {
    const m = this.state.uriPattern.matches(openItem.getURI());

    if (!m.ok()) {
      return null;
    }

    const stub = _stubItem.default.create('generic', openItem.getStubProps(), openItem.getURI());

    const copiedItem = new OpenItem(m, stub.getElement(), stub);
    this.setState(prevState => ({
      currentlyOpen: [...prevState.currentlyOpen, copiedItem]
    }), () => {
      this.registerCloseListener(stub, copiedItem);
      copiedItem.hydrateStub({
        copy: () => this.copyOpenItem(copiedItem)
      });
    });
    return stub;
  }

  registerCloseListener(paneItem, openItem) {
    const sub = this.props.workspace.onDidDestroyPaneItem(({
      item
    }) => {
      if (item === paneItem) {
        sub.dispose();
        this.subs.remove(sub);
        this.setState(prevState => ({
          currentlyOpen: prevState.currentlyOpen.filter(each => each !== openItem)
        }));
      }
    });
    this.subs.add(sub);
  }

}
/**
 * A subtree rendered through a portal onto a detached DOM node for use as the root as a PaneItem.
 */


exports.default = PaneItem;

_defineProperty(PaneItem, "propTypes", {
  workspace: _propTypes.default.object.isRequired,
  children: _propTypes.default.func.isRequired,
  uriPattern: _propTypes.default.string.isRequired,
  className: _propTypes.default.string
});

class OpenItem {
  constructor(match, element = null, stub = null) {
    this.id = this.constructor.nextID;
    this.constructor.nextID++;
    this.domNode = element || document.createElement('div');
    this.domNode.tabIndex = '-1';
    this.domNode.onfocus = this.onFocus.bind(this);
    this.stubItem = stub;
    this.stubProps = stub ? stub.props : {};
    this.match = match;
    this.itemHolder = new _refHolder.default();
  }

  getURI() {
    return this.match.getURI();
  }

  create(extra = {}) {
    const h = this.itemHolder.isEmpty() ? null : this.itemHolder;
    return (0, _helpers.createItem)(this.domNode, h, this.match.getURI(), extra);
  }

  hydrateStub(extra = {}) {
    if (this.stubItem) {
      this.stubItem.setRealItem(this.create(extra));
      this.stubItem = null;
    }
  }

  addClassName(className) {
    this.domNode.classList.add(className);
  }

  getKey() {
    return this.id;
  }

  getStubProps() {
    const itemProps = this.itemHolder.map(item => ({
      title: item.getTitle ? item.getTitle() : null,
      iconName: item.getIconName ? item.getIconName() : null
    }));
    return _objectSpread2({}, this.stubProps, {}, itemProps);
  }

  onFocus() {
    return this.itemHolder.map(item => item.focus && item.focus());
  }

  renderPortal(renderProp) {
    return _reactDom.default.createPortal(renderProp({
      deserialized: this.stubProps,
      itemHolder: this.itemHolder,
      params: this.match.getParams(),
      uri: this.match.getURI()
    }), this.domNode);
  }

}

_defineProperty(OpenItem, "nextID", 0);
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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