"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = exports.COMMIT_GRAMMAR_SCOPE = void 0;

var _path = _interopRequireDefault(require("path"));

var _atom = require("atom");

var _react = _interopRequireDefault(require("react"));

var _propTypes = _interopRequireDefault(require("prop-types"));

var _eventKit = require("event-kit");

var _fsExtra = _interopRequireDefault(require("fs-extra"));

var _commitView = _interopRequireDefault(require("../views/commit-view"));

var _refHolder = _interopRequireDefault(require("../models/ref-holder"));

var _commitPreviewItem = _interopRequireDefault(require("../items/commit-preview-item"));

var _propTypes2 = require("../prop-types");

var _watchWorkspaceItem = require("../watch-workspace-item");

var _helpers = require("../helpers");

var _reporterProxy = require("../reporter-proxy");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _objectSpread2(target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i] != null ? arguments[i] : {}; var ownKeys = Object.keys(source); if (typeof Object.getOwnPropertySymbols === 'function') { ownKeys = ownKeys.concat(Object.getOwnPropertySymbols(source).filter(function (sym) { return Object.getOwnPropertyDescriptor(source, sym).enumerable; })); } ownKeys.forEach(function (key) { _defineProperty(target, key, source[key]); }); } return target; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

const COMMIT_GRAMMAR_SCOPE = 'text.git-commit';
exports.COMMIT_GRAMMAR_SCOPE = COMMIT_GRAMMAR_SCOPE;

class CommitController extends _react.default.Component {
  constructor(props, context) {
    super(props, context);

    _defineProperty(this, "activateCommitPreview", () => {
      const uri = _commitPreviewItem.default.buildURI(this.props.repository.getWorkingDirectoryPath());

      return this.props.workspace.open(uri, {
        searchAllPanes: true,
        pending: true,
        activate: true
      });
    });

    (0, _helpers.autobind)(this, 'commit', 'handleMessageChange', 'toggleExpandedCommitMessageEditor', 'grammarAdded', 'toggleCommitPreview');
    this.subscriptions = new _eventKit.CompositeDisposable();
    this.refCommitView = new _refHolder.default();
    this.commitMessageBuffer = new _atom.TextBuffer({
      text: this.props.repository.getCommitMessage()
    });
    this.subscriptions.add(this.commitMessageBuffer.onDidChange(this.handleMessageChange));
    this.previewWatcher = (0, _watchWorkspaceItem.watchWorkspaceItem)(this.props.workspace, _commitPreviewItem.default.buildURI(this.props.repository.getWorkingDirectoryPath()), this, 'commitPreviewActive');
    this.subscriptions.add(this.previewWatcher);
  }

  componentDidMount() {
    this.subscriptions.add(this.props.workspace.onDidAddTextEditor(({
      textEditor
    }) => {
      if (this.props.repository.isPresent() && textEditor.getPath() === this.getCommitMessagePath()) {
        const grammar = this.props.grammars.grammarForScopeName(COMMIT_GRAMMAR_SCOPE);

        if (grammar) {
          textEditor.setGrammar(grammar);
        }
      }
    }), this.props.workspace.onDidDestroyPaneItem(async ({
      item
    }) => {
      if (this.props.repository.isPresent() && item.getPath && item.getPath() === this.getCommitMessagePath() && this.getCommitMessageEditors().length === 0) {
        // we closed the last editor pointing to the commit message file
        try {
          this.commitMessageBuffer.setText((await _fsExtra.default.readFile(this.getCommitMessagePath(), {
            encoding: 'utf8'
          })));
        } catch (e) {
          if (e.code !== 'ENOENT') {
            throw e;
          }
        }
      }
    }));
  }

  render() {
    const operationStates = this.props.repository.getOperationStates();
    return _react.default.createElement(_commitView.default, {
      ref: this.refCommitView.setter,
      workspace: this.props.workspace,
      tooltips: this.props.tooltips,
      config: this.props.config,
      stagedChangesExist: this.props.stagedChangesExist,
      mergeConflictsExist: this.props.mergeConflictsExist,
      prepareToCommit: this.props.prepareToCommit,
      commit: this.commit,
      abortMerge: this.props.abortMerge,
      commands: this.props.commands,
      maximumCharacterLimit: 72,
      messageBuffer: this.commitMessageBuffer,
      isMerging: this.props.isMerging,
      isCommitting: operationStates.isCommitInProgress(),
      lastCommit: this.props.lastCommit,
      currentBranch: this.props.currentBranch,
      toggleExpandedCommitMessageEditor: this.toggleExpandedCommitMessageEditor,
      deactivateCommitBox: this.isCommitMessageEditorExpanded(),
      userStore: this.props.userStore,
      selectedCoAuthors: this.props.selectedCoAuthors,
      updateSelectedCoAuthors: this.props.updateSelectedCoAuthors,
      toggleCommitPreview: this.toggleCommitPreview,
      activateCommitPreview: this.activateCommitPreview,
      commitPreviewActive: this.state.commitPreviewActive
    });
  }

  componentDidUpdate(prevProps) {
    this.commitMessageBuffer.setTextViaDiff(this.getCommitMessage());

    if (prevProps.repository !== this.props.repository) {
      this.previewWatcher.setPattern(_commitPreviewItem.default.buildURI(this.props.repository.getWorkingDirectoryPath()));
    }
  }

  componentWillUnmount() {
    this.subscriptions.dispose();
  }

  commit(message, coAuthors = [], amend = false) {
    let msg, verbatim;

    if (this.isCommitMessageEditorExpanded()) {
      msg = this.getCommitMessageEditors()[0].getText();
      verbatim = false;
    } else {
      const wrapMessage = this.props.config.get('github.automaticCommitMessageWrapping');
      msg = wrapMessage ? wrapCommitMessage(message) : message;
      verbatim = true;
    }

    return this.props.commit(msg.trim(), {
      amend,
      coAuthors,
      verbatim
    });
  }

  setCommitMessage(message, options) {
    if (!this.props.repository.isPresent()) {
      return;
    }

    this.props.repository.setCommitMessage(message, options);
  }

  getCommitMessage() {
    return this.props.repository.getCommitMessage();
  }

  getCommitMessagePath() {
    return _path.default.join(this.props.repository.getGitDirectoryPath(), 'ATOM_COMMIT_EDITMSG');
  }

  handleMessageChange() {
    if (!this.props.repository.isPresent()) {
      return;
    }

    this.setCommitMessage(this.commitMessageBuffer.getText(), {
      suppressUpdate: true
    });
  }

  getCommitMessageEditors() {
    if (!this.props.repository.isPresent()) {
      return [];
    }

    return this.props.workspace.getTextEditors().filter(editor => editor.getPath() === this.getCommitMessagePath());
  }

  async toggleExpandedCommitMessageEditor(messageFromBox) {
    if (this.isCommitMessageEditorExpanded()) {
      if (this.commitMessageEditorIsInForeground()) {
        await this.closeAllOpenCommitMessageEditors();
        this.forceUpdate();
      } else {
        this.activateCommitMessageEditor();
      }
    } else {
      await this.openCommitMessageEditor(messageFromBox);
      this.forceUpdate();
    }
  }

  isCommitMessageEditorExpanded() {
    return this.getCommitMessageEditors().length > 0;
  }

  commitMessageEditorIsInForeground() {
    const commitMessageEditorsInForeground = this.props.workspace.getPanes().map(pane => pane.getActiveItem()).filter(item => item && item.getPath && item.getPath() === this.getCommitMessagePath());
    return commitMessageEditorsInForeground.length > 0;
  }

  activateCommitMessageEditor() {
    const panes = this.props.workspace.getPanes();
    let editor;
    const paneWithEditor = panes.find(pane => {
      editor = pane.getItems().find(item => item.getPath && item.getPath() === this.getCommitMessagePath());
      return !!editor;
    });
    paneWithEditor.activate();
    paneWithEditor.activateItem(editor);
  }

  closeAllOpenCommitMessageEditors() {
    return Promise.all(this.props.workspace.getPanes().map(pane => {
      return Promise.all(pane.getItems().map(async item => {
        if (item && item.getPath && item.getPath() === this.getCommitMessagePath()) {
          const destroyed = await pane.destroyItem(item);

          if (!destroyed) {
            pane.activateItem(item);
          }
        }
      }));
    }));
  }

  async openCommitMessageEditor(messageFromBox) {
    await _fsExtra.default.writeFile(this.getCommitMessagePath(), messageFromBox, 'utf8');
    const commitEditor = await this.props.workspace.open(this.getCommitMessagePath());
    (0, _reporterProxy.addEvent)('open-commit-message-editor', {
      package: 'github'
    });
    const grammar = this.props.grammars.grammarForScopeName(COMMIT_GRAMMAR_SCOPE);

    if (grammar) {
      commitEditor.setGrammar(grammar);
    } else {
      this.grammarSubscription = this.props.grammars.onDidAddGrammar(this.grammarAdded);
      this.subscriptions.add(this.grammarSubscription);
    }
  }

  grammarAdded(grammar) {
    if (grammar.scopeName !== COMMIT_GRAMMAR_SCOPE) {
      return;
    }

    this.getCommitMessageEditors().forEach(editor => editor.setGrammar(grammar));
    this.grammarSubscription.dispose();
  }

  getFocus(element) {
    return this.refCommitView.map(view => view.getFocus(element)).getOr(null);
  }

  setFocus(focus) {
    return this.refCommitView.map(view => view.setFocus(focus)).getOr(false);
  }

  advanceFocusFrom(...args) {
    return this.refCommitView.map(view => view.advanceFocusFrom(...args)).getOr(false);
  }

  retreatFocusFrom(...args) {
    return this.refCommitView.map(view => view.retreatFocusFrom(...args)).getOr(false);
  }

  toggleCommitPreview() {
    (0, _reporterProxy.addEvent)('toggle-commit-preview', {
      package: 'github'
    });

    const uri = _commitPreviewItem.default.buildURI(this.props.repository.getWorkingDirectoryPath());

    if (this.props.workspace.hide(uri)) {
      return Promise.resolve();
    } else {
      return this.props.workspace.open(uri, {
        searchAllPanes: true,
        pending: true
      });
    }
  }

}

exports.default = CommitController;

_defineProperty(CommitController, "focus", _objectSpread2({}, _commitView.default.focus));

_defineProperty(CommitController, "propTypes", {
  workspace: _propTypes.default.object.isRequired,
  grammars: _propTypes.default.object.isRequired,
  commands: _propTypes.default.object.isRequired,
  config: _propTypes.default.object.isRequired,
  tooltips: _propTypes.default.object.isRequired,
  repository: _propTypes.default.object.isRequired,
  isMerging: _propTypes.default.bool.isRequired,
  mergeConflictsExist: _propTypes.default.bool.isRequired,
  stagedChangesExist: _propTypes.default.bool.isRequired,
  lastCommit: _propTypes.default.object.isRequired,
  currentBranch: _propTypes.default.object.isRequired,
  userStore: _propTypes2.UserStorePropType.isRequired,
  selectedCoAuthors: _propTypes.default.arrayOf(_propTypes2.AuthorPropType),
  updateSelectedCoAuthors: _propTypes.default.func,
  prepareToCommit: _propTypes.default.func.isRequired,
  commit: _propTypes.default.func.isRequired,
  abortMerge: _propTypes.default.func.isRequired
});

function wrapCommitMessage(message) {
  // hard wrap message (except for first line) at 72 characters
  let results = [];
  message.split('\n').forEach((line, index) => {
    if (line.length <= 72 || index === 0) {
      results.push(line);
    } else {
      const matches = line.match(/.{1,72}(\s|$)|\S+?(\s|$)/g).map(match => {
        return match.endsWith('\n') ? match.substr(0, match.length - 1) : match;
      });
      results = results.concat(matches);
    }
  });
  return results.join('\n');
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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