"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

var _eventKit = require("event-kit");

var _react = _interopRequireDefault(require("react"));

var _propTypes = _interopRequireDefault(require("prop-types"));

var _compareSets = _interopRequireDefault(require("compare-sets"));

var _commands = _interopRequireWildcard(require("../atom/commands"));

var _conflict = _interopRequireDefault(require("../models/conflicts/conflict"));

var _conflictController = _interopRequireDefault(require("./conflict-controller"));

var _source = require("../models/conflicts/source");

var _helpers = require("../helpers");

function _interopRequireWildcard(obj) { if (obj && obj.__esModule) { return obj; } else { var newObj = {}; if (obj != null) { for (var key in obj) { if (Object.prototype.hasOwnProperty.call(obj, key)) { var desc = Object.defineProperty && Object.getOwnPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : {}; if (desc.get || desc.set) { Object.defineProperty(newObj, key, desc); } else { newObj[key] = obj[key]; } } } } newObj.default = obj; return newObj; } }

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

/**
 * Render a `ConflictController` for each conflict marker within an open TextEditor.
 */
class EditorConflictController extends _react.default.Component {
  constructor(props, context) {
    super(props, context);
    (0, _helpers.autobind)(this, 'resolveAsCurrent', 'revertConflictModifications', 'dismissCurrent'); // this.layer = props.editor.addMarkerLayer({
    //   maintainHistory: true,
    //   persistent: false,
    // });

    this.layer = props.editor.getDefaultMarkerLayer();
    this.state = {
      conflicts: new Set(_conflict.default.allFromEditor(props.editor, this.layer, props.isRebase))
    };
    this.subscriptions = new _eventKit.CompositeDisposable();
    this.updateMarkerCount();
  }

  componentDidMount() {
    const buffer = this.props.editor.getBuffer();
    this.subscriptions.add(this.props.editor.onDidDestroy(() => this.props.refreshResolutionProgress(this.props.editor.getPath())), buffer.onDidReload(() => this.reparseConflicts()));
    this.scrollToFirstConflict();
  }

  render() {
    this.updateMarkerCount();
    return _react.default.createElement("div", null, this.state.conflicts.size > 0 && _react.default.createElement(_commands.default, {
      registry: this.props.commands,
      target: "atom-text-editor"
    }, _react.default.createElement(_commands.Command, {
      command: "github:resolve-as-ours",
      callback: this.getResolverUsing([_source.OURS])
    }), _react.default.createElement(_commands.Command, {
      command: "github:resolve-as-theirs",
      callback: this.getResolverUsing([_source.THEIRS])
    }), _react.default.createElement(_commands.Command, {
      command: "github:resolve-as-base",
      callback: this.getResolverUsing([_source.BASE])
    }), _react.default.createElement(_commands.Command, {
      command: "github:resolve-as-ours-then-theirs",
      callback: this.getResolverUsing([_source.OURS, _source.THEIRS])
    }), _react.default.createElement(_commands.Command, {
      command: "github:resolve-as-theirs-then-ours",
      callback: this.getResolverUsing([_source.THEIRS, _source.OURS])
    }), _react.default.createElement(_commands.Command, {
      command: "github:resolve-as-current",
      callback: this.resolveAsCurrent
    }), _react.default.createElement(_commands.Command, {
      command: "github:revert-conflict-modifications",
      callback: this.revertConflictModifications
    }), _react.default.createElement(_commands.Command, {
      command: "github:dismiss-conflict",
      callback: this.dismissCurrent
    })), Array.from(this.state.conflicts, c => _react.default.createElement(_conflictController.default, {
      key: c.getKey(),
      editor: this.props.editor,
      conflict: c,
      resolveAsSequence: sources => this.resolveAsSequence(c, sources),
      dismiss: () => this.dismissConflicts([c])
    })));
  }

  componentWillUnmount() {
    // this.layer.destroy();
    this.subscriptions.dispose();
  }
  /*
   * Return an Array containing `Conflict` objects whose marked regions include any cursor position in the current
   * `TextEditor` and the `Sides` that contain a cursor within each.
   *
   * This method is written to have linear complexity with respect to the number of cursors and the number of
   * conflicts, to gracefully handle files with large numbers of both.
   */


  getCurrentConflicts() {
    const cursorPositions = this.props.editor.getCursorBufferPositions();
    cursorPositions.sort((a, b) => a.compare(b));
    const cursorIterator = cursorPositions[Symbol.iterator]();
    const conflictIterator = this.state.conflicts.keys();
    let currentCursor = cursorIterator.next();
    let currentConflict = conflictIterator.next();
    const activeConflicts = [];

    while (!currentCursor.done && !currentConflict.done) {
      // Advance currentCursor to the first cursor beyond the earliest conflict.
      const earliestConflictPosition = currentConflict.value.getRange().start;

      while (!currentCursor.done && currentCursor.value.isLessThan(earliestConflictPosition)) {
        currentCursor = cursorIterator.next();
      } // Advance currentConflict until the first conflict that begins at a position after the current cursor.
      // Compare each to the current cursor, and add it to activeConflicts if it contains it.


      while (!currentConflict.done && !currentCursor.done && currentConflict.value.getRange().start.isLessThan(currentCursor.value)) {
        if (currentConflict.value.includesPoint(currentCursor.value)) {
          // Hit; determine which sides of this conflict contain cursors.
          const conflict = currentConflict.value;
          const endPosition = conflict.getRange().end;
          const sides = new Set();

          while (!currentCursor.done && currentCursor.value.isLessThan(endPosition)) {
            const side = conflict.getSideContaining(currentCursor.value);

            if (side) {
              sides.add(side);
            }

            currentCursor = cursorIterator.next();
          }

          activeConflicts.push({
            conflict,
            sides
          });
        }

        currentConflict = conflictIterator.next();
      }
    }

    return activeConflicts;
  }

  getResolverUsing(sequence) {
    return () => {
      this.getCurrentConflicts().forEach(match => this.resolveAsSequence(match.conflict, sequence));
    };
  }

  resolveAsCurrent() {
    this.getCurrentConflicts().forEach(match => {
      if (match.sides.size === 1) {
        const side = match.sides.keys().next().value;
        this.resolveAs(match.conflict, side.getSource());
      }
    });
  }

  revertConflictModifications() {
    this.getCurrentConflicts().forEach(match => {
      match.sides.forEach(side => {
        side.isModified() && side.revert();
        side.isBannerModified() && side.revertBanner();
      });
    });
  }

  dismissCurrent() {
    this.dismissConflicts(this.getCurrentConflicts().map(match => match.conflict));
  }

  dismissConflicts(conflicts) {
    this.setState(prevState => {
      const {
        added
      } = (0, _compareSets.default)(new Set(conflicts), prevState.conflicts);
      return {
        conflicts: added
      };
    });
  }

  resolveAsSequence(conflict, sources) {
    const [firstSide, ...restOfSides] = sources.map(source => conflict.getSide(source)).filter(side => side);
    const textToAppend = restOfSides.map(side => side.getText()).join('');
    this.props.editor.transact(() => {
      // Append text from all but the first Side to the first Side. Adjust the following DisplayMarker so that only that
      // Side's marker includes the appended text, not the next one.
      const appendedRange = firstSide.appendText(textToAppend);
      const nextMarker = conflict.markerAfter(firstSide.getPosition());

      if (nextMarker) {
        nextMarker.setTailBufferPosition(appendedRange.end);
      }

      this.innerResolveAs(conflict, sources[0]);
    });
  }

  resolveAs(conflict, source) {
    this.props.editor.transact(() => {
      this.innerResolveAs(conflict, source);
    });
  }

  innerResolveAs(conflict, source) {
    conflict.resolveAs(source);
    const chosenSide = conflict.getChosenSide();

    if (!chosenSide.isBannerModified()) {
      chosenSide.deleteBanner();
    }

    const separator = conflict.getSeparator();

    if (!separator.isModified()) {
      separator.delete();
    }

    conflict.getUnchosenSides().forEach(side => {
      side.deleteBanner();
      side.delete();
    });
    this.updateMarkerCount();
  }

  scrollToFirstConflict() {
    let firstConflict = null;

    for (const conflict of this.state.conflicts) {
      if (firstConflict == null || firstConflict.getRange().compare(conflict.getRange()) > 0) {
        firstConflict = conflict;
      }
    }

    if (firstConflict) {
      this.props.editor.scrollToBufferPosition(firstConflict.getRange().start, {
        center: true
      });
    }
  }

  reparseConflicts() {
    const newConflicts = new Set(_conflict.default.allFromEditor(this.props.editor, this.layer, this.props.isRebase));
    this.setState({
      conflicts: newConflicts
    });
  }

  updateMarkerCount() {
    this.props.resolutionProgress.reportMarkerCount(this.props.editor.getPath(), Array.from(this.state.conflicts, c => !c.isResolved()).filter(b => b).length);
  }

}

exports.default = EditorConflictController;

_defineProperty(EditorConflictController, "propTypes", {
  editor: _propTypes.default.object.isRequired,
  commands: _propTypes.default.object.isRequired,
  resolutionProgress: _propTypes.default.object.isRequired,
  isRebase: _propTypes.default.bool.isRequired,
  refreshResolutionProgress: _propTypes.default.func.isRequired
});
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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