"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

var _react = _interopRequireDefault(require("react"));

var _propTypes = _interopRequireDefault(require("prop-types"));

var _path = _interopRequireDefault(require("path"));

var _helpers = require("../helpers");

var _reporterProxy = require("../reporter-proxy");

var _propTypes2 = require("../prop-types");

var _changedFileItem = _interopRequireDefault(require("../items/changed-file-item"));

var _multiFilePatchView = _interopRequireDefault(require("../views/multi-file-patch-view"));

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _extends() { _extends = Object.assign || function (target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i]; for (var key in source) { if (Object.prototype.hasOwnProperty.call(source, key)) { target[key] = source[key]; } } } return target; }; return _extends.apply(this, arguments); }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

class MultiFilePatchController extends _react.default.Component {
  constructor(props) {
    super(props);
    (0, _helpers.autobind)(this, 'selectedRowsChanged', 'undoLastDiscard', 'diveIntoMirrorPatch', 'openFile', 'toggleFile', 'toggleRows', 'toggleModeChange', 'toggleSymlinkChange', 'discardRows');
    this.state = {
      selectionMode: 'hunk',
      selectedRows: new Set(),
      hasMultipleFileSelections: false
    };
    this.mouseSelectionInProgress = false;
    this.stagingOperationInProgress = false;
    this.lastPatchString = null;
    this.patchChangePromise = new Promise(resolve => {
      this.resolvePatchChangePromise = resolve;
    });
  }

  componentDidUpdate(prevProps) {
    if (this.lastPatchString !== null && this.lastPatchString !== this.props.multiFilePatch.toString()) {
      this.resolvePatchChangePromise();
      this.patchChangePromise = new Promise(resolve => {
        this.resolvePatchChangePromise = resolve;
      });
    }
  }

  render() {
    return _react.default.createElement(_multiFilePatchView.default, _extends({}, this.props, {
      selectedRows: this.state.selectedRows,
      selectionMode: this.state.selectionMode,
      hasMultipleFileSelections: this.state.hasMultipleFileSelections,
      selectedRowsChanged: this.selectedRowsChanged,
      diveIntoMirrorPatch: this.diveIntoMirrorPatch,
      openFile: this.openFile,
      toggleFile: this.toggleFile,
      toggleRows: this.toggleRows,
      toggleModeChange: this.toggleModeChange,
      toggleSymlinkChange: this.toggleSymlinkChange,
      undoLastDiscard: this.undoLastDiscard,
      discardRows: this.discardRows,
      selectNextHunk: this.selectNextHunk,
      selectPreviousHunk: this.selectPreviousHunk,
      switchToIssueish: this.props.switchToIssueish
    }));
  }

  undoLastDiscard(filePatch, {
    eventSource
  } = {}) {
    (0, _reporterProxy.addEvent)('undo-last-discard', {
      package: 'github',
      component: this.constructor.name,
      eventSource
    });
    return this.props.undoLastDiscard(filePatch.getPath(), this.props.repository);
  }

  diveIntoMirrorPatch(filePatch) {
    const mirrorStatus = this.withStagingStatus({
      staged: 'unstaged',
      unstaged: 'staged'
    });
    const workingDirectory = this.props.repository.getWorkingDirectoryPath();

    const uri = _changedFileItem.default.buildURI(filePatch.getPath(), workingDirectory, mirrorStatus);

    this.props.destroy();
    return this.props.workspace.open(uri);
  }

  async openFile(filePatch, positions, pending) {
    const absolutePath = _path.default.join(this.props.repository.getWorkingDirectoryPath(), filePatch.getPath());

    const editor = await this.props.workspace.open(absolutePath, {
      pending
    });

    if (positions.length > 0) {
      editor.setCursorBufferPosition(positions[0], {
        autoscroll: false
      });

      for (const position of positions.slice(1)) {
        editor.addCursorAtBufferPosition(position);
      }

      editor.scrollToBufferPosition(positions[positions.length - 1], {
        center: true
      });
    }

    return editor;
  }

  toggleFile(filePatch) {
    return this.stagingOperation(() => {
      const methodName = this.withStagingStatus({
        staged: 'unstageFiles',
        unstaged: 'stageFiles'
      });
      return this.props.repository[methodName]([filePatch.getPath()]);
    });
  }

  async toggleRows(rowSet, nextSelectionMode) {
    let chosenRows = rowSet;

    if (chosenRows) {
      const nextMultipleFileSelections = this.props.multiFilePatch.spansMultipleFiles(chosenRows);
      await this.selectedRowsChanged(chosenRows, nextSelectionMode, nextMultipleFileSelections);
    } else {
      chosenRows = this.state.selectedRows;
    }

    if (chosenRows.size === 0) {
      return Promise.resolve();
    }

    return this.stagingOperation(() => {
      const patch = this.withStagingStatus({
        staged: () => this.props.multiFilePatch.getUnstagePatchForLines(chosenRows),
        unstaged: () => this.props.multiFilePatch.getStagePatchForLines(chosenRows)
      });
      return this.props.repository.applyPatchToIndex(patch);
    });
  }

  toggleModeChange(filePatch) {
    return this.stagingOperation(() => {
      const targetMode = this.withStagingStatus({
        unstaged: filePatch.getNewMode(),
        staged: filePatch.getOldMode()
      });
      return this.props.repository.stageFileModeChange(filePatch.getPath(), targetMode);
    });
  }

  toggleSymlinkChange(filePatch) {
    return this.stagingOperation(() => {
      const relPath = filePatch.getPath();
      const repository = this.props.repository;
      return this.withStagingStatus({
        unstaged: () => {
          if (filePatch.hasTypechange() && filePatch.getStatus() === 'added') {
            return repository.stageFileSymlinkChange(relPath);
          }

          return repository.stageFiles([relPath]);
        },
        staged: () => {
          if (filePatch.hasTypechange() && filePatch.getStatus() === 'deleted') {
            return repository.stageFileSymlinkChange(relPath);
          }

          return repository.unstageFiles([relPath]);
        }
      });
    });
  }

  async discardRows(rowSet, nextSelectionMode, {
    eventSource
  } = {}) {
    // (kuychaco) For now we only support discarding rows for MultiFilePatches that contain a single file patch
    // The only way to access this method from the UI is to be in a ChangedFileItem, which only has a single file patch
    // This check is duplicated in RootController#discardLines. We also want it here to prevent us from sending metrics
    // unnecessarily
    if (this.props.multiFilePatch.getFilePatches().length !== 1) {
      return Promise.resolve(null);
    }

    let chosenRows = rowSet;

    if (chosenRows) {
      const nextMultipleFileSelections = this.props.multiFilePatch.spansMultipleFiles(chosenRows);
      await this.selectedRowsChanged(chosenRows, nextSelectionMode, nextMultipleFileSelections);
    } else {
      chosenRows = this.state.selectedRows;
    }

    (0, _reporterProxy.addEvent)('discard-unstaged-changes', {
      package: 'github',
      component: this.constructor.name,
      lineCount: chosenRows.size,
      eventSource
    });
    return this.props.discardLines(this.props.multiFilePatch, chosenRows, this.props.repository);
  }

  selectedRowsChanged(rows, nextSelectionMode, nextMultipleFileSelections) {
    if ((0, _helpers.equalSets)(this.state.selectedRows, rows) && this.state.selectionMode === nextSelectionMode && this.state.hasMultipleFileSelections === nextMultipleFileSelections) {
      return Promise.resolve();
    }

    return new Promise(resolve => {
      this.setState({
        selectedRows: rows,
        selectionMode: nextSelectionMode,
        hasMultipleFileSelections: nextMultipleFileSelections
      }, resolve);
    });
  }

  withStagingStatus(callbacks) {
    const callback = callbacks[this.props.stagingStatus];
    /* istanbul ignore if */

    if (!callback) {
      throw new Error(`Unknown staging status: ${this.props.stagingStatus}`);
    }

    return callback instanceof Function ? callback() : callback;
  }

  stagingOperation(fn) {
    if (this.stagingOperationInProgress) {
      return null;
    }

    this.stagingOperationInProgress = true;
    this.lastPatchString = this.props.multiFilePatch.toString();
    const operationPromise = fn();
    operationPromise.then(() => this.patchChangePromise).then(() => {
      this.stagingOperationInProgress = false;
      this.lastPatchString = null;
    });
    return operationPromise;
  }

}

exports.default = MultiFilePatchController;

_defineProperty(MultiFilePatchController, "propTypes", {
  repository: _propTypes.default.object.isRequired,
  stagingStatus: _propTypes.default.oneOf(['staged', 'unstaged']),
  multiFilePatch: _propTypes2.MultiFilePatchPropType.isRequired,
  hasUndoHistory: _propTypes.default.bool,
  reviewCommentsLoading: _propTypes.default.bool,
  reviewCommentThreads: _propTypes.default.arrayOf(_propTypes.default.shape({
    thread: _propTypes.default.object.isRequired,
    comments: _propTypes.default.arrayOf(_propTypes.default.object).isRequired
  })),
  workspace: _propTypes.default.object.isRequired,
  commands: _propTypes.default.object.isRequired,
  keymaps: _propTypes.default.object.isRequired,
  tooltips: _propTypes.default.object.isRequired,
  config: _propTypes.default.object.isRequired,
  destroy: _propTypes.default.func.isRequired,
  discardLines: _propTypes.default.func,
  undoLastDiscard: _propTypes.default.func,
  surface: _propTypes.default.func,
  switchToIssueish: _propTypes.default.func
});
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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