"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = exports.BarePullRequestCheckoutController = exports.checkoutStates = void 0;

var _react = _interopRequireDefault(require("react"));

var _propTypes = _interopRequireDefault(require("prop-types"));

var _reactRelay = require("react-relay");

var _enableableOperation = _interopRequireDefault(require("../models/enableable-operation"));

var _gitShellOutStrategy = require("../git-shell-out-strategy");

var _propTypes2 = require("../prop-types");

var _reporterProxy = require("../reporter-proxy");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

class CheckoutState {
  constructor(name) {
    this.name = name;
  }

  when(cases) {
    return cases[this.name] || cases.default;
  }

}

const checkoutStates = {
  HIDDEN: new CheckoutState('hidden'),
  DISABLED: new CheckoutState('disabled'),
  BUSY: new CheckoutState('busy'),
  CURRENT: new CheckoutState('current')
};
exports.checkoutStates = checkoutStates;

class BarePullRequestCheckoutController extends _react.default.Component {
  constructor(props) {
    super(props);
    this.state = {
      checkoutInProgress: false
    };
    this.checkoutOp = new _enableableOperation.default(() => this.checkout().catch(e => {
      if (!(e instanceof _gitShellOutStrategy.GitError)) {
        throw e;
      }
    }));
    this.checkoutOp.toggleState(this, 'checkoutInProgress');
  }

  render() {
    return this.props.children(this.nextCheckoutOp());
  }

  nextCheckoutOp() {
    const {
      repository,
      pullRequest
    } = this.props;

    if (this.props.isAbsent) {
      return this.checkoutOp.disable(checkoutStates.HIDDEN, 'No repository found');
    }

    if (this.props.isLoading) {
      return this.checkoutOp.disable(checkoutStates.DISABLED, 'Loading');
    }

    if (!this.props.isPresent) {
      return this.checkoutOp.disable(checkoutStates.DISABLED, 'No repository found');
    }

    if (this.props.isMerging) {
      return this.checkoutOp.disable(checkoutStates.DISABLED, 'Merge in progress');
    }

    if (this.props.isRebasing) {
      return this.checkoutOp.disable(checkoutStates.DISABLED, 'Rebase in progress');
    }

    if (this.state.checkoutInProgress) {
      return this.checkoutOp.disable(checkoutStates.DISABLED, 'Checking out...');
    } // determine if pullRequest.headRepository is null
    // this can happen if a repository has been deleted.


    if (!pullRequest.headRepository) {
      return this.checkoutOp.disable(checkoutStates.DISABLED, 'Pull request head repository does not exist');
    } // Determine if we already have this PR checked out.


    const headPush = this.props.branches.getHeadBranch().getPush();
    const headRemote = this.props.remotes.withName(headPush.getRemoteName()); // (detect checkout from pull/### refspec)

    const fromPullRefspec = headRemote.getOwner() === repository.owner.login && headRemote.getRepo() === repository.name && headPush.getShortRemoteRef() === `pull/${pullRequest.number}/head`; // (detect checkout from head repository)

    const fromHeadRepo = headRemote.getOwner() === pullRequest.headRepository.owner.login && headRemote.getRepo() === pullRequest.headRepository.name && headPush.getShortRemoteRef() === pullRequest.headRefName;

    if (fromPullRefspec || fromHeadRepo) {
      return this.checkoutOp.disable(checkoutStates.CURRENT, 'Current');
    }

    return this.checkoutOp.enable();
  }

  async checkout() {
    const {
      pullRequest
    } = this.props;
    const {
      headRepository
    } = pullRequest;
    const fullHeadRef = `refs/heads/${pullRequest.headRefName}`;
    let sourceRemoteName, localRefName; // Discover or create a remote pointing to the repo containing the pull request's head ref.
    // If the local repository already has the head repository specified as a remote, that remote will be used, so
    // that any related configuration is picked up for the fetch. Otherwise, the head repository fetch URL is used
    // directly.

    const headRemotes = this.props.remotes.matchingGitHubRepository(headRepository.owner.login, headRepository.name);

    if (headRemotes.length > 0) {
      sourceRemoteName = headRemotes[0].getName();
    } else {
      const url = {
        https: headRepository.url + '.git',
        ssh: headRepository.sshUrl
      }[this.props.remotes.mostUsedProtocol(['https', 'ssh'])]; // This will throw if a remote with this name already exists (and points somewhere else, or we would have found
      // it above). ¯\_(ツ)_/¯

      const remote = await this.props.localRepository.addRemote(headRepository.owner.login, url);
      sourceRemoteName = remote.getName();
    } // Identify an existing local ref that already corresponds to the pull request, if one exists. Otherwise, generate
    // a new local ref name.


    const pullTargets = this.props.branches.getPullTargets(sourceRemoteName, fullHeadRef);

    if (pullTargets.length > 0) {
      localRefName = pullTargets[0].getName(); // Check out the existing local ref.

      await this.props.localRepository.checkout(localRefName);

      try {
        await this.props.localRepository.pull(fullHeadRef, {
          remoteName: sourceRemoteName,
          ffOnly: true
        });
      } finally {
        (0, _reporterProxy.incrementCounter)('checkout-pr');
      }

      return;
    }

    await this.props.localRepository.fetch(fullHeadRef, {
      remoteName: sourceRemoteName
    }); // Check out the local ref and set it up to track the head ref.

    await this.props.localRepository.checkout(`pr-${pullRequest.number}/${headRepository.owner.login}/${pullRequest.headRefName}`, {
      createNew: true,
      track: true,
      startPoint: `refs/remotes/${sourceRemoteName}/${pullRequest.headRefName}`
    });
    (0, _reporterProxy.incrementCounter)('checkout-pr');
  }

}

exports.BarePullRequestCheckoutController = BarePullRequestCheckoutController;

_defineProperty(BarePullRequestCheckoutController, "propTypes", {
  // GraphQL response
  repository: _propTypes.default.shape({
    name: _propTypes.default.string.isRequired,
    owner: _propTypes.default.shape({
      login: _propTypes.default.string.isRequired
    }).isRequired
  }).isRequired,
  pullRequest: _propTypes.default.shape({
    number: _propTypes.default.number.isRequired,
    headRefName: _propTypes.default.string.isRequired,
    headRepository: _propTypes.default.shape({
      name: _propTypes.default.string.isRequired,
      url: _propTypes.default.string.isRequired,
      sshUrl: _propTypes.default.string.isRequired,
      owner: _propTypes.default.shape({
        login: _propTypes.default.string.isRequired
      })
    })
  }).isRequired,
  // Repository model and attributes
  localRepository: _propTypes.default.object.isRequired,
  isAbsent: _propTypes.default.bool.isRequired,
  isLoading: _propTypes.default.bool.isRequired,
  isPresent: _propTypes.default.bool.isRequired,
  isMerging: _propTypes.default.bool.isRequired,
  isRebasing: _propTypes.default.bool.isRequired,
  branches: _propTypes2.BranchSetPropType.isRequired,
  remotes: _propTypes2.RemoteSetPropType.isRequired,
  children: _propTypes.default.func.isRequired
});

var _default = (0, _reactRelay.createFragmentContainer)(BarePullRequestCheckoutController, {
  repository: function () {
    const node = require("./__generated__/prCheckoutController_repository.graphql");

    if (node.hash && node.hash !== "b2212745240c03ff8fc7cb13dfc63183") {
      console.error("The definition of 'prCheckoutController_repository' appears to have changed. Run `relay-compiler` to update the generated files to receive the expected data.");
    }

    return require("./__generated__/prCheckoutController_repository.graphql");
  },
  pullRequest: function () {
    const node = require("./__generated__/prCheckoutController_pullRequest.graphql");

    if (node.hash && node.hash !== "66e001f389a2c4f74c1369cf69b31268") {
      console.error("The definition of 'prCheckoutController_pullRequest' appears to have changed. Run `relay-compiler` to update the generated files to receive the expected data.");
    }

    return require("./__generated__/prCheckoutController_pullRequest.graphql");
  }
});

exports.default = _default;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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