"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

var _path = _interopRequireDefault(require("path"));

var _react = _interopRequireDefault(require("react"));

var _propTypes = _interopRequireDefault(require("prop-types"));

var _yubikiri = _interopRequireDefault(require("yubikiri"));

var _eventKit = require("event-kit");

var _observeModel = _interopRequireDefault(require("../views/observe-model"));

var _resolutionProgress = _interopRequireDefault(require("../models/conflicts/resolution-progress"));

var _editorConflictController = _interopRequireDefault(require("./editor-conflict-controller"));

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

const DEFAULT_REPO_DATA = {
  mergeConflictPaths: [],
  isRebasing: false
};
/**
 * Render an `EditorConflictController` for each `TextEditor` open on a file that contains git conflict markers.
 */

class RepositoryConflictController extends _react.default.Component {
  constructor(props, context) {
    super(props, context);

    _defineProperty(this, "fetchData", repository => {
      return (0, _yubikiri.default)({
        workingDirectoryPath: repository.getWorkingDirectoryPath(),
        mergeConflictPaths: repository.getMergeConflicts().then(conflicts => {
          return conflicts.map(conflict => conflict.filePath);
        }),
        isRebasing: repository.isRebasing()
      });
    });

    this.state = {
      openEditors: this.props.workspace.getTextEditors()
    };
    this.subscriptions = new _eventKit.CompositeDisposable();
  }

  componentDidMount() {
    const updateState = () => {
      this.setState({
        openEditors: this.props.workspace.getTextEditors()
      });
    };

    this.subscriptions.add(this.props.workspace.observeTextEditors(updateState), this.props.workspace.onDidDestroyPaneItem(updateState), this.props.config.observe('github.graphicalConflictResolution', () => this.forceUpdate()));
  }

  render() {
    return _react.default.createElement(_observeModel.default, {
      model: this.props.repository,
      fetchData: this.fetchData
    }, data => this.renderWithData(data || DEFAULT_REPO_DATA));
  }

  renderWithData(repoData) {
    const conflictingEditors = this.getConflictingEditors(repoData);
    return _react.default.createElement("div", null, conflictingEditors.map(editor => _react.default.createElement(_editorConflictController.default, {
      key: editor.id,
      commands: this.props.commands,
      resolutionProgress: this.props.resolutionProgress,
      editor: editor,
      isRebase: repoData.isRebasing,
      refreshResolutionProgress: this.props.refreshResolutionProgress
    })));
  }

  getConflictingEditors(repoData) {
    if (repoData.mergeConflictPaths.length === 0 || this.state.openEditors.length === 0 || !this.props.config.get('github.graphicalConflictResolution')) {
      return [];
    }

    const commonBasePath = this.props.repository.getWorkingDirectoryPath();
    const fullMergeConflictPaths = new Set(repoData.mergeConflictPaths.map(relativePath => _path.default.join(commonBasePath, relativePath)));
    return this.state.openEditors.filter(editor => fullMergeConflictPaths.has(editor.getPath()));
  }

  componentWillUnmount() {
    this.subscriptions.dispose();
  }

}

exports.default = RepositoryConflictController;

_defineProperty(RepositoryConflictController, "propTypes", {
  workspace: _propTypes.default.object.isRequired,
  commands: _propTypes.default.object.isRequired,
  config: _propTypes.default.object.isRequired,
  resolutionProgress: _propTypes.default.object.isRequired,
  repository: _propTypes.default.object.isRequired,
  refreshResolutionProgress: _propTypes.default.func
});

_defineProperty(RepositoryConflictController, "defaultProps", {
  refreshResolutionProgress: () => {},
  resolutionProgress: new _resolutionProgress.default()
});
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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