"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

var _fsExtra = _interopRequireDefault(require("fs-extra"));

var _path = _interopRequireDefault(require("path"));

var _electron = require("electron");

var _react = _interopRequireWildcard(require("react"));

var _propTypes = _interopRequireDefault(require("prop-types"));

var _eventKit = require("event-kit");

var _yubikiri = _interopRequireDefault(require("yubikiri"));

var _statusBar = _interopRequireDefault(require("../atom/status-bar"));

var _paneItem = _interopRequireDefault(require("../atom/pane-item"));

var _openIssueishDialog = require("../views/open-issueish-dialog");

var _openCommitDialog = require("../views/open-commit-dialog");

var _createDialog = require("../views/create-dialog");

var _observeModel = _interopRequireDefault(require("../views/observe-model"));

var _commands = _interopRequireWildcard(require("../atom/commands"));

var _changedFileItem = _interopRequireDefault(require("../items/changed-file-item"));

var _issueishDetailItem = _interopRequireDefault(require("../items/issueish-detail-item"));

var _commitDetailItem = _interopRequireDefault(require("../items/commit-detail-item"));

var _commitPreviewItem = _interopRequireDefault(require("../items/commit-preview-item"));

var _gitTabItem = _interopRequireDefault(require("../items/git-tab-item"));

var _githubTabItem = _interopRequireDefault(require("../items/github-tab-item"));

var _reviewsItem = _interopRequireDefault(require("../items/reviews-item"));

var _commentDecorationsContainer = _interopRequireDefault(require("../containers/comment-decorations-container"));

var _dialogsController = _interopRequireWildcard(require("./dialogs-controller"));

var _statusBarTileController = _interopRequireDefault(require("./status-bar-tile-controller"));

var _repositoryConflictController = _interopRequireDefault(require("./repository-conflict-controller"));

var _relayNetworkLayerManager = _interopRequireDefault(require("../relay-network-layer-manager"));

var _gitCacheView = _interopRequireDefault(require("../views/git-cache-view"));

var _gitTimingsView = _interopRequireDefault(require("../views/git-timings-view"));

var _conflict = _interopRequireDefault(require("../models/conflicts/conflict"));

var _endpoint = require("../models/endpoint");

var _switchboard = _interopRequireDefault(require("../switchboard"));

var _propTypes2 = require("../prop-types");

var _helpers = require("../helpers");

var _gitShellOutStrategy = require("../git-shell-out-strategy");

var _reporterProxy = require("../reporter-proxy");

function _interopRequireWildcard(obj) { if (obj && obj.__esModule) { return obj; } else { var newObj = {}; if (obj != null) { for (var key in obj) { if (Object.prototype.hasOwnProperty.call(obj, key)) { var desc = Object.defineProperty && Object.getOwnPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : {}; if (desc.get || desc.set) { Object.defineProperty(newObj, key, desc); } else { newObj[key] = obj[key]; } } } } newObj.default = obj; return newObj; } }

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

class RootController extends _react.default.Component {
  constructor(props, context) {
    super(props, context);

    _defineProperty(this, "fetchData", repository => (0, _yubikiri.default)({
      isPublishable: repository.isPublishable(),
      remotes: repository.getRemotes()
    }));

    _defineProperty(this, "closeDialog", () => new Promise(resolve => this.setState({
      dialogRequest: _dialogsController.dialogRequests.null
    }, resolve)));

    _defineProperty(this, "openInitializeDialog", async dirPath => {
      if (!dirPath) {
        const activeEditor = this.props.workspace.getActiveTextEditor();

        if (activeEditor) {
          const [projectPath] = this.props.project.relativizePath(activeEditor.getPath());

          if (projectPath) {
            dirPath = projectPath;
          }
        }
      }

      if (!dirPath) {
        const directories = this.props.project.getDirectories();
        const withRepositories = await Promise.all(directories.map(async d => [d, await this.props.project.repositoryForDirectory(d)]));
        const firstUninitialized = withRepositories.find(([d, r]) => !r);

        if (firstUninitialized && firstUninitialized[0]) {
          dirPath = firstUninitialized[0].getPath();
        }
      }

      if (!dirPath) {
        dirPath = this.props.config.get('core.projectHome');
      }

      const dialogRequest = _dialogsController.dialogRequests.init({
        dirPath
      });

      dialogRequest.onProgressingAccept(async chosenPath => {
        await this.props.initialize(chosenPath);
        await this.closeDialog();
      });
      dialogRequest.onCancel(this.closeDialog);
      return new Promise(resolve => this.setState({
        dialogRequest
      }, resolve));
    });

    _defineProperty(this, "openCloneDialog", opts => {
      const dialogRequest = _dialogsController.dialogRequests.clone(opts);

      dialogRequest.onProgressingAccept(async (url, chosenPath) => {
        await this.props.clone(url, chosenPath);
        await this.closeDialog();
      });
      dialogRequest.onCancel(this.closeDialog);
      return new Promise(resolve => this.setState({
        dialogRequest
      }, resolve));
    });

    _defineProperty(this, "openCredentialsDialog", query => {
      return new Promise((resolve, reject) => {
        const dialogRequest = _dialogsController.dialogRequests.credential(query);

        dialogRequest.onProgressingAccept(async result => {
          resolve(result);
          await this.closeDialog();
        });
        dialogRequest.onCancel(async () => {
          reject();
          await this.closeDialog();
        });
        this.setState({
          dialogRequest
        });
      });
    });

    _defineProperty(this, "openIssueishDialog", () => {
      const dialogRequest = _dialogsController.dialogRequests.issueish();

      dialogRequest.onProgressingAccept(async url => {
        await (0, _openIssueishDialog.openIssueishItem)(url, {
          workspace: this.props.workspace,
          workdir: this.props.repository.getWorkingDirectoryPath()
        });
        await this.closeDialog();
      });
      dialogRequest.onCancel(this.closeDialog);
      return new Promise(resolve => this.setState({
        dialogRequest
      }, resolve));
    });

    _defineProperty(this, "openCommitDialog", () => {
      const dialogRequest = _dialogsController.dialogRequests.commit();

      dialogRequest.onProgressingAccept(async ref => {
        await (0, _openCommitDialog.openCommitDetailItem)(ref, {
          workspace: this.props.workspace,
          repository: this.props.repository
        });
        await this.closeDialog();
      });
      dialogRequest.onCancel(this.closeDialog);
      return new Promise(resolve => this.setState({
        dialogRequest
      }, resolve));
    });

    _defineProperty(this, "openCreateDialog", () => {
      const dialogRequest = _dialogsController.dialogRequests.create();

      dialogRequest.onProgressingAccept(async result => {
        const dotcom = (0, _endpoint.getEndpoint)('github.com');

        const relayEnvironment = _relayNetworkLayerManager.default.getEnvironmentForHost(dotcom);

        await (0, _createDialog.createRepository)(result, {
          clone: this.props.clone,
          relayEnvironment
        });
        await this.closeDialog();
      });
      dialogRequest.onCancel(this.closeDialog);
      return new Promise(resolve => this.setState({
        dialogRequest
      }, resolve));
    });

    _defineProperty(this, "openPublishDialog", repository => {
      const dialogRequest = _dialogsController.dialogRequests.publish({
        localDir: repository.getWorkingDirectoryPath()
      });

      dialogRequest.onProgressingAccept(async result => {
        const dotcom = (0, _endpoint.getEndpoint)('github.com');

        const relayEnvironment = _relayNetworkLayerManager.default.getEnvironmentForHost(dotcom);

        await (0, _createDialog.publishRepository)(result, {
          repository,
          relayEnvironment
        });
        await this.closeDialog();
      });
      dialogRequest.onCancel(this.closeDialog);
      return new Promise(resolve => this.setState({
        dialogRequest
      }, resolve));
    });

    _defineProperty(this, "toggleCommitPreviewItem", () => {
      const workdir = this.props.repository.getWorkingDirectoryPath();
      return this.props.workspace.toggle(_commitPreviewItem.default.buildURI(workdir));
    });

    _defineProperty(this, "surfaceFromFileAtPath", (filePath, stagingStatus) => {
      const gitTab = this.gitTabTracker.getComponent();
      return gitTab && gitTab.focusAndSelectStagingItem(filePath, stagingStatus);
    });

    _defineProperty(this, "surfaceToCommitPreviewButton", () => {
      const gitTab = this.gitTabTracker.getComponent();
      return gitTab && gitTab.focusAndSelectCommitPreviewButton();
    });

    _defineProperty(this, "surfaceToRecentCommit", () => {
      const gitTab = this.gitTabTracker.getComponent();
      return gitTab && gitTab.focusAndSelectRecentCommit();
    });

    _defineProperty(this, "reportRelayError", (friendlyMessage, err) => {
      const opts = {
        dismissable: true
      };

      if (err.network) {
        // Offline
        opts.icon = 'alignment-unalign';
        opts.description = "It looks like you're offline right now.";
      } else if (err.responseText) {
        // Transient error like a 500 from the API
        opts.description = 'The GitHub API reported a problem.';
        opts.detail = err.responseText;
      } else if (err.errors) {
        // GraphQL errors
        opts.detail = err.errors.map(e => e.message).join('\n');
      } else {
        opts.detail = err.stack;
      }

      this.props.notificationManager.addError(friendlyMessage, opts);
    });

    (0, _helpers.autobind)(this, 'installReactDevTools', 'clearGithubToken', 'showWaterfallDiagnostics', 'showCacheDiagnostics', 'destroyFilePatchPaneItems', 'destroyEmptyFilePatchPaneItems', 'quietlySelectItem', 'viewUnstagedChangesForCurrentFile', 'viewStagedChangesForCurrentFile', 'openFiles', 'getUnsavedFiles', 'ensureNoUnsavedFiles', 'discardWorkDirChangesForPaths', 'discardLines', 'undoLastDiscard', 'refreshResolutionProgress');
    this.state = {
      dialogRequest: _dialogsController.dialogRequests.null
    };
    this.gitTabTracker = new TabTracker('git', {
      uri: _gitTabItem.default.buildURI(),
      getWorkspace: () => this.props.workspace
    });
    this.githubTabTracker = new TabTracker('github', {
      uri: _githubTabItem.default.buildURI(),
      getWorkspace: () => this.props.workspace
    });
    this.subscription = new _eventKit.CompositeDisposable(this.props.repository.onPullError(this.gitTabTracker.ensureVisible));
    this.props.commands.onDidDispatch(event => {
      if (event.type && event.type.startsWith('github:') && event.detail && event.detail[0] && event.detail[0].contextCommand) {
        (0, _reporterProxy.addEvent)('context-menu-action', {
          package: 'github',
          command: event.type
        });
      }
    });
  }

  componentDidMount() {
    this.openTabs();
  }

  render() {
    return _react.default.createElement(_react.Fragment, null, this.renderCommands(), this.renderStatusBarTile(), this.renderPaneItems(), this.renderDialogs(), this.renderConflictResolver(), this.renderCommentDecorations());
  }

  renderCommands() {
    const devMode = global.atom && global.atom.inDevMode();
    return _react.default.createElement(_react.Fragment, null, _react.default.createElement(_commands.default, {
      registry: this.props.commands,
      target: "atom-workspace"
    }, devMode && _react.default.createElement(_commands.Command, {
      command: "github:install-react-dev-tools",
      callback: this.installReactDevTools
    }), _react.default.createElement(_commands.Command, {
      command: "github:toggle-commit-preview",
      callback: this.toggleCommitPreviewItem
    }), _react.default.createElement(_commands.Command, {
      command: "github:logout",
      callback: this.clearGithubToken
    }), _react.default.createElement(_commands.Command, {
      command: "github:show-waterfall-diagnostics",
      callback: this.showWaterfallDiagnostics
    }), _react.default.createElement(_commands.Command, {
      command: "github:show-cache-diagnostics",
      callback: this.showCacheDiagnostics
    }), _react.default.createElement(_commands.Command, {
      command: "github:toggle-git-tab",
      callback: this.gitTabTracker.toggle
    }), _react.default.createElement(_commands.Command, {
      command: "github:toggle-git-tab-focus",
      callback: this.gitTabTracker.toggleFocus
    }), _react.default.createElement(_commands.Command, {
      command: "github:toggle-github-tab",
      callback: this.githubTabTracker.toggle
    }), _react.default.createElement(_commands.Command, {
      command: "github:toggle-github-tab-focus",
      callback: this.githubTabTracker.toggleFocus
    }), _react.default.createElement(_commands.Command, {
      command: "github:initialize",
      callback: () => this.openInitializeDialog()
    }), _react.default.createElement(_commands.Command, {
      command: "github:clone",
      callback: () => this.openCloneDialog()
    }), _react.default.createElement(_commands.Command, {
      command: "github:open-issue-or-pull-request",
      callback: () => this.openIssueishDialog()
    }), _react.default.createElement(_commands.Command, {
      command: "github:open-commit",
      callback: () => this.openCommitDialog()
    }), _react.default.createElement(_commands.Command, {
      command: "github:create-repository",
      callback: () => this.openCreateDialog()
    }), _react.default.createElement(_commands.Command, {
      command: "github:view-unstaged-changes-for-current-file",
      callback: this.viewUnstagedChangesForCurrentFile
    }), _react.default.createElement(_commands.Command, {
      command: "github:view-staged-changes-for-current-file",
      callback: this.viewStagedChangesForCurrentFile
    }), _react.default.createElement(_commands.Command, {
      command: "github:close-all-diff-views",
      callback: this.destroyFilePatchPaneItems
    }), _react.default.createElement(_commands.Command, {
      command: "github:close-empty-diff-views",
      callback: this.destroyEmptyFilePatchPaneItems
    })), _react.default.createElement(_observeModel.default, {
      model: this.props.repository,
      fetchData: this.fetchData
    }, data => {
      if (!data || !data.isPublishable || !data.remotes.filter(r => r.isGithubRepo()).isEmpty()) {
        return null;
      }

      return _react.default.createElement(_commands.default, {
        registry: this.props.commands,
        target: "atom-workspace"
      }, _react.default.createElement(_commands.Command, {
        command: "github:publish-repository",
        callback: () => this.openPublishDialog(this.props.repository)
      }));
    }));
  }

  renderStatusBarTile() {
    return _react.default.createElement(_statusBar.default, {
      statusBar: this.props.statusBar,
      onConsumeStatusBar: sb => this.onConsumeStatusBar(sb),
      className: "github-StatusBarTileController"
    }, _react.default.createElement(_statusBarTileController.default, {
      pipelineManager: this.props.pipelineManager,
      workspace: this.props.workspace,
      repository: this.props.repository,
      commands: this.props.commands,
      notificationManager: this.props.notificationManager,
      tooltips: this.props.tooltips,
      confirm: this.props.confirm,
      toggleGitTab: this.gitTabTracker.toggle,
      toggleGithubTab: this.githubTabTracker.toggle
    }));
  }

  renderDialogs() {
    return _react.default.createElement(_dialogsController.default, {
      loginModel: this.props.loginModel,
      request: this.state.dialogRequest,
      currentWindow: this.props.currentWindow,
      workspace: this.props.workspace,
      commands: this.props.commands,
      config: this.props.config
    });
  }

  renderCommentDecorations() {
    if (!this.props.repository) {
      return null;
    }

    return _react.default.createElement(_commentDecorationsContainer.default, {
      workspace: this.props.workspace,
      commands: this.props.commands,
      localRepository: this.props.repository,
      loginModel: this.props.loginModel,
      reportRelayError: this.reportRelayError
    });
  }

  renderConflictResolver() {
    if (!this.props.repository) {
      return null;
    }

    return _react.default.createElement(_repositoryConflictController.default, {
      workspace: this.props.workspace,
      config: this.props.config,
      repository: this.props.repository,
      resolutionProgress: this.props.resolutionProgress,
      refreshResolutionProgress: this.refreshResolutionProgress,
      commands: this.props.commands
    });
  }

  renderPaneItems() {
    const {
      workdirContextPool
    } = this.props;
    const getCurrentWorkDirs = workdirContextPool.getCurrentWorkDirs.bind(workdirContextPool);
    const onDidChangeWorkDirs = workdirContextPool.onDidChangePoolContexts.bind(workdirContextPool);
    return _react.default.createElement(_react.Fragment, null, _react.default.createElement(_paneItem.default, {
      workspace: this.props.workspace,
      uriPattern: _gitTabItem.default.uriPattern,
      className: "github-Git-root"
    }, ({
      itemHolder
    }) => _react.default.createElement(_gitTabItem.default, {
      ref: itemHolder.setter,
      workspace: this.props.workspace,
      commands: this.props.commands,
      notificationManager: this.props.notificationManager,
      tooltips: this.props.tooltips,
      grammars: this.props.grammars,
      project: this.props.project,
      confirm: this.props.confirm,
      config: this.props.config,
      repository: this.props.repository,
      loginModel: this.props.loginModel,
      openInitializeDialog: this.openInitializeDialog,
      resolutionProgress: this.props.resolutionProgress,
      ensureGitTab: this.gitTabTracker.ensureVisible,
      openFiles: this.openFiles,
      discardWorkDirChangesForPaths: this.discardWorkDirChangesForPaths,
      undoLastDiscard: this.undoLastDiscard,
      refreshResolutionProgress: this.refreshResolutionProgress,
      currentWorkDir: this.props.currentWorkDir,
      getCurrentWorkDirs: getCurrentWorkDirs,
      onDidChangeWorkDirs: onDidChangeWorkDirs,
      contextLocked: this.props.contextLocked,
      changeWorkingDirectory: this.props.changeWorkingDirectory,
      setContextLock: this.props.setContextLock
    })), _react.default.createElement(_paneItem.default, {
      workspace: this.props.workspace,
      uriPattern: _githubTabItem.default.uriPattern,
      className: "github-GitHub-root"
    }, ({
      itemHolder
    }) => _react.default.createElement(_githubTabItem.default, {
      ref: itemHolder.setter,
      repository: this.props.repository,
      loginModel: this.props.loginModel,
      workspace: this.props.workspace,
      currentWorkDir: this.props.currentWorkDir,
      getCurrentWorkDirs: getCurrentWorkDirs,
      onDidChangeWorkDirs: onDidChangeWorkDirs,
      contextLocked: this.props.contextLocked,
      changeWorkingDirectory: this.props.changeWorkingDirectory,
      setContextLock: this.props.setContextLock,
      openCreateDialog: this.openCreateDialog,
      openPublishDialog: this.openPublishDialog,
      openCloneDialog: this.openCloneDialog,
      openGitTab: this.gitTabTracker.toggleFocus
    })), _react.default.createElement(_paneItem.default, {
      workspace: this.props.workspace,
      uriPattern: _changedFileItem.default.uriPattern
    }, ({
      itemHolder,
      params
    }) => _react.default.createElement(_changedFileItem.default, {
      ref: itemHolder.setter,
      workdirContextPool: this.props.workdirContextPool,
      relPath: _path.default.join(...params.relPath),
      workingDirectory: params.workingDirectory,
      stagingStatus: params.stagingStatus,
      tooltips: this.props.tooltips,
      commands: this.props.commands,
      keymaps: this.props.keymaps,
      workspace: this.props.workspace,
      config: this.props.config,
      discardLines: this.discardLines,
      undoLastDiscard: this.undoLastDiscard,
      surfaceFileAtPath: this.surfaceFromFileAtPath
    })), _react.default.createElement(_paneItem.default, {
      workspace: this.props.workspace,
      uriPattern: _commitPreviewItem.default.uriPattern,
      className: "github-CommitPreview-root"
    }, ({
      itemHolder,
      params
    }) => _react.default.createElement(_commitPreviewItem.default, {
      ref: itemHolder.setter,
      workdirContextPool: this.props.workdirContextPool,
      workingDirectory: params.workingDirectory,
      workspace: this.props.workspace,
      commands: this.props.commands,
      keymaps: this.props.keymaps,
      tooltips: this.props.tooltips,
      config: this.props.config,
      discardLines: this.discardLines,
      undoLastDiscard: this.undoLastDiscard,
      surfaceToCommitPreviewButton: this.surfaceToCommitPreviewButton
    })), _react.default.createElement(_paneItem.default, {
      workspace: this.props.workspace,
      uriPattern: _commitDetailItem.default.uriPattern,
      className: "github-CommitDetail-root"
    }, ({
      itemHolder,
      params
    }) => _react.default.createElement(_commitDetailItem.default, {
      ref: itemHolder.setter,
      workdirContextPool: this.props.workdirContextPool,
      workingDirectory: params.workingDirectory,
      workspace: this.props.workspace,
      commands: this.props.commands,
      keymaps: this.props.keymaps,
      tooltips: this.props.tooltips,
      config: this.props.config,
      sha: params.sha,
      surfaceCommit: this.surfaceToRecentCommit
    })), _react.default.createElement(_paneItem.default, {
      workspace: this.props.workspace,
      uriPattern: _issueishDetailItem.default.uriPattern
    }, ({
      itemHolder,
      params,
      deserialized
    }) => _react.default.createElement(_issueishDetailItem.default, {
      ref: itemHolder.setter,
      host: params.host,
      owner: params.owner,
      repo: params.repo,
      issueishNumber: parseInt(params.issueishNumber, 10),
      workingDirectory: params.workingDirectory,
      workdirContextPool: this.props.workdirContextPool,
      loginModel: this.props.loginModel,
      initSelectedTab: deserialized.initSelectedTab,
      workspace: this.props.workspace,
      commands: this.props.commands,
      keymaps: this.props.keymaps,
      tooltips: this.props.tooltips,
      config: this.props.config,
      reportRelayError: this.reportRelayError
    })), _react.default.createElement(_paneItem.default, {
      workspace: this.props.workspace,
      uriPattern: _reviewsItem.default.uriPattern
    }, ({
      itemHolder,
      params
    }) => _react.default.createElement(_reviewsItem.default, {
      ref: itemHolder.setter,
      host: params.host,
      owner: params.owner,
      repo: params.repo,
      number: parseInt(params.number, 10),
      workdir: params.workdir,
      workdirContextPool: this.props.workdirContextPool,
      loginModel: this.props.loginModel,
      workspace: this.props.workspace,
      tooltips: this.props.tooltips,
      config: this.props.config,
      commands: this.props.commands,
      confirm: this.props.confirm,
      reportRelayError: this.reportRelayError
    })), _react.default.createElement(_paneItem.default, {
      workspace: this.props.workspace,
      uriPattern: _gitTimingsView.default.uriPattern
    }, ({
      itemHolder
    }) => _react.default.createElement(_gitTimingsView.default, {
      ref: itemHolder.setter
    })), _react.default.createElement(_paneItem.default, {
      workspace: this.props.workspace,
      uriPattern: _gitCacheView.default.uriPattern
    }, ({
      itemHolder
    }) => _react.default.createElement(_gitCacheView.default, {
      ref: itemHolder.setter,
      repository: this.props.repository
    })));
  }

  async openTabs() {
    if (this.props.startOpen) {
      await Promise.all([this.gitTabTracker.ensureRendered(false), this.githubTabTracker.ensureRendered(false)]);
    }

    if (this.props.startRevealed) {
      const docks = new Set([_gitTabItem.default.buildURI(), _githubTabItem.default.buildURI()].map(uri => this.props.workspace.paneContainerForURI(uri)).filter(container => container && typeof container.show === 'function'));

      for (const dock of docks) {
        dock.show();
      }
    }
  }

  async installReactDevTools() {
    // Prevent electron-link from attempting to descend into electron-devtools-installer, which is not available
    // when we're bundled in Atom.
    const devToolsName = 'electron-devtools-installer';

    const devTools = require(devToolsName);

    await Promise.all([this.installExtension(devTools.REACT_DEVELOPER_TOOLS.id), // relay developer tools extension id
    this.installExtension('ncedobpgnmkhcmnnkcimnobpfepidadl')]);
    this.props.notificationManager.addSuccess('🌈 Reload your window to start using the React/Relay dev tools!');
  }

  async installExtension(id) {
    const devToolsName = 'electron-devtools-installer';

    const devTools = require(devToolsName);

    const crossUnzipName = 'cross-unzip';

    const unzip = require(crossUnzipName);

    const url = 'https://clients2.google.com/service/update2/crx?' + `response=redirect&x=id%3D${id}%26uc&prodversion=32`;

    const extensionFolder = _path.default.resolve(_electron.remote.app.getPath('userData'), `extensions/${id}`);

    const extensionFile = `${extensionFolder}.crx`;
    await _fsExtra.default.ensureDir(_path.default.dirname(extensionFile));
    const response = await fetch(url, {
      method: 'GET'
    });
    const body = Buffer.from((await response.arrayBuffer()));
    await _fsExtra.default.writeFile(extensionFile, body);
    await new Promise((resolve, reject) => {
      unzip(extensionFile, extensionFolder, async err => {
        if (err && !(await _fsExtra.default.exists(_path.default.join(extensionFolder, 'manifest.json')))) {
          reject(err);
        }

        resolve();
      });
    });
    await _fsExtra.default.ensureDir(extensionFolder, 0o755);
    await devTools.default(id);
  }

  componentWillUnmount() {
    this.subscription.dispose();
  }

  componentDidUpdate() {
    this.subscription.dispose();
    this.subscription = new _eventKit.CompositeDisposable(this.props.repository.onPullError(() => this.gitTabTracker.ensureVisible()));
  }

  onConsumeStatusBar(statusBar) {
    if (statusBar.disableGitInfoTile) {
      statusBar.disableGitInfoTile();
    }
  }

  clearGithubToken() {
    return this.props.loginModel.removeToken('https://api.github.com');
  }

  showWaterfallDiagnostics() {
    this.props.workspace.open(_gitTimingsView.default.buildURI());
  }

  showCacheDiagnostics() {
    this.props.workspace.open(_gitCacheView.default.buildURI());
  }

  destroyFilePatchPaneItems() {
    (0, _helpers.destroyFilePatchPaneItems)({
      onlyStaged: false
    }, this.props.workspace);
  }

  destroyEmptyFilePatchPaneItems() {
    (0, _helpers.destroyEmptyFilePatchPaneItems)(this.props.workspace);
  }

  quietlySelectItem(filePath, stagingStatus) {
    const gitTab = this.gitTabTracker.getComponent();
    return gitTab && gitTab.quietlySelectItem(filePath, stagingStatus);
  }

  async viewChangesForCurrentFile(stagingStatus) {
    const editor = this.props.workspace.getActiveTextEditor();

    if (!editor.getPath()) {
      return;
    }

    const absFilePath = await _fsExtra.default.realpath(editor.getPath());
    const repoPath = this.props.repository.getWorkingDirectoryPath();

    if (repoPath === null) {
      const [projectPath] = this.props.project.relativizePath(editor.getPath());
      const notification = this.props.notificationManager.addInfo("Hmm, there's nothing to compare this file to", {
        description: 'You can create a Git repository to track changes to the files in your project.',
        dismissable: true,
        buttons: [{
          className: 'btn btn-primary',
          text: 'Create a repository now',
          onDidClick: async () => {
            notification.dismiss();
            const createdPath = await this.initializeRepo(projectPath); // If the user confirmed repository creation for this project path,
            // retry the operation that got them here in the first place

            if (createdPath === projectPath) {
              this.viewChangesForCurrentFile(stagingStatus);
            }
          }
        }]
      });
      return;
    }

    if (absFilePath.startsWith(repoPath)) {
      const filePath = absFilePath.slice(repoPath.length + 1);
      this.quietlySelectItem(filePath, stagingStatus);
      const splitDirection = this.props.config.get('github.viewChangesForCurrentFileDiffPaneSplitDirection');
      const pane = this.props.workspace.getActivePane();

      if (splitDirection === 'right') {
        pane.splitRight();
      } else if (splitDirection === 'down') {
        pane.splitDown();
      }

      const lineNum = editor.getCursorBufferPosition().row + 1;
      const item = await this.props.workspace.open(_changedFileItem.default.buildURI(filePath, repoPath, stagingStatus), {
        pending: true,
        activatePane: true,
        activateItem: true
      });
      await item.getRealItemPromise();
      await item.getFilePatchLoadedPromise();
      item.goToDiffLine(lineNum);
      item.focus();
    } else {
      throw new Error(`${absFilePath} does not belong to repo ${repoPath}`);
    }
  }

  viewUnstagedChangesForCurrentFile() {
    return this.viewChangesForCurrentFile('unstaged');
  }

  viewStagedChangesForCurrentFile() {
    return this.viewChangesForCurrentFile('staged');
  }

  openFiles(filePaths, repository = this.props.repository) {
    return Promise.all(filePaths.map(filePath => {
      const absolutePath = _path.default.join(repository.getWorkingDirectoryPath(), filePath);

      return this.props.workspace.open(absolutePath, {
        pending: filePaths.length === 1
      });
    }));
  }

  getUnsavedFiles(filePaths, workdirPath) {
    const isModifiedByPath = new Map();
    this.props.workspace.getTextEditors().forEach(editor => {
      isModifiedByPath.set(editor.getPath(), editor.isModified());
    });
    return filePaths.filter(filePath => {
      const absFilePath = _path.default.join(workdirPath, filePath);

      return isModifiedByPath.get(absFilePath);
    });
  }

  ensureNoUnsavedFiles(filePaths, message, workdirPath = this.props.repository.getWorkingDirectoryPath()) {
    const unsavedFiles = this.getUnsavedFiles(filePaths, workdirPath).map(filePath => `\`${filePath}\``).join('<br>');

    if (unsavedFiles.length) {
      this.props.notificationManager.addError(message, {
        description: `You have unsaved changes in:<br>${unsavedFiles}.`,
        dismissable: true
      });
      return false;
    } else {
      return true;
    }
  }

  async discardWorkDirChangesForPaths(filePaths) {
    const destructiveAction = () => {
      return this.props.repository.discardWorkDirChangesForPaths(filePaths);
    };

    return await this.props.repository.storeBeforeAndAfterBlobs(filePaths, () => this.ensureNoUnsavedFiles(filePaths, 'Cannot discard changes in selected files.'), destructiveAction);
  }

  async discardLines(multiFilePatch, lines, repository = this.props.repository) {
    // (kuychaco) For now we only support discarding rows for MultiFilePatches that contain a single file patch
    // The only way to access this method from the UI is to be in a ChangedFileItem, which only has a single file patch
    if (multiFilePatch.getFilePatches().length !== 1) {
      return Promise.resolve(null);
    }

    const filePath = multiFilePatch.getFilePatches()[0].getPath();

    const destructiveAction = async () => {
      const discardFilePatch = multiFilePatch.getUnstagePatchForLines(lines);
      await repository.applyPatchToWorkdir(discardFilePatch);
    };

    return await repository.storeBeforeAndAfterBlobs([filePath], () => this.ensureNoUnsavedFiles([filePath], 'Cannot discard lines.', repository.getWorkingDirectoryPath()), destructiveAction, filePath);
  }

  getFilePathsForLastDiscard(partialDiscardFilePath = null) {
    let lastSnapshots = this.props.repository.getLastHistorySnapshots(partialDiscardFilePath);

    if (partialDiscardFilePath) {
      lastSnapshots = lastSnapshots ? [lastSnapshots] : [];
    }

    return lastSnapshots.map(snapshot => snapshot.filePath);
  }

  async undoLastDiscard(partialDiscardFilePath = null, repository = this.props.repository) {
    const filePaths = this.getFilePathsForLastDiscard(partialDiscardFilePath);

    try {
      const results = await repository.restoreLastDiscardInTempFiles(() => this.ensureNoUnsavedFiles(filePaths, 'Cannot undo last discard.'), partialDiscardFilePath);

      if (results.length === 0) {
        return;
      }

      await this.proceedOrPromptBasedOnResults(results, partialDiscardFilePath);
    } catch (e) {
      if (e instanceof _gitShellOutStrategy.GitError && e.stdErr.match(/fatal: Not a valid object name/)) {
        this.cleanUpHistoryForFilePaths(filePaths, partialDiscardFilePath);
      } else {
        // eslint-disable-next-line no-console
        console.error(e);
      }
    }
  }

  async proceedOrPromptBasedOnResults(results, partialDiscardFilePath = null) {
    const conflicts = results.filter(({
      conflict
    }) => conflict);

    if (conflicts.length === 0) {
      await this.proceedWithLastDiscardUndo(results, partialDiscardFilePath);
    } else {
      await this.promptAboutConflicts(results, conflicts, partialDiscardFilePath);
    }
  }

  async promptAboutConflicts(results, conflicts, partialDiscardFilePath = null) {
    const conflictedFiles = conflicts.map(({
      filePath
    }) => `\t${filePath}`).join('\n');
    const choice = this.props.confirm({
      message: 'Undoing will result in conflicts...',
      detailedMessage: `for the following files:\n${conflictedFiles}\n` + 'Would you like to apply the changes with merge conflict markers, ' + 'or open the text with merge conflict markers in a new file?',
      buttons: ['Merge with conflict markers', 'Open in new file', 'Cancel']
    });

    if (choice === 0) {
      await this.proceedWithLastDiscardUndo(results, partialDiscardFilePath);
    } else if (choice === 1) {
      await this.openConflictsInNewEditors(conflicts.map(({
        resultPath
      }) => resultPath));
    }
  }

  cleanUpHistoryForFilePaths(filePaths, partialDiscardFilePath = null) {
    this.props.repository.clearDiscardHistory(partialDiscardFilePath);
    const filePathsStr = filePaths.map(filePath => `\`${filePath}\``).join('<br>');
    this.props.notificationManager.addError('Discard history has expired.', {
      description: `Cannot undo discard for<br>${filePathsStr}<br>Stale discard history has been deleted.`,
      dismissable: true
    });
  }

  async proceedWithLastDiscardUndo(results, partialDiscardFilePath = null) {
    const promises = results.map(async result => {
      const {
        filePath,
        resultPath,
        deleted,
        conflict,
        theirsSha,
        commonBaseSha,
        currentSha
      } = result;

      const absFilePath = _path.default.join(this.props.repository.getWorkingDirectoryPath(), filePath);

      if (deleted && resultPath === null) {
        await _fsExtra.default.remove(absFilePath);
      } else {
        await _fsExtra.default.copy(resultPath, absFilePath);
      }

      if (conflict) {
        await this.props.repository.writeMergeConflictToIndex(filePath, commonBaseSha, currentSha, theirsSha);
      }
    });
    await Promise.all(promises);
    await this.props.repository.popDiscardHistory(partialDiscardFilePath);
  }

  async openConflictsInNewEditors(resultPaths) {
    const editorPromises = resultPaths.map(resultPath => {
      return this.props.workspace.open(resultPath);
    });
    return await Promise.all(editorPromises);
  }

  /*
   * Asynchronously count the conflict markers present in a file specified by full path.
   */
  refreshResolutionProgress(fullPath) {
    const readStream = _fsExtra.default.createReadStream(fullPath, {
      encoding: 'utf8'
    });

    return new Promise(resolve => {
      _conflict.default.countFromStream(readStream).then(count => {
        this.props.resolutionProgress.reportMarkerCount(fullPath, count);
      });
    });
  }

}

exports.default = RootController;

_defineProperty(RootController, "propTypes", {
  // Atom enviornment
  workspace: _propTypes.default.object.isRequired,
  commands: _propTypes.default.object.isRequired,
  deserializers: _propTypes.default.object.isRequired,
  notificationManager: _propTypes.default.object.isRequired,
  tooltips: _propTypes.default.object.isRequired,
  keymaps: _propTypes.default.object.isRequired,
  grammars: _propTypes.default.object.isRequired,
  config: _propTypes.default.object.isRequired,
  project: _propTypes.default.object.isRequired,
  confirm: _propTypes.default.func.isRequired,
  currentWindow: _propTypes.default.object.isRequired,
  // Models
  loginModel: _propTypes.default.object.isRequired,
  workdirContextPool: _propTypes2.WorkdirContextPoolPropType.isRequired,
  repository: _propTypes.default.object.isRequired,
  resolutionProgress: _propTypes.default.object.isRequired,
  statusBar: _propTypes.default.object,
  switchboard: _propTypes.default.instanceOf(_switchboard.default),
  pipelineManager: _propTypes.default.object,
  currentWorkDir: _propTypes.default.string,
  // Git actions
  initialize: _propTypes.default.func.isRequired,
  clone: _propTypes.default.func.isRequired,
  // Control
  contextLocked: _propTypes.default.bool.isRequired,
  changeWorkingDirectory: _propTypes.default.func.isRequired,
  setContextLock: _propTypes.default.func.isRequired,
  startOpen: _propTypes.default.bool,
  startRevealed: _propTypes.default.bool
});

_defineProperty(RootController, "defaultProps", {
  switchboard: new _switchboard.default(),
  startOpen: false,
  startRevealed: false
});

class TabTracker {
  constructor(name, {
    getWorkspace,
    uri
  }) {
    (0, _helpers.autobind)(this, 'toggle', 'toggleFocus', 'ensureVisible');
    this.name = name;
    this.getWorkspace = getWorkspace;
    this.uri = uri;
  }

  async toggle() {
    const focusToRestore = document.activeElement;
    let shouldRestoreFocus = false; // Rendered => the dock item is being rendered, whether or not the dock is visible or the item
    //   is visible within its dock.
    // Visible => the item is active and the dock item is active within its dock.

    const wasRendered = this.isRendered();
    const wasVisible = this.isVisible();

    if (!wasRendered || !wasVisible) {
      // Not rendered, or rendered but not an active item in a visible dock.
      await this.reveal();
      shouldRestoreFocus = true;
    } else {
      // Rendered and an active item within a visible dock.
      await this.hide();
      shouldRestoreFocus = false;
    }

    if (shouldRestoreFocus) {
      process.nextTick(() => focusToRestore.focus());
    }
  }

  async toggleFocus() {
    const hadFocus = this.hasFocus();
    await this.ensureVisible();

    if (hadFocus) {
      let workspace = this.getWorkspace();

      if (workspace.getCenter) {
        workspace = workspace.getCenter();
      }

      workspace.getActivePane().activate();
    } else {
      this.focus();
    }
  }

  async ensureVisible() {
    if (!this.isVisible()) {
      await this.reveal();
      return true;
    }

    return false;
  }

  ensureRendered() {
    return this.getWorkspace().open(this.uri, {
      searchAllPanes: true,
      activateItem: false,
      activatePane: false
    });
  }

  reveal() {
    (0, _reporterProxy.incrementCounter)(`${this.name}-tab-open`);
    return this.getWorkspace().open(this.uri, {
      searchAllPanes: true,
      activateItem: true,
      activatePane: true
    });
  }

  hide() {
    (0, _reporterProxy.incrementCounter)(`${this.name}-tab-close`);
    return this.getWorkspace().hide(this.uri);
  }

  focus() {
    this.getComponent().restoreFocus();
  }

  getItem() {
    const pane = this.getWorkspace().paneForURI(this.uri);

    if (!pane) {
      return null;
    }

    const paneItem = pane.itemForURI(this.uri);

    if (!paneItem) {
      return null;
    }

    return paneItem;
  }

  getComponent() {
    const paneItem = this.getItem();

    if (!paneItem) {
      return null;
    }

    if (typeof paneItem.getRealItem !== 'function') {
      return null;
    }

    return paneItem.getRealItem();
  }

  getDOMElement() {
    const paneItem = this.getItem();

    if (!paneItem) {
      return null;
    }

    if (typeof paneItem.getElement !== 'function') {
      return null;
    }

    return paneItem.getElement();
  }

  isRendered() {
    return !!this.getWorkspace().paneForURI(this.uri);
  }

  isVisible() {
    const workspace = this.getWorkspace();
    return workspace.getPaneContainers().filter(container => container === workspace.getCenter() || container.isVisible()).some(container => container.getPanes().some(pane => {
      const item = pane.getActiveItem();
      return item && item.getURI && item.getURI() === this.uri;
    }));
  }

  hasFocus() {
    const root = this.getDOMElement();
    return root && root.contains(document.activeElement);
  }

}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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