"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

var _net = _interopRequireDefault(require("net"));

var _eventKit = require("event-kit");

var _helpers = require("./helpers");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

class GitPromptServer {
  constructor(gitTempDir) {
    this.emitter = new _eventKit.Emitter();
    this.gitTempDir = gitTempDir;
    this.address = null;
  }

  async start(promptForInput) {
    this.promptForInput = promptForInput;
    await this.gitTempDir.ensure();
    this.server = await this.startListening(this.gitTempDir.getSocketOptions());
  }

  getAddress() {
    /* istanbul ignore if */
    if (!this.address) {
      throw new Error('Server is not listening');
    } else if (this.address.port) {
      // TCP socket
      return `tcp:${this.address.port}`;
    } else {
      // Unix domain socket
      return `unix:${(0, _helpers.normalizeGitHelperPath)(this.address)}`;
    }
  }

  startListening(socketOptions) {
    return new Promise(resolve => {
      const server = _net.default.createServer({
        allowHalfOpen: true
      }, connection => {
        connection.setEncoding('utf8');
        let payload = '';
        connection.on('data', data => {
          payload += data;
        });
        connection.on('end', () => {
          this.handleData(connection, payload);
        });
      });

      server.listen(socketOptions, () => {
        this.address = server.address();
        resolve(server);
      });
    });
  }

  async handleData(connection, data) {
    let query;

    try {
      query = JSON.parse(data);
      const answer = await this.promptForInput(query);
      await new Promise(resolve => {
        connection.end(JSON.stringify(answer), 'utf8', resolve);
      });
    } catch (e) {
      this.emitter.emit('did-cancel', query.pid ? {
        handlerPid: query.pid
      } : undefined);
    }
  }

  onDidCancel(cb) {
    return this.emitter.on('did-cancel', cb);
  }

  async terminate() {
    await new Promise(resolve => this.server.close(resolve));
    this.emitter.dispose();
  }

}

exports.default = GitPromptServer;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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