"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = exports.LargeRepoError = exports.GitError = void 0;

var _path = _interopRequireDefault(require("path"));

var _os = _interopRequireDefault(require("os"));

var _child_process = _interopRequireDefault(require("child_process"));

var _fsExtra = _interopRequireDefault(require("fs-extra"));

var _util = _interopRequireDefault(require("util"));

var _electron = require("electron");

var _eventKit = require("event-kit");

var _dugite = require("dugite");

var _whatTheDiff = require("what-the-diff");

var _whatTheStatus = require("what-the-status");

var _gitPromptServer = _interopRequireDefault(require("./git-prompt-server"));

var _gitTempDir = _interopRequireDefault(require("./git-temp-dir"));

var _asyncQueue = _interopRequireDefault(require("./async-queue"));

var _reporterProxy = require("./reporter-proxy");

var _helpers = require("./helpers");

var _gitTimingsView = _interopRequireDefault(require("./views/git-timings-view"));

var _file = _interopRequireDefault(require("./models/patch/file"));

var _workerManager = _interopRequireDefault(require("./worker-manager"));

var _author = _interopRequireDefault(require("./models/author"));

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _objectSpread2(target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i] != null ? arguments[i] : {}; var ownKeys = Object.keys(source); if (typeof Object.getOwnPropertySymbols === 'function') { ownKeys = ownKeys.concat(Object.getOwnPropertySymbols(source).filter(function (sym) { return Object.getOwnPropertyDescriptor(source, sym).enumerable; })); } ownKeys.forEach(function (key) { _defineProperty(target, key, source[key]); }); } return target; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

const MAX_STATUS_OUTPUT_LENGTH = 1024 * 1024 * 10;
let headless = null;
let execPathPromise = null;

class GitError extends Error {
  constructor(message) {
    super(message);
    this.message = message;
    this.stack = new Error().stack;
  }

}

exports.GitError = GitError;

class LargeRepoError extends Error {
  constructor(message) {
    super(message);
    this.message = message;
    this.stack = new Error().stack;
  }

} // ignored for the purposes of usage metrics tracking because they're noisy


exports.LargeRepoError = LargeRepoError;
const IGNORED_GIT_COMMANDS = ['cat-file', 'config', 'diff', 'for-each-ref', 'log', 'rev-parse', 'status'];
const DISABLE_COLOR_FLAGS = ['branch', 'diff', 'showBranch', 'status', 'ui'].reduce((acc, type) => {
  acc.unshift('-c', `color.${type}=false`);
  return acc;
}, []);
/**
 * Expand config path name per
 * https://git-scm.com/docs/git-config#git-config-pathname
 * this regex attempts to get the specified user's home directory
 * Ex: on Mac ~kuychaco/ is expanded to the specified user’s home directory (/Users/kuychaco)
 * Regex translation:
 * ^~ line starts with tilde
 * ([^\\\\/]*)[\\\\/] captures non-slash characters before first slash
 */

const EXPAND_TILDE_REGEX = new RegExp('^~([^\\\\/]*)[\\\\/]');

class GitShellOutStrategy {
  constructor(workingDir, options = {}) {
    this.workingDir = workingDir;

    if (options.queue) {
      this.commandQueue = options.queue;
    } else {
      const parallelism = options.parallelism || Math.max(3, _os.default.cpus().length);
      this.commandQueue = new _asyncQueue.default({
        parallelism
      });
    }

    this.prompt = options.prompt || (query => Promise.reject());

    this.workerManager = options.workerManager;

    if (headless === null) {
      headless = !_electron.remote.getCurrentWindow().isVisible();
    }
  }
  /*
   * Provide an asynchronous callback to be used to request input from the user for git operations.
   *
   * `prompt` must be a callable that accepts a query object `{prompt, includeUsername}` and returns a Promise
   * that either resolves with a result object `{[username], password}` or rejects on cancellation.
   */


  setPromptCallback(prompt) {
    this.prompt = prompt;
  } // Execute a command and read the output using the embedded Git environment


  async exec(args, options = GitShellOutStrategy.defaultExecArgs) {
    /* eslint-disable no-console,no-control-regex */
    const {
      stdin,
      useGitPromptServer,
      useGpgWrapper,
      useGpgAtomPrompt,
      writeOperation
    } = options;
    const commandName = args[0];
    const subscriptions = new _eventKit.CompositeDisposable();
    const diagnosticsEnabled = process.env.ATOM_GITHUB_GIT_DIAGNOSTICS || atom.config.get('github.gitDiagnostics');
    const formattedArgs = `git ${args.join(' ')} in ${this.workingDir}`;

    const timingMarker = _gitTimingsView.default.generateMarker(`git ${args.join(' ')}`);

    timingMarker.mark('queued');
    args.unshift(...DISABLE_COLOR_FLAGS);

    if (execPathPromise === null) {
      // Attempt to collect the --exec-path from a native git installation.
      execPathPromise = new Promise(resolve => {
        _child_process.default.exec('git --exec-path', (error, stdout) => {
          /* istanbul ignore if */
          if (error) {
            // Oh well
            resolve(null);
            return;
          }

          resolve(stdout.trim());
        });
      });
    }

    const execPath = await execPathPromise;
    return this.commandQueue.push(async () => {
      timingMarker.mark('prepare');
      let gitPromptServer;
      const pathParts = [];

      if (process.env.PATH) {
        pathParts.push(process.env.PATH);
      }

      if (execPath) {
        pathParts.push(execPath);
      }

      const env = _objectSpread2({}, process.env, {
        GIT_TERMINAL_PROMPT: '0',
        GIT_OPTIONAL_LOCKS: '0',
        PATH: pathParts.join(_path.default.delimiter)
      });

      const gitTempDir = new _gitTempDir.default();

      if (useGpgWrapper) {
        await gitTempDir.ensure();
        args.unshift('-c', `gpg.program=${gitTempDir.getGpgWrapperSh()}`);
      }

      if (useGitPromptServer) {
        gitPromptServer = new _gitPromptServer.default(gitTempDir);
        await gitPromptServer.start(this.prompt);
        env.ATOM_GITHUB_TMP = gitTempDir.getRootPath();
        env.ATOM_GITHUB_ASKPASS_PATH = (0, _helpers.normalizeGitHelperPath)(gitTempDir.getAskPassJs());
        env.ATOM_GITHUB_CREDENTIAL_PATH = (0, _helpers.normalizeGitHelperPath)(gitTempDir.getCredentialHelperJs());
        env.ATOM_GITHUB_ELECTRON_PATH = (0, _helpers.normalizeGitHelperPath)((0, _helpers.getAtomHelperPath)());
        env.ATOM_GITHUB_SOCK_ADDR = gitPromptServer.getAddress();
        env.ATOM_GITHUB_WORKDIR_PATH = this.workingDir;
        env.ATOM_GITHUB_DUGITE_PATH = (0, _helpers.getDugitePath)();
        env.ATOM_GITHUB_KEYTAR_STRATEGY_PATH = (0, _helpers.getSharedModulePath)('keytar-strategy'); // "ssh" won't respect SSH_ASKPASS unless:
        // (a) it's running without a tty
        // (b) DISPLAY is set to something nonempty
        // But, on a Mac, DISPLAY is unset. Ensure that it is so our SSH_ASKPASS is respected.

        if (!process.env.DISPLAY || process.env.DISPLAY.length === 0) {
          env.DISPLAY = 'atom-github-placeholder';
        }

        env.ATOM_GITHUB_ORIGINAL_PATH = process.env.PATH || '';
        env.ATOM_GITHUB_ORIGINAL_GIT_ASKPASS = process.env.GIT_ASKPASS || '';
        env.ATOM_GITHUB_ORIGINAL_SSH_ASKPASS = process.env.SSH_ASKPASS || '';
        env.ATOM_GITHUB_ORIGINAL_GIT_SSH_COMMAND = process.env.GIT_SSH_COMMAND || '';
        env.ATOM_GITHUB_SPEC_MODE = atom.inSpecMode() ? 'true' : 'false';
        env.SSH_ASKPASS = (0, _helpers.normalizeGitHelperPath)(gitTempDir.getAskPassSh());
        env.GIT_ASKPASS = (0, _helpers.normalizeGitHelperPath)(gitTempDir.getAskPassSh());

        if (process.platform === 'linux') {
          env.GIT_SSH_COMMAND = gitTempDir.getSshWrapperSh();
        } else if (process.env.GIT_SSH_COMMAND) {
          env.GIT_SSH_COMMAND = process.env.GIT_SSH_COMMAND;
        } else {
          env.GIT_SSH = process.env.GIT_SSH;
        }

        const credentialHelperSh = (0, _helpers.normalizeGitHelperPath)(gitTempDir.getCredentialHelperSh());
        args.unshift('-c', `credential.helper=${credentialHelperSh}`);
      }

      if (useGpgWrapper && useGitPromptServer && useGpgAtomPrompt) {
        env.ATOM_GITHUB_GPG_PROMPT = 'true';
      }
      /* istanbul ignore if */


      if (diagnosticsEnabled) {
        env.GIT_TRACE = 'true';
        env.GIT_TRACE_CURL = 'true';
      }

      let opts = {
        env
      };

      if (stdin) {
        opts.stdin = stdin;
        opts.stdinEncoding = 'utf8';
      }
      /* istanbul ignore if */


      if (process.env.PRINT_GIT_TIMES) {
        console.time(`git:${formattedArgs}`);
      }

      return new Promise(async (resolve, reject) => {
        if (options.beforeRun) {
          const newArgsOpts = await options.beforeRun({
            args,
            opts
          });
          args = newArgsOpts.args;
          opts = newArgsOpts.opts;
        }

        const {
          promise,
          cancel
        } = this.executeGitCommand(args, opts, timingMarker);
        let expectCancel = false;

        if (gitPromptServer) {
          subscriptions.add(gitPromptServer.onDidCancel(async ({
            handlerPid
          }) => {
            expectCancel = true;
            await cancel(); // On Windows, the SSH_ASKPASS handler is executed as a non-child process, so the bin\git-askpass-atom.sh
            // process does not terminate when the git process is killed.
            // Kill the handler process *after* the git process has been killed to ensure that git doesn't have a
            // chance to fall back to GIT_ASKPASS from the credential handler.

            await new Promise((resolveKill, rejectKill) => {
              require('tree-kill')(handlerPid, 'SIGTERM', err => {
                /* istanbul ignore if */
                if (err) {
                  rejectKill(err);
                } else {
                  resolveKill();
                }
              });
            });
          }));
        }

        const {
          stdout,
          stderr,
          exitCode,
          signal,
          timing
        } = await promise.catch(err => {
          if (err.signal) {
            return {
              signal: err.signal
            };
          }

          reject(err);
          return {};
        });

        if (timing) {
          const {
            execTime,
            spawnTime,
            ipcTime
          } = timing;
          const now = performance.now();
          timingMarker.mark('nexttick', now - execTime - spawnTime - ipcTime);
          timingMarker.mark('execute', now - execTime - ipcTime);
          timingMarker.mark('ipc', now - ipcTime);
        }

        timingMarker.finalize();
        /* istanbul ignore if */

        if (process.env.PRINT_GIT_TIMES) {
          console.timeEnd(`git:${formattedArgs}`);
        }

        if (gitPromptServer) {
          gitPromptServer.terminate();
        }

        subscriptions.dispose();
        /* istanbul ignore if */

        if (diagnosticsEnabled) {
          const exposeControlCharacters = raw => {
            if (!raw) {
              return '';
            }

            return raw.replace(/\u0000/ug, '<NUL>\n').replace(/\u001F/ug, '<SEP>');
          };

          if (headless) {
            let summary = `git:${formattedArgs}\n`;

            if (exitCode !== undefined) {
              summary += `exit status: ${exitCode}\n`;
            } else if (signal) {
              summary += `exit signal: ${signal}\n`;
            }

            if (stdin && stdin.length !== 0) {
              summary += `stdin:\n${exposeControlCharacters(stdin)}\n`;
            }

            summary += 'stdout:';

            if (stdout.length === 0) {
              summary += ' <empty>\n';
            } else {
              summary += `\n${exposeControlCharacters(stdout)}\n`;
            }

            summary += 'stderr:';

            if (stderr.length === 0) {
              summary += ' <empty>\n';
            } else {
              summary += `\n${exposeControlCharacters(stderr)}\n`;
            }

            console.log(summary);
          } else {
            const headerStyle = 'font-weight: bold; color: blue;';
            console.groupCollapsed(`git:${formattedArgs}`);

            if (exitCode !== undefined) {
              console.log('%cexit status%c %d', headerStyle, 'font-weight: normal; color: black;', exitCode);
            } else if (signal) {
              console.log('%cexit signal%c %s', headerStyle, 'font-weight: normal; color: black;', signal);
            }

            console.log('%cfull arguments%c %s', headerStyle, 'font-weight: normal; color: black;', _util.default.inspect(args, {
              breakLength: Infinity
            }));

            if (stdin && stdin.length !== 0) {
              console.log('%cstdin', headerStyle);
              console.log(exposeControlCharacters(stdin));
            }

            console.log('%cstdout', headerStyle);
            console.log(exposeControlCharacters(stdout));
            console.log('%cstderr', headerStyle);
            console.log(exposeControlCharacters(stderr));
            console.groupEnd();
          }
        }

        if (exitCode !== 0 && !expectCancel) {
          const err = new GitError(`${formattedArgs} exited with code ${exitCode}\nstdout: ${stdout}\nstderr: ${stderr}`);
          err.code = exitCode;
          err.stdErr = stderr;
          err.stdOut = stdout;
          err.command = formattedArgs;
          reject(err);
        }

        if (!IGNORED_GIT_COMMANDS.includes(commandName)) {
          (0, _reporterProxy.incrementCounter)(commandName);
        }

        resolve(stdout);
      });
    }, {
      parallel: !writeOperation
    });
    /* eslint-enable no-console,no-control-regex */
  }

  async gpgExec(args, options) {
    try {
      return await this.exec(args.slice(), _objectSpread2({
        useGpgWrapper: true,
        useGpgAtomPrompt: false
      }, options));
    } catch (e) {
      if (/gpg failed/.test(e.stdErr)) {
        return await this.exec(args, _objectSpread2({
          useGitPromptServer: true,
          useGpgWrapper: true,
          useGpgAtomPrompt: true
        }, options));
      } else {
        throw e;
      }
    }
  }

  executeGitCommand(args, options, marker = null) {
    if (process.env.ATOM_GITHUB_INLINE_GIT_EXEC || !_workerManager.default.getInstance().isReady()) {
      marker && marker.mark('nexttick');
      let childPid;

      options.processCallback = child => {
        childPid = child.pid;
        /* istanbul ignore next */

        child.stdin.on('error', err => {
          throw new Error(`Error writing to stdin: git ${args.join(' ')} in ${this.workingDir}\n${options.stdin}\n${err}`);
        });
      };

      const promise = _dugite.GitProcess.exec(args, this.workingDir, options);

      marker && marker.mark('execute');
      return {
        promise,
        cancel: () => {
          /* istanbul ignore if */
          if (!childPid) {
            return Promise.resolve();
          }

          return new Promise((resolve, reject) => {
            require('tree-kill')(childPid, 'SIGTERM', err => {
              /* istanbul ignore if */
              if (err) {
                reject(err);
              } else {
                resolve();
              }
            });
          });
        }
      };
    } else {
      const workerManager = this.workerManager || _workerManager.default.getInstance();

      return workerManager.request({
        args,
        workingDir: this.workingDir,
        options
      });
    }
  }

  async resolveDotGitDir() {
    try {
      await _fsExtra.default.stat(this.workingDir); // fails if folder doesn't exist

      const output = await this.exec(['rev-parse', '--resolve-git-dir', _path.default.join(this.workingDir, '.git')]);
      const dotGitDir = output.trim();
      return (0, _helpers.toNativePathSep)(dotGitDir);
    } catch (e) {
      return null;
    }
  }

  init() {
    return this.exec(['init', this.workingDir]);
  }
  /**
   * Staging/Unstaging files and patches and committing
   */


  stageFiles(paths) {
    if (paths.length === 0) {
      return Promise.resolve(null);
    }

    const args = ['add'].concat(paths.map(_helpers.toGitPathSep));
    return this.exec(args, {
      writeOperation: true
    });
  }

  async fetchCommitMessageTemplate() {
    let templatePath = await this.getConfig('commit.template');

    if (!templatePath) {
      return null;
    }

    const homeDir = _os.default.homedir();

    templatePath = templatePath.trim().replace(EXPAND_TILDE_REGEX, (_, user) => {
      // if no user is specified, fall back to using the home directory.
      return `${user ? _path.default.join(_path.default.dirname(homeDir), user) : homeDir}/`;
    });
    templatePath = (0, _helpers.toNativePathSep)(templatePath);

    if (!_path.default.isAbsolute(templatePath)) {
      templatePath = _path.default.join(this.workingDir, templatePath);
    }

    if (!(await (0, _helpers.fileExists)(templatePath))) {
      throw new Error(`Invalid commit template path set in Git config: ${templatePath}`);
    }

    return await _fsExtra.default.readFile(templatePath, {
      encoding: 'utf8'
    });
  }

  unstageFiles(paths, commit = 'HEAD') {
    if (paths.length === 0) {
      return Promise.resolve(null);
    }

    const args = ['reset', commit, '--'].concat(paths.map(_helpers.toGitPathSep));
    return this.exec(args, {
      writeOperation: true
    });
  }

  stageFileModeChange(filename, newMode) {
    const indexReadPromise = this.exec(['ls-files', '-s', '--', filename]);
    return this.exec(['update-index', '--cacheinfo', `${newMode},<OID_TBD>,${filename}`], {
      writeOperation: true,
      beforeRun: async function determineArgs({
        args,
        opts
      }) {
        const index = await indexReadPromise;
        const oid = index.substr(7, 40);
        return {
          opts,
          args: ['update-index', '--cacheinfo', `${newMode},${oid},${filename}`]
        };
      }
    });
  }

  stageFileSymlinkChange(filename) {
    return this.exec(['rm', '--cached', filename], {
      writeOperation: true
    });
  }

  applyPatch(patch, {
    index
  } = {}) {
    const args = ['apply', '-'];

    if (index) {
      args.splice(1, 0, '--cached');
    }

    return this.exec(args, {
      stdin: patch,
      writeOperation: true
    });
  }

  async commit(rawMessage, {
    allowEmpty,
    amend,
    coAuthors,
    verbatim
  } = {}) {
    const args = ['commit'];
    let msg; // if amending and no new message is passed, use last commit's message. Ensure that we don't
    // mangle it in the process.

    if (amend && rawMessage.length === 0) {
      const {
        unbornRef,
        messageBody,
        messageSubject
      } = await this.getHeadCommit();

      if (unbornRef) {
        msg = rawMessage;
      } else {
        msg = `${messageSubject}\n\n${messageBody}`.trim();
        verbatim = true;
      }
    } else {
      msg = rawMessage;
    } // if commit template is used, strip commented lines from commit
    // to be consistent with command line git.


    const template = await this.fetchCommitMessageTemplate();

    if (template) {
      // respecting the comment character from user settings or fall back to # as default.
      // https://git-scm.com/docs/git-config#git-config-corecommentChar
      let commentChar = await this.getConfig('core.commentChar');

      if (!commentChar) {
        commentChar = '#';
      }

      msg = msg.split('\n').filter(line => !line.startsWith(commentChar)).join('\n');
    } // Determine the cleanup mode.


    if (verbatim) {
      args.push('--cleanup=verbatim');
    } else {
      const configured = await this.getConfig('commit.cleanup');
      const mode = configured && configured !== 'default' ? configured : 'strip';
      args.push(`--cleanup=${mode}`);
    } // add co-author commit trailers if necessary


    if (coAuthors && coAuthors.length > 0) {
      msg = await this.addCoAuthorsToMessage(msg, coAuthors);
    }

    args.push('-m', msg.trim());

    if (amend) {
      args.push('--amend');
    }

    if (allowEmpty) {
      args.push('--allow-empty');
    }

    return this.gpgExec(args, {
      writeOperation: true
    });
  }

  addCoAuthorsToMessage(message, coAuthors = []) {
    const trailers = coAuthors.map(author => {
      return {
        token: 'Co-Authored-By',
        value: `${author.name} <${author.email}>`
      };
    }); // Ensure that message ends with newline for git-interpret trailers to work

    const msg = `${message.trim()}\n`;
    return trailers.length ? this.mergeTrailers(msg, trailers) : msg;
  }
  /**
   * File Status and Diffs
   */


  async getStatusBundle() {
    const args = ['status', '--porcelain=v2', '--branch', '--untracked-files=all', '--ignore-submodules=dirty', '-z'];
    const output = await this.exec(args);

    if (output.length > MAX_STATUS_OUTPUT_LENGTH) {
      throw new LargeRepoError();
    }

    const results = await (0, _whatTheStatus.parse)(output);

    for (const entryType in results) {
      if (Array.isArray(results[entryType])) {
        this.updateNativePathSepForEntries(results[entryType]);
      }
    }

    return results;
  }

  updateNativePathSepForEntries(entries) {
    entries.forEach(entry => {
      // Normally we would avoid mutating responses from other package's APIs, but we control
      // the `what-the-status` module and know there are no side effects.
      // This is a hot code path and by mutating we avoid creating new objects that will just be GC'ed
      if (entry.filePath) {
        entry.filePath = (0, _helpers.toNativePathSep)(entry.filePath);
      }

      if (entry.origFilePath) {
        entry.origFilePath = (0, _helpers.toNativePathSep)(entry.origFilePath);
      }
    });
  }

  async diffFileStatus(options = {}) {
    const args = ['diff', '--name-status', '--no-renames'];

    if (options.staged) {
      args.push('--staged');
    }

    if (options.target) {
      args.push(options.target);
    }

    const output = await this.exec(args);
    const statusMap = {
      A: 'added',
      M: 'modified',
      D: 'deleted',
      U: 'unmerged'
    };
    const fileStatuses = {};
    output && output.trim().split(_helpers.LINE_ENDING_REGEX).forEach(line => {
      const [status, rawFilePath] = line.split('\t');
      const filePath = (0, _helpers.toNativePathSep)(rawFilePath);
      fileStatuses[filePath] = statusMap[status];
    });

    if (!options.staged) {
      const untracked = await this.getUntrackedFiles();
      untracked.forEach(filePath => {
        fileStatuses[filePath] = 'added';
      });
    }

    return fileStatuses;
  }

  async getUntrackedFiles() {
    const output = await this.exec(['ls-files', '--others', '--exclude-standard']);

    if (output.trim() === '') {
      return [];
    }

    return output.trim().split(_helpers.LINE_ENDING_REGEX).map(_helpers.toNativePathSep);
  }

  async getDiffsForFilePath(filePath, {
    staged,
    baseCommit
  } = {}) {
    let args = ['diff', '--no-prefix', '--no-ext-diff', '--no-renames', '--diff-filter=u'];

    if (staged) {
      args.push('--staged');
    }

    if (baseCommit) {
      args.push(baseCommit);
    }

    args = args.concat(['--', (0, _helpers.toGitPathSep)(filePath)]);
    const output = await this.exec(args);
    let rawDiffs = [];

    if (output) {
      rawDiffs = (0, _whatTheDiff.parse)(output).filter(rawDiff => rawDiff.status !== 'unmerged');

      for (let i = 0; i < rawDiffs.length; i++) {
        const rawDiff = rawDiffs[i];

        if (rawDiff.oldPath) {
          rawDiff.oldPath = (0, _helpers.toNativePathSep)(rawDiff.oldPath);
        }

        if (rawDiff.newPath) {
          rawDiff.newPath = (0, _helpers.toNativePathSep)(rawDiff.newPath);
        }
      }
    }

    if (!staged && (await this.getUntrackedFiles()).includes(filePath)) {
      // add untracked file
      const absPath = _path.default.join(this.workingDir, filePath);

      const executable = await (0, _helpers.isFileExecutable)(absPath);
      const symlink = await (0, _helpers.isFileSymlink)(absPath);
      const contents = await _fsExtra.default.readFile(absPath, {
        encoding: 'utf8'
      });
      const binary = (0, _helpers.isBinary)(contents);
      let mode;
      let realpath;

      if (executable) {
        mode = _file.default.modes.EXECUTABLE;
      } else if (symlink) {
        mode = _file.default.modes.SYMLINK;
        realpath = await _fsExtra.default.realpath(absPath);
      } else {
        mode = _file.default.modes.NORMAL;
      }

      rawDiffs.push(buildAddedFilePatch(filePath, binary ? null : contents, mode, realpath));
    }

    if (rawDiffs.length > 2) {
      throw new Error(`Expected between 0 and 2 diffs for ${filePath} but got ${rawDiffs.length}`);
    }

    return rawDiffs;
  }

  async getStagedChangesPatch() {
    const output = await this.exec(['diff', '--staged', '--no-prefix', '--no-ext-diff', '--no-renames', '--diff-filter=u']);

    if (!output) {
      return [];
    }

    const diffs = (0, _whatTheDiff.parse)(output);

    for (const diff of diffs) {
      if (diff.oldPath) {
        diff.oldPath = (0, _helpers.toNativePathSep)(diff.oldPath);
      }

      if (diff.newPath) {
        diff.newPath = (0, _helpers.toNativePathSep)(diff.newPath);
      }
    }

    return diffs;
  }
  /**
   * Miscellaneous getters
   */


  async getCommit(ref) {
    const [commit] = await this.getCommits({
      max: 1,
      ref,
      includeUnborn: true
    });
    return commit;
  }

  async getHeadCommit() {
    const [headCommit] = await this.getCommits({
      max: 1,
      ref: 'HEAD',
      includeUnborn: true
    });
    return headCommit;
  }

  async getCommits(options = {}) {
    const {
      max,
      ref,
      includeUnborn,
      includePatch
    } = _objectSpread2({
      max: 1,
      ref: 'HEAD',
      includeUnborn: false,
      includePatch: false
    }, options); // https://git-scm.com/docs/git-log#_pretty_formats
    // %x00 - null byte
    // %H - commit SHA
    // %ae - author email
    // %an = author full name
    // %at - timestamp, UNIX timestamp
    // %s - subject
    // %b - body


    const args = ['log', '--pretty=format:%H%x00%ae%x00%an%x00%at%x00%s%x00%b%x00', '--no-abbrev-commit', '--no-prefix', '--no-ext-diff', '--no-renames', '-z', '-n', max, ref];

    if (includePatch) {
      args.push('--patch', '-m', '--first-parent');
    }

    const output = await this.exec(args.concat('--')).catch(err => {
      if (/unknown revision/.test(err.stdErr) || /bad revision 'HEAD'/.test(err.stdErr)) {
        return '';
      } else {
        throw err;
      }
    });

    if (output === '') {
      return includeUnborn ? [{
        sha: '',
        message: '',
        unbornRef: true
      }] : [];
    }

    const fields = output.trim().split('\0');
    const commits = [];

    for (let i = 0; i < fields.length; i += 7) {
      const body = fields[i + 5].trim();
      let patch = [];

      if (includePatch) {
        const diffs = fields[i + 6];
        patch = (0, _whatTheDiff.parse)(diffs.trim());
      }

      const {
        message: messageBody,
        coAuthors
      } = (0, _helpers.extractCoAuthorsAndRawCommitMessage)(body);
      commits.push({
        sha: fields[i] && fields[i].trim(),
        author: new _author.default(fields[i + 1] && fields[i + 1].trim(), fields[i + 2] && fields[i + 2].trim()),
        authorDate: parseInt(fields[i + 3], 10),
        messageSubject: fields[i + 4],
        messageBody,
        coAuthors,
        unbornRef: false,
        patch
      });
    }

    return commits;
  }

  async getAuthors(options = {}) {
    const {
      max,
      ref
    } = _objectSpread2({
      max: 1,
      ref: 'HEAD'
    }, options); // https://git-scm.com/docs/git-log#_pretty_formats
    // %x1F - field separator byte
    // %an - author name
    // %ae - author email
    // %cn - committer name
    // %ce - committer email
    // %(trailers:unfold,only) - the commit message trailers, separated
    //                           by newlines and unfolded (i.e. properly
    //                           formatted and one trailer per line).


    const delimiter = '1F';
    const delimiterString = String.fromCharCode(parseInt(delimiter, 16));
    const fields = ['%an', '%ae', '%cn', '%ce', '%(trailers:unfold,only)'];
    const format = fields.join(`%x${delimiter}`);

    try {
      const output = await this.exec(['log', `--format=${format}`, '-z', '-n', max, ref, '--']);
      return output.split('\0').reduce((acc, line) => {
        if (line.length === 0) {
          return acc;
        }

        const [an, ae, cn, ce, trailers] = line.split(delimiterString);
        trailers.split('\n').map(trailer => trailer.match(_helpers.CO_AUTHOR_REGEX)).filter(match => match !== null).forEach(([_, name, email]) => {
          acc[email] = name;
        });
        acc[ae] = an;
        acc[ce] = cn;
        return acc;
      }, {});
    } catch (err) {
      if (/unknown revision/.test(err.stdErr) || /bad revision 'HEAD'/.test(err.stdErr)) {
        return [];
      } else {
        throw err;
      }
    }
  }

  mergeTrailers(commitMessage, trailers) {
    const args = ['interpret-trailers'];

    for (const trailer of trailers) {
      args.push('--trailer', `${trailer.token}=${trailer.value}`);
    }

    return this.exec(args, {
      stdin: commitMessage
    });
  }

  readFileFromIndex(filePath) {
    return this.exec(['show', `:${(0, _helpers.toGitPathSep)(filePath)}`]);
  }
  /**
   * Merge
   */


  merge(branchName) {
    return this.gpgExec(['merge', branchName], {
      writeOperation: true
    });
  }

  isMerging(dotGitDir) {
    return (0, _helpers.fileExists)(_path.default.join(dotGitDir, 'MERGE_HEAD')).catch(() => false);
  }

  abortMerge() {
    return this.exec(['merge', '--abort'], {
      writeOperation: true
    });
  }

  checkoutSide(side, paths) {
    if (paths.length === 0) {
      return Promise.resolve();
    }

    return this.exec(['checkout', `--${side}`, ...paths.map(_helpers.toGitPathSep)]);
  }
  /**
   * Rebase
   */


  async isRebasing(dotGitDir) {
    const results = await Promise.all([(0, _helpers.fileExists)(_path.default.join(dotGitDir, 'rebase-merge')), (0, _helpers.fileExists)(_path.default.join(dotGitDir, 'rebase-apply'))]);
    return results.some(r => r);
  }
  /**
   * Remote interactions
   */


  clone(remoteUrl, options = {}) {
    const args = ['clone'];

    if (options.noLocal) {
      args.push('--no-local');
    }

    if (options.bare) {
      args.push('--bare');
    }

    if (options.recursive) {
      args.push('--recursive');
    }

    if (options.sourceRemoteName) {
      args.push('--origin', options.remoteName);
    }

    args.push(remoteUrl, this.workingDir);
    return this.exec(args, {
      useGitPromptServer: true,
      writeOperation: true
    });
  }

  fetch(remoteName, branchName) {
    return this.exec(['fetch', remoteName, branchName], {
      useGitPromptServer: true,
      writeOperation: true
    });
  }

  pull(remoteName, branchName, options = {}) {
    const args = ['pull', remoteName, options.refSpec || branchName];

    if (options.ffOnly) {
      args.push('--ff-only');
    }

    return this.gpgExec(args, {
      useGitPromptServer: true,
      writeOperation: true
    });
  }

  push(remoteName, branchName, options = {}) {
    const args = ['push', remoteName || 'origin', options.refSpec || `refs/heads/${branchName}`];

    if (options.setUpstream) {
      args.push('--set-upstream');
    }

    if (options.force) {
      args.push('--force');
    }

    return this.exec(args, {
      useGitPromptServer: true,
      writeOperation: true
    });
  }
  /**
   * Undo Operations
   */


  reset(type, revision = 'HEAD') {
    const validTypes = ['soft'];

    if (!validTypes.includes(type)) {
      throw new Error(`Invalid type ${type}. Must be one of: ${validTypes.join(', ')}`);
    }

    return this.exec(['reset', `--${type}`, revision]);
  }

  deleteRef(ref) {
    return this.exec(['update-ref', '-d', ref]);
  }
  /**
   * Branches
   */


  checkout(branchName, options = {}) {
    const args = ['checkout'];

    if (options.createNew) {
      args.push('-b');
    }

    args.push(branchName);

    if (options.startPoint) {
      if (options.track) {
        args.push('--track');
      }

      args.push(options.startPoint);
    }

    return this.exec(args, {
      writeOperation: true
    });
  }

  async getBranches() {
    const format = ['%(objectname)', '%(HEAD)', '%(refname:short)', '%(upstream)', '%(upstream:remotename)', '%(upstream:remoteref)', '%(push)', '%(push:remotename)', '%(push:remoteref)'].join('%00');
    const output = await this.exec(['for-each-ref', `--format=${format}`, 'refs/heads/**']);
    return output.trim().split(_helpers.LINE_ENDING_REGEX).map(line => {
      const [sha, head, name, upstreamTrackingRef, upstreamRemoteName, upstreamRemoteRef, pushTrackingRef, pushRemoteName, pushRemoteRef] = line.split('\0');
      const branch = {
        name,
        sha,
        head: head === '*'
      };

      if (upstreamTrackingRef || upstreamRemoteName || upstreamRemoteRef) {
        branch.upstream = {
          trackingRef: upstreamTrackingRef,
          remoteName: upstreamRemoteName,
          remoteRef: upstreamRemoteRef
        };
      }

      if (branch.upstream || pushTrackingRef || pushRemoteName || pushRemoteRef) {
        branch.push = {
          trackingRef: pushTrackingRef,
          remoteName: pushRemoteName || branch.upstream && branch.upstream.remoteName,
          remoteRef: pushRemoteRef || branch.upstream && branch.upstream.remoteRef
        };
      }

      return branch;
    });
  }

  async getBranchesWithCommit(sha, option = {}) {
    const args = ['branch', '--format=%(refname)', '--contains', sha];

    if (option.showLocal && option.showRemote) {
      args.splice(1, 0, '--all');
    } else if (option.showRemote) {
      args.splice(1, 0, '--remotes');
    }

    if (option.pattern) {
      args.push(option.pattern);
    }

    return (await this.exec(args)).trim().split(_helpers.LINE_ENDING_REGEX);
  }

  checkoutFiles(paths, revision) {
    if (paths.length === 0) {
      return null;
    }

    const args = ['checkout'];

    if (revision) {
      args.push(revision);
    }

    return this.exec(args.concat('--', paths.map(_helpers.toGitPathSep)), {
      writeOperation: true
    });
  }

  async describeHead() {
    return (await this.exec(['describe', '--contains', '--all', '--always', 'HEAD'])).trim();
  }

  async getConfig(option, {
    local
  } = {}) {
    let output;

    try {
      let args = ['config'];

      if (local) {
        args.push('--local');
      }

      args = args.concat(option);
      output = await this.exec(args);
    } catch (err) {
      if (err.code === 1 || err.code === 128) {
        // No matching config found OR --local can only be used inside a git repository
        return null;
      } else {
        throw err;
      }
    }

    return output.trim();
  }

  setConfig(option, value, {
    replaceAll,
    global
  } = {}) {
    let args = ['config'];

    if (replaceAll) {
      args.push('--replace-all');
    }

    if (global) {
      args.push('--global');
    }

    args = args.concat(option, value);
    return this.exec(args, {
      writeOperation: true
    });
  }

  unsetConfig(option) {
    return this.exec(['config', '--unset', option], {
      writeOperation: true
    });
  }

  async getRemotes() {
    let output = await this.getConfig(['--get-regexp', '^remote\\..*\\.url$'], {
      local: true
    });

    if (output) {
      output = output.trim();

      if (!output.length) {
        return [];
      }

      return output.split('\n').map(line => {
        const match = line.match(/^remote\.(.*)\.url (.*)$/);
        return {
          name: match[1],
          url: match[2]
        };
      });
    } else {
      return [];
    }
  }

  addRemote(name, url) {
    return this.exec(['remote', 'add', name, url]);
  }

  async createBlob({
    filePath,
    stdin
  } = {}) {
    let output;

    if (filePath) {
      try {
        output = (await this.exec(['hash-object', '-w', filePath], {
          writeOperation: true
        })).trim();
      } catch (e) {
        if (e.stdErr && e.stdErr.match(/fatal: Cannot open .*: No such file or directory/)) {
          output = null;
        } else {
          throw e;
        }
      }
    } else if (stdin) {
      output = (await this.exec(['hash-object', '-w', '--stdin'], {
        stdin,
        writeOperation: true
      })).trim();
    } else {
      throw new Error('Must supply file path or stdin');
    }

    return output;
  }

  async expandBlobToFile(absFilePath, sha) {
    const output = await this.exec(['cat-file', '-p', sha]);
    await _fsExtra.default.writeFile(absFilePath, output, {
      encoding: 'utf8'
    });
    return absFilePath;
  }

  async getBlobContents(sha) {
    return await this.exec(['cat-file', '-p', sha]);
  }

  async mergeFile(oursPath, commonBasePath, theirsPath, resultPath) {
    const args = ['merge-file', '-p', oursPath, commonBasePath, theirsPath, '-L', 'current', '-L', 'after discard', '-L', 'before discard'];
    let output;
    let conflict = false;

    try {
      output = await this.exec(args);
    } catch (e) {
      if (e instanceof GitError && e.code === 1) {
        output = e.stdOut;
        conflict = true;
      } else {
        throw e;
      }
    } // Interpret a relative resultPath as relative to the repository working directory for consistency with the
    // other arguments.


    const resolvedResultPath = _path.default.resolve(this.workingDir, resultPath);

    await _fsExtra.default.writeFile(resolvedResultPath, output, {
      encoding: 'utf8'
    });
    return {
      filePath: oursPath,
      resultPath,
      conflict
    };
  }

  async writeMergeConflictToIndex(filePath, commonBaseSha, oursSha, theirsSha) {
    const gitFilePath = (0, _helpers.toGitPathSep)(filePath);
    const fileMode = await this.getFileMode(filePath);
    let indexInfo = `0 0000000000000000000000000000000000000000\t${gitFilePath}\n`;

    if (commonBaseSha) {
      indexInfo += `${fileMode} ${commonBaseSha} 1\t${gitFilePath}\n`;
    }

    if (oursSha) {
      indexInfo += `${fileMode} ${oursSha} 2\t${gitFilePath}\n`;
    }

    if (theirsSha) {
      indexInfo += `${fileMode} ${theirsSha} 3\t${gitFilePath}\n`;
    }

    return this.exec(['update-index', '--index-info'], {
      stdin: indexInfo,
      writeOperation: true
    });
  }

  async getFileMode(filePath) {
    const output = await this.exec(['ls-files', '--stage', '--', (0, _helpers.toGitPathSep)(filePath)]);

    if (output) {
      return output.slice(0, 6);
    } else {
      const executable = await (0, _helpers.isFileExecutable)(_path.default.join(this.workingDir, filePath));
      const symlink = await (0, _helpers.isFileSymlink)(_path.default.join(this.workingDir, filePath));

      if (symlink) {
        return _file.default.modes.SYMLINK;
      } else if (executable) {
        return _file.default.modes.EXECUTABLE;
      } else {
        return _file.default.modes.NORMAL;
      }
    }
  }

  destroy() {
    this.commandQueue.dispose();
  }

}

exports.default = GitShellOutStrategy;

_defineProperty(GitShellOutStrategy, "defaultExecArgs", {
  stdin: null,
  useGitPromptServer: false,
  useGpgWrapper: false,
  useGpgAtomPrompt: false,
  writeOperation: false
});

function buildAddedFilePatch(filePath, contents, mode, realpath) {
  const hunks = [];

  if (contents) {
    let noNewLine;
    let lines;

    if (mode === _file.default.modes.SYMLINK) {
      noNewLine = false;
      lines = [`+${(0, _helpers.toGitPathSep)(realpath)}`, '\\ No newline at end of file'];
    } else {
      noNewLine = contents[contents.length - 1] !== '\n';
      lines = contents.trim().split(_helpers.LINE_ENDING_REGEX).map(line => `+${line}`);
    }

    if (noNewLine) {
      lines.push('\\ No newline at end of file');
    }

    hunks.push({
      lines,
      oldStartLine: 0,
      oldLineCount: 0,
      newStartLine: 1,
      heading: '',
      newLineCount: noNewLine ? lines.length - 1 : lines.length
    });
  }

  return {
    oldPath: null,
    newPath: (0, _helpers.toNativePathSep)(filePath),
    oldMode: null,
    newMode: mode,
    status: 'added',
    hunks
  };
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbImdpdC1zaGVsbC1vdXQtc3RyYXRlZ3kuanMiXSwibmFtZXMiOlsiTUFYX1NUQVRVU19PVVRQVVRfTEVOR1RIIiwiaGVhZGxlc3MiLCJleGVjUGF0aFByb21pc2UiLCJHaXRFcnJvciIsIkVycm9yIiwiY29uc3RydWN0b3IiLCJtZXNzYWdlIiwic3RhY2siLCJMYXJnZVJlcG9FcnJvciIsIklHTk9SRURfR0lUX0NPTU1BTkRTIiwiRElTQUJMRV9DT0xPUl9GTEFHUyIsInJlZHVjZSIsImFjYyIsInR5cGUiLCJ1bnNoaWZ0IiwiRVhQQU5EX1RJTERFX1JFR0VYIiwiUmVnRXhwIiwiR2l0U2hlbGxPdXRTdHJhdGVneSIsIndvcmtpbmdEaXIiLCJvcHRpb25zIiwicXVldWUiLCJjb21tYW5kUXVldWUiLCJwYXJhbGxlbGlzbSIsIk1hdGgiLCJtYXgiLCJvcyIsImNwdXMiLCJsZW5ndGgiLCJBc3luY1F1ZXVlIiwicHJvbXB0IiwicXVlcnkiLCJQcm9taXNlIiwicmVqZWN0Iiwid29ya2VyTWFuYWdlciIsInJlbW90ZSIsImdldEN1cnJlbnRXaW5kb3ciLCJpc1Zpc2libGUiLCJzZXRQcm9tcHRDYWxsYmFjayIsImV4ZWMiLCJhcmdzIiwiZGVmYXVsdEV4ZWNBcmdzIiwic3RkaW4iLCJ1c2VHaXRQcm9tcHRTZXJ2ZXIiLCJ1c2VHcGdXcmFwcGVyIiwidXNlR3BnQXRvbVByb21wdCIsIndyaXRlT3BlcmF0aW9uIiwiY29tbWFuZE5hbWUiLCJzdWJzY3JpcHRpb25zIiwiQ29tcG9zaXRlRGlzcG9zYWJsZSIsImRpYWdub3N0aWNzRW5hYmxlZCIsInByb2Nlc3MiLCJlbnYiLCJBVE9NX0dJVEhVQl9HSVRfRElBR05PU1RJQ1MiLCJhdG9tIiwiY29uZmlnIiwiZ2V0IiwiZm9ybWF0dGVkQXJncyIsImpvaW4iLCJ0aW1pbmdNYXJrZXIiLCJHaXRUaW1pbmdzVmlldyIsImdlbmVyYXRlTWFya2VyIiwibWFyayIsInJlc29sdmUiLCJjaGlsZFByb2Nlc3MiLCJlcnJvciIsInN0ZG91dCIsInRyaW0iLCJleGVjUGF0aCIsInB1c2giLCJnaXRQcm9tcHRTZXJ2ZXIiLCJwYXRoUGFydHMiLCJQQVRIIiwiR0lUX1RFUk1JTkFMX1BST01QVCIsIkdJVF9PUFRJT05BTF9MT0NLUyIsInBhdGgiLCJkZWxpbWl0ZXIiLCJnaXRUZW1wRGlyIiwiR2l0VGVtcERpciIsImVuc3VyZSIsImdldEdwZ1dyYXBwZXJTaCIsIkdpdFByb21wdFNlcnZlciIsInN0YXJ0IiwiQVRPTV9HSVRIVUJfVE1QIiwiZ2V0Um9vdFBhdGgiLCJBVE9NX0dJVEhVQl9BU0tQQVNTX1BBVEgiLCJnZXRBc2tQYXNzSnMiLCJBVE9NX0dJVEhVQl9DUkVERU5USUFMX1BBVEgiLCJnZXRDcmVkZW50aWFsSGVscGVySnMiLCJBVE9NX0dJVEhVQl9FTEVDVFJPTl9QQVRIIiwiQVRPTV9HSVRIVUJfU09DS19BRERSIiwiZ2V0QWRkcmVzcyIsIkFUT01fR0lUSFVCX1dPUktESVJfUEFUSCIsIkFUT01fR0lUSFVCX0RVR0lURV9QQVRIIiwiQVRPTV9HSVRIVUJfS0VZVEFSX1NUUkFURUdZX1BBVEgiLCJESVNQTEFZIiwiQVRPTV9HSVRIVUJfT1JJR0lOQUxfUEFUSCIsIkFUT01fR0lUSFVCX09SSUdJTkFMX0dJVF9BU0tQQVNTIiwiR0lUX0FTS1BBU1MiLCJBVE9NX0dJVEhVQl9PUklHSU5BTF9TU0hfQVNLUEFTUyIsIlNTSF9BU0tQQVNTIiwiQVRPTV9HSVRIVUJfT1JJR0lOQUxfR0lUX1NTSF9DT01NQU5EIiwiR0lUX1NTSF9DT01NQU5EIiwiQVRPTV9HSVRIVUJfU1BFQ19NT0RFIiwiaW5TcGVjTW9kZSIsImdldEFza1Bhc3NTaCIsInBsYXRmb3JtIiwiZ2V0U3NoV3JhcHBlclNoIiwiR0lUX1NTSCIsImNyZWRlbnRpYWxIZWxwZXJTaCIsImdldENyZWRlbnRpYWxIZWxwZXJTaCIsIkFUT01fR0lUSFVCX0dQR19QUk9NUFQiLCJHSVRfVFJBQ0UiLCJHSVRfVFJBQ0VfQ1VSTCIsIm9wdHMiLCJzdGRpbkVuY29kaW5nIiwiUFJJTlRfR0lUX1RJTUVTIiwiY29uc29sZSIsInRpbWUiLCJiZWZvcmVSdW4iLCJuZXdBcmdzT3B0cyIsInByb21pc2UiLCJjYW5jZWwiLCJleGVjdXRlR2l0Q29tbWFuZCIsImV4cGVjdENhbmNlbCIsImFkZCIsIm9uRGlkQ2FuY2VsIiwiaGFuZGxlclBpZCIsInJlc29sdmVLaWxsIiwicmVqZWN0S2lsbCIsInJlcXVpcmUiLCJlcnIiLCJzdGRlcnIiLCJleGl0Q29kZSIsInNpZ25hbCIsInRpbWluZyIsImNhdGNoIiwiZXhlY1RpbWUiLCJzcGF3blRpbWUiLCJpcGNUaW1lIiwibm93IiwicGVyZm9ybWFuY2UiLCJmaW5hbGl6ZSIsInRpbWVFbmQiLCJ0ZXJtaW5hdGUiLCJkaXNwb3NlIiwiZXhwb3NlQ29udHJvbENoYXJhY3RlcnMiLCJyYXciLCJyZXBsYWNlIiwic3VtbWFyeSIsInVuZGVmaW5lZCIsImxvZyIsImhlYWRlclN0eWxlIiwiZ3JvdXBDb2xsYXBzZWQiLCJ1dGlsIiwiaW5zcGVjdCIsImJyZWFrTGVuZ3RoIiwiSW5maW5pdHkiLCJncm91cEVuZCIsImNvZGUiLCJzdGRFcnIiLCJzdGRPdXQiLCJjb21tYW5kIiwiaW5jbHVkZXMiLCJwYXJhbGxlbCIsImdwZ0V4ZWMiLCJzbGljZSIsImUiLCJ0ZXN0IiwibWFya2VyIiwiQVRPTV9HSVRIVUJfSU5MSU5FX0dJVF9FWEVDIiwiV29ya2VyTWFuYWdlciIsImdldEluc3RhbmNlIiwiaXNSZWFkeSIsImNoaWxkUGlkIiwicHJvY2Vzc0NhbGxiYWNrIiwiY2hpbGQiLCJwaWQiLCJvbiIsIkdpdFByb2Nlc3MiLCJyZXF1ZXN0IiwicmVzb2x2ZURvdEdpdERpciIsImZzIiwic3RhdCIsIm91dHB1dCIsImRvdEdpdERpciIsImluaXQiLCJzdGFnZUZpbGVzIiwicGF0aHMiLCJjb25jYXQiLCJtYXAiLCJ0b0dpdFBhdGhTZXAiLCJmZXRjaENvbW1pdE1lc3NhZ2VUZW1wbGF0ZSIsInRlbXBsYXRlUGF0aCIsImdldENvbmZpZyIsImhvbWVEaXIiLCJob21lZGlyIiwiXyIsInVzZXIiLCJkaXJuYW1lIiwiaXNBYnNvbHV0ZSIsInJlYWRGaWxlIiwiZW5jb2RpbmciLCJ1bnN0YWdlRmlsZXMiLCJjb21taXQiLCJzdGFnZUZpbGVNb2RlQ2hhbmdlIiwiZmlsZW5hbWUiLCJuZXdNb2RlIiwiaW5kZXhSZWFkUHJvbWlzZSIsImRldGVybWluZUFyZ3MiLCJpbmRleCIsIm9pZCIsInN1YnN0ciIsInN0YWdlRmlsZVN5bWxpbmtDaGFuZ2UiLCJhcHBseVBhdGNoIiwicGF0Y2giLCJzcGxpY2UiLCJyYXdNZXNzYWdlIiwiYWxsb3dFbXB0eSIsImFtZW5kIiwiY29BdXRob3JzIiwidmVyYmF0aW0iLCJtc2ciLCJ1bmJvcm5SZWYiLCJtZXNzYWdlQm9keSIsIm1lc3NhZ2VTdWJqZWN0IiwiZ2V0SGVhZENvbW1pdCIsInRlbXBsYXRlIiwiY29tbWVudENoYXIiLCJzcGxpdCIsImZpbHRlciIsImxpbmUiLCJzdGFydHNXaXRoIiwiY29uZmlndXJlZCIsIm1vZGUiLCJhZGRDb0F1dGhvcnNUb01lc3NhZ2UiLCJ0cmFpbGVycyIsImF1dGhvciIsInRva2VuIiwidmFsdWUiLCJuYW1lIiwiZW1haWwiLCJtZXJnZVRyYWlsZXJzIiwiZ2V0U3RhdHVzQnVuZGxlIiwicmVzdWx0cyIsImVudHJ5VHlwZSIsIkFycmF5IiwiaXNBcnJheSIsInVwZGF0ZU5hdGl2ZVBhdGhTZXBGb3JFbnRyaWVzIiwiZW50cmllcyIsImZvckVhY2giLCJlbnRyeSIsImZpbGVQYXRoIiwib3JpZ0ZpbGVQYXRoIiwiZGlmZkZpbGVTdGF0dXMiLCJzdGFnZWQiLCJ0YXJnZXQiLCJzdGF0dXNNYXAiLCJBIiwiTSIsIkQiLCJVIiwiZmlsZVN0YXR1c2VzIiwiTElORV9FTkRJTkdfUkVHRVgiLCJzdGF0dXMiLCJyYXdGaWxlUGF0aCIsInVudHJhY2tlZCIsImdldFVudHJhY2tlZEZpbGVzIiwidG9OYXRpdmVQYXRoU2VwIiwiZ2V0RGlmZnNGb3JGaWxlUGF0aCIsImJhc2VDb21taXQiLCJyYXdEaWZmcyIsInJhd0RpZmYiLCJpIiwib2xkUGF0aCIsIm5ld1BhdGgiLCJhYnNQYXRoIiwiZXhlY3V0YWJsZSIsInN5bWxpbmsiLCJjb250ZW50cyIsImJpbmFyeSIsInJlYWxwYXRoIiwiRmlsZSIsIm1vZGVzIiwiRVhFQ1VUQUJMRSIsIlNZTUxJTksiLCJOT1JNQUwiLCJidWlsZEFkZGVkRmlsZVBhdGNoIiwiZ2V0U3RhZ2VkQ2hhbmdlc1BhdGNoIiwiZGlmZnMiLCJkaWZmIiwiZ2V0Q29tbWl0IiwicmVmIiwiZ2V0Q29tbWl0cyIsImluY2x1ZGVVbmJvcm4iLCJoZWFkQ29tbWl0IiwiaW5jbHVkZVBhdGNoIiwic2hhIiwiZmllbGRzIiwiY29tbWl0cyIsImJvZHkiLCJBdXRob3IiLCJhdXRob3JEYXRlIiwicGFyc2VJbnQiLCJnZXRBdXRob3JzIiwiZGVsaW1pdGVyU3RyaW5nIiwiU3RyaW5nIiwiZnJvbUNoYXJDb2RlIiwiZm9ybWF0IiwiYW4iLCJhZSIsImNuIiwiY2UiLCJ0cmFpbGVyIiwibWF0Y2giLCJDT19BVVRIT1JfUkVHRVgiLCJjb21taXRNZXNzYWdlIiwicmVhZEZpbGVGcm9tSW5kZXgiLCJtZXJnZSIsImJyYW5jaE5hbWUiLCJpc01lcmdpbmciLCJhYm9ydE1lcmdlIiwiY2hlY2tvdXRTaWRlIiwic2lkZSIsImlzUmViYXNpbmciLCJhbGwiLCJzb21lIiwiciIsImNsb25lIiwicmVtb3RlVXJsIiwibm9Mb2NhbCIsImJhcmUiLCJyZWN1cnNpdmUiLCJzb3VyY2VSZW1vdGVOYW1lIiwicmVtb3RlTmFtZSIsImZldGNoIiwicHVsbCIsInJlZlNwZWMiLCJmZk9ubHkiLCJzZXRVcHN0cmVhbSIsImZvcmNlIiwicmVzZXQiLCJyZXZpc2lvbiIsInZhbGlkVHlwZXMiLCJkZWxldGVSZWYiLCJjaGVja291dCIsImNyZWF0ZU5ldyIsInN0YXJ0UG9pbnQiLCJ0cmFjayIsImdldEJyYW5jaGVzIiwiaGVhZCIsInVwc3RyZWFtVHJhY2tpbmdSZWYiLCJ1cHN0cmVhbVJlbW90ZU5hbWUiLCJ1cHN0cmVhbVJlbW90ZVJlZiIsInB1c2hUcmFja2luZ1JlZiIsInB1c2hSZW1vdGVOYW1lIiwicHVzaFJlbW90ZVJlZiIsImJyYW5jaCIsInVwc3RyZWFtIiwidHJhY2tpbmdSZWYiLCJyZW1vdGVSZWYiLCJnZXRCcmFuY2hlc1dpdGhDb21taXQiLCJvcHRpb24iLCJzaG93TG9jYWwiLCJzaG93UmVtb3RlIiwicGF0dGVybiIsImNoZWNrb3V0RmlsZXMiLCJkZXNjcmliZUhlYWQiLCJsb2NhbCIsInNldENvbmZpZyIsInJlcGxhY2VBbGwiLCJnbG9iYWwiLCJ1bnNldENvbmZpZyIsImdldFJlbW90ZXMiLCJ1cmwiLCJhZGRSZW1vdGUiLCJjcmVhdGVCbG9iIiwiZXhwYW5kQmxvYlRvRmlsZSIsImFic0ZpbGVQYXRoIiwid3JpdGVGaWxlIiwiZ2V0QmxvYkNvbnRlbnRzIiwibWVyZ2VGaWxlIiwib3Vyc1BhdGgiLCJjb21tb25CYXNlUGF0aCIsInRoZWlyc1BhdGgiLCJyZXN1bHRQYXRoIiwiY29uZmxpY3QiLCJyZXNvbHZlZFJlc3VsdFBhdGgiLCJ3cml0ZU1lcmdlQ29uZmxpY3RUb0luZGV4IiwiY29tbW9uQmFzZVNoYSIsIm91cnNTaGEiLCJ0aGVpcnNTaGEiLCJnaXRGaWxlUGF0aCIsImZpbGVNb2RlIiwiZ2V0RmlsZU1vZGUiLCJpbmRleEluZm8iLCJkZXN0cm95IiwiaHVua3MiLCJub05ld0xpbmUiLCJsaW5lcyIsIm9sZFN0YXJ0TGluZSIsIm9sZExpbmVDb3VudCIsIm5ld1N0YXJ0TGluZSIsImhlYWRpbmciLCJuZXdMaW5lQ291bnQiLCJvbGRNb2RlIl0sIm1hcHBpbmdzIjoiOzs7Ozs7O0FBQUE7O0FBQ0E7O0FBQ0E7O0FBQ0E7O0FBQ0E7O0FBQ0E7O0FBRUE7O0FBQ0E7O0FBQ0E7O0FBQ0E7O0FBRUE7O0FBQ0E7O0FBQ0E7O0FBQ0E7O0FBQ0E7O0FBS0E7O0FBQ0E7O0FBQ0E7O0FBQ0E7Ozs7Ozs7O0FBRUEsTUFBTUEsd0JBQXdCLEdBQUcsT0FBTyxJQUFQLEdBQWMsRUFBL0M7QUFFQSxJQUFJQyxRQUFRLEdBQUcsSUFBZjtBQUNBLElBQUlDLGVBQWUsR0FBRyxJQUF0Qjs7QUFFTyxNQUFNQyxRQUFOLFNBQXVCQyxLQUF2QixDQUE2QjtBQUNsQ0MsRUFBQUEsV0FBVyxDQUFDQyxPQUFELEVBQVU7QUFDbkIsVUFBTUEsT0FBTjtBQUNBLFNBQUtBLE9BQUwsR0FBZUEsT0FBZjtBQUNBLFNBQUtDLEtBQUwsR0FBYSxJQUFJSCxLQUFKLEdBQVlHLEtBQXpCO0FBQ0Q7O0FBTGlDOzs7O0FBUTdCLE1BQU1DLGNBQU4sU0FBNkJKLEtBQTdCLENBQW1DO0FBQ3hDQyxFQUFBQSxXQUFXLENBQUNDLE9BQUQsRUFBVTtBQUNuQixVQUFNQSxPQUFOO0FBQ0EsU0FBS0EsT0FBTCxHQUFlQSxPQUFmO0FBQ0EsU0FBS0MsS0FBTCxHQUFhLElBQUlILEtBQUosR0FBWUcsS0FBekI7QUFDRDs7QUFMdUMsQyxDQVExQzs7OztBQUNBLE1BQU1FLG9CQUFvQixHQUFHLENBQUMsVUFBRCxFQUFhLFFBQWIsRUFBdUIsTUFBdkIsRUFBK0IsY0FBL0IsRUFBK0MsS0FBL0MsRUFBc0QsV0FBdEQsRUFBbUUsUUFBbkUsQ0FBN0I7QUFFQSxNQUFNQyxtQkFBbUIsR0FBRyxDQUMxQixRQUQwQixFQUNoQixNQURnQixFQUNSLFlBRFEsRUFDTSxRQUROLEVBQ2dCLElBRGhCLEVBRTFCQyxNQUYwQixDQUVuQixDQUFDQyxHQUFELEVBQU1DLElBQU4sS0FBZTtBQUN0QkQsRUFBQUEsR0FBRyxDQUFDRSxPQUFKLENBQVksSUFBWixFQUFtQixTQUFRRCxJQUFLLFFBQWhDO0FBQ0EsU0FBT0QsR0FBUDtBQUNELENBTDJCLEVBS3pCLEVBTHlCLENBQTVCO0FBT0E7Ozs7Ozs7Ozs7QUFTQSxNQUFNRyxrQkFBa0IsR0FBRyxJQUFJQyxNQUFKLENBQVcsc0JBQVgsQ0FBM0I7O0FBRWUsTUFBTUMsbUJBQU4sQ0FBMEI7QUFTdkNaLEVBQUFBLFdBQVcsQ0FBQ2EsVUFBRCxFQUFhQyxPQUFPLEdBQUcsRUFBdkIsRUFBMkI7QUFDcEMsU0FBS0QsVUFBTCxHQUFrQkEsVUFBbEI7O0FBQ0EsUUFBSUMsT0FBTyxDQUFDQyxLQUFaLEVBQW1CO0FBQ2pCLFdBQUtDLFlBQUwsR0FBb0JGLE9BQU8sQ0FBQ0MsS0FBNUI7QUFDRCxLQUZELE1BRU87QUFDTCxZQUFNRSxXQUFXLEdBQUdILE9BQU8sQ0FBQ0csV0FBUixJQUF1QkMsSUFBSSxDQUFDQyxHQUFMLENBQVMsQ0FBVCxFQUFZQyxZQUFHQyxJQUFILEdBQVVDLE1BQXRCLENBQTNDO0FBQ0EsV0FBS04sWUFBTCxHQUFvQixJQUFJTyxtQkFBSixDQUFlO0FBQUNOLFFBQUFBO0FBQUQsT0FBZixDQUFwQjtBQUNEOztBQUVELFNBQUtPLE1BQUwsR0FBY1YsT0FBTyxDQUFDVSxNQUFSLEtBQW1CQyxLQUFLLElBQUlDLE9BQU8sQ0FBQ0MsTUFBUixFQUE1QixDQUFkOztBQUNBLFNBQUtDLGFBQUwsR0FBcUJkLE9BQU8sQ0FBQ2MsYUFBN0I7O0FBRUEsUUFBSWhDLFFBQVEsS0FBSyxJQUFqQixFQUF1QjtBQUNyQkEsTUFBQUEsUUFBUSxHQUFHLENBQUNpQyxpQkFBT0MsZ0JBQVAsR0FBMEJDLFNBQTFCLEVBQVo7QUFDRDtBQUNGO0FBRUQ7Ozs7Ozs7O0FBTUFDLEVBQUFBLGlCQUFpQixDQUFDUixNQUFELEVBQVM7QUFDeEIsU0FBS0EsTUFBTCxHQUFjQSxNQUFkO0FBQ0QsR0FsQ3NDLENBb0N2Qzs7O0FBQ0EsUUFBTVMsSUFBTixDQUFXQyxJQUFYLEVBQWlCcEIsT0FBTyxHQUFHRixtQkFBbUIsQ0FBQ3VCLGVBQS9DLEVBQWdFO0FBQzlEO0FBQ0EsVUFBTTtBQUFDQyxNQUFBQSxLQUFEO0FBQVFDLE1BQUFBLGtCQUFSO0FBQTRCQyxNQUFBQSxhQUE1QjtBQUEyQ0MsTUFBQUEsZ0JBQTNDO0FBQTZEQyxNQUFBQTtBQUE3RCxRQUErRTFCLE9BQXJGO0FBQ0EsVUFBTTJCLFdBQVcsR0FBR1AsSUFBSSxDQUFDLENBQUQsQ0FBeEI7QUFDQSxVQUFNUSxhQUFhLEdBQUcsSUFBSUMsNkJBQUosRUFBdEI7QUFDQSxVQUFNQyxrQkFBa0IsR0FBR0MsT0FBTyxDQUFDQyxHQUFSLENBQVlDLDJCQUFaLElBQTJDQyxJQUFJLENBQUNDLE1BQUwsQ0FBWUMsR0FBWixDQUFnQix1QkFBaEIsQ0FBdEU7QUFFQSxVQUFNQyxhQUFhLEdBQUksT0FBTWpCLElBQUksQ0FBQ2tCLElBQUwsQ0FBVSxHQUFWLENBQWUsT0FBTSxLQUFLdkMsVUFBVyxFQUFsRTs7QUFDQSxVQUFNd0MsWUFBWSxHQUFHQyx3QkFBZUMsY0FBZixDQUErQixPQUFNckIsSUFBSSxDQUFDa0IsSUFBTCxDQUFVLEdBQVYsQ0FBZSxFQUFwRCxDQUFyQjs7QUFDQUMsSUFBQUEsWUFBWSxDQUFDRyxJQUFiLENBQWtCLFFBQWxCO0FBRUF0QixJQUFBQSxJQUFJLENBQUN6QixPQUFMLENBQWEsR0FBR0osbUJBQWhCOztBQUVBLFFBQUlSLGVBQWUsS0FBSyxJQUF4QixFQUE4QjtBQUM1QjtBQUNBQSxNQUFBQSxlQUFlLEdBQUcsSUFBSTZCLE9BQUosQ0FBWStCLE9BQU8sSUFBSTtBQUN2Q0MsK0JBQWF6QixJQUFiLENBQWtCLGlCQUFsQixFQUFxQyxDQUFDMEIsS0FBRCxFQUFRQyxNQUFSLEtBQW1CO0FBQ3REO0FBQ0EsY0FBSUQsS0FBSixFQUFXO0FBQ1Q7QUFDQUYsWUFBQUEsT0FBTyxDQUFDLElBQUQsQ0FBUDtBQUNBO0FBQ0Q7O0FBRURBLFVBQUFBLE9BQU8sQ0FBQ0csTUFBTSxDQUFDQyxJQUFQLEVBQUQsQ0FBUDtBQUNELFNBVEQ7QUFVRCxPQVhpQixDQUFsQjtBQVlEOztBQUNELFVBQU1DLFFBQVEsR0FBRyxNQUFNakUsZUFBdkI7QUFFQSxXQUFPLEtBQUttQixZQUFMLENBQWtCK0MsSUFBbEIsQ0FBdUIsWUFBWTtBQUN4Q1YsTUFBQUEsWUFBWSxDQUFDRyxJQUFiLENBQWtCLFNBQWxCO0FBQ0EsVUFBSVEsZUFBSjtBQUVBLFlBQU1DLFNBQVMsR0FBRyxFQUFsQjs7QUFDQSxVQUFJcEIsT0FBTyxDQUFDQyxHQUFSLENBQVlvQixJQUFoQixFQUFzQjtBQUNwQkQsUUFBQUEsU0FBUyxDQUFDRixJQUFWLENBQWVsQixPQUFPLENBQUNDLEdBQVIsQ0FBWW9CLElBQTNCO0FBQ0Q7O0FBQ0QsVUFBSUosUUFBSixFQUFjO0FBQ1pHLFFBQUFBLFNBQVMsQ0FBQ0YsSUFBVixDQUFlRCxRQUFmO0FBQ0Q7O0FBRUQsWUFBTWhCLEdBQUcsc0JBQ0pELE9BQU8sQ0FBQ0MsR0FESjtBQUVQcUIsUUFBQUEsbUJBQW1CLEVBQUUsR0FGZDtBQUdQQyxRQUFBQSxrQkFBa0IsRUFBRSxHQUhiO0FBSVBGLFFBQUFBLElBQUksRUFBRUQsU0FBUyxDQUFDYixJQUFWLENBQWVpQixjQUFLQyxTQUFwQjtBQUpDLFFBQVQ7O0FBT0EsWUFBTUMsVUFBVSxHQUFHLElBQUlDLG1CQUFKLEVBQW5COztBQUVBLFVBQUlsQyxhQUFKLEVBQW1CO0FBQ2pCLGNBQU1pQyxVQUFVLENBQUNFLE1BQVgsRUFBTjtBQUNBdkMsUUFBQUEsSUFBSSxDQUFDekIsT0FBTCxDQUFhLElBQWIsRUFBb0IsZUFBYzhELFVBQVUsQ0FBQ0csZUFBWCxFQUE2QixFQUEvRDtBQUNEOztBQUVELFVBQUlyQyxrQkFBSixFQUF3QjtBQUN0QjJCLFFBQUFBLGVBQWUsR0FBRyxJQUFJVyx3QkFBSixDQUFvQkosVUFBcEIsQ0FBbEI7QUFDQSxjQUFNUCxlQUFlLENBQUNZLEtBQWhCLENBQXNCLEtBQUtwRCxNQUEzQixDQUFOO0FBRUFzQixRQUFBQSxHQUFHLENBQUMrQixlQUFKLEdBQXNCTixVQUFVLENBQUNPLFdBQVgsRUFBdEI7QUFDQWhDLFFBQUFBLEdBQUcsQ0FBQ2lDLHdCQUFKLEdBQStCLHFDQUF1QlIsVUFBVSxDQUFDUyxZQUFYLEVBQXZCLENBQS9CO0FBQ0FsQyxRQUFBQSxHQUFHLENBQUNtQywyQkFBSixHQUFrQyxxQ0FBdUJWLFVBQVUsQ0FBQ1cscUJBQVgsRUFBdkIsQ0FBbEM7QUFDQXBDLFFBQUFBLEdBQUcsQ0FBQ3FDLHlCQUFKLEdBQWdDLHFDQUF1QixpQ0FBdkIsQ0FBaEM7QUFDQXJDLFFBQUFBLEdBQUcsQ0FBQ3NDLHFCQUFKLEdBQTRCcEIsZUFBZSxDQUFDcUIsVUFBaEIsRUFBNUI7QUFFQXZDLFFBQUFBLEdBQUcsQ0FBQ3dDLHdCQUFKLEdBQStCLEtBQUt6RSxVQUFwQztBQUNBaUMsUUFBQUEsR0FBRyxDQUFDeUMsdUJBQUosR0FBOEIsNkJBQTlCO0FBQ0F6QyxRQUFBQSxHQUFHLENBQUMwQyxnQ0FBSixHQUF1QyxrQ0FBb0IsaUJBQXBCLENBQXZDLENBWnNCLENBY3RCO0FBQ0E7QUFDQTtBQUNBOztBQUNBLFlBQUksQ0FBQzNDLE9BQU8sQ0FBQ0MsR0FBUixDQUFZMkMsT0FBYixJQUF3QjVDLE9BQU8sQ0FBQ0MsR0FBUixDQUFZMkMsT0FBWixDQUFvQm5FLE1BQXBCLEtBQStCLENBQTNELEVBQThEO0FBQzVEd0IsVUFBQUEsR0FBRyxDQUFDMkMsT0FBSixHQUFjLHlCQUFkO0FBQ0Q7O0FBRUQzQyxRQUFBQSxHQUFHLENBQUM0Qyx5QkFBSixHQUFnQzdDLE9BQU8sQ0FBQ0MsR0FBUixDQUFZb0IsSUFBWixJQUFvQixFQUFwRDtBQUNBcEIsUUFBQUEsR0FBRyxDQUFDNkMsZ0NBQUosR0FBdUM5QyxPQUFPLENBQUNDLEdBQVIsQ0FBWThDLFdBQVosSUFBMkIsRUFBbEU7QUFDQTlDLFFBQUFBLEdBQUcsQ0FBQytDLGdDQUFKLEdBQXVDaEQsT0FBTyxDQUFDQyxHQUFSLENBQVlnRCxXQUFaLElBQTJCLEVBQWxFO0FBQ0FoRCxRQUFBQSxHQUFHLENBQUNpRCxvQ0FBSixHQUEyQ2xELE9BQU8sQ0FBQ0MsR0FBUixDQUFZa0QsZUFBWixJQUErQixFQUExRTtBQUNBbEQsUUFBQUEsR0FBRyxDQUFDbUQscUJBQUosR0FBNEJqRCxJQUFJLENBQUNrRCxVQUFMLEtBQW9CLE1BQXBCLEdBQTZCLE9BQXpEO0FBRUFwRCxRQUFBQSxHQUFHLENBQUNnRCxXQUFKLEdBQWtCLHFDQUF1QnZCLFVBQVUsQ0FBQzRCLFlBQVgsRUFBdkIsQ0FBbEI7QUFDQXJELFFBQUFBLEdBQUcsQ0FBQzhDLFdBQUosR0FBa0IscUNBQXVCckIsVUFBVSxDQUFDNEIsWUFBWCxFQUF2QixDQUFsQjs7QUFFQSxZQUFJdEQsT0FBTyxDQUFDdUQsUUFBUixLQUFxQixPQUF6QixFQUFrQztBQUNoQ3RELFVBQUFBLEdBQUcsQ0FBQ2tELGVBQUosR0FBc0J6QixVQUFVLENBQUM4QixlQUFYLEVBQXRCO0FBQ0QsU0FGRCxNQUVPLElBQUl4RCxPQUFPLENBQUNDLEdBQVIsQ0FBWWtELGVBQWhCLEVBQWlDO0FBQ3RDbEQsVUFBQUEsR0FBRyxDQUFDa0QsZUFBSixHQUFzQm5ELE9BQU8sQ0FBQ0MsR0FBUixDQUFZa0QsZUFBbEM7QUFDRCxTQUZNLE1BRUE7QUFDTGxELFVBQUFBLEdBQUcsQ0FBQ3dELE9BQUosR0FBY3pELE9BQU8sQ0FBQ0MsR0FBUixDQUFZd0QsT0FBMUI7QUFDRDs7QUFFRCxjQUFNQyxrQkFBa0IsR0FBRyxxQ0FBdUJoQyxVQUFVLENBQUNpQyxxQkFBWCxFQUF2QixDQUEzQjtBQUNBdEUsUUFBQUEsSUFBSSxDQUFDekIsT0FBTCxDQUFhLElBQWIsRUFBb0IscUJBQW9COEYsa0JBQW1CLEVBQTNEO0FBQ0Q7O0FBRUQsVUFBSWpFLGFBQWEsSUFBSUQsa0JBQWpCLElBQXVDRSxnQkFBM0MsRUFBNkQ7QUFDM0RPLFFBQUFBLEdBQUcsQ0FBQzJELHNCQUFKLEdBQTZCLE1BQTdCO0FBQ0Q7QUFFRDs7O0FBQ0EsVUFBSTdELGtCQUFKLEVBQXdCO0FBQ3RCRSxRQUFBQSxHQUFHLENBQUM0RCxTQUFKLEdBQWdCLE1BQWhCO0FBQ0E1RCxRQUFBQSxHQUFHLENBQUM2RCxjQUFKLEdBQXFCLE1BQXJCO0FBQ0Q7O0FBRUQsVUFBSUMsSUFBSSxHQUFHO0FBQUM5RCxRQUFBQTtBQUFELE9BQVg7O0FBRUEsVUFBSVYsS0FBSixFQUFXO0FBQ1R3RSxRQUFBQSxJQUFJLENBQUN4RSxLQUFMLEdBQWFBLEtBQWI7QUFDQXdFLFFBQUFBLElBQUksQ0FBQ0MsYUFBTCxHQUFxQixNQUFyQjtBQUNEO0FBRUQ7OztBQUNBLFVBQUloRSxPQUFPLENBQUNDLEdBQVIsQ0FBWWdFLGVBQWhCLEVBQWlDO0FBQy9CQyxRQUFBQSxPQUFPLENBQUNDLElBQVIsQ0FBYyxPQUFNN0QsYUFBYyxFQUFsQztBQUNEOztBQUVELGFBQU8sSUFBSXpCLE9BQUosQ0FBWSxPQUFPK0IsT0FBUCxFQUFnQjlCLE1BQWhCLEtBQTJCO0FBQzVDLFlBQUliLE9BQU8sQ0FBQ21HLFNBQVosRUFBdUI7QUFDckIsZ0JBQU1DLFdBQVcsR0FBRyxNQUFNcEcsT0FBTyxDQUFDbUcsU0FBUixDQUFrQjtBQUFDL0UsWUFBQUEsSUFBRDtBQUFPMEUsWUFBQUE7QUFBUCxXQUFsQixDQUExQjtBQUNBMUUsVUFBQUEsSUFBSSxHQUFHZ0YsV0FBVyxDQUFDaEYsSUFBbkI7QUFDQTBFLFVBQUFBLElBQUksR0FBR00sV0FBVyxDQUFDTixJQUFuQjtBQUNEOztBQUNELGNBQU07QUFBQ08sVUFBQUEsT0FBRDtBQUFVQyxVQUFBQTtBQUFWLFlBQW9CLEtBQUtDLGlCQUFMLENBQXVCbkYsSUFBdkIsRUFBNkIwRSxJQUE3QixFQUFtQ3ZELFlBQW5DLENBQTFCO0FBQ0EsWUFBSWlFLFlBQVksR0FBRyxLQUFuQjs7QUFDQSxZQUFJdEQsZUFBSixFQUFxQjtBQUNuQnRCLFVBQUFBLGFBQWEsQ0FBQzZFLEdBQWQsQ0FBa0J2RCxlQUFlLENBQUN3RCxXQUFoQixDQUE0QixPQUFPO0FBQUNDLFlBQUFBO0FBQUQsV0FBUCxLQUF3QjtBQUNwRUgsWUFBQUEsWUFBWSxHQUFHLElBQWY7QUFDQSxrQkFBTUYsTUFBTSxFQUFaLENBRm9FLENBSXBFO0FBQ0E7QUFDQTtBQUNBOztBQUNBLGtCQUFNLElBQUkxRixPQUFKLENBQVksQ0FBQ2dHLFdBQUQsRUFBY0MsVUFBZCxLQUE2QjtBQUM3Q0MsY0FBQUEsT0FBTyxDQUFDLFdBQUQsQ0FBUCxDQUFxQkgsVUFBckIsRUFBaUMsU0FBakMsRUFBNENJLEdBQUcsSUFBSTtBQUNqRDtBQUNBLG9CQUFJQSxHQUFKLEVBQVM7QUFBRUYsa0JBQUFBLFVBQVUsQ0FBQ0UsR0FBRCxDQUFWO0FBQWtCLGlCQUE3QixNQUFtQztBQUFFSCxrQkFBQUEsV0FBVztBQUFLO0FBQ3RELGVBSEQ7QUFJRCxhQUxLLENBQU47QUFNRCxXQWRpQixDQUFsQjtBQWVEOztBQUVELGNBQU07QUFBQzlELFVBQUFBLE1BQUQ7QUFBU2tFLFVBQUFBLE1BQVQ7QUFBaUJDLFVBQUFBLFFBQWpCO0FBQTJCQyxVQUFBQSxNQUEzQjtBQUFtQ0MsVUFBQUE7QUFBbkMsWUFBNkMsTUFBTWQsT0FBTyxDQUFDZSxLQUFSLENBQWNMLEdBQUcsSUFBSTtBQUM1RSxjQUFJQSxHQUFHLENBQUNHLE1BQVIsRUFBZ0I7QUFDZCxtQkFBTztBQUFDQSxjQUFBQSxNQUFNLEVBQUVILEdBQUcsQ0FBQ0c7QUFBYixhQUFQO0FBQ0Q7O0FBQ0RyRyxVQUFBQSxNQUFNLENBQUNrRyxHQUFELENBQU47QUFDQSxpQkFBTyxFQUFQO0FBQ0QsU0FOd0QsQ0FBekQ7O0FBUUEsWUFBSUksTUFBSixFQUFZO0FBQ1YsZ0JBQU07QUFBQ0UsWUFBQUEsUUFBRDtBQUFXQyxZQUFBQSxTQUFYO0FBQXNCQyxZQUFBQTtBQUF0QixjQUFpQ0osTUFBdkM7QUFDQSxnQkFBTUssR0FBRyxHQUFHQyxXQUFXLENBQUNELEdBQVosRUFBWjtBQUNBakYsVUFBQUEsWUFBWSxDQUFDRyxJQUFiLENBQWtCLFVBQWxCLEVBQThCOEUsR0FBRyxHQUFHSCxRQUFOLEdBQWlCQyxTQUFqQixHQUE2QkMsT0FBM0Q7QUFDQWhGLFVBQUFBLFlBQVksQ0FBQ0csSUFBYixDQUFrQixTQUFsQixFQUE2QjhFLEdBQUcsR0FBR0gsUUFBTixHQUFpQkUsT0FBOUM7QUFDQWhGLFVBQUFBLFlBQVksQ0FBQ0csSUFBYixDQUFrQixLQUFsQixFQUF5QjhFLEdBQUcsR0FBR0QsT0FBL0I7QUFDRDs7QUFDRGhGLFFBQUFBLFlBQVksQ0FBQ21GLFFBQWI7QUFFQTs7QUFDQSxZQUFJM0YsT0FBTyxDQUFDQyxHQUFSLENBQVlnRSxlQUFoQixFQUFpQztBQUMvQkMsVUFBQUEsT0FBTyxDQUFDMEIsT0FBUixDQUFpQixPQUFNdEYsYUFBYyxFQUFyQztBQUNEOztBQUVELFlBQUlhLGVBQUosRUFBcUI7QUFDbkJBLFVBQUFBLGVBQWUsQ0FBQzBFLFNBQWhCO0FBQ0Q7O0FBQ0RoRyxRQUFBQSxhQUFhLENBQUNpRyxPQUFkO0FBRUE7O0FBQ0EsWUFBSS9GLGtCQUFKLEVBQXdCO0FBQ3RCLGdCQUFNZ0csdUJBQXVCLEdBQUdDLEdBQUcsSUFBSTtBQUNyQyxnQkFBSSxDQUFDQSxHQUFMLEVBQVU7QUFBRSxxQkFBTyxFQUFQO0FBQVk7O0FBRXhCLG1CQUFPQSxHQUFHLENBQ1BDLE9BREksQ0FDSSxVQURKLEVBQ2dCLFNBRGhCLEVBRUpBLE9BRkksQ0FFSSxVQUZKLEVBRWdCLE9BRmhCLENBQVA7QUFHRCxXQU5EOztBQVFBLGNBQUlsSixRQUFKLEVBQWM7QUFDWixnQkFBSW1KLE9BQU8sR0FBSSxPQUFNNUYsYUFBYyxJQUFuQzs7QUFDQSxnQkFBSTRFLFFBQVEsS0FBS2lCLFNBQWpCLEVBQTRCO0FBQzFCRCxjQUFBQSxPQUFPLElBQUssZ0JBQWVoQixRQUFTLElBQXBDO0FBQ0QsYUFGRCxNQUVPLElBQUlDLE1BQUosRUFBWTtBQUNqQmUsY0FBQUEsT0FBTyxJQUFLLGdCQUFlZixNQUFPLElBQWxDO0FBQ0Q7O0FBQ0QsZ0JBQUk1RixLQUFLLElBQUlBLEtBQUssQ0FBQ2QsTUFBTixLQUFpQixDQUE5QixFQUFpQztBQUMvQnlILGNBQUFBLE9BQU8sSUFBSyxXQUFVSCx1QkFBdUIsQ0FBQ3hHLEtBQUQsQ0FBUSxJQUFyRDtBQUNEOztBQUNEMkcsWUFBQUEsT0FBTyxJQUFJLFNBQVg7O0FBQ0EsZ0JBQUluRixNQUFNLENBQUN0QyxNQUFQLEtBQWtCLENBQXRCLEVBQXlCO0FBQ3ZCeUgsY0FBQUEsT0FBTyxJQUFJLFlBQVg7QUFDRCxhQUZELE1BRU87QUFDTEEsY0FBQUEsT0FBTyxJQUFLLEtBQUlILHVCQUF1QixDQUFDaEYsTUFBRCxDQUFTLElBQWhEO0FBQ0Q7O0FBQ0RtRixZQUFBQSxPQUFPLElBQUksU0FBWDs7QUFDQSxnQkFBSWpCLE1BQU0sQ0FBQ3hHLE1BQVAsS0FBa0IsQ0FBdEIsRUFBeUI7QUFDdkJ5SCxjQUFBQSxPQUFPLElBQUksWUFBWDtBQUNELGFBRkQsTUFFTztBQUNMQSxjQUFBQSxPQUFPLElBQUssS0FBSUgsdUJBQXVCLENBQUNkLE1BQUQsQ0FBUyxJQUFoRDtBQUNEOztBQUVEZixZQUFBQSxPQUFPLENBQUNrQyxHQUFSLENBQVlGLE9BQVo7QUFDRCxXQXhCRCxNQXdCTztBQUNMLGtCQUFNRyxXQUFXLEdBQUcsaUNBQXBCO0FBRUFuQyxZQUFBQSxPQUFPLENBQUNvQyxjQUFSLENBQXdCLE9BQU1oRyxhQUFjLEVBQTVDOztBQUNBLGdCQUFJNEUsUUFBUSxLQUFLaUIsU0FBakIsRUFBNEI7QUFDMUJqQyxjQUFBQSxPQUFPLENBQUNrQyxHQUFSLENBQVksb0JBQVosRUFBa0NDLFdBQWxDLEVBQStDLG9DQUEvQyxFQUFxRm5CLFFBQXJGO0FBQ0QsYUFGRCxNQUVPLElBQUlDLE1BQUosRUFBWTtBQUNqQmpCLGNBQUFBLE9BQU8sQ0FBQ2tDLEdBQVIsQ0FBWSxvQkFBWixFQUFrQ0MsV0FBbEMsRUFBK0Msb0NBQS9DLEVBQXFGbEIsTUFBckY7QUFDRDs7QUFDRGpCLFlBQUFBLE9BQU8sQ0FBQ2tDLEdBQVIsQ0FDRSx1QkFERixFQUVFQyxXQUZGLEVBRWUsb0NBRmYsRUFHRUUsY0FBS0MsT0FBTCxDQUFhbkgsSUFBYixFQUFtQjtBQUFDb0gsY0FBQUEsV0FBVyxFQUFFQztBQUFkLGFBQW5CLENBSEY7O0FBS0EsZ0JBQUluSCxLQUFLLElBQUlBLEtBQUssQ0FBQ2QsTUFBTixLQUFpQixDQUE5QixFQUFpQztBQUMvQnlGLGNBQUFBLE9BQU8sQ0FBQ2tDLEdBQVIsQ0FBWSxTQUFaLEVBQXVCQyxXQUF2QjtBQUNBbkMsY0FBQUEsT0FBTyxDQUFDa0MsR0FBUixDQUFZTCx1QkFBdUIsQ0FBQ3hHLEtBQUQsQ0FBbkM7QUFDRDs7QUFDRDJFLFlBQUFBLE9BQU8sQ0FBQ2tDLEdBQVIsQ0FBWSxVQUFaLEVBQXdCQyxXQUF4QjtBQUNBbkMsWUFBQUEsT0FBTyxDQUFDa0MsR0FBUixDQUFZTCx1QkFBdUIsQ0FBQ2hGLE1BQUQsQ0FBbkM7QUFDQW1ELFlBQUFBLE9BQU8sQ0FBQ2tDLEdBQVIsQ0FBWSxVQUFaLEVBQXdCQyxXQUF4QjtBQUNBbkMsWUFBQUEsT0FBTyxDQUFDa0MsR0FBUixDQUFZTCx1QkFBdUIsQ0FBQ2QsTUFBRCxDQUFuQztBQUNBZixZQUFBQSxPQUFPLENBQUN5QyxRQUFSO0FBQ0Q7QUFDRjs7QUFFRCxZQUFJekIsUUFBUSxLQUFLLENBQWIsSUFBa0IsQ0FBQ1QsWUFBdkIsRUFBcUM7QUFDbkMsZ0JBQU1PLEdBQUcsR0FBRyxJQUFJL0gsUUFBSixDQUNULEdBQUVxRCxhQUFjLHFCQUFvQjRFLFFBQVMsYUFBWW5FLE1BQU8sYUFBWWtFLE1BQU8sRUFEMUUsQ0FBWjtBQUdBRCxVQUFBQSxHQUFHLENBQUM0QixJQUFKLEdBQVcxQixRQUFYO0FBQ0FGLFVBQUFBLEdBQUcsQ0FBQzZCLE1BQUosR0FBYTVCLE1BQWI7QUFDQUQsVUFBQUEsR0FBRyxDQUFDOEIsTUFBSixHQUFhL0YsTUFBYjtBQUNBaUUsVUFBQUEsR0FBRyxDQUFDK0IsT0FBSixHQUFjekcsYUFBZDtBQUNBeEIsVUFBQUEsTUFBTSxDQUFDa0csR0FBRCxDQUFOO0FBQ0Q7O0FBRUQsWUFBSSxDQUFDekgsb0JBQW9CLENBQUN5SixRQUFyQixDQUE4QnBILFdBQTlCLENBQUwsRUFBaUQ7QUFDL0MsK0NBQWlCQSxXQUFqQjtBQUNEOztBQUNEZ0IsUUFBQUEsT0FBTyxDQUFDRyxNQUFELENBQVA7QUFDRCxPQWhJTSxDQUFQO0FBaUlELEtBNU5NLEVBNE5KO0FBQUNrRyxNQUFBQSxRQUFRLEVBQUUsQ0FBQ3RIO0FBQVosS0E1TkksQ0FBUDtBQTZOQTtBQUNEOztBQUVELFFBQU11SCxPQUFOLENBQWM3SCxJQUFkLEVBQW9CcEIsT0FBcEIsRUFBNkI7QUFDM0IsUUFBSTtBQUNGLGFBQU8sTUFBTSxLQUFLbUIsSUFBTCxDQUFVQyxJQUFJLENBQUM4SCxLQUFMLEVBQVY7QUFDWDFILFFBQUFBLGFBQWEsRUFBRSxJQURKO0FBRVhDLFFBQUFBLGdCQUFnQixFQUFFO0FBRlAsU0FHUnpCLE9BSFEsRUFBYjtBQUtELEtBTkQsQ0FNRSxPQUFPbUosQ0FBUCxFQUFVO0FBQ1YsVUFBSSxhQUFhQyxJQUFiLENBQWtCRCxDQUFDLENBQUNQLE1BQXBCLENBQUosRUFBaUM7QUFDL0IsZUFBTyxNQUFNLEtBQUt6SCxJQUFMLENBQVVDLElBQVY7QUFDWEcsVUFBQUEsa0JBQWtCLEVBQUUsSUFEVDtBQUVYQyxVQUFBQSxhQUFhLEVBQUUsSUFGSjtBQUdYQyxVQUFBQSxnQkFBZ0IsRUFBRTtBQUhQLFdBSVJ6QixPQUpRLEVBQWI7QUFNRCxPQVBELE1BT087QUFDTCxjQUFNbUosQ0FBTjtBQUNEO0FBQ0Y7QUFDRjs7QUFFRDVDLEVBQUFBLGlCQUFpQixDQUFDbkYsSUFBRCxFQUFPcEIsT0FBUCxFQUFnQnFKLE1BQU0sR0FBRyxJQUF6QixFQUErQjtBQUM5QyxRQUFJdEgsT0FBTyxDQUFDQyxHQUFSLENBQVlzSCwyQkFBWixJQUEyQyxDQUFDQyx1QkFBY0MsV0FBZCxHQUE0QkMsT0FBNUIsRUFBaEQsRUFBdUY7QUFDckZKLE1BQUFBLE1BQU0sSUFBSUEsTUFBTSxDQUFDM0csSUFBUCxDQUFZLFVBQVosQ0FBVjtBQUVBLFVBQUlnSCxRQUFKOztBQUNBMUosTUFBQUEsT0FBTyxDQUFDMkosZUFBUixHQUEwQkMsS0FBSyxJQUFJO0FBQ2pDRixRQUFBQSxRQUFRLEdBQUdFLEtBQUssQ0FBQ0MsR0FBakI7QUFFQTs7QUFDQUQsUUFBQUEsS0FBSyxDQUFDdEksS0FBTixDQUFZd0ksRUFBWixDQUFlLE9BQWYsRUFBd0IvQyxHQUFHLElBQUk7QUFDN0IsZ0JBQU0sSUFBSTlILEtBQUosQ0FDSCwrQkFBOEJtQyxJQUFJLENBQUNrQixJQUFMLENBQVUsR0FBVixDQUFlLE9BQU0sS0FBS3ZDLFVBQVcsS0FBSUMsT0FBTyxDQUFDc0IsS0FBTSxLQUFJeUYsR0FBSSxFQUQxRixDQUFOO0FBRUQsU0FIRDtBQUlELE9BUkQ7O0FBVUEsWUFBTVYsT0FBTyxHQUFHMEQsbUJBQVc1SSxJQUFYLENBQWdCQyxJQUFoQixFQUFzQixLQUFLckIsVUFBM0IsRUFBdUNDLE9BQXZDLENBQWhCOztBQUNBcUosTUFBQUEsTUFBTSxJQUFJQSxNQUFNLENBQUMzRyxJQUFQLENBQVksU0FBWixDQUFWO0FBQ0EsYUFBTztBQUNMMkQsUUFBQUEsT0FESztBQUVMQyxRQUFBQSxNQUFNLEVBQUUsTUFBTTtBQUNaO0FBQ0EsY0FBSSxDQUFDb0QsUUFBTCxFQUFlO0FBQ2IsbUJBQU85SSxPQUFPLENBQUMrQixPQUFSLEVBQVA7QUFDRDs7QUFFRCxpQkFBTyxJQUFJL0IsT0FBSixDQUFZLENBQUMrQixPQUFELEVBQVU5QixNQUFWLEtBQXFCO0FBQ3RDaUcsWUFBQUEsT0FBTyxDQUFDLFdBQUQsQ0FBUCxDQUFxQjRDLFFBQXJCLEVBQStCLFNBQS9CLEVBQTBDM0MsR0FBRyxJQUFJO0FBQy9DO0FBQ0Esa0JBQUlBLEdBQUosRUFBUztBQUFFbEcsZ0JBQUFBLE1BQU0sQ0FBQ2tHLEdBQUQsQ0FBTjtBQUFjLGVBQXpCLE1BQStCO0FBQUVwRSxnQkFBQUEsT0FBTztBQUFLO0FBQzlDLGFBSEQ7QUFJRCxXQUxNLENBQVA7QUFNRDtBQWRJLE9BQVA7QUFnQkQsS0FoQ0QsTUFnQ087QUFDTCxZQUFNN0IsYUFBYSxHQUFHLEtBQUtBLGFBQUwsSUFBc0J5SSx1QkFBY0MsV0FBZCxFQUE1Qzs7QUFDQSxhQUFPMUksYUFBYSxDQUFDa0osT0FBZCxDQUFzQjtBQUMzQjVJLFFBQUFBLElBRDJCO0FBRTNCckIsUUFBQUEsVUFBVSxFQUFFLEtBQUtBLFVBRlU7QUFHM0JDLFFBQUFBO0FBSDJCLE9BQXRCLENBQVA7QUFLRDtBQUNGOztBQUVELFFBQU1pSyxnQkFBTixHQUF5QjtBQUN2QixRQUFJO0FBQ0YsWUFBTUMsaUJBQUdDLElBQUgsQ0FBUSxLQUFLcEssVUFBYixDQUFOLENBREUsQ0FDOEI7O0FBQ2hDLFlBQU1xSyxNQUFNLEdBQUcsTUFBTSxLQUFLakosSUFBTCxDQUFVLENBQUMsV0FBRCxFQUFjLG1CQUFkLEVBQW1Db0MsY0FBS2pCLElBQUwsQ0FBVSxLQUFLdkMsVUFBZixFQUEyQixNQUEzQixDQUFuQyxDQUFWLENBQXJCO0FBQ0EsWUFBTXNLLFNBQVMsR0FBR0QsTUFBTSxDQUFDckgsSUFBUCxFQUFsQjtBQUNBLGFBQU8sOEJBQWdCc0gsU0FBaEIsQ0FBUDtBQUNELEtBTEQsQ0FLRSxPQUFPbEIsQ0FBUCxFQUFVO0FBQ1YsYUFBTyxJQUFQO0FBQ0Q7QUFDRjs7QUFFRG1CLEVBQUFBLElBQUksR0FBRztBQUNMLFdBQU8sS0FBS25KLElBQUwsQ0FBVSxDQUFDLE1BQUQsRUFBUyxLQUFLcEIsVUFBZCxDQUFWLENBQVA7QUFDRDtBQUVEOzs7OztBQUdBd0ssRUFBQUEsVUFBVSxDQUFDQyxLQUFELEVBQVE7QUFDaEIsUUFBSUEsS0FBSyxDQUFDaEssTUFBTixLQUFpQixDQUFyQixFQUF3QjtBQUFFLGFBQU9JLE9BQU8sQ0FBQytCLE9BQVIsQ0FBZ0IsSUFBaEIsQ0FBUDtBQUErQjs7QUFDekQsVUFBTXZCLElBQUksR0FBRyxDQUFDLEtBQUQsRUFBUXFKLE1BQVIsQ0FBZUQsS0FBSyxDQUFDRSxHQUFOLENBQVVDLHFCQUFWLENBQWYsQ0FBYjtBQUNBLFdBQU8sS0FBS3hKLElBQUwsQ0FBVUMsSUFBVixFQUFnQjtBQUFDTSxNQUFBQSxjQUFjLEVBQUU7QUFBakIsS0FBaEIsQ0FBUDtBQUNEOztBQUVELFFBQU1rSiwwQkFBTixHQUFtQztBQUNqQyxRQUFJQyxZQUFZLEdBQUcsTUFBTSxLQUFLQyxTQUFMLENBQWUsaUJBQWYsQ0FBekI7O0FBQ0EsUUFBSSxDQUFDRCxZQUFMLEVBQW1CO0FBQ2pCLGFBQU8sSUFBUDtBQUNEOztBQUVELFVBQU1FLE9BQU8sR0FBR3pLLFlBQUcwSyxPQUFILEVBQWhCOztBQUVBSCxJQUFBQSxZQUFZLEdBQUdBLFlBQVksQ0FBQzlILElBQWIsR0FBb0JpRixPQUFwQixDQUE0QnBJLGtCQUE1QixFQUFnRCxDQUFDcUwsQ0FBRCxFQUFJQyxJQUFKLEtBQWE7QUFDMUU7QUFDQSxhQUFRLEdBQUVBLElBQUksR0FBRzNILGNBQUtqQixJQUFMLENBQVVpQixjQUFLNEgsT0FBTCxDQUFhSixPQUFiLENBQVYsRUFBaUNHLElBQWpDLENBQUgsR0FBNENILE9BQVEsR0FBbEU7QUFDRCxLQUhjLENBQWY7QUFJQUYsSUFBQUEsWUFBWSxHQUFHLDhCQUFnQkEsWUFBaEIsQ0FBZjs7QUFFQSxRQUFJLENBQUN0SCxjQUFLNkgsVUFBTCxDQUFnQlAsWUFBaEIsQ0FBTCxFQUFvQztBQUNsQ0EsTUFBQUEsWUFBWSxHQUFHdEgsY0FBS2pCLElBQUwsQ0FBVSxLQUFLdkMsVUFBZixFQUEyQjhLLFlBQTNCLENBQWY7QUFDRDs7QUFFRCxRQUFJLEVBQUMsTUFBTSx5QkFBV0EsWUFBWCxDQUFQLENBQUosRUFBcUM7QUFDbkMsWUFBTSxJQUFJNUwsS0FBSixDQUFXLG1EQUFrRDRMLFlBQWEsRUFBMUUsQ0FBTjtBQUNEOztBQUNELFdBQU8sTUFBTVgsaUJBQUdtQixRQUFILENBQVlSLFlBQVosRUFBMEI7QUFBQ1MsTUFBQUEsUUFBUSxFQUFFO0FBQVgsS0FBMUIsQ0FBYjtBQUNEOztBQUVEQyxFQUFBQSxZQUFZLENBQUNmLEtBQUQsRUFBUWdCLE1BQU0sR0FBRyxNQUFqQixFQUF5QjtBQUNuQyxRQUFJaEIsS0FBSyxDQUFDaEssTUFBTixLQUFpQixDQUFyQixFQUF3QjtBQUFFLGFBQU9JLE9BQU8sQ0FBQytCLE9BQVIsQ0FBZ0IsSUFBaEIsQ0FBUDtBQUErQjs7QUFDekQsVUFBTXZCLElBQUksR0FBRyxDQUFDLE9BQUQsRUFBVW9LLE1BQVYsRUFBa0IsSUFBbEIsRUFBd0JmLE1BQXhCLENBQStCRCxLQUFLLENBQUNFLEdBQU4sQ0FBVUMscUJBQVYsQ0FBL0IsQ0FBYjtBQUNBLFdBQU8sS0FBS3hKLElBQUwsQ0FBVUMsSUFBVixFQUFnQjtBQUFDTSxNQUFBQSxjQUFjLEVBQUU7QUFBakIsS0FBaEIsQ0FBUDtBQUNEOztBQUVEK0osRUFBQUEsbUJBQW1CLENBQUNDLFFBQUQsRUFBV0MsT0FBWCxFQUFvQjtBQUNyQyxVQUFNQyxnQkFBZ0IsR0FBRyxLQUFLekssSUFBTCxDQUFVLENBQUMsVUFBRCxFQUFhLElBQWIsRUFBbUIsSUFBbkIsRUFBeUJ1SyxRQUF6QixDQUFWLENBQXpCO0FBQ0EsV0FBTyxLQUFLdkssSUFBTCxDQUFVLENBQUMsY0FBRCxFQUFpQixhQUFqQixFQUFpQyxHQUFFd0ssT0FBUSxjQUFhRCxRQUFTLEVBQWpFLENBQVYsRUFBK0U7QUFDcEZoSyxNQUFBQSxjQUFjLEVBQUUsSUFEb0U7QUFFcEZ5RSxNQUFBQSxTQUFTLEVBQUUsZUFBZTBGLGFBQWYsQ0FBNkI7QUFBQ3pLLFFBQUFBLElBQUQ7QUFBTzBFLFFBQUFBO0FBQVAsT0FBN0IsRUFBMkM7QUFDcEQsY0FBTWdHLEtBQUssR0FBRyxNQUFNRixnQkFBcEI7QUFDQSxjQUFNRyxHQUFHLEdBQUdELEtBQUssQ0FBQ0UsTUFBTixDQUFhLENBQWIsRUFBZ0IsRUFBaEIsQ0FBWjtBQUNBLGVBQU87QUFDTGxHLFVBQUFBLElBREs7QUFFTDFFLFVBQUFBLElBQUksRUFBRSxDQUFDLGNBQUQsRUFBaUIsYUFBakIsRUFBaUMsR0FBRXVLLE9BQVEsSUFBR0ksR0FBSSxJQUFHTCxRQUFTLEVBQTlEO0FBRkQsU0FBUDtBQUlEO0FBVG1GLEtBQS9FLENBQVA7QUFXRDs7QUFFRE8sRUFBQUEsc0JBQXNCLENBQUNQLFFBQUQsRUFBVztBQUMvQixXQUFPLEtBQUt2SyxJQUFMLENBQVUsQ0FBQyxJQUFELEVBQU8sVUFBUCxFQUFtQnVLLFFBQW5CLENBQVYsRUFBd0M7QUFBQ2hLLE1BQUFBLGNBQWMsRUFBRTtBQUFqQixLQUF4QyxDQUFQO0FBQ0Q7O0FBRUR3SyxFQUFBQSxVQUFVLENBQUNDLEtBQUQsRUFBUTtBQUFDTCxJQUFBQTtBQUFELE1BQVUsRUFBbEIsRUFBc0I7QUFDOUIsVUFBTTFLLElBQUksR0FBRyxDQUFDLE9BQUQsRUFBVSxHQUFWLENBQWI7O0FBQ0EsUUFBSTBLLEtBQUosRUFBVztBQUFFMUssTUFBQUEsSUFBSSxDQUFDZ0wsTUFBTCxDQUFZLENBQVosRUFBZSxDQUFmLEVBQWtCLFVBQWxCO0FBQWdDOztBQUM3QyxXQUFPLEtBQUtqTCxJQUFMLENBQVVDLElBQVYsRUFBZ0I7QUFBQ0UsTUFBQUEsS0FBSyxFQUFFNkssS0FBUjtBQUFlekssTUFBQUEsY0FBYyxFQUFFO0FBQS9CLEtBQWhCLENBQVA7QUFDRDs7QUFFRCxRQUFNOEosTUFBTixDQUFhYSxVQUFiLEVBQXlCO0FBQUNDLElBQUFBLFVBQUQ7QUFBYUMsSUFBQUEsS0FBYjtBQUFvQkMsSUFBQUEsU0FBcEI7QUFBK0JDLElBQUFBO0FBQS9CLE1BQTJDLEVBQXBFLEVBQXdFO0FBQ3RFLFVBQU1yTCxJQUFJLEdBQUcsQ0FBQyxRQUFELENBQWI7QUFDQSxRQUFJc0wsR0FBSixDQUZzRSxDQUl0RTtBQUNBOztBQUNBLFFBQUlILEtBQUssSUFBSUYsVUFBVSxDQUFDN0wsTUFBWCxLQUFzQixDQUFuQyxFQUFzQztBQUNwQyxZQUFNO0FBQUNtTSxRQUFBQSxTQUFEO0FBQVlDLFFBQUFBLFdBQVo7QUFBeUJDLFFBQUFBO0FBQXpCLFVBQTJDLE1BQU0sS0FBS0MsYUFBTCxFQUF2RDs7QUFDQSxVQUFJSCxTQUFKLEVBQWU7QUFDYkQsUUFBQUEsR0FBRyxHQUFHTCxVQUFOO0FBQ0QsT0FGRCxNQUVPO0FBQ0xLLFFBQUFBLEdBQUcsR0FBSSxHQUFFRyxjQUFlLE9BQU1ELFdBQVksRUFBcEMsQ0FBc0M3SixJQUF0QyxFQUFOO0FBQ0EwSixRQUFBQSxRQUFRLEdBQUcsSUFBWDtBQUNEO0FBQ0YsS0FSRCxNQVFPO0FBQ0xDLE1BQUFBLEdBQUcsR0FBR0wsVUFBTjtBQUNELEtBaEJxRSxDQWtCdEU7QUFDQTs7O0FBQ0EsVUFBTVUsUUFBUSxHQUFHLE1BQU0sS0FBS25DLDBCQUFMLEVBQXZCOztBQUNBLFFBQUltQyxRQUFKLEVBQWM7QUFFWjtBQUNBO0FBQ0EsVUFBSUMsV0FBVyxHQUFHLE1BQU0sS0FBS2xDLFNBQUwsQ0FBZSxrQkFBZixDQUF4Qjs7QUFDQSxVQUFJLENBQUNrQyxXQUFMLEVBQWtCO0FBQ2hCQSxRQUFBQSxXQUFXLEdBQUcsR0FBZDtBQUNEOztBQUNETixNQUFBQSxHQUFHLEdBQUdBLEdBQUcsQ0FBQ08sS0FBSixDQUFVLElBQVYsRUFBZ0JDLE1BQWhCLENBQXVCQyxJQUFJLElBQUksQ0FBQ0EsSUFBSSxDQUFDQyxVQUFMLENBQWdCSixXQUFoQixDQUFoQyxFQUE4RDFLLElBQTlELENBQW1FLElBQW5FLENBQU47QUFDRCxLQTlCcUUsQ0FnQ3RFOzs7QUFDQSxRQUFJbUssUUFBSixFQUFjO0FBQ1pyTCxNQUFBQSxJQUFJLENBQUM2QixJQUFMLENBQVUsb0JBQVY7QUFDRCxLQUZELE1BRU87QUFDTCxZQUFNb0ssVUFBVSxHQUFHLE1BQU0sS0FBS3ZDLFNBQUwsQ0FBZSxnQkFBZixDQUF6QjtBQUNBLFlBQU13QyxJQUFJLEdBQUlELFVBQVUsSUFBSUEsVUFBVSxLQUFLLFNBQTlCLEdBQTJDQSxVQUEzQyxHQUF3RCxPQUFyRTtBQUNBak0sTUFBQUEsSUFBSSxDQUFDNkIsSUFBTCxDQUFXLGFBQVlxSyxJQUFLLEVBQTVCO0FBQ0QsS0F2Q3FFLENBeUN0RTs7O0FBQ0EsUUFBSWQsU0FBUyxJQUFJQSxTQUFTLENBQUNoTSxNQUFWLEdBQW1CLENBQXBDLEVBQXVDO0FBQ3JDa00sTUFBQUEsR0FBRyxHQUFHLE1BQU0sS0FBS2EscUJBQUwsQ0FBMkJiLEdBQTNCLEVBQWdDRixTQUFoQyxDQUFaO0FBQ0Q7O0FBRURwTCxJQUFBQSxJQUFJLENBQUM2QixJQUFMLENBQVUsSUFBVixFQUFnQnlKLEdBQUcsQ0FBQzNKLElBQUosRUFBaEI7O0FBRUEsUUFBSXdKLEtBQUosRUFBVztBQUFFbkwsTUFBQUEsSUFBSSxDQUFDNkIsSUFBTCxDQUFVLFNBQVY7QUFBdUI7O0FBQ3BDLFFBQUlxSixVQUFKLEVBQWdCO0FBQUVsTCxNQUFBQSxJQUFJLENBQUM2QixJQUFMLENBQVUsZUFBVjtBQUE2Qjs7QUFDL0MsV0FBTyxLQUFLZ0csT0FBTCxDQUFhN0gsSUFBYixFQUFtQjtBQUFDTSxNQUFBQSxjQUFjLEVBQUU7QUFBakIsS0FBbkIsQ0FBUDtBQUNEOztBQUVENkwsRUFBQUEscUJBQXFCLENBQUNwTyxPQUFELEVBQVVxTixTQUFTLEdBQUcsRUFBdEIsRUFBMEI7QUFDN0MsVUFBTWdCLFFBQVEsR0FBR2hCLFNBQVMsQ0FBQzlCLEdBQVYsQ0FBYytDLE1BQU0sSUFBSTtBQUN2QyxhQUFPO0FBQ0xDLFFBQUFBLEtBQUssRUFBRSxnQkFERjtBQUVMQyxRQUFBQSxLQUFLLEVBQUcsR0FBRUYsTUFBTSxDQUFDRyxJQUFLLEtBQUlILE1BQU0sQ0FBQ0ksS0FBTTtBQUZsQyxPQUFQO0FBSUQsS0FMZ0IsQ0FBakIsQ0FENkMsQ0FRN0M7O0FBQ0EsVUFBTW5CLEdBQUcsR0FBSSxHQUFFdk4sT0FBTyxDQUFDNEQsSUFBUixFQUFlLElBQTlCO0FBRUEsV0FBT3lLLFFBQVEsQ0FBQ2hOLE1BQVQsR0FBa0IsS0FBS3NOLGFBQUwsQ0FBbUJwQixHQUFuQixFQUF3QmMsUUFBeEIsQ0FBbEIsR0FBc0RkLEdBQTdEO0FBQ0Q7QUFFRDs7Ozs7QUFHQSxRQUFNcUIsZUFBTixHQUF3QjtBQUN0QixVQUFNM00sSUFBSSxHQUFHLENBQUMsUUFBRCxFQUFXLGdCQUFYLEVBQTZCLFVBQTdCLEVBQXlDLHVCQUF6QyxFQUFrRSwyQkFBbEUsRUFBK0YsSUFBL0YsQ0FBYjtBQUNBLFVBQU1nSixNQUFNLEdBQUcsTUFBTSxLQUFLakosSUFBTCxDQUFVQyxJQUFWLENBQXJCOztBQUNBLFFBQUlnSixNQUFNLENBQUM1SixNQUFQLEdBQWdCM0Isd0JBQXBCLEVBQThDO0FBQzVDLFlBQU0sSUFBSVEsY0FBSixFQUFOO0FBQ0Q7O0FBRUQsVUFBTTJPLE9BQU8sR0FBRyxNQUFNLDBCQUFZNUQsTUFBWixDQUF0Qjs7QUFFQSxTQUFLLE1BQU02RCxTQUFYLElBQXdCRCxPQUF4QixFQUFpQztBQUMvQixVQUFJRSxLQUFLLENBQUNDLE9BQU4sQ0FBY0gsT0FBTyxDQUFDQyxTQUFELENBQXJCLENBQUosRUFBdUM7QUFDckMsYUFBS0csNkJBQUwsQ0FBbUNKLE9BQU8sQ0FBQ0MsU0FBRCxDQUExQztBQUNEO0FBQ0Y7O0FBRUQsV0FBT0QsT0FBUDtBQUNEOztBQUVESSxFQUFBQSw2QkFBNkIsQ0FBQ0MsT0FBRCxFQUFVO0FBQ3JDQSxJQUFBQSxPQUFPLENBQUNDLE9BQVIsQ0FBZ0JDLEtBQUssSUFBSTtBQUN2QjtBQUNBO0FBQ0E7QUFDQSxVQUFJQSxLQUFLLENBQUNDLFFBQVYsRUFBb0I7QUFDbEJELFFBQUFBLEtBQUssQ0FBQ0MsUUFBTixHQUFpQiw4QkFBZ0JELEtBQUssQ0FBQ0MsUUFBdEIsQ0FBakI7QUFDRDs7QUFDRCxVQUFJRCxLQUFLLENBQUNFLFlBQVYsRUFBd0I7QUFDdEJGLFFBQUFBLEtBQUssQ0FBQ0UsWUFBTixHQUFxQiw4QkFBZ0JGLEtBQUssQ0FBQ0UsWUFBdEIsQ0FBckI7QUFDRDtBQUNGLEtBVkQ7QUFXRDs7QUFFRCxRQUFNQyxjQUFOLENBQXFCMU8sT0FBTyxHQUFHLEVBQS9CLEVBQW1DO0FBQ2pDLFVBQU1vQixJQUFJLEdBQUcsQ0FBQyxNQUFELEVBQVMsZUFBVCxFQUEwQixjQUExQixDQUFiOztBQUNBLFFBQUlwQixPQUFPLENBQUMyTyxNQUFaLEVBQW9CO0FBQUV2TixNQUFBQSxJQUFJLENBQUM2QixJQUFMLENBQVUsVUFBVjtBQUF3Qjs7QUFDOUMsUUFBSWpELE9BQU8sQ0FBQzRPLE1BQVosRUFBb0I7QUFBRXhOLE1BQUFBLElBQUksQ0FBQzZCLElBQUwsQ0FBVWpELE9BQU8sQ0FBQzRPLE1BQWxCO0FBQTRCOztBQUNsRCxVQUFNeEUsTUFBTSxHQUFHLE1BQU0sS0FBS2pKLElBQUwsQ0FBVUMsSUFBVixDQUFyQjtBQUVBLFVBQU15TixTQUFTLEdBQUc7QUFDaEJDLE1BQUFBLENBQUMsRUFBRSxPQURhO0FBRWhCQyxNQUFBQSxDQUFDLEVBQUUsVUFGYTtBQUdoQkMsTUFBQUEsQ0FBQyxFQUFFLFNBSGE7QUFJaEJDLE1BQUFBLENBQUMsRUFBRTtBQUphLEtBQWxCO0FBT0EsVUFBTUMsWUFBWSxHQUFHLEVBQXJCO0FBQ0E5RSxJQUFBQSxNQUFNLElBQUlBLE1BQU0sQ0FBQ3JILElBQVAsR0FBY2tLLEtBQWQsQ0FBb0JrQywwQkFBcEIsRUFBdUNiLE9BQXZDLENBQStDbkIsSUFBSSxJQUFJO0FBQy9ELFlBQU0sQ0FBQ2lDLE1BQUQsRUFBU0MsV0FBVCxJQUF3QmxDLElBQUksQ0FBQ0YsS0FBTCxDQUFXLElBQVgsQ0FBOUI7QUFDQSxZQUFNdUIsUUFBUSxHQUFHLDhCQUFnQmEsV0FBaEIsQ0FBakI7QUFDQUgsTUFBQUEsWUFBWSxDQUFDVixRQUFELENBQVosR0FBeUJLLFNBQVMsQ0FBQ08sTUFBRCxDQUFsQztBQUNELEtBSlMsQ0FBVjs7QUFLQSxRQUFJLENBQUNwUCxPQUFPLENBQUMyTyxNQUFiLEVBQXFCO0FBQ25CLFlBQU1XLFNBQVMsR0FBRyxNQUFNLEtBQUtDLGlCQUFMLEVBQXhCO0FBQ0FELE1BQUFBLFNBQVMsQ0FBQ2hCLE9BQVYsQ0FBa0JFLFFBQVEsSUFBSTtBQUFFVSxRQUFBQSxZQUFZLENBQUNWLFFBQUQsQ0FBWixHQUF5QixPQUF6QjtBQUFtQyxPQUFuRTtBQUNEOztBQUNELFdBQU9VLFlBQVA7QUFDRDs7QUFFRCxRQUFNSyxpQkFBTixHQUEwQjtBQUN4QixVQUFNbkYsTUFBTSxHQUFHLE1BQU0sS0FBS2pKLElBQUwsQ0FBVSxDQUFDLFVBQUQsRUFBYSxVQUFiLEVBQXlCLG9CQUF6QixDQUFWLENBQXJCOztBQUNBLFFBQUlpSixNQUFNLENBQUNySCxJQUFQLE9BQWtCLEVBQXRCLEVBQTBCO0FBQUUsYUFBTyxFQUFQO0FBQVk7O0FBQ3hDLFdBQU9xSCxNQUFNLENBQUNySCxJQUFQLEdBQWNrSyxLQUFkLENBQW9Ca0MsMEJBQXBCLEVBQXVDekUsR0FBdkMsQ0FBMkM4RSx3QkFBM0MsQ0FBUDtBQUNEOztBQUVELFFBQU1DLG1CQUFOLENBQTBCakIsUUFBMUIsRUFBb0M7QUFBQ0csSUFBQUEsTUFBRDtBQUFTZSxJQUFBQTtBQUFULE1BQXVCLEVBQTNELEVBQStEO0FBQzdELFFBQUl0TyxJQUFJLEdBQUcsQ0FBQyxNQUFELEVBQVMsYUFBVCxFQUF3QixlQUF4QixFQUF5QyxjQUF6QyxFQUF5RCxpQkFBekQsQ0FBWDs7QUFDQSxRQUFJdU4sTUFBSixFQUFZO0FBQUV2TixNQUFBQSxJQUFJLENBQUM2QixJQUFMLENBQVUsVUFBVjtBQUF3Qjs7QUFDdEMsUUFBSXlNLFVBQUosRUFBZ0I7QUFBRXRPLE1BQUFBLElBQUksQ0FBQzZCLElBQUwsQ0FBVXlNLFVBQVY7QUFBd0I7O0FBQzFDdE8sSUFBQUEsSUFBSSxHQUFHQSxJQUFJLENBQUNxSixNQUFMLENBQVksQ0FBQyxJQUFELEVBQU8sMkJBQWErRCxRQUFiLENBQVAsQ0FBWixDQUFQO0FBQ0EsVUFBTXBFLE1BQU0sR0FBRyxNQUFNLEtBQUtqSixJQUFMLENBQVVDLElBQVYsQ0FBckI7QUFFQSxRQUFJdU8sUUFBUSxHQUFHLEVBQWY7O0FBQ0EsUUFBSXZGLE1BQUosRUFBWTtBQUNWdUYsTUFBQUEsUUFBUSxHQUFHLHdCQUFVdkYsTUFBVixFQUNSOEMsTUFEUSxDQUNEMEMsT0FBTyxJQUFJQSxPQUFPLENBQUNSLE1BQVIsS0FBbUIsVUFEN0IsQ0FBWDs7QUFHQSxXQUFLLElBQUlTLENBQUMsR0FBRyxDQUFiLEVBQWdCQSxDQUFDLEdBQUdGLFFBQVEsQ0FBQ25QLE1BQTdCLEVBQXFDcVAsQ0FBQyxFQUF0QyxFQUEwQztBQUN4QyxjQUFNRCxPQUFPLEdBQUdELFFBQVEsQ0FBQ0UsQ0FBRCxDQUF4Qjs7QUFDQSxZQUFJRCxPQUFPLENBQUNFLE9BQVosRUFBcUI7QUFDbkJGLFVBQUFBLE9BQU8sQ0FBQ0UsT0FBUixHQUFrQiw4QkFBZ0JGLE9BQU8sQ0FBQ0UsT0FBeEIsQ0FBbEI7QUFDRDs7QUFDRCxZQUFJRixPQUFPLENBQUNHLE9BQVosRUFBcUI7QUFDbkJILFVBQUFBLE9BQU8sQ0FBQ0csT0FBUixHQUFrQiw4QkFBZ0JILE9BQU8sQ0FBQ0csT0FBeEIsQ0FBbEI7QUFDRDtBQUNGO0FBQ0Y7O0FBRUQsUUFBSSxDQUFDcEIsTUFBRCxJQUFXLENBQUMsTUFBTSxLQUFLWSxpQkFBTCxFQUFQLEVBQWlDeEcsUUFBakMsQ0FBMEN5RixRQUExQyxDQUFmLEVBQW9FO0FBQ2xFO0FBQ0EsWUFBTXdCLE9BQU8sR0FBR3pNLGNBQUtqQixJQUFMLENBQVUsS0FBS3ZDLFVBQWYsRUFBMkJ5TyxRQUEzQixDQUFoQjs7QUFDQSxZQUFNeUIsVUFBVSxHQUFHLE1BQU0sK0JBQWlCRCxPQUFqQixDQUF6QjtBQUNBLFlBQU1FLE9BQU8sR0FBRyxNQUFNLDRCQUFjRixPQUFkLENBQXRCO0FBQ0EsWUFBTUcsUUFBUSxHQUFHLE1BQU1qRyxpQkFBR21CLFFBQUgsQ0FBWTJFLE9BQVosRUFBcUI7QUFBQzFFLFFBQUFBLFFBQVEsRUFBRTtBQUFYLE9BQXJCLENBQXZCO0FBQ0EsWUFBTThFLE1BQU0sR0FBRyx1QkFBU0QsUUFBVCxDQUFmO0FBQ0EsVUFBSTdDLElBQUo7QUFDQSxVQUFJK0MsUUFBSjs7QUFDQSxVQUFJSixVQUFKLEVBQWdCO0FBQ2QzQyxRQUFBQSxJQUFJLEdBQUdnRCxjQUFLQyxLQUFMLENBQVdDLFVBQWxCO0FBQ0QsT0FGRCxNQUVPLElBQUlOLE9BQUosRUFBYTtBQUNsQjVDLFFBQUFBLElBQUksR0FBR2dELGNBQUtDLEtBQUwsQ0FBV0UsT0FBbEI7QUFDQUosUUFBQUEsUUFBUSxHQUFHLE1BQU1uRyxpQkFBR21HLFFBQUgsQ0FBWUwsT0FBWixDQUFqQjtBQUNELE9BSE0sTUFHQTtBQUNMMUMsUUFBQUEsSUFBSSxHQUFHZ0QsY0FBS0MsS0FBTCxDQUFXRyxNQUFsQjtBQUNEOztBQUVEZixNQUFBQSxRQUFRLENBQUMxTSxJQUFULENBQWMwTixtQkFBbUIsQ0FBQ25DLFFBQUQsRUFBVzRCLE1BQU0sR0FBRyxJQUFILEdBQVVELFFBQTNCLEVBQXFDN0MsSUFBckMsRUFBMkMrQyxRQUEzQyxDQUFqQztBQUNEOztBQUNELFFBQUlWLFFBQVEsQ0FBQ25QLE1BQVQsR0FBa0IsQ0FBdEIsRUFBeUI7QUFDdkIsWUFBTSxJQUFJdkIsS0FBSixDQUFXLHNDQUFxQ3VQLFFBQVMsWUFBV21CLFFBQVEsQ0FBQ25QLE1BQU8sRUFBcEYsQ0FBTjtBQUNEOztBQUNELFdBQU9tUCxRQUFQO0FBQ0Q7O0FBRUQsUUFBTWlCLHFCQUFOLEdBQThCO0FBQzVCLFVBQU14RyxNQUFNLEdBQUcsTUFBTSxLQUFLakosSUFBTCxDQUFVLENBQzdCLE1BRDZCLEVBQ3JCLFVBRHFCLEVBQ1QsYUFEUyxFQUNNLGVBRE4sRUFDdUIsY0FEdkIsRUFDdUMsaUJBRHZDLENBQVYsQ0FBckI7O0FBSUEsUUFBSSxDQUFDaUosTUFBTCxFQUFhO0FBQ1gsYUFBTyxFQUFQO0FBQ0Q7O0FBRUQsVUFBTXlHLEtBQUssR0FBRyx3QkFBVXpHLE1BQVYsQ0FBZDs7QUFDQSxTQUFLLE1BQU0wRyxJQUFYLElBQW1CRCxLQUFuQixFQUEwQjtBQUN4QixVQUFJQyxJQUFJLENBQUNoQixPQUFULEVBQWtCO0FBQUVnQixRQUFBQSxJQUFJLENBQUNoQixPQUFMLEdBQWUsOEJBQWdCZ0IsSUFBSSxDQUFDaEIsT0FBckIsQ0FBZjtBQUErQzs7QUFDbkUsVUFBSWdCLElBQUksQ0FBQ2YsT0FBVCxFQUFrQjtBQUFFZSxRQUFBQSxJQUFJLENBQUNmLE9BQUwsR0FBZSw4QkFBZ0JlLElBQUksQ0FBQ2YsT0FBckIsQ0FBZjtBQUErQztBQUNwRTs7QUFDRCxXQUFPYyxLQUFQO0FBQ0Q7QUFFRDs7Ozs7QUFHQSxRQUFNRSxTQUFOLENBQWdCQyxHQUFoQixFQUFxQjtBQUNuQixVQUFNLENBQUN4RixNQUFELElBQVcsTUFBTSxLQUFLeUYsVUFBTCxDQUFnQjtBQUFDNVEsTUFBQUEsR0FBRyxFQUFFLENBQU47QUFBUzJRLE1BQUFBLEdBQVQ7QUFBY0UsTUFBQUEsYUFBYSxFQUFFO0FBQTdCLEtBQWhCLENBQXZCO0FBQ0EsV0FBTzFGLE1BQVA7QUFDRDs7QUFFRCxRQUFNc0IsYUFBTixHQUFzQjtBQUNwQixVQUFNLENBQUNxRSxVQUFELElBQWUsTUFBTSxLQUFLRixVQUFMLENBQWdCO0FBQUM1USxNQUFBQSxHQUFHLEVBQUUsQ0FBTjtBQUFTMlEsTUFBQUEsR0FBRyxFQUFFLE1BQWQ7QUFBc0JFLE1BQUFBLGFBQWEsRUFBRTtBQUFyQyxLQUFoQixDQUEzQjtBQUNBLFdBQU9DLFVBQVA7QUFDRDs7QUFFRCxRQUFNRixVQUFOLENBQWlCalIsT0FBTyxHQUFHLEVBQTNCLEVBQStCO0FBQzdCLFVBQU07QUFBQ0ssTUFBQUEsR0FBRDtBQUFNMlEsTUFBQUEsR0FBTjtBQUFXRSxNQUFBQSxhQUFYO0FBQTBCRSxNQUFBQTtBQUExQjtBQUNKL1EsTUFBQUEsR0FBRyxFQUFFLENBREQ7QUFFSjJRLE1BQUFBLEdBQUcsRUFBRSxNQUZEO0FBR0pFLE1BQUFBLGFBQWEsRUFBRSxLQUhYO0FBSUpFLE1BQUFBLFlBQVksRUFBRTtBQUpWLE9BS0RwUixPQUxDLENBQU4sQ0FENkIsQ0FTN0I7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTs7O0FBQ0EsVUFBTW9CLElBQUksR0FBRyxDQUNYLEtBRFcsRUFFWCx5REFGVyxFQUdYLG9CQUhXLEVBSVgsYUFKVyxFQUtYLGVBTFcsRUFNWCxjQU5XLEVBT1gsSUFQVyxFQVFYLElBUlcsRUFTWGYsR0FUVyxFQVVYMlEsR0FWVyxDQUFiOztBQWFBLFFBQUlJLFlBQUosRUFBa0I7QUFDaEJoUSxNQUFBQSxJQUFJLENBQUM2QixJQUFMLENBQVUsU0FBVixFQUFxQixJQUFyQixFQUEyQixnQkFBM0I7QUFDRDs7QUFFRCxVQUFNbUgsTUFBTSxHQUFHLE1BQU0sS0FBS2pKLElBQUwsQ0FBVUMsSUFBSSxDQUFDcUosTUFBTCxDQUFZLElBQVosQ0FBVixFQUE2QnJELEtBQTdCLENBQW1DTCxHQUFHLElBQUk7QUFDN0QsVUFBSSxtQkFBbUJxQyxJQUFuQixDQUF3QnJDLEdBQUcsQ0FBQzZCLE1BQTVCLEtBQXVDLHNCQUFzQlEsSUFBdEIsQ0FBMkJyQyxHQUFHLENBQUM2QixNQUEvQixDQUEzQyxFQUFtRjtBQUNqRixlQUFPLEVBQVA7QUFDRCxPQUZELE1BRU87QUFDTCxjQUFNN0IsR0FBTjtBQUNEO0FBQ0YsS0FOb0IsQ0FBckI7O0FBUUEsUUFBSXFELE1BQU0sS0FBSyxFQUFmLEVBQW1CO0FBQ2pCLGFBQU84RyxhQUFhLEdBQUcsQ0FBQztBQUFDRyxRQUFBQSxHQUFHLEVBQUUsRUFBTjtBQUFVbFMsUUFBQUEsT0FBTyxFQUFFLEVBQW5CO0FBQXVCd04sUUFBQUEsU0FBUyxFQUFFO0FBQWxDLE9BQUQsQ0FBSCxHQUErQyxFQUFuRTtBQUNEOztBQUVELFVBQU0yRSxNQUFNLEdBQUdsSCxNQUFNLENBQUNySCxJQUFQLEdBQWNrSyxLQUFkLENBQW9CLElBQXBCLENBQWY7QUFFQSxVQUFNc0UsT0FBTyxHQUFHLEVBQWhCOztBQUNBLFNBQUssSUFBSTFCLENBQUMsR0FBRyxDQUFiLEVBQWdCQSxDQUFDLEdBQUd5QixNQUFNLENBQUM5USxNQUEzQixFQUFtQ3FQLENBQUMsSUFBSSxDQUF4QyxFQUEyQztBQUN6QyxZQUFNMkIsSUFBSSxHQUFHRixNQUFNLENBQUN6QixDQUFDLEdBQUcsQ0FBTCxDQUFOLENBQWM5TSxJQUFkLEVBQWI7QUFDQSxVQUFJb0osS0FBSyxHQUFHLEVBQVo7O0FBQ0EsVUFBSWlGLFlBQUosRUFBa0I7QUFDaEIsY0FBTVAsS0FBSyxHQUFHUyxNQUFNLENBQUN6QixDQUFDLEdBQUcsQ0FBTCxDQUFwQjtBQUNBMUQsUUFBQUEsS0FBSyxHQUFHLHdCQUFVMEUsS0FBSyxDQUFDOU4sSUFBTixFQUFWLENBQVI7QUFDRDs7QUFFRCxZQUFNO0FBQUM1RCxRQUFBQSxPQUFPLEVBQUV5TixXQUFWO0FBQXVCSixRQUFBQTtBQUF2QixVQUFvQyxrREFBb0NnRixJQUFwQyxDQUExQztBQUVBRCxNQUFBQSxPQUFPLENBQUN0TyxJQUFSLENBQWE7QUFDWG9PLFFBQUFBLEdBQUcsRUFBRUMsTUFBTSxDQUFDekIsQ0FBRCxDQUFOLElBQWF5QixNQUFNLENBQUN6QixDQUFELENBQU4sQ0FBVTlNLElBQVYsRUFEUDtBQUVYMEssUUFBQUEsTUFBTSxFQUFFLElBQUlnRSxlQUFKLENBQVdILE1BQU0sQ0FBQ3pCLENBQUMsR0FBRyxDQUFMLENBQU4sSUFBaUJ5QixNQUFNLENBQUN6QixDQUFDLEdBQUcsQ0FBTCxDQUFOLENBQWM5TSxJQUFkLEVBQTVCLEVBQWtEdU8sTUFBTSxDQUFDekIsQ0FBQyxHQUFHLENBQUwsQ0FBTixJQUFpQnlCLE1BQU0sQ0FBQ3pCLENBQUMsR0FBRyxDQUFMLENBQU4sQ0FBYzlNLElBQWQsRUFBbkUsQ0FGRztBQUdYMk8sUUFBQUEsVUFBVSxFQUFFQyxRQUFRLENBQUNMLE1BQU0sQ0FBQ3pCLENBQUMsR0FBRyxDQUFMLENBQVAsRUFBZ0IsRUFBaEIsQ0FIVDtBQUlYaEQsUUFBQUEsY0FBYyxFQUFFeUUsTUFBTSxDQUFDekIsQ0FBQyxHQUFHLENBQUwsQ0FKWDtBQUtYakQsUUFBQUEsV0FMVztBQU1YSixRQUFBQSxTQU5XO0FBT1hHLFFBQUFBLFNBQVMsRUFBRSxLQVBBO0FBUVhSLFFBQUFBO0FBUlcsT0FBYjtBQVVEOztBQUNELFdBQU9vRixPQUFQO0FBQ0Q7O0FBRUQsUUFBTUssVUFBTixDQUFpQjVSLE9BQU8sR0FBRyxFQUEzQixFQUErQjtBQUM3QixVQUFNO0FBQUNLLE1BQUFBLEdBQUQ7QUFBTTJRLE1BQUFBO0FBQU47QUFBYzNRLE1BQUFBLEdBQUcsRUFBRSxDQUFuQjtBQUFzQjJRLE1BQUFBLEdBQUcsRUFBRTtBQUEzQixPQUFzQ2hSLE9BQXRDLENBQU4sQ0FENkIsQ0FHN0I7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBOzs7QUFFQSxVQUFNd0QsU0FBUyxHQUFHLElBQWxCO0FBQ0EsVUFBTXFPLGVBQWUsR0FBR0MsTUFBTSxDQUFDQyxZQUFQLENBQW9CSixRQUFRLENBQUNuTyxTQUFELEVBQVksRUFBWixDQUE1QixDQUF4QjtBQUNBLFVBQU04TixNQUFNLEdBQUcsQ0FBQyxLQUFELEVBQVEsS0FBUixFQUFlLEtBQWYsRUFBc0IsS0FBdEIsRUFBNkIseUJBQTdCLENBQWY7QUFDQSxVQUFNVSxNQUFNLEdBQUdWLE1BQU0sQ0FBQ2hQLElBQVAsQ0FBYSxLQUFJa0IsU0FBVSxFQUEzQixDQUFmOztBQUVBLFFBQUk7QUFDRixZQUFNNEcsTUFBTSxHQUFHLE1BQU0sS0FBS2pKLElBQUwsQ0FBVSxDQUM3QixLQUQ2QixFQUNyQixZQUFXNlEsTUFBTyxFQURHLEVBQ0EsSUFEQSxFQUNNLElBRE4sRUFDWTNSLEdBRFosRUFDaUIyUSxHQURqQixFQUNzQixJQUR0QixDQUFWLENBQXJCO0FBSUEsYUFBTzVHLE1BQU0sQ0FBQzZDLEtBQVAsQ0FBYSxJQUFiLEVBQ0p6TixNQURJLENBQ0csQ0FBQ0MsR0FBRCxFQUFNME4sSUFBTixLQUFlO0FBQ3JCLFlBQUlBLElBQUksQ0FBQzNNLE1BQUwsS0FBZ0IsQ0FBcEIsRUFBdUI7QUFBRSxpQkFBT2YsR0FBUDtBQUFhOztBQUV0QyxjQUFNLENBQUN3UyxFQUFELEVBQUtDLEVBQUwsRUFBU0MsRUFBVCxFQUFhQyxFQUFiLEVBQWlCNUUsUUFBakIsSUFBNkJMLElBQUksQ0FBQ0YsS0FBTCxDQUFXNEUsZUFBWCxDQUFuQztBQUNBckUsUUFBQUEsUUFBUSxDQUNMUCxLQURILENBQ1MsSUFEVCxFQUVHdkMsR0FGSCxDQUVPMkgsT0FBTyxJQUFJQSxPQUFPLENBQUNDLEtBQVIsQ0FBY0Msd0JBQWQsQ0FGbEIsRUFHR3JGLE1BSEgsQ0FHVW9GLEtBQUssSUFBSUEsS0FBSyxLQUFLLElBSDdCLEVBSUdoRSxPQUpILENBSVcsQ0FBQyxDQUFDckQsQ0FBRCxFQUFJMkMsSUFBSixFQUFVQyxLQUFWLENBQUQsS0FBc0I7QUFBRXBPLFVBQUFBLEdBQUcsQ0FBQ29PLEtBQUQsQ0FBSCxHQUFhRCxJQUFiO0FBQW9CLFNBSnZEO0FBTUFuTyxRQUFBQSxHQUFHLENBQUN5UyxFQUFELENBQUgsR0FBVUQsRUFBVjtBQUNBeFMsUUFBQUEsR0FBRyxDQUFDMlMsRUFBRCxDQUFILEdBQVVELEVBQVY7QUFFQSxlQUFPMVMsR0FBUDtBQUNELE9BZkksRUFlRixFQWZFLENBQVA7QUFnQkQsS0FyQkQsQ0FxQkUsT0FBT3NILEdBQVAsRUFBWTtBQUNaLFVBQUksbUJBQW1CcUMsSUFBbkIsQ0FBd0JyQyxHQUFHLENBQUM2QixNQUE1QixLQUF1QyxzQkFBc0JRLElBQXRCLENBQTJCckMsR0FBRyxDQUFDNkIsTUFBL0IsQ0FBM0MsRUFBbUY7QUFDakYsZUFBTyxFQUFQO0FBQ0QsT0FGRCxNQUVPO0FBQ0wsY0FBTTdCLEdBQU47QUFDRDtBQUNGO0FBQ0Y7O0FBRUQrRyxFQUFBQSxhQUFhLENBQUMwRSxhQUFELEVBQWdCaEYsUUFBaEIsRUFBMEI7QUFDckMsVUFBTXBNLElBQUksR0FBRyxDQUFDLG9CQUFELENBQWI7O0FBQ0EsU0FBSyxNQUFNaVIsT0FBWCxJQUFzQjdFLFFBQXRCLEVBQWdDO0FBQzlCcE0sTUFBQUEsSUFBSSxDQUFDNkIsSUFBTCxDQUFVLFdBQVYsRUFBd0IsR0FBRW9QLE9BQU8sQ0FBQzNFLEtBQU0sSUFBRzJFLE9BQU8sQ0FBQzFFLEtBQU0sRUFBekQ7QUFDRDs7QUFDRCxXQUFPLEtBQUt4TSxJQUFMLENBQVVDLElBQVYsRUFBZ0I7QUFBQ0UsTUFBQUEsS0FBSyxFQUFFa1I7QUFBUixLQUFoQixDQUFQO0FBQ0Q7O0FBRURDLEVBQUFBLGlCQUFpQixDQUFDakUsUUFBRCxFQUFXO0FBQzFCLFdBQU8sS0FBS3JOLElBQUwsQ0FBVSxDQUFDLE1BQUQsRUFBVSxJQUFHLDJCQUFhcU4sUUFBYixDQUF1QixFQUFwQyxDQUFWLENBQVA7QUFDRDtBQUVEOzs7OztBQUdBa0UsRUFBQUEsS0FBSyxDQUFDQyxVQUFELEVBQWE7QUFDaEIsV0FBTyxLQUFLMUosT0FBTCxDQUFhLENBQUMsT0FBRCxFQUFVMEosVUFBVixDQUFiLEVBQW9DO0FBQUNqUixNQUFBQSxjQUFjLEVBQUU7QUFBakIsS0FBcEMsQ0FBUDtBQUNEOztBQUVEa1IsRUFBQUEsU0FBUyxDQUFDdkksU0FBRCxFQUFZO0FBQ25CLFdBQU8seUJBQVc5RyxjQUFLakIsSUFBTCxDQUFVK0gsU0FBVixFQUFxQixZQUFyQixDQUFYLEVBQStDakQsS0FBL0MsQ0FBcUQsTUFBTSxLQUEzRCxDQUFQO0FBQ0Q7O0FBRUR5TCxFQUFBQSxVQUFVLEdBQUc7QUFDWCxXQUFPLEtBQUsxUixJQUFMLENBQVUsQ0FBQyxPQUFELEVBQVUsU0FBVixDQUFWLEVBQWdDO0FBQUNPLE1BQUFBLGNBQWMsRUFBRTtBQUFqQixLQUFoQyxDQUFQO0FBQ0Q7O0FBRURvUixFQUFBQSxZQUFZLENBQUNDLElBQUQsRUFBT3ZJLEtBQVAsRUFBYztBQUN4QixRQUFJQSxLQUFLLENBQUNoSyxNQUFOLEtBQWlCLENBQXJCLEVBQXdCO0FBQ3RCLGFBQU9JLE9BQU8sQ0FBQytCLE9BQVIsRUFBUDtBQUNEOztBQUVELFdBQU8sS0FBS3hCLElBQUwsQ0FBVSxDQUFDLFVBQUQsRUFBYyxLQUFJNFIsSUFBSyxFQUF2QixFQUEwQixHQUFHdkksS0FBSyxDQUFDRSxHQUFOLENBQVVDLHFCQUFWLENBQTdCLENBQVYsQ0FBUDtBQUNEO0FBRUQ7Ozs7O0FBR0EsUUFBTXFJLFVBQU4sQ0FBaUIzSSxTQUFqQixFQUE0QjtBQUMxQixVQUFNMkQsT0FBTyxHQUFHLE1BQU1wTixPQUFPLENBQUNxUyxHQUFSLENBQVksQ0FDaEMseUJBQVcxUCxjQUFLakIsSUFBTCxDQUFVK0gsU0FBVixFQUFxQixjQUFyQixDQUFYLENBRGdDLEVBRWhDLHlCQUFXOUcsY0FBS2pCLElBQUwsQ0FBVStILFNBQVYsRUFBcUIsY0FBckIsQ0FBWCxDQUZnQyxDQUFaLENBQXRCO0FBSUEsV0FBTzJELE9BQU8sQ0FBQ2tGLElBQVIsQ0FBYUMsQ0FBQyxJQUFJQSxDQUFsQixDQUFQO0FBQ0Q7QUFFRDs7Ozs7QUFHQUMsRUFBQUEsS0FBSyxDQUFDQyxTQUFELEVBQVlyVCxPQUFPLEdBQUcsRUFBdEIsRUFBMEI7QUFDN0IsVUFBTW9CLElBQUksR0FBRyxDQUFDLE9BQUQsQ0FBYjs7QUFDQSxRQUFJcEIsT0FBTyxDQUFDc1QsT0FBWixFQUFxQjtBQUFFbFMsTUFBQUEsSUFBSSxDQUFDNkIsSUFBTCxDQUFVLFlBQVY7QUFBMEI7O0FBQ2pELFFBQUlqRCxPQUFPLENBQUN1VCxJQUFaLEVBQWtCO0FBQUVuUyxNQUFBQSxJQUFJLENBQUM2QixJQUFMLENBQVUsUUFBVjtBQUFzQjs7QUFDMUMsUUFBSWpELE9BQU8sQ0FBQ3dULFNBQVosRUFBdUI7QUFBRXBTLE1BQUFBLElBQUksQ0FBQzZCLElBQUwsQ0FBVSxhQUFWO0FBQTJCOztBQUNwRCxRQUFJakQsT0FBTyxDQUFDeVQsZ0JBQVosRUFBOEI7QUFBRXJTLE1BQUFBLElBQUksQ0FBQzZCLElBQUwsQ0FBVSxVQUFWLEVBQXNCakQsT0FBTyxDQUFDMFQsVUFBOUI7QUFBNEM7O0FBQzVFdFMsSUFBQUEsSUFBSSxDQUFDNkIsSUFBTCxDQUFVb1EsU0FBVixFQUFxQixLQUFLdFQsVUFBMUI7QUFFQSxXQUFPLEtBQUtvQixJQUFMLENBQVVDLElBQVYsRUFBZ0I7QUFBQ0csTUFBQUEsa0JBQWtCLEVBQUUsSUFBckI7QUFBMkJHLE1BQUFBLGNBQWMsRUFBRTtBQUEzQyxLQUFoQixDQUFQO0FBQ0Q7O0FBRURpUyxFQUFBQSxLQUFLLENBQUNELFVBQUQsRUFBYWYsVUFBYixFQUF5QjtBQUM1QixXQUFPLEtBQUt4UixJQUFMLENBQVUsQ0FBQyxPQUFELEVBQVV1UyxVQUFWLEVBQXNCZixVQUF0QixDQUFWLEVBQTZDO0FBQUNwUixNQUFBQSxrQkFBa0IsRUFBRSxJQUFyQjtBQUEyQkcsTUFBQUEsY0FBYyxFQUFFO0FBQTNDLEtBQTdDLENBQVA7QUFDRDs7QUFFRGtTLEVBQUFBLElBQUksQ0FBQ0YsVUFBRCxFQUFhZixVQUFiLEVBQXlCM1MsT0FBTyxHQUFHLEVBQW5DLEVBQXVDO0FBQ3pDLFVBQU1vQixJQUFJLEdBQUcsQ0FBQyxNQUFELEVBQVNzUyxVQUFULEVBQXFCMVQsT0FBTyxDQUFDNlQsT0FBUixJQUFtQmxCLFVBQXhDLENBQWI7O0FBQ0EsUUFBSTNTLE9BQU8sQ0FBQzhULE1BQVosRUFBb0I7QUFDbEIxUyxNQUFBQSxJQUFJLENBQUM2QixJQUFMLENBQVUsV0FBVjtBQUNEOztBQUNELFdBQU8sS0FBS2dHLE9BQUwsQ0FBYTdILElBQWIsRUFBbUI7QUFBQ0csTUFBQUEsa0JBQWtCLEVBQUUsSUFBckI7QUFBMkJHLE1BQUFBLGNBQWMsRUFBRTtBQUEzQyxLQUFuQixDQUFQO0FBQ0Q7O0FBRUR1QixFQUFBQSxJQUFJLENBQUN5USxVQUFELEVBQWFmLFVBQWIsRUFBeUIzUyxPQUFPLEdBQUcsRUFBbkMsRUFBdUM7QUFDekMsVUFBTW9CLElBQUksR0FBRyxDQUFDLE1BQUQsRUFBU3NTLFVBQVUsSUFBSSxRQUF2QixFQUFpQzFULE9BQU8sQ0FBQzZULE9BQVIsSUFBb0IsY0FBYWxCLFVBQVcsRUFBN0UsQ0FBYjs7QUFDQSxRQUFJM1MsT0FBTyxDQUFDK1QsV0FBWixFQUF5QjtBQUFFM1MsTUFBQUEsSUFBSSxDQUFDNkIsSUFBTCxDQUFVLGdCQUFWO0FBQThCOztBQUN6RCxRQUFJakQsT0FBTyxDQUFDZ1UsS0FBWixFQUFtQjtBQUFFNVMsTUFBQUEsSUFBSSxDQUFDNkIsSUFBTCxDQUFVLFNBQVY7QUFBdUI7O0FBQzVDLFdBQU8sS0FBSzlCLElBQUwsQ0FBVUMsSUFBVixFQUFnQjtBQUFDRyxNQUFBQSxrQkFBa0IsRUFBRSxJQUFyQjtBQUEyQkcsTUFBQUEsY0FBYyxFQUFFO0FBQTNDLEtBQWhCLENBQVA7QUFDRDtBQUVEOzs7OztBQUdBdVMsRUFBQUEsS0FBSyxDQUFDdlUsSUFBRCxFQUFPd1UsUUFBUSxHQUFHLE1BQWxCLEVBQTBCO0FBQzdCLFVBQU1DLFVBQVUsR0FBRyxDQUFDLE1BQUQsQ0FBbkI7O0FBQ0EsUUFBSSxDQUFDQSxVQUFVLENBQUNwTCxRQUFYLENBQW9CckosSUFBcEIsQ0FBTCxFQUFnQztBQUM5QixZQUFNLElBQUlULEtBQUosQ0FBVyxnQkFBZVMsSUFBSyxxQkFBb0J5VSxVQUFVLENBQUM3UixJQUFYLENBQWdCLElBQWhCLENBQXNCLEVBQXpFLENBQU47QUFDRDs7QUFDRCxXQUFPLEtBQUtuQixJQUFMLENBQVUsQ0FBQyxPQUFELEVBQVcsS0FBSXpCLElBQUssRUFBcEIsRUFBdUJ3VSxRQUF2QixDQUFWLENBQVA7QUFDRDs7QUFFREUsRUFBQUEsU0FBUyxDQUFDcEQsR0FBRCxFQUFNO0FBQ2IsV0FBTyxLQUFLN1AsSUFBTCxDQUFVLENBQUMsWUFBRCxFQUFlLElBQWYsRUFBcUI2UCxHQUFyQixDQUFWLENBQVA7QUFDRDtBQUVEOzs7OztBQUdBcUQsRUFBQUEsUUFBUSxDQUFDMUIsVUFBRCxFQUFhM1MsT0FBTyxHQUFHLEVBQXZCLEVBQTJCO0FBQ2pDLFVBQU1vQixJQUFJLEdBQUcsQ0FBQyxVQUFELENBQWI7O0FBQ0EsUUFBSXBCLE9BQU8sQ0FBQ3NVLFNBQVosRUFBdUI7QUFDckJsVCxNQUFBQSxJQUFJLENBQUM2QixJQUFMLENBQVUsSUFBVjtBQUNEOztBQUNEN0IsSUFBQUEsSUFBSSxDQUFDNkIsSUFBTCxDQUFVMFAsVUFBVjs7QUFDQSxRQUFJM1MsT0FBTyxDQUFDdVUsVUFBWixFQUF3QjtBQUN0QixVQUFJdlUsT0FBTyxDQUFDd1UsS0FBWixFQUFtQjtBQUFFcFQsUUFBQUEsSUFBSSxDQUFDNkIsSUFBTCxDQUFVLFNBQVY7QUFBdUI7O0FBQzVDN0IsTUFBQUEsSUFBSSxDQUFDNkIsSUFBTCxDQUFVakQsT0FBTyxDQUFDdVUsVUFBbEI7QUFDRDs7QUFFRCxXQUFPLEtBQUtwVCxJQUFMLENBQVVDLElBQVYsRUFBZ0I7QUFBQ00sTUFBQUEsY0FBYyxFQUFFO0FBQWpCLEtBQWhCLENBQVA7QUFDRDs7QUFFRCxRQUFNK1MsV0FBTixHQUFvQjtBQUNsQixVQUFNekMsTUFBTSxHQUFHLENBQ2IsZUFEYSxFQUNJLFNBREosRUFDZSxrQkFEZixFQUViLGFBRmEsRUFFRSx3QkFGRixFQUU0Qix1QkFGNUIsRUFHYixTQUhhLEVBR0Ysb0JBSEUsRUFHb0IsbUJBSHBCLEVBSWIxUCxJQUphLENBSVIsS0FKUSxDQUFmO0FBTUEsVUFBTThILE1BQU0sR0FBRyxNQUFNLEtBQUtqSixJQUFMLENBQVUsQ0FBQyxjQUFELEVBQWtCLFlBQVc2USxNQUFPLEVBQXBDLEVBQXVDLGVBQXZDLENBQVYsQ0FBckI7QUFDQSxXQUFPNUgsTUFBTSxDQUFDckgsSUFBUCxHQUFja0ssS0FBZCxDQUFvQmtDLDBCQUFwQixFQUF1Q3pFLEdBQXZDLENBQTJDeUMsSUFBSSxJQUFJO0FBQ3hELFlBQU0sQ0FDSmtFLEdBREksRUFDQ3FELElBREQsRUFDTzlHLElBRFAsRUFFSitHLG1CQUZJLEVBRWlCQyxrQkFGakIsRUFFcUNDLGlCQUZyQyxFQUdKQyxlQUhJLEVBR2FDLGNBSGIsRUFHNkJDLGFBSDdCLElBSUY3SCxJQUFJLENBQUNGLEtBQUwsQ0FBVyxJQUFYLENBSko7QUFNQSxZQUFNZ0ksTUFBTSxHQUFHO0FBQUNySCxRQUFBQSxJQUFEO0FBQU95RCxRQUFBQSxHQUFQO0FBQVlxRCxRQUFBQSxJQUFJLEVBQUVBLElBQUksS0FBSztBQUEzQixPQUFmOztBQUNBLFVBQUlDLG1CQUFtQixJQUFJQyxrQkFBdkIsSUFBNkNDLGlCQUFqRCxFQUFvRTtBQUNsRUksUUFBQUEsTUFBTSxDQUFDQyxRQUFQLEdBQWtCO0FBQ2hCQyxVQUFBQSxXQUFXLEVBQUVSLG1CQURHO0FBRWhCakIsVUFBQUEsVUFBVSxFQUFFa0Isa0JBRkk7QUFHaEJRLFVBQUFBLFNBQVMsRUFBRVA7QUFISyxTQUFsQjtBQUtEOztBQUNELFVBQUlJLE1BQU0sQ0FBQ0MsUUFBUCxJQUFtQkosZUFBbkIsSUFBc0NDLGNBQXRDLElBQXdEQyxhQUE1RCxFQUEyRTtBQUN6RUMsUUFBQUEsTUFBTSxDQUFDaFMsSUFBUCxHQUFjO0FBQ1prUyxVQUFBQSxXQUFXLEVBQUVMLGVBREQ7QUFFWnBCLFVBQUFBLFVBQVUsRUFBRXFCLGNBQWMsSUFBS0UsTUFBTSxDQUFDQyxRQUFQLElBQW1CRCxNQUFNLENBQUNDLFFBQVAsQ0FBZ0J4QixVQUZ0RDtBQUdaMEIsVUFBQUEsU0FBUyxFQUFFSixhQUFhLElBQUtDLE1BQU0sQ0FBQ0MsUUFBUCxJQUFtQkQsTUFBTSxDQUFDQyxRQUFQLENBQWdCRTtBQUhwRCxTQUFkO0FBS0Q7O0FBQ0QsYUFBT0gsTUFBUDtBQUNELEtBdkJNLENBQVA7QUF3QkQ7O0FBRUQsUUFBTUkscUJBQU4sQ0FBNEJoRSxHQUE1QixFQUFpQ2lFLE1BQU0sR0FBRyxFQUExQyxFQUE4QztBQUM1QyxVQUFNbFUsSUFBSSxHQUFHLENBQUMsUUFBRCxFQUFXLHFCQUFYLEVBQWtDLFlBQWxDLEVBQWdEaVEsR0FBaEQsQ0FBYjs7QUFDQSxRQUFJaUUsTUFBTSxDQUFDQyxTQUFQLElBQW9CRCxNQUFNLENBQUNFLFVBQS9CLEVBQTJDO0FBQ3pDcFUsTUFBQUEsSUFBSSxDQUFDZ0wsTUFBTCxDQUFZLENBQVosRUFBZSxDQUFmLEVBQWtCLE9BQWxCO0FBQ0QsS0FGRCxNQUVPLElBQUlrSixNQUFNLENBQUNFLFVBQVgsRUFBdUI7QUFDNUJwVSxNQUFBQSxJQUFJLENBQUNnTCxNQUFMLENBQVksQ0FBWixFQUFlLENBQWYsRUFBa0IsV0FBbEI7QUFDRDs7QUFDRCxRQUFJa0osTUFBTSxDQUFDRyxPQUFYLEVBQW9CO0FBQ2xCclUsTUFBQUEsSUFBSSxDQUFDNkIsSUFBTCxDQUFVcVMsTUFBTSxDQUFDRyxPQUFqQjtBQUNEOztBQUNELFdBQU8sQ0FBQyxNQUFNLEtBQUt0VSxJQUFMLENBQVVDLElBQVYsQ0FBUCxFQUF3QjJCLElBQXhCLEdBQStCa0ssS0FBL0IsQ0FBcUNrQywwQkFBckMsQ0FBUDtBQUNEOztBQUVEdUcsRUFBQUEsYUFBYSxDQUFDbEwsS0FBRCxFQUFRMEosUUFBUixFQUFrQjtBQUM3QixRQUFJMUosS0FBSyxDQUFDaEssTUFBTixLQUFpQixDQUFyQixFQUF3QjtBQUFFLGFBQU8sSUFBUDtBQUFjOztBQUN4QyxVQUFNWSxJQUFJLEdBQUcsQ0FBQyxVQUFELENBQWI7O0FBQ0EsUUFBSThTLFFBQUosRUFBYztBQUFFOVMsTUFBQUEsSUFBSSxDQUFDNkIsSUFBTCxDQUFVaVIsUUFBVjtBQUFzQjs7QUFDdEMsV0FBTyxLQUFLL1MsSUFBTCxDQUFVQyxJQUFJLENBQUNxSixNQUFMLENBQVksSUFBWixFQUFrQkQsS0FBSyxDQUFDRSxHQUFOLENBQVVDLHFCQUFWLENBQWxCLENBQVYsRUFBc0Q7QUFBQ2pKLE1BQUFBLGNBQWMsRUFBRTtBQUFqQixLQUF0RCxDQUFQO0FBQ0Q7O0FBRUQsUUFBTWlVLFlBQU4sR0FBcUI7QUFDbkIsV0FBTyxDQUFDLE1BQU0sS0FBS3hVLElBQUwsQ0FBVSxDQUFDLFVBQUQsRUFBYSxZQUFiLEVBQTJCLE9BQTNCLEVBQW9DLFVBQXBDLEVBQWdELE1BQWhELENBQVYsQ0FBUCxFQUEyRTRCLElBQTNFLEVBQVA7QUFDRDs7QUFFRCxRQUFNK0gsU0FBTixDQUFnQndLLE1BQWhCLEVBQXdCO0FBQUNNLElBQUFBO0FBQUQsTUFBVSxFQUFsQyxFQUFzQztBQUNwQyxRQUFJeEwsTUFBSjs7QUFDQSxRQUFJO0FBQ0YsVUFBSWhKLElBQUksR0FBRyxDQUFDLFFBQUQsQ0FBWDs7QUFDQSxVQUFJd1UsS0FBSixFQUFXO0FBQUV4VSxRQUFBQSxJQUFJLENBQUM2QixJQUFMLENBQVUsU0FBVjtBQUF1Qjs7QUFDcEM3QixNQUFBQSxJQUFJLEdBQUdBLElBQUksQ0FBQ3FKLE1BQUwsQ0FBWTZLLE1BQVosQ0FBUDtBQUNBbEwsTUFBQUEsTUFBTSxHQUFHLE1BQU0sS0FBS2pKLElBQUwsQ0FBVUMsSUFBVixDQUFmO0FBQ0QsS0FMRCxDQUtFLE9BQU8yRixHQUFQLEVBQVk7QUFDWixVQUFJQSxHQUFHLENBQUM0QixJQUFKLEtBQWEsQ0FBYixJQUFrQjVCLEdBQUcsQ0FBQzRCLElBQUosS0FBYSxHQUFuQyxFQUF3QztBQUN0QztBQUNBLGVBQU8sSUFBUDtBQUNELE9BSEQsTUFHTztBQUNMLGNBQU01QixHQUFOO0FBQ0Q7QUFDRjs7QUFFRCxXQUFPcUQsTUFBTSxDQUFDckgsSUFBUCxFQUFQO0FBQ0Q7O0FBRUQ4UyxFQUFBQSxTQUFTLENBQUNQLE1BQUQsRUFBUzNILEtBQVQsRUFBZ0I7QUFBQ21JLElBQUFBLFVBQUQ7QUFBYUMsSUFBQUE7QUFBYixNQUF1QixFQUF2QyxFQUEyQztBQUNsRCxRQUFJM1UsSUFBSSxHQUFHLENBQUMsUUFBRCxDQUFYOztBQUNBLFFBQUkwVSxVQUFKLEVBQWdCO0FBQUUxVSxNQUFBQSxJQUFJLENBQUM2QixJQUFMLENBQVUsZUFBVjtBQUE2Qjs7QUFDL0MsUUFBSThTLE1BQUosRUFBWTtBQUFFM1UsTUFBQUEsSUFBSSxDQUFDNkIsSUFBTCxDQUFVLFVBQVY7QUFBd0I7O0FBQ3RDN0IsSUFBQUEsSUFBSSxHQUFHQSxJQUFJLENBQUNxSixNQUFMLENBQVk2SyxNQUFaLEVBQW9CM0gsS0FBcEIsQ0FBUDtBQUNBLFdBQU8sS0FBS3hNLElBQUwsQ0FBVUMsSUFBVixFQUFnQjtBQUFDTSxNQUFBQSxjQUFjLEVBQUU7QUFBakIsS0FBaEIsQ0FBUDtBQUNEOztBQUVEc1UsRUFBQUEsV0FBVyxDQUFDVixNQUFELEVBQVM7QUFDbEIsV0FBTyxLQUFLblUsSUFBTCxDQUFVLENBQUMsUUFBRCxFQUFXLFNBQVgsRUFBc0JtVSxNQUF0QixDQUFWLEVBQXlDO0FBQUM1VCxNQUFBQSxjQUFjLEVBQUU7QUFBakIsS0FBekMsQ0FBUDtBQUNEOztBQUVELFFBQU11VSxVQUFOLEdBQW1CO0FBQ2pCLFFBQUk3TCxNQUFNLEdBQUcsTUFBTSxLQUFLVSxTQUFMLENBQWUsQ0FBQyxjQUFELEVBQWlCLHFCQUFqQixDQUFmLEVBQXdEO0FBQUM4SyxNQUFBQSxLQUFLLEVBQUU7QUFBUixLQUF4RCxDQUFuQjs7QUFDQSxRQUFJeEwsTUFBSixFQUFZO0FBQ1ZBLE1BQUFBLE1BQU0sR0FBR0EsTUFBTSxDQUFDckgsSUFBUCxFQUFUOztBQUNBLFVBQUksQ0FBQ3FILE1BQU0sQ0FBQzVKLE1BQVosRUFBb0I7QUFBRSxlQUFPLEVBQVA7QUFBWTs7QUFDbEMsYUFBTzRKLE1BQU0sQ0FBQzZDLEtBQVAsQ0FBYSxJQUFiLEVBQW1CdkMsR0FBbkIsQ0FBdUJ5QyxJQUFJLElBQUk7QUFDcEMsY0FBTW1GLEtBQUssR0FBR25GLElBQUksQ0FBQ21GLEtBQUwsQ0FBVywwQkFBWCxDQUFkO0FBQ0EsZUFBTztBQUNMMUUsVUFBQUEsSUFBSSxFQUFFMEUsS0FBSyxDQUFDLENBQUQsQ0FETjtBQUVMNEQsVUFBQUEsR0FBRyxFQUFFNUQsS0FBSyxDQUFDLENBQUQ7QUFGTCxTQUFQO0FBSUQsT0FOTSxDQUFQO0FBT0QsS0FWRCxNQVVPO0FBQ0wsYUFBTyxFQUFQO0FBQ0Q7QUFDRjs7QUFFRDZELEVBQUFBLFNBQVMsQ0FBQ3ZJLElBQUQsRUFBT3NJLEdBQVAsRUFBWTtBQUNuQixXQUFPLEtBQUsvVSxJQUFMLENBQVUsQ0FBQyxRQUFELEVBQVcsS0FBWCxFQUFrQnlNLElBQWxCLEVBQXdCc0ksR0FBeEIsQ0FBVixDQUFQO0FBQ0Q7O0FBRUQsUUFBTUUsVUFBTixDQUFpQjtBQUFDNUgsSUFBQUEsUUFBRDtBQUFXbE4sSUFBQUE7QUFBWCxNQUFvQixFQUFyQyxFQUF5QztBQUN2QyxRQUFJOEksTUFBSjs7QUFDQSxRQUFJb0UsUUFBSixFQUFjO0FBQ1osVUFBSTtBQUNGcEUsUUFBQUEsTUFBTSxHQUFHLENBQUMsTUFBTSxLQUFLakosSUFBTCxDQUFVLENBQUMsYUFBRCxFQUFnQixJQUFoQixFQUFzQnFOLFFBQXRCLENBQVYsRUFBMkM7QUFBQzlNLFVBQUFBLGNBQWMsRUFBRTtBQUFqQixTQUEzQyxDQUFQLEVBQTJFcUIsSUFBM0UsRUFBVDtBQUNELE9BRkQsQ0FFRSxPQUFPb0csQ0FBUCxFQUFVO0FBQ1YsWUFBSUEsQ0FBQyxDQUFDUCxNQUFGLElBQVlPLENBQUMsQ0FBQ1AsTUFBRixDQUFTMEosS0FBVCxDQUFlLGtEQUFmLENBQWhCLEVBQW9GO0FBQ2xGbEksVUFBQUEsTUFBTSxHQUFHLElBQVQ7QUFDRCxTQUZELE1BRU87QUFDTCxnQkFBTWpCLENBQU47QUFDRDtBQUNGO0FBQ0YsS0FWRCxNQVVPLElBQUk3SCxLQUFKLEVBQVc7QUFDaEI4SSxNQUFBQSxNQUFNLEdBQUcsQ0FBQyxNQUFNLEtBQUtqSixJQUFMLENBQVUsQ0FBQyxhQUFELEVBQWdCLElBQWhCLEVBQXNCLFNBQXRCLENBQVYsRUFBNEM7QUFBQ0csUUFBQUEsS0FBRDtBQUFRSSxRQUFBQSxjQUFjLEVBQUU7QUFBeEIsT0FBNUMsQ0FBUCxFQUFtRnFCLElBQW5GLEVBQVQ7QUFDRCxLQUZNLE1BRUE7QUFDTCxZQUFNLElBQUk5RCxLQUFKLENBQVUsZ0NBQVYsQ0FBTjtBQUNEOztBQUNELFdBQU9tTCxNQUFQO0FBQ0Q7O0FBRUQsUUFBTWlNLGdCQUFOLENBQXVCQyxXQUF2QixFQUFvQ2pGLEdBQXBDLEVBQXlDO0FBQ3ZDLFVBQU1qSCxNQUFNLEdBQUcsTUFBTSxLQUFLakosSUFBTCxDQUFVLENBQUMsVUFBRCxFQUFhLElBQWIsRUFBbUJrUSxHQUFuQixDQUFWLENBQXJCO0FBQ0EsVUFBTW5ILGlCQUFHcU0sU0FBSCxDQUFhRCxXQUFiLEVBQTBCbE0sTUFBMUIsRUFBa0M7QUFBQ2tCLE1BQUFBLFFBQVEsRUFBRTtBQUFYLEtBQWxDLENBQU47QUFDQSxXQUFPZ0wsV0FBUDtBQUNEOztBQUVELFFBQU1FLGVBQU4sQ0FBc0JuRixHQUF0QixFQUEyQjtBQUN6QixXQUFPLE1BQU0sS0FBS2xRLElBQUwsQ0FBVSxDQUFDLFVBQUQsRUFBYSxJQUFiLEVBQW1Ca1EsR0FBbkIsQ0FBVixDQUFiO0FBQ0Q7O0FBRUQsUUFBTW9GLFNBQU4sQ0FBZ0JDLFFBQWhCLEVBQTBCQyxjQUExQixFQUEwQ0MsVUFBMUMsRUFBc0RDLFVBQXRELEVBQWtFO0FBQ2hFLFVBQU16VixJQUFJLEdBQUcsQ0FDWCxZQURXLEVBQ0csSUFESCxFQUNTc1YsUUFEVCxFQUNtQkMsY0FEbkIsRUFDbUNDLFVBRG5DLEVBRVgsSUFGVyxFQUVMLFNBRkssRUFFTSxJQUZOLEVBRVksZUFGWixFQUU2QixJQUY3QixFQUVtQyxnQkFGbkMsQ0FBYjtBQUlBLFFBQUl4TSxNQUFKO0FBQ0EsUUFBSTBNLFFBQVEsR0FBRyxLQUFmOztBQUNBLFFBQUk7QUFDRjFNLE1BQUFBLE1BQU0sR0FBRyxNQUFNLEtBQUtqSixJQUFMLENBQVVDLElBQVYsQ0FBZjtBQUNELEtBRkQsQ0FFRSxPQUFPK0gsQ0FBUCxFQUFVO0FBQ1YsVUFBSUEsQ0FBQyxZQUFZbkssUUFBYixJQUF5Qm1LLENBQUMsQ0FBQ1IsSUFBRixLQUFXLENBQXhDLEVBQTJDO0FBQ3pDeUIsUUFBQUEsTUFBTSxHQUFHakIsQ0FBQyxDQUFDTixNQUFYO0FBQ0FpTyxRQUFBQSxRQUFRLEdBQUcsSUFBWDtBQUNELE9BSEQsTUFHTztBQUNMLGNBQU0zTixDQUFOO0FBQ0Q7QUFDRixLQWhCK0QsQ0FrQmhFO0FBQ0E7OztBQUNBLFVBQU00TixrQkFBa0IsR0FBR3hULGNBQUtaLE9BQUwsQ0FBYSxLQUFLNUMsVUFBbEIsRUFBOEI4VyxVQUE5QixDQUEzQjs7QUFDQSxVQUFNM00saUJBQUdxTSxTQUFILENBQWFRLGtCQUFiLEVBQWlDM00sTUFBakMsRUFBeUM7QUFBQ2tCLE1BQUFBLFFBQVEsRUFBRTtBQUFYLEtBQXpDLENBQU47QUFFQSxXQUFPO0FBQUNrRCxNQUFBQSxRQUFRLEVBQUVrSSxRQUFYO0FBQXFCRyxNQUFBQSxVQUFyQjtBQUFpQ0MsTUFBQUE7QUFBakMsS0FBUDtBQUNEOztBQUVELFFBQU1FLHlCQUFOLENBQWdDeEksUUFBaEMsRUFBMEN5SSxhQUExQyxFQUF5REMsT0FBekQsRUFBa0VDLFNBQWxFLEVBQTZFO0FBQzNFLFVBQU1DLFdBQVcsR0FBRywyQkFBYTVJLFFBQWIsQ0FBcEI7QUFDQSxVQUFNNkksUUFBUSxHQUFHLE1BQU0sS0FBS0MsV0FBTCxDQUFpQjlJLFFBQWpCLENBQXZCO0FBQ0EsUUFBSStJLFNBQVMsR0FBSSwrQ0FBOENILFdBQVksSUFBM0U7O0FBQ0EsUUFBSUgsYUFBSixFQUFtQjtBQUFFTSxNQUFBQSxTQUFTLElBQUssR0FBRUYsUUFBUyxJQUFHSixhQUFjLE9BQU1HLFdBQVksSUFBNUQ7QUFBa0U7O0FBQ3ZGLFFBQUlGLE9BQUosRUFBYTtBQUFFSyxNQUFBQSxTQUFTLElBQUssR0FBRUYsUUFBUyxJQUFHSCxPQUFRLE9BQU1FLFdBQVksSUFBdEQ7QUFBNEQ7O0FBQzNFLFFBQUlELFNBQUosRUFBZTtBQUFFSSxNQUFBQSxTQUFTLElBQUssR0FBRUYsUUFBUyxJQUFHRixTQUFVLE9BQU1DLFdBQVksSUFBeEQ7QUFBOEQ7O0FBQy9FLFdBQU8sS0FBS2pXLElBQUwsQ0FBVSxDQUFDLGNBQUQsRUFBaUIsY0FBakIsQ0FBVixFQUE0QztBQUFDRyxNQUFBQSxLQUFLLEVBQUVpVyxTQUFSO0FBQW1CN1YsTUFBQUEsY0FBYyxFQUFFO0FBQW5DLEtBQTVDLENBQVA7QUFDRDs7QUFFRCxRQUFNNFYsV0FBTixDQUFrQjlJLFFBQWxCLEVBQTRCO0FBQzFCLFVBQU1wRSxNQUFNLEdBQUcsTUFBTSxLQUFLakosSUFBTCxDQUFVLENBQUMsVUFBRCxFQUFhLFNBQWIsRUFBd0IsSUFBeEIsRUFBOEIsMkJBQWFxTixRQUFiLENBQTlCLENBQVYsQ0FBckI7O0FBQ0EsUUFBSXBFLE1BQUosRUFBWTtBQUNWLGFBQU9BLE1BQU0sQ0FBQ2xCLEtBQVAsQ0FBYSxDQUFiLEVBQWdCLENBQWhCLENBQVA7QUFDRCxLQUZELE1BRU87QUFDTCxZQUFNK0csVUFBVSxHQUFHLE1BQU0sK0JBQWlCMU0sY0FBS2pCLElBQUwsQ0FBVSxLQUFLdkMsVUFBZixFQUEyQnlPLFFBQTNCLENBQWpCLENBQXpCO0FBQ0EsWUFBTTBCLE9BQU8sR0FBRyxNQUFNLDRCQUFjM00sY0FBS2pCLElBQUwsQ0FBVSxLQUFLdkMsVUFBZixFQUEyQnlPLFFBQTNCLENBQWQsQ0FBdEI7O0FBQ0EsVUFBSTBCLE9BQUosRUFBYTtBQUNYLGVBQU9JLGNBQUtDLEtBQUwsQ0FBV0UsT0FBbEI7QUFDRCxPQUZELE1BRU8sSUFBSVIsVUFBSixFQUFnQjtBQUNyQixlQUFPSyxjQUFLQyxLQUFMLENBQVdDLFVBQWxCO0FBQ0QsT0FGTSxNQUVBO0FBQ0wsZUFBT0YsY0FBS0MsS0FBTCxDQUFXRyxNQUFsQjtBQUNEO0FBQ0Y7QUFDRjs7QUFFRDhHLEVBQUFBLE9BQU8sR0FBRztBQUNSLFNBQUt0WCxZQUFMLENBQWtCMkgsT0FBbEI7QUFDRDs7QUFsakNzQzs7OztnQkFBcEIvSCxtQixxQkFDTTtBQUN2QndCLEVBQUFBLEtBQUssRUFBRSxJQURnQjtBQUV2QkMsRUFBQUEsa0JBQWtCLEVBQUUsS0FGRztBQUd2QkMsRUFBQUEsYUFBYSxFQUFFLEtBSFE7QUFJdkJDLEVBQUFBLGdCQUFnQixFQUFFLEtBSks7QUFLdkJDLEVBQUFBLGNBQWMsRUFBRTtBQUxPLEM7O0FBb2pDM0IsU0FBU2lQLG1CQUFULENBQTZCbkMsUUFBN0IsRUFBdUMyQixRQUF2QyxFQUFpRDdDLElBQWpELEVBQXVEK0MsUUFBdkQsRUFBaUU7QUFDL0QsUUFBTW9ILEtBQUssR0FBRyxFQUFkOztBQUNBLE1BQUl0SCxRQUFKLEVBQWM7QUFDWixRQUFJdUgsU0FBSjtBQUNBLFFBQUlDLEtBQUo7O0FBQ0EsUUFBSXJLLElBQUksS0FBS2dELGNBQUtDLEtBQUwsQ0FBV0UsT0FBeEIsRUFBaUM7QUFDL0JpSCxNQUFBQSxTQUFTLEdBQUcsS0FBWjtBQUNBQyxNQUFBQSxLQUFLLEdBQUcsQ0FBRSxJQUFHLDJCQUFhdEgsUUFBYixDQUF1QixFQUE1QixFQUErQiw4QkFBL0IsQ0FBUjtBQUNELEtBSEQsTUFHTztBQUNMcUgsTUFBQUEsU0FBUyxHQUFHdkgsUUFBUSxDQUFDQSxRQUFRLENBQUMzUCxNQUFULEdBQWtCLENBQW5CLENBQVIsS0FBa0MsSUFBOUM7QUFDQW1YLE1BQUFBLEtBQUssR0FBR3hILFFBQVEsQ0FBQ3BOLElBQVQsR0FBZ0JrSyxLQUFoQixDQUFzQmtDLDBCQUF0QixFQUF5Q3pFLEdBQXpDLENBQTZDeUMsSUFBSSxJQUFLLElBQUdBLElBQUssRUFBOUQsQ0FBUjtBQUNEOztBQUNELFFBQUl1SyxTQUFKLEVBQWU7QUFBRUMsTUFBQUEsS0FBSyxDQUFDMVUsSUFBTixDQUFXLDhCQUFYO0FBQTZDOztBQUM5RHdVLElBQUFBLEtBQUssQ0FBQ3hVLElBQU4sQ0FBVztBQUNUMFUsTUFBQUEsS0FEUztBQUVUQyxNQUFBQSxZQUFZLEVBQUUsQ0FGTDtBQUdUQyxNQUFBQSxZQUFZLEVBQUUsQ0FITDtBQUlUQyxNQUFBQSxZQUFZLEVBQUUsQ0FKTDtBQUtUQyxNQUFBQSxPQUFPLEVBQUUsRUFMQTtBQU1UQyxNQUFBQSxZQUFZLEVBQUVOLFNBQVMsR0FBR0MsS0FBSyxDQUFDblgsTUFBTixHQUFlLENBQWxCLEdBQXNCbVgsS0FBSyxDQUFDblg7QUFOMUMsS0FBWDtBQVFEOztBQUNELFNBQU87QUFDTHNQLElBQUFBLE9BQU8sRUFBRSxJQURKO0FBRUxDLElBQUFBLE9BQU8sRUFBRSw4QkFBZ0J2QixRQUFoQixDQUZKO0FBR0x5SixJQUFBQSxPQUFPLEVBQUUsSUFISjtBQUlMdE0sSUFBQUEsT0FBTyxFQUFFMkIsSUFKSjtBQUtMOEIsSUFBQUEsTUFBTSxFQUFFLE9BTEg7QUFNTHFJLElBQUFBO0FBTkssR0FBUDtBQVFEIiwic291cmNlUm9vdCI6Ii9idWlsZC9hdG9tL3NyYy9hdG9tL291dC9hcHAvbm9kZV9tb2R1bGVzL2dpdGh1YiIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCBwYXRoIGZyb20gJ3BhdGgnO1xuaW1wb3J0IG9zIGZyb20gJ29zJztcbmltcG9ydCBjaGlsZFByb2Nlc3MgZnJvbSAnY2hpbGRfcHJvY2Vzcyc7XG5pbXBvcnQgZnMgZnJvbSAnZnMtZXh0cmEnO1xuaW1wb3J0IHV0aWwgZnJvbSAndXRpbCc7XG5pbXBvcnQge3JlbW90ZX0gZnJvbSAnZWxlY3Ryb24nO1xuXG5pbXBvcnQge0NvbXBvc2l0ZURpc3Bvc2FibGV9IGZyb20gJ2V2ZW50LWtpdCc7XG5pbXBvcnQge0dpdFByb2Nlc3N9IGZyb20gJ2R1Z2l0ZSc7XG5pbXBvcnQge3BhcnNlIGFzIHBhcnNlRGlmZn0gZnJvbSAnd2hhdC10aGUtZGlmZic7XG5pbXBvcnQge3BhcnNlIGFzIHBhcnNlU3RhdHVzfSBmcm9tICd3aGF0LXRoZS1zdGF0dXMnO1xuXG5pbXBvcnQgR2l0UHJvbXB0U2VydmVyIGZyb20gJy4vZ2l0LXByb21wdC1zZXJ2ZXInO1xuaW1wb3J0IEdpdFRlbXBEaXIgZnJvbSAnLi9naXQtdGVtcC1kaXInO1xuaW1wb3J0IEFzeW5jUXVldWUgZnJvbSAnLi9hc3luYy1xdWV1ZSc7XG5pbXBvcnQge2luY3JlbWVudENvdW50ZXJ9IGZyb20gJy4vcmVwb3J0ZXItcHJveHknO1xuaW1wb3J0IHtcbiAgZ2V0RHVnaXRlUGF0aCwgZ2V0U2hhcmVkTW9kdWxlUGF0aCwgZ2V0QXRvbUhlbHBlclBhdGgsXG4gIGV4dHJhY3RDb0F1dGhvcnNBbmRSYXdDb21taXRNZXNzYWdlLCBmaWxlRXhpc3RzLCBpc0ZpbGVFeGVjdXRhYmxlLCBpc0ZpbGVTeW1saW5rLCBpc0JpbmFyeSxcbiAgbm9ybWFsaXplR2l0SGVscGVyUGF0aCwgdG9OYXRpdmVQYXRoU2VwLCB0b0dpdFBhdGhTZXAsIExJTkVfRU5ESU5HX1JFR0VYLCBDT19BVVRIT1JfUkVHRVgsXG59IGZyb20gJy4vaGVscGVycyc7XG5pbXBvcnQgR2l0VGltaW5nc1ZpZXcgZnJvbSAnLi92aWV3cy9naXQtdGltaW5ncy12aWV3JztcbmltcG9ydCBGaWxlIGZyb20gJy4vbW9kZWxzL3BhdGNoL2ZpbGUnO1xuaW1wb3J0IFdvcmtlck1hbmFnZXIgZnJvbSAnLi93b3JrZXItbWFuYWdlcic7XG5pbXBvcnQgQXV0aG9yIGZyb20gJy4vbW9kZWxzL2F1dGhvcic7XG5cbmNvbnN0IE1BWF9TVEFUVVNfT1VUUFVUX0xFTkdUSCA9IDEwMjQgKiAxMDI0ICogMTA7XG5cbmxldCBoZWFkbGVzcyA9IG51bGw7XG5sZXQgZXhlY1BhdGhQcm9taXNlID0gbnVsbDtcblxuZXhwb3J0IGNsYXNzIEdpdEVycm9yIGV4dGVuZHMgRXJyb3Ige1xuICBjb25zdHJ1Y3RvcihtZXNzYWdlKSB7XG4gICAgc3VwZXIobWVzc2FnZSk7XG4gICAgdGhpcy5tZXNzYWdlID0gbWVzc2FnZTtcbiAgICB0aGlzLnN0YWNrID0gbmV3IEVycm9yKCkuc3RhY2s7XG4gIH1cbn1cblxuZXhwb3J0IGNsYXNzIExhcmdlUmVwb0Vycm9yIGV4dGVuZHMgRXJyb3Ige1xuICBjb25zdHJ1Y3RvcihtZXNzYWdlKSB7XG4gICAgc3VwZXIobWVzc2FnZSk7XG4gICAgdGhpcy5tZXNzYWdlID0gbWVzc2FnZTtcbiAgICB0aGlzLnN0YWNrID0gbmV3IEVycm9yKCkuc3RhY2s7XG4gIH1cbn1cblxuLy8gaWdub3JlZCBmb3IgdGhlIHB1cnBvc2VzIG9mIHVzYWdlIG1ldHJpY3MgdHJhY2tpbmcgYmVjYXVzZSB0aGV5J3JlIG5vaXN5XG5jb25zdCBJR05PUkVEX0dJVF9DT01NQU5EUyA9IFsnY2F0LWZpbGUnLCAnY29uZmlnJywgJ2RpZmYnLCAnZm9yLWVhY2gtcmVmJywgJ2xvZycsICdyZXYtcGFyc2UnLCAnc3RhdHVzJ107XG5cbmNvbnN0IERJU0FCTEVfQ09MT1JfRkxBR1MgPSBbXG4gICdicmFuY2gnLCAnZGlmZicsICdzaG93QnJhbmNoJywgJ3N0YXR1cycsICd1aScsXG5dLnJlZHVjZSgoYWNjLCB0eXBlKSA9PiB7XG4gIGFjYy51bnNoaWZ0KCctYycsIGBjb2xvci4ke3R5cGV9PWZhbHNlYCk7XG4gIHJldHVybiBhY2M7XG59LCBbXSk7XG5cbi8qKlxuICogRXhwYW5kIGNvbmZpZyBwYXRoIG5hbWUgcGVyXG4gKiBodHRwczovL2dpdC1zY20uY29tL2RvY3MvZ2l0LWNvbmZpZyNnaXQtY29uZmlnLXBhdGhuYW1lXG4gKiB0aGlzIHJlZ2V4IGF0dGVtcHRzIHRvIGdldCB0aGUgc3BlY2lmaWVkIHVzZXIncyBob21lIGRpcmVjdG9yeVxuICogRXg6IG9uIE1hYyB+a3V5Y2hhY28vIGlzIGV4cGFuZGVkIHRvIHRoZSBzcGVjaWZpZWQgdXNlcuKAmXMgaG9tZSBkaXJlY3RvcnkgKC9Vc2Vycy9rdXljaGFjbylcbiAqIFJlZ2V4IHRyYW5zbGF0aW9uOlxuICogXn4gbGluZSBzdGFydHMgd2l0aCB0aWxkZVxuICogKFteXFxcXFxcXFwvXSopW1xcXFxcXFxcL10gY2FwdHVyZXMgbm9uLXNsYXNoIGNoYXJhY3RlcnMgYmVmb3JlIGZpcnN0IHNsYXNoXG4gKi9cbmNvbnN0IEVYUEFORF9USUxERV9SRUdFWCA9IG5ldyBSZWdFeHAoJ15+KFteXFxcXFxcXFwvXSopW1xcXFxcXFxcL10nKTtcblxuZXhwb3J0IGRlZmF1bHQgY2xhc3MgR2l0U2hlbGxPdXRTdHJhdGVneSB7XG4gIHN0YXRpYyBkZWZhdWx0RXhlY0FyZ3MgPSB7XG4gICAgc3RkaW46IG51bGwsXG4gICAgdXNlR2l0UHJvbXB0U2VydmVyOiBmYWxzZSxcbiAgICB1c2VHcGdXcmFwcGVyOiBmYWxzZSxcbiAgICB1c2VHcGdBdG9tUHJvbXB0OiBmYWxzZSxcbiAgICB3cml0ZU9wZXJhdGlvbjogZmFsc2UsXG4gIH1cblxuICBjb25zdHJ1Y3Rvcih3b3JraW5nRGlyLCBvcHRpb25zID0ge30pIHtcbiAgICB0aGlzLndvcmtpbmdEaXIgPSB3b3JraW5nRGlyO1xuICAgIGlmIChvcHRpb25zLnF1ZXVlKSB7XG4gICAgICB0aGlzLmNvbW1hbmRRdWV1ZSA9IG9wdGlvbnMucXVldWU7XG4gICAgfSBlbHNlIHtcbiAgICAgIGNvbnN0IHBhcmFsbGVsaXNtID0gb3B0aW9ucy5wYXJhbGxlbGlzbSB8fCBNYXRoLm1heCgzLCBvcy5jcHVzKCkubGVuZ3RoKTtcbiAgICAgIHRoaXMuY29tbWFuZFF1ZXVlID0gbmV3IEFzeW5jUXVldWUoe3BhcmFsbGVsaXNtfSk7XG4gICAgfVxuXG4gICAgdGhpcy5wcm9tcHQgPSBvcHRpb25zLnByb21wdCB8fCAocXVlcnkgPT4gUHJvbWlzZS5yZWplY3QoKSk7XG4gICAgdGhpcy53b3JrZXJNYW5hZ2VyID0gb3B0aW9ucy53b3JrZXJNYW5hZ2VyO1xuXG4gICAgaWYgKGhlYWRsZXNzID09PSBudWxsKSB7XG4gICAgICBoZWFkbGVzcyA9ICFyZW1vdGUuZ2V0Q3VycmVudFdpbmRvdygpLmlzVmlzaWJsZSgpO1xuICAgIH1cbiAgfVxuXG4gIC8qXG4gICAqIFByb3ZpZGUgYW4gYXN5bmNocm9ub3VzIGNhbGxiYWNrIHRvIGJlIHVzZWQgdG8gcmVxdWVzdCBpbnB1dCBmcm9tIHRoZSB1c2VyIGZvciBnaXQgb3BlcmF0aW9ucy5cbiAgICpcbiAgICogYHByb21wdGAgbXVzdCBiZSBhIGNhbGxhYmxlIHRoYXQgYWNjZXB0cyBhIHF1ZXJ5IG9iamVjdCBge3Byb21wdCwgaW5jbHVkZVVzZXJuYW1lfWAgYW5kIHJldHVybnMgYSBQcm9taXNlXG4gICAqIHRoYXQgZWl0aGVyIHJlc29sdmVzIHdpdGggYSByZXN1bHQgb2JqZWN0IGB7W3VzZXJuYW1lXSwgcGFzc3dvcmR9YCBvciByZWplY3RzIG9uIGNhbmNlbGxhdGlvbi5cbiAgICovXG4gIHNldFByb21wdENhbGxiYWNrKHByb21wdCkge1xuICAgIHRoaXMucHJvbXB0ID0gcHJvbXB0O1xuICB9XG5cbiAgLy8gRXhlY3V0ZSBhIGNvbW1hbmQgYW5kIHJlYWQgdGhlIG91dHB1dCB1c2luZyB0aGUgZW1iZWRkZWQgR2l0IGVudmlyb25tZW50XG4gIGFzeW5jIGV4ZWMoYXJncywgb3B0aW9ucyA9IEdpdFNoZWxsT3V0U3RyYXRlZ3kuZGVmYXVsdEV4ZWNBcmdzKSB7XG4gICAgLyogZXNsaW50LWRpc2FibGUgbm8tY29uc29sZSxuby1jb250cm9sLXJlZ2V4ICovXG4gICAgY29uc3Qge3N0ZGluLCB1c2VHaXRQcm9tcHRTZXJ2ZXIsIHVzZUdwZ1dyYXBwZXIsIHVzZUdwZ0F0b21Qcm9tcHQsIHdyaXRlT3BlcmF0aW9ufSA9IG9wdGlvbnM7XG4gICAgY29uc3QgY29tbWFuZE5hbWUgPSBhcmdzWzBdO1xuICAgIGNvbnN0IHN1YnNjcmlwdGlvbnMgPSBuZXcgQ29tcG9zaXRlRGlzcG9zYWJsZSgpO1xuICAgIGNvbnN0IGRpYWdub3N0aWNzRW5hYmxlZCA9IHByb2Nlc3MuZW52LkFUT01fR0lUSFVCX0dJVF9ESUFHTk9TVElDUyB8fCBhdG9tLmNvbmZpZy5nZXQoJ2dpdGh1Yi5naXREaWFnbm9zdGljcycpO1xuXG4gICAgY29uc3QgZm9ybWF0dGVkQXJncyA9IGBnaXQgJHthcmdzLmpvaW4oJyAnKX0gaW4gJHt0aGlzLndvcmtpbmdEaXJ9YDtcbiAgICBjb25zdCB0aW1pbmdNYXJrZXIgPSBHaXRUaW1pbmdzVmlldy5nZW5lcmF0ZU1hcmtlcihgZ2l0ICR7YXJncy5qb2luKCcgJyl9YCk7XG4gICAgdGltaW5nTWFya2VyLm1hcmsoJ3F1ZXVlZCcpO1xuXG4gICAgYXJncy51bnNoaWZ0KC4uLkRJU0FCTEVfQ09MT1JfRkxBR1MpO1xuXG4gICAgaWYgKGV4ZWNQYXRoUHJvbWlzZSA9PT0gbnVsbCkge1xuICAgICAgLy8gQXR0ZW1wdCB0byBjb2xsZWN0IHRoZSAtLWV4ZWMtcGF0aCBmcm9tIGEgbmF0aXZlIGdpdCBpbnN0YWxsYXRpb24uXG4gICAgICBleGVjUGF0aFByb21pc2UgPSBuZXcgUHJvbWlzZShyZXNvbHZlID0+IHtcbiAgICAgICAgY2hpbGRQcm9jZXNzLmV4ZWMoJ2dpdCAtLWV4ZWMtcGF0aCcsIChlcnJvciwgc3Rkb3V0KSA9PiB7XG4gICAgICAgICAgLyogaXN0YW5idWwgaWdub3JlIGlmICovXG4gICAgICAgICAgaWYgKGVycm9yKSB7XG4gICAgICAgICAgICAvLyBPaCB3ZWxsXG4gICAgICAgICAgICByZXNvbHZlKG51bGwpO1xuICAgICAgICAgICAgcmV0dXJuO1xuICAgICAgICAgIH1cblxuICAgICAgICAgIHJlc29sdmUoc3Rkb3V0LnRyaW0oKSk7XG4gICAgICAgIH0pO1xuICAgICAgfSk7XG4gICAgfVxuICAgIGNvbnN0IGV4ZWNQYXRoID0gYXdhaXQgZXhlY1BhdGhQcm9taXNlO1xuXG4gICAgcmV0dXJuIHRoaXMuY29tbWFuZFF1ZXVlLnB1c2goYXN5bmMgKCkgPT4ge1xuICAgICAgdGltaW5nTWFya2VyLm1hcmsoJ3ByZXBhcmUnKTtcbiAgICAgIGxldCBnaXRQcm9tcHRTZXJ2ZXI7XG5cbiAgICAgIGNvbnN0IHBhdGhQYXJ0cyA9IFtdO1xuICAgICAgaWYgKHByb2Nlc3MuZW52LlBBVEgpIHtcbiAgICAgICAgcGF0aFBhcnRzLnB1c2gocHJvY2Vzcy5lbnYuUEFUSCk7XG4gICAgICB9XG4gICAgICBpZiAoZXhlY1BhdGgpIHtcbiAgICAgICAgcGF0aFBhcnRzLnB1c2goZXhlY1BhdGgpO1xuICAgICAgfVxuXG4gICAgICBjb25zdCBlbnYgPSB7XG4gICAgICAgIC4uLnByb2Nlc3MuZW52LFxuICAgICAgICBHSVRfVEVSTUlOQUxfUFJPTVBUOiAnMCcsXG4gICAgICAgIEdJVF9PUFRJT05BTF9MT0NLUzogJzAnLFxuICAgICAgICBQQVRIOiBwYXRoUGFydHMuam9pbihwYXRoLmRlbGltaXRlciksXG4gICAgICB9O1xuXG4gICAgICBjb25zdCBnaXRUZW1wRGlyID0gbmV3IEdpdFRlbXBEaXIoKTtcblxuICAgICAgaWYgKHVzZUdwZ1dyYXBwZXIpIHtcbiAgICAgICAgYXdhaXQgZ2l0VGVtcERpci5lbnN1cmUoKTtcbiAgICAgICAgYXJncy51bnNoaWZ0KCctYycsIGBncGcucHJvZ3JhbT0ke2dpdFRlbXBEaXIuZ2V0R3BnV3JhcHBlclNoKCl9YCk7XG4gICAgICB9XG5cbiAgICAgIGlmICh1c2VHaXRQcm9tcHRTZXJ2ZXIpIHtcbiAgICAgICAgZ2l0UHJvbXB0U2VydmVyID0gbmV3IEdpdFByb21wdFNlcnZlcihnaXRUZW1wRGlyKTtcbiAgICAgICAgYXdhaXQgZ2l0UHJvbXB0U2VydmVyLnN0YXJ0KHRoaXMucHJvbXB0KTtcblxuICAgICAgICBlbnYuQVRPTV9HSVRIVUJfVE1QID0gZ2l0VGVtcERpci5nZXRSb290UGF0aCgpO1xuICAgICAgICBlbnYuQVRPTV9HSVRIVUJfQVNLUEFTU19QQVRIID0gbm9ybWFsaXplR2l0SGVscGVyUGF0aChnaXRUZW1wRGlyLmdldEFza1Bhc3NKcygpKTtcbiAgICAgICAgZW52LkFUT01fR0lUSFVCX0NSRURFTlRJQUxfUEFUSCA9IG5vcm1hbGl6ZUdpdEhlbHBlclBhdGgoZ2l0VGVtcERpci5nZXRDcmVkZW50aWFsSGVscGVySnMoKSk7XG4gICAgICAgIGVudi5BVE9NX0dJVEhVQl9FTEVDVFJPTl9QQVRIID0gbm9ybWFsaXplR2l0SGVscGVyUGF0aChnZXRBdG9tSGVscGVyUGF0aCgpKTtcbiAgICAgICAgZW52LkFUT01fR0lUSFVCX1NPQ0tfQUREUiA9IGdpdFByb21wdFNlcnZlci5nZXRBZGRyZXNzKCk7XG5cbiAgICAgICAgZW52LkFUT01fR0lUSFVCX1dPUktESVJfUEFUSCA9IHRoaXMud29ya2luZ0RpcjtcbiAgICAgICAgZW52LkFUT01fR0lUSFVCX0RVR0lURV9QQVRIID0gZ2V0RHVnaXRlUGF0aCgpO1xuICAgICAgICBlbnYuQVRPTV9HSVRIVUJfS0VZVEFSX1NUUkFURUdZX1BBVEggPSBnZXRTaGFyZWRNb2R1bGVQYXRoKCdrZXl0YXItc3RyYXRlZ3knKTtcblxuICAgICAgICAvLyBcInNzaFwiIHdvbid0IHJlc3BlY3QgU1NIX0FTS1BBU1MgdW5sZXNzOlxuICAgICAgICAvLyAoYSkgaXQncyBydW5uaW5nIHdpdGhvdXQgYSB0dHlcbiAgICAgICAgLy8gKGIpIERJU1BMQVkgaXMgc2V0IHRvIHNvbWV0aGluZyBub25lbXB0eVxuICAgICAgICAvLyBCdXQsIG9uIGEgTWFjLCBESVNQTEFZIGlzIHVuc2V0LiBFbnN1cmUgdGhhdCBpdCBpcyBzbyBvdXIgU1NIX0FTS1BBU1MgaXMgcmVzcGVjdGVkLlxuICAgICAgICBpZiAoIXByb2Nlc3MuZW52LkRJU1BMQVkgfHwgcHJvY2Vzcy5lbnYuRElTUExBWS5sZW5ndGggPT09IDApIHtcbiAgICAgICAgICBlbnYuRElTUExBWSA9ICdhdG9tLWdpdGh1Yi1wbGFjZWhvbGRlcic7XG4gICAgICAgIH1cblxuICAgICAgICBlbnYuQVRPTV9HSVRIVUJfT1JJR0lOQUxfUEFUSCA9IHByb2Nlc3MuZW52LlBBVEggfHwgJyc7XG4gICAgICAgIGVudi5BVE9NX0dJVEhVQl9PUklHSU5BTF9HSVRfQVNLUEFTUyA9IHByb2Nlc3MuZW52LkdJVF9BU0tQQVNTIHx8ICcnO1xuICAgICAgICBlbnYuQVRPTV9HSVRIVUJfT1JJR0lOQUxfU1NIX0FTS1BBU1MgPSBwcm9jZXNzLmVudi5TU0hfQVNLUEFTUyB8fCAnJztcbiAgICAgICAgZW52LkFUT01fR0lUSFVCX09SSUdJTkFMX0dJVF9TU0hfQ09NTUFORCA9IHByb2Nlc3MuZW52LkdJVF9TU0hfQ09NTUFORCB8fCAnJztcbiAgICAgICAgZW52LkFUT01fR0lUSFVCX1NQRUNfTU9ERSA9IGF0b20uaW5TcGVjTW9kZSgpID8gJ3RydWUnIDogJ2ZhbHNlJztcblxuICAgICAgICBlbnYuU1NIX0FTS1BBU1MgPSBub3JtYWxpemVHaXRIZWxwZXJQYXRoKGdpdFRlbXBEaXIuZ2V0QXNrUGFzc1NoKCkpO1xuICAgICAgICBlbnYuR0lUX0FTS1BBU1MgPSBub3JtYWxpemVHaXRIZWxwZXJQYXRoKGdpdFRlbXBEaXIuZ2V0QXNrUGFzc1NoKCkpO1xuXG4gICAgICAgIGlmIChwcm9jZXNzLnBsYXRmb3JtID09PSAnbGludXgnKSB7XG4gICAgICAgICAgZW52LkdJVF9TU0hfQ09NTUFORCA9IGdpdFRlbXBEaXIuZ2V0U3NoV3JhcHBlclNoKCk7XG4gICAgICAgIH0gZWxzZSBpZiAocHJvY2Vzcy5lbnYuR0lUX1NTSF9DT01NQU5EKSB7XG4gICAgICAgICAgZW52LkdJVF9TU0hfQ09NTUFORCA9IHByb2Nlc3MuZW52LkdJVF9TU0hfQ09NTUFORDtcbiAgICAgICAgfSBlbHNlIHtcbiAgICAgICAgICBlbnYuR0lUX1NTSCA9IHByb2Nlc3MuZW52LkdJVF9TU0g7XG4gICAgICAgIH1cblxuICAgICAgICBjb25zdCBjcmVkZW50aWFsSGVscGVyU2ggPSBub3JtYWxpemVHaXRIZWxwZXJQYXRoKGdpdFRlbXBEaXIuZ2V0Q3JlZGVudGlhbEhlbHBlclNoKCkpO1xuICAgICAgICBhcmdzLnVuc2hpZnQoJy1jJywgYGNyZWRlbnRpYWwuaGVscGVyPSR7Y3JlZGVudGlhbEhlbHBlclNofWApO1xuICAgICAgfVxuXG4gICAgICBpZiAodXNlR3BnV3JhcHBlciAmJiB1c2VHaXRQcm9tcHRTZXJ2ZXIgJiYgdXNlR3BnQXRvbVByb21wdCkge1xuICAgICAgICBlbnYuQVRPTV9HSVRIVUJfR1BHX1BST01QVCA9ICd0cnVlJztcbiAgICAgIH1cblxuICAgICAgLyogaXN0YW5idWwgaWdub3JlIGlmICovXG4gICAgICBpZiAoZGlhZ25vc3RpY3NFbmFibGVkKSB7XG4gICAgICAgIGVudi5HSVRfVFJBQ0UgPSAndHJ1ZSc7XG4gICAgICAgIGVudi5HSVRfVFJBQ0VfQ1VSTCA9ICd0cnVlJztcbiAgICAgIH1cblxuICAgICAgbGV0IG9wdHMgPSB7ZW52fTtcblxuICAgICAgaWYgKHN0ZGluKSB7XG4gICAgICAgIG9wdHMuc3RkaW4gPSBzdGRpbjtcbiAgICAgICAgb3B0cy5zdGRpbkVuY29kaW5nID0gJ3V0ZjgnO1xuICAgICAgfVxuXG4gICAgICAvKiBpc3RhbmJ1bCBpZ25vcmUgaWYgKi9cbiAgICAgIGlmIChwcm9jZXNzLmVudi5QUklOVF9HSVRfVElNRVMpIHtcbiAgICAgICAgY29uc29sZS50aW1lKGBnaXQ6JHtmb3JtYXR0ZWRBcmdzfWApO1xuICAgICAgfVxuXG4gICAgICByZXR1cm4gbmV3IFByb21pc2UoYXN5bmMgKHJlc29sdmUsIHJlamVjdCkgPT4ge1xuICAgICAgICBpZiAob3B0aW9ucy5iZWZvcmVSdW4pIHtcbiAgICAgICAgICBjb25zdCBuZXdBcmdzT3B0cyA9IGF3YWl0IG9wdGlvbnMuYmVmb3JlUnVuKHthcmdzLCBvcHRzfSk7XG4gICAgICAgICAgYXJncyA9IG5ld0FyZ3NPcHRzLmFyZ3M7XG4gICAgICAgICAgb3B0cyA9IG5ld0FyZ3NPcHRzLm9wdHM7XG4gICAgICAgIH1cbiAgICAgICAgY29uc3Qge3Byb21pc2UsIGNhbmNlbH0gPSB0aGlzLmV4ZWN1dGVHaXRDb21tYW5kKGFyZ3MsIG9wdHMsIHRpbWluZ01hcmtlcik7XG4gICAgICAgIGxldCBleHBlY3RDYW5jZWwgPSBmYWxzZTtcbiAgICAgICAgaWYgKGdpdFByb21wdFNlcnZlcikge1xuICAgICAgICAgIHN1YnNjcmlwdGlvbnMuYWRkKGdpdFByb21wdFNlcnZlci5vbkRpZENhbmNlbChhc3luYyAoe2hhbmRsZXJQaWR9KSA9PiB7XG4gICAgICAgICAgICBleHBlY3RDYW5jZWwgPSB0cnVlO1xuICAgICAgICAgICAgYXdhaXQgY2FuY2VsKCk7XG5cbiAgICAgICAgICAgIC8vIE9uIFdpbmRvd3MsIHRoZSBTU0hfQVNLUEFTUyBoYW5kbGVyIGlzIGV4ZWN1dGVkIGFzIGEgbm9uLWNoaWxkIHByb2Nlc3MsIHNvIHRoZSBiaW5cXGdpdC1hc2twYXNzLWF0b20uc2hcbiAgICAgICAgICAgIC8vIHByb2Nlc3MgZG9lcyBub3QgdGVybWluYXRlIHdoZW4gdGhlIGdpdCBwcm9jZXNzIGlzIGtpbGxlZC5cbiAgICAgICAgICAgIC8vIEtpbGwgdGhlIGhhbmRsZXIgcHJvY2VzcyAqYWZ0ZXIqIHRoZSBnaXQgcHJvY2VzcyBoYXMgYmVlbiBraWxsZWQgdG8gZW5zdXJlIHRoYXQgZ2l0IGRvZXNuJ3QgaGF2ZSBhXG4gICAgICAgICAgICAvLyBjaGFuY2UgdG8gZmFsbCBiYWNrIHRvIEdJVF9BU0tQQVNTIGZyb20gdGhlIGNyZWRlbnRpYWwgaGFuZGxlci5cbiAgICAgICAgICAgIGF3YWl0IG5ldyBQcm9taXNlKChyZXNvbHZlS2lsbCwgcmVqZWN0S2lsbCkgPT4ge1xuICAgICAgICAgICAgICByZXF1aXJlKCd0cmVlLWtpbGwnKShoYW5kbGVyUGlkLCAnU0lHVEVSTScsIGVyciA9PiB7XG4gICAgICAgICAgICAgICAgLyogaXN0YW5idWwgaWdub3JlIGlmICovXG4gICAgICAgICAgICAgICAgaWYgKGVycikgeyByZWplY3RLaWxsKGVycik7IH0gZWxzZSB7IHJlc29sdmVLaWxsKCk7IH1cbiAgICAgICAgICAgICAgfSk7XG4gICAgICAgICAgICB9KTtcbiAgICAgICAgICB9KSk7XG4gICAgICAgIH1cblxuICAgICAgICBjb25zdCB7c3Rkb3V0LCBzdGRlcnIsIGV4aXRDb2RlLCBzaWduYWwsIHRpbWluZ30gPSBhd2FpdCBwcm9taXNlLmNhdGNoKGVyciA9PiB7XG4gICAgICAgICAgaWYgKGVyci5zaWduYWwpIHtcbiAgICAgICAgICAgIHJldHVybiB7c2lnbmFsOiBlcnIuc2lnbmFsfTtcbiAgICAgICAgICB9XG4gICAgICAgICAgcmVqZWN0KGVycik7XG4gICAgICAgICAgcmV0dXJuIHt9O1xuICAgICAgICB9KTtcblxuICAgICAgICBpZiAodGltaW5nKSB7XG4gICAgICAgICAgY29uc3Qge2V4ZWNUaW1lLCBzcGF3blRpbWUsIGlwY1RpbWV9ID0gdGltaW5nO1xuICAgICAgICAgIGNvbnN0IG5vdyA9IHBlcmZvcm1hbmNlLm5vdygpO1xuICAgICAgICAgIHRpbWluZ01hcmtlci5tYXJrKCduZXh0dGljaycsIG5vdyAtIGV4ZWNUaW1lIC0gc3Bhd25UaW1lIC0gaXBjVGltZSk7XG4gICAgICAgICAgdGltaW5nTWFya2VyLm1hcmsoJ2V4ZWN1dGUnLCBub3cgLSBleGVjVGltZSAtIGlwY1RpbWUpO1xuICAgICAgICAgIHRpbWluZ01hcmtlci5tYXJrKCdpcGMnLCBub3cgLSBpcGNUaW1lKTtcbiAgICAgICAgfVxuICAgICAgICB0aW1pbmdNYXJrZXIuZmluYWxpemUoKTtcblxuICAgICAgICAvKiBpc3RhbmJ1bCBpZ25vcmUgaWYgKi9cbiAgICAgICAgaWYgKHByb2Nlc3MuZW52LlBSSU5UX0dJVF9USU1FUykge1xuICAgICAgICAgIGNvbnNvbGUudGltZUVuZChgZ2l0OiR7Zm9ybWF0dGVkQXJnc31gKTtcbiAgICAgICAgfVxuXG4gICAgICAgIGlmIChnaXRQcm9tcHRTZXJ2ZXIpIHtcbiAgICAgICAgICBnaXRQcm9tcHRTZXJ2ZXIudGVybWluYXRlKCk7XG4gICAgICAgIH1cbiAgICAgICAgc3Vic2NyaXB0aW9ucy5kaXNwb3NlKCk7XG5cbiAgICAgICAgLyogaXN0YW5idWwgaWdub3JlIGlmICovXG4gICAgICAgIGlmIChkaWFnbm9zdGljc0VuYWJsZWQpIHtcbiAgICAgICAgICBjb25zdCBleHBvc2VDb250cm9sQ2hhcmFjdGVycyA9IHJhdyA9PiB7XG4gICAgICAgICAgICBpZiAoIXJhdykgeyByZXR1cm4gJyc7IH1cblxuICAgICAgICAgICAgcmV0dXJuIHJhd1xuICAgICAgICAgICAgICAucmVwbGFjZSgvXFx1MDAwMC91ZywgJzxOVUw+XFxuJylcbiAgICAgICAgICAgICAgLnJlcGxhY2UoL1xcdTAwMUYvdWcsICc8U0VQPicpO1xuICAgICAgICAgIH07XG5cbiAgICAgICAgICBpZiAoaGVhZGxlc3MpIHtcbiAgICAgICAgICAgIGxldCBzdW1tYXJ5ID0gYGdpdDoke2Zvcm1hdHRlZEFyZ3N9XFxuYDtcbiAgICAgICAgICAgIGlmIChleGl0Q29kZSAhPT0gdW5kZWZpbmVkKSB7XG4gICAgICAgICAgICAgIHN1bW1hcnkgKz0gYGV4aXQgc3RhdHVzOiAke2V4aXRDb2RlfVxcbmA7XG4gICAgICAgICAgICB9IGVsc2UgaWYgKHNpZ25hbCkge1xuICAgICAgICAgICAgICBzdW1tYXJ5ICs9IGBleGl0IHNpZ25hbDogJHtzaWduYWx9XFxuYDtcbiAgICAgICAgICAgIH1cbiAgICAgICAgICAgIGlmIChzdGRpbiAmJiBzdGRpbi5sZW5ndGggIT09IDApIHtcbiAgICAgICAgICAgICAgc3VtbWFyeSArPSBgc3RkaW46XFxuJHtleHBvc2VDb250cm9sQ2hhcmFjdGVycyhzdGRpbil9XFxuYDtcbiAgICAgICAgICAgIH1cbiAgICAgICAgICAgIHN1bW1hcnkgKz0gJ3N0ZG91dDonO1xuICAgICAgICAgICAgaWYgKHN0ZG91dC5sZW5ndGggPT09IDApIHtcbiAgICAgICAgICAgICAgc3VtbWFyeSArPSAnIDxlbXB0eT5cXG4nO1xuICAgICAgICAgICAgfSBlbHNlIHtcbiAgICAgICAgICAgICAgc3VtbWFyeSArPSBgXFxuJHtleHBvc2VDb250cm9sQ2hhcmFjdGVycyhzdGRvdXQpfVxcbmA7XG4gICAgICAgICAgICB9XG4gICAgICAgICAgICBzdW1tYXJ5ICs9ICdzdGRlcnI6JztcbiAgICAgICAgICAgIGlmIChzdGRlcnIubGVuZ3RoID09PSAwKSB7XG4gICAgICAgICAgICAgIHN1bW1hcnkgKz0gJyA8ZW1wdHk+XFxuJztcbiAgICAgICAgICAgIH0gZWxzZSB7XG4gICAgICAgICAgICAgIHN1bW1hcnkgKz0gYFxcbiR7ZXhwb3NlQ29udHJvbENoYXJhY3RlcnMoc3RkZXJyKX1cXG5gO1xuICAgICAgICAgICAgfVxuXG4gICAgICAgICAgICBjb25zb2xlLmxvZyhzdW1tYXJ5KTtcbiAgICAgICAgICB9IGVsc2Uge1xuICAgICAgICAgICAgY29uc3QgaGVhZGVyU3R5bGUgPSAnZm9udC13ZWlnaHQ6IGJvbGQ7IGNvbG9yOiBibHVlOyc7XG5cbiAgICAgICAgICAgIGNvbnNvbGUuZ3JvdXBDb2xsYXBzZWQoYGdpdDoke2Zvcm1hdHRlZEFyZ3N9YCk7XG4gICAgICAgICAgICBpZiAoZXhpdENvZGUgIT09IHVuZGVmaW5lZCkge1xuICAgICAgICAgICAgICBjb25zb2xlLmxvZygnJWNleGl0IHN0YXR1cyVjICVkJywgaGVhZGVyU3R5bGUsICdmb250LXdlaWdodDogbm9ybWFsOyBjb2xvcjogYmxhY2s7JywgZXhpdENvZGUpO1xuICAgICAgICAgICAgfSBlbHNlIGlmIChzaWduYWwpIHtcbiAgICAgICAgICAgICAgY29uc29sZS5sb2coJyVjZXhpdCBzaWduYWwlYyAlcycsIGhlYWRlclN0eWxlLCAnZm9udC13ZWlnaHQ6IG5vcm1hbDsgY29sb3I6IGJsYWNrOycsIHNpZ25hbCk7XG4gICAgICAgICAgICB9XG4gICAgICAgICAgICBjb25zb2xlLmxvZyhcbiAgICAgICAgICAgICAgJyVjZnVsbCBhcmd1bWVudHMlYyAlcycsXG4gICAgICAgICAgICAgIGhlYWRlclN0eWxlLCAnZm9udC13ZWlnaHQ6IG5vcm1hbDsgY29sb3I6IGJsYWNrOycsXG4gICAgICAgICAgICAgIHV0aWwuaW5zcGVjdChhcmdzLCB7YnJlYWtMZW5ndGg6IEluZmluaXR5fSksXG4gICAgICAgICAgICApO1xuICAgICAgICAgICAgaWYgKHN0ZGluICYmIHN0ZGluLmxlbmd0aCAhPT0gMCkge1xuICAgICAgICAgICAgICBjb25zb2xlLmxvZygnJWNzdGRpbicsIGhlYWRlclN0eWxlKTtcbiAgICAgICAgICAgICAgY29uc29sZS5sb2coZXhwb3NlQ29udHJvbENoYXJhY3RlcnMoc3RkaW4pKTtcbiAgICAgICAgICAgIH1cbiAgICAgICAgICAgIGNvbnNvbGUubG9nKCclY3N0ZG91dCcsIGhlYWRlclN0eWxlKTtcbiAgICAgICAgICAgIGNvbnNvbGUubG9nKGV4cG9zZUNvbnRyb2xDaGFyYWN0ZXJzKHN0ZG91dCkpO1xuICAgICAgICAgICAgY29uc29sZS5sb2coJyVjc3RkZXJyJywgaGVhZGVyU3R5bGUpO1xuICAgICAgICAgICAgY29uc29sZS5sb2coZXhwb3NlQ29udHJvbENoYXJhY3RlcnMoc3RkZXJyKSk7XG4gICAgICAgICAgICBjb25zb2xlLmdyb3VwRW5kKCk7XG4gICAgICAgICAgfVxuICAgICAgICB9XG5cbiAgICAgICAgaWYgKGV4aXRDb2RlICE9PSAwICYmICFleHBlY3RDYW5jZWwpIHtcbiAgICAgICAgICBjb25zdCBlcnIgPSBuZXcgR2l0RXJyb3IoXG4gICAgICAgICAgICBgJHtmb3JtYXR0ZWRBcmdzfSBleGl0ZWQgd2l0aCBjb2RlICR7ZXhpdENvZGV9XFxuc3Rkb3V0OiAke3N0ZG91dH1cXG5zdGRlcnI6ICR7c3RkZXJyfWAsXG4gICAgICAgICAgKTtcbiAgICAgICAgICBlcnIuY29kZSA9IGV4aXRDb2RlO1xuICAgICAgICAgIGVyci5zdGRFcnIgPSBzdGRlcnI7XG4gICAgICAgICAgZXJyLnN0ZE91dCA9IHN0ZG91dDtcbiAgICAgICAgICBlcnIuY29tbWFuZCA9IGZvcm1hdHRlZEFyZ3M7XG4gICAgICAgICAgcmVqZWN0KGVycik7XG4gICAgICAgIH1cblxuICAgICAgICBpZiAoIUlHTk9SRURfR0lUX0NPTU1BTkRTLmluY2x1ZGVzKGNvbW1hbmROYW1lKSkge1xuICAgICAgICAgIGluY3JlbWVudENvdW50ZXIoY29tbWFuZE5hbWUpO1xuICAgICAgICB9XG4gICAgICAgIHJlc29sdmUoc3Rkb3V0KTtcbiAgICAgIH0pO1xuICAgIH0sIHtwYXJhbGxlbDogIXdyaXRlT3BlcmF0aW9ufSk7XG4gICAgLyogZXNsaW50LWVuYWJsZSBuby1jb25zb2xlLG5vLWNvbnRyb2wtcmVnZXggKi9cbiAgfVxuXG4gIGFzeW5jIGdwZ0V4ZWMoYXJncywgb3B0aW9ucykge1xuICAgIHRyeSB7XG4gICAgICByZXR1cm4gYXdhaXQgdGhpcy5leGVjKGFyZ3Muc2xpY2UoKSwge1xuICAgICAgICB1c2VHcGdXcmFwcGVyOiB0cnVlLFxuICAgICAgICB1c2VHcGdBdG9tUHJvbXB0OiBmYWxzZSxcbiAgICAgICAgLi4ub3B0aW9ucyxcbiAgICAgIH0pO1xuICAgIH0gY2F0Y2ggKGUpIHtcbiAgICAgIGlmICgvZ3BnIGZhaWxlZC8udGVzdChlLnN0ZEVycikpIHtcbiAgICAgICAgcmV0dXJuIGF3YWl0IHRoaXMuZXhlYyhhcmdzLCB7XG4gICAgICAgICAgdXNlR2l0UHJvbXB0U2VydmVyOiB0cnVlLFxuICAgICAgICAgIHVzZUdwZ1dyYXBwZXI6IHRydWUsXG4gICAgICAgICAgdXNlR3BnQXRvbVByb21wdDogdHJ1ZSxcbiAgICAgICAgICAuLi5vcHRpb25zLFxuICAgICAgICB9KTtcbiAgICAgIH0gZWxzZSB7XG4gICAgICAgIHRocm93IGU7XG4gICAgICB9XG4gICAgfVxuICB9XG5cbiAgZXhlY3V0ZUdpdENvbW1hbmQoYXJncywgb3B0aW9ucywgbWFya2VyID0gbnVsbCkge1xuICAgIGlmIChwcm9jZXNzLmVudi5BVE9NX0dJVEhVQl9JTkxJTkVfR0lUX0VYRUMgfHwgIVdvcmtlck1hbmFnZXIuZ2V0SW5zdGFuY2UoKS5pc1JlYWR5KCkpIHtcbiAgICAgIG1hcmtlciAmJiBtYXJrZXIubWFyaygnbmV4dHRpY2snKTtcblxuICAgICAgbGV0IGNoaWxkUGlkO1xuICAgICAgb3B0aW9ucy5wcm9jZXNzQ2FsbGJhY2sgPSBjaGlsZCA9PiB7XG4gICAgICAgIGNoaWxkUGlkID0gY2hpbGQucGlkO1xuXG4gICAgICAgIC8qIGlzdGFuYnVsIGlnbm9yZSBuZXh0ICovXG4gICAgICAgIGNoaWxkLnN0ZGluLm9uKCdlcnJvcicsIGVyciA9PiB7XG4gICAgICAgICAgdGhyb3cgbmV3IEVycm9yKFxuICAgICAgICAgICAgYEVycm9yIHdyaXRpbmcgdG8gc3RkaW46IGdpdCAke2FyZ3Muam9pbignICcpfSBpbiAke3RoaXMud29ya2luZ0Rpcn1cXG4ke29wdGlvbnMuc3RkaW59XFxuJHtlcnJ9YCk7XG4gICAgICAgIH0pO1xuICAgICAgfTtcblxuICAgICAgY29uc3QgcHJvbWlzZSA9IEdpdFByb2Nlc3MuZXhlYyhhcmdzLCB0aGlzLndvcmtpbmdEaXIsIG9wdGlvbnMpO1xuICAgICAgbWFya2VyICYmIG1hcmtlci5tYXJrKCdleGVjdXRlJyk7XG4gICAgICByZXR1cm4ge1xuICAgICAgICBwcm9taXNlLFxuICAgICAgICBjYW5jZWw6ICgpID0+IHtcbiAgICAgICAgICAvKiBpc3RhbmJ1bCBpZ25vcmUgaWYgKi9cbiAgICAgICAgICBpZiAoIWNoaWxkUGlkKSB7XG4gICAgICAgICAgICByZXR1cm4gUHJvbWlzZS5yZXNvbHZlKCk7XG4gICAgICAgICAgfVxuXG4gICAgICAgICAgcmV0dXJuIG5ldyBQcm9taXNlKChyZXNvbHZlLCByZWplY3QpID0+IHtcbiAgICAgICAgICAgIHJlcXVpcmUoJ3RyZWUta2lsbCcpKGNoaWxkUGlkLCAnU0lHVEVSTScsIGVyciA9PiB7XG4gICAgICAgICAgICAgIC8qIGlzdGFuYnVsIGlnbm9yZSBpZiAqL1xuICAgICAgICAgICAgICBpZiAoZXJyKSB7IHJlamVjdChlcnIpOyB9IGVsc2UgeyByZXNvbHZlKCk7IH1cbiAgICAgICAgICAgIH0pO1xuICAgICAgICAgIH0pO1xuICAgICAgICB9LFxuICAgICAgfTtcbiAgICB9IGVsc2Uge1xuICAgICAgY29uc3Qgd29ya2VyTWFuYWdlciA9IHRoaXMud29ya2VyTWFuYWdlciB8fCBXb3JrZXJNYW5hZ2VyLmdldEluc3RhbmNlKCk7XG4gICAgICByZXR1cm4gd29ya2VyTWFuYWdlci5yZXF1ZXN0KHtcbiAgICAgICAgYXJncyxcbiAgICAgICAgd29ya2luZ0RpcjogdGhpcy53b3JraW5nRGlyLFxuICAgICAgICBvcHRpb25zLFxuICAgICAgfSk7XG4gICAgfVxuICB9XG5cbiAgYXN5bmMgcmVzb2x2ZURvdEdpdERpcigpIHtcbiAgICB0cnkge1xuICAgICAgYXdhaXQgZnMuc3RhdCh0aGlzLndvcmtpbmdEaXIpOyAvLyBmYWlscyBpZiBmb2xkZXIgZG9lc24ndCBleGlzdFxuICAgICAgY29uc3Qgb3V0cHV0ID0gYXdhaXQgdGhpcy5leGVjKFsncmV2LXBhcnNlJywgJy0tcmVzb2x2ZS1naXQtZGlyJywgcGF0aC5qb2luKHRoaXMud29ya2luZ0RpciwgJy5naXQnKV0pO1xuICAgICAgY29uc3QgZG90R2l0RGlyID0gb3V0cHV0LnRyaW0oKTtcbiAgICAgIHJldHVybiB0b05hdGl2ZVBhdGhTZXAoZG90R2l0RGlyKTtcbiAgICB9IGNhdGNoIChlKSB7XG4gICAgICByZXR1cm4gbnVsbDtcbiAgICB9XG4gIH1cblxuICBpbml0KCkge1xuICAgIHJldHVybiB0aGlzLmV4ZWMoWydpbml0JywgdGhpcy53b3JraW5nRGlyXSk7XG4gIH1cblxuICAvKipcbiAgICogU3RhZ2luZy9VbnN0YWdpbmcgZmlsZXMgYW5kIHBhdGNoZXMgYW5kIGNvbW1pdHRpbmdcbiAgICovXG4gIHN0YWdlRmlsZXMocGF0aHMpIHtcbiAgICBpZiAocGF0aHMubGVuZ3RoID09PSAwKSB7IHJldHVybiBQcm9taXNlLnJlc29sdmUobnVsbCk7IH1cbiAgICBjb25zdCBhcmdzID0gWydhZGQnXS5jb25jYXQocGF0aHMubWFwKHRvR2l0UGF0aFNlcCkpO1xuICAgIHJldHVybiB0aGlzLmV4ZWMoYXJncywge3dyaXRlT3BlcmF0aW9uOiB0cnVlfSk7XG4gIH1cblxuICBhc3luYyBmZXRjaENvbW1pdE1lc3NhZ2VUZW1wbGF0ZSgpIHtcbiAgICBsZXQgdGVtcGxhdGVQYXRoID0gYXdhaXQgdGhpcy5nZXRDb25maWcoJ2NvbW1pdC50ZW1wbGF0ZScpO1xuICAgIGlmICghdGVtcGxhdGVQYXRoKSB7XG4gICAgICByZXR1cm4gbnVsbDtcbiAgICB9XG5cbiAgICBjb25zdCBob21lRGlyID0gb3MuaG9tZWRpcigpO1xuXG4gICAgdGVtcGxhdGVQYXRoID0gdGVtcGxhdGVQYXRoLnRyaW0oKS5yZXBsYWNlKEVYUEFORF9USUxERV9SRUdFWCwgKF8sIHVzZXIpID0+IHtcbiAgICAgIC8vIGlmIG5vIHVzZXIgaXMgc3BlY2lmaWVkLCBmYWxsIGJhY2sgdG8gdXNpbmcgdGhlIGhvbWUgZGlyZWN0b3J5LlxuICAgICAgcmV0dXJuIGAke3VzZXIgPyBwYXRoLmpvaW4ocGF0aC5kaXJuYW1lKGhvbWVEaXIpLCB1c2VyKSA6IGhvbWVEaXJ9L2A7XG4gICAgfSk7XG4gICAgdGVtcGxhdGVQYXRoID0gdG9OYXRpdmVQYXRoU2VwKHRlbXBsYXRlUGF0aCk7XG5cbiAgICBpZiAoIXBhdGguaXNBYnNvbHV0ZSh0ZW1wbGF0ZVBhdGgpKSB7XG4gICAgICB0ZW1wbGF0ZVBhdGggPSBwYXRoLmpvaW4odGhpcy53b3JraW5nRGlyLCB0ZW1wbGF0ZVBhdGgpO1xuICAgIH1cblxuICAgIGlmICghYXdhaXQgZmlsZUV4aXN0cyh0ZW1wbGF0ZVBhdGgpKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoYEludmFsaWQgY29tbWl0IHRlbXBsYXRlIHBhdGggc2V0IGluIEdpdCBjb25maWc6ICR7dGVtcGxhdGVQYXRofWApO1xuICAgIH1cbiAgICByZXR1cm4gYXdhaXQgZnMucmVhZEZpbGUodGVtcGxhdGVQYXRoLCB7ZW5jb2Rpbmc6ICd1dGY4J30pO1xuICB9XG5cbiAgdW5zdGFnZUZpbGVzKHBhdGhzLCBjb21taXQgPSAnSEVBRCcpIHtcbiAgICBpZiAocGF0aHMubGVuZ3RoID09PSAwKSB7IHJldHVybiBQcm9taXNlLnJlc29sdmUobnVsbCk7IH1cbiAgICBjb25zdCBhcmdzID0gWydyZXNldCcsIGNvbW1pdCwgJy0tJ10uY29uY2F0KHBhdGhzLm1hcCh0b0dpdFBhdGhTZXApKTtcbiAgICByZXR1cm4gdGhpcy5leGVjKGFyZ3MsIHt3cml0ZU9wZXJhdGlvbjogdHJ1ZX0pO1xuICB9XG5cbiAgc3RhZ2VGaWxlTW9kZUNoYW5nZShmaWxlbmFtZSwgbmV3TW9kZSkge1xuICAgIGNvbnN0IGluZGV4UmVhZFByb21pc2UgPSB0aGlzLmV4ZWMoWydscy1maWxlcycsICctcycsICctLScsIGZpbGVuYW1lXSk7XG4gICAgcmV0dXJuIHRoaXMuZXhlYyhbJ3VwZGF0ZS1pbmRleCcsICctLWNhY2hlaW5mbycsIGAke25ld01vZGV9LDxPSURfVEJEPiwke2ZpbGVuYW1lfWBdLCB7XG4gICAgICB3cml0ZU9wZXJhdGlvbjogdHJ1ZSxcbiAgICAgIGJlZm9yZVJ1bjogYXN5bmMgZnVuY3Rpb24gZGV0ZXJtaW5lQXJncyh7YXJncywgb3B0c30pIHtcbiAgICAgICAgY29uc3QgaW5kZXggPSBhd2FpdCBpbmRleFJlYWRQcm9taXNlO1xuICAgICAgICBjb25zdCBvaWQgPSBpbmRleC5zdWJzdHIoNywgNDApO1xuICAgICAgICByZXR1cm4ge1xuICAgICAgICAgIG9wdHMsXG4gICAgICAgICAgYXJnczogWyd1cGRhdGUtaW5kZXgnLCAnLS1jYWNoZWluZm8nLCBgJHtuZXdNb2RlfSwke29pZH0sJHtmaWxlbmFtZX1gXSxcbiAgICAgICAgfTtcbiAgICAgIH0sXG4gICAgfSk7XG4gIH1cblxuICBzdGFnZUZpbGVTeW1saW5rQ2hhbmdlKGZpbGVuYW1lKSB7XG4gICAgcmV0dXJuIHRoaXMuZXhlYyhbJ3JtJywgJy0tY2FjaGVkJywgZmlsZW5hbWVdLCB7d3JpdGVPcGVyYXRpb246IHRydWV9KTtcbiAgfVxuXG4gIGFwcGx5UGF0Y2gocGF0Y2gsIHtpbmRleH0gPSB7fSkge1xuICAgIGNvbnN0IGFyZ3MgPSBbJ2FwcGx5JywgJy0nXTtcbiAgICBpZiAoaW5kZXgpIHsgYXJncy5zcGxpY2UoMSwgMCwgJy0tY2FjaGVkJyk7IH1cbiAgICByZXR1cm4gdGhpcy5leGVjKGFyZ3MsIHtzdGRpbjogcGF0Y2gsIHdyaXRlT3BlcmF0aW9uOiB0cnVlfSk7XG4gIH1cblxuICBhc3luYyBjb21taXQocmF3TWVzc2FnZSwge2FsbG93RW1wdHksIGFtZW5kLCBjb0F1dGhvcnMsIHZlcmJhdGltfSA9IHt9KSB7XG4gICAgY29uc3QgYXJncyA9IFsnY29tbWl0J107XG4gICAgbGV0IG1zZztcblxuICAgIC8vIGlmIGFtZW5kaW5nIGFuZCBubyBuZXcgbWVzc2FnZSBpcyBwYXNzZWQsIHVzZSBsYXN0IGNvbW1pdCdzIG1lc3NhZ2UuIEVuc3VyZSB0aGF0IHdlIGRvbid0XG4gICAgLy8gbWFuZ2xlIGl0IGluIHRoZSBwcm9jZXNzLlxuICAgIGlmIChhbWVuZCAmJiByYXdNZXNzYWdlLmxlbmd0aCA9PT0gMCkge1xuICAgICAgY29uc3Qge3VuYm9yblJlZiwgbWVzc2FnZUJvZHksIG1lc3NhZ2VTdWJqZWN0fSA9IGF3YWl0IHRoaXMuZ2V0SGVhZENvbW1pdCgpO1xuICAgICAgaWYgKHVuYm9yblJlZikge1xuICAgICAgICBtc2cgPSByYXdNZXNzYWdlO1xuICAgICAgfSBlbHNlIHtcbiAgICAgICAgbXNnID0gYCR7bWVzc2FnZVN1YmplY3R9XFxuXFxuJHttZXNzYWdlQm9keX1gLnRyaW0oKTtcbiAgICAgICAgdmVyYmF0aW0gPSB0cnVlO1xuICAgICAgfVxuICAgIH0gZWxzZSB7XG4gICAgICBtc2cgPSByYXdNZXNzYWdlO1xuICAgIH1cblxuICAgIC8vIGlmIGNvbW1pdCB0ZW1wbGF0ZSBpcyB1c2VkLCBzdHJpcCBjb21tZW50ZWQgbGluZXMgZnJvbSBjb21taXRcbiAgICAvLyB0byBiZSBjb25zaXN0ZW50IHdpdGggY29tbWFuZCBsaW5lIGdpdC5cbiAgICBjb25zdCB0ZW1wbGF0ZSA9IGF3YWl0IHRoaXMuZmV0Y2hDb21taXRNZXNzYWdlVGVtcGxhdGUoKTtcbiAgICBpZiAodGVtcGxhdGUpIHtcblxuICAgICAgLy8gcmVzcGVjdGluZyB0aGUgY29tbWVudCBjaGFyYWN0ZXIgZnJvbSB1c2VyIHNldHRpbmdzIG9yIGZhbGwgYmFjayB0byAjIGFzIGRlZmF1bHQuXG4gICAgICAvLyBodHRwczovL2dpdC1zY20uY29tL2RvY3MvZ2l0LWNvbmZpZyNnaXQtY29uZmlnLWNvcmVjb21tZW50Q2hhclxuICAgICAgbGV0IGNvbW1lbnRDaGFyID0gYXdhaXQgdGhpcy5nZXRDb25maWcoJ2NvcmUuY29tbWVudENoYXInKTtcbiAgICAgIGlmICghY29tbWVudENoYXIpIHtcbiAgICAgICAgY29tbWVudENoYXIgPSAnIyc7XG4gICAgICB9XG4gICAgICBtc2cgPSBtc2cuc3BsaXQoJ1xcbicpLmZpbHRlcihsaW5lID0+ICFsaW5lLnN0YXJ0c1dpdGgoY29tbWVudENoYXIpKS5qb2luKCdcXG4nKTtcbiAgICB9XG5cbiAgICAvLyBEZXRlcm1pbmUgdGhlIGNsZWFudXAgbW9kZS5cbiAgICBpZiAodmVyYmF0aW0pIHtcbiAgICAgIGFyZ3MucHVzaCgnLS1jbGVhbnVwPXZlcmJhdGltJyk7XG4gICAgfSBlbHNlIHtcbiAgICAgIGNvbnN0IGNvbmZpZ3VyZWQgPSBhd2FpdCB0aGlzLmdldENvbmZpZygnY29tbWl0LmNsZWFudXAnKTtcbiAgICAgIGNvbnN0IG1vZGUgPSAoY29uZmlndXJlZCAmJiBjb25maWd1cmVkICE9PSAnZGVmYXVsdCcpID8gY29uZmlndXJlZCA6ICdzdHJpcCc7XG4gICAgICBhcmdzLnB1c2goYC0tY2xlYW51cD0ke21vZGV9YCk7XG4gICAgfVxuXG4gICAgLy8gYWRkIGNvLWF1dGhvciBjb21taXQgdHJhaWxlcnMgaWYgbmVjZXNzYXJ5XG4gICAgaWYgKGNvQXV0aG9ycyAmJiBjb0F1dGhvcnMubGVuZ3RoID4gMCkge1xuICAgICAgbXNnID0gYXdhaXQgdGhpcy5hZGRDb0F1dGhvcnNUb01lc3NhZ2UobXNnLCBjb0F1dGhvcnMpO1xuICAgIH1cblxuICAgIGFyZ3MucHVzaCgnLW0nLCBtc2cudHJpbSgpKTtcblxuICAgIGlmIChhbWVuZCkgeyBhcmdzLnB1c2goJy0tYW1lbmQnKTsgfVxuICAgIGlmIChhbGxvd0VtcHR5KSB7IGFyZ3MucHVzaCgnLS1hbGxvdy1lbXB0eScpOyB9XG4gICAgcmV0dXJuIHRoaXMuZ3BnRXhlYyhhcmdzLCB7d3JpdGVPcGVyYXRpb246IHRydWV9KTtcbiAgfVxuXG4gIGFkZENvQXV0aG9yc1RvTWVzc2FnZShtZXNzYWdlLCBjb0F1dGhvcnMgPSBbXSkge1xuICAgIGNvbnN0IHRyYWlsZXJzID0gY29BdXRob3JzLm1hcChhdXRob3IgPT4ge1xuICAgICAgcmV0dXJuIHtcbiAgICAgICAgdG9rZW46ICdDby1BdXRob3JlZC1CeScsXG4gICAgICAgIHZhbHVlOiBgJHthdXRob3IubmFtZX0gPCR7YXV0aG9yLmVtYWlsfT5gLFxuICAgICAgfTtcbiAgICB9KTtcblxuICAgIC8vIEVuc3VyZSB0aGF0IG1lc3NhZ2UgZW5kcyB3aXRoIG5ld2xpbmUgZm9yIGdpdC1pbnRlcnByZXQgdHJhaWxlcnMgdG8gd29ya1xuICAgIGNvbnN0IG1zZyA9IGAke21lc3NhZ2UudHJpbSgpfVxcbmA7XG5cbiAgICByZXR1cm4gdHJhaWxlcnMubGVuZ3RoID8gdGhpcy5tZXJnZVRyYWlsZXJzKG1zZywgdHJhaWxlcnMpIDogbXNnO1xuICB9XG5cbiAgLyoqXG4gICAqIEZpbGUgU3RhdHVzIGFuZCBEaWZmc1xuICAgKi9cbiAgYXN5bmMgZ2V0U3RhdHVzQnVuZGxlKCkge1xuICAgIGNvbnN0IGFyZ3MgPSBbJ3N0YXR1cycsICctLXBvcmNlbGFpbj12MicsICctLWJyYW5jaCcsICctLXVudHJhY2tlZC1maWxlcz1hbGwnLCAnLS1pZ25vcmUtc3VibW9kdWxlcz1kaXJ0eScsICcteiddO1xuICAgIGNvbnN0IG91dHB1dCA9IGF3YWl0IHRoaXMuZXhlYyhhcmdzKTtcbiAgICBpZiAob3V0cHV0Lmxlbmd0aCA+IE1BWF9TVEFUVVNfT1VUUFVUX0xFTkdUSCkge1xuICAgICAgdGhyb3cgbmV3IExhcmdlUmVwb0Vycm9yKCk7XG4gICAgfVxuXG4gICAgY29uc3QgcmVzdWx0cyA9IGF3YWl0IHBhcnNlU3RhdHVzKG91dHB1dCk7XG5cbiAgICBmb3IgKGNvbnN0IGVudHJ5VHlwZSBpbiByZXN1bHRzKSB7XG4gICAgICBpZiAoQXJyYXkuaXNBcnJheShyZXN1bHRzW2VudHJ5VHlwZV0pKSB7XG4gICAgICAgIHRoaXMudXBkYXRlTmF0aXZlUGF0aFNlcEZvckVudHJpZXMocmVzdWx0c1tlbnRyeVR5cGVdKTtcbiAgICAgIH1cbiAgICB9XG5cbiAgICByZXR1cm4gcmVzdWx0cztcbiAgfVxuXG4gIHVwZGF0ZU5hdGl2ZVBhdGhTZXBGb3JFbnRyaWVzKGVudHJpZXMpIHtcbiAgICBlbnRyaWVzLmZvckVhY2goZW50cnkgPT4ge1xuICAgICAgLy8gTm9ybWFsbHkgd2Ugd291bGQgYXZvaWQgbXV0YXRpbmcgcmVzcG9uc2VzIGZyb20gb3RoZXIgcGFja2FnZSdzIEFQSXMsIGJ1dCB3ZSBjb250cm9sXG4gICAgICAvLyB0aGUgYHdoYXQtdGhlLXN0YXR1c2AgbW9kdWxlIGFuZCBrbm93IHRoZXJlIGFyZSBubyBzaWRlIGVmZmVjdHMuXG4gICAgICAvLyBUaGlzIGlzIGEgaG90IGNvZGUgcGF0aCBhbmQgYnkgbXV0YXRpbmcgd2UgYXZvaWQgY3JlYXRpbmcgbmV3IG9iamVjdHMgdGhhdCB3aWxsIGp1c3QgYmUgR0MnZWRcbiAgICAgIGlmIChlbnRyeS5maWxlUGF0aCkge1xuICAgICAgICBlbnRyeS5maWxlUGF0aCA9IHRvTmF0aXZlUGF0aFNlcChlbnRyeS5maWxlUGF0aCk7XG4gICAgICB9XG4gICAgICBpZiAoZW50cnkub3JpZ0ZpbGVQYXRoKSB7XG4gICAgICAgIGVudHJ5Lm9yaWdGaWxlUGF0aCA9IHRvTmF0aXZlUGF0aFNlcChlbnRyeS5vcmlnRmlsZVBhdGgpO1xuICAgICAgfVxuICAgIH0pO1xuICB9XG5cbiAgYXN5bmMgZGlmZkZpbGVTdGF0dXMob3B0aW9ucyA9IHt9KSB7XG4gICAgY29uc3QgYXJncyA9IFsnZGlmZicsICctLW5hbWUtc3RhdHVzJywgJy0tbm8tcmVuYW1lcyddO1xuICAgIGlmIChvcHRpb25zLnN0YWdlZCkgeyBhcmdzLnB1c2goJy0tc3RhZ2VkJyk7IH1cbiAgICBpZiAob3B0aW9ucy50YXJnZXQpIHsgYXJncy5wdXNoKG9wdGlvbnMudGFyZ2V0KTsgfVxuICAgIGNvbnN0IG91dHB1dCA9IGF3YWl0IHRoaXMuZXhlYyhhcmdzKTtcblxuICAgIGNvbnN0IHN0YXR1c01hcCA9IHtcbiAgICAgIEE6ICdhZGRlZCcsXG4gICAgICBNOiAnbW9kaWZpZWQnLFxuICAgICAgRDogJ2RlbGV0ZWQnLFxuICAgICAgVTogJ3VubWVyZ2VkJyxcbiAgICB9O1xuXG4gICAgY29uc3QgZmlsZVN0YXR1c2VzID0ge307XG4gICAgb3V0cHV0ICYmIG91dHB1dC50cmltKCkuc3BsaXQoTElORV9FTkRJTkdfUkVHRVgpLmZvckVhY2gobGluZSA9PiB7XG4gICAgICBjb25zdCBbc3RhdHVzLCByYXdGaWxlUGF0aF0gPSBsaW5lLnNwbGl0KCdcXHQnKTtcbiAgICAgIGNvbnN0IGZpbGVQYXRoID0gdG9OYXRpdmVQYXRoU2VwKHJhd0ZpbGVQYXRoKTtcbiAgICAgIGZpbGVTdGF0dXNlc1tmaWxlUGF0aF0gPSBzdGF0dXNNYXBbc3RhdHVzXTtcbiAgICB9KTtcbiAgICBpZiAoIW9wdGlvbnMuc3RhZ2VkKSB7XG4gICAgICBjb25zdCB1bnRyYWNrZWQgPSBhd2FpdCB0aGlzLmdldFVudHJhY2tlZEZpbGVzKCk7XG4gICAgICB1bnRyYWNrZWQuZm9yRWFjaChmaWxlUGF0aCA9PiB7IGZpbGVTdGF0dXNlc1tmaWxlUGF0aF0gPSAnYWRkZWQnOyB9KTtcbiAgICB9XG4gICAgcmV0dXJuIGZpbGVTdGF0dXNlcztcbiAgfVxuXG4gIGFzeW5jIGdldFVudHJhY2tlZEZpbGVzKCkge1xuICAgIGNvbnN0IG91dHB1dCA9IGF3YWl0IHRoaXMuZXhlYyhbJ2xzLWZpbGVzJywgJy0tb3RoZXJzJywgJy0tZXhjbHVkZS1zdGFuZGFyZCddKTtcbiAgICBpZiAob3V0cHV0LnRyaW0oKSA9PT0gJycpIHsgcmV0dXJuIFtdOyB9XG4gICAgcmV0dXJuIG91dHB1dC50cmltKCkuc3BsaXQoTElORV9FTkRJTkdfUkVHRVgpLm1hcCh0b05hdGl2ZVBhdGhTZXApO1xuICB9XG5cbiAgYXN5bmMgZ2V0RGlmZnNGb3JGaWxlUGF0aChmaWxlUGF0aCwge3N0YWdlZCwgYmFzZUNvbW1pdH0gPSB7fSkge1xuICAgIGxldCBhcmdzID0gWydkaWZmJywgJy0tbm8tcHJlZml4JywgJy0tbm8tZXh0LWRpZmYnLCAnLS1uby1yZW5hbWVzJywgJy0tZGlmZi1maWx0ZXI9dSddO1xuICAgIGlmIChzdGFnZWQpIHsgYXJncy5wdXNoKCctLXN0YWdlZCcpOyB9XG4gICAgaWYgKGJhc2VDb21taXQpIHsgYXJncy5wdXNoKGJhc2VDb21taXQpOyB9XG4gICAgYXJncyA9IGFyZ3MuY29uY2F0KFsnLS0nLCB0b0dpdFBhdGhTZXAoZmlsZVBhdGgpXSk7XG4gICAgY29uc3Qgb3V0cHV0ID0gYXdhaXQgdGhpcy5leGVjKGFyZ3MpO1xuXG4gICAgbGV0IHJhd0RpZmZzID0gW107XG4gICAgaWYgKG91dHB1dCkge1xuICAgICAgcmF3RGlmZnMgPSBwYXJzZURpZmYob3V0cHV0KVxuICAgICAgICAuZmlsdGVyKHJhd0RpZmYgPT4gcmF3RGlmZi5zdGF0dXMgIT09ICd1bm1lcmdlZCcpO1xuXG4gICAgICBmb3IgKGxldCBpID0gMDsgaSA8IHJhd0RpZmZzLmxlbmd0aDsgaSsrKSB7XG4gICAgICAgIGNvbnN0IHJhd0RpZmYgPSByYXdEaWZmc1tpXTtcbiAgICAgICAgaWYgKHJhd0RpZmYub2xkUGF0aCkge1xuICAgICAgICAgIHJhd0RpZmYub2xkUGF0aCA9IHRvTmF0aXZlUGF0aFNlcChyYXdEaWZmLm9sZFBhdGgpO1xuICAgICAgICB9XG4gICAgICAgIGlmIChyYXdEaWZmLm5ld1BhdGgpIHtcbiAgICAgICAgICByYXdEaWZmLm5ld1BhdGggPSB0b05hdGl2ZVBhdGhTZXAocmF3RGlmZi5uZXdQYXRoKTtcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH1cblxuICAgIGlmICghc3RhZ2VkICYmIChhd2FpdCB0aGlzLmdldFVudHJhY2tlZEZpbGVzKCkpLmluY2x1ZGVzKGZpbGVQYXRoKSkge1xuICAgICAgLy8gYWRkIHVudHJhY2tlZCBmaWxlXG4gICAgICBjb25zdCBhYnNQYXRoID0gcGF0aC5qb2luKHRoaXMud29ya2luZ0RpciwgZmlsZVBhdGgpO1xuICAgICAgY29uc3QgZXhlY3V0YWJsZSA9IGF3YWl0IGlzRmlsZUV4ZWN1dGFibGUoYWJzUGF0aCk7XG4gICAgICBjb25zdCBzeW1saW5rID0gYXdhaXQgaXNGaWxlU3ltbGluayhhYnNQYXRoKTtcbiAgICAgIGNvbnN0IGNvbnRlbnRzID0gYXdhaXQgZnMucmVhZEZpbGUoYWJzUGF0aCwge2VuY29kaW5nOiAndXRmOCd9KTtcbiAgICAgIGNvbnN0IGJpbmFyeSA9IGlzQmluYXJ5KGNvbnRlbnRzKTtcbiAgICAgIGxldCBtb2RlO1xuICAgICAgbGV0IHJlYWxwYXRoO1xuICAgICAgaWYgKGV4ZWN1dGFibGUpIHtcbiAgICAgICAgbW9kZSA9IEZpbGUubW9kZXMuRVhFQ1VUQUJMRTtcbiAgICAgIH0gZWxzZSBpZiAoc3ltbGluaykge1xuICAgICAgICBtb2RlID0gRmlsZS5tb2Rlcy5TWU1MSU5LO1xuICAgICAgICByZWFscGF0aCA9IGF3YWl0IGZzLnJlYWxwYXRoKGFic1BhdGgpO1xuICAgICAgfSBlbHNlIHtcbiAgICAgICAgbW9kZSA9IEZpbGUubW9kZXMuTk9STUFMO1xuICAgICAgfVxuXG4gICAgICByYXdEaWZmcy5wdXNoKGJ1aWxkQWRkZWRGaWxlUGF0Y2goZmlsZVBhdGgsIGJpbmFyeSA/IG51bGwgOiBjb250ZW50cywgbW9kZSwgcmVhbHBhdGgpKTtcbiAgICB9XG4gICAgaWYgKHJhd0RpZmZzLmxlbmd0aCA+IDIpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihgRXhwZWN0ZWQgYmV0d2VlbiAwIGFuZCAyIGRpZmZzIGZvciAke2ZpbGVQYXRofSBidXQgZ290ICR7cmF3RGlmZnMubGVuZ3RofWApO1xuICAgIH1cbiAgICByZXR1cm4gcmF3RGlmZnM7XG4gIH1cblxuICBhc3luYyBnZXRTdGFnZWRDaGFuZ2VzUGF0Y2goKSB7XG4gICAgY29uc3Qgb3V0cHV0ID0gYXdhaXQgdGhpcy5leGVjKFtcbiAgICAgICdkaWZmJywgJy0tc3RhZ2VkJywgJy0tbm8tcHJlZml4JywgJy0tbm8tZXh0LWRpZmYnLCAnLS1uby1yZW5hbWVzJywgJy0tZGlmZi1maWx0ZXI9dScsXG4gICAgXSk7XG5cbiAgICBpZiAoIW91dHB1dCkge1xuICAgICAgcmV0dXJuIFtdO1xuICAgIH1cblxuICAgIGNvbnN0IGRpZmZzID0gcGFyc2VEaWZmKG91dHB1dCk7XG4gICAgZm9yIChjb25zdCBkaWZmIG9mIGRpZmZzKSB7XG4gICAgICBpZiAoZGlmZi5vbGRQYXRoKSB7IGRpZmYub2xkUGF0aCA9IHRvTmF0aXZlUGF0aFNlcChkaWZmLm9sZFBhdGgpOyB9XG4gICAgICBpZiAoZGlmZi5uZXdQYXRoKSB7IGRpZmYubmV3UGF0aCA9IHRvTmF0aXZlUGF0aFNlcChkaWZmLm5ld1BhdGgpOyB9XG4gICAgfVxuICAgIHJldHVybiBkaWZmcztcbiAgfVxuXG4gIC8qKlxuICAgKiBNaXNjZWxsYW5lb3VzIGdldHRlcnNcbiAgICovXG4gIGFzeW5jIGdldENvbW1pdChyZWYpIHtcbiAgICBjb25zdCBbY29tbWl0XSA9IGF3YWl0IHRoaXMuZ2V0Q29tbWl0cyh7bWF4OiAxLCByZWYsIGluY2x1ZGVVbmJvcm46IHRydWV9KTtcbiAgICByZXR1cm4gY29tbWl0O1xuICB9XG5cbiAgYXN5bmMgZ2V0SGVhZENvbW1pdCgpIHtcbiAgICBjb25zdCBbaGVhZENvbW1pdF0gPSBhd2FpdCB0aGlzLmdldENvbW1pdHMoe21heDogMSwgcmVmOiAnSEVBRCcsIGluY2x1ZGVVbmJvcm46IHRydWV9KTtcbiAgICByZXR1cm4gaGVhZENvbW1pdDtcbiAgfVxuXG4gIGFzeW5jIGdldENvbW1pdHMob3B0aW9ucyA9IHt9KSB7XG4gICAgY29uc3Qge21heCwgcmVmLCBpbmNsdWRlVW5ib3JuLCBpbmNsdWRlUGF0Y2h9ID0ge1xuICAgICAgbWF4OiAxLFxuICAgICAgcmVmOiAnSEVBRCcsXG4gICAgICBpbmNsdWRlVW5ib3JuOiBmYWxzZSxcbiAgICAgIGluY2x1ZGVQYXRjaDogZmFsc2UsXG4gICAgICAuLi5vcHRpb25zLFxuICAgIH07XG5cbiAgICAvLyBodHRwczovL2dpdC1zY20uY29tL2RvY3MvZ2l0LWxvZyNfcHJldHR5X2Zvcm1hdHNcbiAgICAvLyAleDAwIC0gbnVsbCBieXRlXG4gICAgLy8gJUggLSBjb21taXQgU0hBXG4gICAgLy8gJWFlIC0gYXV0aG9yIGVtYWlsXG4gICAgLy8gJWFuID0gYXV0aG9yIGZ1bGwgbmFtZVxuICAgIC8vICVhdCAtIHRpbWVzdGFtcCwgVU5JWCB0aW1lc3RhbXBcbiAgICAvLyAlcyAtIHN1YmplY3RcbiAgICAvLyAlYiAtIGJvZHlcbiAgICBjb25zdCBhcmdzID0gW1xuICAgICAgJ2xvZycsXG4gICAgICAnLS1wcmV0dHk9Zm9ybWF0OiVIJXgwMCVhZSV4MDAlYW4leDAwJWF0JXgwMCVzJXgwMCViJXgwMCcsXG4gICAgICAnLS1uby1hYmJyZXYtY29tbWl0JyxcbiAgICAgICctLW5vLXByZWZpeCcsXG4gICAgICAnLS1uby1leHQtZGlmZicsXG4gICAgICAnLS1uby1yZW5hbWVzJyxcbiAgICAgICcteicsXG4gICAgICAnLW4nLFxuICAgICAgbWF4LFxuICAgICAgcmVmLFxuICAgIF07XG5cbiAgICBpZiAoaW5jbHVkZVBhdGNoKSB7XG4gICAgICBhcmdzLnB1c2goJy0tcGF0Y2gnLCAnLW0nLCAnLS1maXJzdC1wYXJlbnQnKTtcbiAgICB9XG5cbiAgICBjb25zdCBvdXRwdXQgPSBhd2FpdCB0aGlzLmV4ZWMoYXJncy5jb25jYXQoJy0tJykpLmNhdGNoKGVyciA9PiB7XG4gICAgICBpZiAoL3Vua25vd24gcmV2aXNpb24vLnRlc3QoZXJyLnN0ZEVycikgfHwgL2JhZCByZXZpc2lvbiAnSEVBRCcvLnRlc3QoZXJyLnN0ZEVycikpIHtcbiAgICAgICAgcmV0dXJuICcnO1xuICAgICAgfSBlbHNlIHtcbiAgICAgICAgdGhyb3cgZXJyO1xuICAgICAgfVxuICAgIH0pO1xuXG4gICAgaWYgKG91dHB1dCA9PT0gJycpIHtcbiAgICAgIHJldHVybiBpbmNsdWRlVW5ib3JuID8gW3tzaGE6ICcnLCBtZXNzYWdlOiAnJywgdW5ib3JuUmVmOiB0cnVlfV0gOiBbXTtcbiAgICB9XG5cbiAgICBjb25zdCBmaWVsZHMgPSBvdXRwdXQudHJpbSgpLnNwbGl0KCdcXDAnKTtcblxuICAgIGNvbnN0IGNvbW1pdHMgPSBbXTtcbiAgICBmb3IgKGxldCBpID0gMDsgaSA8IGZpZWxkcy5sZW5ndGg7IGkgKz0gNykge1xuICAgICAgY29uc3QgYm9keSA9IGZpZWxkc1tpICsgNV0udHJpbSgpO1xuICAgICAgbGV0IHBhdGNoID0gW107XG4gICAgICBpZiAoaW5jbHVkZVBhdGNoKSB7XG4gICAgICAgIGNvbnN0IGRpZmZzID0gZmllbGRzW2kgKyA2XTtcbiAgICAgICAgcGF0Y2ggPSBwYXJzZURpZmYoZGlmZnMudHJpbSgpKTtcbiAgICAgIH1cblxuICAgICAgY29uc3Qge21lc3NhZ2U6IG1lc3NhZ2VCb2R5LCBjb0F1dGhvcnN9ID0gZXh0cmFjdENvQXV0aG9yc0FuZFJhd0NvbW1pdE1lc3NhZ2UoYm9keSk7XG5cbiAgICAgIGNvbW1pdHMucHVzaCh7XG4gICAgICAgIHNoYTogZmllbGRzW2ldICYmIGZpZWxkc1tpXS50cmltKCksXG4gICAgICAgIGF1dGhvcjogbmV3IEF1dGhvcihmaWVsZHNbaSArIDFdICYmIGZpZWxkc1tpICsgMV0udHJpbSgpLCBmaWVsZHNbaSArIDJdICYmIGZpZWxkc1tpICsgMl0udHJpbSgpKSxcbiAgICAgICAgYXV0aG9yRGF0ZTogcGFyc2VJbnQoZmllbGRzW2kgKyAzXSwgMTApLFxuICAgICAgICBtZXNzYWdlU3ViamVjdDogZmllbGRzW2kgKyA0XSxcbiAgICAgICAgbWVzc2FnZUJvZHksXG4gICAgICAgIGNvQXV0aG9ycyxcbiAgICAgICAgdW5ib3JuUmVmOiBmYWxzZSxcbiAgICAgICAgcGF0Y2gsXG4gICAgICB9KTtcbiAgICB9XG4gICAgcmV0dXJuIGNvbW1pdHM7XG4gIH1cblxuICBhc3luYyBnZXRBdXRob3JzKG9wdGlvbnMgPSB7fSkge1xuICAgIGNvbnN0IHttYXgsIHJlZn0gPSB7bWF4OiAxLCByZWY6ICdIRUFEJywgLi4ub3B0aW9uc307XG5cbiAgICAvLyBodHRwczovL2dpdC1zY20uY29tL2RvY3MvZ2l0LWxvZyNfcHJldHR5X2Zvcm1hdHNcbiAgICAvLyAleDFGIC0gZmllbGQgc2VwYXJhdG9yIGJ5dGVcbiAgICAvLyAlYW4gLSBhdXRob3IgbmFtZVxuICAgIC8vICVhZSAtIGF1dGhvciBlbWFpbFxuICAgIC8vICVjbiAtIGNvbW1pdHRlciBuYW1lXG4gICAgLy8gJWNlIC0gY29tbWl0dGVyIGVtYWlsXG4gICAgLy8gJSh0cmFpbGVyczp1bmZvbGQsb25seSkgLSB0aGUgY29tbWl0IG1lc3NhZ2UgdHJhaWxlcnMsIHNlcGFyYXRlZFxuICAgIC8vICAgICAgICAgICAgICAgICAgICAgICAgICAgYnkgbmV3bGluZXMgYW5kIHVuZm9sZGVkIChpLmUuIHByb3Blcmx5XG4gICAgLy8gICAgICAgICAgICAgICAgICAgICAgICAgICBmb3JtYXR0ZWQgYW5kIG9uZSB0cmFpbGVyIHBlciBsaW5lKS5cblxuICAgIGNvbnN0IGRlbGltaXRlciA9ICcxRic7XG4gICAgY29uc3QgZGVsaW1pdGVyU3RyaW5nID0gU3RyaW5nLmZyb21DaGFyQ29kZShwYXJzZUludChkZWxpbWl0ZXIsIDE2KSk7XG4gICAgY29uc3QgZmllbGRzID0gWyclYW4nLCAnJWFlJywgJyVjbicsICclY2UnLCAnJSh0cmFpbGVyczp1bmZvbGQsb25seSknXTtcbiAgICBjb25zdCBmb3JtYXQgPSBmaWVsZHMuam9pbihgJXgke2RlbGltaXRlcn1gKTtcblxuICAgIHRyeSB7XG4gICAgICBjb25zdCBvdXRwdXQgPSBhd2FpdCB0aGlzLmV4ZWMoW1xuICAgICAgICAnbG9nJywgYC0tZm9ybWF0PSR7Zm9ybWF0fWAsICcteicsICctbicsIG1heCwgcmVmLCAnLS0nLFxuICAgICAgXSk7XG5cbiAgICAgIHJldHVybiBvdXRwdXQuc3BsaXQoJ1xcMCcpXG4gICAgICAgIC5yZWR1Y2UoKGFjYywgbGluZSkgPT4ge1xuICAgICAgICAgIGlmIChsaW5lLmxlbmd0aCA9PT0gMCkgeyByZXR1cm4gYWNjOyB9XG5cbiAgICAgICAgICBjb25zdCBbYW4sIGFlLCBjbiwgY2UsIHRyYWlsZXJzXSA9IGxpbmUuc3BsaXQoZGVsaW1pdGVyU3RyaW5nKTtcbiAgICAgICAgICB0cmFpbGVyc1xuICAgICAgICAgICAgLnNwbGl0KCdcXG4nKVxuICAgICAgICAgICAgLm1hcCh0cmFpbGVyID0+IHRyYWlsZXIubWF0Y2goQ09fQVVUSE9SX1JFR0VYKSlcbiAgICAgICAgICAgIC5maWx0ZXIobWF0Y2ggPT4gbWF0Y2ggIT09IG51bGwpXG4gICAgICAgICAgICAuZm9yRWFjaCgoW18sIG5hbWUsIGVtYWlsXSkgPT4geyBhY2NbZW1haWxdID0gbmFtZTsgfSk7XG5cbiAgICAgICAgICBhY2NbYWVdID0gYW47XG4gICAgICAgICAgYWNjW2NlXSA9IGNuO1xuXG4gICAgICAgICAgcmV0dXJuIGFjYztcbiAgICAgICAgfSwge30pO1xuICAgIH0gY2F0Y2ggKGVycikge1xuICAgICAgaWYgKC91bmtub3duIHJldmlzaW9uLy50ZXN0KGVyci5zdGRFcnIpIHx8IC9iYWQgcmV2aXNpb24gJ0hFQUQnLy50ZXN0KGVyci5zdGRFcnIpKSB7XG4gICAgICAgIHJldHVybiBbXTtcbiAgICAgIH0gZWxzZSB7XG4gICAgICAgIHRocm93IGVycjtcbiAgICAgIH1cbiAgICB9XG4gIH1cblxuICBtZXJnZVRyYWlsZXJzKGNvbW1pdE1lc3NhZ2UsIHRyYWlsZXJzKSB7XG4gICAgY29uc3QgYXJncyA9IFsnaW50ZXJwcmV0LXRyYWlsZXJzJ107XG4gICAgZm9yIChjb25zdCB0cmFpbGVyIG9mIHRyYWlsZXJzKSB7XG4gICAgICBhcmdzLnB1c2goJy0tdHJhaWxlcicsIGAke3RyYWlsZXIudG9rZW59PSR7dHJhaWxlci52YWx1ZX1gKTtcbiAgICB9XG4gICAgcmV0dXJuIHRoaXMuZXhlYyhhcmdzLCB7c3RkaW46IGNvbW1pdE1lc3NhZ2V9KTtcbiAgfVxuXG4gIHJlYWRGaWxlRnJvbUluZGV4KGZpbGVQYXRoKSB7XG4gICAgcmV0dXJuIHRoaXMuZXhlYyhbJ3Nob3cnLCBgOiR7dG9HaXRQYXRoU2VwKGZpbGVQYXRoKX1gXSk7XG4gIH1cblxuICAvKipcbiAgICogTWVyZ2VcbiAgICovXG4gIG1lcmdlKGJyYW5jaE5hbWUpIHtcbiAgICByZXR1cm4gdGhpcy5ncGdFeGVjKFsnbWVyZ2UnLCBicmFuY2hOYW1lXSwge3dyaXRlT3BlcmF0aW9uOiB0cnVlfSk7XG4gIH1cblxuICBpc01lcmdpbmcoZG90R2l0RGlyKSB7XG4gICAgcmV0dXJuIGZpbGVFeGlzdHMocGF0aC5qb2luKGRvdEdpdERpciwgJ01FUkdFX0hFQUQnKSkuY2F0Y2goKCkgPT4gZmFsc2UpO1xuICB9XG5cbiAgYWJvcnRNZXJnZSgpIHtcbiAgICByZXR1cm4gdGhpcy5leGVjKFsnbWVyZ2UnLCAnLS1hYm9ydCddLCB7d3JpdGVPcGVyYXRpb246IHRydWV9KTtcbiAgfVxuXG4gIGNoZWNrb3V0U2lkZShzaWRlLCBwYXRocykge1xuICAgIGlmIChwYXRocy5sZW5ndGggPT09IDApIHtcbiAgICAgIHJldHVybiBQcm9taXNlLnJlc29sdmUoKTtcbiAgICB9XG5cbiAgICByZXR1cm4gdGhpcy5leGVjKFsnY2hlY2tvdXQnLCBgLS0ke3NpZGV9YCwgLi4ucGF0aHMubWFwKHRvR2l0UGF0aFNlcCldKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBSZWJhc2VcbiAgICovXG4gIGFzeW5jIGlzUmViYXNpbmcoZG90R2l0RGlyKSB7XG4gICAgY29uc3QgcmVzdWx0cyA9IGF3YWl0IFByb21pc2UuYWxsKFtcbiAgICAgIGZpbGVFeGlzdHMocGF0aC5qb2luKGRvdEdpdERpciwgJ3JlYmFzZS1tZXJnZScpKSxcbiAgICAgIGZpbGVFeGlzdHMocGF0aC5qb2luKGRvdEdpdERpciwgJ3JlYmFzZS1hcHBseScpKSxcbiAgICBdKTtcbiAgICByZXR1cm4gcmVzdWx0cy5zb21lKHIgPT4gcik7XG4gIH1cblxuICAvKipcbiAgICogUmVtb3RlIGludGVyYWN0aW9uc1xuICAgKi9cbiAgY2xvbmUocmVtb3RlVXJsLCBvcHRpb25zID0ge30pIHtcbiAgICBjb25zdCBhcmdzID0gWydjbG9uZSddO1xuICAgIGlmIChvcHRpb25zLm5vTG9jYWwpIHsgYXJncy5wdXNoKCctLW5vLWxvY2FsJyk7IH1cbiAgICBpZiAob3B0aW9ucy5iYXJlKSB7IGFyZ3MucHVzaCgnLS1iYXJlJyk7IH1cbiAgICBpZiAob3B0aW9ucy5yZWN1cnNpdmUpIHsgYXJncy5wdXNoKCctLXJlY3Vyc2l2ZScpOyB9XG4gICAgaWYgKG9wdGlvbnMuc291cmNlUmVtb3RlTmFtZSkgeyBhcmdzLnB1c2goJy0tb3JpZ2luJywgb3B0aW9ucy5yZW1vdGVOYW1lKTsgfVxuICAgIGFyZ3MucHVzaChyZW1vdGVVcmwsIHRoaXMud29ya2luZ0Rpcik7XG5cbiAgICByZXR1cm4gdGhpcy5leGVjKGFyZ3MsIHt1c2VHaXRQcm9tcHRTZXJ2ZXI6IHRydWUsIHdyaXRlT3BlcmF0aW9uOiB0cnVlfSk7XG4gIH1cblxuICBmZXRjaChyZW1vdGVOYW1lLCBicmFuY2hOYW1lKSB7XG4gICAgcmV0dXJuIHRoaXMuZXhlYyhbJ2ZldGNoJywgcmVtb3RlTmFtZSwgYnJhbmNoTmFtZV0sIHt1c2VHaXRQcm9tcHRTZXJ2ZXI6IHRydWUsIHdyaXRlT3BlcmF0aW9uOiB0cnVlfSk7XG4gIH1cblxuICBwdWxsKHJlbW90ZU5hbWUsIGJyYW5jaE5hbWUsIG9wdGlvbnMgPSB7fSkge1xuICAgIGNvbnN0IGFyZ3MgPSBbJ3B1bGwnLCByZW1vdGVOYW1lLCBvcHRpb25zLnJlZlNwZWMgfHwgYnJhbmNoTmFtZV07XG4gICAgaWYgKG9wdGlvbnMuZmZPbmx5KSB7XG4gICAgICBhcmdzLnB1c2goJy0tZmYtb25seScpO1xuICAgIH1cbiAgICByZXR1cm4gdGhpcy5ncGdFeGVjKGFyZ3MsIHt1c2VHaXRQcm9tcHRTZXJ2ZXI6IHRydWUsIHdyaXRlT3BlcmF0aW9uOiB0cnVlfSk7XG4gIH1cblxuICBwdXNoKHJlbW90ZU5hbWUsIGJyYW5jaE5hbWUsIG9wdGlvbnMgPSB7fSkge1xuICAgIGNvbnN0IGFyZ3MgPSBbJ3B1c2gnLCByZW1vdGVOYW1lIHx8ICdvcmlnaW4nLCBvcHRpb25zLnJlZlNwZWMgfHwgYHJlZnMvaGVhZHMvJHticmFuY2hOYW1lfWBdO1xuICAgIGlmIChvcHRpb25zLnNldFVwc3RyZWFtKSB7IGFyZ3MucHVzaCgnLS1zZXQtdXBzdHJlYW0nKTsgfVxuICAgIGlmIChvcHRpb25zLmZvcmNlKSB7IGFyZ3MucHVzaCgnLS1mb3JjZScpOyB9XG4gICAgcmV0dXJuIHRoaXMuZXhlYyhhcmdzLCB7dXNlR2l0UHJvbXB0U2VydmVyOiB0cnVlLCB3cml0ZU9wZXJhdGlvbjogdHJ1ZX0pO1xuICB9XG5cbiAgLyoqXG4gICAqIFVuZG8gT3BlcmF0aW9uc1xuICAgKi9cbiAgcmVzZXQodHlwZSwgcmV2aXNpb24gPSAnSEVBRCcpIHtcbiAgICBjb25zdCB2YWxpZFR5cGVzID0gWydzb2Z0J107XG4gICAgaWYgKCF2YWxpZFR5cGVzLmluY2x1ZGVzKHR5cGUpKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoYEludmFsaWQgdHlwZSAke3R5cGV9LiBNdXN0IGJlIG9uZSBvZjogJHt2YWxpZFR5cGVzLmpvaW4oJywgJyl9YCk7XG4gICAgfVxuICAgIHJldHVybiB0aGlzLmV4ZWMoWydyZXNldCcsIGAtLSR7dHlwZX1gLCByZXZpc2lvbl0pO1xuICB9XG5cbiAgZGVsZXRlUmVmKHJlZikge1xuICAgIHJldHVybiB0aGlzLmV4ZWMoWyd1cGRhdGUtcmVmJywgJy1kJywgcmVmXSk7XG4gIH1cblxuICAvKipcbiAgICogQnJhbmNoZXNcbiAgICovXG4gIGNoZWNrb3V0KGJyYW5jaE5hbWUsIG9wdGlvbnMgPSB7fSkge1xuICAgIGNvbnN0IGFyZ3MgPSBbJ2NoZWNrb3V0J107XG4gICAgaWYgKG9wdGlvbnMuY3JlYXRlTmV3KSB7XG4gICAgICBhcmdzLnB1c2goJy1iJyk7XG4gICAgfVxuICAgIGFyZ3MucHVzaChicmFuY2hOYW1lKTtcbiAgICBpZiAob3B0aW9ucy5zdGFydFBvaW50KSB7XG4gICAgICBpZiAob3B0aW9ucy50cmFjaykgeyBhcmdzLnB1c2goJy0tdHJhY2snKTsgfVxuICAgICAgYXJncy5wdXNoKG9wdGlvbnMuc3RhcnRQb2ludCk7XG4gICAgfVxuXG4gICAgcmV0dXJuIHRoaXMuZXhlYyhhcmdzLCB7d3JpdGVPcGVyYXRpb246IHRydWV9KTtcbiAgfVxuXG4gIGFzeW5jIGdldEJyYW5jaGVzKCkge1xuICAgIGNvbnN0IGZvcm1hdCA9IFtcbiAgICAgICclKG9iamVjdG5hbWUpJywgJyUoSEVBRCknLCAnJShyZWZuYW1lOnNob3J0KScsXG4gICAgICAnJSh1cHN0cmVhbSknLCAnJSh1cHN0cmVhbTpyZW1vdGVuYW1lKScsICclKHVwc3RyZWFtOnJlbW90ZXJlZiknLFxuICAgICAgJyUocHVzaCknLCAnJShwdXNoOnJlbW90ZW5hbWUpJywgJyUocHVzaDpyZW1vdGVyZWYpJyxcbiAgICBdLmpvaW4oJyUwMCcpO1xuXG4gICAgY29uc3Qgb3V0cHV0ID0gYXdhaXQgdGhpcy5leGVjKFsnZm9yLWVhY2gtcmVmJywgYC0tZm9ybWF0PSR7Zm9ybWF0fWAsICdyZWZzL2hlYWRzLyoqJ10pO1xuICAgIHJldHVybiBvdXRwdXQudHJpbSgpLnNwbGl0KExJTkVfRU5ESU5HX1JFR0VYKS5tYXAobGluZSA9PiB7XG4gICAgICBjb25zdCBbXG4gICAgICAgIHNoYSwgaGVhZCwgbmFtZSxcbiAgICAgICAgdXBzdHJlYW1UcmFja2luZ1JlZiwgdXBzdHJlYW1SZW1vdGVOYW1lLCB1cHN0cmVhbVJlbW90ZVJlZixcbiAgICAgICAgcHVzaFRyYWNraW5nUmVmLCBwdXNoUmVtb3RlTmFtZSwgcHVzaFJlbW90ZVJlZixcbiAgICAgIF0gPSBsaW5lLnNwbGl0KCdcXDAnKTtcblxuICAgICAgY29uc3QgYnJhbmNoID0ge25hbWUsIHNoYSwgaGVhZDogaGVhZCA9PT0gJyonfTtcbiAgICAgIGlmICh1cHN0cmVhbVRyYWNraW5nUmVmIHx8IHVwc3RyZWFtUmVtb3RlTmFtZSB8fCB1cHN0cmVhbVJlbW90ZVJlZikge1xuICAgICAgICBicmFuY2gudXBzdHJlYW0gPSB7XG4gICAgICAgICAgdHJhY2tpbmdSZWY6IHVwc3RyZWFtVHJhY2tpbmdSZWYsXG4gICAgICAgICAgcmVtb3RlTmFtZTogdXBzdHJlYW1SZW1vdGVOYW1lLFxuICAgICAgICAgIHJlbW90ZVJlZjogdXBzdHJlYW1SZW1vdGVSZWYsXG4gICAgICAgIH07XG4gICAgICB9XG4gICAgICBpZiAoYnJhbmNoLnVwc3RyZWFtIHx8IHB1c2hUcmFja2luZ1JlZiB8fCBwdXNoUmVtb3RlTmFtZSB8fCBwdXNoUmVtb3RlUmVmKSB7XG4gICAgICAgIGJyYW5jaC5wdXNoID0ge1xuICAgICAgICAgIHRyYWNraW5nUmVmOiBwdXNoVHJhY2tpbmdSZWYsXG4gICAgICAgICAgcmVtb3RlTmFtZTogcHVzaFJlbW90ZU5hbWUgfHwgKGJyYW5jaC51cHN0cmVhbSAmJiBicmFuY2gudXBzdHJlYW0ucmVtb3RlTmFtZSksXG4gICAgICAgICAgcmVtb3RlUmVmOiBwdXNoUmVtb3RlUmVmIHx8IChicmFuY2gudXBzdHJlYW0gJiYgYnJhbmNoLnVwc3RyZWFtLnJlbW90ZVJlZiksXG4gICAgICAgIH07XG4gICAgICB9XG4gICAgICByZXR1cm4gYnJhbmNoO1xuICAgIH0pO1xuICB9XG5cbiAgYXN5bmMgZ2V0QnJhbmNoZXNXaXRoQ29tbWl0KHNoYSwgb3B0aW9uID0ge30pIHtcbiAgICBjb25zdCBhcmdzID0gWydicmFuY2gnLCAnLS1mb3JtYXQ9JShyZWZuYW1lKScsICctLWNvbnRhaW5zJywgc2hhXTtcbiAgICBpZiAob3B0aW9uLnNob3dMb2NhbCAmJiBvcHRpb24uc2hvd1JlbW90ZSkge1xuICAgICAgYXJncy5zcGxpY2UoMSwgMCwgJy0tYWxsJyk7XG4gICAgfSBlbHNlIGlmIChvcHRpb24uc2hvd1JlbW90ZSkge1xuICAgICAgYXJncy5zcGxpY2UoMSwgMCwgJy0tcmVtb3RlcycpO1xuICAgIH1cbiAgICBpZiAob3B0aW9uLnBhdHRlcm4pIHtcbiAgICAgIGFyZ3MucHVzaChvcHRpb24ucGF0dGVybik7XG4gICAgfVxuICAgIHJldHVybiAoYXdhaXQgdGhpcy5leGVjKGFyZ3MpKS50cmltKCkuc3BsaXQoTElORV9FTkRJTkdfUkVHRVgpO1xuICB9XG5cbiAgY2hlY2tvdXRGaWxlcyhwYXRocywgcmV2aXNpb24pIHtcbiAgICBpZiAocGF0aHMubGVuZ3RoID09PSAwKSB7IHJldHVybiBudWxsOyB9XG4gICAgY29uc3QgYXJncyA9IFsnY2hlY2tvdXQnXTtcbiAgICBpZiAocmV2aXNpb24pIHsgYXJncy5wdXNoKHJldmlzaW9uKTsgfVxuICAgIHJldHVybiB0aGlzLmV4ZWMoYXJncy5jb25jYXQoJy0tJywgcGF0aHMubWFwKHRvR2l0UGF0aFNlcCkpLCB7d3JpdGVPcGVyYXRpb246IHRydWV9KTtcbiAgfVxuXG4gIGFzeW5jIGRlc2NyaWJlSGVhZCgpIHtcbiAgICByZXR1cm4gKGF3YWl0IHRoaXMuZXhlYyhbJ2Rlc2NyaWJlJywgJy0tY29udGFpbnMnLCAnLS1hbGwnLCAnLS1hbHdheXMnLCAnSEVBRCddKSkudHJpbSgpO1xuICB9XG5cbiAgYXN5bmMgZ2V0Q29uZmlnKG9wdGlvbiwge2xvY2FsfSA9IHt9KSB7XG4gICAgbGV0IG91dHB1dDtcbiAgICB0cnkge1xuICAgICAgbGV0IGFyZ3MgPSBbJ2NvbmZpZyddO1xuICAgICAgaWYgKGxvY2FsKSB7IGFyZ3MucHVzaCgnLS1sb2NhbCcpOyB9XG4gICAgICBhcmdzID0gYXJncy5jb25jYXQob3B0aW9uKTtcbiAgICAgIG91dHB1dCA9IGF3YWl0IHRoaXMuZXhlYyhhcmdzKTtcbiAgICB9IGNhdGNoIChlcnIpIHtcbiAgICAgIGlmIChlcnIuY29kZSA9PT0gMSB8fCBlcnIuY29kZSA9PT0gMTI4KSB7XG4gICAgICAgIC8vIE5vIG1hdGNoaW5nIGNvbmZpZyBmb3VuZCBPUiAtLWxvY2FsIGNhbiBvbmx5IGJlIHVzZWQgaW5zaWRlIGEgZ2l0IHJlcG9zaXRvcnlcbiAgICAgICAgcmV0dXJuIG51bGw7XG4gICAgICB9IGVsc2Uge1xuICAgICAgICB0aHJvdyBlcnI7XG4gICAgICB9XG4gICAgfVxuXG4gICAgcmV0dXJuIG91dHB1dC50cmltKCk7XG4gIH1cblxuICBzZXRDb25maWcob3B0aW9uLCB2YWx1ZSwge3JlcGxhY2VBbGwsIGdsb2JhbH0gPSB7fSkge1xuICAgIGxldCBhcmdzID0gWydjb25maWcnXTtcbiAgICBpZiAocmVwbGFjZUFsbCkgeyBhcmdzLnB1c2goJy0tcmVwbGFjZS1hbGwnKTsgfVxuICAgIGlmIChnbG9iYWwpIHsgYXJncy5wdXNoKCctLWdsb2JhbCcpOyB9XG4gICAgYXJncyA9IGFyZ3MuY29uY2F0KG9wdGlvbiwgdmFsdWUpO1xuICAgIHJldHVybiB0aGlzLmV4ZWMoYXJncywge3dyaXRlT3BlcmF0aW9uOiB0cnVlfSk7XG4gIH1cblxuICB1bnNldENvbmZpZyhvcHRpb24pIHtcbiAgICByZXR1cm4gdGhpcy5leGVjKFsnY29uZmlnJywgJy0tdW5zZXQnLCBvcHRpb25dLCB7d3JpdGVPcGVyYXRpb246IHRydWV9KTtcbiAgfVxuXG4gIGFzeW5jIGdldFJlbW90ZXMoKSB7XG4gICAgbGV0IG91dHB1dCA9IGF3YWl0IHRoaXMuZ2V0Q29uZmlnKFsnLS1nZXQtcmVnZXhwJywgJ15yZW1vdGVcXFxcLi4qXFxcXC51cmwkJ10sIHtsb2NhbDogdHJ1ZX0pO1xuICAgIGlmIChvdXRwdXQpIHtcbiAgICAgIG91dHB1dCA9IG91dHB1dC50cmltKCk7XG4gICAgICBpZiAoIW91dHB1dC5sZW5ndGgpIHsgcmV0dXJuIFtdOyB9XG4gICAgICByZXR1cm4gb3V0cHV0LnNwbGl0KCdcXG4nKS5tYXAobGluZSA9PiB7XG4gICAgICAgIGNvbnN0IG1hdGNoID0gbGluZS5tYXRjaCgvXnJlbW90ZVxcLiguKilcXC51cmwgKC4qKSQvKTtcbiAgICAgICAgcmV0dXJuIHtcbiAgICAgICAgICBuYW1lOiBtYXRjaFsxXSxcbiAgICAgICAgICB1cmw6IG1hdGNoWzJdLFxuICAgICAgICB9O1xuICAgICAgfSk7XG4gICAgfSBlbHNlIHtcbiAgICAgIHJldHVybiBbXTtcbiAgICB9XG4gIH1cblxuICBhZGRSZW1vdGUobmFtZSwgdXJsKSB7XG4gICAgcmV0dXJuIHRoaXMuZXhlYyhbJ3JlbW90ZScsICdhZGQnLCBuYW1lLCB1cmxdKTtcbiAgfVxuXG4gIGFzeW5jIGNyZWF0ZUJsb2Ioe2ZpbGVQYXRoLCBzdGRpbn0gPSB7fSkge1xuICAgIGxldCBvdXRwdXQ7XG4gICAgaWYgKGZpbGVQYXRoKSB7XG4gICAgICB0cnkge1xuICAgICAgICBvdXRwdXQgPSAoYXdhaXQgdGhpcy5leGVjKFsnaGFzaC1vYmplY3QnLCAnLXcnLCBmaWxlUGF0aF0sIHt3cml0ZU9wZXJhdGlvbjogdHJ1ZX0pKS50cmltKCk7XG4gICAgICB9IGNhdGNoIChlKSB7XG4gICAgICAgIGlmIChlLnN0ZEVyciAmJiBlLnN0ZEVyci5tYXRjaCgvZmF0YWw6IENhbm5vdCBvcGVuIC4qOiBObyBzdWNoIGZpbGUgb3IgZGlyZWN0b3J5LykpIHtcbiAgICAgICAgICBvdXRwdXQgPSBudWxsO1xuICAgICAgICB9IGVsc2Uge1xuICAgICAgICAgIHRocm93IGU7XG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9IGVsc2UgaWYgKHN0ZGluKSB7XG4gICAgICBvdXRwdXQgPSAoYXdhaXQgdGhpcy5leGVjKFsnaGFzaC1vYmplY3QnLCAnLXcnLCAnLS1zdGRpbiddLCB7c3RkaW4sIHdyaXRlT3BlcmF0aW9uOiB0cnVlfSkpLnRyaW0oKTtcbiAgICB9IGVsc2Uge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdNdXN0IHN1cHBseSBmaWxlIHBhdGggb3Igc3RkaW4nKTtcbiAgICB9XG4gICAgcmV0dXJuIG91dHB1dDtcbiAgfVxuXG4gIGFzeW5jIGV4cGFuZEJsb2JUb0ZpbGUoYWJzRmlsZVBhdGgsIHNoYSkge1xuICAgIGNvbnN0IG91dHB1dCA9IGF3YWl0IHRoaXMuZXhlYyhbJ2NhdC1maWxlJywgJy1wJywgc2hhXSk7XG4gICAgYXdhaXQgZnMud3JpdGVGaWxlKGFic0ZpbGVQYXRoLCBvdXRwdXQsIHtlbmNvZGluZzogJ3V0ZjgnfSk7XG4gICAgcmV0dXJuIGFic0ZpbGVQYXRoO1xuICB9XG5cbiAgYXN5bmMgZ2V0QmxvYkNvbnRlbnRzKHNoYSkge1xuICAgIHJldHVybiBhd2FpdCB0aGlzLmV4ZWMoWydjYXQtZmlsZScsICctcCcsIHNoYV0pO1xuICB9XG5cbiAgYXN5bmMgbWVyZ2VGaWxlKG91cnNQYXRoLCBjb21tb25CYXNlUGF0aCwgdGhlaXJzUGF0aCwgcmVzdWx0UGF0aCkge1xuICAgIGNvbnN0IGFyZ3MgPSBbXG4gICAgICAnbWVyZ2UtZmlsZScsICctcCcsIG91cnNQYXRoLCBjb21tb25CYXNlUGF0aCwgdGhlaXJzUGF0aCxcbiAgICAgICctTCcsICdjdXJyZW50JywgJy1MJywgJ2FmdGVyIGRpc2NhcmQnLCAnLUwnLCAnYmVmb3JlIGRpc2NhcmQnLFxuICAgIF07XG4gICAgbGV0IG91dHB1dDtcbiAgICBsZXQgY29uZmxpY3QgPSBmYWxzZTtcbiAgICB0cnkge1xuICAgICAgb3V0cHV0ID0gYXdhaXQgdGhpcy5leGVjKGFyZ3MpO1xuICAgIH0gY2F0Y2ggKGUpIHtcbiAgICAgIGlmIChlIGluc3RhbmNlb2YgR2l0RXJyb3IgJiYgZS5jb2RlID09PSAxKSB7XG4gICAgICAgIG91dHB1dCA9IGUuc3RkT3V0O1xuICAgICAgICBjb25mbGljdCA9IHRydWU7XG4gICAgICB9IGVsc2Uge1xuICAgICAgICB0aHJvdyBlO1xuICAgICAgfVxuICAgIH1cblxuICAgIC8vIEludGVycHJldCBhIHJlbGF0aXZlIHJlc3VsdFBhdGggYXMgcmVsYXRpdmUgdG8gdGhlIHJlcG9zaXRvcnkgd29ya2luZyBkaXJlY3RvcnkgZm9yIGNvbnNpc3RlbmN5IHdpdGggdGhlXG4gICAgLy8gb3RoZXIgYXJndW1lbnRzLlxuICAgIGNvbnN0IHJlc29sdmVkUmVzdWx0UGF0aCA9IHBhdGgucmVzb2x2ZSh0aGlzLndvcmtpbmdEaXIsIHJlc3VsdFBhdGgpO1xuICAgIGF3YWl0IGZzLndyaXRlRmlsZShyZXNvbHZlZFJlc3VsdFBhdGgsIG91dHB1dCwge2VuY29kaW5nOiAndXRmOCd9KTtcblxuICAgIHJldHVybiB7ZmlsZVBhdGg6IG91cnNQYXRoLCByZXN1bHRQYXRoLCBjb25mbGljdH07XG4gIH1cblxuICBhc3luYyB3cml0ZU1lcmdlQ29uZmxpY3RUb0luZGV4KGZpbGVQYXRoLCBjb21tb25CYXNlU2hhLCBvdXJzU2hhLCB0aGVpcnNTaGEpIHtcbiAgICBjb25zdCBnaXRGaWxlUGF0aCA9IHRvR2l0UGF0aFNlcChmaWxlUGF0aCk7XG4gICAgY29uc3QgZmlsZU1vZGUgPSBhd2FpdCB0aGlzLmdldEZpbGVNb2RlKGZpbGVQYXRoKTtcbiAgICBsZXQgaW5kZXhJbmZvID0gYDAgMDAwMDAwMDAwMDAwMDAwMDAwMDAwMDAwMDAwMDAwMDAwMDAwMDAwMFxcdCR7Z2l0RmlsZVBhdGh9XFxuYDtcbiAgICBpZiAoY29tbW9uQmFzZVNoYSkgeyBpbmRleEluZm8gKz0gYCR7ZmlsZU1vZGV9ICR7Y29tbW9uQmFzZVNoYX0gMVxcdCR7Z2l0RmlsZVBhdGh9XFxuYDsgfVxuICAgIGlmIChvdXJzU2hhKSB7IGluZGV4SW5mbyArPSBgJHtmaWxlTW9kZX0gJHtvdXJzU2hhfSAyXFx0JHtnaXRGaWxlUGF0aH1cXG5gOyB9XG4gICAgaWYgKHRoZWlyc1NoYSkgeyBpbmRleEluZm8gKz0gYCR7ZmlsZU1vZGV9ICR7dGhlaXJzU2hhfSAzXFx0JHtnaXRGaWxlUGF0aH1cXG5gOyB9XG4gICAgcmV0dXJuIHRoaXMuZXhlYyhbJ3VwZGF0ZS1pbmRleCcsICctLWluZGV4LWluZm8nXSwge3N0ZGluOiBpbmRleEluZm8sIHdyaXRlT3BlcmF0aW9uOiB0cnVlfSk7XG4gIH1cblxuICBhc3luYyBnZXRGaWxlTW9kZShmaWxlUGF0aCkge1xuICAgIGNvbnN0IG91dHB1dCA9IGF3YWl0IHRoaXMuZXhlYyhbJ2xzLWZpbGVzJywgJy0tc3RhZ2UnLCAnLS0nLCB0b0dpdFBhdGhTZXAoZmlsZVBhdGgpXSk7XG4gICAgaWYgKG91dHB1dCkge1xuICAgICAgcmV0dXJuIG91dHB1dC5zbGljZSgwLCA2KTtcbiAgICB9IGVsc2Uge1xuICAgICAgY29uc3QgZXhlY3V0YWJsZSA9IGF3YWl0IGlzRmlsZUV4ZWN1dGFibGUocGF0aC5qb2luKHRoaXMud29ya2luZ0RpciwgZmlsZVBhdGgpKTtcbiAgICAgIGNvbnN0IHN5bWxpbmsgPSBhd2FpdCBpc0ZpbGVTeW1saW5rKHBhdGguam9pbih0aGlzLndvcmtpbmdEaXIsIGZpbGVQYXRoKSk7XG4gICAgICBpZiAoc3ltbGluaykge1xuICAgICAgICByZXR1cm4gRmlsZS5tb2Rlcy5TWU1MSU5LO1xuICAgICAgfSBlbHNlIGlmIChleGVjdXRhYmxlKSB7XG4gICAgICAgIHJldHVybiBGaWxlLm1vZGVzLkVYRUNVVEFCTEU7XG4gICAgICB9IGVsc2Uge1xuICAgICAgICByZXR1cm4gRmlsZS5tb2Rlcy5OT1JNQUw7XG4gICAgICB9XG4gICAgfVxuICB9XG5cbiAgZGVzdHJveSgpIHtcbiAgICB0aGlzLmNvbW1hbmRRdWV1ZS5kaXNwb3NlKCk7XG4gIH1cbn1cblxuZnVuY3Rpb24gYnVpbGRBZGRlZEZpbGVQYXRjaChmaWxlUGF0aCwgY29udGVudHMsIG1vZGUsIHJlYWxwYXRoKSB7XG4gIGNvbnN0IGh1bmtzID0gW107XG4gIGlmIChjb250ZW50cykge1xuICAgIGxldCBub05ld0xpbmU7XG4gICAgbGV0IGxpbmVzO1xuICAgIGlmIChtb2RlID09PSBGaWxlLm1vZGVzLlNZTUxJTkspIHtcbiAgICAgIG5vTmV3TGluZSA9IGZhbHNlO1xuICAgICAgbGluZXMgPSBbYCske3RvR2l0UGF0aFNlcChyZWFscGF0aCl9YCwgJ1xcXFwgTm8gbmV3bGluZSBhdCBlbmQgb2YgZmlsZSddO1xuICAgIH0gZWxzZSB7XG4gICAgICBub05ld0xpbmUgPSBjb250ZW50c1tjb250ZW50cy5sZW5ndGggLSAxXSAhPT0gJ1xcbic7XG4gICAgICBsaW5lcyA9IGNvbnRlbnRzLnRyaW0oKS5zcGxpdChMSU5FX0VORElOR19SRUdFWCkubWFwKGxpbmUgPT4gYCske2xpbmV9YCk7XG4gICAgfVxuICAgIGlmIChub05ld0xpbmUpIHsgbGluZXMucHVzaCgnXFxcXCBObyBuZXdsaW5lIGF0IGVuZCBvZiBmaWxlJyk7IH1cbiAgICBodW5rcy5wdXNoKHtcbiAgICAgIGxpbmVzLFxuICAgICAgb2xkU3RhcnRMaW5lOiAwLFxuICAgICAgb2xkTGluZUNvdW50OiAwLFxuICAgICAgbmV3U3RhcnRMaW5lOiAxLFxuICAgICAgaGVhZGluZzogJycsXG4gICAgICBuZXdMaW5lQ291bnQ6IG5vTmV3TGluZSA/IGxpbmVzLmxlbmd0aCAtIDEgOiBsaW5lcy5sZW5ndGgsXG4gICAgfSk7XG4gIH1cbiAgcmV0dXJuIHtcbiAgICBvbGRQYXRoOiBudWxsLFxuICAgIG5ld1BhdGg6IHRvTmF0aXZlUGF0aFNlcChmaWxlUGF0aCksXG4gICAgb2xkTW9kZTogbnVsbCxcbiAgICBuZXdNb2RlOiBtb2RlLFxuICAgIHN0YXR1czogJ2FkZGVkJyxcbiAgICBodW5rcyxcbiAgfTtcbn1cbiJdfQ==