"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

var _eventKit = require("event-kit");

let key = 0;

class StubItem {
  // StubItems should only be created by `create` and never constructed directly.
  static create(name, props, uri = '') {
    const stub = new StubItem(name, props, uri);
    const override = {
      _getStub: () => stub,
      getElement: () => stub.getElement(),
      destroy: stub.destroy.bind(stub)
    };
    const proxy = new Proxy(override, {
      get(target, propName) {
        const item = stub.getRealItem();

        if (Reflect.has(target, propName)) {
          return target[propName];
        } else if (item && Reflect.has(item, propName)) {
          let val = item[propName];

          if (typeof val === 'function') {
            val = val.bind(item);
          }

          return val;
        } else {
          let val = stub[propName];

          if (typeof val === 'function') {
            val = val.bind(stub);
          }

          return val;
        }
      }

    });
    return proxy;
  }

  constructor(name, props = {}, uri) {
    this.emitter = new _eventKit.Emitter();
    this.subscriptions = new _eventKit.CompositeDisposable();
    this.name = name;
    this.props = props;
    this.uri = uri;
    this.key = ++key;
    this.element = document.createElement('div');
    this.element.classList.add(`github-StubItem-${name}`);
    this.realItem = null;
    this.realItemPromise = new Promise(res => {
      this.resolveRealItemPromise = res;
    });
  }

  setRealItem(item) {
    this.realItem = item;

    if (this.realItem.getRealItemPromise) {
      this.realItem.getRealItemPromise().then(this.resolveRealItemPromise);
    } else {
      this.resolveRealItemPromise(this.realItem);
    }

    this.emitter.emit('did-change-title');
    this.emitter.emit('did-change-icon');

    if (item.onDidChangeTitle) {
      this.subscriptions.add(item.onDidChangeTitle((...args) => this.emitter.emit('did-change-title', ...args)));
    }

    if (item.onDidChangeIcon) {
      this.subscriptions.add(item.onDidChangeIcon((...args) => this.emitter.emit('did-change-icon', ...args)));
    }

    if (item.onDidDestroy) {
      this.subscriptions.add(item.onDidDestroy((...args) => {
        this.realItem = null;
        this.emitter.emit('did-destroy', ...args);
      }));
    }
  }

  getRealItemPromise() {
    return this.realItemPromise;
  }

  getRealItem() {
    return this.realItem;
  }

  getURI() {
    return this.uri;
  }

  getTitle() {
    return this.props.title || null;
  }

  getIconName() {
    return this.props.iconName || null;
  }

  onDidChangeTitle(cb) {
    return this.emitter.on('did-change-title', cb);
  }

  onDidChangeIcon(cb) {
    return this.emitter.on('did-change-icon', cb);
  }

  getElement() {
    return this.element;
  }

  onDidDestroy(cb) {
    return this.emitter.on('did-destroy', cb);
  }

  destroy() {
    this.resolveRealItemPromise(null);
    this.subscriptions.dispose();
    this.emitter.dispose();

    if (this.realItem) {
      this.realItem.destroy && this.realItem.destroy();
    }
  }

}

exports.default = StubItem;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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