"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.nullAuthor = exports.default = exports.NO_REPLY_GITHUB_EMAIL = void 0;
const NEW = Symbol('new');
const NO_REPLY_GITHUB_EMAIL = 'noreply@github.com';
exports.NO_REPLY_GITHUB_EMAIL = NO_REPLY_GITHUB_EMAIL;

class Author {
  constructor(email, fullName, login = null, isNew = null, avatarUrl = null) {
    if (avatarUrl == null) {
      const match = (email || '').match(/^(\d+)\+[^@]+@users.noreply.github.com$/);

      if (match) {
        avatarUrl = 'https://avatars.githubusercontent.com/u/' + match[1] + '?s=32';
      } else if (email && email !== '') {
        avatarUrl = 'https://avatars.githubusercontent.com/u/e?email=' + encodeURIComponent(email) + '&s=32';
      } else {
        avatarUrl = '';
      }
    }

    this.email = email;
    this.fullName = fullName;
    this.login = login;
    this.new = isNew === NEW;
    this.avatarUrl = avatarUrl;
  }

  static createNew(email, fullName) {
    return new this(email, fullName, null, NEW);
  }

  getEmail() {
    return this.email;
  }

  getAvatarUrl() {
    return this.avatarUrl;
  }

  getFullName() {
    return this.fullName;
  }

  getLogin() {
    return this.login;
  }

  isNoReply() {
    return this.email === NO_REPLY_GITHUB_EMAIL;
  }

  hasLogin() {
    return this.login !== null;
  }

  isNew() {
    return this.new;
  }

  isPresent() {
    return true;
  }

  matches(other) {
    return this.getEmail() === other.getEmail();
  }

  toString() {
    let s = `${this.fullName} <${this.email}>`;

    if (this.hasLogin()) {
      s += ` @${this.login}`;
    }

    return s;
  }

  static compare(a, b) {
    if (a.getFullName() < b.getFullName()) {
      return -1;
    }

    if (a.getFullName() > b.getFullName()) {
      return 1;
    }

    return 0;
  }

}

exports.default = Author;
const nullAuthor = {
  getEmail() {
    return '';
  },

  getAvatarUrl() {
    return '';
  },

  getFullName() {
    return '';
  },

  getLogin() {
    return null;
  },

  isNoReply() {
    return false;
  },

  hasLogin() {
    return false;
  },

  isNew() {
    return false;
  },

  isPresent() {
    return false;
  },

  matches(other) {
    return other === this;
  },

  toString() {
    return 'null author';
  }

};
exports.nullAuthor = nullAuthor;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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