"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

var _util = _interopRequireDefault(require("util"));

var _branch = require("./branch");

var _helpers = require("../helpers");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

// Store and index a set of Branches in a repository.
class BranchSet {
  constructor(all = []) {
    this.all = [];
    this.head = _branch.nullBranch;
    this.byUpstreamRef = new Map();
    this.byPushRef = new Map();

    for (const branch of all) {
      this.add(branch);
    }
  }

  add(branch) {
    this.all.push(branch);

    if (branch.isHead()) {
      this.head = branch;
    }

    const u = branch.getUpstream();

    if (u.isPresent() && u.isRemoteTracking()) {
      const k = `${u.getRemoteName()}\0${u.getRemoteRef()}`;
      (0, _helpers.pushAtKey)(this.byUpstreamRef, k, branch);
    }

    const p = branch.getPush();

    if (p.isPresent() && p.isRemoteTracking()) {
      const k = `${p.getRemoteName()}\0${p.getRemoteRef()}`;
      (0, _helpers.pushAtKey)(this.byPushRef, k, branch);
    }
  }

  getNames() {
    return this.all.map(branch => branch.getName());
  } // Return the HEAD branch, or `nullBranch` if HEAD is not a branch. This can happen if HEAD is unborn (the repository
  // was just initialized) or if HEAD is detached.


  getHeadBranch() {
    return this.head;
  } // Return an Array of Branches that would be updated from a given remote ref with a `git pull`. This corresponds with
  // git's notion of an _upstream_ and takes into account the current `branch.remote` setting and `remote.<name>.fetch`
  // refspec.


  getPullTargets(remoteName, remoteRefName) {
    return this.byUpstreamRef.get(`${remoteName}\0${remoteRefName}`) || [];
  } // Return an Array of Branches that will update a given remote ref on an unqualified `git push`. This accounts for
  // the current `branch.pushRemote` setting and `remote.<name>.push` refspec.


  getPushSources(remoteName, remoteRefName) {
    return this.byPushRef.get(`${remoteName}\0${remoteRefName}`) || [];
  }

  inspect(depth, options) {
    return `BranchSet {${_util.default.inspect(this.all)}}`;
  }

}

exports.default = BranchSet;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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