"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.nullBranch = exports.default = void 0;
const DETACHED = Symbol('detached');
const REMOTE_TRACKING = Symbol('remote-tracking');

class Branch {
  constructor(name, upstream = nullBranch, push = upstream, head = false, attributes = {}) {
    this.name = name;
    this.upstream = upstream;
    this.push = push;
    this.head = head;
    this.attributes = attributes;
  }

  static createDetached(describe) {
    return new Branch(describe, nullBranch, nullBranch, true, {
      [DETACHED]: true
    });
  }

  static createRemoteTracking(refName, remoteName, remoteRef) {
    return new Branch(refName, nullBranch, nullBranch, false, {
      [REMOTE_TRACKING]: {
        remoteName,
        remoteRef
      }
    });
  }

  getName() {
    return this.name;
  }

  getShortRef() {
    return this.getName().replace(/^(refs\/)?((heads|remotes)\/)?/, '');
  }

  getFullRef() {
    if (this.isDetached()) {
      return '';
    }

    if (this.isRemoteTracking()) {
      if (this.name.startsWith('refs/')) {
        return this.name;
      } else if (this.name.startsWith('remotes/')) {
        return `refs/${this.name}`;
      }

      return `refs/remotes/${this.name}`;
    }

    if (this.name.startsWith('refs/')) {
      return this.name;
    } else if (this.name.startsWith('heads/')) {
      return `refs/${this.name}`;
    } else {
      return `refs/heads/${this.name}`;
    }
  }

  getRemoteName() {
    if (!this.isRemoteTracking()) {
      return '';
    }

    return this.attributes[REMOTE_TRACKING].remoteName || '';
  }

  getRemoteRef() {
    if (!this.isRemoteTracking()) {
      return '';
    }

    return this.attributes[REMOTE_TRACKING].remoteRef || '';
  }

  getShortRemoteRef() {
    return this.getRemoteRef().replace(/^(refs\/)?((heads|remotes)\/)?/, '');
  }

  getRefSpec(action) {
    if (this.isRemoteTracking()) {
      return '';
    }

    const remoteBranch = action === 'PUSH' ? this.push : this.upstream;
    const remoteBranchName = remoteBranch.getShortRemoteRef();
    const localBranchName = this.getName();

    if (remoteBranchName && remoteBranchName !== localBranchName) {
      if (action === 'PUSH') {
        return `${localBranchName}:${remoteBranchName}`;
      } else if (action === 'PULL') {
        return `${remoteBranchName}:${localBranchName}`;
      }
    }

    return localBranchName;
  }

  getSha() {
    return this.attributes.sha || '';
  }

  getUpstream() {
    return this.upstream;
  }

  getPush() {
    return this.push;
  }

  isHead() {
    return this.head;
  }

  isDetached() {
    return this.attributes[DETACHED] !== undefined;
  }

  isRemoteTracking() {
    return this.attributes[REMOTE_TRACKING] !== undefined;
  }

  isPresent() {
    return true;
  }

}

exports.default = Branch;
const nullBranch = {
  getName() {
    return '';
  },

  getShortRef() {
    return '';
  },

  getFullRef() {
    return '';
  },

  getSha() {
    return '';
  },

  getUpstream() {
    return this;
  },

  getPush() {
    return this;
  },

  isHead() {
    return false;
  },

  getRemoteName() {
    return '';
  },

  getRemoteRef() {
    return '';
  },

  getShortRemoteRef() {
    return '';
  },

  isDetached() {
    return false;
  },

  isRemoteTracking() {
    return false;
  },

  isPresent() {
    return false;
  },

  inspect(depth, options) {
    return '{nullBranch}';
  }

};
exports.nullBranch = nullBranch;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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