"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.nullCommit = exports.default = void 0;

var _patch = require("./patch");

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

const UNBORN = Symbol('unborn'); // Truncation elipsis styles

const WORD_ELIPSES = '...';
const NEWLINE_ELIPSES = '\n...';
const PARAGRAPH_ELIPSES = '\n\n...';

class Commit {
  static createUnborn() {
    return new Commit({
      unbornRef: UNBORN
    });
  }

  constructor({
    sha,
    author,
    coAuthors,
    authorDate,
    messageSubject,
    messageBody,
    unbornRef,
    patch
  }) {
    this.sha = sha;
    this.author = author;
    this.coAuthors = coAuthors || [];
    this.authorDate = authorDate;
    this.messageSubject = messageSubject;
    this.messageBody = messageBody;
    this.unbornRef = unbornRef === UNBORN;
    this.multiFileDiff = patch ? (0, _patch.buildMultiFilePatch)(patch) : (0, _patch.buildMultiFilePatch)([]);
  }

  getSha() {
    return this.sha;
  }

  getAuthor() {
    return this.author;
  }

  getAuthorEmail() {
    return this.author.getEmail();
  }

  getAuthorAvatarUrl() {
    return this.author.getAvatarUrl();
  }

  getAuthorName() {
    return this.author.getFullName();
  }

  getAuthorDate() {
    return this.authorDate;
  }

  getCoAuthors() {
    return this.coAuthors;
  }

  getMessageSubject() {
    return this.messageSubject;
  }

  getMessageBody() {
    return this.messageBody;
  }

  isBodyLong() {
    if (this.getMessageBody().length > this.constructor.LONG_MESSAGE_THRESHOLD) {
      return true;
    }

    if ((this.getMessageBody().match(/\r?\n/g) || []).length > this.constructor.NEWLINE_THRESHOLD) {
      return true;
    }

    return false;
  }

  getFullMessage() {
    return `${this.getMessageSubject()}\n\n${this.getMessageBody()}`.trim();
  }
  /*
   * Return the messageBody truncated to at most LONG_MESSAGE_THRESHOLD characters or NEWLINE_THRESHOLD newlines,
   * whichever comes first.
   *
   * If NEWLINE_THRESHOLD newlines are encountered before LONG_MESSAGE_THRESHOLD characters, the body will be truncated
   * at the last counted newline and elipses added.
   *
   * If a paragraph boundary is found before LONG_MESSAGE_THRESHOLD characters, the message will be truncated at the end
   * of the previous paragraph and an elipses added. If no paragraph boundary is found, but a word boundary is, the text
   * is truncated at the last word boundary and an elipsis added. If neither are found, the text is truncated hard at
   * LONG_MESSAGE_THRESHOLD - 3 characters and an elipsis is added.
   */


  abbreviatedBody() {
    if (!this.isBodyLong()) {
      return this.getMessageBody();
    }

    const {
      LONG_MESSAGE_THRESHOLD,
      NEWLINE_THRESHOLD
    } = this.constructor;
    let lastNewlineCutoff = null;
    let lastParagraphCutoff = null;
    let lastWordCutoff = null;
    const searchText = this.getMessageBody().substring(0, LONG_MESSAGE_THRESHOLD);
    const boundaryRx = /\s+/g;
    let result;
    let lineCount = 0;

    while ((result = boundaryRx.exec(searchText)) !== null) {
      const newlineCount = (result[0].match(/\r?\n/g) || []).length;
      lineCount += newlineCount;

      if (lineCount > NEWLINE_THRESHOLD) {
        lastNewlineCutoff = result.index;
        break;
      }

      if (newlineCount < 2 && result.index <= LONG_MESSAGE_THRESHOLD - WORD_ELIPSES.length) {
        lastWordCutoff = result.index;
      } else if (result.index < LONG_MESSAGE_THRESHOLD - PARAGRAPH_ELIPSES.length) {
        lastParagraphCutoff = result.index;
      }
    }

    let elipses = WORD_ELIPSES;
    let cutoffIndex = LONG_MESSAGE_THRESHOLD - WORD_ELIPSES.length;

    if (lastNewlineCutoff !== null) {
      elipses = NEWLINE_ELIPSES;
      cutoffIndex = lastNewlineCutoff;
    } else if (lastParagraphCutoff !== null) {
      elipses = PARAGRAPH_ELIPSES;
      cutoffIndex = lastParagraphCutoff;
    } else if (lastWordCutoff !== null) {
      cutoffIndex = lastWordCutoff;
    }

    return this.getMessageBody().substring(0, cutoffIndex) + elipses;
  }

  setMultiFileDiff(multiFileDiff) {
    this.multiFileDiff = multiFileDiff;
  }

  getMultiFileDiff() {
    return this.multiFileDiff;
  }

  isUnbornRef() {
    return this.unbornRef;
  }

  isPresent() {
    return true;
  }

  isEqual(other) {
    // Directly comparable properties
    const properties = ['sha', 'authorDate', 'messageSubject', 'messageBody', 'unbornRef'];

    for (const property of properties) {
      if (this[property] !== other[property]) {
        return false;
      }
    } // Author


    if (this.author.getEmail() !== other.getAuthorEmail() || this.author.getFullName() !== other.getAuthorName()) {
      return false;
    } // Co-author array


    if (this.coAuthors.length !== other.coAuthors.length) {
      return false;
    }

    for (let i = 0; i < this.coAuthors.length; i++) {
      const thisCoAuthor = this.coAuthors[i];
      const otherCoAuthor = other.coAuthors[i];

      if (thisCoAuthor.getFullName() !== otherCoAuthor.getFullName() || thisCoAuthor.getEmail() !== otherCoAuthor.getEmail()) {
        return false;
      }
    } // Multi-file patch


    if (!this.multiFileDiff.isEqual(other.multiFileDiff)) {
      return false;
    }

    return true;
  }

}

exports.default = Commit;

_defineProperty(Commit, "LONG_MESSAGE_THRESHOLD", 400);

_defineProperty(Commit, "NEWLINE_THRESHOLD", 5);

const nullCommit = {
  getSha() {
    return '';
  },

  getMessageSubject() {
    return '';
  },

  isUnbornRef() {
    return false;
  },

  isPresent() {
    return false;
  },

  isBodyLong() {
    return false;
  }

};
exports.nullCommit = nullCommit;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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