"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

var _position = require("./position");

var _source = require("./source");

var _side = _interopRequireDefault(require("./side"));

var _banner = _interopRequireDefault(require("./banner"));

var _separator = _interopRequireDefault(require("./separator"));

var _parser = require("./parser");

var _adapter = require("./parser/adapter");

var _noopVisitor = require("./parser/noop-visitor");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

// Regular expression that matches the beginning of a potential conflict.
const CONFLICT_START_REGEX = /^<{7} ([^\r\n]+)\r?\n/g;
/*
 * Conflict parser visitor that marks each buffer range and assembles a Conflict from the
 * pieces.
 */

class ConflictVisitor {
  /*
   * editor - [TextEditor] displaying the conflicting text.
   * layer - [DisplayMarkerLayer] to created conflict markers on.
   */
  constructor(editor, layer) {
    this.editor = editor;
    this.layer = layer;
    this.ours = null;
    this.base = null;
    this.separator = null;
    this.theirs = null;
  }
  /*
   * position - [Position] one of TOP or BOTTOM.
   * bannerRow - [Integer] of the buffer row that contains our side's banner.
   * textRowStart - [Integer] of the first buffer row that contain this side's text.
   * textRowEnd - [Integer] of the first buffer row beyond the extend of this side's text.
   */


  visitOurSide(position, bannerRow, textRowStart, textRowEnd) {
    this.ours = this.markSide(position, _source.OURS, bannerRow, textRowStart, textRowEnd);
  }
  /*
   * bannerRow - [Integer] the buffer row that contains our side's banner.
   * textRowStart - [Integer] first buffer row that contain this side's text.
   * textRowEnd - [Integer] first buffer row beyond the extend of this side's text.
   */


  visitBaseSide(bannerRow, textRowStart, textRowEnd) {
    this.base = this.markSide(_position.MIDDLE, _source.BASE, bannerRow, textRowStart, textRowEnd);
  }
  /*
   * sepRowStart - [Integer] buffer row that contains the "=======" separator.
   */


  visitSeparator(sepRowStart) {
    const marker = this.layer.markBufferRange([[sepRowStart, 0], [sepRowStart + 1, 0]], {
      invalidate: 'surround',
      exclusive: true
    });
    this.separator = new _separator.default(this.editor, marker);
  }
  /*
   * position - [Position] alignment within the conflict marker: TOP or BOTTOM.
   * bannerRow - [Integer] the buffer row that contains our side's banner.
   * textRowStart - [Integer] first buffer row that contain this side's text.
   * textRowEnd - [Integer] first buffer row beyond the extent of this side's text.
   */


  visitTheirSide(position, bannerRow, textRowStart, textRowEnd) {
    this.theirs = this.markSide(position, _source.THEIRS, bannerRow, textRowStart, textRowEnd);
  }

  markSide(position, source, bannerRow, textRowStart, textRowEnd) {
    const blockCol = position.when({
      top: () => 0,
      middle: () => 0,
      bottom: () => this.editor.lineTextForBufferRow(bannerRow).length
    });
    const blockRange = [[bannerRow, blockCol], [bannerRow, blockCol]];
    const blockMarker = this.layer.markBufferRange(blockRange, {
      invalidate: 'surround',
      exclusive: true
    });
    const description = this.sideDescription(bannerRow);
    const bannerRange = [[bannerRow, 0], [bannerRow + 1, 0]];
    const bannerMarker = this.layer.markBufferRange(bannerRange, {
      invalidate: 'surround',
      exclusive: true
    });
    const originalBannerText = this.editor.getTextInBufferRange(bannerRange);
    const banner = new _banner.default(this.editor, bannerMarker, description, originalBannerText);
    const textRange = [[textRowStart, 0], [textRowEnd, 0]];
    const sideMarker = this.layer.markBufferRange(textRange, {
      invalidate: 'surround',
      exclusive: false
    });
    const originalText = this.editor.getTextInBufferRange(textRange);
    return new _side.default(this.editor, sideMarker, blockMarker, source, position, banner, originalText);
  }
  /*
   * Parse the banner description for the current side from a banner row.
   *
   * bannerRow - [Integer] buffer row containing the <, |, or > marker
   */


  sideDescription(bannerRow) {
    return this.editor.lineTextForBufferRow(bannerRow).match(/^[<|>]{7} (.*)$/)[1];
  }

  conflict() {
    return new Conflict(this.ours, this.separator, this.base, this.theirs);
  }

}

class Conflict {
  constructor(ours, separator, base, theirs) {
    this.separator = separator;
    this.bySource = {};
    this.byPosition = {};
    [ours, base, theirs].forEach(side => {
      if (!side) {
        return;
      }

      this.bySource[side.getSource().getName()] = side;
      this.byPosition[side.getPosition().getName()] = side;
    });
    this.resolution = null;
  }

  getKey() {
    return this.getSide(_source.OURS).getMarker().id;
  }

  isResolved() {
    return this.resolution !== null;
  }

  resolveAs(source) {
    this.resolution = this.getSide(source);
  }

  getSides() {
    return ['ours', 'base', 'theirs'].map(sourceName => this.bySource[sourceName]).filter(side => side);
  }

  getChosenSide() {
    return this.resolution;
  }

  getUnchosenSides() {
    return this.getSides().filter(side => side !== this.resolution);
  }

  getSide(source) {
    return this.bySource[source.getName()];
  }
  /*
   * Return a `Side` containing a buffer point, or `undefined` if none do.
   */


  getSideContaining(point) {
    return this.getSides().find(side => side.includesPoint(point));
  }
  /*
   * Return a `Range` that encompasses the entire Conflict region.
   */


  getRange() {
    const topRange = this.byPosition.top.getRange();
    const bottomRange = this.byPosition.bottom.getRange();
    return topRange.union(bottomRange);
  }
  /*
   * Determine whether or not a buffer position is contained within this conflict.
   */


  includesPoint(point) {
    return this.getRange().containsPoint(point);
  }
  /*
   * Return the `DisplayMarker` that immediately follows the `Side` in a given `Position`. Return `null` if no such
   * marker exists.
   */


  markerAfter(position) {
    return position.when({
      top: () => this.byPosition.middle ? this.byPosition.middle.getBannerMarker() : this.getSeparator().getMarker(),
      middle: () => this.getSeparator().getMarker(),
      bottom: () => this.byPosition.bottom.getBannerMarker()
    });
  }

  getSeparator() {
    return this.separator;
  }
  /*
   * Parse any conflict markers in a TextEditor's buffer and return a Conflict that contains markers corresponding to
   * each.
   *
   * editor [TextEditor] The editor to search.
   * layer [DisplayMarkerLayer] Marker layer to create markers on.
   * return [Array<Conflict>] A (possibly empty) collection of parsed Conflicts.
   */


  static allFromEditor(editor, layer, isRebase) {
    const conflicts = [];
    let lastRow = -1;
    editor.getBuffer().scan(CONFLICT_START_REGEX, m => {
      const conflictStartRow = m.range.start.row;

      if (conflictStartRow < lastRow) {
        // Match within an already-parsed conflict.
        return;
      }

      const adapter = new _adapter.EditorAdapter(editor, conflictStartRow);
      const visitor = new ConflictVisitor(editor, layer);
      const parser = new _parser.ConflictParser(adapter, visitor, isRebase);

      if (parser.parse().wasSuccessful()) {
        conflicts.push(visitor.conflict());
      }

      lastRow = adapter.getCurrentRow();
    });
    return conflicts;
  }
  /*
   * Return the number of conflict markers present in a streamed file.
   */


  static countFromStream(stream, isRebase) {
    return new Promise((resolve, reject) => {
      let count = 0;
      let lastResult = null;
      let lastPartialMarker = '';
      stream.on('data', chunk => {
        const adapter = new _adapter.ChunkAdapter(lastPartialMarker + chunk);

        if (!lastResult) {
          if (!adapter.advanceTo(CONFLICT_START_REGEX)) {
            lastPartialMarker = adapter.getLastPartialMarker();
            return;
          }
        }

        do {
          const parser = new _parser.ConflictParser(adapter, new _noopVisitor.NoopVisitor(), isRebase);
          const result = lastResult ? parser.continueFrom(lastResult) : parser.parse();

          if (result.wasSuccessful()) {
            count++;
          } else {
            lastResult = result;
          }
        } while (adapter.advanceTo(CONFLICT_START_REGEX));

        lastPartialMarker = adapter.getLastPartialMarker();
      });
      stream.on('error', reject);
      stream.on('end', () => resolve(count));
    });
  }

}

exports.default = Conflict;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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