"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ChunkAdapter = exports.EditorAdapter = void 0;

/*
 * Input adapter to facilitate parsing conflicts from text loaded into an Editor.
 */
class EditorAdapter {
  constructor(editor, startRow) {
    this.editor = editor;
    this.currentRow = startRow;
  }

  getCurrentRow() {
    return this.currentRow;
  }

  getCurrentLine() {
    return this.editor.lineTextForBufferRow(this.currentRow);
  }

  advanceRow() {
    this.currentRow++;
  }

  isAtEnd() {
    return this.currentRow > this.editor.getLastBufferRow();
  }

}
/*
 * Input adapter for parsing conflicts from a chunk of text arriving from a ReadStream.
 */


exports.EditorAdapter = EditorAdapter;

class ChunkAdapter {
  constructor(chunk) {
    this.chunk = chunk;
    this.lineEndRx = /\r?\n/g;
    this.lineStartPosition = 0;
    this.eof = false;
    this.advanceRow();
  }

  advanceTo(pattern) {
    if (this.eof) {
      return false;
    }

    const rx = new RegExp(pattern.source, 'gm');
    rx.lastIndex = this.lineStartPosition;
    const match = rx.exec(this.chunk);

    if (match) {
      this.lineEndRx.lastIndex = match.index;
      return true;
    } else {
      return false;
    }
  }

  getCurrentRow() {
    return undefined;
  }

  getCurrentLine() {
    return this.currentLine;
  }

  advanceRow() {
    this.lineStartPosition = this.lineEndRx.lastIndex;

    if (this.lineEndRx.test(this.chunk)) {
      this.currentLine = this.chunk.slice(this.lineStartPosition, this.lineEndRx.lastIndex);
    } else {
      this.currentLine = this.chunk.slice(this.lineStartPosition);
      this.eof = true;
    }
  }

  isAtEnd() {
    return this.eof;
  }

  getLastPartialMarker() {
    const match = /[<|>=]{1,7} ?[^\r\n]*\r?$/.exec(this.chunk);
    return match ? match[0] : '';
  }

}

exports.ChunkAdapter = ChunkAdapter;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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