"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;
const DISABLEMENT = Symbol('disablement');
const ENABLED = Symbol('enabled');
const NO_REASON = Symbol('no-reason'); // Track an operation that may be either enabled or disabled with a message and a reason. EnableableOperation instances
// are immutable to aid passing them as React component props; call `.enable()` or `.disable()` to derive a new
// operation instance with the same callback.

class EnableableOperation {
  constructor(op, options = {}) {
    this.beforeOp = null;
    this.op = op;
    this.afterOp = null;
    this.disablement = options[DISABLEMENT] || ENABLED;
  }

  toggleState(component, stateKey) {
    this.beforeOp = () => {
      component.setState(prevState => {
        return !prevState[stateKey] ? {
          [stateKey]: true
        } : {};
      });
    };

    this.afterOp = () => {
      return new Promise(resolve => {
        component.setState(prevState => {
          return prevState[stateKey] ? {
            [stateKey]: false
          } : {};
        }, resolve);
      });
    };
  }

  isEnabled() {
    return this.disablement === ENABLED;
  }

  async run() {
    if (!this.isEnabled()) {
      throw new Error(this.disablement.message);
    }

    if (this.beforeOp) {
      this.beforeOp();
    }

    let result = undefined;

    try {
      result = await this.op();
    } finally {
      if (this.afterOp) {
        await this.afterOp();
      }
    }

    return result;
  }

  getMessage() {
    return this.disablement.message;
  }

  why() {
    return this.disablement.reason;
  }

  disable(reason = NO_REASON, message = 'disabled') {
    if (!this.isEnabled() && this.disablement.reason === reason && this.disablement.message === message) {
      return this;
    }

    return new this.constructor(this.op, {
      [DISABLEMENT]: {
        reason,
        message
      }
    });
  }

  enable() {
    if (this.isEnabled()) {
      return this;
    }

    return new this.constructor(this.op, {
      [DISABLEMENT]: ENABLED
    });
  }

}

exports.default = EnableableOperation;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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