"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

var _crypto = _interopRequireDefault(require("crypto"));

var _eventKit = require("event-kit");

var _keytarStrategy = require("../shared/keytar-strategy");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

let instance = null;

class GithubLoginModel {
  // Be sure that we're requesting at least this many scopes on the token we grant through github.atom.io or we'll
  // give everyone a really frustrating experience ;-)
  static get() {
    if (!instance) {
      instance = new GithubLoginModel();
    }

    return instance;
  }

  constructor(Strategy) {
    this._Strategy = Strategy;
    this._strategy = null;
    this.emitter = new _eventKit.Emitter();
    this.checked = new Map();
  }

  async getStrategy() {
    if (this._strategy) {
      return this._strategy;
    }

    if (this._Strategy) {
      this._strategy = new this._Strategy();
      return this._strategy;
    }

    this._strategy = await (0, _keytarStrategy.createStrategy)();
    return this._strategy;
  }

  async getToken(account) {
    const strategy = await this.getStrategy();
    const password = await strategy.getPassword('atom-github', account);

    if (!password || password === _keytarStrategy.UNAUTHENTICATED) {
      // User is not logged in
      return _keytarStrategy.UNAUTHENTICATED;
    }

    if (/^https?:\/\//.test(account)) {
      // Avoid storing tokens in memory longer than necessary. Let's cache token scope checks by storing a set of
      // checksums instead.
      const hash = _crypto.default.createHash('md5');

      hash.update(password);
      const fingerprint = hash.digest('base64');
      const outcome = this.checked.get(fingerprint);

      if (outcome === _keytarStrategy.UNAUTHENTICATED || outcome === _keytarStrategy.INSUFFICIENT) {
        // Cached failure
        return outcome;
      } else if (!outcome) {
        // No cached outcome. Query for scopes.
        try {
          const scopes = await this.getScopes(account, password);

          if (scopes === _keytarStrategy.UNAUTHORIZED) {
            // Password is incorrect. Treat it as though you aren't authenticated at all.
            this.checked.set(fingerprint, _keytarStrategy.UNAUTHENTICATED);
            return _keytarStrategy.UNAUTHENTICATED;
          }

          const scopeSet = new Set(scopes);

          for (const scope of this.constructor.REQUIRED_SCOPES) {
            if (!scopeSet.has(scope)) {
              // Token doesn't have enough OAuth scopes, need to reauthenticate
              this.checked.set(fingerprint, _keytarStrategy.INSUFFICIENT);
              return _keytarStrategy.INSUFFICIENT;
            }
          } // Successfully authenticated and had all required scopes.


          this.checked.set(fingerprint, true);
        } catch (e) {
          // Most likely a network error. Do not cache the failure.
          return e;
        }
      }
    }

    return password;
  }

  async setToken(account, token) {
    const strategy = await this.getStrategy();
    await strategy.replacePassword('atom-github', account, token);
    this.didUpdate();
  }

  async removeToken(account) {
    const strategy = await this.getStrategy();
    await strategy.deletePassword('atom-github', account);
    this.didUpdate();
  }
  /* istanbul ignore next */


  async getScopes(host, token) {
    if (atom.inSpecMode()) {
      if (token === 'good-token') {
        return this.constructor.REQUIRED_SCOPES;
      }

      throw new Error('Attempt to check token scopes in specs');
    }

    let response;

    try {
      response = await fetch(host, {
        method: 'HEAD',
        headers: {
          Authorization: `bearer ${token}`
        }
      });
    } catch (e) {
      e.network = true;
      throw e;
    }

    if (response.status === 401) {
      return _keytarStrategy.UNAUTHORIZED;
    }

    if (response.status !== 200) {
      const e = new Error(`Unable to check token for OAuth scopes against ${host}`);
      e.response = response;
      e.responseText = await response.text();
      throw e;
    }

    return response.headers.get('X-OAuth-Scopes').split(/\s*,\s*/);
  }

  didUpdate() {
    this.emitter.emit('did-update');
  }

  onDidUpdate(cb) {
    return this.emitter.on('did-update', cb);
  }

  destroy() {
    this.emitter.dispose();
  }

}

exports.default = GithubLoginModel;

_defineProperty(GithubLoginModel, "REQUIRED_SCOPES", ['repo', 'read:org', 'user:email']);
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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