"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

var _helpers = require("../helpers");

const COPY = Symbol('copy');

class ListSelection {
  constructor(options = {}) {
    (0, _helpers.autobind)(this, 'isItemSelectable');

    if (options._copy !== COPY) {
      this.options = {
        isItemSelectable: options.isItemSelectable || (item => !!item)
      };
      this.items = options.items || [];
      this.selections = this.items.length > 0 ? [{
        head: 0,
        tail: 0
      }] : [];
    } else {
      this.options = {
        isItemSelectable: options.isItemSelectable
      };
      this.items = options.items;
      this.selections = options.selections;
    }
  }

  copy(options = {}) {
    return new ListSelection({
      _copy: COPY,
      isItemSelectable: options.isItemSelectable || this.options.isItemSelectable,
      items: options.items || this.items,
      selections: options.selections || this.selections
    });
  }

  isItemSelectable(item) {
    return this.options.isItemSelectable(item);
  }

  setItems(items) {
    let newSelectionIndex;

    if (this.selections.length > 0) {
      const [{
        head,
        tail
      }] = this.selections;
      newSelectionIndex = Math.min(head, tail, items.length - 1);
    } else {
      newSelectionIndex = 0;
    }

    const newSelections = items.length > 0 ? [{
      head: newSelectionIndex,
      tail: newSelectionIndex
    }] : [];
    return this.copy({
      items,
      selections: newSelections
    });
  }

  getItems() {
    return this.items;
  }

  getLastItem() {
    return this.items[this.items.length - 1];
  }

  selectFirstItem(preserveTail) {
    for (let i = 0; i < this.items.length; i++) {
      const item = this.items[i];

      if (this.isItemSelectable(item)) {
        return this.selectItem(item, preserveTail);
      }
    }

    return this;
  }

  selectLastItem(preserveTail) {
    for (let i = this.items.length - 1; i > 0; i--) {
      const item = this.items[i];

      if (this.isItemSelectable(item)) {
        return this.selectItem(item, preserveTail);
      }
    }

    return this;
  }

  selectAllItems() {
    return this.selectFirstItem().selectLastItem(true);
  }

  selectNextItem(preserveTail) {
    if (this.selections.length === 0) {
      return this.selectFirstItem();
    }

    let itemIndex = this.selections[0].head;
    let nextItemIndex = itemIndex;

    while (itemIndex < this.items.length - 1) {
      itemIndex++;

      if (this.isItemSelectable(this.items[itemIndex])) {
        nextItemIndex = itemIndex;
        break;
      }
    }

    return this.selectItem(this.items[nextItemIndex], preserveTail);
  }

  selectPreviousItem(preserveTail) {
    if (this.selections.length === 0) {
      return this.selectLastItem();
    }

    let itemIndex = this.selections[0].head;
    let previousItemIndex = itemIndex;

    while (itemIndex > 0) {
      itemIndex--;

      if (this.isItemSelectable(this.items[itemIndex])) {
        previousItemIndex = itemIndex;
        break;
      }
    }

    return this.selectItem(this.items[previousItemIndex], preserveTail);
  }

  selectItem(item, preserveTail, addOrSubtract) {
    if (addOrSubtract && preserveTail) {
      throw new Error('addOrSubtract and preserveTail cannot both be true at the same time');
    }

    const itemIndex = this.items.indexOf(item);

    if (preserveTail && this.selections[0]) {
      const newSelections = [{
        head: itemIndex,
        tail: this.selections[0].tail,
        negate: this.selections[0].negate
      }, ...this.selections.slice(1)];
      return this.copy({
        selections: newSelections
      });
    } else {
      const selection = {
        head: itemIndex,
        tail: itemIndex
      };

      if (addOrSubtract) {
        if (this.getSelectedItems().has(item)) {
          selection.negate = true;
        }

        return this.copy({
          selections: [selection, ...this.selections]
        });
      } else {
        return this.copy({
          selections: [selection]
        });
      }
    }
  }

  addOrSubtractSelection(item) {
    return this.selectItem(item, false, true);
  }

  coalesce() {
    if (this.selections.length === 0) {
      return this;
    }

    const mostRecent = this.selections[0];
    let mostRecentStart = Math.min(mostRecent.head, mostRecent.tail);
    let mostRecentEnd = Math.max(mostRecent.head, mostRecent.tail);

    while (mostRecentStart > 0 && !this.isItemSelectable(this.items[mostRecentStart - 1])) {
      mostRecentStart--;
    }

    while (mostRecentEnd < this.items.length - 1 && !this.isItemSelectable(this.items[mostRecentEnd + 1])) {
      mostRecentEnd++;
    }

    let changed = false;
    const newSelections = [mostRecent];

    for (let i = 1; i < this.selections.length;) {
      const current = this.selections[i];
      const currentStart = Math.min(current.head, current.tail);
      const currentEnd = Math.max(current.head, current.tail);

      if (mostRecentStart <= currentEnd + 1 && currentStart - 1 <= mostRecentEnd) {
        if (mostRecent.negate) {
          if (current.head > current.tail) {
            if (currentEnd > mostRecentEnd) {
              // suffix
              newSelections.push({
                tail: mostRecentEnd + 1,
                head: currentEnd
              });
            }

            if (currentStart < mostRecentStart) {
              // prefix
              newSelections.push({
                tail: currentStart,
                head: mostRecentStart - 1
              });
            }
          } else {
            if (currentStart < mostRecentStart) {
              // prefix
              newSelections.push({
                head: currentStart,
                tail: mostRecentStart - 1
              });
            }

            if (currentEnd > mostRecentEnd) {
              // suffix
              newSelections.push({
                head: mostRecentEnd + 1,
                tail: currentEnd
              });
            }
          }

          changed = true;
          i++;
        } else {
          mostRecentStart = Math.min(mostRecentStart, currentStart);
          mostRecentEnd = Math.max(mostRecentEnd, currentEnd);

          if (mostRecent.head >= mostRecent.tail) {
            mostRecent.head = mostRecentEnd;
            mostRecent.tail = mostRecentStart;
          } else {
            mostRecent.head = mostRecentStart;
            mostRecent.tail = mostRecentEnd;
          }

          changed = true;
          i++;
        }
      } else {
        newSelections.push(current);
        i++;
      }
    }

    if (mostRecent.negate) {
      changed = true;
      newSelections.shift();
    }

    return changed ? this.copy({
      selections: newSelections
    }) : this;
  }

  getSelectedItems() {
    const selectedItems = new Set();

    for (const {
      head,
      tail,
      negate
    } of this.selections.slice().reverse()) {
      const start = Math.min(head, tail);
      const end = Math.max(head, tail);

      for (let i = start; i <= end; i++) {
        const item = this.items[i];

        if (this.isItemSelectable(item)) {
          if (negate) {
            selectedItems.delete(item);
          } else {
            selectedItems.add(item);
          }
        }
      }
    }

    return selectedItems;
  }

  getHeadItem() {
    return this.selections.length > 0 ? this.items[this.selections[0].head] : null;
  }

  getMostRecentSelectionStartIndex() {
    const selection = this.selections[0];
    return Math.min(selection.head, selection.tail);
  }

  getTailIndex() {
    return this.selections[0] ? this.selections[0].tail : null;
  }

}

exports.default = ListSelection;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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