"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.buildFilePatch = buildFilePatch;
exports.buildMultiFilePatch = buildMultiFilePatch;
exports.DEFAULT_OPTIONS = void 0;

var _patchBuffer = _interopRequireDefault(require("./patch-buffer"));

var _hunk = _interopRequireDefault(require("./hunk"));

var _file = _interopRequireWildcard(require("./file"));

var _patch = _interopRequireWildcard(require("./patch"));

var _region = require("./region");

var _filePatch = _interopRequireDefault(require("./file-patch"));

var _multiFilePatch = _interopRequireDefault(require("./multi-file-patch"));

function _interopRequireWildcard(obj) { if (obj && obj.__esModule) { return obj; } else { var newObj = {}; if (obj != null) { for (var key in obj) { if (Object.prototype.hasOwnProperty.call(obj, key)) { var desc = Object.defineProperty && Object.getOwnPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : {}; if (desc.get || desc.set) { Object.defineProperty(newObj, key, desc); } else { newObj[key] = obj[key]; } } } } newObj.default = obj; return newObj; } }

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _objectSpread2(target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i] != null ? arguments[i] : {}; var ownKeys = Object.keys(source); if (typeof Object.getOwnPropertySymbols === 'function') { ownKeys = ownKeys.concat(Object.getOwnPropertySymbols(source).filter(function (sym) { return Object.getOwnPropertyDescriptor(source, sym).enumerable; })); } ownKeys.forEach(function (key) { _defineProperty(target, key, source[key]); }); } return target; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

const DEFAULT_OPTIONS = {
  // Number of lines after which we consider the diff "large"
  largeDiffThreshold: 800,
  // Map of file path (relative to repository root) to Patch render status (EXPANDED, COLLAPSED, DEFERRED)
  renderStatusOverrides: {},
  // Existing patch buffer to render onto
  patchBuffer: null,
  // Store off what-the-diff file patch
  preserveOriginal: false,
  // Paths of file patches that have been removed from the patch before parsing
  removed: new Set()
};
exports.DEFAULT_OPTIONS = DEFAULT_OPTIONS;

function buildFilePatch(diffs, options) {
  const opts = _objectSpread2({}, DEFAULT_OPTIONS, {}, options);

  const patchBuffer = new _patchBuffer.default();
  let filePatch;

  if (diffs.length === 0) {
    filePatch = emptyDiffFilePatch();
  } else if (diffs.length === 1) {
    filePatch = singleDiffFilePatch(diffs[0], patchBuffer, opts);
  } else if (diffs.length === 2) {
    filePatch = dualDiffFilePatch(diffs[0], diffs[1], patchBuffer, opts);
  } else {
    throw new Error(`Unexpected number of diffs: ${diffs.length}`);
  } // Delete the trailing newline.


  patchBuffer.deleteLastNewline();
  return new _multiFilePatch.default({
    patchBuffer,
    filePatches: [filePatch]
  });
}

function buildMultiFilePatch(diffs, options) {
  const opts = _objectSpread2({}, DEFAULT_OPTIONS, {}, options);

  const patchBuffer = new _patchBuffer.default();
  const byPath = new Map();
  const actions = [];
  let index = 0;

  for (const diff of diffs) {
    const thePath = diff.oldPath || diff.newPath;

    if (diff.status === 'added' || diff.status === 'deleted') {
      // Potential paired diff. Either a symlink deletion + content addition or a symlink addition +
      // content deletion.
      const otherHalf = byPath.get(thePath);

      if (otherHalf) {
        // The second half. Complete the paired diff, or fail if they have unexpected statuses or modes.
        const [otherDiff, otherIndex] = otherHalf;

        actions[otherIndex] = function (_diff, _otherDiff) {
          return () => dualDiffFilePatch(_diff, _otherDiff, patchBuffer, opts);
        }(diff, otherDiff);

        byPath.delete(thePath);
      } else {
        // The first half we've seen.
        byPath.set(thePath, [diff, index]);
        index++;
      }
    } else {
      actions[index] = function (_diff) {
        return () => singleDiffFilePatch(_diff, patchBuffer, opts);
      }(diff);

      index++;
    }
  } // Populate unpaired diffs that looked like they could be part of a pair, but weren't.


  for (const [unpairedDiff, originalIndex] of byPath.values()) {
    actions[originalIndex] = function (_unpairedDiff) {
      return () => singleDiffFilePatch(_unpairedDiff, patchBuffer, opts);
    }(unpairedDiff);
  }

  const filePatches = actions.map(action => action()); // Delete the final trailing newline from the last non-empty patch.

  patchBuffer.deleteLastNewline(); // Append hidden patches corresponding to each removed file.

  for (const removedPath of opts.removed) {
    const removedFile = new _file.default({
      path: removedPath
    });
    const removedMarker = patchBuffer.markPosition(_patch.default.layerName, patchBuffer.getBuffer().getEndPosition(), {
      invalidate: 'never',
      exclusive: false
    });
    filePatches.push(_filePatch.default.createHiddenFilePatch(removedFile, removedFile, removedMarker, _patch.REMOVED,
    /* istanbul ignore next */
    () => {
      throw new Error(`Attempt to expand removed file patch ${removedPath}`);
    }));
  }

  return new _multiFilePatch.default({
    patchBuffer,
    filePatches
  });
}

function emptyDiffFilePatch() {
  return _filePatch.default.createNull();
}

function singleDiffFilePatch(diff, patchBuffer, opts) {
  const wasSymlink = diff.oldMode === _file.default.modes.SYMLINK;
  const isSymlink = diff.newMode === _file.default.modes.SYMLINK;
  let oldSymlink = null;
  let newSymlink = null;

  if (wasSymlink && !isSymlink) {
    oldSymlink = diff.hunks[0].lines[0].slice(1);
  } else if (!wasSymlink && isSymlink) {
    newSymlink = diff.hunks[0].lines[0].slice(1);
  } else if (wasSymlink && isSymlink) {
    oldSymlink = diff.hunks[0].lines[0].slice(1);
    newSymlink = diff.hunks[0].lines[2].slice(1);
  }

  const oldFile = diff.oldPath !== null || diff.oldMode !== null ? new _file.default({
    path: diff.oldPath,
    mode: diff.oldMode,
    symlink: oldSymlink
  }) : _file.nullFile;
  const newFile = diff.newPath !== null || diff.newMode !== null ? new _file.default({
    path: diff.newPath,
    mode: diff.newMode,
    symlink: newSymlink
  }) : _file.nullFile;
  const renderStatusOverride = oldFile.isPresent() && opts.renderStatusOverrides[oldFile.getPath()] || newFile.isPresent() && opts.renderStatusOverrides[newFile.getPath()] || undefined;

  const renderStatus = renderStatusOverride || isDiffLarge([diff], opts) && _patch.DEFERRED || _patch.EXPANDED;

  if (!renderStatus.isVisible()) {
    const patchMarker = patchBuffer.markPosition(_patch.default.layerName, patchBuffer.getBuffer().getEndPosition(), {
      invalidate: 'never',
      exclusive: false
    });
    return _filePatch.default.createHiddenFilePatch(oldFile, newFile, patchMarker, renderStatus, () => {
      const subPatchBuffer = new _patchBuffer.default();
      const [hunks, nextPatchMarker] = buildHunks(diff, subPatchBuffer);
      const nextPatch = new _patch.default({
        status: diff.status,
        hunks,
        marker: nextPatchMarker
      });
      subPatchBuffer.deleteLastNewline();
      return {
        patch: nextPatch,
        patchBuffer: subPatchBuffer
      };
    });
  } else {
    const [hunks, patchMarker] = buildHunks(diff, patchBuffer);
    const patch = new _patch.default({
      status: diff.status,
      hunks,
      marker: patchMarker
    });
    const rawPatches = opts.preserveOriginal ? {
      content: diff
    } : null;
    return new _filePatch.default(oldFile, newFile, patch, rawPatches);
  }
}

function dualDiffFilePatch(diff1, diff2, patchBuffer, opts) {
  let modeChangeDiff, contentChangeDiff;

  if (diff1.oldMode === _file.default.modes.SYMLINK || diff1.newMode === _file.default.modes.SYMLINK) {
    modeChangeDiff = diff1;
    contentChangeDiff = diff2;
  } else {
    modeChangeDiff = diff2;
    contentChangeDiff = diff1;
  }

  const filePath = contentChangeDiff.oldPath || contentChangeDiff.newPath;
  const symlink = modeChangeDiff.hunks[0].lines[0].slice(1);
  let status;
  let oldMode, newMode;
  let oldSymlink = null;
  let newSymlink = null;

  if (modeChangeDiff.status === 'added') {
    // contents were deleted and replaced with symlink
    status = 'deleted';
    oldMode = contentChangeDiff.oldMode;
    newMode = modeChangeDiff.newMode;
    newSymlink = symlink;
  } else if (modeChangeDiff.status === 'deleted') {
    // contents were added after symlink was deleted
    status = 'added';
    oldMode = modeChangeDiff.oldMode;
    oldSymlink = symlink;
    newMode = contentChangeDiff.newMode;
  } else {
    throw new Error(`Invalid mode change diff status: ${modeChangeDiff.status}`);
  }

  const oldFile = new _file.default({
    path: filePath,
    mode: oldMode,
    symlink: oldSymlink
  });
  const newFile = new _file.default({
    path: filePath,
    mode: newMode,
    symlink: newSymlink
  });

  const renderStatus = opts.renderStatusOverrides[filePath] || isDiffLarge([contentChangeDiff], opts) && _patch.DEFERRED || _patch.EXPANDED;

  if (!renderStatus.isVisible()) {
    const patchMarker = patchBuffer.markPosition(_patch.default.layerName, patchBuffer.getBuffer().getEndPosition(), {
      invalidate: 'never',
      exclusive: false
    });
    return _filePatch.default.createHiddenFilePatch(oldFile, newFile, patchMarker, renderStatus, () => {
      const subPatchBuffer = new _patchBuffer.default();
      const [hunks, nextPatchMarker] = buildHunks(contentChangeDiff, subPatchBuffer);
      const nextPatch = new _patch.default({
        status,
        hunks,
        marker: nextPatchMarker
      });
      subPatchBuffer.deleteLastNewline();
      return {
        patch: nextPatch,
        patchBuffer: subPatchBuffer
      };
    });
  } else {
    const [hunks, patchMarker] = buildHunks(contentChangeDiff, patchBuffer);
    const patch = new _patch.default({
      status,
      hunks,
      marker: patchMarker
    });
    const rawPatches = opts.preserveOriginal ? {
      content: contentChangeDiff,
      mode: modeChangeDiff
    } : null;
    return new _filePatch.default(oldFile, newFile, patch, rawPatches);
  }
}

const CHANGEKIND = {
  '+': _region.Addition,
  '-': _region.Deletion,
  ' ': _region.Unchanged,
  '\\': _region.NoNewline
};

function buildHunks(diff, patchBuffer) {
  const inserter = patchBuffer.createInserterAtEnd().keepBefore(patchBuffer.findAllMarkers({
    endPosition: patchBuffer.getInsertionPoint()
  }));
  let patchMarker = null;
  let firstHunk = true;
  const hunks = [];
  inserter.markWhile(_patch.default.layerName, () => {
    for (const rawHunk of diff.hunks) {
      let firstRegion = true;
      const regions = []; // Separate hunks with an unmarked newline

      if (firstHunk) {
        firstHunk = false;
      } else {
        inserter.insert('\n');
      }

      inserter.markWhile(_hunk.default.layerName, () => {
        let firstRegionLine = true;
        let currentRegionText = '';
        let CurrentRegionKind = null;

        function finishRegion() {
          if (CurrentRegionKind === null) {
            return;
          } // Separate regions with an unmarked newline


          if (firstRegion) {
            firstRegion = false;
          } else {
            inserter.insert('\n');
          }

          inserter.insertMarked(currentRegionText, CurrentRegionKind.layerName, {
            invalidate: 'never',
            exclusive: false,
            callback: function (_regions, _CurrentRegionKind) {
              return regionMarker => {
                _regions.push(new _CurrentRegionKind(regionMarker));
              };
            }(regions, CurrentRegionKind)
          });
        }

        for (const rawLine of rawHunk.lines) {
          const NextRegionKind = CHANGEKIND[rawLine[0]];

          if (NextRegionKind === undefined) {
            throw new Error(`Unknown diff status character: "${rawLine[0]}"`);
          }

          const nextLine = rawLine.slice(1);
          let separator = '';

          if (firstRegionLine) {
            firstRegionLine = false;
          } else {
            separator = '\n';
          }

          if (NextRegionKind === CurrentRegionKind) {
            currentRegionText += separator + nextLine;
            continue;
          } else {
            finishRegion();
            CurrentRegionKind = NextRegionKind;
            currentRegionText = nextLine;
          }
        }

        finishRegion();
      }, {
        invalidate: 'never',
        exclusive: false,
        callback: function (_hunks, _rawHunk, _regions) {
          return hunkMarker => {
            _hunks.push(new _hunk.default({
              oldStartRow: _rawHunk.oldStartLine,
              newStartRow: _rawHunk.newStartLine,
              oldRowCount: _rawHunk.oldLineCount,
              newRowCount: _rawHunk.newLineCount,
              sectionHeading: _rawHunk.heading,
              marker: hunkMarker,
              regions: _regions
            }));
          };
        }(hunks, rawHunk, regions)
      });
    }
  }, {
    invalidate: 'never',
    exclusive: false,
    callback: marker => {
      patchMarker = marker;
    }
  }); // Separate multiple non-empty patches on the same buffer with an unmarked newline. The newline after the final
  // non-empty patch (if there is one) should be deleted before MultiFilePatch construction.

  if (diff.hunks.length > 0) {
    inserter.insert('\n');
  }

  inserter.apply();
  return [hunks, patchMarker];
}

function isDiffLarge(diffs, opts) {
  const size = diffs.reduce((diffSizeCounter, diff) => {
    return diffSizeCounter + diff.hunks.reduce((hunkSizeCounter, hunk) => {
      return hunkSizeCounter + hunk.lines.length;
    }, 0);
  }, 0);
  return size > opts.largeDiffThreshold;
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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