"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

var _eventKit = require("event-kit");

var _file = require("./file");

var _patch = _interopRequireWildcard(require("./patch"));

var _helpers = require("../../helpers");

function _interopRequireWildcard(obj) { if (obj && obj.__esModule) { return obj; } else { var newObj = {}; if (obj != null) { for (var key in obj) { if (Object.prototype.hasOwnProperty.call(obj, key)) { var desc = Object.defineProperty && Object.getOwnPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : {}; if (desc.get || desc.set) { Object.defineProperty(newObj, key, desc); } else { newObj[key] = obj[key]; } } } } newObj.default = obj; return newObj; } }

function _objectSpread2(target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i] != null ? arguments[i] : {}; var ownKeys = Object.keys(source); if (typeof Object.getOwnPropertySymbols === 'function') { ownKeys = ownKeys.concat(Object.getOwnPropertySymbols(source).filter(function (sym) { return Object.getOwnPropertyDescriptor(source, sym).enumerable; })); } ownKeys.forEach(function (key) { _defineProperty(target, key, source[key]); }); } return target; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

class FilePatch {
  static createNull() {
    return new this(_file.nullFile, _file.nullFile, _patch.default.createNull());
  }

  static createHiddenFilePatch(oldFile, newFile, marker, renderStatus, showFn) {
    return new this(oldFile, newFile, _patch.default.createHiddenPatch(marker, renderStatus, showFn));
  }

  constructor(oldFile, newFile, patch, rawPatches) {
    this.oldFile = oldFile;
    this.newFile = newFile;
    this.patch = patch;
    this.rawPatches = rawPatches;
    this.emitter = new _eventKit.Emitter();
  }

  isPresent() {
    return this.oldFile.isPresent() || this.newFile.isPresent() || this.patch.isPresent();
  }

  getRenderStatus() {
    return this.patch.getRenderStatus();
  }

  getOldFile() {
    return this.oldFile;
  }

  getNewFile() {
    return this.newFile;
  }

  getRawContentPatch() {
    if (!this.rawPatches) {
      throw new Error('FilePatch was not parsed with {perserveOriginal: true}');
    }

    return this.rawPatches.content;
  }

  getPatch() {
    return this.patch;
  }

  getMarker() {
    return this.getPatch().getMarker();
  }

  getStartRange() {
    return this.getPatch().getStartRange();
  }

  getOldPath() {
    return this.getOldFile().getPath();
  }

  getNewPath() {
    return this.getNewFile().getPath();
  }

  getOldMode() {
    return this.getOldFile().getMode();
  }

  getNewMode() {
    return this.getNewFile().getMode();
  }

  getOldSymlink() {
    return this.getOldFile().getSymlink();
  }

  getNewSymlink() {
    return this.getNewFile().getSymlink();
  }

  getFirstChangeRange() {
    return this.getPatch().getFirstChangeRange();
  }

  getMaxLineNumberWidth() {
    return this.getPatch().getMaxLineNumberWidth();
  }

  containsRow(row) {
    return this.getPatch().containsRow(row);
  }

  didChangeExecutableMode() {
    if (!this.oldFile.isPresent() || !this.newFile.isPresent()) {
      return false;
    }

    return this.oldFile.isExecutable() && !this.newFile.isExecutable() || !this.oldFile.isExecutable() && this.newFile.isExecutable();
  }

  hasSymlink() {
    return Boolean(this.getOldFile().getSymlink() || this.getNewFile().getSymlink());
  }

  hasTypechange() {
    if (!this.oldFile.isPresent() || !this.newFile.isPresent()) {
      return false;
    }

    return this.oldFile.isSymlink() && !this.newFile.isSymlink() || !this.oldFile.isSymlink() && this.newFile.isSymlink();
  }

  getPath() {
    return this.getOldPath() || this.getNewPath();
  }

  getStatus() {
    return this.getPatch().getStatus();
  }

  getHunks() {
    return this.getPatch().getHunks();
  }

  updateMarkers(map) {
    return this.patch.updateMarkers(map);
  }

  triggerCollapseIn(patchBuffer, {
    before,
    after
  }) {
    if (!this.patch.getRenderStatus().isVisible()) {
      return false;
    }

    const oldPatch = this.patch;
    const oldRange = oldPatch.getRange().copy();
    const insertionPosition = oldRange.start;
    const exclude = new Set([...before, ...after]);
    const {
      patchBuffer: subPatchBuffer,
      markerMap
    } = patchBuffer.extractPatchBuffer(oldRange, {
      exclude
    });
    oldPatch.destroyMarkers();
    oldPatch.updateMarkers(markerMap); // Delete the separating newline after the collapsing patch, if any.

    if (!oldRange.isEmpty()) {
      patchBuffer.getBuffer().deleteRow(insertionPosition.row);
    }

    const patchMarker = patchBuffer.markPosition(_patch.default.layerName, insertionPosition, {
      invalidate: 'never',
      exclusive: true
    });
    this.patch = _patch.default.createHiddenPatch(patchMarker, _patch.COLLAPSED, () => {
      return {
        patch: oldPatch,
        patchBuffer: subPatchBuffer
      };
    });
    this.didChangeRenderStatus();
    return true;
  }

  triggerExpandIn(patchBuffer, {
    before,
    after
  }) {
    if (this.patch.getRenderStatus().isVisible()) {
      return false;
    }

    const {
      patch: nextPatch,
      patchBuffer: subPatchBuffer
    } = this.patch.show();
    const atStart = this.patch.getInsertionPoint().isEqual([0, 0]);
    const atEnd = this.patch.getInsertionPoint().isEqual(patchBuffer.getBuffer().getEndPosition());
    const willHaveContent = !subPatchBuffer.getBuffer().isEmpty(); // The expanding patch's insertion point is just after the unmarked newline that separates adjacent visible
    // patches:
    // <p0> '\n' * <p1> '\n' <p2>
    //
    // If it's to become the first (visible) patch, its insertion point is at [0, 0]:
    // * <p0> '\n' <p1> '\n' <p2>
    //
    // If it's to become the final (visible) patch, its insertion point is at the buffer end:
    // <p0> '\n' <p1> '\n' <p2> *
    //
    // Insert a newline *before* the expanding patch if we're inserting at the buffer's end, but the buffer is non-empty
    // (so it isn't also the end of the buffer). Insert a newline *after* the expanding patch when inserting anywhere
    // but the buffer's end.

    if (willHaveContent && atEnd && !atStart) {
      const beforeNewline = [];
      const afterNewline = after.slice();

      for (const marker of before) {
        if (marker.getRange().isEmpty()) {
          afterNewline.push(marker);
        } else {
          beforeNewline.push(marker);
        }
      }

      patchBuffer.createInserterAt(this.patch.getInsertionPoint()).keepBefore(beforeNewline).keepAfter(afterNewline).insert('\n').apply();
    }

    patchBuffer.createInserterAt(this.patch.getInsertionPoint()).keepBefore(before).keepAfter(after).insertPatchBuffer(subPatchBuffer, {
      callback: map => nextPatch.updateMarkers(map)
    }).insert(!atEnd ? '\n' : '').apply();
    this.patch.destroyMarkers();
    this.patch = nextPatch;
    this.didChangeRenderStatus();
    return true;
  }

  didChangeRenderStatus() {
    return this.emitter.emit('change-render-status', this);
  }

  onDidChangeRenderStatus(callback) {
    return this.emitter.on('change-render-status', callback);
  }

  clone(opts = {}) {
    return new this.constructor(opts.oldFile !== undefined ? opts.oldFile : this.oldFile, opts.newFile !== undefined ? opts.newFile : this.newFile, opts.patch !== undefined ? opts.patch : this.patch);
  }

  getStartingMarkers() {
    return this.patch.getStartingMarkers();
  }

  getEndingMarkers() {
    return this.patch.getEndingMarkers();
  }

  buildStagePatchForLines(originalBuffer, nextPatchBuffer, selectedLineSet) {
    let newFile = this.getNewFile();

    if (this.getStatus() === 'deleted') {
      if (this.patch.getChangedLineCount() === selectedLineSet.size && Array.from(selectedLineSet, row => this.patch.containsRow(row)).every(Boolean)) {
        // Whole file deletion staged.
        newFile = _file.nullFile;
      } else {
        // Partial file deletion, which becomes a modification.
        newFile = this.getOldFile();
      }
    }

    const patch = this.patch.buildStagePatchForLines(originalBuffer, nextPatchBuffer, selectedLineSet);
    return this.clone({
      newFile,
      patch
    });
  }

  buildUnstagePatchForLines(originalBuffer, nextPatchBuffer, selectedLineSet) {
    const nonNullFile = this.getNewFile().isPresent() ? this.getNewFile() : this.getOldFile();
    let oldFile = this.getNewFile();
    let newFile = nonNullFile;

    if (this.getStatus() === 'added') {
      if (selectedLineSet.size === this.patch.getChangedLineCount() && Array.from(selectedLineSet, row => this.patch.containsRow(row)).every(Boolean)) {
        // Ensure that newFile is null if the patch is an addition because we're deleting the entire file from the
        // index. If a symlink was deleted and replaced by a non-symlink file, we don't want the symlink entry to muck
        // up the patch.
        oldFile = nonNullFile;
        newFile = _file.nullFile;
      }
    } else if (this.getStatus() === 'deleted') {
      if (selectedLineSet.size === this.patch.getChangedLineCount() && Array.from(selectedLineSet, row => this.patch.containsRow(row)).every(Boolean)) {
        oldFile = _file.nullFile;
        newFile = nonNullFile;
      }
    }

    const patch = this.patch.buildUnstagePatchForLines(originalBuffer, nextPatchBuffer, selectedLineSet);
    return this.clone({
      oldFile,
      newFile,
      patch
    });
  }

  toStringIn(buffer) {
    if (!this.isPresent()) {
      return '';
    }

    if (this.hasTypechange()) {
      const left = this.clone({
        newFile: _file.nullFile,
        patch: this.getOldSymlink() ? this.getPatch().clone({
          status: 'deleted'
        }) : this.getPatch()
      });
      const right = this.clone({
        oldFile: _file.nullFile,
        patch: this.getNewSymlink() ? this.getPatch().clone({
          status: 'added'
        }) : this.getPatch()
      });
      return left.toStringIn(buffer) + right.toStringIn(buffer);
    } else if (this.getStatus() === 'added' && this.getNewFile().isSymlink()) {
      const symlinkPath = this.getNewSymlink();
      return this.getHeaderString() + `@@ -0,0 +1 @@\n+${symlinkPath}\n\\ No newline at end of file\n`;
    } else if (this.getStatus() === 'deleted' && this.getOldFile().isSymlink()) {
      const symlinkPath = this.getOldSymlink();
      return this.getHeaderString() + `@@ -1 +0,0 @@\n-${symlinkPath}\n\\ No newline at end of file\n`;
    } else {
      return this.getHeaderString() + this.getPatch().toStringIn(buffer);
    }
  }
  /*
   * Construct a String containing diagnostic information about the internal state of this FilePatch.
   */

  /* istanbul ignore next */


  inspect(opts = {}) {
    const options = _objectSpread2({
      indent: 0
    }, opts);

    let indentation = '';

    for (let i = 0; i < options.indent; i++) {
      indentation += ' ';
    }

    let inspectString = `${indentation}(FilePatch `;

    if (this.getOldPath() !== this.getNewPath()) {
      inspectString += `oldPath=${this.getOldPath()} newPath=${this.getNewPath()}`;
    } else {
      inspectString += `path=${this.getPath()}`;
    }

    inspectString += '\n';
    inspectString += this.patch.inspect({
      indent: options.indent + 2
    });
    inspectString += `${indentation})\n`;
    return inspectString;
  }

  getHeaderString() {
    const fromPath = this.getOldPath() || this.getNewPath();
    const toPath = this.getNewPath() || this.getOldPath();
    let header = `diff --git a/${(0, _helpers.toGitPathSep)(fromPath)} b/${(0, _helpers.toGitPathSep)(toPath)}`;
    header += '\n';

    if (this.getStatus() === 'added') {
      header += `new file mode ${this.getNewMode()}`;
      header += '\n';
    } else if (this.getStatus() === 'deleted') {
      header += `deleted file mode ${this.getOldMode()}`;
      header += '\n';
    }

    header += this.getOldPath() ? `--- a/${(0, _helpers.toGitPathSep)(this.getOldPath())}` : '--- /dev/null';
    header += '\n';
    header += this.getNewPath() ? `+++ b/${(0, _helpers.toGitPathSep)(this.getNewPath())}` : '+++ /dev/null';
    header += '\n';
    return header;
  }

}

exports.default = FilePatch;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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