"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

var _atom = require("atom");

var _bintrees = require("bintrees");

var _patchBuffer = _interopRequireDefault(require("./patch-buffer"));

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

class MultiFilePatch {
  static createNull() {
    return new this({
      patchBuffer: new _patchBuffer.default(),
      filePatches: []
    });
  }

  constructor({
    patchBuffer,
    filePatches
  }) {
    _defineProperty(this, "isPatchVisible", filePatchPath => {
      const patch = this.filePatchesByPath.get(filePatchPath);

      if (!patch) {
        return false;
      }

      return patch.getRenderStatus().isVisible();
    });

    _defineProperty(this, "getBufferRowForDiffPosition", (fileName, diffRow) => {
      const offsetIndex = this.diffRowOffsetIndices.get(fileName);

      if (!offsetIndex) {
        // eslint-disable-next-line no-console
        console.error('Attempt to compute buffer row for invalid diff position: file not included', {
          fileName,
          diffRow,
          validFileNames: Array.from(this.diffRowOffsetIndices.keys())
        });
        return null;
      }

      const {
        startBufferRow,
        index
      } = offsetIndex;
      const result = index.lowerBound({
        diffRow
      }).data();

      if (!result) {
        // eslint-disable-next-line no-console
        console.error('Attempt to compute buffer row for invalid diff position: diff row out of range', {
          fileName,
          diffRow
        });
        return null;
      }

      const {
        offset
      } = result;
      return startBufferRow + diffRow - offset;
    });

    this.patchBuffer = patchBuffer;
    this.filePatches = filePatches;
    this.filePatchesByMarker = new Map();
    this.filePatchesByPath = new Map();
    this.hunksByMarker = new Map(); // Store a map of {diffRow, offset} for each FilePatch where offset is the number of Hunk headers within the current
    // FilePatch that occur before this row in the original diff output.

    this.diffRowOffsetIndices = new Map();

    for (const filePatch of this.filePatches) {
      this.filePatchesByPath.set(filePatch.getPath(), filePatch);
      this.filePatchesByMarker.set(filePatch.getMarker(), filePatch);
      this.populateDiffRowOffsetIndices(filePatch);
    }
  }

  clone(opts = {}) {
    return new this.constructor({
      patchBuffer: opts.patchBuffer !== undefined ? opts.patchBuffer : this.getPatchBuffer(),
      filePatches: opts.filePatches !== undefined ? opts.filePatches : this.getFilePatches()
    });
  }

  getPatchBuffer() {
    return this.patchBuffer;
  }

  getBuffer() {
    return this.getPatchBuffer().getBuffer();
  }

  getPatchLayer() {
    return this.getPatchBuffer().getLayer('patch');
  }

  getHunkLayer() {
    return this.getPatchBuffer().getLayer('hunk');
  }

  getUnchangedLayer() {
    return this.getPatchBuffer().getLayer('unchanged');
  }

  getAdditionLayer() {
    return this.getPatchBuffer().getLayer('addition');
  }

  getDeletionLayer() {
    return this.getPatchBuffer().getLayer('deletion');
  }

  getNoNewlineLayer() {
    return this.getPatchBuffer().getLayer('nonewline');
  }

  getFilePatches() {
    return this.filePatches;
  }

  getPatchForPath(path) {
    return this.filePatchesByPath.get(path);
  }

  getPathSet() {
    return this.getFilePatches().reduce((pathSet, filePatch) => {
      for (const file of [filePatch.getOldFile(), filePatch.getNewFile()]) {
        if (file.isPresent()) {
          pathSet.add(file.getPath());
        }
      }

      return pathSet;
    }, new Set());
  }

  getFilePatchAt(bufferRow) {
    if (bufferRow < 0 || bufferRow > this.patchBuffer.getBuffer().getLastRow()) {
      return undefined;
    }

    const [marker] = this.patchBuffer.findMarkers('patch', {
      intersectsRow: bufferRow
    });
    return this.filePatchesByMarker.get(marker);
  }

  getHunkAt(bufferRow) {
    if (bufferRow < 0) {
      return undefined;
    }

    const [marker] = this.patchBuffer.findMarkers('hunk', {
      intersectsRow: bufferRow
    });
    return this.hunksByMarker.get(marker);
  }

  getStagePatchForLines(selectedLineSet) {
    const nextPatchBuffer = new _patchBuffer.default();
    const nextFilePatches = this.getFilePatchesContaining(selectedLineSet).map(fp => {
      return fp.buildStagePatchForLines(this.getBuffer(), nextPatchBuffer, selectedLineSet);
    });
    return this.clone({
      patchBuffer: nextPatchBuffer,
      filePatches: nextFilePatches
    });
  }

  getStagePatchForHunk(hunk) {
    return this.getStagePatchForLines(new Set(hunk.getBufferRows()));
  }

  getUnstagePatchForLines(selectedLineSet) {
    const nextPatchBuffer = new _patchBuffer.default();
    const nextFilePatches = this.getFilePatchesContaining(selectedLineSet).map(fp => {
      return fp.buildUnstagePatchForLines(this.getBuffer(), nextPatchBuffer, selectedLineSet);
    });
    return this.clone({
      patchBuffer: nextPatchBuffer,
      filePatches: nextFilePatches
    });
  }

  getUnstagePatchForHunk(hunk) {
    return this.getUnstagePatchForLines(new Set(hunk.getBufferRows()));
  }

  getMaxSelectionIndex(selectedRows) {
    if (selectedRows.size === 0) {
      return 0;
    }

    const lastMax = Math.max(...selectedRows);
    let selectionIndex = 0; // counts unselected lines in changed regions from the old patch
    // until we get to the bottom-most selected line from the old patch (lastMax).

    patchLoop: for (const filePatch of this.getFilePatches()) {
      for (const hunk of filePatch.getHunks()) {
        let includesMax = false;

        for (const change of hunk.getChanges()) {
          for (const {
            intersection,
            gap
          } of change.intersectRows(selectedRows, true)) {
            // Only include a partial range if this intersection includes the last selected buffer row.
            includesMax = intersection.intersectsRow(lastMax);
            const delta = includesMax ? lastMax - intersection.start.row + 1 : intersection.getRowCount();

            if (gap) {
              // Range of unselected changes.
              selectionIndex += delta;
            }

            if (includesMax) {
              break patchLoop;
            }
          }
        }
      }
    }

    return selectionIndex;
  }

  getSelectionRangeForIndex(selectionIndex) {
    // Iterate over changed lines in this patch in order to find the
    // new row to be selected based on the last selection index.
    // As we walk through the changed lines, we whittle down the
    // remaining lines until we reach the row that corresponds to the
    // last selected index.
    let selectionRow = 0;
    let remainingChangedLines = selectionIndex;
    let foundRow = false;
    let lastChangedRow = 0;

    patchLoop: for (const filePatch of this.getFilePatches()) {
      for (const hunk of filePatch.getHunks()) {
        for (const change of hunk.getChanges()) {
          if (remainingChangedLines < change.bufferRowCount()) {
            selectionRow = change.getStartBufferRow() + remainingChangedLines;
            foundRow = true;
            break patchLoop;
          } else {
            remainingChangedLines -= change.bufferRowCount();
            lastChangedRow = change.getEndBufferRow();
          }
        }
      }
    } // If we never got to the last selected index, that means it is
    // no longer present in the new patch (ie. we staged the last line of the file).
    // In this case we want the next selected line to be the last changed row in the file


    if (!foundRow) {
      selectionRow = lastChangedRow;
    }

    return _atom.Range.fromObject([[selectionRow, 0], [selectionRow, Infinity]]);
  }

  isDiffRowOffsetIndexEmpty(filePatchPath) {
    const diffRowOffsetIndex = this.diffRowOffsetIndices.get(filePatchPath);
    return diffRowOffsetIndex.index.size === 0;
  }

  populateDiffRowOffsetIndices(filePatch) {
    let diffRow = 1;
    const index = new _bintrees.RBTree((a, b) => a.diffRow - b.diffRow);
    this.diffRowOffsetIndices.set(filePatch.getPath(), {
      startBufferRow: filePatch.getStartRange().start.row,
      index
    });

    for (let hunkIndex = 0; hunkIndex < filePatch.getHunks().length; hunkIndex++) {
      const hunk = filePatch.getHunks()[hunkIndex];
      this.hunksByMarker.set(hunk.getMarker(), hunk); // Advance past the hunk body

      diffRow += hunk.bufferRowCount();
      index.insert({
        diffRow,
        offset: hunkIndex + 1
      }); // Advance past the next hunk header

      diffRow++;
    }
  }

  adoptBuffer(nextPatchBuffer) {
    nextPatchBuffer.clearAllLayers();
    this.filePatchesByMarker.clear();
    this.hunksByMarker.clear();
    const markerMap = nextPatchBuffer.adopt(this.patchBuffer);

    for (const filePatch of this.getFilePatches()) {
      filePatch.updateMarkers(markerMap);
      this.filePatchesByMarker.set(filePatch.getMarker(), filePatch);

      for (const hunk of filePatch.getHunks()) {
        this.hunksByMarker.set(hunk.getMarker(), hunk);
      }
    }

    this.patchBuffer = nextPatchBuffer;
  }
  /*
   * Efficiently locate the FilePatch instances that contain at least one row from a Set.
   */


  getFilePatchesContaining(rowSet) {
    const sortedRowSet = Array.from(rowSet);
    sortedRowSet.sort((a, b) => a - b);
    const filePatches = [];
    let lastFilePatch = null;

    for (const row of sortedRowSet) {
      // Because the rows are sorted, consecutive rows will almost certainly belong to the same patch, so we can save
      // many avoidable marker index lookups by comparing with the last.
      if (lastFilePatch && lastFilePatch.containsRow(row)) {
        continue;
      }

      lastFilePatch = this.getFilePatchAt(row);
      filePatches.push(lastFilePatch);
    }

    return filePatches;
  }

  anyPresent() {
    return this.patchBuffer !== null && this.filePatches.some(fp => fp.isPresent());
  }

  didAnyChangeExecutableMode() {
    for (const filePatch of this.getFilePatches()) {
      if (filePatch.didChangeExecutableMode()) {
        return true;
      }
    }

    return false;
  }

  anyHaveTypechange() {
    return this.getFilePatches().some(fp => fp.hasTypechange());
  }

  getMaxLineNumberWidth() {
    return this.getFilePatches().reduce((maxWidth, filePatch) => {
      const width = filePatch.getMaxLineNumberWidth();
      return maxWidth >= width ? maxWidth : width;
    }, 0);
  }

  spansMultipleFiles(rows) {
    let lastFilePatch = null;

    for (const row of rows) {
      if (lastFilePatch) {
        if (lastFilePatch.containsRow(row)) {
          continue;
        }

        return true;
      } else {
        lastFilePatch = this.getFilePatchAt(row);
      }
    }

    return false;
  }

  collapseFilePatch(filePatch) {
    const index = this.filePatches.indexOf(filePatch);
    this.filePatchesByMarker.delete(filePatch.getMarker());

    for (const hunk of filePatch.getHunks()) {
      this.hunksByMarker.delete(hunk.getMarker());
    }

    const before = this.getMarkersBefore(index);
    const after = this.getMarkersAfter(index);
    filePatch.triggerCollapseIn(this.patchBuffer, {
      before,
      after
    });
    this.filePatchesByMarker.set(filePatch.getMarker(), filePatch); // This hunk collection should be empty, but let's iterate anyway just in case filePatch was already collapsed

    /* istanbul ignore next */

    for (const hunk of filePatch.getHunks()) {
      this.hunksByMarker.set(hunk.getMarker(), hunk);
    }
  }

  expandFilePatch(filePatch) {
    const index = this.filePatches.indexOf(filePatch);
    this.filePatchesByMarker.delete(filePatch.getMarker());

    for (const hunk of filePatch.getHunks()) {
      this.hunksByMarker.delete(hunk.getMarker());
    }

    const before = this.getMarkersBefore(index);
    const after = this.getMarkersAfter(index);
    filePatch.triggerExpandIn(this.patchBuffer, {
      before,
      after
    });
    this.filePatchesByMarker.set(filePatch.getMarker(), filePatch);

    for (const hunk of filePatch.getHunks()) {
      this.hunksByMarker.set(hunk.getMarker(), hunk);
    } // if the patch was initially collapsed, we need to calculate
    // the diffRowOffsetIndices to calculate comment position.


    if (this.isDiffRowOffsetIndexEmpty(filePatch.getPath())) {
      this.populateDiffRowOffsetIndices(filePatch);
    }
  }

  getMarkersBefore(filePatchIndex) {
    const before = [];
    let beforeIndex = filePatchIndex - 1;

    while (beforeIndex >= 0) {
      const beforeFilePatch = this.filePatches[beforeIndex];
      before.push(...beforeFilePatch.getEndingMarkers());

      if (!beforeFilePatch.getMarker().getRange().isEmpty()) {
        break;
      }

      beforeIndex--;
    }

    return before;
  }

  getMarkersAfter(filePatchIndex) {
    const after = [];
    let afterIndex = filePatchIndex + 1;

    while (afterIndex < this.filePatches.length) {
      const afterFilePatch = this.filePatches[afterIndex];
      after.push(...afterFilePatch.getStartingMarkers());

      if (!afterFilePatch.getMarker().getRange().isEmpty()) {
        break;
      }

      afterIndex++;
    }

    return after;
  }

  getPreviewPatchBuffer(fileName, diffRow, maxRowCount) {
    const bufferRow = this.getBufferRowForDiffPosition(fileName, diffRow);

    if (bufferRow === null) {
      return new _patchBuffer.default();
    }

    const filePatch = this.getFilePatchAt(bufferRow);
    const filePatchIndex = this.filePatches.indexOf(filePatch);
    const hunk = this.getHunkAt(bufferRow);
    const previewStartRow = Math.max(bufferRow - maxRowCount + 1, hunk.getRange().start.row);
    const previewEndRow = bufferRow;
    const before = this.getMarkersBefore(filePatchIndex);
    const after = this.getMarkersAfter(filePatchIndex);
    const exclude = new Set([...before, ...after]);
    return this.patchBuffer.createSubBuffer([[previewStartRow, 0], [previewEndRow, Infinity]], {
      exclude
    }).patchBuffer;
  }
  /*
   * Construct an apply-able patch String.
   */


  toString() {
    return this.filePatches.map(fp => fp.toStringIn(this.getBuffer())).join('') + '\n';
  }
  /*
   * Construct a string of diagnostic information useful for debugging.
   */

  /* istanbul ignore next */


  inspect() {
    let inspectString = '(MultiFilePatch';
    inspectString += ` filePatchesByMarker=(${Array.from(this.filePatchesByMarker.keys(), m => m.id).join(', ')})`;
    inspectString += ` hunksByMarker=(${Array.from(this.hunksByMarker.keys(), m => m.id).join(', ')})\n`;

    for (const filePatch of this.filePatches) {
      inspectString += filePatch.inspect({
        indent: 2
      });
    }

    inspectString += ')\n';
    return inspectString;
  }
  /* istanbul ignore next */


  isEqual(other) {
    return this.toString() === other.toString();
  }

}

exports.default = MultiFilePatch;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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