"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

var _atom = require("atom");

var _util = require("util");

function _objectSpread2(target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i] != null ? arguments[i] : {}; var ownKeys = Object.keys(source); if (typeof Object.getOwnPropertySymbols === 'function') { ownKeys = ownKeys.concat(Object.getOwnPropertySymbols(source).filter(function (sym) { return Object.getOwnPropertyDescriptor(source, sym).enumerable; })); } ownKeys.forEach(function (key) { _defineProperty(target, key, source[key]); }); } return target; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

const LAYER_NAMES = ['unchanged', 'addition', 'deletion', 'nonewline', 'hunk', 'patch'];

class PatchBuffer {
  constructor() {
    this.buffer = new _atom.TextBuffer();
    this.buffer.retain();
    this.layers = LAYER_NAMES.reduce((map, layerName) => {
      map[layerName] = this.buffer.addMarkerLayer();
      return map;
    }, {});
  }

  getBuffer() {
    return this.buffer;
  }

  getInsertionPoint() {
    return this.buffer.getEndPosition();
  }

  getLayer(layerName) {
    return this.layers[layerName];
  }

  findMarkers(layerName, ...args) {
    return this.layers[layerName].findMarkers(...args);
  }

  findAllMarkers(...args) {
    return LAYER_NAMES.reduce((arr, layerName) => {
      arr.push(...this.findMarkers(layerName, ...args));
      return arr;
    }, []);
  }

  markPosition(layerName, ...args) {
    return this.layers[layerName].markPosition(...args);
  }

  markRange(layerName, ...args) {
    return this.layers[layerName].markRange(...args);
  }

  clearAllLayers() {
    for (const layerName of LAYER_NAMES) {
      this.layers[layerName].clear();
    }
  }

  createInserterAt(insertionPoint) {
    return new Inserter(this, _atom.Point.fromObject(insertionPoint));
  }

  createInserterAtEnd() {
    return this.createInserterAt(this.getInsertionPoint());
  }

  createSubBuffer(rangeLike, options = {}) {
    const opts = _objectSpread2({
      exclude: new Set()
    }, options);

    const range = _atom.Range.fromObject(rangeLike);

    const baseOffset = range.start.negate();
    const includedMarkersByLayer = LAYER_NAMES.reduce((map, layerName) => {
      map[layerName] = this.layers[layerName].findMarkers({
        intersectsRange: range
      }).filter(m => !opts.exclude.has(m));
      return map;
    }, {});
    const markerMap = new Map();
    const subBuffer = new PatchBuffer();
    subBuffer.getBuffer().setText(this.buffer.getTextInRange(range));

    for (const layerName of LAYER_NAMES) {
      for (const oldMarker of includedMarkersByLayer[layerName]) {
        const oldRange = oldMarker.getRange();
        const clippedStart = oldRange.start.isLessThanOrEqual(range.start) ? range.start : oldRange.start;
        const clippedEnd = oldRange.end.isGreaterThanOrEqual(range.end) ? range.end : oldRange.end; // Exclude non-empty markers that intersect *only* at the range start or end

        if (clippedStart.isEqual(clippedEnd) && !oldRange.start.isEqual(oldRange.end)) {
          continue;
        }

        const startOffset = clippedStart.row === range.start.row ? baseOffset : [baseOffset.row, 0];
        const endOffset = clippedEnd.row === range.start.row ? baseOffset : [baseOffset.row, 0];
        const newMarker = subBuffer.markRange(layerName, [clippedStart.translate(startOffset), clippedEnd.translate(endOffset)], oldMarker.getProperties());
        markerMap.set(oldMarker, newMarker);
      }
    }

    return {
      patchBuffer: subBuffer,
      markerMap
    };
  }

  extractPatchBuffer(rangeLike, options = {}) {
    const {
      patchBuffer: subBuffer,
      markerMap
    } = this.createSubBuffer(rangeLike, options);

    for (const oldMarker of markerMap.keys()) {
      oldMarker.destroy();
    }

    this.buffer.setTextInRange(rangeLike, '');
    return {
      patchBuffer: subBuffer,
      markerMap
    };
  }

  deleteLastNewline() {
    if (this.buffer.getLastLine() === '') {
      this.buffer.deleteRow(this.buffer.getLastRow());
    }

    return this;
  }

  adopt(original) {
    this.clearAllLayers();
    this.buffer.setText(original.getBuffer().getText());
    const markerMap = new Map();

    for (const layerName of LAYER_NAMES) {
      for (const originalMarker of original.getLayer(layerName).getMarkers()) {
        const newMarker = this.markRange(layerName, originalMarker.getRange(), originalMarker.getProperties());
        markerMap.set(originalMarker, newMarker);
      }
    }

    return markerMap;
  }
  /* istanbul ignore next */


  inspect(opts = {}) {
    /* istanbul ignore next */
    const options = _objectSpread2({
      layerNames: LAYER_NAMES
    }, opts);

    let inspectString = '';
    const increasingMarkers = [];

    for (const layerName of options.layerNames) {
      for (const marker of this.findMarkers(layerName, {})) {
        increasingMarkers.push({
          layerName,
          point: marker.getRange().start,
          start: true,
          id: marker.id
        });
        increasingMarkers.push({
          layerName,
          point: marker.getRange().end,
          end: true,
          id: marker.id
        });
      }
    }

    increasingMarkers.sort((a, b) => {
      const cmp = a.point.compare(b.point);

      if (cmp !== 0) {
        return cmp;
      } else if (a.start && b.start) {
        return 0;
      } else if (a.start && !b.start) {
        return -1;
      } else if (!a.start && b.start) {
        return 1;
      } else {
        return 0;
      }
    });

    let inspectPoint = _atom.Point.fromObject([0, 0]);

    for (const marker of increasingMarkers) {
      if (!marker.point.isEqual(inspectPoint)) {
        inspectString += (0, _util.inspect)(this.buffer.getTextInRange([inspectPoint, marker.point])) + '\n';
      }

      if (marker.start) {
        inspectString += `  start ${marker.layerName}@${marker.id}\n`;
      } else if (marker.end) {
        inspectString += `  end ${marker.layerName}@${marker.id}\n`;
      }

      inspectPoint = marker.point;
    }

    return inspectString;
  }

}

exports.default = PatchBuffer;

class Inserter {
  constructor(patchBuffer, insertionPoint) {
    const clipped = patchBuffer.getBuffer().clipPosition(insertionPoint);
    this.patchBuffer = patchBuffer;
    this.startPoint = clipped.copy();
    this.insertionPoint = clipped.copy();
    this.markerBlueprints = [];
    this.markerMapCallbacks = [];
    this.markersBefore = new Set();
    this.markersAfter = new Set();
  }

  keepBefore(markers) {
    for (const marker of markers) {
      if (marker.getRange().end.isEqual(this.startPoint)) {
        this.markersBefore.add(marker);
      }
    }

    return this;
  }

  keepAfter(markers) {
    for (const marker of markers) {
      if (marker.getRange().start.isEqual(this.startPoint)) {
        this.markersAfter.add(marker);
      }
    }

    return this;
  }

  markWhile(layerName, block, markerOpts) {
    const start = this.insertionPoint.copy();
    block();
    const end = this.insertionPoint.copy();
    this.markerBlueprints.push({
      layerName,
      range: new _atom.Range(start, end),
      markerOpts
    });
    return this;
  }

  insert(text) {
    const insertedRange = this.patchBuffer.getBuffer().insert(this.insertionPoint, text);
    this.insertionPoint = insertedRange.end;
    return this;
  }

  insertMarked(text, layerName, markerOpts) {
    return this.markWhile(layerName, () => this.insert(text), markerOpts);
  }

  insertPatchBuffer(subPatchBuffer, opts) {
    const baseOffset = this.insertionPoint.copy();
    this.insert(subPatchBuffer.getBuffer().getText());
    const subMarkerMap = new Map();

    for (const layerName of LAYER_NAMES) {
      for (const oldMarker of subPatchBuffer.findMarkers(layerName, {})) {
        const startOffset = oldMarker.getRange().start.row === 0 ? baseOffset : [baseOffset.row, 0];
        const endOffset = oldMarker.getRange().end.row === 0 ? baseOffset : [baseOffset.row, 0];
        const range = oldMarker.getRange().translate(startOffset, endOffset);

        const markerOpts = _objectSpread2({}, oldMarker.getProperties(), {
          callback: newMarker => {
            subMarkerMap.set(oldMarker, newMarker);
          }
        });

        this.markerBlueprints.push({
          layerName,
          range,
          markerOpts
        });
      }
    }

    this.markerMapCallbacks.push({
      markerMap: subMarkerMap,
      callback: opts.callback
    });
    return this;
  }

  apply() {
    for (const {
      layerName,
      range,
      markerOpts
    } of this.markerBlueprints) {
      const callback = markerOpts.callback;
      delete markerOpts.callback;
      const marker = this.patchBuffer.markRange(layerName, range, markerOpts);

      if (callback) {
        callback(marker);
      }
    }

    for (const {
      markerMap,
      callback
    } of this.markerMapCallbacks) {
      callback(markerMap);
    }

    for (const beforeMarker of this.markersBefore) {
      const isEmpty = beforeMarker.getRange().isEmpty();

      if (!beforeMarker.isReversed()) {
        beforeMarker.setHeadPosition(this.startPoint);

        if (isEmpty) {
          beforeMarker.setTailPosition(this.startPoint);
        }
      } else {
        beforeMarker.setTailPosition(this.startPoint);

        if (isEmpty) {
          beforeMarker.setHeadPosition(this.startPoint);
        }
      }
    }

    for (const afterMarker of this.markersAfter) {
      const isEmpty = afterMarker.getRange().isEmpty();

      if (!afterMarker.isReversed()) {
        afterMarker.setTailPosition(this.insertionPoint);

        if (isEmpty) {
          afterMarker.setHeadPosition(this.insertionPoint);
        }
      } else {
        afterMarker.setHeadPosition(this.insertionPoint);

        if (isEmpty) {
          afterMarker.setTailPosition(this.insertionPoint);
        }
      }
    }
  }

}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbInBhdGNoLWJ1ZmZlci5qcyJdLCJuYW1lcyI6WyJMQVlFUl9OQU1FUyIsIlBhdGNoQnVmZmVyIiwiY29uc3RydWN0b3IiLCJidWZmZXIiLCJUZXh0QnVmZmVyIiwicmV0YWluIiwibGF5ZXJzIiwicmVkdWNlIiwibWFwIiwibGF5ZXJOYW1lIiwiYWRkTWFya2VyTGF5ZXIiLCJnZXRCdWZmZXIiLCJnZXRJbnNlcnRpb25Qb2ludCIsImdldEVuZFBvc2l0aW9uIiwiZ2V0TGF5ZXIiLCJmaW5kTWFya2VycyIsImFyZ3MiLCJmaW5kQWxsTWFya2VycyIsImFyciIsInB1c2giLCJtYXJrUG9zaXRpb24iLCJtYXJrUmFuZ2UiLCJjbGVhckFsbExheWVycyIsImNsZWFyIiwiY3JlYXRlSW5zZXJ0ZXJBdCIsImluc2VydGlvblBvaW50IiwiSW5zZXJ0ZXIiLCJQb2ludCIsImZyb21PYmplY3QiLCJjcmVhdGVJbnNlcnRlckF0RW5kIiwiY3JlYXRlU3ViQnVmZmVyIiwicmFuZ2VMaWtlIiwib3B0aW9ucyIsIm9wdHMiLCJleGNsdWRlIiwiU2V0IiwicmFuZ2UiLCJSYW5nZSIsImJhc2VPZmZzZXQiLCJzdGFydCIsIm5lZ2F0ZSIsImluY2x1ZGVkTWFya2Vyc0J5TGF5ZXIiLCJpbnRlcnNlY3RzUmFuZ2UiLCJmaWx0ZXIiLCJtIiwiaGFzIiwibWFya2VyTWFwIiwiTWFwIiwic3ViQnVmZmVyIiwic2V0VGV4dCIsImdldFRleHRJblJhbmdlIiwib2xkTWFya2VyIiwib2xkUmFuZ2UiLCJnZXRSYW5nZSIsImNsaXBwZWRTdGFydCIsImlzTGVzc1RoYW5PckVxdWFsIiwiY2xpcHBlZEVuZCIsImVuZCIsImlzR3JlYXRlclRoYW5PckVxdWFsIiwiaXNFcXVhbCIsInN0YXJ0T2Zmc2V0Iiwicm93IiwiZW5kT2Zmc2V0IiwibmV3TWFya2VyIiwidHJhbnNsYXRlIiwiZ2V0UHJvcGVydGllcyIsInNldCIsInBhdGNoQnVmZmVyIiwiZXh0cmFjdFBhdGNoQnVmZmVyIiwia2V5cyIsImRlc3Ryb3kiLCJzZXRUZXh0SW5SYW5nZSIsImRlbGV0ZUxhc3ROZXdsaW5lIiwiZ2V0TGFzdExpbmUiLCJkZWxldGVSb3ciLCJnZXRMYXN0Um93IiwiYWRvcHQiLCJvcmlnaW5hbCIsImdldFRleHQiLCJvcmlnaW5hbE1hcmtlciIsImdldE1hcmtlcnMiLCJpbnNwZWN0IiwibGF5ZXJOYW1lcyIsImluc3BlY3RTdHJpbmciLCJpbmNyZWFzaW5nTWFya2VycyIsIm1hcmtlciIsInBvaW50IiwiaWQiLCJzb3J0IiwiYSIsImIiLCJjbXAiLCJjb21wYXJlIiwiaW5zcGVjdFBvaW50IiwiY2xpcHBlZCIsImNsaXBQb3NpdGlvbiIsInN0YXJ0UG9pbnQiLCJjb3B5IiwibWFya2VyQmx1ZXByaW50cyIsIm1hcmtlck1hcENhbGxiYWNrcyIsIm1hcmtlcnNCZWZvcmUiLCJtYXJrZXJzQWZ0ZXIiLCJrZWVwQmVmb3JlIiwibWFya2VycyIsImFkZCIsImtlZXBBZnRlciIsIm1hcmtXaGlsZSIsImJsb2NrIiwibWFya2VyT3B0cyIsImluc2VydCIsInRleHQiLCJpbnNlcnRlZFJhbmdlIiwiaW5zZXJ0TWFya2VkIiwiaW5zZXJ0UGF0Y2hCdWZmZXIiLCJzdWJQYXRjaEJ1ZmZlciIsInN1Yk1hcmtlck1hcCIsImNhbGxiYWNrIiwiYXBwbHkiLCJiZWZvcmVNYXJrZXIiLCJpc0VtcHR5IiwiaXNSZXZlcnNlZCIsInNldEhlYWRQb3NpdGlvbiIsInNldFRhaWxQb3NpdGlvbiIsImFmdGVyTWFya2VyIl0sIm1hcHBpbmdzIjoiOzs7Ozs7O0FBQUE7O0FBQ0E7Ozs7OztBQUVBLE1BQU1BLFdBQVcsR0FBRyxDQUFDLFdBQUQsRUFBYyxVQUFkLEVBQTBCLFVBQTFCLEVBQXNDLFdBQXRDLEVBQW1ELE1BQW5ELEVBQTJELE9BQTNELENBQXBCOztBQUVlLE1BQU1DLFdBQU4sQ0FBa0I7QUFDL0JDLEVBQUFBLFdBQVcsR0FBRztBQUNaLFNBQUtDLE1BQUwsR0FBYyxJQUFJQyxnQkFBSixFQUFkO0FBQ0EsU0FBS0QsTUFBTCxDQUFZRSxNQUFaO0FBRUEsU0FBS0MsTUFBTCxHQUFjTixXQUFXLENBQUNPLE1BQVosQ0FBbUIsQ0FBQ0MsR0FBRCxFQUFNQyxTQUFOLEtBQW9CO0FBQ25ERCxNQUFBQSxHQUFHLENBQUNDLFNBQUQsQ0FBSCxHQUFpQixLQUFLTixNQUFMLENBQVlPLGNBQVosRUFBakI7QUFDQSxhQUFPRixHQUFQO0FBQ0QsS0FIYSxFQUdYLEVBSFcsQ0FBZDtBQUlEOztBQUVERyxFQUFBQSxTQUFTLEdBQUc7QUFDVixXQUFPLEtBQUtSLE1BQVo7QUFDRDs7QUFFRFMsRUFBQUEsaUJBQWlCLEdBQUc7QUFDbEIsV0FBTyxLQUFLVCxNQUFMLENBQVlVLGNBQVosRUFBUDtBQUNEOztBQUVEQyxFQUFBQSxRQUFRLENBQUNMLFNBQUQsRUFBWTtBQUNsQixXQUFPLEtBQUtILE1BQUwsQ0FBWUcsU0FBWixDQUFQO0FBQ0Q7O0FBRURNLEVBQUFBLFdBQVcsQ0FBQ04sU0FBRCxFQUFZLEdBQUdPLElBQWYsRUFBcUI7QUFDOUIsV0FBTyxLQUFLVixNQUFMLENBQVlHLFNBQVosRUFBdUJNLFdBQXZCLENBQW1DLEdBQUdDLElBQXRDLENBQVA7QUFDRDs7QUFFREMsRUFBQUEsY0FBYyxDQUFDLEdBQUdELElBQUosRUFBVTtBQUN0QixXQUFPaEIsV0FBVyxDQUFDTyxNQUFaLENBQW1CLENBQUNXLEdBQUQsRUFBTVQsU0FBTixLQUFvQjtBQUM1Q1MsTUFBQUEsR0FBRyxDQUFDQyxJQUFKLENBQVMsR0FBRyxLQUFLSixXQUFMLENBQWlCTixTQUFqQixFQUE0QixHQUFHTyxJQUEvQixDQUFaO0FBQ0EsYUFBT0UsR0FBUDtBQUNELEtBSE0sRUFHSixFQUhJLENBQVA7QUFJRDs7QUFFREUsRUFBQUEsWUFBWSxDQUFDWCxTQUFELEVBQVksR0FBR08sSUFBZixFQUFxQjtBQUMvQixXQUFPLEtBQUtWLE1BQUwsQ0FBWUcsU0FBWixFQUF1QlcsWUFBdkIsQ0FBb0MsR0FBR0osSUFBdkMsQ0FBUDtBQUNEOztBQUVESyxFQUFBQSxTQUFTLENBQUNaLFNBQUQsRUFBWSxHQUFHTyxJQUFmLEVBQXFCO0FBQzVCLFdBQU8sS0FBS1YsTUFBTCxDQUFZRyxTQUFaLEVBQXVCWSxTQUF2QixDQUFpQyxHQUFHTCxJQUFwQyxDQUFQO0FBQ0Q7O0FBRURNLEVBQUFBLGNBQWMsR0FBRztBQUNmLFNBQUssTUFBTWIsU0FBWCxJQUF3QlQsV0FBeEIsRUFBcUM7QUFDbkMsV0FBS00sTUFBTCxDQUFZRyxTQUFaLEVBQXVCYyxLQUF2QjtBQUNEO0FBQ0Y7O0FBRURDLEVBQUFBLGdCQUFnQixDQUFDQyxjQUFELEVBQWlCO0FBQy9CLFdBQU8sSUFBSUMsUUFBSixDQUFhLElBQWIsRUFBbUJDLFlBQU1DLFVBQU4sQ0FBaUJILGNBQWpCLENBQW5CLENBQVA7QUFDRDs7QUFFREksRUFBQUEsbUJBQW1CLEdBQUc7QUFDcEIsV0FBTyxLQUFLTCxnQkFBTCxDQUFzQixLQUFLWixpQkFBTCxFQUF0QixDQUFQO0FBQ0Q7O0FBRURrQixFQUFBQSxlQUFlLENBQUNDLFNBQUQsRUFBWUMsT0FBTyxHQUFHLEVBQXRCLEVBQTBCO0FBQ3ZDLFVBQU1DLElBQUk7QUFDUkMsTUFBQUEsT0FBTyxFQUFFLElBQUlDLEdBQUo7QUFERCxPQUVMSCxPQUZLLENBQVY7O0FBS0EsVUFBTUksS0FBSyxHQUFHQyxZQUFNVCxVQUFOLENBQWlCRyxTQUFqQixDQUFkOztBQUNBLFVBQU1PLFVBQVUsR0FBR0YsS0FBSyxDQUFDRyxLQUFOLENBQVlDLE1BQVosRUFBbkI7QUFDQSxVQUFNQyxzQkFBc0IsR0FBR3pDLFdBQVcsQ0FBQ08sTUFBWixDQUFtQixDQUFDQyxHQUFELEVBQU1DLFNBQU4sS0FBb0I7QUFDcEVELE1BQUFBLEdBQUcsQ0FBQ0MsU0FBRCxDQUFILEdBQWlCLEtBQUtILE1BQUwsQ0FBWUcsU0FBWixFQUNkTSxXQURjLENBQ0Y7QUFBQzJCLFFBQUFBLGVBQWUsRUFBRU47QUFBbEIsT0FERSxFQUVkTyxNQUZjLENBRVBDLENBQUMsSUFBSSxDQUFDWCxJQUFJLENBQUNDLE9BQUwsQ0FBYVcsR0FBYixDQUFpQkQsQ0FBakIsQ0FGQyxDQUFqQjtBQUdBLGFBQU9wQyxHQUFQO0FBQ0QsS0FMOEIsRUFLNUIsRUFMNEIsQ0FBL0I7QUFNQSxVQUFNc0MsU0FBUyxHQUFHLElBQUlDLEdBQUosRUFBbEI7QUFFQSxVQUFNQyxTQUFTLEdBQUcsSUFBSS9DLFdBQUosRUFBbEI7QUFDQStDLElBQUFBLFNBQVMsQ0FBQ3JDLFNBQVYsR0FBc0JzQyxPQUF0QixDQUE4QixLQUFLOUMsTUFBTCxDQUFZK0MsY0FBWixDQUEyQmQsS0FBM0IsQ0FBOUI7O0FBRUEsU0FBSyxNQUFNM0IsU0FBWCxJQUF3QlQsV0FBeEIsRUFBcUM7QUFDbkMsV0FBSyxNQUFNbUQsU0FBWCxJQUF3QlYsc0JBQXNCLENBQUNoQyxTQUFELENBQTlDLEVBQTJEO0FBQ3pELGNBQU0yQyxRQUFRLEdBQUdELFNBQVMsQ0FBQ0UsUUFBVixFQUFqQjtBQUVBLGNBQU1DLFlBQVksR0FBR0YsUUFBUSxDQUFDYixLQUFULENBQWVnQixpQkFBZixDQUFpQ25CLEtBQUssQ0FBQ0csS0FBdkMsSUFBZ0RILEtBQUssQ0FBQ0csS0FBdEQsR0FBOERhLFFBQVEsQ0FBQ2IsS0FBNUY7QUFDQSxjQUFNaUIsVUFBVSxHQUFHSixRQUFRLENBQUNLLEdBQVQsQ0FBYUMsb0JBQWIsQ0FBa0N0QixLQUFLLENBQUNxQixHQUF4QyxJQUErQ3JCLEtBQUssQ0FBQ3FCLEdBQXJELEdBQTJETCxRQUFRLENBQUNLLEdBQXZGLENBSnlELENBTXpEOztBQUNBLFlBQUlILFlBQVksQ0FBQ0ssT0FBYixDQUFxQkgsVUFBckIsS0FBb0MsQ0FBQ0osUUFBUSxDQUFDYixLQUFULENBQWVvQixPQUFmLENBQXVCUCxRQUFRLENBQUNLLEdBQWhDLENBQXpDLEVBQStFO0FBQzdFO0FBQ0Q7O0FBRUQsY0FBTUcsV0FBVyxHQUFHTixZQUFZLENBQUNPLEdBQWIsS0FBcUJ6QixLQUFLLENBQUNHLEtBQU4sQ0FBWXNCLEdBQWpDLEdBQXVDdkIsVUFBdkMsR0FBb0QsQ0FBQ0EsVUFBVSxDQUFDdUIsR0FBWixFQUFpQixDQUFqQixDQUF4RTtBQUNBLGNBQU1DLFNBQVMsR0FBR04sVUFBVSxDQUFDSyxHQUFYLEtBQW1CekIsS0FBSyxDQUFDRyxLQUFOLENBQVlzQixHQUEvQixHQUFxQ3ZCLFVBQXJDLEdBQWtELENBQUNBLFVBQVUsQ0FBQ3VCLEdBQVosRUFBaUIsQ0FBakIsQ0FBcEU7QUFFQSxjQUFNRSxTQUFTLEdBQUdmLFNBQVMsQ0FBQzNCLFNBQVYsQ0FDaEJaLFNBRGdCLEVBRWhCLENBQUM2QyxZQUFZLENBQUNVLFNBQWIsQ0FBdUJKLFdBQXZCLENBQUQsRUFBc0NKLFVBQVUsQ0FBQ1EsU0FBWCxDQUFxQkYsU0FBckIsQ0FBdEMsQ0FGZ0IsRUFHaEJYLFNBQVMsQ0FBQ2MsYUFBVixFQUhnQixDQUFsQjtBQUtBbkIsUUFBQUEsU0FBUyxDQUFDb0IsR0FBVixDQUFjZixTQUFkLEVBQXlCWSxTQUF6QjtBQUNEO0FBQ0Y7O0FBRUQsV0FBTztBQUFDSSxNQUFBQSxXQUFXLEVBQUVuQixTQUFkO0FBQXlCRixNQUFBQTtBQUF6QixLQUFQO0FBQ0Q7O0FBRURzQixFQUFBQSxrQkFBa0IsQ0FBQ3JDLFNBQUQsRUFBWUMsT0FBTyxHQUFHLEVBQXRCLEVBQTBCO0FBQzFDLFVBQU07QUFBQ21DLE1BQUFBLFdBQVcsRUFBRW5CLFNBQWQ7QUFBeUJGLE1BQUFBO0FBQXpCLFFBQXNDLEtBQUtoQixlQUFMLENBQXFCQyxTQUFyQixFQUFnQ0MsT0FBaEMsQ0FBNUM7O0FBRUEsU0FBSyxNQUFNbUIsU0FBWCxJQUF3QkwsU0FBUyxDQUFDdUIsSUFBVixFQUF4QixFQUEwQztBQUN4Q2xCLE1BQUFBLFNBQVMsQ0FBQ21CLE9BQVY7QUFDRDs7QUFFRCxTQUFLbkUsTUFBTCxDQUFZb0UsY0FBWixDQUEyQnhDLFNBQTNCLEVBQXNDLEVBQXRDO0FBQ0EsV0FBTztBQUFDb0MsTUFBQUEsV0FBVyxFQUFFbkIsU0FBZDtBQUF5QkYsTUFBQUE7QUFBekIsS0FBUDtBQUNEOztBQUVEMEIsRUFBQUEsaUJBQWlCLEdBQUc7QUFDbEIsUUFBSSxLQUFLckUsTUFBTCxDQUFZc0UsV0FBWixPQUE4QixFQUFsQyxFQUFzQztBQUNwQyxXQUFLdEUsTUFBTCxDQUFZdUUsU0FBWixDQUFzQixLQUFLdkUsTUFBTCxDQUFZd0UsVUFBWixFQUF0QjtBQUNEOztBQUVELFdBQU8sSUFBUDtBQUNEOztBQUVEQyxFQUFBQSxLQUFLLENBQUNDLFFBQUQsRUFBVztBQUNkLFNBQUt2RCxjQUFMO0FBQ0EsU0FBS25CLE1BQUwsQ0FBWThDLE9BQVosQ0FBb0I0QixRQUFRLENBQUNsRSxTQUFULEdBQXFCbUUsT0FBckIsRUFBcEI7QUFFQSxVQUFNaEMsU0FBUyxHQUFHLElBQUlDLEdBQUosRUFBbEI7O0FBQ0EsU0FBSyxNQUFNdEMsU0FBWCxJQUF3QlQsV0FBeEIsRUFBcUM7QUFDbkMsV0FBSyxNQUFNK0UsY0FBWCxJQUE2QkYsUUFBUSxDQUFDL0QsUUFBVCxDQUFrQkwsU0FBbEIsRUFBNkJ1RSxVQUE3QixFQUE3QixFQUF3RTtBQUN0RSxjQUFNakIsU0FBUyxHQUFHLEtBQUsxQyxTQUFMLENBQWVaLFNBQWYsRUFBMEJzRSxjQUFjLENBQUMxQixRQUFmLEVBQTFCLEVBQXFEMEIsY0FBYyxDQUFDZCxhQUFmLEVBQXJELENBQWxCO0FBQ0FuQixRQUFBQSxTQUFTLENBQUNvQixHQUFWLENBQWNhLGNBQWQsRUFBOEJoQixTQUE5QjtBQUNEO0FBQ0Y7O0FBQ0QsV0FBT2pCLFNBQVA7QUFDRDtBQUVEOzs7QUFDQW1DLEVBQUFBLE9BQU8sQ0FBQ2hELElBQUksR0FBRyxFQUFSLEVBQVk7QUFDakI7QUFDQSxVQUFNRCxPQUFPO0FBQ1hrRCxNQUFBQSxVQUFVLEVBQUVsRjtBQURELE9BRVJpQyxJQUZRLENBQWI7O0FBS0EsUUFBSWtELGFBQWEsR0FBRyxFQUFwQjtBQUVBLFVBQU1DLGlCQUFpQixHQUFHLEVBQTFCOztBQUNBLFNBQUssTUFBTTNFLFNBQVgsSUFBd0J1QixPQUFPLENBQUNrRCxVQUFoQyxFQUE0QztBQUMxQyxXQUFLLE1BQU1HLE1BQVgsSUFBcUIsS0FBS3RFLFdBQUwsQ0FBaUJOLFNBQWpCLEVBQTRCLEVBQTVCLENBQXJCLEVBQXNEO0FBQ3BEMkUsUUFBQUEsaUJBQWlCLENBQUNqRSxJQUFsQixDQUF1QjtBQUFDVixVQUFBQSxTQUFEO0FBQVk2RSxVQUFBQSxLQUFLLEVBQUVELE1BQU0sQ0FBQ2hDLFFBQVAsR0FBa0JkLEtBQXJDO0FBQTRDQSxVQUFBQSxLQUFLLEVBQUUsSUFBbkQ7QUFBeURnRCxVQUFBQSxFQUFFLEVBQUVGLE1BQU0sQ0FBQ0U7QUFBcEUsU0FBdkI7QUFDQUgsUUFBQUEsaUJBQWlCLENBQUNqRSxJQUFsQixDQUF1QjtBQUFDVixVQUFBQSxTQUFEO0FBQVk2RSxVQUFBQSxLQUFLLEVBQUVELE1BQU0sQ0FBQ2hDLFFBQVAsR0FBa0JJLEdBQXJDO0FBQTBDQSxVQUFBQSxHQUFHLEVBQUUsSUFBL0M7QUFBcUQ4QixVQUFBQSxFQUFFLEVBQUVGLE1BQU0sQ0FBQ0U7QUFBaEUsU0FBdkI7QUFDRDtBQUNGOztBQUNESCxJQUFBQSxpQkFBaUIsQ0FBQ0ksSUFBbEIsQ0FBdUIsQ0FBQ0MsQ0FBRCxFQUFJQyxDQUFKLEtBQVU7QUFDL0IsWUFBTUMsR0FBRyxHQUFHRixDQUFDLENBQUNILEtBQUYsQ0FBUU0sT0FBUixDQUFnQkYsQ0FBQyxDQUFDSixLQUFsQixDQUFaOztBQUNBLFVBQUlLLEdBQUcsS0FBSyxDQUFaLEVBQWU7QUFDYixlQUFPQSxHQUFQO0FBQ0QsT0FGRCxNQUVPLElBQUlGLENBQUMsQ0FBQ2xELEtBQUYsSUFBV21ELENBQUMsQ0FBQ25ELEtBQWpCLEVBQXdCO0FBQzdCLGVBQU8sQ0FBUDtBQUNELE9BRk0sTUFFQSxJQUFJa0QsQ0FBQyxDQUFDbEQsS0FBRixJQUFXLENBQUNtRCxDQUFDLENBQUNuRCxLQUFsQixFQUF5QjtBQUM5QixlQUFPLENBQUMsQ0FBUjtBQUNELE9BRk0sTUFFQSxJQUFJLENBQUNrRCxDQUFDLENBQUNsRCxLQUFILElBQVltRCxDQUFDLENBQUNuRCxLQUFsQixFQUF5QjtBQUM5QixlQUFPLENBQVA7QUFDRCxPQUZNLE1BRUE7QUFDTCxlQUFPLENBQVA7QUFDRDtBQUNGLEtBYkQ7O0FBZUEsUUFBSXNELFlBQVksR0FBR2xFLFlBQU1DLFVBQU4sQ0FBaUIsQ0FBQyxDQUFELEVBQUksQ0FBSixDQUFqQixDQUFuQjs7QUFDQSxTQUFLLE1BQU15RCxNQUFYLElBQXFCRCxpQkFBckIsRUFBd0M7QUFDdEMsVUFBSSxDQUFDQyxNQUFNLENBQUNDLEtBQVAsQ0FBYTNCLE9BQWIsQ0FBcUJrQyxZQUFyQixDQUFMLEVBQXlDO0FBQ3ZDVixRQUFBQSxhQUFhLElBQUksbUJBQVEsS0FBS2hGLE1BQUwsQ0FBWStDLGNBQVosQ0FBMkIsQ0FBQzJDLFlBQUQsRUFBZVIsTUFBTSxDQUFDQyxLQUF0QixDQUEzQixDQUFSLElBQW9FLElBQXJGO0FBQ0Q7O0FBRUQsVUFBSUQsTUFBTSxDQUFDOUMsS0FBWCxFQUFrQjtBQUNoQjRDLFFBQUFBLGFBQWEsSUFBSyxXQUFVRSxNQUFNLENBQUM1RSxTQUFVLElBQUc0RSxNQUFNLENBQUNFLEVBQUcsSUFBMUQ7QUFDRCxPQUZELE1BRU8sSUFBSUYsTUFBTSxDQUFDNUIsR0FBWCxFQUFnQjtBQUNyQjBCLFFBQUFBLGFBQWEsSUFBSyxTQUFRRSxNQUFNLENBQUM1RSxTQUFVLElBQUc0RSxNQUFNLENBQUNFLEVBQUcsSUFBeEQ7QUFDRDs7QUFFRE0sTUFBQUEsWUFBWSxHQUFHUixNQUFNLENBQUNDLEtBQXRCO0FBQ0Q7O0FBRUQsV0FBT0gsYUFBUDtBQUNEOztBQXZMOEI7Ozs7QUEwTGpDLE1BQU16RCxRQUFOLENBQWU7QUFDYnhCLEVBQUFBLFdBQVcsQ0FBQ2lFLFdBQUQsRUFBYzFDLGNBQWQsRUFBOEI7QUFDdkMsVUFBTXFFLE9BQU8sR0FBRzNCLFdBQVcsQ0FBQ3hELFNBQVosR0FBd0JvRixZQUF4QixDQUFxQ3RFLGNBQXJDLENBQWhCO0FBRUEsU0FBSzBDLFdBQUwsR0FBbUJBLFdBQW5CO0FBQ0EsU0FBSzZCLFVBQUwsR0FBa0JGLE9BQU8sQ0FBQ0csSUFBUixFQUFsQjtBQUNBLFNBQUt4RSxjQUFMLEdBQXNCcUUsT0FBTyxDQUFDRyxJQUFSLEVBQXRCO0FBQ0EsU0FBS0MsZ0JBQUwsR0FBd0IsRUFBeEI7QUFDQSxTQUFLQyxrQkFBTCxHQUEwQixFQUExQjtBQUVBLFNBQUtDLGFBQUwsR0FBcUIsSUFBSWpFLEdBQUosRUFBckI7QUFDQSxTQUFLa0UsWUFBTCxHQUFvQixJQUFJbEUsR0FBSixFQUFwQjtBQUNEOztBQUVEbUUsRUFBQUEsVUFBVSxDQUFDQyxPQUFELEVBQVU7QUFDbEIsU0FBSyxNQUFNbEIsTUFBWCxJQUFxQmtCLE9BQXJCLEVBQThCO0FBQzVCLFVBQUlsQixNQUFNLENBQUNoQyxRQUFQLEdBQWtCSSxHQUFsQixDQUFzQkUsT0FBdEIsQ0FBOEIsS0FBS3FDLFVBQW5DLENBQUosRUFBb0Q7QUFDbEQsYUFBS0ksYUFBTCxDQUFtQkksR0FBbkIsQ0FBdUJuQixNQUF2QjtBQUNEO0FBQ0Y7O0FBQ0QsV0FBTyxJQUFQO0FBQ0Q7O0FBRURvQixFQUFBQSxTQUFTLENBQUNGLE9BQUQsRUFBVTtBQUNqQixTQUFLLE1BQU1sQixNQUFYLElBQXFCa0IsT0FBckIsRUFBOEI7QUFDNUIsVUFBSWxCLE1BQU0sQ0FBQ2hDLFFBQVAsR0FBa0JkLEtBQWxCLENBQXdCb0IsT0FBeEIsQ0FBZ0MsS0FBS3FDLFVBQXJDLENBQUosRUFBc0Q7QUFDcEQsYUFBS0ssWUFBTCxDQUFrQkcsR0FBbEIsQ0FBc0JuQixNQUF0QjtBQUNEO0FBQ0Y7O0FBQ0QsV0FBTyxJQUFQO0FBQ0Q7O0FBRURxQixFQUFBQSxTQUFTLENBQUNqRyxTQUFELEVBQVlrRyxLQUFaLEVBQW1CQyxVQUFuQixFQUErQjtBQUN0QyxVQUFNckUsS0FBSyxHQUFHLEtBQUtkLGNBQUwsQ0FBb0J3RSxJQUFwQixFQUFkO0FBQ0FVLElBQUFBLEtBQUs7QUFDTCxVQUFNbEQsR0FBRyxHQUFHLEtBQUtoQyxjQUFMLENBQW9Cd0UsSUFBcEIsRUFBWjtBQUNBLFNBQUtDLGdCQUFMLENBQXNCL0UsSUFBdEIsQ0FBMkI7QUFBQ1YsTUFBQUEsU0FBRDtBQUFZMkIsTUFBQUEsS0FBSyxFQUFFLElBQUlDLFdBQUosQ0FBVUUsS0FBVixFQUFpQmtCLEdBQWpCLENBQW5CO0FBQTBDbUQsTUFBQUE7QUFBMUMsS0FBM0I7QUFDQSxXQUFPLElBQVA7QUFDRDs7QUFFREMsRUFBQUEsTUFBTSxDQUFDQyxJQUFELEVBQU87QUFDWCxVQUFNQyxhQUFhLEdBQUcsS0FBSzVDLFdBQUwsQ0FBaUJ4RCxTQUFqQixHQUE2QmtHLE1BQTdCLENBQW9DLEtBQUtwRixjQUF6QyxFQUF5RHFGLElBQXpELENBQXRCO0FBQ0EsU0FBS3JGLGNBQUwsR0FBc0JzRixhQUFhLENBQUN0RCxHQUFwQztBQUNBLFdBQU8sSUFBUDtBQUNEOztBQUVEdUQsRUFBQUEsWUFBWSxDQUFDRixJQUFELEVBQU9yRyxTQUFQLEVBQWtCbUcsVUFBbEIsRUFBOEI7QUFDeEMsV0FBTyxLQUFLRixTQUFMLENBQWVqRyxTQUFmLEVBQTBCLE1BQU0sS0FBS29HLE1BQUwsQ0FBWUMsSUFBWixDQUFoQyxFQUFtREYsVUFBbkQsQ0FBUDtBQUNEOztBQUVESyxFQUFBQSxpQkFBaUIsQ0FBQ0MsY0FBRCxFQUFpQmpGLElBQWpCLEVBQXVCO0FBQ3RDLFVBQU1LLFVBQVUsR0FBRyxLQUFLYixjQUFMLENBQW9Cd0UsSUFBcEIsRUFBbkI7QUFDQSxTQUFLWSxNQUFMLENBQVlLLGNBQWMsQ0FBQ3ZHLFNBQWYsR0FBMkJtRSxPQUEzQixFQUFaO0FBRUEsVUFBTXFDLFlBQVksR0FBRyxJQUFJcEUsR0FBSixFQUFyQjs7QUFDQSxTQUFLLE1BQU10QyxTQUFYLElBQXdCVCxXQUF4QixFQUFxQztBQUNuQyxXQUFLLE1BQU1tRCxTQUFYLElBQXdCK0QsY0FBYyxDQUFDbkcsV0FBZixDQUEyQk4sU0FBM0IsRUFBc0MsRUFBdEMsQ0FBeEIsRUFBbUU7QUFDakUsY0FBTW1ELFdBQVcsR0FBR1QsU0FBUyxDQUFDRSxRQUFWLEdBQXFCZCxLQUFyQixDQUEyQnNCLEdBQTNCLEtBQW1DLENBQW5DLEdBQXVDdkIsVUFBdkMsR0FBb0QsQ0FBQ0EsVUFBVSxDQUFDdUIsR0FBWixFQUFpQixDQUFqQixDQUF4RTtBQUNBLGNBQU1DLFNBQVMsR0FBR1gsU0FBUyxDQUFDRSxRQUFWLEdBQXFCSSxHQUFyQixDQUF5QkksR0FBekIsS0FBaUMsQ0FBakMsR0FBcUN2QixVQUFyQyxHQUFrRCxDQUFDQSxVQUFVLENBQUN1QixHQUFaLEVBQWlCLENBQWpCLENBQXBFO0FBRUEsY0FBTXpCLEtBQUssR0FBR2UsU0FBUyxDQUFDRSxRQUFWLEdBQXFCVyxTQUFyQixDQUErQkosV0FBL0IsRUFBNENFLFNBQTVDLENBQWQ7O0FBQ0EsY0FBTThDLFVBQVUsc0JBQ1h6RCxTQUFTLENBQUNjLGFBQVYsRUFEVztBQUVkbUQsVUFBQUEsUUFBUSxFQUFFckQsU0FBUyxJQUFJO0FBQUVvRCxZQUFBQSxZQUFZLENBQUNqRCxHQUFiLENBQWlCZixTQUFqQixFQUE0QlksU0FBNUI7QUFBeUM7QUFGcEQsVUFBaEI7O0FBSUEsYUFBS21DLGdCQUFMLENBQXNCL0UsSUFBdEIsQ0FBMkI7QUFBQ1YsVUFBQUEsU0FBRDtBQUFZMkIsVUFBQUEsS0FBWjtBQUFtQndFLFVBQUFBO0FBQW5CLFNBQTNCO0FBQ0Q7QUFDRjs7QUFFRCxTQUFLVCxrQkFBTCxDQUF3QmhGLElBQXhCLENBQTZCO0FBQUMyQixNQUFBQSxTQUFTLEVBQUVxRSxZQUFaO0FBQTBCQyxNQUFBQSxRQUFRLEVBQUVuRixJQUFJLENBQUNtRjtBQUF6QyxLQUE3QjtBQUVBLFdBQU8sSUFBUDtBQUNEOztBQUVEQyxFQUFBQSxLQUFLLEdBQUc7QUFDTixTQUFLLE1BQU07QUFBQzVHLE1BQUFBLFNBQUQ7QUFBWTJCLE1BQUFBLEtBQVo7QUFBbUJ3RSxNQUFBQTtBQUFuQixLQUFYLElBQTZDLEtBQUtWLGdCQUFsRCxFQUFvRTtBQUNsRSxZQUFNa0IsUUFBUSxHQUFHUixVQUFVLENBQUNRLFFBQTVCO0FBQ0EsYUFBT1IsVUFBVSxDQUFDUSxRQUFsQjtBQUVBLFlBQU0vQixNQUFNLEdBQUcsS0FBS2xCLFdBQUwsQ0FBaUI5QyxTQUFqQixDQUEyQlosU0FBM0IsRUFBc0MyQixLQUF0QyxFQUE2Q3dFLFVBQTdDLENBQWY7O0FBQ0EsVUFBSVEsUUFBSixFQUFjO0FBQ1pBLFFBQUFBLFFBQVEsQ0FBQy9CLE1BQUQsQ0FBUjtBQUNEO0FBQ0Y7O0FBRUQsU0FBSyxNQUFNO0FBQUN2QyxNQUFBQSxTQUFEO0FBQVlzRSxNQUFBQTtBQUFaLEtBQVgsSUFBb0MsS0FBS2pCLGtCQUF6QyxFQUE2RDtBQUMzRGlCLE1BQUFBLFFBQVEsQ0FBQ3RFLFNBQUQsQ0FBUjtBQUNEOztBQUVELFNBQUssTUFBTXdFLFlBQVgsSUFBMkIsS0FBS2xCLGFBQWhDLEVBQStDO0FBQzdDLFlBQU1tQixPQUFPLEdBQUdELFlBQVksQ0FBQ2pFLFFBQWIsR0FBd0JrRSxPQUF4QixFQUFoQjs7QUFFQSxVQUFJLENBQUNELFlBQVksQ0FBQ0UsVUFBYixFQUFMLEVBQWdDO0FBQzlCRixRQUFBQSxZQUFZLENBQUNHLGVBQWIsQ0FBNkIsS0FBS3pCLFVBQWxDOztBQUNBLFlBQUl1QixPQUFKLEVBQWE7QUFDWEQsVUFBQUEsWUFBWSxDQUFDSSxlQUFiLENBQTZCLEtBQUsxQixVQUFsQztBQUNEO0FBQ0YsT0FMRCxNQUtPO0FBQ0xzQixRQUFBQSxZQUFZLENBQUNJLGVBQWIsQ0FBNkIsS0FBSzFCLFVBQWxDOztBQUNBLFlBQUl1QixPQUFKLEVBQWE7QUFDWEQsVUFBQUEsWUFBWSxDQUFDRyxlQUFiLENBQTZCLEtBQUt6QixVQUFsQztBQUNEO0FBQ0Y7QUFDRjs7QUFFRCxTQUFLLE1BQU0yQixXQUFYLElBQTBCLEtBQUt0QixZQUEvQixFQUE2QztBQUMzQyxZQUFNa0IsT0FBTyxHQUFHSSxXQUFXLENBQUN0RSxRQUFaLEdBQXVCa0UsT0FBdkIsRUFBaEI7O0FBRUEsVUFBSSxDQUFDSSxXQUFXLENBQUNILFVBQVosRUFBTCxFQUErQjtBQUM3QkcsUUFBQUEsV0FBVyxDQUFDRCxlQUFaLENBQTRCLEtBQUtqRyxjQUFqQzs7QUFDQSxZQUFJOEYsT0FBSixFQUFhO0FBQ1hJLFVBQUFBLFdBQVcsQ0FBQ0YsZUFBWixDQUE0QixLQUFLaEcsY0FBakM7QUFDRDtBQUNGLE9BTEQsTUFLTztBQUNMa0csUUFBQUEsV0FBVyxDQUFDRixlQUFaLENBQTRCLEtBQUtoRyxjQUFqQzs7QUFDQSxZQUFJOEYsT0FBSixFQUFhO0FBQ1hJLFVBQUFBLFdBQVcsQ0FBQ0QsZUFBWixDQUE0QixLQUFLakcsY0FBakM7QUFDRDtBQUNGO0FBQ0Y7QUFDRjs7QUF4SFkiLCJzb3VyY2VSb290IjoiL2J1aWxkL2F0b20vc3JjL2F0b20vb3V0L2FwcC9ub2RlX21vZHVsZXMvZ2l0aHViIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHtUZXh0QnVmZmVyLCBSYW5nZSwgUG9pbnR9IGZyb20gJ2F0b20nO1xuaW1wb3J0IHtpbnNwZWN0fSBmcm9tICd1dGlsJztcblxuY29uc3QgTEFZRVJfTkFNRVMgPSBbJ3VuY2hhbmdlZCcsICdhZGRpdGlvbicsICdkZWxldGlvbicsICdub25ld2xpbmUnLCAnaHVuaycsICdwYXRjaCddO1xuXG5leHBvcnQgZGVmYXVsdCBjbGFzcyBQYXRjaEJ1ZmZlciB7XG4gIGNvbnN0cnVjdG9yKCkge1xuICAgIHRoaXMuYnVmZmVyID0gbmV3IFRleHRCdWZmZXIoKTtcbiAgICB0aGlzLmJ1ZmZlci5yZXRhaW4oKTtcblxuICAgIHRoaXMubGF5ZXJzID0gTEFZRVJfTkFNRVMucmVkdWNlKChtYXAsIGxheWVyTmFtZSkgPT4ge1xuICAgICAgbWFwW2xheWVyTmFtZV0gPSB0aGlzLmJ1ZmZlci5hZGRNYXJrZXJMYXllcigpO1xuICAgICAgcmV0dXJuIG1hcDtcbiAgICB9LCB7fSk7XG4gIH1cblxuICBnZXRCdWZmZXIoKSB7XG4gICAgcmV0dXJuIHRoaXMuYnVmZmVyO1xuICB9XG5cbiAgZ2V0SW5zZXJ0aW9uUG9pbnQoKSB7XG4gICAgcmV0dXJuIHRoaXMuYnVmZmVyLmdldEVuZFBvc2l0aW9uKCk7XG4gIH1cblxuICBnZXRMYXllcihsYXllck5hbWUpIHtcbiAgICByZXR1cm4gdGhpcy5sYXllcnNbbGF5ZXJOYW1lXTtcbiAgfVxuXG4gIGZpbmRNYXJrZXJzKGxheWVyTmFtZSwgLi4uYXJncykge1xuICAgIHJldHVybiB0aGlzLmxheWVyc1tsYXllck5hbWVdLmZpbmRNYXJrZXJzKC4uLmFyZ3MpO1xuICB9XG5cbiAgZmluZEFsbE1hcmtlcnMoLi4uYXJncykge1xuICAgIHJldHVybiBMQVlFUl9OQU1FUy5yZWR1Y2UoKGFyciwgbGF5ZXJOYW1lKSA9PiB7XG4gICAgICBhcnIucHVzaCguLi50aGlzLmZpbmRNYXJrZXJzKGxheWVyTmFtZSwgLi4uYXJncykpO1xuICAgICAgcmV0dXJuIGFycjtcbiAgICB9LCBbXSk7XG4gIH1cblxuICBtYXJrUG9zaXRpb24obGF5ZXJOYW1lLCAuLi5hcmdzKSB7XG4gICAgcmV0dXJuIHRoaXMubGF5ZXJzW2xheWVyTmFtZV0ubWFya1Bvc2l0aW9uKC4uLmFyZ3MpO1xuICB9XG5cbiAgbWFya1JhbmdlKGxheWVyTmFtZSwgLi4uYXJncykge1xuICAgIHJldHVybiB0aGlzLmxheWVyc1tsYXllck5hbWVdLm1hcmtSYW5nZSguLi5hcmdzKTtcbiAgfVxuXG4gIGNsZWFyQWxsTGF5ZXJzKCkge1xuICAgIGZvciAoY29uc3QgbGF5ZXJOYW1lIG9mIExBWUVSX05BTUVTKSB7XG4gICAgICB0aGlzLmxheWVyc1tsYXllck5hbWVdLmNsZWFyKCk7XG4gICAgfVxuICB9XG5cbiAgY3JlYXRlSW5zZXJ0ZXJBdChpbnNlcnRpb25Qb2ludCkge1xuICAgIHJldHVybiBuZXcgSW5zZXJ0ZXIodGhpcywgUG9pbnQuZnJvbU9iamVjdChpbnNlcnRpb25Qb2ludCkpO1xuICB9XG5cbiAgY3JlYXRlSW5zZXJ0ZXJBdEVuZCgpIHtcbiAgICByZXR1cm4gdGhpcy5jcmVhdGVJbnNlcnRlckF0KHRoaXMuZ2V0SW5zZXJ0aW9uUG9pbnQoKSk7XG4gIH1cblxuICBjcmVhdGVTdWJCdWZmZXIocmFuZ2VMaWtlLCBvcHRpb25zID0ge30pIHtcbiAgICBjb25zdCBvcHRzID0ge1xuICAgICAgZXhjbHVkZTogbmV3IFNldCgpLFxuICAgICAgLi4ub3B0aW9ucyxcbiAgICB9O1xuXG4gICAgY29uc3QgcmFuZ2UgPSBSYW5nZS5mcm9tT2JqZWN0KHJhbmdlTGlrZSk7XG4gICAgY29uc3QgYmFzZU9mZnNldCA9IHJhbmdlLnN0YXJ0Lm5lZ2F0ZSgpO1xuICAgIGNvbnN0IGluY2x1ZGVkTWFya2Vyc0J5TGF5ZXIgPSBMQVlFUl9OQU1FUy5yZWR1Y2UoKG1hcCwgbGF5ZXJOYW1lKSA9PiB7XG4gICAgICBtYXBbbGF5ZXJOYW1lXSA9IHRoaXMubGF5ZXJzW2xheWVyTmFtZV1cbiAgICAgICAgLmZpbmRNYXJrZXJzKHtpbnRlcnNlY3RzUmFuZ2U6IHJhbmdlfSlcbiAgICAgICAgLmZpbHRlcihtID0+ICFvcHRzLmV4Y2x1ZGUuaGFzKG0pKTtcbiAgICAgIHJldHVybiBtYXA7XG4gICAgfSwge30pO1xuICAgIGNvbnN0IG1hcmtlck1hcCA9IG5ldyBNYXAoKTtcblxuICAgIGNvbnN0IHN1YkJ1ZmZlciA9IG5ldyBQYXRjaEJ1ZmZlcigpO1xuICAgIHN1YkJ1ZmZlci5nZXRCdWZmZXIoKS5zZXRUZXh0KHRoaXMuYnVmZmVyLmdldFRleHRJblJhbmdlKHJhbmdlKSk7XG5cbiAgICBmb3IgKGNvbnN0IGxheWVyTmFtZSBvZiBMQVlFUl9OQU1FUykge1xuICAgICAgZm9yIChjb25zdCBvbGRNYXJrZXIgb2YgaW5jbHVkZWRNYXJrZXJzQnlMYXllcltsYXllck5hbWVdKSB7XG4gICAgICAgIGNvbnN0IG9sZFJhbmdlID0gb2xkTWFya2VyLmdldFJhbmdlKCk7XG5cbiAgICAgICAgY29uc3QgY2xpcHBlZFN0YXJ0ID0gb2xkUmFuZ2Uuc3RhcnQuaXNMZXNzVGhhbk9yRXF1YWwocmFuZ2Uuc3RhcnQpID8gcmFuZ2Uuc3RhcnQgOiBvbGRSYW5nZS5zdGFydDtcbiAgICAgICAgY29uc3QgY2xpcHBlZEVuZCA9IG9sZFJhbmdlLmVuZC5pc0dyZWF0ZXJUaGFuT3JFcXVhbChyYW5nZS5lbmQpID8gcmFuZ2UuZW5kIDogb2xkUmFuZ2UuZW5kO1xuXG4gICAgICAgIC8vIEV4Y2x1ZGUgbm9uLWVtcHR5IG1hcmtlcnMgdGhhdCBpbnRlcnNlY3QgKm9ubHkqIGF0IHRoZSByYW5nZSBzdGFydCBvciBlbmRcbiAgICAgICAgaWYgKGNsaXBwZWRTdGFydC5pc0VxdWFsKGNsaXBwZWRFbmQpICYmICFvbGRSYW5nZS5zdGFydC5pc0VxdWFsKG9sZFJhbmdlLmVuZCkpIHtcbiAgICAgICAgICBjb250aW51ZTtcbiAgICAgICAgfVxuXG4gICAgICAgIGNvbnN0IHN0YXJ0T2Zmc2V0ID0gY2xpcHBlZFN0YXJ0LnJvdyA9PT0gcmFuZ2Uuc3RhcnQucm93ID8gYmFzZU9mZnNldCA6IFtiYXNlT2Zmc2V0LnJvdywgMF07XG4gICAgICAgIGNvbnN0IGVuZE9mZnNldCA9IGNsaXBwZWRFbmQucm93ID09PSByYW5nZS5zdGFydC5yb3cgPyBiYXNlT2Zmc2V0IDogW2Jhc2VPZmZzZXQucm93LCAwXTtcblxuICAgICAgICBjb25zdCBuZXdNYXJrZXIgPSBzdWJCdWZmZXIubWFya1JhbmdlKFxuICAgICAgICAgIGxheWVyTmFtZSxcbiAgICAgICAgICBbY2xpcHBlZFN0YXJ0LnRyYW5zbGF0ZShzdGFydE9mZnNldCksIGNsaXBwZWRFbmQudHJhbnNsYXRlKGVuZE9mZnNldCldLFxuICAgICAgICAgIG9sZE1hcmtlci5nZXRQcm9wZXJ0aWVzKCksXG4gICAgICAgICk7XG4gICAgICAgIG1hcmtlck1hcC5zZXQob2xkTWFya2VyLCBuZXdNYXJrZXIpO1xuICAgICAgfVxuICAgIH1cblxuICAgIHJldHVybiB7cGF0Y2hCdWZmZXI6IHN1YkJ1ZmZlciwgbWFya2VyTWFwfTtcbiAgfVxuXG4gIGV4dHJhY3RQYXRjaEJ1ZmZlcihyYW5nZUxpa2UsIG9wdGlvbnMgPSB7fSkge1xuICAgIGNvbnN0IHtwYXRjaEJ1ZmZlcjogc3ViQnVmZmVyLCBtYXJrZXJNYXB9ID0gdGhpcy5jcmVhdGVTdWJCdWZmZXIocmFuZ2VMaWtlLCBvcHRpb25zKTtcblxuICAgIGZvciAoY29uc3Qgb2xkTWFya2VyIG9mIG1hcmtlck1hcC5rZXlzKCkpIHtcbiAgICAgIG9sZE1hcmtlci5kZXN0cm95KCk7XG4gICAgfVxuXG4gICAgdGhpcy5idWZmZXIuc2V0VGV4dEluUmFuZ2UocmFuZ2VMaWtlLCAnJyk7XG4gICAgcmV0dXJuIHtwYXRjaEJ1ZmZlcjogc3ViQnVmZmVyLCBtYXJrZXJNYXB9O1xuICB9XG5cbiAgZGVsZXRlTGFzdE5ld2xpbmUoKSB7XG4gICAgaWYgKHRoaXMuYnVmZmVyLmdldExhc3RMaW5lKCkgPT09ICcnKSB7XG4gICAgICB0aGlzLmJ1ZmZlci5kZWxldGVSb3codGhpcy5idWZmZXIuZ2V0TGFzdFJvdygpKTtcbiAgICB9XG5cbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIGFkb3B0KG9yaWdpbmFsKSB7XG4gICAgdGhpcy5jbGVhckFsbExheWVycygpO1xuICAgIHRoaXMuYnVmZmVyLnNldFRleHQob3JpZ2luYWwuZ2V0QnVmZmVyKCkuZ2V0VGV4dCgpKTtcblxuICAgIGNvbnN0IG1hcmtlck1hcCA9IG5ldyBNYXAoKTtcbiAgICBmb3IgKGNvbnN0IGxheWVyTmFtZSBvZiBMQVlFUl9OQU1FUykge1xuICAgICAgZm9yIChjb25zdCBvcmlnaW5hbE1hcmtlciBvZiBvcmlnaW5hbC5nZXRMYXllcihsYXllck5hbWUpLmdldE1hcmtlcnMoKSkge1xuICAgICAgICBjb25zdCBuZXdNYXJrZXIgPSB0aGlzLm1hcmtSYW5nZShsYXllck5hbWUsIG9yaWdpbmFsTWFya2VyLmdldFJhbmdlKCksIG9yaWdpbmFsTWFya2VyLmdldFByb3BlcnRpZXMoKSk7XG4gICAgICAgIG1hcmtlck1hcC5zZXQob3JpZ2luYWxNYXJrZXIsIG5ld01hcmtlcik7XG4gICAgICB9XG4gICAgfVxuICAgIHJldHVybiBtYXJrZXJNYXA7XG4gIH1cblxuICAvKiBpc3RhbmJ1bCBpZ25vcmUgbmV4dCAqL1xuICBpbnNwZWN0KG9wdHMgPSB7fSkge1xuICAgIC8qIGlzdGFuYnVsIGlnbm9yZSBuZXh0ICovXG4gICAgY29uc3Qgb3B0aW9ucyA9IHtcbiAgICAgIGxheWVyTmFtZXM6IExBWUVSX05BTUVTLFxuICAgICAgLi4ub3B0cyxcbiAgICB9O1xuXG4gICAgbGV0IGluc3BlY3RTdHJpbmcgPSAnJztcblxuICAgIGNvbnN0IGluY3JlYXNpbmdNYXJrZXJzID0gW107XG4gICAgZm9yIChjb25zdCBsYXllck5hbWUgb2Ygb3B0aW9ucy5sYXllck5hbWVzKSB7XG4gICAgICBmb3IgKGNvbnN0IG1hcmtlciBvZiB0aGlzLmZpbmRNYXJrZXJzKGxheWVyTmFtZSwge30pKSB7XG4gICAgICAgIGluY3JlYXNpbmdNYXJrZXJzLnB1c2goe2xheWVyTmFtZSwgcG9pbnQ6IG1hcmtlci5nZXRSYW5nZSgpLnN0YXJ0LCBzdGFydDogdHJ1ZSwgaWQ6IG1hcmtlci5pZH0pO1xuICAgICAgICBpbmNyZWFzaW5nTWFya2Vycy5wdXNoKHtsYXllck5hbWUsIHBvaW50OiBtYXJrZXIuZ2V0UmFuZ2UoKS5lbmQsIGVuZDogdHJ1ZSwgaWQ6IG1hcmtlci5pZH0pO1xuICAgICAgfVxuICAgIH1cbiAgICBpbmNyZWFzaW5nTWFya2Vycy5zb3J0KChhLCBiKSA9PiB7XG4gICAgICBjb25zdCBjbXAgPSBhLnBvaW50LmNvbXBhcmUoYi5wb2ludCk7XG4gICAgICBpZiAoY21wICE9PSAwKSB7XG4gICAgICAgIHJldHVybiBjbXA7XG4gICAgICB9IGVsc2UgaWYgKGEuc3RhcnQgJiYgYi5zdGFydCkge1xuICAgICAgICByZXR1cm4gMDtcbiAgICAgIH0gZWxzZSBpZiAoYS5zdGFydCAmJiAhYi5zdGFydCkge1xuICAgICAgICByZXR1cm4gLTE7XG4gICAgICB9IGVsc2UgaWYgKCFhLnN0YXJ0ICYmIGIuc3RhcnQpIHtcbiAgICAgICAgcmV0dXJuIDE7XG4gICAgICB9IGVsc2Uge1xuICAgICAgICByZXR1cm4gMDtcbiAgICAgIH1cbiAgICB9KTtcblxuICAgIGxldCBpbnNwZWN0UG9pbnQgPSBQb2ludC5mcm9tT2JqZWN0KFswLCAwXSk7XG4gICAgZm9yIChjb25zdCBtYXJrZXIgb2YgaW5jcmVhc2luZ01hcmtlcnMpIHtcbiAgICAgIGlmICghbWFya2VyLnBvaW50LmlzRXF1YWwoaW5zcGVjdFBvaW50KSkge1xuICAgICAgICBpbnNwZWN0U3RyaW5nICs9IGluc3BlY3QodGhpcy5idWZmZXIuZ2V0VGV4dEluUmFuZ2UoW2luc3BlY3RQb2ludCwgbWFya2VyLnBvaW50XSkpICsgJ1xcbic7XG4gICAgICB9XG5cbiAgICAgIGlmIChtYXJrZXIuc3RhcnQpIHtcbiAgICAgICAgaW5zcGVjdFN0cmluZyArPSBgICBzdGFydCAke21hcmtlci5sYXllck5hbWV9QCR7bWFya2VyLmlkfVxcbmA7XG4gICAgICB9IGVsc2UgaWYgKG1hcmtlci5lbmQpIHtcbiAgICAgICAgaW5zcGVjdFN0cmluZyArPSBgICBlbmQgJHttYXJrZXIubGF5ZXJOYW1lfUAke21hcmtlci5pZH1cXG5gO1xuICAgICAgfVxuXG4gICAgICBpbnNwZWN0UG9pbnQgPSBtYXJrZXIucG9pbnQ7XG4gICAgfVxuXG4gICAgcmV0dXJuIGluc3BlY3RTdHJpbmc7XG4gIH1cbn1cblxuY2xhc3MgSW5zZXJ0ZXIge1xuICBjb25zdHJ1Y3RvcihwYXRjaEJ1ZmZlciwgaW5zZXJ0aW9uUG9pbnQpIHtcbiAgICBjb25zdCBjbGlwcGVkID0gcGF0Y2hCdWZmZXIuZ2V0QnVmZmVyKCkuY2xpcFBvc2l0aW9uKGluc2VydGlvblBvaW50KTtcblxuICAgIHRoaXMucGF0Y2hCdWZmZXIgPSBwYXRjaEJ1ZmZlcjtcbiAgICB0aGlzLnN0YXJ0UG9pbnQgPSBjbGlwcGVkLmNvcHkoKTtcbiAgICB0aGlzLmluc2VydGlvblBvaW50ID0gY2xpcHBlZC5jb3B5KCk7XG4gICAgdGhpcy5tYXJrZXJCbHVlcHJpbnRzID0gW107XG4gICAgdGhpcy5tYXJrZXJNYXBDYWxsYmFja3MgPSBbXTtcblxuICAgIHRoaXMubWFya2Vyc0JlZm9yZSA9IG5ldyBTZXQoKTtcbiAgICB0aGlzLm1hcmtlcnNBZnRlciA9IG5ldyBTZXQoKTtcbiAgfVxuXG4gIGtlZXBCZWZvcmUobWFya2Vycykge1xuICAgIGZvciAoY29uc3QgbWFya2VyIG9mIG1hcmtlcnMpIHtcbiAgICAgIGlmIChtYXJrZXIuZ2V0UmFuZ2UoKS5lbmQuaXNFcXVhbCh0aGlzLnN0YXJ0UG9pbnQpKSB7XG4gICAgICAgIHRoaXMubWFya2Vyc0JlZm9yZS5hZGQobWFya2VyKTtcbiAgICAgIH1cbiAgICB9XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICBrZWVwQWZ0ZXIobWFya2Vycykge1xuICAgIGZvciAoY29uc3QgbWFya2VyIG9mIG1hcmtlcnMpIHtcbiAgICAgIGlmIChtYXJrZXIuZ2V0UmFuZ2UoKS5zdGFydC5pc0VxdWFsKHRoaXMuc3RhcnRQb2ludCkpIHtcbiAgICAgICAgdGhpcy5tYXJrZXJzQWZ0ZXIuYWRkKG1hcmtlcik7XG4gICAgICB9XG4gICAgfVxuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgbWFya1doaWxlKGxheWVyTmFtZSwgYmxvY2ssIG1hcmtlck9wdHMpIHtcbiAgICBjb25zdCBzdGFydCA9IHRoaXMuaW5zZXJ0aW9uUG9pbnQuY29weSgpO1xuICAgIGJsb2NrKCk7XG4gICAgY29uc3QgZW5kID0gdGhpcy5pbnNlcnRpb25Qb2ludC5jb3B5KCk7XG4gICAgdGhpcy5tYXJrZXJCbHVlcHJpbnRzLnB1c2goe2xheWVyTmFtZSwgcmFuZ2U6IG5ldyBSYW5nZShzdGFydCwgZW5kKSwgbWFya2VyT3B0c30pO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgaW5zZXJ0KHRleHQpIHtcbiAgICBjb25zdCBpbnNlcnRlZFJhbmdlID0gdGhpcy5wYXRjaEJ1ZmZlci5nZXRCdWZmZXIoKS5pbnNlcnQodGhpcy5pbnNlcnRpb25Qb2ludCwgdGV4dCk7XG4gICAgdGhpcy5pbnNlcnRpb25Qb2ludCA9IGluc2VydGVkUmFuZ2UuZW5kO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgaW5zZXJ0TWFya2VkKHRleHQsIGxheWVyTmFtZSwgbWFya2VyT3B0cykge1xuICAgIHJldHVybiB0aGlzLm1hcmtXaGlsZShsYXllck5hbWUsICgpID0+IHRoaXMuaW5zZXJ0KHRleHQpLCBtYXJrZXJPcHRzKTtcbiAgfVxuXG4gIGluc2VydFBhdGNoQnVmZmVyKHN1YlBhdGNoQnVmZmVyLCBvcHRzKSB7XG4gICAgY29uc3QgYmFzZU9mZnNldCA9IHRoaXMuaW5zZXJ0aW9uUG9pbnQuY29weSgpO1xuICAgIHRoaXMuaW5zZXJ0KHN1YlBhdGNoQnVmZmVyLmdldEJ1ZmZlcigpLmdldFRleHQoKSk7XG5cbiAgICBjb25zdCBzdWJNYXJrZXJNYXAgPSBuZXcgTWFwKCk7XG4gICAgZm9yIChjb25zdCBsYXllck5hbWUgb2YgTEFZRVJfTkFNRVMpIHtcbiAgICAgIGZvciAoY29uc3Qgb2xkTWFya2VyIG9mIHN1YlBhdGNoQnVmZmVyLmZpbmRNYXJrZXJzKGxheWVyTmFtZSwge30pKSB7XG4gICAgICAgIGNvbnN0IHN0YXJ0T2Zmc2V0ID0gb2xkTWFya2VyLmdldFJhbmdlKCkuc3RhcnQucm93ID09PSAwID8gYmFzZU9mZnNldCA6IFtiYXNlT2Zmc2V0LnJvdywgMF07XG4gICAgICAgIGNvbnN0IGVuZE9mZnNldCA9IG9sZE1hcmtlci5nZXRSYW5nZSgpLmVuZC5yb3cgPT09IDAgPyBiYXNlT2Zmc2V0IDogW2Jhc2VPZmZzZXQucm93LCAwXTtcblxuICAgICAgICBjb25zdCByYW5nZSA9IG9sZE1hcmtlci5nZXRSYW5nZSgpLnRyYW5zbGF0ZShzdGFydE9mZnNldCwgZW5kT2Zmc2V0KTtcbiAgICAgICAgY29uc3QgbWFya2VyT3B0cyA9IHtcbiAgICAgICAgICAuLi5vbGRNYXJrZXIuZ2V0UHJvcGVydGllcygpLFxuICAgICAgICAgIGNhbGxiYWNrOiBuZXdNYXJrZXIgPT4geyBzdWJNYXJrZXJNYXAuc2V0KG9sZE1hcmtlciwgbmV3TWFya2VyKTsgfSxcbiAgICAgICAgfTtcbiAgICAgICAgdGhpcy5tYXJrZXJCbHVlcHJpbnRzLnB1c2goe2xheWVyTmFtZSwgcmFuZ2UsIG1hcmtlck9wdHN9KTtcbiAgICAgIH1cbiAgICB9XG5cbiAgICB0aGlzLm1hcmtlck1hcENhbGxiYWNrcy5wdXNoKHttYXJrZXJNYXA6IHN1Yk1hcmtlck1hcCwgY2FsbGJhY2s6IG9wdHMuY2FsbGJhY2t9KTtcblxuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgYXBwbHkoKSB7XG4gICAgZm9yIChjb25zdCB7bGF5ZXJOYW1lLCByYW5nZSwgbWFya2VyT3B0c30gb2YgdGhpcy5tYXJrZXJCbHVlcHJpbnRzKSB7XG4gICAgICBjb25zdCBjYWxsYmFjayA9IG1hcmtlck9wdHMuY2FsbGJhY2s7XG4gICAgICBkZWxldGUgbWFya2VyT3B0cy5jYWxsYmFjaztcblxuICAgICAgY29uc3QgbWFya2VyID0gdGhpcy5wYXRjaEJ1ZmZlci5tYXJrUmFuZ2UobGF5ZXJOYW1lLCByYW5nZSwgbWFya2VyT3B0cyk7XG4gICAgICBpZiAoY2FsbGJhY2spIHtcbiAgICAgICAgY2FsbGJhY2sobWFya2VyKTtcbiAgICAgIH1cbiAgICB9XG5cbiAgICBmb3IgKGNvbnN0IHttYXJrZXJNYXAsIGNhbGxiYWNrfSBvZiB0aGlzLm1hcmtlck1hcENhbGxiYWNrcykge1xuICAgICAgY2FsbGJhY2sobWFya2VyTWFwKTtcbiAgICB9XG5cbiAgICBmb3IgKGNvbnN0IGJlZm9yZU1hcmtlciBvZiB0aGlzLm1hcmtlcnNCZWZvcmUpIHtcbiAgICAgIGNvbnN0IGlzRW1wdHkgPSBiZWZvcmVNYXJrZXIuZ2V0UmFuZ2UoKS5pc0VtcHR5KCk7XG5cbiAgICAgIGlmICghYmVmb3JlTWFya2VyLmlzUmV2ZXJzZWQoKSkge1xuICAgICAgICBiZWZvcmVNYXJrZXIuc2V0SGVhZFBvc2l0aW9uKHRoaXMuc3RhcnRQb2ludCk7XG4gICAgICAgIGlmIChpc0VtcHR5KSB7XG4gICAgICAgICAgYmVmb3JlTWFya2VyLnNldFRhaWxQb3NpdGlvbih0aGlzLnN0YXJ0UG9pbnQpO1xuICAgICAgICB9XG4gICAgICB9IGVsc2Uge1xuICAgICAgICBiZWZvcmVNYXJrZXIuc2V0VGFpbFBvc2l0aW9uKHRoaXMuc3RhcnRQb2ludCk7XG4gICAgICAgIGlmIChpc0VtcHR5KSB7XG4gICAgICAgICAgYmVmb3JlTWFya2VyLnNldEhlYWRQb3NpdGlvbih0aGlzLnN0YXJ0UG9pbnQpO1xuICAgICAgICB9XG4gICAgICB9XG4gICAgfVxuXG4gICAgZm9yIChjb25zdCBhZnRlck1hcmtlciBvZiB0aGlzLm1hcmtlcnNBZnRlcikge1xuICAgICAgY29uc3QgaXNFbXB0eSA9IGFmdGVyTWFya2VyLmdldFJhbmdlKCkuaXNFbXB0eSgpO1xuXG4gICAgICBpZiAoIWFmdGVyTWFya2VyLmlzUmV2ZXJzZWQoKSkge1xuICAgICAgICBhZnRlck1hcmtlci5zZXRUYWlsUG9zaXRpb24odGhpcy5pbnNlcnRpb25Qb2ludCk7XG4gICAgICAgIGlmIChpc0VtcHR5KSB7XG4gICAgICAgICAgYWZ0ZXJNYXJrZXIuc2V0SGVhZFBvc2l0aW9uKHRoaXMuaW5zZXJ0aW9uUG9pbnQpO1xuICAgICAgICB9XG4gICAgICB9IGVsc2Uge1xuICAgICAgICBhZnRlck1hcmtlci5zZXRIZWFkUG9zaXRpb24odGhpcy5pbnNlcnRpb25Qb2ludCk7XG4gICAgICAgIGlmIChpc0VtcHR5KSB7XG4gICAgICAgICAgYWZ0ZXJNYXJrZXIuc2V0VGFpbFBvc2l0aW9uKHRoaXMuaW5zZXJ0aW9uUG9pbnQpO1xuICAgICAgICB9XG4gICAgICB9XG4gICAgfVxuICB9XG59XG4iXX0=