"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.NoNewline = exports.Unchanged = exports.Deletion = exports.Addition = void 0;

var _atom = require("atom");

function _objectSpread2(target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i] != null ? arguments[i] : {}; var ownKeys = Object.keys(source); if (typeof Object.getOwnPropertySymbols === 'function') { ownKeys = ownKeys.concat(Object.getOwnPropertySymbols(source).filter(function (sym) { return Object.getOwnPropertyDescriptor(source, sym).enumerable; })); } ownKeys.forEach(function (key) { _defineProperty(target, key, source[key]); }); } return target; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

class Region {
  constructor(marker) {
    this.marker = marker;
  }

  getMarker() {
    return this.marker;
  }

  getRange() {
    return this.marker.getRange();
  }

  getStartBufferRow() {
    return this.getRange().start.row;
  }

  getEndBufferRow() {
    return this.getRange().end.row;
  }

  includesBufferRow(row) {
    return this.getRange().intersectsRow(row);
  }
  /*
   * intersectRows breaks a Region into runs of rows that are included in
   * rowSet and rows that are not. For example:
   *  @this Region        row 10-20
   *  @param rowSet       row 11, 12, 13, 17, 19
   *  @param includeGaps  true (whether the result will include gaps or not)
   *  @return an array of regions like this:
   *    (10, gap = true) (11, 12, 13, gap = false) (14, 15, 16, gap = true)
   *    (17, gap = false) (18, gap = true) (19, gap = false) (20, gap = true)
   */


  intersectRows(rowSet, includeGaps) {
    const intersections = [];
    let withinIntersection = false;
    let currentRow = this.getRange().start.row;
    let nextStartRow = currentRow;

    const finishRowRange = isGap => {
      if (isGap && !includeGaps) {
        nextStartRow = currentRow;
        return;
      }

      if (currentRow <= this.getRange().start.row) {
        return;
      }

      intersections.push({
        intersection: _atom.Range.fromObject([[nextStartRow, 0], [currentRow - 1, Infinity]]),
        gap: isGap
      });
      nextStartRow = currentRow;
    };

    while (currentRow <= this.getRange().end.row) {
      if (rowSet.has(currentRow) && !withinIntersection) {
        // One row past the end of a gap. Start of intersecting row range.
        finishRowRange(true);
        withinIntersection = true;
      } else if (!rowSet.has(currentRow) && withinIntersection) {
        // One row past the end of intersecting row range. Start of the next gap.
        finishRowRange(false);
        withinIntersection = false;
      }

      currentRow++;
    }

    finishRowRange(!withinIntersection);
    return intersections;
  }

  isAddition() {
    return false;
  }

  isDeletion() {
    return false;
  }

  isUnchanged() {
    return false;
  }

  isNoNewline() {
    return false;
  }

  getBufferRows() {
    return this.getRange().getRows();
  }

  bufferRowCount() {
    return this.getRange().getRowCount();
  }

  when(callbacks) {
    const callback = callbacks[this.constructor.name.toLowerCase()] || callbacks.default || (() => undefined);

    return callback();
  }

  updateMarkers(map) {
    this.marker = map.get(this.marker) || this.marker;
  }

  destroyMarkers() {
    this.marker.destroy();
  }

  toStringIn(buffer) {
    const raw = buffer.getTextInRange(this.getRange());
    return this.constructor.origin + raw.replace(/\r?\n/g, '$&' + this.constructor.origin) + buffer.lineEndingForRow(this.getRange().end.row);
  }
  /*
   * Construct a String containing internal diagnostic information.
   */

  /* istanbul ignore next */


  inspect(opts = {}) {
    const options = _objectSpread2({
      indent: 0
    }, opts);

    let indentation = '';

    for (let i = 0; i < options.indent; i++) {
      indentation += ' ';
    }

    let inspectString = `${indentation}(${this.constructor.name} marker=${this.marker.id})`;

    if (this.marker.isDestroyed()) {
      inspectString += ' [destroyed]';
    }

    if (!this.marker.isValid()) {
      inspectString += ' [invalid]';
    }

    return inspectString + '\n';
  }

  isChange() {
    return true;
  }

}

class Addition extends Region {
  isAddition() {
    return true;
  }

  invertIn(nextBuffer) {
    return new Deletion(nextBuffer.markRange(this.getRange()));
  }

}

exports.Addition = Addition;

_defineProperty(Addition, "origin", '+');

_defineProperty(Addition, "layerName", 'addition');

class Deletion extends Region {
  isDeletion() {
    return true;
  }

  invertIn(nextBuffer) {
    return new Addition(nextBuffer.markRange(this.getRange()));
  }

}

exports.Deletion = Deletion;

_defineProperty(Deletion, "origin", '-');

_defineProperty(Deletion, "layerName", 'deletion');

class Unchanged extends Region {
  isUnchanged() {
    return true;
  }

  isChange() {
    return false;
  }

  invertIn(nextBuffer) {
    return new Unchanged(nextBuffer.markRange(this.getRange()));
  }

}

exports.Unchanged = Unchanged;

_defineProperty(Unchanged, "origin", ' ');

_defineProperty(Unchanged, "layerName", 'unchanged');

class NoNewline extends Region {
  isNoNewline() {
    return true;
  }

  isChange() {
    return false;
  }

  invertIn(nextBuffer) {
    return new NoNewline(nextBuffer.markRange(this.getRange()));
  }

}

exports.NoNewline = NoNewline;

_defineProperty(NoNewline, "origin", '\\');

_defineProperty(NoNewline, "layerName", 'nonewline');
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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