"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.nullRemote = exports.default = void 0;

var _endpoint = require("./endpoint");

class Remote {
  constructor(name, url) {
    this.name = name;
    this.url = url;
    const {
      isGithubRepo,
      domain,
      protocol,
      owner,
      repo
    } = githubInfoFromRemote(url);
    this.githubRepo = isGithubRepo;
    this.domain = domain;
    this.protocol = protocol;
    this.owner = owner;
    this.repo = repo;
  }

  getName() {
    return this.name;
  }

  getUrl() {
    return this.url;
  }

  isGithubRepo() {
    return this.githubRepo;
  }

  getProtocol() {
    return this.protocol;
  }

  getDomain() {
    return this.domain;
  }

  getOwner() {
    return this.owner;
  }

  getRepo() {
    return this.repo;
  }

  getNameOr() {
    return this.getName();
  }

  getSlug() {
    if (this.owner === null || this.repo === null) {
      return null;
    }

    return `${this.owner}/${this.repo}`;
  }

  getEndpoint() {
    return this.domain === null ? null : (0, _endpoint.getEndpoint)(this.domain);
  }

  getEndpointOrDotcom() {
    return this.getEndpoint() || _endpoint.DOTCOM;
  }

  isPresent() {
    return true;
  }

}

exports.default = Remote;

function githubInfoFromRemote(remoteUrl) {
  if (!remoteUrl) {
    return {
      isGithubRepo: false,
      domain: null,
      owner: null,
      repo: null
    };
  } //             proto         login   domain           owner    repo


  const regex = /(?:(.+):\/\/)?(?:.+@)?(github\.com)[:/]\/?([^/]+)\/(.+)/;
  const match = remoteUrl.match(regex);

  if (match) {
    return {
      isGithubRepo: true,
      protocol: match[1] || 'ssh',
      domain: match[2],
      owner: match[3],
      repo: match[4].replace(/\.git$/, '')
    };
  } else {
    return {
      isGithubRepo: false,
      protocol: null,
      domain: null,
      owner: null,
      repo: null
    };
  }
}

const nullRemote = {
  getName() {
    return '';
  },

  getUrl() {
    return '';
  },

  isGithubRepo() {
    return false;
  },

  getDomain() {
    return null;
  },

  getProtocol() {
    return null;
  },

  getOwner() {
    return null;
  },

  getRepo() {
    return null;
  },

  getNameOr(fallback) {
    return fallback;
  },

  getSlug() {
    return null;
  },

  getEndpoint() {
    return null;
  },

  getEndpointOrDotcom() {
    return _endpoint.DOTCOM;
  },

  isPresent() {
    return false;
  }

};
exports.nullRemote = nullRemote;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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