"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.Keys = void 0;

class CacheKey {
  constructor(primary, groups = []) {
    this.primary = primary;
    this.groups = groups;
  }

  getPrimary() {
    return this.primary;
  }

  getGroups() {
    return this.groups;
  }

  removeFromCache(cache, withoutGroup = null) {
    cache.removePrimary(this.getPrimary());
    const groups = this.getGroups();

    for (let i = 0; i < groups.length; i++) {
      const group = groups[i];

      if (group === withoutGroup) {
        continue;
      }

      cache.removeFromGroup(group, this);
    }
  }
  /* istanbul ignore next */


  toString() {
    return `CacheKey(${this.primary})`;
  }

}

class GroupKey {
  constructor(group) {
    this.group = group;
  }

  removeFromCache(cache) {
    for (const matchingKey of cache.keysInGroup(this.group)) {
      matchingKey.removeFromCache(cache, this.group);
    }
  }
  /* istanbul ignore next */


  toString() {
    return `GroupKey(${this.group})`;
  }

}

const Keys = {
  statusBundle: new CacheKey('status-bundle'),
  stagedChanges: new CacheKey('staged-changes'),
  filePatch: {
    _optKey: ({
      staged
    }) => staged ? 's' : 'u',
    oneWith: (fileName, options) => {
      // <-- Keys.filePatch
      const optKey = Keys.filePatch._optKey(options);

      const baseCommit = options.baseCommit || 'head';
      const extraGroups = [];

      if (options.baseCommit) {
        extraGroups.push(`file-patch:base-nonhead:path-${fileName}`);
        extraGroups.push('file-patch:base-nonhead');
      } else {
        extraGroups.push('file-patch:base-head');
      }

      return new CacheKey(`file-patch:${optKey}:${baseCommit}:${fileName}`, ['file-patch', `file-patch:opt-${optKey}`, `file-patch:opt-${optKey}:path-${fileName}`, ...extraGroups]);
    },
    eachWithFileOpts: (fileNames, opts) => {
      const keys = [];

      for (let i = 0; i < fileNames.length; i++) {
        for (let j = 0; j < opts.length; j++) {
          keys.push(new GroupKey(`file-patch:opt-${Keys.filePatch._optKey(opts[j])}:path-${fileNames[i]}`));
        }
      }

      return keys;
    },
    eachNonHeadWithFiles: fileNames => {
      return fileNames.map(fileName => new GroupKey(`file-patch:base-nonhead:path-${fileName}`));
    },
    allAgainstNonHead: new GroupKey('file-patch:base-nonhead'),
    eachWithOpts: (...opts) => opts.map(opt => new GroupKey(`file-patch:opt-${Keys.filePatch._optKey(opt)}`)),
    all: new GroupKey('file-patch')
  },
  index: {
    oneWith: fileName => new CacheKey(`index:${fileName}`, ['index']),
    all: new GroupKey('index')
  },
  lastCommit: new CacheKey('last-commit'),
  recentCommits: new CacheKey('recent-commits'),
  authors: new CacheKey('authors'),
  branches: new CacheKey('branches'),
  headDescription: new CacheKey('head-description'),
  remotes: new CacheKey('remotes'),
  config: {
    _optKey: options => options.local ? 'l' : '',
    oneWith: (setting, options) => {
      const optKey = Keys.config._optKey(options);

      return new CacheKey(`config:${optKey}:${setting}`, ['config', `config:${optKey}`]);
    },
    eachWithSetting: setting => [Keys.config.oneWith(setting, {
      local: true
    }), Keys.config.oneWith(setting, {
      local: false
    })],
    all: new GroupKey('config')
  },
  blob: {
    oneWith: sha => new CacheKey(`blob:${sha}`, ['blob'])
  },
  // Common collections of keys and patterns for use with invalidate().
  workdirOperationKeys: fileNames => [Keys.statusBundle, ...Keys.filePatch.eachWithFileOpts(fileNames, [{
    staged: false
  }])],
  cacheOperationKeys: fileNames => [...Keys.workdirOperationKeys(fileNames), ...Keys.filePatch.eachWithFileOpts(fileNames, [{
    staged: true
  }]), ...fileNames.map(Keys.index.oneWith), Keys.stagedChanges],
  headOperationKeys: () => [Keys.headDescription, Keys.branches, ...Keys.filePatch.eachWithOpts({
    staged: true
  }), Keys.filePatch.allAgainstNonHead, Keys.stagedChanges, Keys.lastCommit, Keys.recentCommits, Keys.authors, Keys.statusBundle]
};
exports.Keys = Keys;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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