"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

var _path = _interopRequireDefault(require("path"));

var _eventKit = require("event-kit");

var _fsExtra = _interopRequireDefault(require("fs-extra"));

var _state = _interopRequireDefault(require("./state"));

var _keys = require("./cache/keys");

var _gitShellOutStrategy = require("../../git-shell-out-strategy");

var _workspaceChangeObserver = require("../workspace-change-observer");

var _patch = require("../patch");

var _discardHistory = _interopRequireDefault(require("../discard-history"));

var _branch = _interopRequireWildcard(require("../branch"));

var _author = _interopRequireDefault(require("../author"));

var _branchSet = _interopRequireDefault(require("../branch-set"));

var _remote = _interopRequireDefault(require("../remote"));

var _remoteSet = _interopRequireDefault(require("../remote-set"));

var _commit = _interopRequireDefault(require("../commit"));

var _operationStates = _interopRequireDefault(require("../operation-states"));

var _reporterProxy = require("../../reporter-proxy");

var _helpers = require("../../helpers");

function _interopRequireWildcard(obj) { if (obj && obj.__esModule) { return obj; } else { var newObj = {}; if (obj != null) { for (var key in obj) { if (Object.prototype.hasOwnProperty.call(obj, key)) { var desc = Object.defineProperty && Object.getOwnPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : {}; if (desc.get || desc.set) { Object.defineProperty(newObj, key, desc); } else { newObj[key] = obj[key]; } } } } newObj.default = obj; return newObj; } }

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _objectSpread2(target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i] != null ? arguments[i] : {}; var ownKeys = Object.keys(source); if (typeof Object.getOwnPropertySymbols === 'function') { ownKeys = ownKeys.concat(Object.getOwnPropertySymbols(source).filter(function (sym) { return Object.getOwnPropertyDescriptor(source, sym).enumerable; })); } ownKeys.forEach(function (key) { _defineProperty(target, key, source[key]); }); } return target; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

/**
 * State used when the working directory contains a valid git repository and can be interacted with. Performs
 * actual git operations, caching the results, and broadcasts `onDidUpdate` events when write actions are
 * performed.
 */
class Present extends _state.default {
  constructor(repository, history) {
    super(repository);
    this.cache = new Cache();
    this.discardHistory = new _discardHistory.default(this.createBlob.bind(this), this.expandBlobToFile.bind(this), this.mergeFile.bind(this), this.workdir(), {
      maxHistoryLength: 60
    });
    this.operationStates = new _operationStates.default({
      didUpdate: this.didUpdate.bind(this)
    });
    this.commitMessage = '';
    this.commitMessageTemplate = null;
    this.fetchInitialMessage();
    /* istanbul ignore else */

    if (history) {
      this.discardHistory.updateHistory(history);
    }
  }

  setCommitMessage(message, {
    suppressUpdate
  } = {
    suppressUpdate: false
  }) {
    this.commitMessage = message;

    if (!suppressUpdate) {
      this.didUpdate();
    }
  }

  setCommitMessageTemplate(template) {
    this.commitMessageTemplate = template;
  }

  async fetchInitialMessage() {
    const mergeMessage = await this.repository.getMergeMessage();
    const template = await this.fetchCommitMessageTemplate();

    if (template) {
      this.commitMessageTemplate = template;
    }

    if (mergeMessage) {
      this.setCommitMessage(mergeMessage);
    } else if (template) {
      this.setCommitMessage(template);
    }
  }

  getCommitMessage() {
    return this.commitMessage;
  }

  fetchCommitMessageTemplate() {
    return this.git().fetchCommitMessageTemplate();
  }

  getOperationStates() {
    return this.operationStates;
  }

  isPresent() {
    return true;
  }

  destroy() {
    this.cache.destroy();
    super.destroy();
  }

  showStatusBarTiles() {
    return true;
  }

  isPublishable() {
    return true;
  }

  acceptInvalidation(spec, {
    globally
  } = {}) {
    this.cache.invalidate(spec());
    this.didUpdate();

    if (globally) {
      this.didGloballyInvalidate(spec);
    }
  }

  invalidateCacheAfterFilesystemChange(events) {
    const paths = events.map(e => e.special || e.path);
    const keys = new Set();

    for (let i = 0; i < paths.length; i++) {
      const fullPath = paths[i];

      if (fullPath === _workspaceChangeObserver.FOCUS) {
        keys.add(_keys.Keys.statusBundle);

        for (const k of _keys.Keys.filePatch.eachWithOpts({
          staged: false
        })) {
          keys.add(k);
        }

        continue;
      }

      const includes = (...segments) => fullPath.includes(_path.default.join(...segments));

      if ((0, _helpers.filePathEndsWith)(fullPath, '.git', 'index')) {
        keys.add(_keys.Keys.stagedChanges);
        keys.add(_keys.Keys.filePatch.all);
        keys.add(_keys.Keys.index.all);
        keys.add(_keys.Keys.statusBundle);
        continue;
      }

      if ((0, _helpers.filePathEndsWith)(fullPath, '.git', 'HEAD')) {
        keys.add(_keys.Keys.branches);
        keys.add(_keys.Keys.lastCommit);
        keys.add(_keys.Keys.recentCommits);
        keys.add(_keys.Keys.statusBundle);
        keys.add(_keys.Keys.headDescription);
        keys.add(_keys.Keys.authors);
        continue;
      }

      if (includes('.git', 'refs', 'heads')) {
        keys.add(_keys.Keys.branches);
        keys.add(_keys.Keys.lastCommit);
        keys.add(_keys.Keys.recentCommits);
        keys.add(_keys.Keys.headDescription);
        keys.add(_keys.Keys.authors);
        continue;
      }

      if (includes('.git', 'refs', 'remotes')) {
        keys.add(_keys.Keys.remotes);
        keys.add(_keys.Keys.statusBundle);
        keys.add(_keys.Keys.headDescription);
        continue;
      }

      if ((0, _helpers.filePathEndsWith)(fullPath, '.git', 'config')) {
        keys.add(_keys.Keys.remotes);
        keys.add(_keys.Keys.config.all);
        keys.add(_keys.Keys.statusBundle);
        continue;
      } // File change within the working directory


      const relativePath = _path.default.relative(this.workdir(), fullPath);

      for (const key of _keys.Keys.filePatch.eachWithFileOpts([relativePath], [{
        staged: false
      }])) {
        keys.add(key);
      }

      keys.add(_keys.Keys.statusBundle);
    }
    /* istanbul ignore else */


    if (keys.size > 0) {
      this.cache.invalidate(Array.from(keys));
      this.didUpdate();
    }
  }

  isCommitMessageClean() {
    if (this.commitMessage.trim() === '') {
      return true;
    } else if (this.commitMessageTemplate) {
      return this.commitMessage === this.commitMessageTemplate;
    }

    return false;
  }

  async updateCommitMessageAfterFileSystemChange(events) {
    for (let i = 0; i < events.length; i++) {
      const event = events[i];

      if (!event.path) {
        continue;
      }

      if ((0, _helpers.filePathEndsWith)(event.path, '.git', 'MERGE_HEAD')) {
        if (event.action === 'created') {
          if (this.isCommitMessageClean()) {
            this.setCommitMessage((await this.repository.getMergeMessage()));
          }
        } else if (event.action === 'deleted') {
          this.setCommitMessage(this.commitMessageTemplate || '');
        }
      }

      if ((0, _helpers.filePathEndsWith)(event.path, '.git', 'config')) {
        // this won't catch changes made to the template file itself...
        const template = await this.fetchCommitMessageTemplate();

        if (template === null) {
          this.setCommitMessage('');
        } else if (this.commitMessageTemplate !== template) {
          this.setCommitMessage(template);
        }

        this.setCommitMessageTemplate(template);
      }
    }
  }

  observeFilesystemChange(events) {
    this.invalidateCacheAfterFilesystemChange(events);
    this.updateCommitMessageAfterFileSystemChange(events);
  }

  refresh() {
    this.cache.clear();
    this.didUpdate();
  }

  init() {
    return super.init().catch(e => {
      e.stdErr = 'This directory already contains a git repository';
      return Promise.reject(e);
    });
  }

  clone() {
    return super.clone().catch(e => {
      e.stdErr = 'This directory already contains a git repository';
      return Promise.reject(e);
    });
  } // Git operations ////////////////////////////////////////////////////////////////////////////////////////////////////
  // Staging and unstaging


  stageFiles(paths) {
    return this.invalidate(() => _keys.Keys.cacheOperationKeys(paths), () => this.git().stageFiles(paths));
  }

  unstageFiles(paths) {
    return this.invalidate(() => _keys.Keys.cacheOperationKeys(paths), () => this.git().unstageFiles(paths));
  }

  stageFilesFromParentCommit(paths) {
    return this.invalidate(() => _keys.Keys.cacheOperationKeys(paths), () => this.git().unstageFiles(paths, 'HEAD~'));
  }

  stageFileModeChange(filePath, fileMode) {
    return this.invalidate(() => _keys.Keys.cacheOperationKeys([filePath]), () => this.git().stageFileModeChange(filePath, fileMode));
  }

  stageFileSymlinkChange(filePath) {
    return this.invalidate(() => _keys.Keys.cacheOperationKeys([filePath]), () => this.git().stageFileSymlinkChange(filePath));
  }

  applyPatchToIndex(multiFilePatch) {
    return this.invalidate(() => _keys.Keys.cacheOperationKeys(Array.from(multiFilePatch.getPathSet())), () => {
      const patchStr = multiFilePatch.toString();
      return this.git().applyPatch(patchStr, {
        index: true
      });
    });
  }

  applyPatchToWorkdir(multiFilePatch) {
    return this.invalidate(() => _keys.Keys.workdirOperationKeys(Array.from(multiFilePatch.getPathSet())), () => {
      const patchStr = multiFilePatch.toString();
      return this.git().applyPatch(patchStr);
    });
  } // Committing


  commit(message, options) {
    return this.invalidate(_keys.Keys.headOperationKeys, // eslint-disable-next-line no-shadow
    () => this.executePipelineAction('COMMIT', async (message, options = {}) => {
      const coAuthors = options.coAuthors;
      const opts = !coAuthors ? options : _objectSpread2({}, options, {
        coAuthors: coAuthors.map(author => {
          return {
            email: author.getEmail(),
            name: author.getFullName()
          };
        })
      });
      await this.git().commit(message, opts); // Collect commit metadata metrics
      // note: in GitShellOutStrategy we have counters for all git commands, including `commit`, but here we have
      //       access to additional metadata (unstaged file count) so it makes sense to collect commit events here

      const {
        unstagedFiles,
        mergeConflictFiles
      } = await this.getStatusesForChangedFiles();
      const unstagedCount = Object.keys(_objectSpread2({}, unstagedFiles, {}, mergeConflictFiles)).length;
      (0, _reporterProxy.addEvent)('commit', {
        package: 'github',
        partial: unstagedCount > 0,
        amend: !!options.amend,
        coAuthorCount: coAuthors ? coAuthors.length : 0
      });
    }, message, options));
  } // Merging


  merge(branchName) {
    return this.invalidate(() => [..._keys.Keys.headOperationKeys(), _keys.Keys.index.all, _keys.Keys.headDescription], () => this.git().merge(branchName));
  }

  abortMerge() {
    return this.invalidate(() => [_keys.Keys.statusBundle, _keys.Keys.stagedChanges, _keys.Keys.filePatch.all, _keys.Keys.index.all], async () => {
      await this.git().abortMerge();
      this.setCommitMessage(this.commitMessageTemplate || '');
    });
  }

  checkoutSide(side, paths) {
    return this.git().checkoutSide(side, paths);
  }

  mergeFile(oursPath, commonBasePath, theirsPath, resultPath) {
    return this.git().mergeFile(oursPath, commonBasePath, theirsPath, resultPath);
  }

  writeMergeConflictToIndex(filePath, commonBaseSha, oursSha, theirsSha) {
    return this.invalidate(() => [_keys.Keys.statusBundle, _keys.Keys.stagedChanges, ..._keys.Keys.filePatch.eachWithFileOpts([filePath], [{
      staged: false
    }, {
      staged: true
    }]), _keys.Keys.index.oneWith(filePath)], () => this.git().writeMergeConflictToIndex(filePath, commonBaseSha, oursSha, theirsSha));
  } // Checkout


  checkout(revision, options = {}) {
    return this.invalidate(() => [_keys.Keys.stagedChanges, _keys.Keys.lastCommit, _keys.Keys.recentCommits, _keys.Keys.authors, _keys.Keys.statusBundle, _keys.Keys.index.all, ..._keys.Keys.filePatch.eachWithOpts({
      staged: true
    }), _keys.Keys.filePatch.allAgainstNonHead, _keys.Keys.headDescription, _keys.Keys.branches], // eslint-disable-next-line no-shadow
    () => this.executePipelineAction('CHECKOUT', (revision, options) => {
      return this.git().checkout(revision, options);
    }, revision, options));
  }

  checkoutPathsAtRevision(paths, revision = 'HEAD') {
    return this.invalidate(() => [_keys.Keys.statusBundle, _keys.Keys.stagedChanges, ...paths.map(fileName => _keys.Keys.index.oneWith(fileName)), ..._keys.Keys.filePatch.eachWithFileOpts(paths, [{
      staged: true
    }]), ..._keys.Keys.filePatch.eachNonHeadWithFiles(paths)], () => this.git().checkoutFiles(paths, revision));
  } // Reset


  undoLastCommit() {
    return this.invalidate(() => [_keys.Keys.stagedChanges, _keys.Keys.lastCommit, _keys.Keys.recentCommits, _keys.Keys.authors, _keys.Keys.statusBundle, _keys.Keys.index.all, ..._keys.Keys.filePatch.eachWithOpts({
      staged: true
    }), _keys.Keys.headDescription], async () => {
      try {
        await this.git().reset('soft', 'HEAD~');
        (0, _reporterProxy.addEvent)('undo-last-commit', {
          package: 'github'
        });
      } catch (e) {
        if (/unknown revision/.test(e.stdErr)) {
          // Initial commit
          await this.git().deleteRef('HEAD');
        } else {
          throw e;
        }
      }
    });
  } // Remote interactions


  fetch(branchName, options = {}) {
    return this.invalidate(() => [_keys.Keys.statusBundle, _keys.Keys.headDescription], // eslint-disable-next-line no-shadow
    () => this.executePipelineAction('FETCH', async branchName => {
      let finalRemoteName = options.remoteName;

      if (!finalRemoteName) {
        const remote = await this.getRemoteForBranch(branchName);

        if (!remote.isPresent()) {
          return null;
        }

        finalRemoteName = remote.getName();
      }

      return this.git().fetch(finalRemoteName, branchName);
    }, branchName));
  }

  pull(branchName, options = {}) {
    return this.invalidate(() => [..._keys.Keys.headOperationKeys(), _keys.Keys.index.all, _keys.Keys.headDescription, _keys.Keys.branches], // eslint-disable-next-line no-shadow
    () => this.executePipelineAction('PULL', async branchName => {
      let finalRemoteName = options.remoteName;

      if (!finalRemoteName) {
        const remote = await this.getRemoteForBranch(branchName);

        if (!remote.isPresent()) {
          return null;
        }

        finalRemoteName = remote.getName();
      }

      return this.git().pull(finalRemoteName, branchName, options);
    }, branchName));
  }

  push(branchName, options = {}) {
    return this.invalidate(() => {
      const keys = [_keys.Keys.statusBundle, _keys.Keys.headDescription];

      if (options.setUpstream) {
        keys.push(_keys.Keys.branches);
        keys.push(..._keys.Keys.config.eachWithSetting(`branch.${branchName}.remote`));
      }

      return keys;
    }, // eslint-disable-next-line no-shadow
    () => this.executePipelineAction('PUSH', async (branchName, options) => {
      const remote = options.remote || (await this.getRemoteForBranch(branchName));
      return this.git().push(remote.getNameOr('origin'), branchName, options);
    }, branchName, options));
  } // Configuration


  setConfig(setting, value, options = {}) {
    return this.invalidate(() => _keys.Keys.config.eachWithSetting(setting), () => this.git().setConfig(setting, value, options), {
      globally: options.global
    });
  }

  unsetConfig(setting) {
    return this.invalidate(() => _keys.Keys.config.eachWithSetting(setting), () => this.git().unsetConfig(setting));
  } // Direct blob interactions


  createBlob(options) {
    return this.git().createBlob(options);
  }

  expandBlobToFile(absFilePath, sha) {
    return this.git().expandBlobToFile(absFilePath, sha);
  } // Discard history


  createDiscardHistoryBlob() {
    return this.discardHistory.createHistoryBlob();
  }

  async updateDiscardHistory() {
    const history = await this.loadHistoryPayload();
    this.discardHistory.updateHistory(history);
  }

  async storeBeforeAndAfterBlobs(filePaths, isSafe, destructiveAction, partialDiscardFilePath = null) {
    const snapshots = await this.discardHistory.storeBeforeAndAfterBlobs(filePaths, isSafe, destructiveAction, partialDiscardFilePath);
    /* istanbul ignore else */

    if (snapshots) {
      await this.saveDiscardHistory();
    }

    return snapshots;
  }

  restoreLastDiscardInTempFiles(isSafe, partialDiscardFilePath = null) {
    return this.discardHistory.restoreLastDiscardInTempFiles(isSafe, partialDiscardFilePath);
  }

  async popDiscardHistory(partialDiscardFilePath = null) {
    const removed = await this.discardHistory.popHistory(partialDiscardFilePath);

    if (removed) {
      await this.saveDiscardHistory();
    }
  }

  clearDiscardHistory(partialDiscardFilePath = null) {
    this.discardHistory.clearHistory(partialDiscardFilePath);
    return this.saveDiscardHistory();
  }

  discardWorkDirChangesForPaths(paths) {
    return this.invalidate(() => [_keys.Keys.statusBundle, ...paths.map(filePath => _keys.Keys.filePatch.oneWith(filePath, {
      staged: false
    })), ..._keys.Keys.filePatch.eachNonHeadWithFiles(paths)], async () => {
      const untrackedFiles = await this.git().getUntrackedFiles();
      const [filesToRemove, filesToCheckout] = partition(paths, f => untrackedFiles.includes(f));
      await this.git().checkoutFiles(filesToCheckout);
      await Promise.all(filesToRemove.map(filePath => {
        const absPath = _path.default.join(this.workdir(), filePath);

        return _fsExtra.default.remove(absPath);
      }));
    });
  } // Accessors /////////////////////////////////////////////////////////////////////////////////////////////////////////
  // Index queries


  getStatusBundle() {
    return this.cache.getOrSet(_keys.Keys.statusBundle, async () => {
      try {
        const bundle = await this.git().getStatusBundle();
        const results = await this.formatChangedFiles(bundle);
        results.branch = bundle.branch;
        return results;
      } catch (err) {
        if (err instanceof _gitShellOutStrategy.LargeRepoError) {
          this.transitionTo('TooLarge');
          return {
            branch: {},
            stagedFiles: {},
            unstagedFiles: {},
            mergeConflictFiles: {}
          };
        } else {
          throw err;
        }
      }
    });
  }

  async formatChangedFiles({
    changedEntries,
    untrackedEntries,
    renamedEntries,
    unmergedEntries
  }) {
    const statusMap = {
      A: 'added',
      M: 'modified',
      D: 'deleted',
      U: 'modified',
      T: 'typechange'
    };
    const stagedFiles = {};
    const unstagedFiles = {};
    const mergeConflictFiles = {};
    changedEntries.forEach(entry => {
      if (entry.stagedStatus) {
        stagedFiles[entry.filePath] = statusMap[entry.stagedStatus];
      }

      if (entry.unstagedStatus) {
        unstagedFiles[entry.filePath] = statusMap[entry.unstagedStatus];
      }
    });
    untrackedEntries.forEach(entry => {
      unstagedFiles[entry.filePath] = statusMap.A;
    });
    renamedEntries.forEach(entry => {
      if (entry.stagedStatus === 'R') {
        stagedFiles[entry.filePath] = statusMap.A;
        stagedFiles[entry.origFilePath] = statusMap.D;
      }

      if (entry.unstagedStatus === 'R') {
        unstagedFiles[entry.filePath] = statusMap.A;
        unstagedFiles[entry.origFilePath] = statusMap.D;
      }

      if (entry.stagedStatus === 'C') {
        stagedFiles[entry.filePath] = statusMap.A;
      }

      if (entry.unstagedStatus === 'C') {
        unstagedFiles[entry.filePath] = statusMap.A;
      }
    });
    let statusToHead;

    for (let i = 0; i < unmergedEntries.length; i++) {
      const {
        stagedStatus,
        unstagedStatus,
        filePath
      } = unmergedEntries[i];

      if (stagedStatus === 'U' || unstagedStatus === 'U' || stagedStatus === 'A' && unstagedStatus === 'A') {
        // Skipping this check here because we only run a single `await`
        // and we only run it in the main, synchronous body of the for loop.
        // eslint-disable-next-line no-await-in-loop
        if (!statusToHead) {
          statusToHead = await this.git().diffFileStatus({
            target: 'HEAD'
          });
        }

        mergeConflictFiles[filePath] = {
          ours: statusMap[stagedStatus],
          theirs: statusMap[unstagedStatus],
          file: statusToHead[filePath] || 'equivalent'
        };
      }
    }

    return {
      stagedFiles,
      unstagedFiles,
      mergeConflictFiles
    };
  }

  async getStatusesForChangedFiles() {
    const {
      stagedFiles,
      unstagedFiles,
      mergeConflictFiles
    } = await this.getStatusBundle();
    return {
      stagedFiles,
      unstagedFiles,
      mergeConflictFiles
    };
  }

  getFilePatchForPath(filePath, options) {
    const opts = _objectSpread2({
      staged: false,
      patchBuffer: null,
      builder: {},
      before: () => {},
      after: () => {}
    }, options);

    return this.cache.getOrSet(_keys.Keys.filePatch.oneWith(filePath, {
      staged: opts.staged
    }), async () => {
      const diffs = await this.git().getDiffsForFilePath(filePath, {
        staged: opts.staged
      });
      const payload = opts.before();
      const patch = (0, _patch.buildFilePatch)(diffs, opts.builder);

      if (opts.patchBuffer !== null) {
        patch.adoptBuffer(opts.patchBuffer);
      }

      opts.after(patch, payload);
      return patch;
    });
  }

  getDiffsForFilePath(filePath, baseCommit) {
    return this.cache.getOrSet(_keys.Keys.filePatch.oneWith(filePath, {
      baseCommit
    }), () => {
      return this.git().getDiffsForFilePath(filePath, {
        baseCommit
      });
    });
  }

  getStagedChangesPatch(options) {
    const opts = _objectSpread2({
      builder: {},
      patchBuffer: null,
      before: () => {},
      after: () => {}
    }, options);

    return this.cache.getOrSet(_keys.Keys.stagedChanges, async () => {
      const diffs = await this.git().getStagedChangesPatch();
      const payload = opts.before();
      const patch = (0, _patch.buildMultiFilePatch)(diffs, opts.builder);

      if (opts.patchBuffer !== null) {
        patch.adoptBuffer(opts.patchBuffer);
      }

      opts.after(patch, payload);
      return patch;
    });
  }

  readFileFromIndex(filePath) {
    return this.cache.getOrSet(_keys.Keys.index.oneWith(filePath), () => {
      return this.git().readFileFromIndex(filePath);
    });
  } // Commit access


  getLastCommit() {
    return this.cache.getOrSet(_keys.Keys.lastCommit, async () => {
      const headCommit = await this.git().getHeadCommit();
      return headCommit.unbornRef ? _commit.default.createUnborn() : new _commit.default(headCommit);
    });
  }

  getCommit(sha) {
    return this.cache.getOrSet(_keys.Keys.blob.oneWith(sha), async () => {
      const [rawCommit] = await this.git().getCommits({
        max: 1,
        ref: sha,
        includePatch: true
      });
      const commit = new _commit.default(rawCommit);
      return commit;
    });
  }

  getRecentCommits(options) {
    return this.cache.getOrSet(_keys.Keys.recentCommits, async () => {
      const commits = await this.git().getCommits(_objectSpread2({
        ref: 'HEAD'
      }, options));
      return commits.map(commit => new _commit.default(commit));
    });
  }

  async isCommitPushed(sha) {
    const currentBranch = await this.repository.getCurrentBranch();
    const upstream = currentBranch.getPush();

    if (!upstream.isPresent()) {
      return false;
    }

    const contained = await this.git().getBranchesWithCommit(sha, {
      showLocal: false,
      showRemote: true,
      pattern: upstream.getShortRef()
    });
    return contained.some(ref => ref.length > 0);
  } // Author information


  getAuthors(options) {
    // For now we'll do the naive thing and invalidate anytime HEAD moves. This ensures that we get new authors
    // introduced by newly created commits or pulled commits.
    // This means that we are constantly re-fetching data. If performance becomes a concern we can optimize
    return this.cache.getOrSet(_keys.Keys.authors, async () => {
      const authorMap = await this.git().getAuthors(options);
      return Object.keys(authorMap).map(email => new _author.default(email, authorMap[email]));
    });
  } // Branches


  getBranches() {
    return this.cache.getOrSet(_keys.Keys.branches, async () => {
      const payloads = await this.git().getBranches();
      const branches = new _branchSet.default();

      for (const payload of payloads) {
        let upstream = _branch.nullBranch;

        if (payload.upstream) {
          upstream = payload.upstream.remoteName ? _branch.default.createRemoteTracking(payload.upstream.trackingRef, payload.upstream.remoteName, payload.upstream.remoteRef) : new _branch.default(payload.upstream.trackingRef);
        }

        let push = upstream;

        if (payload.push) {
          push = payload.push.remoteName ? _branch.default.createRemoteTracking(payload.push.trackingRef, payload.push.remoteName, payload.push.remoteRef) : new _branch.default(payload.push.trackingRef);
        }

        branches.add(new _branch.default(payload.name, upstream, push, payload.head, {
          sha: payload.sha
        }));
      }

      return branches;
    });
  }

  getHeadDescription() {
    return this.cache.getOrSet(_keys.Keys.headDescription, () => {
      return this.git().describeHead();
    });
  } // Merging and rebasing status


  isMerging() {
    return this.git().isMerging(this.repository.getGitDirectoryPath());
  }

  isRebasing() {
    return this.git().isRebasing(this.repository.getGitDirectoryPath());
  } // Remotes


  getRemotes() {
    return this.cache.getOrSet(_keys.Keys.remotes, async () => {
      const remotesInfo = await this.git().getRemotes();
      return new _remoteSet.default(remotesInfo.map(({
        name,
        url
      }) => new _remote.default(name, url)));
    });
  }

  addRemote(name, url) {
    return this.invalidate(() => [..._keys.Keys.config.eachWithSetting(`remote.${name}.url`), ..._keys.Keys.config.eachWithSetting(`remote.${name}.fetch`), _keys.Keys.remotes], // eslint-disable-next-line no-shadow
    () => this.executePipelineAction('ADDREMOTE', async (name, url) => {
      await this.git().addRemote(name, url);
      return new _remote.default(name, url);
    }, name, url));
  }

  async getAheadCount(branchName) {
    const bundle = await this.getStatusBundle();
    return bundle.branch.aheadBehind.ahead;
  }

  async getBehindCount(branchName) {
    const bundle = await this.getStatusBundle();
    return bundle.branch.aheadBehind.behind;
  }

  getConfig(option, {
    local
  } = {
    local: false
  }) {
    return this.cache.getOrSet(_keys.Keys.config.oneWith(option, {
      local
    }), () => {
      return this.git().getConfig(option, {
        local
      });
    });
  }

  directGetConfig(key, options) {
    return this.getConfig(key, options);
  } // Direct blob access


  getBlobContents(sha) {
    return this.cache.getOrSet(_keys.Keys.blob.oneWith(sha), () => {
      return this.git().getBlobContents(sha);
    });
  }

  directGetBlobContents(sha) {
    return this.getBlobContents(sha);
  } // Discard history


  hasDiscardHistory(partialDiscardFilePath = null) {
    return this.discardHistory.hasHistory(partialDiscardFilePath);
  }

  getDiscardHistory(partialDiscardFilePath = null) {
    return this.discardHistory.getHistory(partialDiscardFilePath);
  }

  getLastHistorySnapshots(partialDiscardFilePath = null) {
    return this.discardHistory.getLastSnapshots(partialDiscardFilePath);
  } // Cache

  /* istanbul ignore next */


  getCache() {
    return this.cache;
  }

  invalidate(spec, body, options = {}) {
    return body().then(result => {
      this.acceptInvalidation(spec, options);
      return result;
    }, err => {
      this.acceptInvalidation(spec, options);
      return Promise.reject(err);
    });
  }

}

exports.default = Present;

_state.default.register(Present);

function partition(array, predicate) {
  const matches = [];
  const nonmatches = [];
  array.forEach(item => {
    if (predicate(item)) {
      matches.push(item);
    } else {
      nonmatches.push(item);
    }
  });
  return [matches, nonmatches];
}

class Cache {
  constructor() {
    this.storage = new Map();
    this.byGroup = new Map();
    this.emitter = new _eventKit.Emitter();
  }

  getOrSet(key, operation) {
    const primary = key.getPrimary();
    const existing = this.storage.get(primary);

    if (existing !== undefined) {
      existing.hits++;
      return existing.promise;
    }

    const created = operation();
    this.storage.set(primary, {
      createdAt: performance.now(),
      hits: 0,
      promise: created
    });
    const groups = key.getGroups();

    for (let i = 0; i < groups.length; i++) {
      const group = groups[i];
      let groupSet = this.byGroup.get(group);

      if (groupSet === undefined) {
        groupSet = new Set();
        this.byGroup.set(group, groupSet);
      }

      groupSet.add(key);
    }

    this.didUpdate();
    return created;
  }

  invalidate(keys) {
    for (let i = 0; i < keys.length; i++) {
      keys[i].removeFromCache(this);
    }

    if (keys.length > 0) {
      this.didUpdate();
    }
  }

  keysInGroup(group) {
    return this.byGroup.get(group) || [];
  }

  removePrimary(primary) {
    this.storage.delete(primary);
    this.didUpdate();
  }

  removeFromGroup(group, key) {
    const groupSet = this.byGroup.get(group);
    groupSet && groupSet.delete(key);
    this.didUpdate();
  }
  /* istanbul ignore next */


  [Symbol.iterator]() {
    return this.storage[Symbol.iterator]();
  }

  clear() {
    this.storage.clear();
    this.byGroup.clear();
    this.didUpdate();
  }

  didUpdate() {
    this.emitter.emit('did-update');
  }
  /* istanbul ignore next */


  onDidUpdate(callback) {
    return this.emitter.on('did-update', callback);
  }

  destroy() {
    this.emitter.dispose();
  }

}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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