"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

var _path = _interopRequireDefault(require("path"));

var _commit = require("../commit");

var _branchSet = _interopRequireDefault(require("../branch-set"));

var _remoteSet = _interopRequireDefault(require("../remote-set"));

var _operationStates = require("../operation-states");

var _multiFilePatch = _interopRequireDefault(require("../patch/multi-file-patch"));

var _compositeGitStrategy = _interopRequireDefault(require("../../composite-git-strategy"));

var _keys = require("./cache/keys");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/**
 * Map of registered subclasses to allow states to transition to one another without circular dependencies.
 * Subclasses of State should call `State.register` to add themselves here.
 */
const stateConstructors = new Map();
/**
 * Base class for Repository states. Implements default "null" behavior.
 */

class State {
  constructor(repository) {
    this.repository = repository;
  }

  static register(Subclass) {
    stateConstructors.set(Subclass.name, Subclass);
  } // This state has just been entered. Perform any asynchronous initialization that needs to occur.


  start() {
    return Promise.resolve();
  } // State probe predicates ////////////////////////////////////////////////////////////////////////////////////////////
  // Allow external callers to identify which state a Repository is in if necessary.


  isLoadingGuess() {
    return false;
  }

  isAbsentGuess() {
    return false;
  }

  isAbsent() {
    return false;
  }

  isLoading() {
    return false;
  }

  isEmpty() {
    return false;
  }

  isPresent() {
    return false;
  }

  isTooLarge() {
    return false;
  }

  isDestroyed() {
    return false;
  } // Behavior probe predicates /////////////////////////////////////////////////////////////////////////////////////////
  // Determine specific rendering behavior based on the current state.


  isUndetermined() {
    return false;
  }

  showGitTabInit() {
    return false;
  }

  showGitTabInitInProgress() {
    return false;
  }

  showGitTabLoading() {
    return false;
  }

  showStatusBarTiles() {
    return false;
  }

  hasDirectory() {
    return true;
  }

  isPublishable() {
    return false;
  } // Lifecycle actions /////////////////////////////////////////////////////////////////////////////////////////////////
  // These generally default to rejecting a Promise with an error.


  init() {
    return unsupportedOperationPromise(this, 'init');
  }

  clone(remoteUrl) {
    return unsupportedOperationPromise(this, 'clone');
  }

  destroy() {
    return this.transitionTo('Destroyed');
  }
  /* istanbul ignore next */


  refresh() {} // No-op

  /* istanbul ignore next */


  observeFilesystemChange(events) {
    this.repository.refresh();
  }
  /* istanbul ignore next */


  updateCommitMessageAfterFileSystemChange() {
    // this is only used in unit tests, we don't need no stinkin coverage
    this.repository.refresh();
  } // Git operations ////////////////////////////////////////////////////////////////////////////////////////////////////
  // These default to rejecting a Promise with an error stating that the operation is not supported in the current
  // state.
  // Staging and unstaging


  stageFiles(paths) {
    return unsupportedOperationPromise(this, 'stageFiles');
  }

  unstageFiles(paths) {
    return unsupportedOperationPromise(this, 'unstageFiles');
  }

  stageFilesFromParentCommit(paths) {
    return unsupportedOperationPromise(this, 'stageFilesFromParentCommit');
  }

  applyPatchToIndex(patch) {
    return unsupportedOperationPromise(this, 'applyPatchToIndex');
  }

  applyPatchToWorkdir(patch) {
    return unsupportedOperationPromise(this, 'applyPatchToWorkdir');
  } // Committing


  commit(message, options) {
    return unsupportedOperationPromise(this, 'commit');
  } // Merging


  merge(branchName) {
    return unsupportedOperationPromise(this, 'merge');
  }

  abortMerge() {
    return unsupportedOperationPromise(this, 'abortMerge');
  }

  checkoutSide(side, paths) {
    return unsupportedOperationPromise(this, 'checkoutSide');
  }

  mergeFile(oursPath, commonBasePath, theirsPath, resultPath) {
    return unsupportedOperationPromise(this, 'mergeFile');
  }

  writeMergeConflictToIndex(filePath, commonBaseSha, oursSha, theirsSha) {
    return unsupportedOperationPromise(this, 'writeMergeConflictToIndex');
  } // Checkout


  checkout(revision, options = {}) {
    return unsupportedOperationPromise(this, 'checkout');
  }

  checkoutPathsAtRevision(paths, revision = 'HEAD') {
    return unsupportedOperationPromise(this, 'checkoutPathsAtRevision');
  } // Reset


  undoLastCommit() {
    return unsupportedOperationPromise(this, 'undoLastCommit');
  } // Remote interactions


  fetch(branchName) {
    return unsupportedOperationPromise(this, 'fetch');
  }

  pull(branchName) {
    return unsupportedOperationPromise(this, 'pull');
  }

  push(branchName) {
    return unsupportedOperationPromise(this, 'push');
  } // Configuration


  async setConfig(optionName, value, options = {}) {
    await this.workdirlessGit().setConfig(optionName, value, options);
    this.didUpdate();

    if (options.global) {
      this.didGloballyInvalidate(() => _keys.Keys.config.eachWithSetting(optionName));
    }
  }

  unsetConfig(option) {
    return unsupportedOperationPromise(this, 'unsetConfig');
  } // Direct blob interactions


  createBlob({
    filePath,
    stdin
  } = {}) {
    return unsupportedOperationPromise(this, 'createBlob');
  }

  expandBlobToFile(absFilePath, sha) {
    return unsupportedOperationPromise(this, 'expandBlobToFile');
  } // Discard history


  createDiscardHistoryBlob() {
    return unsupportedOperationPromise(this, 'createDiscardHistoryBlob');
  }

  updateDiscardHistory() {
    return unsupportedOperationPromise(this, 'updateDiscardHistory');
  }

  storeBeforeAndAfterBlobs(filePaths, isSafe, destructiveAction, partialDiscardFilePath = null) {
    return unsupportedOperationPromise(this, 'storeBeforeAndAfterBlobs');
  }

  restoreLastDiscardInTempFiles(isSafe, partialDiscardFilePath = null) {
    return unsupportedOperationPromise(this, 'restoreLastDiscardInTempFiles');
  }

  popDiscardHistory(partialDiscardFilePath = null) {
    return unsupportedOperationPromise(this, 'popDiscardHistory');
  }

  clearDiscardHistory(partialDiscardFilePath = null) {
    return unsupportedOperationPromise(this, 'clearDiscardHistory');
  }

  discardWorkDirChangesForPaths(paths) {
    return unsupportedOperationPromise(this, 'discardWorkDirChangesForPaths');
  } // Accessors /////////////////////////////////////////////////////////////////////////////////////////////////////////
  // When possible, these default to "empty" results when invoked in states that don't have information available, or
  // fail in a way that's consistent with the requested information not being found.
  // Index queries


  getStatusBundle() {
    return Promise.resolve({
      stagedFiles: {},
      unstagedFiles: {},
      mergeConflictFiles: {},
      branch: {
        oid: null,
        head: null,
        upstream: null,
        aheadBehind: {
          ahead: null,
          behind: null
        }
      }
    });
  }

  getStatusesForChangedFiles() {
    return Promise.resolve({
      stagedFiles: [],
      unstagedFiles: [],
      mergeConflictFiles: []
    });
  }

  getFilePatchForPath(filePath, options = {}) {
    return Promise.resolve(_multiFilePatch.default.createNull());
  }

  getDiffsForFilePath(filePath, options = {}) {
    return Promise.resolve([]);
  }

  getStagedChangesPatch() {
    return Promise.resolve(_multiFilePatch.default.createNull());
  }

  readFileFromIndex(filePath) {
    return Promise.reject(new Error(`fatal: Path ${filePath} does not exist (neither on disk nor in the index).`));
  } // Commit access


  getLastCommit() {
    return Promise.resolve(_commit.nullCommit);
  }

  getCommit() {
    return Promise.resolve(_commit.nullCommit);
  }

  getRecentCommits() {
    return Promise.resolve([]);
  }

  isCommitPushed(sha) {
    return false;
  } // Author information


  getAuthors() {
    return Promise.resolve([]);
  } // Branches


  getBranches() {
    return Promise.resolve(new _branchSet.default());
  }

  getHeadDescription() {
    return Promise.resolve('(no repository)');
  } // Merging and rebasing status


  isMerging() {
    return Promise.resolve(false);
  }

  isRebasing() {
    return Promise.resolve(false);
  } // Remotes


  getRemotes() {
    return Promise.resolve(new _remoteSet.default([]));
  }

  addRemote() {
    return unsupportedOperationPromise(this, 'addRemote');
  }

  getAheadCount(branchName) {
    return Promise.resolve(0);
  }

  getBehindCount(branchName) {
    return Promise.resolve(0);
  }

  getConfig(optionName, options) {
    return this.workdirlessGit().getConfig(optionName, options);
  } // Direct blob access


  getBlobContents(sha) {
    return Promise.reject(new Error(`fatal: Not a valid object name ${sha}`));
  } // Discard history


  hasDiscardHistory(partialDiscardFilePath = null) {
    return false;
  }

  getDiscardHistory(partialDiscardFilePath = null) {
    return [];
  }

  getLastHistorySnapshots(partialDiscardFilePath = null) {
    return null;
  } // Atom repo state


  getOperationStates() {
    return _operationStates.nullOperationStates;
  }

  setCommitMessage(message) {
    return unsupportedOperationPromise(this, 'setCommitMessage');
  }

  getCommitMessage() {
    return '';
  }

  fetchCommitMessageTemplate() {
    return unsupportedOperationPromise(this, 'fetchCommitMessageTemplate');
  } // Cache


  getCache() {
    return null;
  }

  acceptInvalidation() {
    return null;
  } // Internal //////////////////////////////////////////////////////////////////////////////////////////////////////////
  // Non-delegated methods that provide subclasses with convenient access to Repository properties.


  git() {
    return this.repository.git;
  }

  workdir() {
    return this.repository.getWorkingDirectoryPath();
  } // Call methods on the active Repository state, even if the state has transitioned beneath you.
  // Use this to perform operations within `start()` methods to guard against interrupted state transitions.


  current() {
    return this.repository.state;
  } // pipeline


  executePipelineAction(...args) {
    return this.repository.executePipelineAction(...args);
  } // Return a Promise that will resolve once the state transitions from Loading.


  getLoadPromise() {
    return this.repository.getLoadPromise();
  }

  getRemoteForBranch(branchName) {
    return this.repository.getRemoteForBranch(branchName);
  }

  saveDiscardHistory() {
    return this.repository.saveDiscardHistory();
  } // Initiate a transition to another state.


  transitionTo(stateName, ...payload) {
    const StateConstructor = stateConstructors.get(stateName);
    /* istanbul ignore if */

    if (StateConstructor === undefined) {
      throw new Error(`Attempt to transition to unrecognized state ${stateName}`);
    }

    return this.repository.transition(this, StateConstructor, ...payload);
  } // Event broadcast


  didDestroy() {
    return this.repository.emitter.emit('did-destroy');
  }

  didUpdate() {
    return this.repository.emitter.emit('did-update');
  }

  didGloballyInvalidate(spec) {
    return this.repository.emitter.emit('did-globally-invalidate', spec);
  } // Direct git access
  // Non-delegated git operations for internal use within states.


  workdirlessGit() {
    // We want to report config values from the global or system level, but never local ones (unless we're in the
    // present state, which overrides this).
    // The filesystem root is the most likely and convenient place for this to be true.
    const {
      root
    } = _path.default.parse(process.cwd());

    return _compositeGitStrategy.default.create(root);
  }
  /* istanbul ignore next */


  directResolveDotGitDir() {
    return Promise.resolve(null);
  }
  /* istanbul ignore next */


  directGetConfig(key, options = {}) {
    return Promise.resolve(null);
  }
  /* istanbul ignore next */


  directGetBlobContents() {
    return Promise.reject(new Error('Not a valid object name'));
  }
  /* istanbul ignore next */


  directInit() {
    return Promise.resolve();
  }
  /* istanbul ignore next */


  directClone(remoteUrl, options) {
    return Promise.resolve();
  } // Deferred operations
  // Direct raw git operations to the current state, even if the state has been changed. Use these methods within
  // start() methods.


  resolveDotGitDir() {
    return this.current().directResolveDotGitDir();
  }

  doInit(workdir) {
    return this.current().directInit();
  }

  doClone(remoteUrl, options) {
    return this.current().directClone(remoteUrl, options);
  } // Parse a DiscardHistory payload from the SHA recorded in config.


  async loadHistoryPayload() {
    const historySha = await this.current().directGetConfig('atomGithub.historySha');

    if (!historySha) {
      return {};
    }

    let blob;

    try {
      blob = await this.current().directGetBlobContents(historySha);
    } catch (e) {
      if (/Not a valid object name/.test(e.stdErr)) {
        return {};
      }

      throw e;
    }

    try {
      return JSON.parse(blob);
    } catch (e) {
      return {};
    }
  } // Debugging assistance.


  toString() {
    return this.constructor.name;
  }

}

exports.default = State;

function unsupportedOperationPromise(self, opName) {
  return Promise.reject(new Error(`${opName} is not available in ${self} state`));
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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