"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

var _path = _interopRequireDefault(require("path"));

var _eventKit = require("event-kit");

var _fsExtra = _interopRequireDefault(require("fs-extra"));

var _yubikiri = _interopRequireDefault(require("yubikiri"));

var _actionPipeline = require("../action-pipeline");

var _compositeGitStrategy = _interopRequireDefault(require("../composite-git-strategy"));

var _author = _interopRequireWildcard(require("./author"));

var _branch = _interopRequireDefault(require("./branch"));

var _repositoryStates = require("./repository-states");

function _interopRequireWildcard(obj) { if (obj && obj.__esModule) { return obj; } else { var newObj = {}; if (obj != null) { for (var key in obj) { if (Object.prototype.hasOwnProperty.call(obj, key)) { var desc = Object.defineProperty && Object.getOwnPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : {}; if (desc.get || desc.set) { Object.defineProperty(newObj, key, desc); } else { newObj[key] = obj[key]; } } } } newObj.default = obj; return newObj; } }

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _objectSpread2(target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i] != null ? arguments[i] : {}; var ownKeys = Object.keys(source); if (typeof Object.getOwnPropertySymbols === 'function') { ownKeys = ownKeys.concat(Object.getOwnPropertySymbols(source).filter(function (sym) { return Object.getOwnPropertyDescriptor(source, sym).enumerable; })); } ownKeys.forEach(function (key) { _defineProperty(target, key, source[key]); }); } return target; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

const MERGE_MARKER_REGEX = /^(>|<){7} \S+$/m; // Internal option keys used to designate the desired initial state of a Repository.

const initialStateSym = Symbol('initialState');

class Repository {
  constructor(workingDirectoryPath, gitStrategy = null, options = {}) {
    this.workingDirectoryPath = workingDirectoryPath;
    this.git = gitStrategy || _compositeGitStrategy.default.create(workingDirectoryPath);
    this.emitter = new _eventKit.Emitter();
    this.loadPromise = new Promise(resolve => {
      const sub = this.onDidChangeState(() => {
        if (!this.isLoading()) {
          resolve();
          sub.dispose();
        } else if (this.isDestroyed()) {
          sub.dispose();
        }
      });
    });
    this.pipelineManager = options.pipelineManager || (0, _actionPipeline.getNullActionPipelineManager)();
    this.transitionTo(options[initialStateSym] || _repositoryStates.Loading);
  }

  static absent(options) {
    return new Repository(null, null, _objectSpread2({
      [initialStateSym]: _repositoryStates.Absent
    }, options));
  }

  static loadingGuess(options) {
    return new Repository(null, null, _objectSpread2({
      [initialStateSym]: _repositoryStates.LoadingGuess
    }, options));
  }

  static absentGuess(options) {
    return new Repository(null, null, _objectSpread2({
      [initialStateSym]: _repositoryStates.AbsentGuess
    }, options));
  } // State management //////////////////////////////////////////////////////////////////////////////////////////////////


  transition(currentState, StateConstructor, ...payload) {
    if (currentState !== this.state) {
      // Attempted transition from a non-active state, most likely from an asynchronous start() method.
      return Promise.resolve();
    }

    const nextState = new StateConstructor(this, ...payload);
    this.state = nextState;
    this.emitter.emit('did-change-state', {
      from: currentState,
      to: this.state
    });

    if (!this.isDestroyed()) {
      this.emitter.emit('did-update');
    }

    return this.state.start();
  }

  transitionTo(StateConstructor, ...payload) {
    return this.transition(this.state, StateConstructor, ...payload);
  }

  getLoadPromise() {
    return this.isAbsent() ? Promise.reject(new Error('An absent repository will never load')) : this.loadPromise;
  }
  /*
   * Use `callback` to request user input from all git strategies.
   */


  setPromptCallback(callback) {
    this.git.getImplementers().forEach(strategy => strategy.setPromptCallback(callback));
  } // Pipeline


  getPipeline(actionName) {
    const actionKey = this.pipelineManager.actionKeys[actionName];
    return this.pipelineManager.getPipeline(actionKey);
  }

  executePipelineAction(actionName, fn, ...args) {
    const pipeline = this.getPipeline(actionName);
    return pipeline.run(fn, this, ...args);
  } // Event subscription ////////////////////////////////////////////////////////////////////////////////////////////////


  onDidDestroy(callback) {
    return this.emitter.on('did-destroy', callback);
  }

  onDidChangeState(callback) {
    return this.emitter.on('did-change-state', callback);
  }

  onDidUpdate(callback) {
    return this.emitter.on('did-update', callback);
  }

  onDidGloballyInvalidate(callback) {
    return this.emitter.on('did-globally-invalidate', callback);
  }

  onPullError(callback) {
    return this.emitter.on('pull-error', callback);
  }

  didPullError() {
    return this.emitter.emit('pull-error');
  } // State-independent actions /////////////////////////////////////////////////////////////////////////////////////////
  // Actions that use direct filesystem access or otherwise don't need `this.git` to be available.


  async pathHasMergeMarkers(relativePath) {
    try {
      const contents = await _fsExtra.default.readFile(_path.default.join(this.getWorkingDirectoryPath(), relativePath), {
        encoding: 'utf8'
      });
      return MERGE_MARKER_REGEX.test(contents);
    } catch (e) {
      // EISDIR implies this is a submodule
      if (e.code === 'ENOENT' || e.code === 'EISDIR') {
        return false;
      } else {
        throw e;
      }
    }
  }

  async getMergeMessage() {
    try {
      const contents = await _fsExtra.default.readFile(_path.default.join(this.getGitDirectoryPath(), 'MERGE_MSG'), {
        encoding: 'utf8'
      });
      return contents.split(/\n/).filter(line => line.length > 0 && !line.startsWith('#')).join('\n');
    } catch (e) {
      return null;
    }
  } // State-independent accessors ///////////////////////////////////////////////////////////////////////////////////////


  getWorkingDirectoryPath() {
    return this.workingDirectoryPath;
  }

  setGitDirectoryPath(gitDirectoryPath) {
    this._gitDirectoryPath = gitDirectoryPath;
  }

  getGitDirectoryPath() {
    if (this._gitDirectoryPath) {
      return this._gitDirectoryPath;
    } else if (this.getWorkingDirectoryPath()) {
      return _path.default.join(this.getWorkingDirectoryPath(), '.git');
    } else {
      // Absent/Loading/etc.
      return null;
    }
  }

  isInState(stateName) {
    return this.state.constructor.name === stateName;
  }

  toString() {
    return `Repository(state=${this.state.constructor.name}, workdir="${this.getWorkingDirectoryPath()}")`;
  } // Compound Getters //////////////////////////////////////////////////////////////////////////////////////////////////
  // Accessor methods for data derived from other, state-provided getters.


  async getCurrentBranch() {
    const branches = await this.getBranches();
    const head = branches.getHeadBranch();

    if (head.isPresent()) {
      return head;
    }

    const description = await this.getHeadDescription();
    return _branch.default.createDetached(description || 'no branch');
  }

  async getUnstagedChanges() {
    const {
      unstagedFiles
    } = await this.getStatusBundle();
    return Object.keys(unstagedFiles).sort().map(filePath => {
      return {
        filePath,
        status: unstagedFiles[filePath]
      };
    });
  }

  async getStagedChanges() {
    const {
      stagedFiles
    } = await this.getStatusBundle();
    return Object.keys(stagedFiles).sort().map(filePath => {
      return {
        filePath,
        status: stagedFiles[filePath]
      };
    });
  }

  async getMergeConflicts() {
    const {
      mergeConflictFiles
    } = await this.getStatusBundle();
    return Object.keys(mergeConflictFiles).map(filePath => {
      return {
        filePath,
        status: mergeConflictFiles[filePath]
      };
    });
  }

  async isPartiallyStaged(fileName) {
    const {
      unstagedFiles,
      stagedFiles
    } = await this.getStatusBundle();
    const u = unstagedFiles[fileName];
    const s = stagedFiles[fileName];
    return u === 'modified' && s === 'modified' || u === 'modified' && s === 'added' || u === 'added' && s === 'deleted' || u === 'deleted' && s === 'modified';
  }

  async getRemoteForBranch(branchName) {
    const name = await this.getConfig(`branch.${branchName}.remote`);
    return (await this.getRemotes()).withName(name);
  }

  async saveDiscardHistory() {
    if (this.isDestroyed()) {
      return;
    }

    const historySha = await this.createDiscardHistoryBlob();

    if (this.isDestroyed()) {
      return;
    }

    await this.setConfig('atomGithub.historySha', historySha);
  }

  async getCommitter(options = {}) {
    const committer = await (0, _yubikiri.default)({
      email: this.getConfig('user.email', options),
      name: this.getConfig('user.name', options)
    });
    return committer.name !== null && committer.email !== null ? new _author.default(committer.email, committer.name) : _author.nullAuthor;
  } // todo (@annthurium, 3/2019): refactor GitHubTabController etc to use this method.


  async getCurrentGitHubRemote() {
    let currentRemote = null;
    const remotes = await this.getRemotes();
    const gitHubRemotes = remotes.filter(remote => remote.isGithubRepo());
    const selectedRemoteName = await this.getConfig('atomGithub.currentRemote');
    currentRemote = gitHubRemotes.withName(selectedRemoteName);

    if (!currentRemote.isPresent() && gitHubRemotes.size() === 1) {
      currentRemote = Array.from(gitHubRemotes)[0];
    } // todo: handle the case where multiple remotes are available and no chosen remote is set.


    return currentRemote;
  }

  async hasGitHubRemote(host, owner, name) {
    const remotes = await this.getRemotes();
    return remotes.matchingGitHubRepository(owner, name).length > 0;
  }

} // The methods named here will be delegated to the current State.
//
// Duplicated here rather than just using `expectedDelegates` directly so that this file is grep-friendly for answering
// the question of "what all can a Repository do exactly".


exports.default = Repository;
const delegates = ['isLoadingGuess', 'isAbsentGuess', 'isAbsent', 'isLoading', 'isEmpty', 'isPresent', 'isTooLarge', 'isDestroyed', 'isUndetermined', 'showGitTabInit', 'showGitTabInitInProgress', 'showGitTabLoading', 'showStatusBarTiles', 'hasDirectory', 'isPublishable', 'init', 'clone', 'destroy', 'refresh', 'observeFilesystemChange', 'updateCommitMessageAfterFileSystemChange', 'stageFiles', 'unstageFiles', 'stageFilesFromParentCommit', 'stageFileModeChange', 'stageFileSymlinkChange', 'applyPatchToIndex', 'applyPatchToWorkdir', 'commit', 'merge', 'abortMerge', 'checkoutSide', 'mergeFile', 'writeMergeConflictToIndex', 'checkout', 'checkoutPathsAtRevision', 'undoLastCommit', 'fetch', 'pull', 'push', 'setConfig', 'createBlob', 'expandBlobToFile', 'createDiscardHistoryBlob', 'updateDiscardHistory', 'storeBeforeAndAfterBlobs', 'restoreLastDiscardInTempFiles', 'popDiscardHistory', 'clearDiscardHistory', 'discardWorkDirChangesForPaths', 'getStatusBundle', 'getStatusesForChangedFiles', 'getFilePatchForPath', 'getDiffsForFilePath', 'getStagedChangesPatch', 'readFileFromIndex', 'getLastCommit', 'getCommit', 'getRecentCommits', 'isCommitPushed', 'getAuthors', 'getBranches', 'getHeadDescription', 'isMerging', 'isRebasing', 'getRemotes', 'addRemote', 'getAheadCount', 'getBehindCount', 'getConfig', 'unsetConfig', 'getBlobContents', 'hasDiscardHistory', 'getDiscardHistory', 'getLastHistorySnapshots', 'getOperationStates', 'setCommitMessage', 'getCommitMessage', 'fetchCommitMessageTemplate', 'getCache', 'acceptInvalidation'];

for (let i = 0; i < delegates.length; i++) {
  const delegate = delegates[i];

  Repository.prototype[delegate] = function (...args) {
    return this.state[delegate](...args);
  };
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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