"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = exports.source = void 0;

var _yubikiri = _interopRequireDefault(require("yubikiri"));

var _eventKit = require("event-kit");

var _relayNetworkLayerManager = _interopRequireDefault(require("../relay-network-layer-manager"));

var _author = _interopRequireWildcard(require("./author"));

var _keytarStrategy = require("../shared/keytar-strategy");

var _modelObserver = _interopRequireDefault(require("./model-observer"));

function _interopRequireWildcard(obj) { if (obj && obj.__esModule) { return obj; } else { var newObj = {}; if (obj != null) { for (var key in obj) { if (Object.prototype.hasOwnProperty.call(obj, key)) { var desc = Object.defineProperty && Object.getOwnPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : {}; if (desc.get || desc.set) { Object.defineProperty(newObj, key, desc); } else { newObj[key] = obj[key]; } } } } newObj.default = obj; return newObj; } }

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

// This is a guess about what a reasonable value is. Can adjust if performance is poor.
const MAX_COMMITS = 5000;
const source = {
  PENDING: Symbol('pending'),
  GITLOG: Symbol('git log'),
  GITHUBAPI: Symbol('github API')
};
exports.source = source;

class GraphQLCache {
  // One hour
  constructor() {
    this.bySlug = new Map();
  }

  get(remote) {
    const slug = remote.getSlug();
    const {
      ts,
      data
    } = this.bySlug.get(slug) || {
      ts: -Infinity,
      data: {}
    };

    if (Date.now() - ts > this.constructor.MAX_AGE_MS) {
      this.bySlug.delete(slug);
      return null;
    }

    return data;
  }

  set(remote, data) {
    this.bySlug.set(remote.getSlug(), {
      ts: Date.now(),
      data
    });
  }

}

_defineProperty(GraphQLCache, "MAX_AGE_MS", 3.6e6);

class UserStore {
  constructor({
    repository,
    login,
    config
  }) {
    this.emitter = new _eventKit.Emitter();
    this.subs = new _eventKit.CompositeDisposable(); // TODO: [ku 3/2018] Consider using Dexie (indexDB wrapper) like Desktop and persist users across sessions

    this.allUsers = new Map();
    this.excludedUsers = new Set();
    this.users = [];
    this.committer = _author.nullAuthor;
    this.last = {
      source: source.PENDING,
      repository: null,
      excludedUsers: this.excludedUsers
    };
    this.cache = new GraphQLCache();
    this.repositoryObserver = new _modelObserver.default({
      fetchData: r => (0, _yubikiri.default)({
        committer: r.getCommitter(),
        authors: r.getAuthors({
          max: MAX_COMMITS
        }),
        remotes: r.getRemotes()
      }),
      didUpdate: () => this.loadUsers()
    });
    this.repositoryObserver.setActiveModel(repository);
    this.loginObserver = new _modelObserver.default({
      didUpdate: () => this.loadUsers()
    });
    this.loginObserver.setActiveModel(login);
    this.subs.add(config.observe('github.excludedUsers', value => {
      this.excludedUsers = new Set((value || '').split(/\s*,\s*/).filter(each => each.length > 0));
      return this.loadUsers();
    }));
  }

  dispose() {
    this.subs.dispose();
    this.emitter.dispose();
  }

  async loadUsers() {
    const data = this.repositoryObserver.getActiveModelData();

    if (!data) {
      return;
    }

    this.setCommitter(data.committer);
    const githubRemotes = Array.from(data.remotes).filter(remote => remote.isGithubRepo());

    if (githubRemotes.length > 0) {
      await this.loadUsersFromGraphQL(githubRemotes);
    } else {
      this.addUsers(data.authors, source.GITLOG);
    } // if for whatever reason, no committers can be added, fall back to
    // using git log committers as the last resort


    if (this.allUsers.size === 0) {
      this.addUsers(data.authors, source.GITLOG);
    }
  }

  loadUsersFromGraphQL(remotes) {
    return Promise.all(Array.from(remotes, remote => this.loadMentionableUsers(remote)));
  }

  async getToken(loginModel, loginAccount) {
    if (!loginModel) {
      return null;
    }

    const token = await loginModel.getToken(loginAccount);

    if (token === _keytarStrategy.UNAUTHENTICATED || token === _keytarStrategy.INSUFFICIENT || token instanceof Error) {
      return null;
    }

    return token;
  }

  async loadMentionableUsers(remote) {
    const cached = this.cache.get(remote);

    if (cached !== null) {
      this.addUsers(cached, source.GITHUBAPI);
      return;
    }

    const endpoint = remote.getEndpoint();
    const token = await this.getToken(this.loginObserver.getActiveModel(), endpoint.getLoginAccount());

    if (!token) {
      return;
    }

    const fetchQuery = _relayNetworkLayerManager.default.getFetchQuery(endpoint, token);

    let hasMore = true;
    let cursor = null;
    const remoteUsers = [];

    while (hasMore) {
      const response = await fetchQuery({
        name: 'GetMentionableUsers',
        text: `
          query GetMentionableUsers($owner: String!, $name: String!, $first: Int!, $after: String) {
            repository(owner: $owner, name: $name) {
              mentionableUsers(first: $first, after: $after) {
                nodes {
                  login
                  email
                  name
                }
                pageInfo {
                  hasNextPage
                  endCursor
                }
              }
            }
          }
        `
      }, {
        owner: remote.getOwner(),
        name: remote.getRepo(),
        first: 100,
        after: cursor
      });
      /* istanbul ignore if */

      if (response.errors && response.errors.length > 1) {
        // eslint-disable-next-line no-console
        console.error(`Error fetching mentionable users:\n${response.errors.map(e => e.message).join('\n')}`);
      }

      if (!response.data || !response.data.repository) {
        break;
      }

      const connection = response.data.repository.mentionableUsers;
      const authors = connection.nodes.map(node => {
        if (node.email === '') {
          node.email = `${node.login}@users.noreply.github.com`;
        }

        return new _author.default(node.email, node.name, node.login);
      });
      this.addUsers(authors, source.GITHUBAPI);
      remoteUsers.push(...authors);
      cursor = connection.pageInfo.endCursor;
      hasMore = connection.pageInfo.hasNextPage;
    }

    this.cache.set(remote, remoteUsers);
  }

  addUsers(users, nextSource) {
    let changed = false;

    if (nextSource !== this.last.source || this.repositoryObserver.getActiveModel() !== this.last.repository || this.excludedUsers !== this.last.excludedUsers) {
      changed = true;
      this.allUsers.clear();
    }

    for (const author of users) {
      if (!this.allUsers.has(author.getEmail())) {
        changed = true;
      }

      this.allUsers.set(author.getEmail(), author);
    }

    if (changed) {
      this.finalize();
    }

    this.last.source = nextSource;
    this.last.repository = this.repositoryObserver.getActiveModel();
    this.last.excludedUsers = this.excludedUsers;
  }

  finalize() {
    // TODO: [ku 3/2018] consider sorting based on most recent authors or commit frequency
    const users = [];

    for (const author of this.allUsers.values()) {
      if (author.matches(this.committer)) {
        continue;
      }

      if (author.isNoReply()) {
        continue;
      }

      if (this.excludedUsers.has(author.getEmail())) {
        continue;
      }

      users.push(author);
    }

    users.sort(_author.default.compare);
    this.users = users;
    this.didUpdate();
  }

  setRepository(repository) {
    this.repositoryObserver.setActiveModel(repository);
  }

  setLoginModel(login) {
    this.loginObserver.setActiveModel(login);
  }

  setCommitter(committer) {
    const changed = !this.committer.matches(committer);
    this.committer = committer;

    if (changed) {
      this.finalize();
    }
  }

  didUpdate() {
    this.emitter.emit('did-update', this.getUsers());
  }

  onDidUpdate(callback) {
    return this.emitter.on('did-update', callback);
  }

  getUsers() {
    return this.users;
  }

}

exports.default = UserStore;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbInVzZXItc3RvcmUuanMiXSwibmFtZXMiOlsiTUFYX0NPTU1JVFMiLCJzb3VyY2UiLCJQRU5ESU5HIiwiU3ltYm9sIiwiR0lUTE9HIiwiR0lUSFVCQVBJIiwiR3JhcGhRTENhY2hlIiwiY29uc3RydWN0b3IiLCJieVNsdWciLCJNYXAiLCJnZXQiLCJyZW1vdGUiLCJzbHVnIiwiZ2V0U2x1ZyIsInRzIiwiZGF0YSIsIkluZmluaXR5IiwiRGF0ZSIsIm5vdyIsIk1BWF9BR0VfTVMiLCJkZWxldGUiLCJzZXQiLCJVc2VyU3RvcmUiLCJyZXBvc2l0b3J5IiwibG9naW4iLCJjb25maWciLCJlbWl0dGVyIiwiRW1pdHRlciIsInN1YnMiLCJDb21wb3NpdGVEaXNwb3NhYmxlIiwiYWxsVXNlcnMiLCJleGNsdWRlZFVzZXJzIiwiU2V0IiwidXNlcnMiLCJjb21taXR0ZXIiLCJudWxsQXV0aG9yIiwibGFzdCIsImNhY2hlIiwicmVwb3NpdG9yeU9ic2VydmVyIiwiTW9kZWxPYnNlcnZlciIsImZldGNoRGF0YSIsInIiLCJnZXRDb21taXR0ZXIiLCJhdXRob3JzIiwiZ2V0QXV0aG9ycyIsIm1heCIsInJlbW90ZXMiLCJnZXRSZW1vdGVzIiwiZGlkVXBkYXRlIiwibG9hZFVzZXJzIiwic2V0QWN0aXZlTW9kZWwiLCJsb2dpbk9ic2VydmVyIiwiYWRkIiwib2JzZXJ2ZSIsInZhbHVlIiwic3BsaXQiLCJmaWx0ZXIiLCJlYWNoIiwibGVuZ3RoIiwiZGlzcG9zZSIsImdldEFjdGl2ZU1vZGVsRGF0YSIsInNldENvbW1pdHRlciIsImdpdGh1YlJlbW90ZXMiLCJBcnJheSIsImZyb20iLCJpc0dpdGh1YlJlcG8iLCJsb2FkVXNlcnNGcm9tR3JhcGhRTCIsImFkZFVzZXJzIiwic2l6ZSIsIlByb21pc2UiLCJhbGwiLCJsb2FkTWVudGlvbmFibGVVc2VycyIsImdldFRva2VuIiwibG9naW5Nb2RlbCIsImxvZ2luQWNjb3VudCIsInRva2VuIiwiVU5BVVRIRU5USUNBVEVEIiwiSU5TVUZGSUNJRU5UIiwiRXJyb3IiLCJjYWNoZWQiLCJlbmRwb2ludCIsImdldEVuZHBvaW50IiwiZ2V0QWN0aXZlTW9kZWwiLCJnZXRMb2dpbkFjY291bnQiLCJmZXRjaFF1ZXJ5IiwiUmVsYXlOZXR3b3JrTGF5ZXJNYW5hZ2VyIiwiZ2V0RmV0Y2hRdWVyeSIsImhhc01vcmUiLCJjdXJzb3IiLCJyZW1vdGVVc2VycyIsInJlc3BvbnNlIiwibmFtZSIsInRleHQiLCJvd25lciIsImdldE93bmVyIiwiZ2V0UmVwbyIsImZpcnN0IiwiYWZ0ZXIiLCJlcnJvcnMiLCJjb25zb2xlIiwiZXJyb3IiLCJtYXAiLCJlIiwibWVzc2FnZSIsImpvaW4iLCJjb25uZWN0aW9uIiwibWVudGlvbmFibGVVc2VycyIsIm5vZGVzIiwibm9kZSIsImVtYWlsIiwiQXV0aG9yIiwicHVzaCIsInBhZ2VJbmZvIiwiZW5kQ3Vyc29yIiwiaGFzTmV4dFBhZ2UiLCJuZXh0U291cmNlIiwiY2hhbmdlZCIsImNsZWFyIiwiYXV0aG9yIiwiaGFzIiwiZ2V0RW1haWwiLCJmaW5hbGl6ZSIsInZhbHVlcyIsIm1hdGNoZXMiLCJpc05vUmVwbHkiLCJzb3J0IiwiY29tcGFyZSIsInNldFJlcG9zaXRvcnkiLCJzZXRMb2dpbk1vZGVsIiwiZW1pdCIsImdldFVzZXJzIiwib25EaWRVcGRhdGUiLCJjYWxsYmFjayIsIm9uIl0sIm1hcHBpbmdzIjoiOzs7Ozs7O0FBQUE7O0FBQ0E7O0FBRUE7O0FBQ0E7O0FBQ0E7O0FBQ0E7Ozs7Ozs7O0FBRUE7QUFDQSxNQUFNQSxXQUFXLEdBQUcsSUFBcEI7QUFFTyxNQUFNQyxNQUFNLEdBQUc7QUFDcEJDLEVBQUFBLE9BQU8sRUFBRUMsTUFBTSxDQUFDLFNBQUQsQ0FESztBQUVwQkMsRUFBQUEsTUFBTSxFQUFFRCxNQUFNLENBQUMsU0FBRCxDQUZNO0FBR3BCRSxFQUFBQSxTQUFTLEVBQUVGLE1BQU0sQ0FBQyxZQUFEO0FBSEcsQ0FBZjs7O0FBTVAsTUFBTUcsWUFBTixDQUFtQjtBQUNqQjtBQUdBQyxFQUFBQSxXQUFXLEdBQUc7QUFDWixTQUFLQyxNQUFMLEdBQWMsSUFBSUMsR0FBSixFQUFkO0FBQ0Q7O0FBRURDLEVBQUFBLEdBQUcsQ0FBQ0MsTUFBRCxFQUFTO0FBQ1YsVUFBTUMsSUFBSSxHQUFHRCxNQUFNLENBQUNFLE9BQVAsRUFBYjtBQUNBLFVBQU07QUFBQ0MsTUFBQUEsRUFBRDtBQUFLQyxNQUFBQTtBQUFMLFFBQWEsS0FBS1AsTUFBTCxDQUFZRSxHQUFaLENBQWdCRSxJQUFoQixLQUF5QjtBQUMxQ0UsTUFBQUEsRUFBRSxFQUFFLENBQUNFLFFBRHFDO0FBRTFDRCxNQUFBQSxJQUFJLEVBQUU7QUFGb0MsS0FBNUM7O0FBS0EsUUFBSUUsSUFBSSxDQUFDQyxHQUFMLEtBQWFKLEVBQWIsR0FBa0IsS0FBS1AsV0FBTCxDQUFpQlksVUFBdkMsRUFBbUQ7QUFDakQsV0FBS1gsTUFBTCxDQUFZWSxNQUFaLENBQW1CUixJQUFuQjtBQUNBLGFBQU8sSUFBUDtBQUNEOztBQUNELFdBQU9HLElBQVA7QUFDRDs7QUFFRE0sRUFBQUEsR0FBRyxDQUFDVixNQUFELEVBQVNJLElBQVQsRUFBZTtBQUNoQixTQUFLUCxNQUFMLENBQVlhLEdBQVosQ0FBZ0JWLE1BQU0sQ0FBQ0UsT0FBUCxFQUFoQixFQUFrQztBQUFDQyxNQUFBQSxFQUFFLEVBQUVHLElBQUksQ0FBQ0MsR0FBTCxFQUFMO0FBQWlCSCxNQUFBQTtBQUFqQixLQUFsQztBQUNEOztBQXhCZ0I7O2dCQUFiVCxZLGdCQUVnQixLOztBQXlCUCxNQUFNZ0IsU0FBTixDQUFnQjtBQUM3QmYsRUFBQUEsV0FBVyxDQUFDO0FBQUNnQixJQUFBQSxVQUFEO0FBQWFDLElBQUFBLEtBQWI7QUFBb0JDLElBQUFBO0FBQXBCLEdBQUQsRUFBOEI7QUFDdkMsU0FBS0MsT0FBTCxHQUFlLElBQUlDLGlCQUFKLEVBQWY7QUFDQSxTQUFLQyxJQUFMLEdBQVksSUFBSUMsNkJBQUosRUFBWixDQUZ1QyxDQUl2Qzs7QUFDQSxTQUFLQyxRQUFMLEdBQWdCLElBQUlyQixHQUFKLEVBQWhCO0FBQ0EsU0FBS3NCLGFBQUwsR0FBcUIsSUFBSUMsR0FBSixFQUFyQjtBQUNBLFNBQUtDLEtBQUwsR0FBYSxFQUFiO0FBQ0EsU0FBS0MsU0FBTCxHQUFpQkMsa0JBQWpCO0FBRUEsU0FBS0MsSUFBTCxHQUFZO0FBQ1ZuQyxNQUFBQSxNQUFNLEVBQUVBLE1BQU0sQ0FBQ0MsT0FETDtBQUVWcUIsTUFBQUEsVUFBVSxFQUFFLElBRkY7QUFHVlEsTUFBQUEsYUFBYSxFQUFFLEtBQUtBO0FBSFYsS0FBWjtBQUtBLFNBQUtNLEtBQUwsR0FBYSxJQUFJL0IsWUFBSixFQUFiO0FBRUEsU0FBS2dDLGtCQUFMLEdBQTBCLElBQUlDLHNCQUFKLENBQWtCO0FBQzFDQyxNQUFBQSxTQUFTLEVBQUVDLENBQUMsSUFBSSx1QkFBUztBQUN2QlAsUUFBQUEsU0FBUyxFQUFFTyxDQUFDLENBQUNDLFlBQUYsRUFEWTtBQUV2QkMsUUFBQUEsT0FBTyxFQUFFRixDQUFDLENBQUNHLFVBQUYsQ0FBYTtBQUFDQyxVQUFBQSxHQUFHLEVBQUU3QztBQUFOLFNBQWIsQ0FGYztBQUd2QjhDLFFBQUFBLE9BQU8sRUFBRUwsQ0FBQyxDQUFDTSxVQUFGO0FBSGMsT0FBVCxDQUQwQjtBQU0xQ0MsTUFBQUEsU0FBUyxFQUFFLE1BQU0sS0FBS0MsU0FBTDtBQU55QixLQUFsQixDQUExQjtBQVFBLFNBQUtYLGtCQUFMLENBQXdCWSxjQUF4QixDQUF1QzNCLFVBQXZDO0FBRUEsU0FBSzRCLGFBQUwsR0FBcUIsSUFBSVosc0JBQUosQ0FBa0I7QUFDckNTLE1BQUFBLFNBQVMsRUFBRSxNQUFNLEtBQUtDLFNBQUw7QUFEb0IsS0FBbEIsQ0FBckI7QUFHQSxTQUFLRSxhQUFMLENBQW1CRCxjQUFuQixDQUFrQzFCLEtBQWxDO0FBRUEsU0FBS0ksSUFBTCxDQUFVd0IsR0FBVixDQUNFM0IsTUFBTSxDQUFDNEIsT0FBUCxDQUFlLHNCQUFmLEVBQXVDQyxLQUFLLElBQUk7QUFDOUMsV0FBS3ZCLGFBQUwsR0FBcUIsSUFBSUMsR0FBSixDQUNuQixDQUFDc0IsS0FBSyxJQUFJLEVBQVYsRUFBY0MsS0FBZCxDQUFvQixTQUFwQixFQUErQkMsTUFBL0IsQ0FBc0NDLElBQUksSUFBSUEsSUFBSSxDQUFDQyxNQUFMLEdBQWMsQ0FBNUQsQ0FEbUIsQ0FBckI7QUFHQSxhQUFPLEtBQUtULFNBQUwsRUFBUDtBQUNELEtBTEQsQ0FERjtBQVFEOztBQUVEVSxFQUFBQSxPQUFPLEdBQUc7QUFDUixTQUFLL0IsSUFBTCxDQUFVK0IsT0FBVjtBQUNBLFNBQUtqQyxPQUFMLENBQWFpQyxPQUFiO0FBQ0Q7O0FBRUQsUUFBTVYsU0FBTixHQUFrQjtBQUNoQixVQUFNbEMsSUFBSSxHQUFHLEtBQUt1QixrQkFBTCxDQUF3QnNCLGtCQUF4QixFQUFiOztBQUVBLFFBQUksQ0FBQzdDLElBQUwsRUFBVztBQUNUO0FBQ0Q7O0FBRUQsU0FBSzhDLFlBQUwsQ0FBa0I5QyxJQUFJLENBQUNtQixTQUF2QjtBQUNBLFVBQU00QixhQUFhLEdBQUdDLEtBQUssQ0FBQ0MsSUFBTixDQUFXakQsSUFBSSxDQUFDK0IsT0FBaEIsRUFBeUJVLE1BQXpCLENBQWdDN0MsTUFBTSxJQUFJQSxNQUFNLENBQUNzRCxZQUFQLEVBQTFDLENBQXRCOztBQUVBLFFBQUlILGFBQWEsQ0FBQ0osTUFBZCxHQUF1QixDQUEzQixFQUE4QjtBQUM1QixZQUFNLEtBQUtRLG9CQUFMLENBQTBCSixhQUExQixDQUFOO0FBQ0QsS0FGRCxNQUVPO0FBQ0wsV0FBS0ssUUFBTCxDQUFjcEQsSUFBSSxDQUFDNEIsT0FBbkIsRUFBNEIxQyxNQUFNLENBQUNHLE1BQW5DO0FBQ0QsS0FkZSxDQWdCaEI7QUFDQTs7O0FBQ0EsUUFBSSxLQUFLMEIsUUFBTCxDQUFjc0MsSUFBZCxLQUF1QixDQUEzQixFQUE4QjtBQUM1QixXQUFLRCxRQUFMLENBQWNwRCxJQUFJLENBQUM0QixPQUFuQixFQUE0QjFDLE1BQU0sQ0FBQ0csTUFBbkM7QUFDRDtBQUNGOztBQUVEOEQsRUFBQUEsb0JBQW9CLENBQUNwQixPQUFELEVBQVU7QUFDNUIsV0FBT3VCLE9BQU8sQ0FBQ0MsR0FBUixDQUNMUCxLQUFLLENBQUNDLElBQU4sQ0FBV2xCLE9BQVgsRUFBb0JuQyxNQUFNLElBQUksS0FBSzRELG9CQUFMLENBQTBCNUQsTUFBMUIsQ0FBOUIsQ0FESyxDQUFQO0FBR0Q7O0FBRUQsUUFBTTZELFFBQU4sQ0FBZUMsVUFBZixFQUEyQkMsWUFBM0IsRUFBeUM7QUFDdkMsUUFBSSxDQUFDRCxVQUFMLEVBQWlCO0FBQ2YsYUFBTyxJQUFQO0FBQ0Q7O0FBQ0QsVUFBTUUsS0FBSyxHQUFHLE1BQU1GLFVBQVUsQ0FBQ0QsUUFBWCxDQUFvQkUsWUFBcEIsQ0FBcEI7O0FBQ0EsUUFBSUMsS0FBSyxLQUFLQywrQkFBVixJQUE2QkQsS0FBSyxLQUFLRSw0QkFBdkMsSUFBdURGLEtBQUssWUFBWUcsS0FBNUUsRUFBbUY7QUFDakYsYUFBTyxJQUFQO0FBQ0Q7O0FBQ0QsV0FBT0gsS0FBUDtBQUNEOztBQUVELFFBQU1KLG9CQUFOLENBQTJCNUQsTUFBM0IsRUFBbUM7QUFDakMsVUFBTW9FLE1BQU0sR0FBRyxLQUFLMUMsS0FBTCxDQUFXM0IsR0FBWCxDQUFlQyxNQUFmLENBQWY7O0FBQ0EsUUFBSW9FLE1BQU0sS0FBSyxJQUFmLEVBQXFCO0FBQ25CLFdBQUtaLFFBQUwsQ0FBY1ksTUFBZCxFQUFzQjlFLE1BQU0sQ0FBQ0ksU0FBN0I7QUFDQTtBQUNEOztBQUVELFVBQU0yRSxRQUFRLEdBQUdyRSxNQUFNLENBQUNzRSxXQUFQLEVBQWpCO0FBQ0EsVUFBTU4sS0FBSyxHQUFHLE1BQU0sS0FBS0gsUUFBTCxDQUFjLEtBQUtyQixhQUFMLENBQW1CK0IsY0FBbkIsRUFBZCxFQUFtREYsUUFBUSxDQUFDRyxlQUFULEVBQW5ELENBQXBCOztBQUNBLFFBQUksQ0FBQ1IsS0FBTCxFQUFZO0FBQ1Y7QUFDRDs7QUFFRCxVQUFNUyxVQUFVLEdBQUdDLGtDQUF5QkMsYUFBekIsQ0FBdUNOLFFBQXZDLEVBQWlETCxLQUFqRCxDQUFuQjs7QUFFQSxRQUFJWSxPQUFPLEdBQUcsSUFBZDtBQUNBLFFBQUlDLE1BQU0sR0FBRyxJQUFiO0FBQ0EsVUFBTUMsV0FBVyxHQUFHLEVBQXBCOztBQUVBLFdBQU9GLE9BQVAsRUFBZ0I7QUFDZCxZQUFNRyxRQUFRLEdBQUcsTUFBTU4sVUFBVSxDQUFDO0FBQ2hDTyxRQUFBQSxJQUFJLEVBQUUscUJBRDBCO0FBRWhDQyxRQUFBQSxJQUFJLEVBQUc7Ozs7Ozs7Ozs7Ozs7Ozs7O0FBRnlCLE9BQUQsRUFtQjlCO0FBQ0RDLFFBQUFBLEtBQUssRUFBRWxGLE1BQU0sQ0FBQ21GLFFBQVAsRUFETjtBQUVESCxRQUFBQSxJQUFJLEVBQUVoRixNQUFNLENBQUNvRixPQUFQLEVBRkw7QUFHREMsUUFBQUEsS0FBSyxFQUFFLEdBSE47QUFJREMsUUFBQUEsS0FBSyxFQUFFVDtBQUpOLE9BbkI4QixDQUFqQztBQTBCQTs7QUFDQSxVQUFJRSxRQUFRLENBQUNRLE1BQVQsSUFBbUJSLFFBQVEsQ0FBQ1EsTUFBVCxDQUFnQnhDLE1BQWhCLEdBQXlCLENBQWhELEVBQW1EO0FBQ2pEO0FBQ0F5QyxRQUFBQSxPQUFPLENBQUNDLEtBQVIsQ0FBZSxzQ0FBcUNWLFFBQVEsQ0FBQ1EsTUFBVCxDQUFnQkcsR0FBaEIsQ0FBb0JDLENBQUMsSUFBSUEsQ0FBQyxDQUFDQyxPQUEzQixFQUFvQ0MsSUFBcEMsQ0FBeUMsSUFBekMsQ0FBK0MsRUFBbkc7QUFDRDs7QUFFRCxVQUFJLENBQUNkLFFBQVEsQ0FBQzNFLElBQVYsSUFBa0IsQ0FBQzJFLFFBQVEsQ0FBQzNFLElBQVQsQ0FBY1EsVUFBckMsRUFBaUQ7QUFDL0M7QUFDRDs7QUFFRCxZQUFNa0YsVUFBVSxHQUFHZixRQUFRLENBQUMzRSxJQUFULENBQWNRLFVBQWQsQ0FBeUJtRixnQkFBNUM7QUFDQSxZQUFNL0QsT0FBTyxHQUFHOEQsVUFBVSxDQUFDRSxLQUFYLENBQWlCTixHQUFqQixDQUFxQk8sSUFBSSxJQUFJO0FBQzNDLFlBQUlBLElBQUksQ0FBQ0MsS0FBTCxLQUFlLEVBQW5CLEVBQXVCO0FBQ3JCRCxVQUFBQSxJQUFJLENBQUNDLEtBQUwsR0FBYyxHQUFFRCxJQUFJLENBQUNwRixLQUFNLDJCQUEzQjtBQUNEOztBQUVELGVBQU8sSUFBSXNGLGVBQUosQ0FBV0YsSUFBSSxDQUFDQyxLQUFoQixFQUF1QkQsSUFBSSxDQUFDakIsSUFBNUIsRUFBa0NpQixJQUFJLENBQUNwRixLQUF2QyxDQUFQO0FBQ0QsT0FOZSxDQUFoQjtBQU9BLFdBQUsyQyxRQUFMLENBQWN4QixPQUFkLEVBQXVCMUMsTUFBTSxDQUFDSSxTQUE5QjtBQUNBb0YsTUFBQUEsV0FBVyxDQUFDc0IsSUFBWixDQUFpQixHQUFHcEUsT0FBcEI7QUFFQTZDLE1BQUFBLE1BQU0sR0FBR2lCLFVBQVUsQ0FBQ08sUUFBWCxDQUFvQkMsU0FBN0I7QUFDQTFCLE1BQUFBLE9BQU8sR0FBR2tCLFVBQVUsQ0FBQ08sUUFBWCxDQUFvQkUsV0FBOUI7QUFDRDs7QUFFRCxTQUFLN0UsS0FBTCxDQUFXaEIsR0FBWCxDQUFlVixNQUFmLEVBQXVCOEUsV0FBdkI7QUFDRDs7QUFFRHRCLEVBQUFBLFFBQVEsQ0FBQ2xDLEtBQUQsRUFBUWtGLFVBQVIsRUFBb0I7QUFDMUIsUUFBSUMsT0FBTyxHQUFHLEtBQWQ7O0FBRUEsUUFDRUQsVUFBVSxLQUFLLEtBQUsvRSxJQUFMLENBQVVuQyxNQUF6QixJQUNBLEtBQUtxQyxrQkFBTCxDQUF3QjRDLGNBQXhCLE9BQTZDLEtBQUs5QyxJQUFMLENBQVViLFVBRHZELElBRUEsS0FBS1EsYUFBTCxLQUF1QixLQUFLSyxJQUFMLENBQVVMLGFBSG5DLEVBSUU7QUFDQXFGLE1BQUFBLE9BQU8sR0FBRyxJQUFWO0FBQ0EsV0FBS3RGLFFBQUwsQ0FBY3VGLEtBQWQ7QUFDRDs7QUFFRCxTQUFLLE1BQU1DLE1BQVgsSUFBcUJyRixLQUFyQixFQUE0QjtBQUMxQixVQUFJLENBQUMsS0FBS0gsUUFBTCxDQUFjeUYsR0FBZCxDQUFrQkQsTUFBTSxDQUFDRSxRQUFQLEVBQWxCLENBQUwsRUFBMkM7QUFDekNKLFFBQUFBLE9BQU8sR0FBRyxJQUFWO0FBQ0Q7O0FBQ0QsV0FBS3RGLFFBQUwsQ0FBY1QsR0FBZCxDQUFrQmlHLE1BQU0sQ0FBQ0UsUUFBUCxFQUFsQixFQUFxQ0YsTUFBckM7QUFDRDs7QUFFRCxRQUFJRixPQUFKLEVBQWE7QUFDWCxXQUFLSyxRQUFMO0FBQ0Q7O0FBQ0QsU0FBS3JGLElBQUwsQ0FBVW5DLE1BQVYsR0FBbUJrSCxVQUFuQjtBQUNBLFNBQUsvRSxJQUFMLENBQVViLFVBQVYsR0FBdUIsS0FBS2Usa0JBQUwsQ0FBd0I0QyxjQUF4QixFQUF2QjtBQUNBLFNBQUs5QyxJQUFMLENBQVVMLGFBQVYsR0FBMEIsS0FBS0EsYUFBL0I7QUFDRDs7QUFFRDBGLEVBQUFBLFFBQVEsR0FBRztBQUNUO0FBQ0EsVUFBTXhGLEtBQUssR0FBRyxFQUFkOztBQUNBLFNBQUssTUFBTXFGLE1BQVgsSUFBcUIsS0FBS3hGLFFBQUwsQ0FBYzRGLE1BQWQsRUFBckIsRUFBNkM7QUFDM0MsVUFBSUosTUFBTSxDQUFDSyxPQUFQLENBQWUsS0FBS3pGLFNBQXBCLENBQUosRUFBb0M7QUFBRTtBQUFXOztBQUNqRCxVQUFJb0YsTUFBTSxDQUFDTSxTQUFQLEVBQUosRUFBd0I7QUFBRTtBQUFXOztBQUNyQyxVQUFJLEtBQUs3RixhQUFMLENBQW1Cd0YsR0FBbkIsQ0FBdUJELE1BQU0sQ0FBQ0UsUUFBUCxFQUF2QixDQUFKLEVBQStDO0FBQUU7QUFBVzs7QUFFNUR2RixNQUFBQSxLQUFLLENBQUM4RSxJQUFOLENBQVdPLE1BQVg7QUFDRDs7QUFDRHJGLElBQUFBLEtBQUssQ0FBQzRGLElBQU4sQ0FBV2YsZ0JBQU9nQixPQUFsQjtBQUNBLFNBQUs3RixLQUFMLEdBQWFBLEtBQWI7QUFDQSxTQUFLZSxTQUFMO0FBQ0Q7O0FBRUQrRSxFQUFBQSxhQUFhLENBQUN4RyxVQUFELEVBQWE7QUFDeEIsU0FBS2Usa0JBQUwsQ0FBd0JZLGNBQXhCLENBQXVDM0IsVUFBdkM7QUFDRDs7QUFFRHlHLEVBQUFBLGFBQWEsQ0FBQ3hHLEtBQUQsRUFBUTtBQUNuQixTQUFLMkIsYUFBTCxDQUFtQkQsY0FBbkIsQ0FBa0MxQixLQUFsQztBQUNEOztBQUVEcUMsRUFBQUEsWUFBWSxDQUFDM0IsU0FBRCxFQUFZO0FBQ3RCLFVBQU1rRixPQUFPLEdBQUcsQ0FBQyxLQUFLbEYsU0FBTCxDQUFleUYsT0FBZixDQUF1QnpGLFNBQXZCLENBQWpCO0FBQ0EsU0FBS0EsU0FBTCxHQUFpQkEsU0FBakI7O0FBQ0EsUUFBSWtGLE9BQUosRUFBYTtBQUNYLFdBQUtLLFFBQUw7QUFDRDtBQUNGOztBQUVEekUsRUFBQUEsU0FBUyxHQUFHO0FBQ1YsU0FBS3RCLE9BQUwsQ0FBYXVHLElBQWIsQ0FBa0IsWUFBbEIsRUFBZ0MsS0FBS0MsUUFBTCxFQUFoQztBQUNEOztBQUVEQyxFQUFBQSxXQUFXLENBQUNDLFFBQUQsRUFBVztBQUNwQixXQUFPLEtBQUsxRyxPQUFMLENBQWEyRyxFQUFiLENBQWdCLFlBQWhCLEVBQThCRCxRQUE5QixDQUFQO0FBQ0Q7O0FBRURGLEVBQUFBLFFBQVEsR0FBRztBQUNULFdBQU8sS0FBS2pHLEtBQVo7QUFDRDs7QUF0TzRCIiwic291cmNlUm9vdCI6Ii9idWlsZC9hdG9tL3NyYy9hdG9tL291dC9hcHAvbm9kZV9tb2R1bGVzL2dpdGh1YiIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB5dWJpa2lyaSBmcm9tICd5dWJpa2lyaSc7XG5pbXBvcnQge0VtaXR0ZXIsIENvbXBvc2l0ZURpc3Bvc2FibGV9IGZyb20gJ2V2ZW50LWtpdCc7XG5cbmltcG9ydCBSZWxheU5ldHdvcmtMYXllck1hbmFnZXIgZnJvbSAnLi4vcmVsYXktbmV0d29yay1sYXllci1tYW5hZ2VyJztcbmltcG9ydCBBdXRob3IsIHtudWxsQXV0aG9yfSBmcm9tICcuL2F1dGhvcic7XG5pbXBvcnQge1VOQVVUSEVOVElDQVRFRCwgSU5TVUZGSUNJRU5UfSBmcm9tICcuLi9zaGFyZWQva2V5dGFyLXN0cmF0ZWd5JztcbmltcG9ydCBNb2RlbE9ic2VydmVyIGZyb20gJy4vbW9kZWwtb2JzZXJ2ZXInO1xuXG4vLyBUaGlzIGlzIGEgZ3Vlc3MgYWJvdXQgd2hhdCBhIHJlYXNvbmFibGUgdmFsdWUgaXMuIENhbiBhZGp1c3QgaWYgcGVyZm9ybWFuY2UgaXMgcG9vci5cbmNvbnN0IE1BWF9DT01NSVRTID0gNTAwMDtcblxuZXhwb3J0IGNvbnN0IHNvdXJjZSA9IHtcbiAgUEVORElORzogU3ltYm9sKCdwZW5kaW5nJyksXG4gIEdJVExPRzogU3ltYm9sKCdnaXQgbG9nJyksXG4gIEdJVEhVQkFQSTogU3ltYm9sKCdnaXRodWIgQVBJJyksXG59O1xuXG5jbGFzcyBHcmFwaFFMQ2FjaGUge1xuICAvLyBPbmUgaG91clxuICBzdGF0aWMgTUFYX0FHRV9NUyA9IDMuNmU2XG5cbiAgY29uc3RydWN0b3IoKSB7XG4gICAgdGhpcy5ieVNsdWcgPSBuZXcgTWFwKCk7XG4gIH1cblxuICBnZXQocmVtb3RlKSB7XG4gICAgY29uc3Qgc2x1ZyA9IHJlbW90ZS5nZXRTbHVnKCk7XG4gICAgY29uc3Qge3RzLCBkYXRhfSA9IHRoaXMuYnlTbHVnLmdldChzbHVnKSB8fCB7XG4gICAgICB0czogLUluZmluaXR5LFxuICAgICAgZGF0YToge30sXG4gICAgfTtcblxuICAgIGlmIChEYXRlLm5vdygpIC0gdHMgPiB0aGlzLmNvbnN0cnVjdG9yLk1BWF9BR0VfTVMpIHtcbiAgICAgIHRoaXMuYnlTbHVnLmRlbGV0ZShzbHVnKTtcbiAgICAgIHJldHVybiBudWxsO1xuICAgIH1cbiAgICByZXR1cm4gZGF0YTtcbiAgfVxuXG4gIHNldChyZW1vdGUsIGRhdGEpIHtcbiAgICB0aGlzLmJ5U2x1Zy5zZXQocmVtb3RlLmdldFNsdWcoKSwge3RzOiBEYXRlLm5vdygpLCBkYXRhfSk7XG4gIH1cbn1cblxuZXhwb3J0IGRlZmF1bHQgY2xhc3MgVXNlclN0b3JlIHtcbiAgY29uc3RydWN0b3Ioe3JlcG9zaXRvcnksIGxvZ2luLCBjb25maWd9KSB7XG4gICAgdGhpcy5lbWl0dGVyID0gbmV3IEVtaXR0ZXIoKTtcbiAgICB0aGlzLnN1YnMgPSBuZXcgQ29tcG9zaXRlRGlzcG9zYWJsZSgpO1xuXG4gICAgLy8gVE9ETzogW2t1IDMvMjAxOF0gQ29uc2lkZXIgdXNpbmcgRGV4aWUgKGluZGV4REIgd3JhcHBlcikgbGlrZSBEZXNrdG9wIGFuZCBwZXJzaXN0IHVzZXJzIGFjcm9zcyBzZXNzaW9uc1xuICAgIHRoaXMuYWxsVXNlcnMgPSBuZXcgTWFwKCk7XG4gICAgdGhpcy5leGNsdWRlZFVzZXJzID0gbmV3IFNldCgpO1xuICAgIHRoaXMudXNlcnMgPSBbXTtcbiAgICB0aGlzLmNvbW1pdHRlciA9IG51bGxBdXRob3I7XG5cbiAgICB0aGlzLmxhc3QgPSB7XG4gICAgICBzb3VyY2U6IHNvdXJjZS5QRU5ESU5HLFxuICAgICAgcmVwb3NpdG9yeTogbnVsbCxcbiAgICAgIGV4Y2x1ZGVkVXNlcnM6IHRoaXMuZXhjbHVkZWRVc2VycyxcbiAgICB9O1xuICAgIHRoaXMuY2FjaGUgPSBuZXcgR3JhcGhRTENhY2hlKCk7XG5cbiAgICB0aGlzLnJlcG9zaXRvcnlPYnNlcnZlciA9IG5ldyBNb2RlbE9ic2VydmVyKHtcbiAgICAgIGZldGNoRGF0YTogciA9PiB5dWJpa2lyaSh7XG4gICAgICAgIGNvbW1pdHRlcjogci5nZXRDb21taXR0ZXIoKSxcbiAgICAgICAgYXV0aG9yczogci5nZXRBdXRob3JzKHttYXg6IE1BWF9DT01NSVRTfSksXG4gICAgICAgIHJlbW90ZXM6IHIuZ2V0UmVtb3RlcygpLFxuICAgICAgfSksXG4gICAgICBkaWRVcGRhdGU6ICgpID0+IHRoaXMubG9hZFVzZXJzKCksXG4gICAgfSk7XG4gICAgdGhpcy5yZXBvc2l0b3J5T2JzZXJ2ZXIuc2V0QWN0aXZlTW9kZWwocmVwb3NpdG9yeSk7XG5cbiAgICB0aGlzLmxvZ2luT2JzZXJ2ZXIgPSBuZXcgTW9kZWxPYnNlcnZlcih7XG4gICAgICBkaWRVcGRhdGU6ICgpID0+IHRoaXMubG9hZFVzZXJzKCksXG4gICAgfSk7XG4gICAgdGhpcy5sb2dpbk9ic2VydmVyLnNldEFjdGl2ZU1vZGVsKGxvZ2luKTtcblxuICAgIHRoaXMuc3Vicy5hZGQoXG4gICAgICBjb25maWcub2JzZXJ2ZSgnZ2l0aHViLmV4Y2x1ZGVkVXNlcnMnLCB2YWx1ZSA9PiB7XG4gICAgICAgIHRoaXMuZXhjbHVkZWRVc2VycyA9IG5ldyBTZXQoXG4gICAgICAgICAgKHZhbHVlIHx8ICcnKS5zcGxpdCgvXFxzKixcXHMqLykuZmlsdGVyKGVhY2ggPT4gZWFjaC5sZW5ndGggPiAwKSxcbiAgICAgICAgKTtcbiAgICAgICAgcmV0dXJuIHRoaXMubG9hZFVzZXJzKCk7XG4gICAgICB9KSxcbiAgICApO1xuICB9XG5cbiAgZGlzcG9zZSgpIHtcbiAgICB0aGlzLnN1YnMuZGlzcG9zZSgpO1xuICAgIHRoaXMuZW1pdHRlci5kaXNwb3NlKCk7XG4gIH1cblxuICBhc3luYyBsb2FkVXNlcnMoKSB7XG4gICAgY29uc3QgZGF0YSA9IHRoaXMucmVwb3NpdG9yeU9ic2VydmVyLmdldEFjdGl2ZU1vZGVsRGF0YSgpO1xuXG4gICAgaWYgKCFkYXRhKSB7XG4gICAgICByZXR1cm47XG4gICAgfVxuXG4gICAgdGhpcy5zZXRDb21taXR0ZXIoZGF0YS5jb21taXR0ZXIpO1xuICAgIGNvbnN0IGdpdGh1YlJlbW90ZXMgPSBBcnJheS5mcm9tKGRhdGEucmVtb3RlcykuZmlsdGVyKHJlbW90ZSA9PiByZW1vdGUuaXNHaXRodWJSZXBvKCkpO1xuXG4gICAgaWYgKGdpdGh1YlJlbW90ZXMubGVuZ3RoID4gMCkge1xuICAgICAgYXdhaXQgdGhpcy5sb2FkVXNlcnNGcm9tR3JhcGhRTChnaXRodWJSZW1vdGVzKTtcbiAgICB9IGVsc2Uge1xuICAgICAgdGhpcy5hZGRVc2VycyhkYXRhLmF1dGhvcnMsIHNvdXJjZS5HSVRMT0cpO1xuICAgIH1cblxuICAgIC8vIGlmIGZvciB3aGF0ZXZlciByZWFzb24sIG5vIGNvbW1pdHRlcnMgY2FuIGJlIGFkZGVkLCBmYWxsIGJhY2sgdG9cbiAgICAvLyB1c2luZyBnaXQgbG9nIGNvbW1pdHRlcnMgYXMgdGhlIGxhc3QgcmVzb3J0XG4gICAgaWYgKHRoaXMuYWxsVXNlcnMuc2l6ZSA9PT0gMCkge1xuICAgICAgdGhpcy5hZGRVc2VycyhkYXRhLmF1dGhvcnMsIHNvdXJjZS5HSVRMT0cpO1xuICAgIH1cbiAgfVxuXG4gIGxvYWRVc2Vyc0Zyb21HcmFwaFFMKHJlbW90ZXMpIHtcbiAgICByZXR1cm4gUHJvbWlzZS5hbGwoXG4gICAgICBBcnJheS5mcm9tKHJlbW90ZXMsIHJlbW90ZSA9PiB0aGlzLmxvYWRNZW50aW9uYWJsZVVzZXJzKHJlbW90ZSkpLFxuICAgICk7XG4gIH1cblxuICBhc3luYyBnZXRUb2tlbihsb2dpbk1vZGVsLCBsb2dpbkFjY291bnQpIHtcbiAgICBpZiAoIWxvZ2luTW9kZWwpIHtcbiAgICAgIHJldHVybiBudWxsO1xuICAgIH1cbiAgICBjb25zdCB0b2tlbiA9IGF3YWl0IGxvZ2luTW9kZWwuZ2V0VG9rZW4obG9naW5BY2NvdW50KTtcbiAgICBpZiAodG9rZW4gPT09IFVOQVVUSEVOVElDQVRFRCB8fCB0b2tlbiA9PT0gSU5TVUZGSUNJRU5UIHx8IHRva2VuIGluc3RhbmNlb2YgRXJyb3IpIHtcbiAgICAgIHJldHVybiBudWxsO1xuICAgIH1cbiAgICByZXR1cm4gdG9rZW47XG4gIH1cblxuICBhc3luYyBsb2FkTWVudGlvbmFibGVVc2VycyhyZW1vdGUpIHtcbiAgICBjb25zdCBjYWNoZWQgPSB0aGlzLmNhY2hlLmdldChyZW1vdGUpO1xuICAgIGlmIChjYWNoZWQgIT09IG51bGwpIHtcbiAgICAgIHRoaXMuYWRkVXNlcnMoY2FjaGVkLCBzb3VyY2UuR0lUSFVCQVBJKTtcbiAgICAgIHJldHVybjtcbiAgICB9XG5cbiAgICBjb25zdCBlbmRwb2ludCA9IHJlbW90ZS5nZXRFbmRwb2ludCgpO1xuICAgIGNvbnN0IHRva2VuID0gYXdhaXQgdGhpcy5nZXRUb2tlbih0aGlzLmxvZ2luT2JzZXJ2ZXIuZ2V0QWN0aXZlTW9kZWwoKSwgZW5kcG9pbnQuZ2V0TG9naW5BY2NvdW50KCkpO1xuICAgIGlmICghdG9rZW4pIHtcbiAgICAgIHJldHVybjtcbiAgICB9XG5cbiAgICBjb25zdCBmZXRjaFF1ZXJ5ID0gUmVsYXlOZXR3b3JrTGF5ZXJNYW5hZ2VyLmdldEZldGNoUXVlcnkoZW5kcG9pbnQsIHRva2VuKTtcblxuICAgIGxldCBoYXNNb3JlID0gdHJ1ZTtcbiAgICBsZXQgY3Vyc29yID0gbnVsbDtcbiAgICBjb25zdCByZW1vdGVVc2VycyA9IFtdO1xuXG4gICAgd2hpbGUgKGhhc01vcmUpIHtcbiAgICAgIGNvbnN0IHJlc3BvbnNlID0gYXdhaXQgZmV0Y2hRdWVyeSh7XG4gICAgICAgIG5hbWU6ICdHZXRNZW50aW9uYWJsZVVzZXJzJyxcbiAgICAgICAgdGV4dDogYFxuICAgICAgICAgIHF1ZXJ5IEdldE1lbnRpb25hYmxlVXNlcnMoJG93bmVyOiBTdHJpbmchLCAkbmFtZTogU3RyaW5nISwgJGZpcnN0OiBJbnQhLCAkYWZ0ZXI6IFN0cmluZykge1xuICAgICAgICAgICAgcmVwb3NpdG9yeShvd25lcjogJG93bmVyLCBuYW1lOiAkbmFtZSkge1xuICAgICAgICAgICAgICBtZW50aW9uYWJsZVVzZXJzKGZpcnN0OiAkZmlyc3QsIGFmdGVyOiAkYWZ0ZXIpIHtcbiAgICAgICAgICAgICAgICBub2RlcyB7XG4gICAgICAgICAgICAgICAgICBsb2dpblxuICAgICAgICAgICAgICAgICAgZW1haWxcbiAgICAgICAgICAgICAgICAgIG5hbWVcbiAgICAgICAgICAgICAgICB9XG4gICAgICAgICAgICAgICAgcGFnZUluZm8ge1xuICAgICAgICAgICAgICAgICAgaGFzTmV4dFBhZ2VcbiAgICAgICAgICAgICAgICAgIGVuZEN1cnNvclxuICAgICAgICAgICAgICAgIH1cbiAgICAgICAgICAgICAgfVxuICAgICAgICAgICAgfVxuICAgICAgICAgIH1cbiAgICAgICAgYCxcbiAgICAgIH0sIHtcbiAgICAgICAgb3duZXI6IHJlbW90ZS5nZXRPd25lcigpLFxuICAgICAgICBuYW1lOiByZW1vdGUuZ2V0UmVwbygpLFxuICAgICAgICBmaXJzdDogMTAwLFxuICAgICAgICBhZnRlcjogY3Vyc29yLFxuICAgICAgfSk7XG5cbiAgICAgIC8qIGlzdGFuYnVsIGlnbm9yZSBpZiAqL1xuICAgICAgaWYgKHJlc3BvbnNlLmVycm9ycyAmJiByZXNwb25zZS5lcnJvcnMubGVuZ3RoID4gMSkge1xuICAgICAgICAvLyBlc2xpbnQtZGlzYWJsZS1uZXh0LWxpbmUgbm8tY29uc29sZVxuICAgICAgICBjb25zb2xlLmVycm9yKGBFcnJvciBmZXRjaGluZyBtZW50aW9uYWJsZSB1c2VyczpcXG4ke3Jlc3BvbnNlLmVycm9ycy5tYXAoZSA9PiBlLm1lc3NhZ2UpLmpvaW4oJ1xcbicpfWApO1xuICAgICAgfVxuXG4gICAgICBpZiAoIXJlc3BvbnNlLmRhdGEgfHwgIXJlc3BvbnNlLmRhdGEucmVwb3NpdG9yeSkge1xuICAgICAgICBicmVhaztcbiAgICAgIH1cblxuICAgICAgY29uc3QgY29ubmVjdGlvbiA9IHJlc3BvbnNlLmRhdGEucmVwb3NpdG9yeS5tZW50aW9uYWJsZVVzZXJzO1xuICAgICAgY29uc3QgYXV0aG9ycyA9IGNvbm5lY3Rpb24ubm9kZXMubWFwKG5vZGUgPT4ge1xuICAgICAgICBpZiAobm9kZS5lbWFpbCA9PT0gJycpIHtcbiAgICAgICAgICBub2RlLmVtYWlsID0gYCR7bm9kZS5sb2dpbn1AdXNlcnMubm9yZXBseS5naXRodWIuY29tYDtcbiAgICAgICAgfVxuXG4gICAgICAgIHJldHVybiBuZXcgQXV0aG9yKG5vZGUuZW1haWwsIG5vZGUubmFtZSwgbm9kZS5sb2dpbik7XG4gICAgICB9KTtcbiAgICAgIHRoaXMuYWRkVXNlcnMoYXV0aG9ycywgc291cmNlLkdJVEhVQkFQSSk7XG4gICAgICByZW1vdGVVc2Vycy5wdXNoKC4uLmF1dGhvcnMpO1xuXG4gICAgICBjdXJzb3IgPSBjb25uZWN0aW9uLnBhZ2VJbmZvLmVuZEN1cnNvcjtcbiAgICAgIGhhc01vcmUgPSBjb25uZWN0aW9uLnBhZ2VJbmZvLmhhc05leHRQYWdlO1xuICAgIH1cblxuICAgIHRoaXMuY2FjaGUuc2V0KHJlbW90ZSwgcmVtb3RlVXNlcnMpO1xuICB9XG5cbiAgYWRkVXNlcnModXNlcnMsIG5leHRTb3VyY2UpIHtcbiAgICBsZXQgY2hhbmdlZCA9IGZhbHNlO1xuXG4gICAgaWYgKFxuICAgICAgbmV4dFNvdXJjZSAhPT0gdGhpcy5sYXN0LnNvdXJjZSB8fFxuICAgICAgdGhpcy5yZXBvc2l0b3J5T2JzZXJ2ZXIuZ2V0QWN0aXZlTW9kZWwoKSAhPT0gdGhpcy5sYXN0LnJlcG9zaXRvcnkgfHxcbiAgICAgIHRoaXMuZXhjbHVkZWRVc2VycyAhPT0gdGhpcy5sYXN0LmV4Y2x1ZGVkVXNlcnNcbiAgICApIHtcbiAgICAgIGNoYW5nZWQgPSB0cnVlO1xuICAgICAgdGhpcy5hbGxVc2Vycy5jbGVhcigpO1xuICAgIH1cblxuICAgIGZvciAoY29uc3QgYXV0aG9yIG9mIHVzZXJzKSB7XG4gICAgICBpZiAoIXRoaXMuYWxsVXNlcnMuaGFzKGF1dGhvci5nZXRFbWFpbCgpKSkge1xuICAgICAgICBjaGFuZ2VkID0gdHJ1ZTtcbiAgICAgIH1cbiAgICAgIHRoaXMuYWxsVXNlcnMuc2V0KGF1dGhvci5nZXRFbWFpbCgpLCBhdXRob3IpO1xuICAgIH1cblxuICAgIGlmIChjaGFuZ2VkKSB7XG4gICAgICB0aGlzLmZpbmFsaXplKCk7XG4gICAgfVxuICAgIHRoaXMubGFzdC5zb3VyY2UgPSBuZXh0U291cmNlO1xuICAgIHRoaXMubGFzdC5yZXBvc2l0b3J5ID0gdGhpcy5yZXBvc2l0b3J5T2JzZXJ2ZXIuZ2V0QWN0aXZlTW9kZWwoKTtcbiAgICB0aGlzLmxhc3QuZXhjbHVkZWRVc2VycyA9IHRoaXMuZXhjbHVkZWRVc2VycztcbiAgfVxuXG4gIGZpbmFsaXplKCkge1xuICAgIC8vIFRPRE86IFtrdSAzLzIwMThdIGNvbnNpZGVyIHNvcnRpbmcgYmFzZWQgb24gbW9zdCByZWNlbnQgYXV0aG9ycyBvciBjb21taXQgZnJlcXVlbmN5XG4gICAgY29uc3QgdXNlcnMgPSBbXTtcbiAgICBmb3IgKGNvbnN0IGF1dGhvciBvZiB0aGlzLmFsbFVzZXJzLnZhbHVlcygpKSB7XG4gICAgICBpZiAoYXV0aG9yLm1hdGNoZXModGhpcy5jb21taXR0ZXIpKSB7IGNvbnRpbnVlOyB9XG4gICAgICBpZiAoYXV0aG9yLmlzTm9SZXBseSgpKSB7IGNvbnRpbnVlOyB9XG4gICAgICBpZiAodGhpcy5leGNsdWRlZFVzZXJzLmhhcyhhdXRob3IuZ2V0RW1haWwoKSkpIHsgY29udGludWU7IH1cblxuICAgICAgdXNlcnMucHVzaChhdXRob3IpO1xuICAgIH1cbiAgICB1c2Vycy5zb3J0KEF1dGhvci5jb21wYXJlKTtcbiAgICB0aGlzLnVzZXJzID0gdXNlcnM7XG4gICAgdGhpcy5kaWRVcGRhdGUoKTtcbiAgfVxuXG4gIHNldFJlcG9zaXRvcnkocmVwb3NpdG9yeSkge1xuICAgIHRoaXMucmVwb3NpdG9yeU9ic2VydmVyLnNldEFjdGl2ZU1vZGVsKHJlcG9zaXRvcnkpO1xuICB9XG5cbiAgc2V0TG9naW5Nb2RlbChsb2dpbikge1xuICAgIHRoaXMubG9naW5PYnNlcnZlci5zZXRBY3RpdmVNb2RlbChsb2dpbik7XG4gIH1cblxuICBzZXRDb21taXR0ZXIoY29tbWl0dGVyKSB7XG4gICAgY29uc3QgY2hhbmdlZCA9ICF0aGlzLmNvbW1pdHRlci5tYXRjaGVzKGNvbW1pdHRlcik7XG4gICAgdGhpcy5jb21taXR0ZXIgPSBjb21taXR0ZXI7XG4gICAgaWYgKGNoYW5nZWQpIHtcbiAgICAgIHRoaXMuZmluYWxpemUoKTtcbiAgICB9XG4gIH1cblxuICBkaWRVcGRhdGUoKSB7XG4gICAgdGhpcy5lbWl0dGVyLmVtaXQoJ2RpZC11cGRhdGUnLCB0aGlzLmdldFVzZXJzKCkpO1xuICB9XG5cbiAgb25EaWRVcGRhdGUoY2FsbGJhY2spIHtcbiAgICByZXR1cm4gdGhpcy5lbWl0dGVyLm9uKCdkaWQtdXBkYXRlJywgY2FsbGJhY2spO1xuICB9XG5cbiAgZ2V0VXNlcnMoKSB7XG4gICAgcmV0dXJuIHRoaXMudXNlcnM7XG4gIH1cbn1cbiJdfQ==