"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

var _path = _interopRequireDefault(require("path"));

var _fsExtra = _interopRequireDefault(require("fs-extra"));

var _compositeGitStrategy = _interopRequireDefault(require("../composite-git-strategy"));

var _helpers = require("../helpers");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/**
 * Locate the nearest git working directory above a given starting point, caching results.
 */
class WorkdirCache {
  constructor(maxSize = 1000) {
    this.maxSize = maxSize;
    this.known = new Map();
  }

  async find(startPath) {
    const cached = this.known.get(startPath);

    if (cached !== undefined) {
      return cached;
    }

    const workDir = await this.revParse(startPath);

    if (this.known.size >= this.maxSize) {
      this.known.clear();
    }

    this.known.set(startPath, workDir);
    return workDir;
  }

  invalidate() {
    this.known.clear();
  }

  async revParse(startPath) {
    try {
      const startDir = (await _fsExtra.default.stat(startPath)).isDirectory() ? startPath : _path.default.dirname(startPath); // Within a git worktree, return a non-empty string containing the path to the worktree root.
      // Throw if a gitdir, outside of a worktree, or startDir does not exist.

      const topLevel = await _compositeGitStrategy.default.create(startDir).exec(['rev-parse', '--show-toplevel']).catch(e => {
        if (/this operation must be run in a work tree/.test(e.stdErr)) {
          return null;
        }

        throw e;
      });

      if (topLevel !== null) {
        return (0, _helpers.toNativePathSep)(topLevel.trim());
      } // Within a gitdir, return the absolute path to the gitdir.
      // Outside of a gitdir or worktree, throw.


      const gitDir = await _compositeGitStrategy.default.create(startDir).exec(['rev-parse', '--absolute-git-dir']);
      return this.revParse(_path.default.resolve(gitDir, '..'));
    } catch (e) {
      /* istanbul ignore if */
      if (atom.config.get('github.reportCannotLocateWorkspaceError')) {
        // eslint-disable-next-line no-console
        console.error(`Unable to locate git workspace root for ${startPath}. Expected if ${startPath} is not in a git repository.`, e);
      }

      return null;
    }
  }

}

exports.default = WorkdirCache;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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