"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

var _compareSets = _interopRequireDefault(require("compare-sets"));

var _eventKit = require("event-kit");

var _workdirContext = _interopRequireDefault(require("./workdir-context"));

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _objectSpread2(target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i] != null ? arguments[i] : {}; var ownKeys = Object.keys(source); if (typeof Object.getOwnPropertySymbols === 'function') { ownKeys = ownKeys.concat(Object.getOwnPropertySymbols(source).filter(function (sym) { return Object.getOwnPropertyDescriptor(source, sym).enumerable; })); } ownKeys.forEach(function (key) { _defineProperty(target, key, source[key]); }); } return target; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

/**
 * Manage a WorkdirContext for each open directory.
 */
class WorkdirContextPool {
  /**
   * Options will be passed to each `WorkdirContext` as it is created.
   */
  constructor(options = {}) {
    this.options = options;
    this.contexts = new Map();
    this.emitter = new _eventKit.Emitter();
  }

  size() {
    return this.contexts.size;
  }
  /**
   * Access the context mapped to a known directory.
   */


  getContext(directory) {
    const {
      pipelineManager
    } = this.options;
    return this.contexts.get(directory) || _workdirContext.default.absent({
      pipelineManager
    });
  }
  /**
   * Return a WorkdirContext whose Repository has at least one remote configured to push to the named GitHub repository.
   * Returns a null context if zero or more than one contexts match.
   */


  async getMatchingContext(host, owner, repo) {
    const matches = await Promise.all(this.withResidentContexts(async (_workdir, context) => {
      const match = await context.getRepository().hasGitHubRemote(host, owner, repo);
      return match ? context : null;
    }));
    const filtered = matches.filter(Boolean);
    return filtered.length === 1 ? filtered[0] : _workdirContext.default.absent(_objectSpread2({}, this.options));
  }

  add(directory, options = {}, silenceEmitter = false) {
    if (this.contexts.has(directory)) {
      return this.getContext(directory);
    }

    const context = new _workdirContext.default(directory, _objectSpread2({}, this.options, {}, options));
    this.contexts.set(directory, context);
    const disposable = context.subs;

    const forwardEvent = (subMethod, emitEventName) => {
      const emit = () => this.emitter.emit(emitEventName, context);

      disposable.add(context[subMethod](emit));
    };

    forwardEvent('onDidStartObserver', 'did-start-observer');
    forwardEvent('onDidChangeWorkdirOrHead', 'did-change-workdir-or-head');
    forwardEvent('onDidChangeRepositoryState', 'did-change-repository-state');
    forwardEvent('onDidUpdateRepository', 'did-update-repository');
    forwardEvent('onDidDestroyRepository', 'did-destroy-repository'); // Propagate global cache invalidations across all resident contexts

    disposable.add(context.getRepository().onDidGloballyInvalidate(spec => {
      this.withResidentContexts((_workdir, eachContext) => {
        if (eachContext !== context) {
          eachContext.getRepository().acceptInvalidation(spec);
        }
      });
    }));

    if (!silenceEmitter) {
      this.emitter.emit('did-change-contexts', {
        added: new Set([directory])
      });
    }

    return context;
  }

  replace(directory, options = {}, silenceEmitter = false) {
    this.remove(directory, true);
    this.add(directory, options, true);

    if (!silenceEmitter) {
      this.emitter.emit('did-change-contexts', {
        altered: new Set([directory])
      });
    }
  }

  remove(directory, silenceEmitter = false) {
    const existing = this.contexts.get(directory);
    this.contexts.delete(directory);

    if (existing) {
      existing.destroy();

      if (!silenceEmitter) {
        this.emitter.emit('did-change-contexts', {
          removed: new Set([directory])
        });
      }
    }
  }

  set(directories, options = {}) {
    const previous = new Set(this.contexts.keys());
    const {
      added,
      removed
    } = (0, _compareSets.default)(previous, directories);

    for (const directory of added) {
      this.add(directory, options, true);
    }

    for (const directory of removed) {
      this.remove(directory, true);
    }

    if (added.size !== 0 || removed.size !== 0) {
      this.emitter.emit('did-change-contexts', {
        added,
        removed
      });
    }
  }

  getCurrentWorkDirs() {
    return this.contexts.keys();
  }

  withResidentContexts(callback) {
    const results = [];

    for (const [workdir, context] of this.contexts) {
      results.push(callback(workdir, context));
    }

    return results;
  }

  onDidStartObserver(callback) {
    return this.emitter.on('did-start-observer', callback);
  }

  onDidChangePoolContexts(callback) {
    return this.emitter.on('did-change-contexts', callback);
  }

  onDidChangeWorkdirOrHead(callback) {
    return this.emitter.on('did-change-workdir-or-head', callback);
  }

  onDidChangeRepositoryState(callback) {
    return this.emitter.on('did-change-repository-state', callback);
  }

  onDidUpdateRepository(callback) {
    return this.emitter.on('did-update-repository', callback);
  }

  onDidDestroyRepository(callback) {
    return this.emitter.on('did-destroy-repository', callback);
  }

  clear() {
    const workdirs = new Set();
    this.withResidentContexts(workdir => {
      this.remove(workdir, true);
      workdirs.add(workdir);
    });

    _workdirContext.default.destroyAbsent();

    if (workdirs.size !== 0) {
      this.emitter.emit('did-change-contexts', {
        removed: workdirs
      });
    }
  }

}

exports.default = WorkdirContextPool;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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