"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = exports.FOCUS = void 0;

var _path = _interopRequireDefault(require("path"));

var _eventKit = require("event-kit");

var _atom = require("atom");

var _eventLogger = _interopRequireDefault(require("./event-logger"));

var _helpers = require("../helpers");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

const FOCUS = Symbol('focus');
exports.FOCUS = FOCUS;

class WorkspaceChangeObserver {
  constructor(window, workspace, repository) {
    (0, _helpers.autobind)(this, 'observeTextEditor');
    this.window = window;
    this.repository = repository;
    this.workspace = workspace;
    this.observedBuffers = new WeakSet();
    this.emitter = new _eventKit.Emitter();
    this.disposables = new _eventKit.CompositeDisposable();
    this.logger = new _eventLogger.default('workspace watcher');
    this.started = false;
  }

  async start() {
    const focusHandler = event => {
      if (this.repository) {
        this.logger.showFocusEvent();
        this.didChange([{
          special: FOCUS
        }]);
      }
    };

    this.window.addEventListener('focus', focusHandler);
    this.disposables.add(this.workspace.observeTextEditors(this.observeTextEditor), new _eventKit.Disposable(() => this.window.removeEventListener('focus', focusHandler)));
    await this.watchActiveRepositoryGitDirectory();
    this.started = true;
    return this;
  }

  async destroy() {
    this.started = false;
    this.observedBuffers = new WeakSet();
    this.emitter.dispose();
    this.disposables.dispose();
    await this.stopCurrentFileWatcher();
  }

  isStarted() {
    return this.started;
  }

  didChange(payload) {
    this.emitter.emit('did-change', payload);
  }

  didChangeWorkdirOrHead() {
    this.emitter.emit('did-change-workdir-or-head');
  }

  onDidChange(callback) {
    return this.emitter.on('did-change', callback);
  }

  onDidChangeWorkdirOrHead(callback) {
    return this.emitter.on('did-change-workdir-or-head', callback);
  }

  getRepository() {
    return this.repository;
  }

  async watchActiveRepositoryGitDirectory() {
    const repository = this.getRepository();
    const gitDirectoryPath = repository.getGitDirectoryPath();
    const basenamesOfInterest = ['config', 'index', 'HEAD', 'MERGE_HEAD'];
    const workdirOrHeadBasenames = ['config', 'index'];

    const eventPaths = event => {
      const ps = [event.path];

      if (event.oldPath) {
        ps.push(event.oldPath);
      }

      return ps;
    };

    const acceptEvent = event => {
      return eventPaths(event).some(eventPath => {
        return basenamesOfInterest.includes(_path.default.basename(eventPath)) || _path.default.dirname(eventPath).includes(_path.default.join('.git', 'refs'));
      });
    };

    const isWorkdirOrHeadEvent = event => {
      return eventPaths(event).some(eventPath => workdirOrHeadBasenames.includes(_path.default.basename(eventPath)));
    };

    this.currentFileWatcher = await (0, _atom.watchPath)(gitDirectoryPath, {}, events => {
      const filteredEvents = events.filter(acceptEvent);

      if (filteredEvents.length) {
        this.logger.showEvents(filteredEvents);
        this.didChange(filteredEvents);

        if (filteredEvents.some(isWorkdirOrHeadEvent)) {
          this.logger.showWorkdirOrHeadEvents();
          this.didChangeWorkdirOrHead();
        }
      }
    });
    this.currentFileWatcher.onDidError(error => {
      const workingDirectory = repository.getWorkingDirectoryPath(); // eslint-disable-next-line no-console

      console.warn(`Error in WorkspaceChangeObserver in ${workingDirectory}:`, error);
      this.stopCurrentFileWatcher();
    });
    this.logger.showStarted(gitDirectoryPath, 'workspace emulated');
  }

  stopCurrentFileWatcher() {
    if (this.currentFileWatcher) {
      this.currentFileWatcher.dispose();
      this.currentFileWatcher = null;
      this.logger.showStopped();
    }

    return Promise.resolve();
  }

  activeRepositoryContainsPath(filePath) {
    const repository = this.getRepository();

    if (filePath && repository) {
      return filePath.indexOf(repository.getWorkingDirectoryPath()) !== -1;
    } else {
      return false;
    }
  }

  observeTextEditor(editor) {
    const buffer = editor.getBuffer();

    if (!this.observedBuffers.has(buffer)) {
      let lastPath = buffer.getPath();

      const didChange = () => {
        const currentPath = buffer.getPath();
        const events = currentPath === lastPath ? [{
          action: 'modified',
          path: currentPath
        }] : [{
          action: 'renamed',
          path: currentPath,
          oldPath: lastPath
        }];
        lastPath = currentPath;
        this.logger.showEvents(events);
        this.didChange(events);
      };

      this.observedBuffers.add(buffer);
      const disposables = new _eventKit.CompositeDisposable(buffer.onDidSave(() => {
        if (this.activeRepositoryContainsPath(buffer.getPath())) {
          didChange();
        }
      }), buffer.onDidReload(() => {
        if (this.activeRepositoryContainsPath(buffer.getPath())) {
          didChange();
        }
      }), buffer.onDidDestroy(() => {
        if (this.activeRepositoryContainsPath(buffer.getPath())) {
          didChange();
        }

        disposables.dispose();
      }));
      this.disposables.add(disposables);
    }
  }

}

exports.default = WorkspaceChangeObserver;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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