"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.expectRelayQuery = expectRelayQuery;
exports.clearRelayExpectations = clearRelayExpectations;
exports.default = void 0;

var _util = _interopRequireDefault(require("util"));

var _relayRuntime = require("relay-runtime");

var _moment = _interopRequireDefault(require("moment"));

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

const LODASH_ISEQUAL = 'lodash.isequal';
let isEqual = null;
const relayEnvironmentPerURL = new Map();
const tokenPerURL = new Map();
const fetchPerURL = new Map();
const responsesByQuery = new Map();

function logRatelimitApi(headers) {
  const remaining = headers.get('x-ratelimit-remaining');
  const total = headers.get('x-ratelimit-limit');
  const resets = headers.get('x-ratelimit-reset');

  const resetsIn = _moment.default.unix(parseInt(resets, 10)).from(); // eslint-disable-next-line no-console


  console.debug(`GitHub API Rate Limit: ${remaining}/${total} — resets ${resetsIn}`);
}

function expectRelayQuery(operationPattern, response) {
  let resolve, reject;
  const handler = typeof response === 'function' ? response : () => ({
    data: response
  });
  const promise = new Promise((resolve0, reject0) => {
    resolve = resolve0;
    reject = reject0;
  });
  const existing = responsesByQuery.get(operationPattern.name) || [];
  existing.push({
    promise,
    handler,
    variables: operationPattern.variables || {},
    trace: operationPattern.trace
  });
  responsesByQuery.set(operationPattern.name, existing);

  const disable = () => responsesByQuery.delete(operationPattern.name);

  return {
    promise,
    resolve,
    reject,
    disable
  };
}

function clearRelayExpectations() {
  responsesByQuery.clear();
  relayEnvironmentPerURL.clear();
  tokenPerURL.clear();
  fetchPerURL.clear();
  responsesByQuery.clear();
}

function createFetchQuery(url) {
  if (atom.inSpecMode()) {
    return function specFetchQuery(operation, variables, _cacheConfig, _uploadables) {
      const expectations = responsesByQuery.get(operation.name) || [];
      const match = expectations.find(expectation => {
        if (isEqual === null) {
          // Lazily require lodash.isequal so we can keep it as a dev dependency.
          // Require indirectly to trick electron-link into not following this.
          isEqual = require(LODASH_ISEQUAL);
        }

        return isEqual(expectation.variables, variables);
      });

      if (!match) {
        // eslint-disable-next-line no-console
        console.log(`GraphQL query ${operation.name} was:\n  ${operation.text.replace(/\n/g, '\n  ')}\n` + _util.default.inspect(variables));
        const e = new Error(`Unexpected GraphQL query: ${operation.name}`);
        e.rawStack = e.stack;
        throw e;
      }

      const responsePromise = match.promise.then(() => {
        return match.handler(operation);
      });

      if (match.trace) {
        // eslint-disable-next-line no-console
        console.log(`[Relay] query "${operation.name}":\n${operation.text}`);
        responsePromise.then(result => {
          // eslint-disable-next-line no-console
          console.log(`[Relay] response "${operation.name}":`, result);
        }, err => {
          // eslint-disable-next-line no-console
          console.error(`[Relay] error "${operation.name}":\n${err.stack || err}`);
          throw err;
        });
      }

      return responsePromise;
    };
  }

  return async function fetchQuery(operation, variables, _cacheConfig, _uploadables) {
    const currentToken = tokenPerURL.get(url);
    let response;

    try {
      response = await fetch(url, {
        method: 'POST',
        headers: {
          'content-type': 'application/json',
          'Authorization': `bearer ${currentToken}`,
          'Accept': 'application/vnd.github.antiope-preview+json'
        },
        body: JSON.stringify({
          query: operation.text,
          variables
        })
      });
    } catch (e) {
      // A network error was encountered. Mark it so that QueryErrorView and ErrorView can distinguish these, because
      // the errors from "fetch" are TypeErrors without much information.
      e.network = true;
      e.rawStack = e.stack;
      throw e;
    }

    try {
      atom && atom.inDevMode() && logRatelimitApi(response.headers);
    } catch (_e) {
      /* do nothing */
    }

    if (response.status !== 200) {
      const e = new Error(`GraphQL API endpoint at ${url} returned ${response.status}`);
      e.response = response;
      e.responseText = await response.text();
      e.rawStack = e.stack;
      throw e;
    }

    const payload = await response.json();

    if (payload && payload.errors && payload.errors.length > 0) {
      const e = new Error(`GraphQL API endpoint at ${url} returned an error for query ${operation.name}.`);
      e.response = response;
      e.errors = payload.errors;
      e.rawStack = e.stack;
      throw e;
    }

    return payload;
  };
}

class RelayNetworkLayerManager {
  static getEnvironmentForHost(endpoint, token) {
    const url = endpoint.getGraphQLRoot();
    let {
      environment,
      network
    } = relayEnvironmentPerURL.get(url) || {};
    tokenPerURL.set(url, token);

    if (!environment) {
      if (!token) {
        throw new Error(`You must authenticate to ${endpoint.getHost()} first.`);
      }

      const source = new _relayRuntime.RecordSource();
      const store = new _relayRuntime.Store(source);
      network = _relayRuntime.Network.create(this.getFetchQuery(endpoint, token));
      environment = new _relayRuntime.Environment({
        network,
        store
      });
      relayEnvironmentPerURL.set(url, {
        environment,
        network
      });
    }

    return environment;
  }

  static getFetchQuery(endpoint, token) {
    const url = endpoint.getGraphQLRoot();
    tokenPerURL.set(url, token);
    let fetch = fetchPerURL.get(url);

    if (!fetch) {
      fetch = createFetchQuery(url);
      fetchPerURL.set(fetch);
    }

    return fetch;
  }

}

exports.default = RelayNetworkLayerManager;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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