"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

var _react = _interopRequireDefault(require("react"));

var _propTypes = _interopRequireDefault(require("prop-types"));

var _util = require("util");

var _observeModel = _interopRequireDefault(require("./observe-model"));

var _helpers = require("../helpers");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

const sortOrders = {
  'by key': (a, b) => a.key.localeCompare(b.key),
  'oldest first': (a, b) => b.age - a.age,
  'newest first': (a, b) => a.age - b.age,
  'most hits': (a, b) => b.hits - a.hits,
  'fewest hits': (a, b) => a.hits - b.hits
};

class GitCacheView extends _react.default.Component {
  static buildURI() {
    return this.uriPattern;
  }

  constructor(props, context) {
    super(props, context);
    (0, _helpers.autobind)(this, 'fetchRepositoryData', 'fetchCacheData', 'renderCache', 'didSelectItem', 'clearCache');
    this.state = {
      order: 'by key'
    };
  }

  getURI() {
    return 'atom-github://debug/cache';
  }

  getTitle() {
    return 'GitHub Package Cache View';
  }

  serialize() {
    return null;
  }

  fetchRepositoryData(repository) {
    return repository.getCache();
  }

  fetchCacheData(cache) {
    const cached = {};
    const promises = [];
    const now = performance.now();

    for (const [key, value] of cache) {
      cached[key] = {
        hits: value.hits,
        age: now - value.createdAt
      };
      promises.push(value.promise.then(payload => (0, _util.inspect)(payload, {
        depth: 3,
        breakLength: 30
      }), err => `${err.message}\n${err.stack}`).then(resolved => {
        cached[key].value = resolved;
      }));
    }

    return Promise.all(promises).then(() => cached);
  }

  render() {
    return _react.default.createElement(_observeModel.default, {
      model: this.props.repository,
      fetchData: this.fetchRepositoryData
    }, cache => _react.default.createElement(_observeModel.default, {
      model: cache,
      fetchData: this.fetchCacheData
    }, this.renderCache));
  }

  renderCache(contents) {
    const rows = Object.keys(contents || {}).map(key => {
      return {
        key,
        age: contents[key].age,
        hits: contents[key].hits,
        content: contents[key].value
      };
    });
    rows.sort(sortOrders[this.state.order]);
    const orders = Object.keys(sortOrders);
    return _react.default.createElement("div", {
      className: "github-CacheView"
    }, _react.default.createElement("header", null, _react.default.createElement("h1", null, "Cache contents"), _react.default.createElement("p", null, _react.default.createElement("span", {
      className: "badge"
    }, rows.length), " cached items")), _react.default.createElement("main", null, _react.default.createElement("p", {
      className: "github-CacheView-Controls"
    }, _react.default.createElement("span", {
      className: "github-CacheView-Order"
    }, "order", _react.default.createElement("select", {
      className: "input-select",
      onChange: this.didSelectItem,
      value: this.state.order
    }, orders.map(order => {
      return _react.default.createElement("option", {
        key: order,
        value: order
      }, order);
    }))), _react.default.createElement("span", {
      className: "github-CacheView-Clear"
    }, _react.default.createElement("button", {
      className: "btn icon icon-trashcan",
      onClick: this.clearCache
    }, "Clear"))), _react.default.createElement("table", null, _react.default.createElement("thead", null, _react.default.createElement("tr", null, _react.default.createElement("td", {
      className: "github-CacheView-Key"
    }, "key"), _react.default.createElement("td", {
      className: "github-CacheView-Age"
    }, "age"), _react.default.createElement("td", {
      className: "github-CacheView-Hits"
    }, "hits"), _react.default.createElement("td", {
      className: "github-CacheView-Content"
    }, "content"))), _react.default.createElement("tbody", null, rows.map(row => _react.default.createElement("tr", {
      key: row.key,
      className: "github-CacheView-Row"
    }, _react.default.createElement("td", {
      className: "github-CacheView-Key"
    }, _react.default.createElement("button", {
      className: "btn",
      onClick: () => this.didClickKey(row.key)
    }, row.key)), _react.default.createElement("td", {
      className: "github-CacheView-Age"
    }, this.formatAge(row.age)), _react.default.createElement("td", {
      className: "github-CacheView-Hits"
    }, row.hits), _react.default.createElement("td", {
      className: "github-CacheView-Content"
    }, _react.default.createElement("code", null, row.content))))))));
  }

  formatAge(ageMs) {
    let remaining = ageMs;
    const parts = [];

    if (remaining > 3600000) {
      const hours = Math.floor(remaining / 3600000);
      parts.push(`${hours}h`);
      remaining -= 3600000 * hours;
    }

    if (remaining > 60000) {
      const minutes = Math.floor(remaining / 60000);
      parts.push(`${minutes}m`);
      remaining -= 60000 * minutes;
    }

    if (remaining > 1000) {
      const seconds = Math.floor(remaining / 1000);
      parts.push(`${seconds}s`);
      remaining -= 1000 * seconds;
    }

    parts.push(`${Math.floor(remaining)}ms`);
    return parts.slice(parts.length - 2).join(' ');
  }

  didSelectItem(event) {
    this.setState({
      order: event.target.value
    });
  }

  didClickKey(key) {
    const cache = this.props.repository.getCache();

    if (!cache) {
      return;
    }

    cache.removePrimary(key);
  }

  clearCache() {
    const cache = this.props.repository.getCache();

    if (!cache) {
      return;
    }

    cache.clear();
  }

}

exports.default = GitCacheView;

_defineProperty(GitCacheView, "uriPattern", 'atom-github://debug/cache');

_defineProperty(GitCacheView, "propTypes", {
  repository: _propTypes.default.object.isRequired
});
//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbImdpdC1jYWNoZS12aWV3LmpzIl0sIm5hbWVzIjpbInNvcnRPcmRlcnMiLCJhIiwiYiIsImtleSIsImxvY2FsZUNvbXBhcmUiLCJhZ2UiLCJoaXRzIiwiR2l0Q2FjaGVWaWV3IiwiUmVhY3QiLCJDb21wb25lbnQiLCJidWlsZFVSSSIsInVyaVBhdHRlcm4iLCJjb25zdHJ1Y3RvciIsInByb3BzIiwiY29udGV4dCIsInN0YXRlIiwib3JkZXIiLCJnZXRVUkkiLCJnZXRUaXRsZSIsInNlcmlhbGl6ZSIsImZldGNoUmVwb3NpdG9yeURhdGEiLCJyZXBvc2l0b3J5IiwiZ2V0Q2FjaGUiLCJmZXRjaENhY2hlRGF0YSIsImNhY2hlIiwiY2FjaGVkIiwicHJvbWlzZXMiLCJub3ciLCJwZXJmb3JtYW5jZSIsInZhbHVlIiwiY3JlYXRlZEF0IiwicHVzaCIsInByb21pc2UiLCJ0aGVuIiwicGF5bG9hZCIsImRlcHRoIiwiYnJlYWtMZW5ndGgiLCJlcnIiLCJtZXNzYWdlIiwic3RhY2siLCJyZXNvbHZlZCIsIlByb21pc2UiLCJhbGwiLCJyZW5kZXIiLCJyZW5kZXJDYWNoZSIsImNvbnRlbnRzIiwicm93cyIsIk9iamVjdCIsImtleXMiLCJtYXAiLCJjb250ZW50Iiwic29ydCIsIm9yZGVycyIsImxlbmd0aCIsImRpZFNlbGVjdEl0ZW0iLCJjbGVhckNhY2hlIiwicm93IiwiZGlkQ2xpY2tLZXkiLCJmb3JtYXRBZ2UiLCJhZ2VNcyIsInJlbWFpbmluZyIsInBhcnRzIiwiaG91cnMiLCJNYXRoIiwiZmxvb3IiLCJtaW51dGVzIiwic2Vjb25kcyIsInNsaWNlIiwiam9pbiIsImV2ZW50Iiwic2V0U3RhdGUiLCJ0YXJnZXQiLCJyZW1vdmVQcmltYXJ5IiwiY2xlYXIiLCJQcm9wVHlwZXMiLCJvYmplY3QiLCJpc1JlcXVpcmVkIl0sIm1hcHBpbmdzIjoiOzs7Ozs7O0FBQUE7O0FBQ0E7O0FBQ0E7O0FBRUE7O0FBQ0E7Ozs7OztBQUVBLE1BQU1BLFVBQVUsR0FBRztBQUNqQixZQUFVLENBQUNDLENBQUQsRUFBSUMsQ0FBSixLQUFVRCxDQUFDLENBQUNFLEdBQUYsQ0FBTUMsYUFBTixDQUFvQkYsQ0FBQyxDQUFDQyxHQUF0QixDQURIO0FBRWpCLGtCQUFnQixDQUFDRixDQUFELEVBQUlDLENBQUosS0FBVUEsQ0FBQyxDQUFDRyxHQUFGLEdBQVFKLENBQUMsQ0FBQ0ksR0FGbkI7QUFHakIsa0JBQWdCLENBQUNKLENBQUQsRUFBSUMsQ0FBSixLQUFVRCxDQUFDLENBQUNJLEdBQUYsR0FBUUgsQ0FBQyxDQUFDRyxHQUhuQjtBQUlqQixlQUFhLENBQUNKLENBQUQsRUFBSUMsQ0FBSixLQUFVQSxDQUFDLENBQUNJLElBQUYsR0FBU0wsQ0FBQyxDQUFDSyxJQUpqQjtBQUtqQixpQkFBZSxDQUFDTCxDQUFELEVBQUlDLENBQUosS0FBVUQsQ0FBQyxDQUFDSyxJQUFGLEdBQVNKLENBQUMsQ0FBQ0k7QUFMbkIsQ0FBbkI7O0FBUWUsTUFBTUMsWUFBTixTQUEyQkMsZUFBTUMsU0FBakMsQ0FBMkM7QUFHeEQsU0FBT0MsUUFBUCxHQUFrQjtBQUNoQixXQUFPLEtBQUtDLFVBQVo7QUFDRDs7QUFNREMsRUFBQUEsV0FBVyxDQUFDQyxLQUFELEVBQVFDLE9BQVIsRUFBaUI7QUFDMUIsVUFBTUQsS0FBTixFQUFhQyxPQUFiO0FBQ0EsMkJBQVMsSUFBVCxFQUFlLHFCQUFmLEVBQXNDLGdCQUF0QyxFQUF3RCxhQUF4RCxFQUF1RSxlQUF2RSxFQUF3RixZQUF4RjtBQUVBLFNBQUtDLEtBQUwsR0FBYTtBQUNYQyxNQUFBQSxLQUFLLEVBQUU7QUFESSxLQUFiO0FBR0Q7O0FBRURDLEVBQUFBLE1BQU0sR0FBRztBQUNQLFdBQU8sMkJBQVA7QUFDRDs7QUFFREMsRUFBQUEsUUFBUSxHQUFHO0FBQ1QsV0FBTywyQkFBUDtBQUNEOztBQUVEQyxFQUFBQSxTQUFTLEdBQUc7QUFDVixXQUFPLElBQVA7QUFDRDs7QUFFREMsRUFBQUEsbUJBQW1CLENBQUNDLFVBQUQsRUFBYTtBQUM5QixXQUFPQSxVQUFVLENBQUNDLFFBQVgsRUFBUDtBQUNEOztBQUVEQyxFQUFBQSxjQUFjLENBQUNDLEtBQUQsRUFBUTtBQUNwQixVQUFNQyxNQUFNLEdBQUcsRUFBZjtBQUNBLFVBQU1DLFFBQVEsR0FBRyxFQUFqQjtBQUNBLFVBQU1DLEdBQUcsR0FBR0MsV0FBVyxDQUFDRCxHQUFaLEVBQVo7O0FBRUEsU0FBSyxNQUFNLENBQUN4QixHQUFELEVBQU0wQixLQUFOLENBQVgsSUFBMkJMLEtBQTNCLEVBQWtDO0FBQ2hDQyxNQUFBQSxNQUFNLENBQUN0QixHQUFELENBQU4sR0FBYztBQUNaRyxRQUFBQSxJQUFJLEVBQUV1QixLQUFLLENBQUN2QixJQURBO0FBRVpELFFBQUFBLEdBQUcsRUFBRXNCLEdBQUcsR0FBR0UsS0FBSyxDQUFDQztBQUZMLE9BQWQ7QUFLQUosTUFBQUEsUUFBUSxDQUFDSyxJQUFULENBQ0VGLEtBQUssQ0FBQ0csT0FBTixDQUNHQyxJQURILENBRUlDLE9BQU8sSUFBSSxtQkFBUUEsT0FBUixFQUFpQjtBQUFDQyxRQUFBQSxLQUFLLEVBQUUsQ0FBUjtBQUFXQyxRQUFBQSxXQUFXLEVBQUU7QUFBeEIsT0FBakIsQ0FGZixFQUdJQyxHQUFHLElBQUssR0FBRUEsR0FBRyxDQUFDQyxPQUFRLEtBQUlELEdBQUcsQ0FBQ0UsS0FBTSxFQUh4QyxFQUtHTixJQUxILENBS1FPLFFBQVEsSUFBSTtBQUFFZixRQUFBQSxNQUFNLENBQUN0QixHQUFELENBQU4sQ0FBWTBCLEtBQVosR0FBb0JXLFFBQXBCO0FBQStCLE9BTHJELENBREY7QUFRRDs7QUFFRCxXQUFPQyxPQUFPLENBQUNDLEdBQVIsQ0FBWWhCLFFBQVosRUFBc0JPLElBQXRCLENBQTJCLE1BQU1SLE1BQWpDLENBQVA7QUFDRDs7QUFFRGtCLEVBQUFBLE1BQU0sR0FBRztBQUNQLFdBQ0UsNkJBQUMscUJBQUQ7QUFBYyxNQUFBLEtBQUssRUFBRSxLQUFLOUIsS0FBTCxDQUFXUSxVQUFoQztBQUE0QyxNQUFBLFNBQVMsRUFBRSxLQUFLRDtBQUE1RCxPQUNHSSxLQUFLLElBQ0osNkJBQUMscUJBQUQ7QUFBYyxNQUFBLEtBQUssRUFBRUEsS0FBckI7QUFBNEIsTUFBQSxTQUFTLEVBQUUsS0FBS0Q7QUFBNUMsT0FDRyxLQUFLcUIsV0FEUixDQUZKLENBREY7QUFTRDs7QUFFREEsRUFBQUEsV0FBVyxDQUFDQyxRQUFELEVBQVc7QUFDcEIsVUFBTUMsSUFBSSxHQUFHQyxNQUFNLENBQUNDLElBQVAsQ0FBWUgsUUFBUSxJQUFJLEVBQXhCLEVBQTRCSSxHQUE1QixDQUFnQzlDLEdBQUcsSUFBSTtBQUNsRCxhQUFPO0FBQ0xBLFFBQUFBLEdBREs7QUFFTEUsUUFBQUEsR0FBRyxFQUFFd0MsUUFBUSxDQUFDMUMsR0FBRCxDQUFSLENBQWNFLEdBRmQ7QUFHTEMsUUFBQUEsSUFBSSxFQUFFdUMsUUFBUSxDQUFDMUMsR0FBRCxDQUFSLENBQWNHLElBSGY7QUFJTDRDLFFBQUFBLE9BQU8sRUFBRUwsUUFBUSxDQUFDMUMsR0FBRCxDQUFSLENBQWMwQjtBQUpsQixPQUFQO0FBTUQsS0FQWSxDQUFiO0FBU0FpQixJQUFBQSxJQUFJLENBQUNLLElBQUwsQ0FBVW5ELFVBQVUsQ0FBQyxLQUFLZSxLQUFMLENBQVdDLEtBQVosQ0FBcEI7QUFFQSxVQUFNb0MsTUFBTSxHQUFHTCxNQUFNLENBQUNDLElBQVAsQ0FBWWhELFVBQVosQ0FBZjtBQUVBLFdBQ0U7QUFBSyxNQUFBLFNBQVMsRUFBQztBQUFmLE9BQ0UsNkNBQ0UsMERBREYsRUFFRSx3Q0FDRTtBQUFNLE1BQUEsU0FBUyxFQUFDO0FBQWhCLE9BQXlCOEMsSUFBSSxDQUFDTyxNQUE5QixDQURGLGtCQUZGLENBREYsRUFPRSwyQ0FDRTtBQUFHLE1BQUEsU0FBUyxFQUFDO0FBQWIsT0FDRTtBQUFNLE1BQUEsU0FBUyxFQUFDO0FBQWhCLGdCQUVFO0FBQ0UsTUFBQSxTQUFTLEVBQUMsY0FEWjtBQUVFLE1BQUEsUUFBUSxFQUFFLEtBQUtDLGFBRmpCO0FBR0UsTUFBQSxLQUFLLEVBQUUsS0FBS3ZDLEtBQUwsQ0FBV0M7QUFIcEIsT0FJR29DLE1BQU0sQ0FBQ0gsR0FBUCxDQUFXakMsS0FBSyxJQUFJO0FBQ25CLGFBQU87QUFBUSxRQUFBLEdBQUcsRUFBRUEsS0FBYjtBQUFvQixRQUFBLEtBQUssRUFBRUE7QUFBM0IsU0FBbUNBLEtBQW5DLENBQVA7QUFDRCxLQUZBLENBSkgsQ0FGRixDQURGLEVBWUU7QUFBTSxNQUFBLFNBQVMsRUFBQztBQUFoQixPQUNFO0FBQVEsTUFBQSxTQUFTLEVBQUMsd0JBQWxCO0FBQTJDLE1BQUEsT0FBTyxFQUFFLEtBQUt1QztBQUF6RCxlQURGLENBWkYsQ0FERixFQW1CRSw0Q0FDRSw0Q0FDRSx5Q0FDRTtBQUFJLE1BQUEsU0FBUyxFQUFDO0FBQWQsYUFERixFQUVFO0FBQUksTUFBQSxTQUFTLEVBQUM7QUFBZCxhQUZGLEVBR0U7QUFBSSxNQUFBLFNBQVMsRUFBQztBQUFkLGNBSEYsRUFJRTtBQUFJLE1BQUEsU0FBUyxFQUFDO0FBQWQsaUJBSkYsQ0FERixDQURGLEVBU0UsNENBQ0dULElBQUksQ0FBQ0csR0FBTCxDQUFTTyxHQUFHLElBQ1g7QUFBSSxNQUFBLEdBQUcsRUFBRUEsR0FBRyxDQUFDckQsR0FBYjtBQUFrQixNQUFBLFNBQVMsRUFBQztBQUE1QixPQUNFO0FBQUksTUFBQSxTQUFTLEVBQUM7QUFBZCxPQUNFO0FBQVEsTUFBQSxTQUFTLEVBQUMsS0FBbEI7QUFBd0IsTUFBQSxPQUFPLEVBQUUsTUFBTSxLQUFLc0QsV0FBTCxDQUFpQkQsR0FBRyxDQUFDckQsR0FBckI7QUFBdkMsT0FDR3FELEdBQUcsQ0FBQ3JELEdBRFAsQ0FERixDQURGLEVBTUU7QUFBSSxNQUFBLFNBQVMsRUFBQztBQUFkLE9BQ0csS0FBS3VELFNBQUwsQ0FBZUYsR0FBRyxDQUFDbkQsR0FBbkIsQ0FESCxDQU5GLEVBU0U7QUFBSSxNQUFBLFNBQVMsRUFBQztBQUFkLE9BQ0dtRCxHQUFHLENBQUNsRCxJQURQLENBVEYsRUFZRTtBQUFJLE1BQUEsU0FBUyxFQUFDO0FBQWQsT0FDRSwyQ0FBT2tELEdBQUcsQ0FBQ04sT0FBWCxDQURGLENBWkYsQ0FERCxDQURILENBVEYsQ0FuQkYsQ0FQRixDQURGO0FBNEREOztBQUVEUSxFQUFBQSxTQUFTLENBQUNDLEtBQUQsRUFBUTtBQUNmLFFBQUlDLFNBQVMsR0FBR0QsS0FBaEI7QUFDQSxVQUFNRSxLQUFLLEdBQUcsRUFBZDs7QUFFQSxRQUFJRCxTQUFTLEdBQUcsT0FBaEIsRUFBeUI7QUFDdkIsWUFBTUUsS0FBSyxHQUFHQyxJQUFJLENBQUNDLEtBQUwsQ0FBV0osU0FBUyxHQUFHLE9BQXZCLENBQWQ7QUFDQUMsTUFBQUEsS0FBSyxDQUFDOUIsSUFBTixDQUFZLEdBQUUrQixLQUFNLEdBQXBCO0FBQ0FGLE1BQUFBLFNBQVMsSUFBSyxVQUFVRSxLQUF4QjtBQUNEOztBQUVELFFBQUlGLFNBQVMsR0FBRyxLQUFoQixFQUF1QjtBQUNyQixZQUFNSyxPQUFPLEdBQUdGLElBQUksQ0FBQ0MsS0FBTCxDQUFXSixTQUFTLEdBQUcsS0FBdkIsQ0FBaEI7QUFDQUMsTUFBQUEsS0FBSyxDQUFDOUIsSUFBTixDQUFZLEdBQUVrQyxPQUFRLEdBQXRCO0FBQ0FMLE1BQUFBLFNBQVMsSUFBSyxRQUFRSyxPQUF0QjtBQUNEOztBQUVELFFBQUlMLFNBQVMsR0FBRyxJQUFoQixFQUFzQjtBQUNwQixZQUFNTSxPQUFPLEdBQUdILElBQUksQ0FBQ0MsS0FBTCxDQUFXSixTQUFTLEdBQUcsSUFBdkIsQ0FBaEI7QUFDQUMsTUFBQUEsS0FBSyxDQUFDOUIsSUFBTixDQUFZLEdBQUVtQyxPQUFRLEdBQXRCO0FBQ0FOLE1BQUFBLFNBQVMsSUFBSyxPQUFPTSxPQUFyQjtBQUNEOztBQUVETCxJQUFBQSxLQUFLLENBQUM5QixJQUFOLENBQVksR0FBRWdDLElBQUksQ0FBQ0MsS0FBTCxDQUFXSixTQUFYLENBQXNCLElBQXBDO0FBRUEsV0FBT0MsS0FBSyxDQUFDTSxLQUFOLENBQVlOLEtBQUssQ0FBQ1IsTUFBTixHQUFlLENBQTNCLEVBQThCZSxJQUE5QixDQUFtQyxHQUFuQyxDQUFQO0FBQ0Q7O0FBRURkLEVBQUFBLGFBQWEsQ0FBQ2UsS0FBRCxFQUFRO0FBQ25CLFNBQUtDLFFBQUwsQ0FBYztBQUFDdEQsTUFBQUEsS0FBSyxFQUFFcUQsS0FBSyxDQUFDRSxNQUFOLENBQWExQztBQUFyQixLQUFkO0FBQ0Q7O0FBRUQ0QixFQUFBQSxXQUFXLENBQUN0RCxHQUFELEVBQU07QUFDZixVQUFNcUIsS0FBSyxHQUFHLEtBQUtYLEtBQUwsQ0FBV1EsVUFBWCxDQUFzQkMsUUFBdEIsRUFBZDs7QUFDQSxRQUFJLENBQUNFLEtBQUwsRUFBWTtBQUNWO0FBQ0Q7O0FBRURBLElBQUFBLEtBQUssQ0FBQ2dELGFBQU4sQ0FBb0JyRSxHQUFwQjtBQUNEOztBQUVEb0QsRUFBQUEsVUFBVSxHQUFHO0FBQ1gsVUFBTS9CLEtBQUssR0FBRyxLQUFLWCxLQUFMLENBQVdRLFVBQVgsQ0FBc0JDLFFBQXRCLEVBQWQ7O0FBQ0EsUUFBSSxDQUFDRSxLQUFMLEVBQVk7QUFDVjtBQUNEOztBQUVEQSxJQUFBQSxLQUFLLENBQUNpRCxLQUFOO0FBQ0Q7O0FBbk11RDs7OztnQkFBckNsRSxZLGdCQUNDLDJCOztnQkFEREEsWSxlQU9BO0FBQ2pCYyxFQUFBQSxVQUFVLEVBQUVxRCxtQkFBVUMsTUFBVixDQUFpQkM7QUFEWixDIiwic291cmNlUm9vdCI6Ii9idWlsZC9hdG9tL3NyYy9hdG9tL291dC9hcHAvbm9kZV9tb2R1bGVzL2dpdGh1YiIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCBSZWFjdCBmcm9tICdyZWFjdCc7XG5pbXBvcnQgUHJvcFR5cGVzIGZyb20gJ3Byb3AtdHlwZXMnO1xuaW1wb3J0IHtpbnNwZWN0fSBmcm9tICd1dGlsJztcblxuaW1wb3J0IE9ic2VydmVNb2RlbCBmcm9tICcuL29ic2VydmUtbW9kZWwnO1xuaW1wb3J0IHthdXRvYmluZH0gZnJvbSAnLi4vaGVscGVycyc7XG5cbmNvbnN0IHNvcnRPcmRlcnMgPSB7XG4gICdieSBrZXknOiAoYSwgYikgPT4gYS5rZXkubG9jYWxlQ29tcGFyZShiLmtleSksXG4gICdvbGRlc3QgZmlyc3QnOiAoYSwgYikgPT4gYi5hZ2UgLSBhLmFnZSxcbiAgJ25ld2VzdCBmaXJzdCc6IChhLCBiKSA9PiBhLmFnZSAtIGIuYWdlLFxuICAnbW9zdCBoaXRzJzogKGEsIGIpID0+IGIuaGl0cyAtIGEuaGl0cyxcbiAgJ2Zld2VzdCBoaXRzJzogKGEsIGIpID0+IGEuaGl0cyAtIGIuaGl0cyxcbn07XG5cbmV4cG9ydCBkZWZhdWx0IGNsYXNzIEdpdENhY2hlVmlldyBleHRlbmRzIFJlYWN0LkNvbXBvbmVudCB7XG4gIHN0YXRpYyB1cmlQYXR0ZXJuID0gJ2F0b20tZ2l0aHViOi8vZGVidWcvY2FjaGUnXG5cbiAgc3RhdGljIGJ1aWxkVVJJKCkge1xuICAgIHJldHVybiB0aGlzLnVyaVBhdHRlcm47XG4gIH1cblxuICBzdGF0aWMgcHJvcFR5cGVzID0ge1xuICAgIHJlcG9zaXRvcnk6IFByb3BUeXBlcy5vYmplY3QuaXNSZXF1aXJlZCxcbiAgfVxuXG4gIGNvbnN0cnVjdG9yKHByb3BzLCBjb250ZXh0KSB7XG4gICAgc3VwZXIocHJvcHMsIGNvbnRleHQpO1xuICAgIGF1dG9iaW5kKHRoaXMsICdmZXRjaFJlcG9zaXRvcnlEYXRhJywgJ2ZldGNoQ2FjaGVEYXRhJywgJ3JlbmRlckNhY2hlJywgJ2RpZFNlbGVjdEl0ZW0nLCAnY2xlYXJDYWNoZScpO1xuXG4gICAgdGhpcy5zdGF0ZSA9IHtcbiAgICAgIG9yZGVyOiAnYnkga2V5JyxcbiAgICB9O1xuICB9XG5cbiAgZ2V0VVJJKCkge1xuICAgIHJldHVybiAnYXRvbS1naXRodWI6Ly9kZWJ1Zy9jYWNoZSc7XG4gIH1cblxuICBnZXRUaXRsZSgpIHtcbiAgICByZXR1cm4gJ0dpdEh1YiBQYWNrYWdlIENhY2hlIFZpZXcnO1xuICB9XG5cbiAgc2VyaWFsaXplKCkge1xuICAgIHJldHVybiBudWxsO1xuICB9XG5cbiAgZmV0Y2hSZXBvc2l0b3J5RGF0YShyZXBvc2l0b3J5KSB7XG4gICAgcmV0dXJuIHJlcG9zaXRvcnkuZ2V0Q2FjaGUoKTtcbiAgfVxuXG4gIGZldGNoQ2FjaGVEYXRhKGNhY2hlKSB7XG4gICAgY29uc3QgY2FjaGVkID0ge307XG4gICAgY29uc3QgcHJvbWlzZXMgPSBbXTtcbiAgICBjb25zdCBub3cgPSBwZXJmb3JtYW5jZS5ub3coKTtcblxuICAgIGZvciAoY29uc3QgW2tleSwgdmFsdWVdIG9mIGNhY2hlKSB7XG4gICAgICBjYWNoZWRba2V5XSA9IHtcbiAgICAgICAgaGl0czogdmFsdWUuaGl0cyxcbiAgICAgICAgYWdlOiBub3cgLSB2YWx1ZS5jcmVhdGVkQXQsXG4gICAgICB9O1xuXG4gICAgICBwcm9taXNlcy5wdXNoKFxuICAgICAgICB2YWx1ZS5wcm9taXNlXG4gICAgICAgICAgLnRoZW4oXG4gICAgICAgICAgICBwYXlsb2FkID0+IGluc3BlY3QocGF5bG9hZCwge2RlcHRoOiAzLCBicmVha0xlbmd0aDogMzB9KSxcbiAgICAgICAgICAgIGVyciA9PiBgJHtlcnIubWVzc2FnZX1cXG4ke2Vyci5zdGFja31gLFxuICAgICAgICAgIClcbiAgICAgICAgICAudGhlbihyZXNvbHZlZCA9PiB7IGNhY2hlZFtrZXldLnZhbHVlID0gcmVzb2x2ZWQ7IH0pLFxuICAgICAgKTtcbiAgICB9XG5cbiAgICByZXR1cm4gUHJvbWlzZS5hbGwocHJvbWlzZXMpLnRoZW4oKCkgPT4gY2FjaGVkKTtcbiAgfVxuXG4gIHJlbmRlcigpIHtcbiAgICByZXR1cm4gKFxuICAgICAgPE9ic2VydmVNb2RlbCBtb2RlbD17dGhpcy5wcm9wcy5yZXBvc2l0b3J5fSBmZXRjaERhdGE9e3RoaXMuZmV0Y2hSZXBvc2l0b3J5RGF0YX0+XG4gICAgICAgIHtjYWNoZSA9PiAoXG4gICAgICAgICAgPE9ic2VydmVNb2RlbCBtb2RlbD17Y2FjaGV9IGZldGNoRGF0YT17dGhpcy5mZXRjaENhY2hlRGF0YX0+XG4gICAgICAgICAgICB7dGhpcy5yZW5kZXJDYWNoZX1cbiAgICAgICAgICA8L09ic2VydmVNb2RlbD5cbiAgICAgICAgKX1cbiAgICAgIDwvT2JzZXJ2ZU1vZGVsPlxuICAgICk7XG4gIH1cblxuICByZW5kZXJDYWNoZShjb250ZW50cykge1xuICAgIGNvbnN0IHJvd3MgPSBPYmplY3Qua2V5cyhjb250ZW50cyB8fCB7fSkubWFwKGtleSA9PiB7XG4gICAgICByZXR1cm4ge1xuICAgICAgICBrZXksXG4gICAgICAgIGFnZTogY29udGVudHNba2V5XS5hZ2UsXG4gICAgICAgIGhpdHM6IGNvbnRlbnRzW2tleV0uaGl0cyxcbiAgICAgICAgY29udGVudDogY29udGVudHNba2V5XS52YWx1ZSxcbiAgICAgIH07XG4gICAgfSk7XG5cbiAgICByb3dzLnNvcnQoc29ydE9yZGVyc1t0aGlzLnN0YXRlLm9yZGVyXSk7XG5cbiAgICBjb25zdCBvcmRlcnMgPSBPYmplY3Qua2V5cyhzb3J0T3JkZXJzKTtcblxuICAgIHJldHVybiAoXG4gICAgICA8ZGl2IGNsYXNzTmFtZT1cImdpdGh1Yi1DYWNoZVZpZXdcIj5cbiAgICAgICAgPGhlYWRlcj5cbiAgICAgICAgICA8aDE+Q2FjaGUgY29udGVudHM8L2gxPlxuICAgICAgICAgIDxwPlxuICAgICAgICAgICAgPHNwYW4gY2xhc3NOYW1lPVwiYmFkZ2VcIj57cm93cy5sZW5ndGh9PC9zcGFuPiBjYWNoZWQgaXRlbXNcbiAgICAgICAgICA8L3A+XG4gICAgICAgIDwvaGVhZGVyPlxuICAgICAgICA8bWFpbj5cbiAgICAgICAgICA8cCBjbGFzc05hbWU9XCJnaXRodWItQ2FjaGVWaWV3LUNvbnRyb2xzXCI+XG4gICAgICAgICAgICA8c3BhbiBjbGFzc05hbWU9XCJnaXRodWItQ2FjaGVWaWV3LU9yZGVyXCI+XG4gICAgICAgICAgICAgIG9yZGVyXG4gICAgICAgICAgICAgIDxzZWxlY3RcbiAgICAgICAgICAgICAgICBjbGFzc05hbWU9XCJpbnB1dC1zZWxlY3RcIlxuICAgICAgICAgICAgICAgIG9uQ2hhbmdlPXt0aGlzLmRpZFNlbGVjdEl0ZW19XG4gICAgICAgICAgICAgICAgdmFsdWU9e3RoaXMuc3RhdGUub3JkZXJ9PlxuICAgICAgICAgICAgICAgIHtvcmRlcnMubWFwKG9yZGVyID0+IHtcbiAgICAgICAgICAgICAgICAgIHJldHVybiA8b3B0aW9uIGtleT17b3JkZXJ9IHZhbHVlPXtvcmRlcn0+e29yZGVyfTwvb3B0aW9uPjtcbiAgICAgICAgICAgICAgICB9KX1cbiAgICAgICAgICAgICAgPC9zZWxlY3Q+XG4gICAgICAgICAgICA8L3NwYW4+XG4gICAgICAgICAgICA8c3BhbiBjbGFzc05hbWU9XCJnaXRodWItQ2FjaGVWaWV3LUNsZWFyXCI+XG4gICAgICAgICAgICAgIDxidXR0b24gY2xhc3NOYW1lPVwiYnRuIGljb24gaWNvbi10cmFzaGNhblwiIG9uQ2xpY2s9e3RoaXMuY2xlYXJDYWNoZX0+XG4gICAgICAgICAgICAgICAgQ2xlYXJcbiAgICAgICAgICAgICAgPC9idXR0b24+XG4gICAgICAgICAgICA8L3NwYW4+XG4gICAgICAgICAgPC9wPlxuICAgICAgICAgIDx0YWJsZT5cbiAgICAgICAgICAgIDx0aGVhZD5cbiAgICAgICAgICAgICAgPHRyPlxuICAgICAgICAgICAgICAgIDx0ZCBjbGFzc05hbWU9XCJnaXRodWItQ2FjaGVWaWV3LUtleVwiPmtleTwvdGQ+XG4gICAgICAgICAgICAgICAgPHRkIGNsYXNzTmFtZT1cImdpdGh1Yi1DYWNoZVZpZXctQWdlXCI+YWdlPC90ZD5cbiAgICAgICAgICAgICAgICA8dGQgY2xhc3NOYW1lPVwiZ2l0aHViLUNhY2hlVmlldy1IaXRzXCI+aGl0czwvdGQ+XG4gICAgICAgICAgICAgICAgPHRkIGNsYXNzTmFtZT1cImdpdGh1Yi1DYWNoZVZpZXctQ29udGVudFwiPmNvbnRlbnQ8L3RkPlxuICAgICAgICAgICAgICA8L3RyPlxuICAgICAgICAgICAgPC90aGVhZD5cbiAgICAgICAgICAgIDx0Ym9keT5cbiAgICAgICAgICAgICAge3Jvd3MubWFwKHJvdyA9PiAoXG4gICAgICAgICAgICAgICAgPHRyIGtleT17cm93LmtleX0gY2xhc3NOYW1lPVwiZ2l0aHViLUNhY2hlVmlldy1Sb3dcIj5cbiAgICAgICAgICAgICAgICAgIDx0ZCBjbGFzc05hbWU9XCJnaXRodWItQ2FjaGVWaWV3LUtleVwiPlxuICAgICAgICAgICAgICAgICAgICA8YnV0dG9uIGNsYXNzTmFtZT1cImJ0blwiIG9uQ2xpY2s9eygpID0+IHRoaXMuZGlkQ2xpY2tLZXkocm93LmtleSl9PlxuICAgICAgICAgICAgICAgICAgICAgIHtyb3cua2V5fVxuICAgICAgICAgICAgICAgICAgICA8L2J1dHRvbj5cbiAgICAgICAgICAgICAgICAgIDwvdGQ+XG4gICAgICAgICAgICAgICAgICA8dGQgY2xhc3NOYW1lPVwiZ2l0aHViLUNhY2hlVmlldy1BZ2VcIj5cbiAgICAgICAgICAgICAgICAgICAge3RoaXMuZm9ybWF0QWdlKHJvdy5hZ2UpfVxuICAgICAgICAgICAgICAgICAgPC90ZD5cbiAgICAgICAgICAgICAgICAgIDx0ZCBjbGFzc05hbWU9XCJnaXRodWItQ2FjaGVWaWV3LUhpdHNcIj5cbiAgICAgICAgICAgICAgICAgICAge3Jvdy5oaXRzfVxuICAgICAgICAgICAgICAgICAgPC90ZD5cbiAgICAgICAgICAgICAgICAgIDx0ZCBjbGFzc05hbWU9XCJnaXRodWItQ2FjaGVWaWV3LUNvbnRlbnRcIj5cbiAgICAgICAgICAgICAgICAgICAgPGNvZGU+e3Jvdy5jb250ZW50fTwvY29kZT5cbiAgICAgICAgICAgICAgICAgIDwvdGQ+XG4gICAgICAgICAgICAgICAgPC90cj5cbiAgICAgICAgICAgICAgKSl9XG4gICAgICAgICAgICA8L3Rib2R5PlxuICAgICAgICAgIDwvdGFibGU+XG4gICAgICAgIDwvbWFpbj5cbiAgICAgIDwvZGl2PlxuICAgICk7XG4gIH1cblxuICBmb3JtYXRBZ2UoYWdlTXMpIHtcbiAgICBsZXQgcmVtYWluaW5nID0gYWdlTXM7XG4gICAgY29uc3QgcGFydHMgPSBbXTtcblxuICAgIGlmIChyZW1haW5pbmcgPiAzNjAwMDAwKSB7XG4gICAgICBjb25zdCBob3VycyA9IE1hdGguZmxvb3IocmVtYWluaW5nIC8gMzYwMDAwMCk7XG4gICAgICBwYXJ0cy5wdXNoKGAke2hvdXJzfWhgKTtcbiAgICAgIHJlbWFpbmluZyAtPSAoMzYwMDAwMCAqIGhvdXJzKTtcbiAgICB9XG5cbiAgICBpZiAocmVtYWluaW5nID4gNjAwMDApIHtcbiAgICAgIGNvbnN0IG1pbnV0ZXMgPSBNYXRoLmZsb29yKHJlbWFpbmluZyAvIDYwMDAwKTtcbiAgICAgIHBhcnRzLnB1c2goYCR7bWludXRlc31tYCk7XG4gICAgICByZW1haW5pbmcgLT0gKDYwMDAwICogbWludXRlcyk7XG4gICAgfVxuXG4gICAgaWYgKHJlbWFpbmluZyA+IDEwMDApIHtcbiAgICAgIGNvbnN0IHNlY29uZHMgPSBNYXRoLmZsb29yKHJlbWFpbmluZyAvIDEwMDApO1xuICAgICAgcGFydHMucHVzaChgJHtzZWNvbmRzfXNgKTtcbiAgICAgIHJlbWFpbmluZyAtPSAoMTAwMCAqIHNlY29uZHMpO1xuICAgIH1cblxuICAgIHBhcnRzLnB1c2goYCR7TWF0aC5mbG9vcihyZW1haW5pbmcpfW1zYCk7XG5cbiAgICByZXR1cm4gcGFydHMuc2xpY2UocGFydHMubGVuZ3RoIC0gMikuam9pbignICcpO1xuICB9XG5cbiAgZGlkU2VsZWN0SXRlbShldmVudCkge1xuICAgIHRoaXMuc2V0U3RhdGUoe29yZGVyOiBldmVudC50YXJnZXQudmFsdWV9KTtcbiAgfVxuXG4gIGRpZENsaWNrS2V5KGtleSkge1xuICAgIGNvbnN0IGNhY2hlID0gdGhpcy5wcm9wcy5yZXBvc2l0b3J5LmdldENhY2hlKCk7XG4gICAgaWYgKCFjYWNoZSkge1xuICAgICAgcmV0dXJuO1xuICAgIH1cblxuICAgIGNhY2hlLnJlbW92ZVByaW1hcnkoa2V5KTtcbiAgfVxuXG4gIGNsZWFyQ2FjaGUoKSB7XG4gICAgY29uc3QgY2FjaGUgPSB0aGlzLnByb3BzLnJlcG9zaXRvcnkuZ2V0Q2FjaGUoKTtcbiAgICBpZiAoIWNhY2hlKSB7XG4gICAgICByZXR1cm47XG4gICAgfVxuXG4gICAgY2FjaGUuY2xlYXIoKTtcbiAgfVxufVxuIl19