"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = IssueishLink;
exports.handleClickEvent = handleClickEvent;
exports.openIssueishLinkInNewTab = openIssueishLinkInNewTab;
exports.openLinkInBrowser = openLinkInBrowser;
exports.getDataFromGithubUrl = getDataFromGithubUrl;

var _url = _interopRequireDefault(require("url"));

var _electron = require("electron");

var _react = _interopRequireDefault(require("react"));

var _propTypes = _interopRequireDefault(require("prop-types"));

var _issueishDetailItem = _interopRequireDefault(require("../items/issueish-detail-item"));

var _reporterProxy = require("../reporter-proxy");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _extends() { _extends = Object.assign || function (target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i]; for (var key in source) { if (Object.prototype.hasOwnProperty.call(source, key)) { target[key] = source[key]; } } } return target; }; return _extends.apply(this, arguments); }

function _objectWithoutProperties(source, excluded) { if (source == null) return {}; var target = _objectWithoutPropertiesLoose(source, excluded); var key, i; if (Object.getOwnPropertySymbols) { var sourceSymbolKeys = Object.getOwnPropertySymbols(source); for (i = 0; i < sourceSymbolKeys.length; i++) { key = sourceSymbolKeys[i]; if (excluded.indexOf(key) >= 0) continue; if (!Object.prototype.propertyIsEnumerable.call(source, key)) continue; target[key] = source[key]; } } return target; }

function _objectWithoutPropertiesLoose(source, excluded) { if (source == null) return {}; var target = {}; var sourceKeys = Object.keys(source); var key, i; for (i = 0; i < sourceKeys.length; i++) { key = sourceKeys[i]; if (excluded.indexOf(key) >= 0) continue; target[key] = source[key]; } return target; }

// eslint-disable-next-line no-shadow
function IssueishLink(_ref) {
  let {
    url,
    children
  } = _ref,
      others = _objectWithoutProperties(_ref, ["url", "children"]);

  function clickHandler(event) {
    handleClickEvent(event, url);
  }

  return _react.default.createElement("a", _extends({}, others, {
    onClick: clickHandler
  }), children);
}

IssueishLink.propTypes = {
  url: _propTypes.default.string.isRequired,
  children: _propTypes.default.node
}; // eslint-disable-next-line no-shadow

function handleClickEvent(event, url) {
  event.preventDefault();
  event.stopPropagation();

  if (!event.shiftKey) {
    return openIssueishLinkInNewTab(url, {
      activate: !(event.metaKey || event.ctrlKey)
    });
  } else {
    // Open in browser if shift key held
    return openLinkInBrowser(url);
  }
} // eslint-disable-next-line no-shadow


function openIssueishLinkInNewTab(url, options = {}) {
  const uri = getAtomUriForGithubUrl(url);

  if (uri) {
    return openInNewTab(uri, options);
  } else {
    return null;
  }
}

async function openLinkInBrowser(uri) {
  await _electron.shell.openExternal(uri);
  (0, _reporterProxy.addEvent)('open-issueish-in-browser', {
    package: 'github',
    from: 'issueish-link'
  });
}

function getAtomUriForGithubUrl(githubUrl) {
  return getUriForData(getDataFromGithubUrl(githubUrl));
}

function getDataFromGithubUrl(githubUrl) {
  const {
    hostname,
    pathname
  } = _url.default.parse(githubUrl);

  const [repoOwner, repoName, type, issueishNumber] = pathname.split('/').filter(s => s);
  return {
    hostname,
    repoOwner,
    repoName,
    type,
    issueishNumber: parseInt(issueishNumber, 10)
  };
}

function getUriForData({
  hostname,
  repoOwner,
  repoName,
  type,
  issueishNumber
}) {
  if (hostname !== 'github.com' || !['pull', 'issues'].includes(type) || !issueishNumber || isNaN(issueishNumber)) {
    return null;
  } else {
    return _issueishDetailItem.default.buildURI({
      host: 'github.com',
      owner: repoOwner,
      repo: repoName,
      number: issueishNumber
    });
  }
}

function openInNewTab(uri, {
  activate
} = {
  activate: true
}) {
  return atom.workspace.open(uri, {
    activateItem: activate
  }).then(() => {
    (0, _reporterProxy.addEvent)('open-issueish-in-pane', {
      package: 'github',
      from: 'issueish-link',
      target: 'new-tab'
    });
  });
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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