"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

var _react = _interopRequireWildcard(require("react"));

var _propTypes = _interopRequireDefault(require("prop-types"));

var _classnames = _interopRequireDefault(require("classnames"));

var _atom = require("atom");

var _eventKit = require("event-kit");

var _helpers = require("../helpers");

var _reporterProxy = require("../reporter-proxy");

var _propTypes2 = require("../prop-types");

var _atomTextEditor = _interopRequireDefault(require("../atom/atom-text-editor"));

var _marker = _interopRequireDefault(require("../atom/marker"));

var _markerLayer = _interopRequireDefault(require("../atom/marker-layer"));

var _decoration = _interopRequireDefault(require("../atom/decoration"));

var _gutter = _interopRequireDefault(require("../atom/gutter"));

var _commands = _interopRequireWildcard(require("../atom/commands"));

var _filePatchHeaderView = _interopRequireDefault(require("./file-patch-header-view"));

var _filePatchMetaView = _interopRequireDefault(require("./file-patch-meta-view"));

var _hunkHeaderView = _interopRequireDefault(require("./hunk-header-view"));

var _refHolder = _interopRequireDefault(require("../models/ref-holder"));

var _changedFileItem = _interopRequireDefault(require("../items/changed-file-item"));

var _commitDetailItem = _interopRequireDefault(require("../items/commit-detail-item"));

var _commentGutterDecorationController = _interopRequireDefault(require("../controllers/comment-gutter-decoration-controller"));

var _issueishDetailItem = _interopRequireDefault(require("../items/issueish-detail-item"));

var _file = _interopRequireDefault(require("../models/patch/file"));

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _interopRequireWildcard(obj) { if (obj && obj.__esModule) { return obj; } else { var newObj = {}; if (obj != null) { for (var key in obj) { if (Object.prototype.hasOwnProperty.call(obj, key)) { var desc = Object.defineProperty && Object.getOwnPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : {}; if (desc.get || desc.set) { Object.defineProperty(newObj, key, desc); } else { newObj[key] = obj[key]; } } } } newObj.default = obj; return newObj; } }

function _objectSpread2(target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i] != null ? arguments[i] : {}; var ownKeys = Object.keys(source); if (typeof Object.getOwnPropertySymbols === 'function') { ownKeys = ownKeys.concat(Object.getOwnPropertySymbols(source).filter(function (sym) { return Object.getOwnPropertyDescriptor(source, sym).enumerable; })); } ownKeys.forEach(function (key) { _defineProperty(target, key, source[key]); }); } return target; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

const executableText = {
  [_file.default.modes.NORMAL]: 'non executable',
  [_file.default.modes.EXECUTABLE]: 'executable'
};

class MultiFilePatchView extends _react.default.Component {
  constructor(props) {
    super(props);

    _defineProperty(this, "renderFilePatchDecorations", (filePatch, index) => {
      const isCollapsed = !filePatch.getRenderStatus().isVisible();
      const isEmpty = filePatch.getMarker().getRange().isEmpty();
      const isExpandable = filePatch.getRenderStatus().isExpandable();
      const isUnavailable = isCollapsed && !isExpandable;
      const atEnd = filePatch.getStartRange().start.isEqual(this.props.multiFilePatch.getBuffer().getEndPosition());
      const position = isEmpty && atEnd ? 'after' : 'before';
      return _react.default.createElement(_react.Fragment, {
        key: filePatch.getPath()
      }, _react.default.createElement(_marker.default, {
        invalidate: "never",
        bufferRange: filePatch.getStartRange()
      }, _react.default.createElement(_decoration.default, {
        type: "block",
        position: position,
        order: index,
        className: "github-FilePatchView-controlBlock"
      }, _react.default.createElement(_filePatchHeaderView.default, {
        itemType: this.props.itemType,
        relPath: filePatch.getPath(),
        newPath: filePatch.getStatus() === 'renamed' ? filePatch.getNewPath() : null,
        stagingStatus: this.props.stagingStatus,
        isPartiallyStaged: this.props.isPartiallyStaged,
        hasUndoHistory: this.props.hasUndoHistory,
        hasMultipleFileSelections: this.props.hasMultipleFileSelections,
        tooltips: this.props.tooltips,
        undoLastDiscard: () => this.undoLastDiscardFromButton(filePatch),
        diveIntoMirrorPatch: () => this.props.diveIntoMirrorPatch(filePatch),
        openFile: () => this.didOpenFile({
          selectedFilePatch: filePatch
        }),
        toggleFile: () => this.props.toggleFile(filePatch),
        isCollapsed: isCollapsed,
        triggerCollapse: () => this.props.multiFilePatch.collapseFilePatch(filePatch),
        triggerExpand: () => this.props.multiFilePatch.expandFilePatch(filePatch)
      }), !isCollapsed && this.renderSymlinkChangeMeta(filePatch), !isCollapsed && this.renderExecutableModeChangeMeta(filePatch))), isExpandable && this.renderDiffGate(filePatch, position, index), isUnavailable && this.renderDiffUnavailable(filePatch, position, index), this.renderHunkHeaders(filePatch, index));
    });

    _defineProperty(this, "undoLastDiscardFromCoreUndo", () => {
      if (this.props.hasUndoHistory) {
        const selectedFilePatches = Array.from(this.getSelectedFilePatches());
        /* istanbul ignore else */

        if (this.props.itemType === _changedFileItem.default) {
          this.props.undoLastDiscard(selectedFilePatches[0], {
            eventSource: {
              command: 'core:undo'
            }
          });
        }
      }
    });

    _defineProperty(this, "undoLastDiscardFromButton", filePatch => {
      this.props.undoLastDiscard(filePatch, {
        eventSource: 'button'
      });
    });

    _defineProperty(this, "discardSelectionFromCommand", () => {
      return this.props.discardRows(this.props.selectedRows, this.props.selectionMode, {
        eventSource: {
          command: 'github:discard-selected-lines'
        }
      });
    });

    _defineProperty(this, "didToggleModeChange", () => {
      return Promise.all(Array.from(this.getSelectedFilePatches()).filter(fp => fp.didChangeExecutableMode()).map(this.props.toggleModeChange));
    });

    _defineProperty(this, "didToggleSymlinkChange", () => {
      return Promise.all(Array.from(this.getSelectedFilePatches()).filter(fp => fp.hasTypechange()).map(this.props.toggleSymlinkChange));
    });

    _defineProperty(this, "scrollToFile", ({
      changedFilePath,
      changedFilePosition
    }) => {
      /* istanbul ignore next */
      this.refEditor.map(e => {
        const row = this.props.multiFilePatch.getBufferRowForDiffPosition(changedFilePath, changedFilePosition);

        if (row === null) {
          return null;
        }

        e.scrollToBufferPosition({
          row,
          column: 0
        }, {
          center: true
        });
        e.setCursorBufferPosition({
          row,
          column: 0
        });
        return null;
      });
    });

    (0, _helpers.autobind)(this, 'didMouseDownOnHeader', 'didMouseDownOnLineNumber', 'didMouseMoveOnLineNumber', 'didMouseUp', 'didConfirm', 'didToggleSelectionMode', 'selectNextHunk', 'selectPreviousHunk', 'didOpenFile', 'didAddSelection', 'didChangeSelectionRange', 'didDestroySelection', 'oldLineNumberLabel', 'newLineNumberLabel');
    this.mouseSelectionInProgress = false;
    this.lastMouseMoveLine = null;
    this.nextSelectionMode = null;
    this.refRoot = new _refHolder.default();
    this.refEditor = new _refHolder.default();
    this.refEditorElement = new _refHolder.default();
    this.mounted = false;
    this.subs = new _eventKit.CompositeDisposable();
    this.subs.add(this.refEditor.observe(editor => {
      this.refEditorElement.setter(editor.getElement());

      if (this.props.refEditor) {
        this.props.refEditor.setter(editor);
      }
    }), this.refEditorElement.observe(element => {
      this.props.refInitialFocus && this.props.refInitialFocus.setter(element);
    })); // Synchronously maintain the editor's scroll position and logical selection across buffer updates.

    this.suppressChanges = false;
    let lastScrollTop = null;
    let lastScrollLeft = null;
    let lastSelectionIndex = null;
    this.subs.add(this.props.onWillUpdatePatch(() => {
      this.suppressChanges = true;
      this.refEditor.map(editor => {
        lastSelectionIndex = this.props.multiFilePatch.getMaxSelectionIndex(this.props.selectedRows);
        lastScrollTop = editor.getElement().getScrollTop();
        lastScrollLeft = editor.getElement().getScrollLeft();
        return null;
      });
    }), this.props.onDidUpdatePatch(nextPatch => {
      this.refEditor.map(editor => {
        /* istanbul ignore else */
        if (lastSelectionIndex !== null) {
          const nextSelectionRange = nextPatch.getSelectionRangeForIndex(lastSelectionIndex);

          if (this.props.selectionMode === 'line') {
            this.nextSelectionMode = 'line';
            editor.setSelectedBufferRange(nextSelectionRange);
          } else {
            const nextHunks = new Set(_atom.Range.fromObject(nextSelectionRange).getRows().map(row => nextPatch.getHunkAt(row)).filter(Boolean));
            /* istanbul ignore next */

            const nextRanges = nextHunks.size > 0 ? Array.from(nextHunks, hunk => hunk.getRange()) : [[[0, 0], [0, 0]]];
            this.nextSelectionMode = 'hunk';
            editor.setSelectedBufferRanges(nextRanges);
          }
        }
        /* istanbul ignore else */


        if (lastScrollTop !== null) {
          editor.getElement().setScrollTop(lastScrollTop);
        }
        /* istanbul ignore else */


        if (lastScrollLeft !== null) {
          editor.getElement().setScrollLeft(lastScrollLeft);
        }

        return null;
      });
      this.suppressChanges = false;
      this.didChangeSelectedRows();
    }));
  }

  componentDidMount() {
    this.mounted = true;
    this.measurePerformance('mount');
    window.addEventListener('mouseup', this.didMouseUp);
    this.refEditor.map(editor => {
      // this.props.multiFilePatch is guaranteed to contain at least one FilePatch if <AtomTextEditor> is rendered.
      const [firstPatch] = this.props.multiFilePatch.getFilePatches();
      const [firstHunk] = firstPatch.getHunks();

      if (!firstHunk) {
        return null;
      }

      this.nextSelectionMode = 'hunk';
      editor.setSelectedBufferRange(firstHunk.getRange());
      return null;
    });
    this.subs.add(this.props.config.onDidChange('github.showDiffIconGutter', () => this.forceUpdate()));
    const {
      initChangedFilePath,
      initChangedFilePosition
    } = this.props;
    /* istanbul ignore next */

    if (initChangedFilePath && initChangedFilePosition >= 0) {
      this.scrollToFile({
        changedFilePath: initChangedFilePath,
        changedFilePosition: initChangedFilePosition
      });
    }
    /* istanbul ignore if */


    if (this.props.onOpenFilesTab) {
      this.subs.add(this.props.onOpenFilesTab(this.scrollToFile));
    }
  }

  componentDidUpdate(prevProps) {
    this.measurePerformance('update');

    if (prevProps.refInitialFocus !== this.props.refInitialFocus) {
      prevProps.refInitialFocus && prevProps.refInitialFocus.setter(null);
      this.props.refInitialFocus && this.refEditorElement.map(this.props.refInitialFocus.setter);
    }

    if (this.props.multiFilePatch === prevProps.multiFilePatch) {
      this.nextSelectionMode = null;
    }
  }

  componentWillUnmount() {
    window.removeEventListener('mouseup', this.didMouseUp);
    this.subs.dispose();
    this.mounted = false;
    performance.clearMarks();
    performance.clearMeasures();
  }

  render() {
    const rootClass = (0, _classnames.default)('github-FilePatchView', {
      [`github-FilePatchView--${this.props.stagingStatus}`]: this.props.stagingStatus
    }, {
      'github-FilePatchView--blank': !this.props.multiFilePatch.anyPresent()
    }, {
      'github-FilePatchView--hunkMode': this.props.selectionMode === 'hunk'
    });

    if (this.mounted) {
      performance.mark('MultiFilePatchView-update-start');
    } else {
      performance.mark('MultiFilePatchView-mount-start');
    }

    return _react.default.createElement("div", {
      className: rootClass,
      ref: this.refRoot.setter
    }, this.renderCommands(), _react.default.createElement("main", {
      className: "github-FilePatchView-container"
    }, this.props.multiFilePatch.anyPresent() ? this.renderNonEmptyPatch() : this.renderEmptyPatch()));
  }

  renderCommands() {
    if (this.props.itemType === _commitDetailItem.default || this.props.itemType === _issueishDetailItem.default) {
      return _react.default.createElement(_commands.default, {
        registry: this.props.commands,
        target: this.refRoot
      }, _react.default.createElement(_commands.Command, {
        command: "github:select-next-hunk",
        callback: this.selectNextHunk
      }), _react.default.createElement(_commands.Command, {
        command: "github:select-previous-hunk",
        callback: this.selectPreviousHunk
      }), _react.default.createElement(_commands.Command, {
        command: "github:toggle-patch-selection-mode",
        callback: this.didToggleSelectionMode
      }));
    }

    let stageModeCommand = null;
    let stageSymlinkCommand = null;

    if (this.props.multiFilePatch.didAnyChangeExecutableMode()) {
      const command = this.props.stagingStatus === 'unstaged' ? 'github:stage-file-mode-change' : 'github:unstage-file-mode-change';
      stageModeCommand = _react.default.createElement(_commands.Command, {
        command: command,
        callback: this.didToggleModeChange
      });
    }

    if (this.props.multiFilePatch.anyHaveTypechange()) {
      const command = this.props.stagingStatus === 'unstaged' ? 'github:stage-symlink-change' : 'github:unstage-symlink-change';
      stageSymlinkCommand = _react.default.createElement(_commands.Command, {
        command: command,
        callback: this.didToggleSymlinkChange
      });
    }

    return _react.default.createElement(_commands.default, {
      registry: this.props.commands,
      target: this.refRoot
    }, _react.default.createElement(_commands.Command, {
      command: "github:select-next-hunk",
      callback: this.selectNextHunk
    }), _react.default.createElement(_commands.Command, {
      command: "github:select-previous-hunk",
      callback: this.selectPreviousHunk
    }), _react.default.createElement(_commands.Command, {
      command: "core:confirm",
      callback: this.didConfirm
    }), _react.default.createElement(_commands.Command, {
      command: "core:undo",
      callback: this.undoLastDiscardFromCoreUndo
    }), _react.default.createElement(_commands.Command, {
      command: "github:discard-selected-lines",
      callback: this.discardSelectionFromCommand
    }), _react.default.createElement(_commands.Command, {
      command: "github:jump-to-file",
      callback: this.didOpenFile
    }), _react.default.createElement(_commands.Command, {
      command: "github:surface",
      callback: this.props.surface
    }), _react.default.createElement(_commands.Command, {
      command: "github:toggle-patch-selection-mode",
      callback: this.didToggleSelectionMode
    }), stageModeCommand, stageSymlinkCommand,
    /* istanbul ignore next */
    atom.inDevMode() && _react.default.createElement(_commands.Command, {
      command: "github:inspect-patch",
      callback: () => {
        // eslint-disable-next-line no-console
        console.log(this.props.multiFilePatch.getPatchBuffer().inspect({
          layerNames: ['patch', 'hunk']
        }));
      }
    }),
    /* istanbul ignore next */
    atom.inDevMode() && _react.default.createElement(_commands.Command, {
      command: "github:inspect-regions",
      callback: () => {
        // eslint-disable-next-line no-console
        console.log(this.props.multiFilePatch.getPatchBuffer().inspect({
          layerNames: ['unchanged', 'deletion', 'addition', 'nonewline']
        }));
      }
    }),
    /* istanbul ignore next */
    atom.inDevMode() && _react.default.createElement(_commands.Command, {
      command: "github:inspect-mfp",
      callback: () => {
        // eslint-disable-next-line no-console
        console.log(this.props.multiFilePatch.inspect());
      }
    }));
  }

  renderEmptyPatch() {
    return _react.default.createElement("p", {
      className: "github-FilePatchView-message icon icon-info"
    }, "No changes to display");
  }

  renderNonEmptyPatch() {
    return _react.default.createElement(_atomTextEditor.default, {
      workspace: this.props.workspace,
      buffer: this.props.multiFilePatch.getBuffer(),
      lineNumberGutterVisible: false,
      autoWidth: false,
      autoHeight: false,
      readOnly: true,
      softWrapped: true,
      didAddSelection: this.didAddSelection,
      didChangeSelectionRange: this.didChangeSelectionRange,
      didDestroySelection: this.didDestroySelection,
      refModel: this.refEditor,
      hideEmptiness: true
    }, _react.default.createElement(_gutter.default, {
      name: "old-line-numbers",
      priority: 1,
      className: "old",
      type: "line-number",
      labelFn: this.oldLineNumberLabel,
      onMouseDown: this.didMouseDownOnLineNumber,
      onMouseMove: this.didMouseMoveOnLineNumber
    }), _react.default.createElement(_gutter.default, {
      name: "new-line-numbers",
      priority: 2,
      className: "new",
      type: "line-number",
      labelFn: this.newLineNumberLabel,
      onMouseDown: this.didMouseDownOnLineNumber,
      onMouseMove: this.didMouseMoveOnLineNumber
    }), _react.default.createElement(_gutter.default, {
      name: "github-comment-icon",
      priority: 3,
      className: "comment",
      type: "decorated"
    }), this.props.config.get('github.showDiffIconGutter') && _react.default.createElement(_gutter.default, {
      name: "diff-icons",
      priority: 4,
      type: "line-number",
      className: "icons",
      labelFn: _helpers.blankLabel,
      onMouseDown: this.didMouseDownOnLineNumber,
      onMouseMove: this.didMouseMoveOnLineNumber
    }), this.renderPRCommentIcons(), this.props.multiFilePatch.getFilePatches().map(this.renderFilePatchDecorations), this.renderLineDecorations(Array.from(this.props.selectedRows, row => _atom.Range.fromObject([[row, 0], [row, Infinity]])), 'github-FilePatchView-line--selected', {
      gutter: true,
      icon: true,
      line: true
    }), this.renderDecorationsOnLayer(this.props.multiFilePatch.getAdditionLayer(), 'github-FilePatchView-line--added', {
      icon: true,
      line: true
    }), this.renderDecorationsOnLayer(this.props.multiFilePatch.getDeletionLayer(), 'github-FilePatchView-line--deleted', {
      icon: true,
      line: true
    }), this.renderDecorationsOnLayer(this.props.multiFilePatch.getNoNewlineLayer(), 'github-FilePatchView-line--nonewline', {
      icon: true,
      line: true
    }));
  }

  renderPRCommentIcons() {
    if (this.props.itemType !== _issueishDetailItem.default || this.props.reviewCommentsLoading) {
      return null;
    }

    return this.props.reviewCommentThreads.map(({
      comments,
      thread
    }) => {
      const {
        path,
        position
      } = comments[0];

      if (!this.props.multiFilePatch.getPatchForPath(path)) {
        return null;
      }

      const row = this.props.multiFilePatch.getBufferRowForDiffPosition(path, position);

      if (row === null) {
        return null;
      }

      const isRowSelected = this.props.selectedRows.has(row);
      return _react.default.createElement(_commentGutterDecorationController.default, {
        key: `github-comment-gutter-decoration-${thread.id}`,
        commentRow: row,
        threadId: thread.id,
        workspace: this.props.workspace,
        endpoint: this.props.endpoint,
        owner: this.props.owner,
        repo: this.props.repo,
        number: this.props.number,
        workdir: this.props.workdirPath,
        extraClasses: isRowSelected ? ['github-FilePatchView-line--selected'] : [],
        parent: this.constructor.name
      });
    });
  }

  renderDiffGate(filePatch, position, orderOffset) {
    const showDiff = () => {
      (0, _reporterProxy.addEvent)('expand-file-patch', {
        component: this.constructor.name,
        package: 'github'
      });
      this.props.multiFilePatch.expandFilePatch(filePatch);
    };

    return _react.default.createElement(_marker.default, {
      invalidate: "never",
      bufferRange: filePatch.getStartRange()
    }, _react.default.createElement(_decoration.default, {
      type: "block",
      order: orderOffset + 0.1,
      position: position,
      className: "github-FilePatchView-controlBlock"
    }, _react.default.createElement("p", {
      className: "github-FilePatchView-message icon icon-info"
    }, "Large diffs are collapsed by default for performance reasons.", _react.default.createElement("br", null), _react.default.createElement("button", {
      className: "github-FilePatchView-showDiffButton",
      onClick: showDiff
    }, " Load Diff"))));
  }

  renderDiffUnavailable(filePatch, position, orderOffset) {
    return _react.default.createElement(_marker.default, {
      invalidate: "never",
      bufferRange: filePatch.getStartRange()
    }, _react.default.createElement(_decoration.default, {
      type: "block",
      order: orderOffset + 0.1,
      position: position,
      className: "github-FilePatchView-controlBlock"
    }, _react.default.createElement("p", {
      className: "github-FilePatchView-message icon icon-warning"
    }, "This diff is too large to load at all. Use the command-line to view it.")));
  }

  renderExecutableModeChangeMeta(filePatch) {
    if (!filePatch.didChangeExecutableMode()) {
      return null;
    }

    const oldMode = filePatch.getOldMode();
    const newMode = filePatch.getNewMode();
    const attrs = this.props.stagingStatus === 'unstaged' ? {
      actionIcon: 'icon-move-down',
      actionText: 'Stage Mode Change'
    } : {
      actionIcon: 'icon-move-up',
      actionText: 'Unstage Mode Change'
    };
    return _react.default.createElement(_filePatchMetaView.default, {
      title: "Mode change",
      actionIcon: attrs.actionIcon,
      actionText: attrs.actionText,
      itemType: this.props.itemType,
      action: () => this.props.toggleModeChange(filePatch)
    }, _react.default.createElement(_react.Fragment, null, "File changed mode", _react.default.createElement("span", {
      className: "github-FilePatchView-metaDiff github-FilePatchView-metaDiff--removed"
    }, "from ", executableText[oldMode], " ", _react.default.createElement("code", null, oldMode)), _react.default.createElement("span", {
      className: "github-FilePatchView-metaDiff github-FilePatchView-metaDiff--added"
    }, "to ", executableText[newMode], " ", _react.default.createElement("code", null, newMode))));
  }

  renderSymlinkChangeMeta(filePatch) {
    if (!filePatch.hasSymlink()) {
      return null;
    }

    let detail = _react.default.createElement("div", null);

    let title = '';
    const oldSymlink = filePatch.getOldSymlink();
    const newSymlink = filePatch.getNewSymlink();

    if (oldSymlink && newSymlink) {
      detail = _react.default.createElement(_react.Fragment, null, "Symlink changed", _react.default.createElement("span", {
        className: (0, _classnames.default)('github-FilePatchView-metaDiff', 'github-FilePatchView-metaDiff--fullWidth', 'github-FilePatchView-metaDiff--removed')
      }, "from ", _react.default.createElement("code", null, oldSymlink)), _react.default.createElement("span", {
        className: (0, _classnames.default)('github-FilePatchView-metaDiff', 'github-FilePatchView-metaDiff--fullWidth', 'github-FilePatchView-metaDiff--added')
      }, "to ", _react.default.createElement("code", null, newSymlink)), ".");
      title = 'Symlink changed';
    } else if (oldSymlink && !newSymlink) {
      detail = _react.default.createElement(_react.Fragment, null, "Symlink", _react.default.createElement("span", {
        className: "github-FilePatchView-metaDiff github-FilePatchView-metaDiff--removed"
      }, "to ", _react.default.createElement("code", null, oldSymlink)), "deleted.");
      title = 'Symlink deleted';
    } else {
      detail = _react.default.createElement(_react.Fragment, null, "Symlink", _react.default.createElement("span", {
        className: "github-FilePatchView-metaDiff github-FilePatchView-metaDiff--added"
      }, "to ", _react.default.createElement("code", null, newSymlink)), "created.");
      title = 'Symlink created';
    }

    const attrs = this.props.stagingStatus === 'unstaged' ? {
      actionIcon: 'icon-move-down',
      actionText: 'Stage Symlink Change'
    } : {
      actionIcon: 'icon-move-up',
      actionText: 'Unstage Symlink Change'
    };
    return _react.default.createElement(_filePatchMetaView.default, {
      title: title,
      actionIcon: attrs.actionIcon,
      actionText: attrs.actionText,
      itemType: this.props.itemType,
      action: () => this.props.toggleSymlinkChange(filePatch)
    }, _react.default.createElement(_react.Fragment, null, detail));
  }

  renderHunkHeaders(filePatch, orderOffset) {
    const toggleVerb = this.props.stagingStatus === 'unstaged' ? 'Stage' : 'Unstage';
    const selectedHunks = new Set(Array.from(this.props.selectedRows, row => this.props.multiFilePatch.getHunkAt(row)));
    return _react.default.createElement(_react.Fragment, null, _react.default.createElement(_markerLayer.default, null, filePatch.getHunks().map((hunk, index) => {
      const containsSelection = this.props.selectionMode === 'line' && selectedHunks.has(hunk);
      const isSelected = this.props.selectionMode === 'hunk' && selectedHunks.has(hunk);
      let buttonSuffix = '';

      if (containsSelection) {
        buttonSuffix += 'Selected Line';

        if (this.props.selectedRows.size > 1) {
          buttonSuffix += 's';
        }
      } else {
        buttonSuffix += 'Hunk';

        if (selectedHunks.size > 1) {
          buttonSuffix += 's';
        }
      }

      const toggleSelectionLabel = `${toggleVerb} ${buttonSuffix}`;
      const discardSelectionLabel = `Discard ${buttonSuffix}`;
      const startPoint = hunk.getRange().start;
      const startRange = new _atom.Range(startPoint, startPoint);
      return _react.default.createElement(_marker.default, {
        key: `hunkHeader-${index}`,
        bufferRange: startRange,
        invalidate: "never"
      }, _react.default.createElement(_decoration.default, {
        type: "block",
        order: orderOffset + 0.2,
        className: "github-FilePatchView-controlBlock"
      }, _react.default.createElement(_hunkHeaderView.default, {
        refTarget: this.refEditorElement,
        hunk: hunk,
        isSelected: isSelected,
        stagingStatus: this.props.stagingStatus,
        selectionMode: "line",
        toggleSelectionLabel: toggleSelectionLabel,
        discardSelectionLabel: discardSelectionLabel,
        tooltips: this.props.tooltips,
        keymaps: this.props.keymaps,
        toggleSelection: () => this.toggleHunkSelection(hunk, containsSelection),
        discardSelection: () => this.discardHunkSelection(hunk, containsSelection),
        mouseDown: this.didMouseDownOnHeader,
        itemType: this.props.itemType
      })));
    })));
  }

  renderLineDecorations(ranges, lineClass, {
    line,
    gutter,
    icon,
    refHolder
  }) {
    if (ranges.length === 0) {
      return null;
    }

    const holder = refHolder || new _refHolder.default();
    return _react.default.createElement(_markerLayer.default, {
      handleLayer: holder.setter
    }, ranges.map((range, index) => {
      return _react.default.createElement(_marker.default, {
        key: `line-${lineClass}-${index}`,
        bufferRange: range,
        invalidate: "never"
      });
    }), this.renderDecorations(lineClass, {
      line,
      gutter,
      icon
    }));
  }

  renderDecorationsOnLayer(layer, lineClass, {
    line,
    gutter,
    icon
  }) {
    if (layer.getMarkerCount() === 0) {
      return null;
    }

    return _react.default.createElement(_markerLayer.default, {
      external: layer
    }, this.renderDecorations(lineClass, {
      line,
      gutter,
      icon
    }));
  }

  renderDecorations(lineClass, {
    line,
    gutter,
    icon
  }) {
    return _react.default.createElement(_react.Fragment, null, line && _react.default.createElement(_decoration.default, {
      type: "line",
      className: lineClass,
      omitEmptyLastRow: false
    }), gutter && _react.default.createElement(_react.Fragment, null, _react.default.createElement(_decoration.default, {
      type: "line-number",
      gutterName: "old-line-numbers",
      className: lineClass,
      omitEmptyLastRow: false
    }), _react.default.createElement(_decoration.default, {
      type: "line-number",
      gutterName: "new-line-numbers",
      className: lineClass,
      omitEmptyLastRow: false
    }), _react.default.createElement(_decoration.default, {
      type: "gutter",
      gutterName: "github-comment-icon",
      className: `github-editorCommentGutterIcon empty ${lineClass}`,
      omitEmptyLastRow: false
    })), icon && _react.default.createElement(_decoration.default, {
      type: "line-number",
      gutterName: "diff-icons",
      className: lineClass,
      omitEmptyLastRow: false
    }));
  }

  toggleHunkSelection(hunk, containsSelection) {
    if (containsSelection) {
      return this.props.toggleRows(this.props.selectedRows, this.props.selectionMode, {
        eventSource: 'button'
      });
    } else {
      const changeRows = new Set(hunk.getChanges().reduce((rows, change) => {
        rows.push(...change.getBufferRows());
        return rows;
      }, []));
      return this.props.toggleRows(changeRows, 'hunk', {
        eventSource: 'button'
      });
    }
  }

  discardHunkSelection(hunk, containsSelection) {
    if (containsSelection) {
      return this.props.discardRows(this.props.selectedRows, this.props.selectionMode, {
        eventSource: 'button'
      });
    } else {
      const changeRows = new Set(hunk.getChanges().reduce((rows, change) => {
        rows.push(...change.getBufferRows());
        return rows;
      }, []));
      return this.props.discardRows(changeRows, 'hunk', {
        eventSource: 'button'
      });
    }
  }

  didMouseDownOnHeader(event, hunk) {
    this.nextSelectionMode = 'hunk';
    this.handleSelectionEvent(event, hunk.getRange());
  }

  didMouseDownOnLineNumber(event) {
    const line = event.bufferRow;

    if (line === undefined || isNaN(line)) {
      return;
    }

    this.nextSelectionMode = 'line';

    if (this.handleSelectionEvent(event.domEvent, [[line, 0], [line, Infinity]])) {
      this.mouseSelectionInProgress = true;
    }
  }

  didMouseMoveOnLineNumber(event) {
    if (!this.mouseSelectionInProgress) {
      return;
    }

    const line = event.bufferRow;

    if (this.lastMouseMoveLine === line || line === undefined || isNaN(line)) {
      return;
    }

    this.lastMouseMoveLine = line;
    this.nextSelectionMode = 'line';
    this.handleSelectionEvent(event.domEvent, [[line, 0], [line, Infinity]], {
      add: true
    });
  }

  didMouseUp() {
    this.mouseSelectionInProgress = false;
  }

  handleSelectionEvent(event, rangeLike, opts) {
    if (event.button !== 0) {
      return false;
    }

    const isWindows = process.platform === 'win32';

    if (event.ctrlKey && !isWindows) {
      // Allow the context menu to open.
      return false;
    }

    const options = _objectSpread2({
      add: false
    }, opts); // Normalize the target selection range


    const converted = _atom.Range.fromObject(rangeLike);

    const range = this.refEditor.map(editor => editor.clipBufferRange(converted)).getOr(converted);

    if (event.metaKey ||
    /* istanbul ignore next */
    event.ctrlKey && isWindows) {
      this.refEditor.map(editor => {
        let intersects = false;
        let without = null;

        for (const selection of editor.getSelections()) {
          if (selection.intersectsBufferRange(range)) {
            // Remove range from this selection by truncating it to the "near edge" of the range and creating a
            // new selection from the "far edge" to the previous end. Omit either side if it is empty.
            intersects = true;
            const selectionRange = selection.getBufferRange();
            const newRanges = [];

            if (!range.start.isEqual(selectionRange.start)) {
              // Include the bit from the selection's previous start to the range's start.
              let nudged = range.start;

              if (range.start.column === 0) {
                const lastColumn = editor.getBuffer().lineLengthForRow(range.start.row - 1);
                nudged = [range.start.row - 1, lastColumn];
              }

              newRanges.push([selectionRange.start, nudged]);
            }

            if (!range.end.isEqual(selectionRange.end)) {
              // Include the bit from the range's end to the selection's end.
              let nudged = range.end;
              const lastColumn = editor.getBuffer().lineLengthForRow(range.end.row);

              if (range.end.column === lastColumn) {
                nudged = [range.end.row + 1, 0];
              }

              newRanges.push([nudged, selectionRange.end]);
            }

            if (newRanges.length > 0) {
              selection.setBufferRange(newRanges[0]);

              for (const newRange of newRanges.slice(1)) {
                editor.addSelectionForBufferRange(newRange, {
                  reversed: selection.isReversed()
                });
              }
            } else {
              without = selection;
            }
          }
        }

        if (without !== null) {
          const replacementRanges = editor.getSelections().filter(each => each !== without).map(each => each.getBufferRange());

          if (replacementRanges.length > 0) {
            editor.setSelectedBufferRanges(replacementRanges);
          }
        }

        if (!intersects) {
          // Add this range as a new, distinct selection.
          editor.addSelectionForBufferRange(range);
        }

        return null;
      });
    } else if (options.add || event.shiftKey) {
      // Extend the existing selection to encompass this range.
      this.refEditor.map(editor => {
        const lastSelection = editor.getLastSelection();
        const lastSelectionRange = lastSelection.getBufferRange(); // You are now entering the wall of ternery operators. This is your last exit before the tollbooth

        const isBefore = range.start.isLessThan(lastSelectionRange.start);
        const farEdge = isBefore ? range.start : range.end;
        const newRange = isBefore ? [farEdge, lastSelectionRange.end] : [lastSelectionRange.start, farEdge];
        lastSelection.setBufferRange(newRange, {
          reversed: isBefore
        });
        return null;
      });
    } else {
      this.refEditor.map(editor => editor.setSelectedBufferRange(range));
    }

    return true;
  }

  didConfirm() {
    return this.props.toggleRows(this.props.selectedRows, this.props.selectionMode);
  }

  didToggleSelectionMode() {
    const selectedHunks = this.getSelectedHunks();
    this.withSelectionMode({
      line: () => {
        const hunkRanges = selectedHunks.map(hunk => hunk.getRange());
        this.nextSelectionMode = 'hunk';
        this.refEditor.map(editor => editor.setSelectedBufferRanges(hunkRanges));
      },
      hunk: () => {
        let firstChangeRow = Infinity;

        for (const hunk of selectedHunks) {
          const [firstChange] = hunk.getChanges();
          /* istanbul ignore else */

          if (firstChange && (!firstChangeRow || firstChange.getStartBufferRow() < firstChangeRow)) {
            firstChangeRow = firstChange.getStartBufferRow();
          }
        }

        this.nextSelectionMode = 'line';
        this.refEditor.map(editor => {
          editor.setSelectedBufferRanges([[[firstChangeRow, 0], [firstChangeRow, Infinity]]]);
          return null;
        });
      }
    });
  }

  selectNextHunk() {
    this.refEditor.map(editor => {
      const nextHunks = new Set(this.withSelectedHunks(hunk => this.getHunkAfter(hunk) || hunk));
      const nextRanges = Array.from(nextHunks, hunk => hunk.getRange());
      this.nextSelectionMode = 'hunk';
      editor.setSelectedBufferRanges(nextRanges);
      return null;
    });
  }

  selectPreviousHunk() {
    this.refEditor.map(editor => {
      const nextHunks = new Set(this.withSelectedHunks(hunk => this.getHunkBefore(hunk) || hunk));
      const nextRanges = Array.from(nextHunks, hunk => hunk.getRange());
      this.nextSelectionMode = 'hunk';
      editor.setSelectedBufferRanges(nextRanges);
      return null;
    });
  }

  didOpenFile({
    selectedFilePatch
  }) {
    const cursorsByFilePatch = new Map();
    this.refEditor.map(editor => {
      const placedRows = new Set();

      for (const cursor of editor.getCursors()) {
        const cursorRow = cursor.getBufferPosition().row;
        const hunk = this.props.multiFilePatch.getHunkAt(cursorRow);
        const filePatch = this.props.multiFilePatch.getFilePatchAt(cursorRow);
        /* istanbul ignore next */

        if (!hunk) {
          continue;
        }

        let newRow = hunk.getNewRowAt(cursorRow);
        let newColumn = cursor.getBufferPosition().column;

        if (newRow === null) {
          let nearestRow = hunk.getNewStartRow();

          for (const region of hunk.getRegions()) {
            if (!region.includesBufferRow(cursorRow)) {
              region.when({
                unchanged: () => {
                  nearestRow += region.bufferRowCount();
                },
                addition: () => {
                  nearestRow += region.bufferRowCount();
                }
              });
            } else {
              break;
            }
          }

          if (!placedRows.has(nearestRow)) {
            newRow = nearestRow;
            newColumn = 0;
            placedRows.add(nearestRow);
          }
        }

        if (newRow !== null) {
          // Why is this needed? I _think_ everything is in terms of buffer position
          // so there shouldn't be an off-by-one issue
          newRow -= 1;
          const cursors = cursorsByFilePatch.get(filePatch);

          if (!cursors) {
            cursorsByFilePatch.set(filePatch, [[newRow, newColumn]]);
          } else {
            cursors.push([newRow, newColumn]);
          }
        }
      }

      return null;
    });
    const filePatchesWithCursors = new Set(cursorsByFilePatch.keys());

    if (selectedFilePatch && !filePatchesWithCursors.has(selectedFilePatch)) {
      const [firstHunk] = selectedFilePatch.getHunks();
      const cursorRow = firstHunk ? firstHunk.getNewStartRow() - 1 :
      /* istanbul ignore next */
      0;
      return this.props.openFile(selectedFilePatch, [[cursorRow, 0]], true);
    } else {
      const pending = cursorsByFilePatch.size === 1;
      return Promise.all(Array.from(cursorsByFilePatch, value => {
        const [filePatch, cursors] = value;
        return this.props.openFile(filePatch, cursors, pending);
      }));
    }
  }

  getSelectedRows() {
    return this.refEditor.map(editor => {
      return new Set(editor.getSelections().map(selection => selection.getBufferRange()).reduce((acc, range) => {
        for (const row of range.getRows()) {
          if (this.isChangeRow(row)) {
            acc.push(row);
          }
        }

        return acc;
      }, []));
    }).getOr(new Set());
  }

  didAddSelection() {
    this.didChangeSelectedRows();
  }

  didChangeSelectionRange(event) {
    if (!event || event.oldBufferRange.start.row !== event.newBufferRange.start.row || event.oldBufferRange.end.row !== event.newBufferRange.end.row) {
      this.didChangeSelectedRows();
    }
  }

  didDestroySelection() {
    this.didChangeSelectedRows();
  }

  didChangeSelectedRows() {
    if (this.suppressChanges) {
      return;
    }

    const nextCursorRows = this.refEditor.map(editor => {
      return editor.getCursorBufferPositions().map(position => position.row);
    }).getOr([]);
    const hasMultipleFileSelections = this.props.multiFilePatch.spansMultipleFiles(nextCursorRows);
    this.props.selectedRowsChanged(this.getSelectedRows(), this.nextSelectionMode || 'line', hasMultipleFileSelections);
  }

  oldLineNumberLabel({
    bufferRow,
    softWrapped
  }) {
    const hunk = this.props.multiFilePatch.getHunkAt(bufferRow);

    if (hunk === undefined) {
      return this.pad('');
    }

    const oldRow = hunk.getOldRowAt(bufferRow);

    if (softWrapped) {
      return this.pad(oldRow === null ? '' : '•');
    }

    return this.pad(oldRow);
  }

  newLineNumberLabel({
    bufferRow,
    softWrapped
  }) {
    const hunk = this.props.multiFilePatch.getHunkAt(bufferRow);

    if (hunk === undefined) {
      return this.pad('');
    }

    const newRow = hunk.getNewRowAt(bufferRow);

    if (softWrapped) {
      return this.pad(newRow === null ? '' : '•');
    }

    return this.pad(newRow);
  }
  /*
   * Return a Set of the Hunks that include at least one editor selection. The selection need not contain an actual
   * change row.
   */


  getSelectedHunks() {
    return this.withSelectedHunks(each => each);
  }

  withSelectedHunks(callback) {
    return this.refEditor.map(editor => {
      const seen = new Set();
      return editor.getSelectedBufferRanges().reduce((acc, range) => {
        for (const row of range.getRows()) {
          const hunk = this.props.multiFilePatch.getHunkAt(row);

          if (!hunk || seen.has(hunk)) {
            continue;
          }

          seen.add(hunk);
          acc.push(callback(hunk));
        }

        return acc;
      }, []);
    }).getOr([]);
  }
  /*
   * Return a Set of FilePatches that include at least one editor selection. The selection need not contain an actual
   * change row.
   */


  getSelectedFilePatches() {
    return this.refEditor.map(editor => {
      const patches = new Set();

      for (const range of editor.getSelectedBufferRanges()) {
        for (const row of range.getRows()) {
          const patch = this.props.multiFilePatch.getFilePatchAt(row);
          patches.add(patch);
        }
      }

      return patches;
    }).getOr(new Set());
  }

  getHunkBefore(hunk) {
    const prevRow = hunk.getRange().start.row - 1;
    return this.props.multiFilePatch.getHunkAt(prevRow);
  }

  getHunkAfter(hunk) {
    const nextRow = hunk.getRange().end.row + 1;
    return this.props.multiFilePatch.getHunkAt(nextRow);
  }

  isChangeRow(bufferRow) {
    const changeLayers = [this.props.multiFilePatch.getAdditionLayer(), this.props.multiFilePatch.getDeletionLayer()];
    return changeLayers.some(layer => layer.findMarkers({
      intersectsRow: bufferRow
    }).length > 0);
  }

  withSelectionMode(callbacks) {
    const callback = callbacks[this.props.selectionMode];
    /* istanbul ignore if */

    if (!callback) {
      throw new Error(`Unknown selection mode: ${this.props.selectionMode}`);
    }

    return callback();
  }

  pad(num) {
    const maxDigits = this.props.multiFilePatch.getMaxLineNumberWidth();

    if (num === null) {
      return _helpers.NBSP_CHARACTER.repeat(maxDigits);
    } else {
      return _helpers.NBSP_CHARACTER.repeat(maxDigits - num.toString().length) + num.toString();
    }
  }

  measurePerformance(action) {
    /* istanbul ignore else */
    if ((action === 'update' || action === 'mount') && performance.getEntriesByName(`MultiFilePatchView-${action}-start`).length > 0) {
      performance.mark(`MultiFilePatchView-${action}-end`);
      performance.measure(`MultiFilePatchView-${action}`, `MultiFilePatchView-${action}-start`, `MultiFilePatchView-${action}-end`);
      const perf = performance.getEntriesByName(`MultiFilePatchView-${action}`)[0];
      performance.clearMarks(`MultiFilePatchView-${action}-start`);
      performance.clearMarks(`MultiFilePatchView-${action}-end`);
      performance.clearMeasures(`MultiFilePatchView-${action}`);
      (0, _reporterProxy.addEvent)(`MultiFilePatchView-${action}`, {
        package: 'github',
        filePatchesLineCounts: this.props.multiFilePatch.getFilePatches().map(fp => fp.getPatch().getChangedLineCount()),
        duration: perf.duration
      });
    }
  }

}

exports.default = MultiFilePatchView;

_defineProperty(MultiFilePatchView, "propTypes", {
  // Behavior controls
  stagingStatus: _propTypes.default.oneOf(['staged', 'unstaged']),
  isPartiallyStaged: _propTypes.default.bool,
  itemType: _propTypes2.ItemTypePropType.isRequired,
  // Models
  repository: _propTypes.default.object.isRequired,
  multiFilePatch: _propTypes2.MultiFilePatchPropType.isRequired,
  selectionMode: _propTypes.default.oneOf(['hunk', 'line']).isRequired,
  selectedRows: _propTypes.default.object.isRequired,
  hasMultipleFileSelections: _propTypes.default.bool.isRequired,
  hasUndoHistory: _propTypes.default.bool,
  // Review comments
  reviewCommentsLoading: _propTypes.default.bool,
  reviewCommentThreads: _propTypes.default.arrayOf(_propTypes.default.shape({
    thread: _propTypes.default.object.isRequired,
    comments: _propTypes.default.arrayOf(_propTypes.default.object).isRequired
  })),
  // Atom environment
  workspace: _propTypes.default.object.isRequired,
  commands: _propTypes.default.object.isRequired,
  keymaps: _propTypes.default.object.isRequired,
  tooltips: _propTypes.default.object.isRequired,
  config: _propTypes.default.object.isRequired,
  pullRequest: _propTypes.default.object,
  // Callbacks
  selectedRowsChanged: _propTypes.default.func,
  // Action methods
  switchToIssueish: _propTypes.default.func,
  diveIntoMirrorPatch: _propTypes.default.func,
  surface: _propTypes.default.func,
  openFile: _propTypes.default.func,
  toggleFile: _propTypes.default.func,
  toggleRows: _propTypes.default.func,
  toggleModeChange: _propTypes.default.func,
  toggleSymlinkChange: _propTypes.default.func,
  undoLastDiscard: _propTypes.default.func,
  discardRows: _propTypes.default.func,
  onWillUpdatePatch: _propTypes.default.func,
  onDidUpdatePatch: _propTypes.default.func,
  // External refs
  refEditor: _propTypes2.RefHolderPropType,
  refInitialFocus: _propTypes2.RefHolderPropType,
  // for navigating the PR changed files tab
  onOpenFilesTab: _propTypes.default.func,
  initChangedFilePath: _propTypes.default.string,
  initChangedFilePosition: _propTypes.default.number,
  // for opening the reviews dock item
  endpoint: _propTypes2.EndpointPropType,
  owner: _propTypes.default.string,
  repo: _propTypes.default.string,
  number: _propTypes.default.number,
  workdirPath: _propTypes.default.string
});

_defineProperty(MultiFilePatchView, "defaultProps", {
  onWillUpdatePatch: () => new _eventKit.Disposable(),
  onDidUpdatePatch: () => new _eventKit.Disposable(),
  reviewCommentsLoading: false,
  reviewCommentThreads: []
});
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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