"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.openIssueishItem = openIssueishItem;
exports.default = void 0;

var _react = _interopRequireDefault(require("react"));

var _propTypes = _interopRequireDefault(require("prop-types"));

var _atom = require("atom");

var _issueishDetailItem = _interopRequireDefault(require("../items/issueish-detail-item"));

var _tabGroup = _interopRequireDefault(require("../tab-group"));

var _dialogView = _interopRequireDefault(require("./dialog-view"));

var _tabbable = require("./tabbable");

var _reporterProxy = require("../reporter-proxy");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

const ISSUEISH_URL_REGEX = /^(?:https?:\/\/)?(github.com)\/([^/]+)\/([^/]+)\/(?:issues|pull)\/(\d+)/;

class OpenIssueishDialog extends _react.default.Component {
  constructor(props) {
    super(props);

    _defineProperty(this, "accept", () => {
      const issueishURL = this.url.getText();

      if (issueishURL.length === 0) {
        return Promise.resolve();
      }

      return this.props.request.accept(issueishURL);
    });

    _defineProperty(this, "didChangeURL", () => {
      const enabled = !this.url.isEmpty();

      if (this.state.acceptEnabled !== enabled) {
        this.setState({
          acceptEnabled: enabled
        });
      }
    });

    this.url = new _atom.TextBuffer();
    this.state = {
      acceptEnabled: false
    };
    this.sub = this.url.onDidChange(this.didChangeURL);
    this.tabGroup = new _tabGroup.default();
  }

  render() {
    return _react.default.createElement(_dialogView.default, {
      acceptEnabled: this.state.acceptEnabled,
      acceptClassName: "icon icon-git-pull-request",
      acceptText: "Open Issue or Pull Request",
      accept: this.accept,
      cancel: this.props.request.cancel,
      tabGroup: this.tabGroup,
      inProgress: this.props.inProgress,
      error: this.props.error,
      workspace: this.props.workspace,
      commands: this.props.commands
    }, _react.default.createElement("label", {
      className: "github-DialogLabel"
    }, "Issue or pull request URL:", _react.default.createElement(_tabbable.TabbableTextEditor, {
      tabGroup: this.tabGroup,
      commands: this.props.commands,
      autofocus: true,
      mini: true,
      className: "github-OpenIssueish-url",
      buffer: this.url
    })));
  }

  componentDidMount() {
    this.tabGroup.autofocus();
  }

  componentWillUnmount() {
    this.sub.dispose();
  }

  parseUrl() {
    const url = this.getIssueishUrl();
    const matches = url.match(ISSUEISH_URL_REGEX);

    if (!matches) {
      return false;
    }

    const [_full, repoOwner, repoName, issueishNumber] = matches; // eslint-disable-line no-unused-vars

    return {
      repoOwner,
      repoName,
      issueishNumber
    };
  }

}

exports.default = OpenIssueishDialog;

_defineProperty(OpenIssueishDialog, "propTypes", {
  // Model
  request: _propTypes.default.shape({
    getParams: _propTypes.default.func.isRequired,
    accept: _propTypes.default.func.isRequired,
    cancel: _propTypes.default.func.isRequired
  }).isRequired,
  inProgress: _propTypes.default.bool,
  error: _propTypes.default.instanceOf(Error),
  // Atom environment
  workspace: _propTypes.default.object.isRequired,
  commands: _propTypes.default.object.isRequired
});

async function openIssueishItem(issueishURL, {
  workspace,
  workdir
}) {
  const matches = ISSUEISH_URL_REGEX.exec(issueishURL);

  if (!matches) {
    throw new Error('Not a valid issue or pull request URL');
  }

  const [, host, owner, repo, number] = matches;

  const uri = _issueishDetailItem.default.buildURI({
    host,
    owner,
    repo,
    number,
    workdir
  });

  const item = await workspace.open(uri, {
    searchAllPanes: true
  });
  (0, _reporterProxy.addEvent)('open-issueish-in-pane', {
    package: 'github',
    from: 'dialog'
  });
  return item;
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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