"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.Operation = exports.RendererProcess = exports.Worker = exports.default = void 0;

var _path = _interopRequireDefault(require("path"));

var _querystring = _interopRequireDefault(require("querystring"));

var _electron = require("electron");

var _eventKit = require("event-kit");

var _helpers = require("./helpers");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _objectSpread2(target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i] != null ? arguments[i] : {}; var ownKeys = Object.keys(source); if (typeof Object.getOwnPropertySymbols === 'function') { ownKeys = ownKeys.concat(Object.getOwnPropertySymbols(source).filter(function (sym) { return Object.getOwnPropertyDescriptor(source, sym).enumerable; })); } ownKeys.forEach(function (key) { _defineProperty(target, key, source[key]); }); } return target; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

const {
  BrowserWindow
} = _electron.remote;

class WorkerManager {
  static getInstance() {
    if (!this.instance) {
      this.instance = new WorkerManager();
    }

    return this.instance;
  }

  static reset(force) {
    if (this.instance) {
      this.instance.destroy(force);
    }

    this.instance = null;
  }

  constructor() {
    (0, _helpers.autobind)(this, 'onDestroyed', 'onCrashed', 'onSick');
    this.workers = new Set();
    this.activeWorker = null;
    this.createNewWorker();
  }

  isReady() {
    return this.activeWorker.isReady();
  }

  request(data) {
    if (this.destroyed) {
      throw new Error('Worker is destroyed');
    }

    let operation;
    const requestPromise = new Promise((resolve, reject) => {
      operation = new Operation(data, resolve, reject);
      return this.activeWorker.executeOperation(operation);
    });
    operation.setPromise(requestPromise);
    return {
      cancel: () => this.activeWorker.cancelOperation(operation),
      promise: requestPromise
    };
  }

  createNewWorker({
    operationCountLimit
  } = {
    operationCountLimit: 10
  }) {
    if (this.destroyed) {
      return;
    }

    this.activeWorker = new Worker({
      operationCountLimit,
      onDestroyed: this.onDestroyed,
      onCrashed: this.onCrashed,
      onSick: this.onSick
    });
    this.workers.add(this.activeWorker);
  }

  onDestroyed(destroyedWorker) {
    this.workers.delete(destroyedWorker);
  }

  onCrashed(crashedWorker) {
    if (crashedWorker === this.getActiveWorker()) {
      this.createNewWorker({
        operationCountLimit: crashedWorker.getOperationCountLimit()
      });
    }

    crashedWorker.getRemainingOperations().forEach(operation => this.activeWorker.executeOperation(operation));
  }

  onSick(sickWorker) {
    if (!atom.inSpecMode()) {
      // eslint-disable-next-line no-console
      console.warn(`Sick worker detected.
        operationCountLimit: ${sickWorker.getOperationCountLimit()},
        completed operation count: ${sickWorker.getCompletedOperationCount()}`);
    }

    const operationCountLimit = this.calculateNewOperationCountLimit(sickWorker);
    return this.createNewWorker({
      operationCountLimit
    });
  }

  calculateNewOperationCountLimit(lastWorker) {
    let operationCountLimit = 10;

    if (lastWorker.getOperationCountLimit() >= lastWorker.getCompletedOperationCount()) {
      operationCountLimit = Math.min(lastWorker.getOperationCountLimit() * 2, 100);
    }

    return operationCountLimit;
  }

  getActiveWorker() {
    return this.activeWorker;
  }

  getReadyPromise() {
    return this.activeWorker.getReadyPromise();
  }

  destroy(force) {
    this.destroyed = true;
    this.workers.forEach(worker => worker.destroy(force));
  }

}

exports.default = WorkerManager;

_defineProperty(WorkerManager, "instance", null);

class Worker {
  constructor({
    operationCountLimit,
    onSick,
    onCrashed,
    onDestroyed
  }) {
    (0, _helpers.autobind)(this, 'handleDataReceived', 'onOperationComplete', 'handleCancelled', 'handleExecStarted', 'handleSpawnError', 'handleStdinError', 'handleSick', 'handleCrashed');
    this.operationCountLimit = operationCountLimit;
    this.onSick = onSick;
    this.onCrashed = onCrashed;
    this.onDestroyed = onDestroyed;
    this.operationsById = new Map();
    this.completedOperationCount = 0;
    this.sick = false;
    this.rendererProcess = new RendererProcess({
      loadUrl: this.getLoadUrl(operationCountLimit),
      onData: this.handleDataReceived,
      onCancelled: this.handleCancelled,
      onExecStarted: this.handleExecStarted,
      onSpawnError: this.handleSpawnError,
      onStdinError: this.handleStdinError,
      onSick: this.handleSick,
      onCrashed: this.handleCrashed,
      onDestroyed: this.destroy
    });
  }

  isReady() {
    return this.rendererProcess.isReady();
  }

  getLoadUrl(operationCountLimit) {
    const htmlPath = _path.default.join((0, _helpers.getPackageRoot)(), 'lib', 'renderer.html');

    const rendererJsPath = _path.default.join((0, _helpers.getPackageRoot)(), 'lib', 'worker.js');

    const qs = _querystring.default.stringify({
      js: rendererJsPath,
      managerWebContentsId: this.getWebContentsId(),
      operationCountLimit,
      channelName: Worker.channelName
    });

    return `file://${htmlPath}?${qs}`;
  }

  getWebContentsId() {
    return _electron.remote.getCurrentWebContents().id;
  }

  executeOperation(operation) {
    this.operationsById.set(operation.id, operation);
    operation.onComplete(this.onOperationComplete);
    return this.rendererProcess.executeOperation(operation);
  }

  cancelOperation(operation) {
    return this.rendererProcess.cancelOperation(operation);
  }

  handleDataReceived({
    id,
    results
  }) {
    const operation = this.operationsById.get(id);
    operation.complete(results, data => {
      const {
        timing
      } = data;
      const totalInternalTime = timing.execTime + timing.spawnTime;
      const ipcTime = operation.getExecutionTime() - totalInternalTime;
      data.timing.ipcTime = ipcTime;
      return data;
    });
  }

  onOperationComplete(operation) {
    this.completedOperationCount++;
    this.operationsById.delete(operation.id);

    if (this.sick && this.operationsById.size === 0) {
      this.destroy();
    }
  }

  handleCancelled({
    id
  }) {
    const operation = this.operationsById.get(id);

    if (operation) {
      // handleDataReceived() can be received before handleCancelled()
      operation.wasCancelled();
    }
  }

  handleExecStarted({
    id
  }) {
    const operation = this.operationsById.get(id);
    operation.setInProgress();
  }

  handleSpawnError({
    id,
    err
  }) {
    const operation = this.operationsById.get(id);
    operation.error(err);
  }

  handleStdinError({
    id,
    stdin,
    err
  }) {
    const operation = this.operationsById.get(id);
    operation.error(err);
  }

  handleSick() {
    this.sick = true;
    this.onSick(this);
  }

  handleCrashed() {
    this.onCrashed(this);
    this.destroy();
  }

  getOperationCountLimit() {
    return this.operationCountLimit;
  }

  getCompletedOperationCount() {
    return this.completedOperationCount;
  }

  getRemainingOperations() {
    return Array.from(this.operationsById.values());
  }

  getPid() {
    return this.rendererProcess.getPid();
  }

  getReadyPromise() {
    return this.rendererProcess.getReadyPromise();
  }

  async destroy(force) {
    this.onDestroyed(this);

    if (this.operationsById.size > 0 && !force) {
      const remainingOperationPromises = this.getRemainingOperations().map(operation => operation.getPromise().catch(() => null));
      await Promise.all(remainingOperationPromises);
    }

    this.rendererProcess.destroy();
  }

}
/*
Sends operations to renderer processes
*/


exports.Worker = Worker;

_defineProperty(Worker, "channelName", 'github:renderer-ipc');

class RendererProcess {
  constructor({
    loadUrl,
    onDestroyed,
    onCrashed,
    onSick,
    onData,
    onCancelled,
    onSpawnError,
    onStdinError,
    onExecStarted
  }) {
    (0, _helpers.autobind)(this, 'handleDestroy');
    this.onDestroyed = onDestroyed;
    this.onCrashed = onCrashed;
    this.onSick = onSick;
    this.onData = onData;
    this.onCancelled = onCancelled;
    this.onSpawnError = onSpawnError;
    this.onStdinError = onStdinError;
    this.onExecStarted = onExecStarted;
    this.win = new BrowserWindow({
      show: !!process.env.ATOM_GITHUB_SHOW_RENDERER_WINDOW,
      webPreferences: {
        nodeIntegration: true
      }
    });
    this.webContents = this.win.webContents; // this.webContents.openDevTools();

    this.emitter = new _eventKit.Emitter();
    this.subscriptions = new _eventKit.CompositeDisposable();
    this.registerListeners();
    this.win.loadURL(loadUrl);
    this.win.webContents.on('crashed', this.handleDestroy);
    this.win.webContents.on('destroyed', this.handleDestroy);
    this.subscriptions.add(new _eventKit.Disposable(() => {
      if (!this.win.isDestroyed()) {
        this.win.webContents.removeListener('crashed', this.handleDestroy);
        this.win.webContents.removeListener('destroyed', this.handleDestroy);
        this.win.destroy();
      }
    }), this.emitter);
    this.ready = false;
    this.readyPromise = new Promise(resolve => {
      this.resolveReady = resolve;
    });
  }

  isReady() {
    return this.ready;
  }

  handleDestroy(...args) {
    this.destroy();
    this.onCrashed(...args);
  }

  registerListeners() {
    const handleMessages = (event, {
      sourceWebContentsId,
      type,
      data
    }) => {
      if (sourceWebContentsId === this.win.webContents.id) {
        this.emitter.emit(type, data);
      }
    };

    _electron.ipcRenderer.on(Worker.channelName, handleMessages);

    this.emitter.on('renderer-ready', ({
      pid
    }) => {
      this.pid = pid;
      this.ready = true;
      this.resolveReady();
    });
    this.emitter.on('git-data', this.onData);
    this.emitter.on('git-cancelled', this.onCancelled);
    this.emitter.on('git-spawn-error', this.onSpawnError);
    this.emitter.on('git-stdin-error', this.onStdinError);
    this.emitter.on('slow-spawns', this.onSick); // not currently used to avoid clogging up ipc channel
    // keeping it around as it's potentially useful for avoiding duplicate write operations upon renderer crashing

    this.emitter.on('exec-started', this.onExecStarted);
    this.subscriptions.add(new _eventKit.Disposable(() => _electron.ipcRenderer.removeListener(Worker.channelName, handleMessages)));
  }

  executeOperation(operation) {
    return operation.execute(payload => {
      if (this.destroyed) {
        return null;
      }

      return this.webContents.send(Worker.channelName, {
        type: 'git-exec',
        data: payload
      });
    });
  }

  cancelOperation(operation) {
    return operation.cancel(payload => {
      if (this.destroyed) {
        return null;
      }

      return this.webContents.send(Worker.channelName, {
        type: 'git-cancel',
        data: payload
      });
    });
  }

  getPid() {
    return this.pid;
  }

  getReadyPromise() {
    return this.readyPromise;
  }

  destroy() {
    this.destroyed = true;
    this.subscriptions.dispose();
  }

}

exports.RendererProcess = RendererProcess;

class Operation {
  constructor(data, resolve, reject) {
    this.id = Operation.id++;
    this.data = data;
    this.resolve = resolve;
    this.reject = reject;
    this.promise = null;

    this.cancellationResolve = () => {};

    this.startTime = null;
    this.endTime = null;
    this.status = Operation.status.PENDING;
    this.results = null;
    this.emitter = new _eventKit.Emitter();
  }

  onComplete(cb) {
    return this.emitter.on('complete', cb);
  }

  setPromise(promise) {
    this.promise = promise;
  }

  getPromise() {
    return this.promise;
  }

  setInProgress() {
    // after exec has been called but before results a received
    this.status = Operation.status.INPROGRESS;
  }

  getExecutionTime() {
    if (!this.startTime || !this.endTime) {
      return NaN;
    } else {
      return this.endTime - this.startTime;
    }
  }

  complete(results, mutate = data => data) {
    this.endTime = performance.now();
    this.results = results;
    this.resolve(mutate(results));
    this.cancellationResolve();
    this.status = Operation.status.COMPLETE;
    this.emitter.emit('complete', this);
    this.emitter.dispose();
  }

  wasCancelled() {
    this.status = Operation.status.CANCELLED;
    this.cancellationResolve();
  }

  error(results) {
    this.endTime = performance.now();
    const err = new Error(results.message, results.fileName, results.lineNumber);
    err.stack = results.stack;
    this.reject(err);
  }

  execute(execFn) {
    this.startTime = performance.now();
    return execFn(_objectSpread2({}, this.data, {
      id: this.id
    }));
  }

  cancel(execFn) {
    return new Promise(resolve => {
      this.status = Operation.status.CANCELLING;
      this.cancellationResolve = resolve;
      execFn({
        id: this.id
      });
    });
  }

}

exports.Operation = Operation;

_defineProperty(Operation, "status", {
  PENDING: Symbol('pending'),
  INPROGRESS: Symbol('in-progress'),
  COMPLETE: Symbol('complete'),
  CANCELLING: Symbol('cancelling'),
  CANCELLED: Symbol('canceled')
});

_defineProperty(Operation, "id", 0);
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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