"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const natural = require("natural");
const SpellingManager_1 = require("./SpellingManager");
const TokenCheckStatus_1 = require("./TokenCheckStatus");
/**
 * A token-based spelling manager that uses non-processed list of words to
 * provides correctness testing and suggestions. This has both case-sensitive
 * and -insensitive methods along with suggestions that are capitalized based
 * on the incorrect word.
 */
class TokenSpellingManager extends SpellingManager_1.SpellingManager {
    constructor() {
        super(...arguments);
        this.maximumDistance = 0.9;
        this.sensitive = {};
        this.insensitive = {};
    }
    /**
     * Adds a word to the manager. If the word is in all lowercase, then it is
     * added as a case insensitive word, otherwise it is added as a case
     * sensitive result. If a token starts with "!", then it is automatically
     * case-sensitive.
     */
    add(token) {
        // If we aren't an array, then wrap it in an array.
        let tokens;
        if (typeof token === "string") {
            tokens = [token];
        }
        else {
            tokens = token;
        }
        // Loop through all the tokens and add each one.
        for (const t of tokens) {
            if (t && t.trim() !== "") {
                // If we have at least one uppercase character, we are
                // considered case sensitive. We don't test for lowercase
                // because we want to ignore things like single quotes for
                // posessives or contractions.
                if (/[A-Z]/.test(t)) {
                    this.addCaseSensitive(t);
                }
                else if (/^!/.test(t)) {
                    this.addCaseSensitive(t.substring(1));
                }
                else {
                    this.addCaseInsensitive(t);
                }
            }
        }
    }
    /**
     * Adds a case-sensitive token, if it hasn't already been added.
     *
     * There is no check to see if this token is already in the case-insensitive
     * list.
     */
    addCaseSensitive(token) {
        if (token && token.trim() !== "") {
            this.sensitive[token] = true;
        }
    }
    /**
     * Adds a case-insensitive token, if it hasn't already been added.
     *
     * There is no check to see if this token is already in the case-sensitive
     * list.
     */
    addCaseInsensitive(token) {
        if (token && token.trim() !== "") {
            this.insensitive[token.toLowerCase()] = true;
        }
    }
    /**
     * Checks the token to determine if it is correct or incorrect.
     */
    check(token) {
        if (token in this.sensitive) {
            return TokenCheckStatus_1.TokenCheckStatus.Correct;
        }
        if (token.toLowerCase() in this.insensitive) {
            return TokenCheckStatus_1.TokenCheckStatus.Correct;
        }
        return TokenCheckStatus_1.TokenCheckStatus.Unknown;
    }
    /**
     * Lists all of the words in a combined list appropriate for adding back
     * into the manager.
     */
    list() {
        // Gather up the list of sensitive items, prefixing with "!" for those
        // which would normally be in the case-insensitive list if they were
        // re-add()ed.
        const list = new Array();
        for (const token in this.sensitive) {
            if (token === token.toLowerCase()) {
                list.push("!" + token);
            }
            else {
                list.push(token);
            }
        }
        // Add in the insensitive items.
        for (const token in this.insensitive) {
            list.push(token);
        }
        // Sort the results because we always produce sorted results. Then
        // return it.
        list.sort();
        return list;
    }
    /**
     * Removes tokens, if it has been added.
     */
    remove(token) {
        if (token && token.trim() !== "") {
            this.removeCaseSensitive(token);
            this.removeCaseInsensitive(token.toLowerCase());
        }
    }
    /**
     * Removes a case-sensitive token, if it has been added.
     */
    removeCaseSensitive(token) {
        if (token && token.trim() !== "") {
            delete this.sensitive[token];
        }
    }
    /**
     * Removes a case-insensitive token, if it has been added.
     */
    removeCaseInsensitive(token) {
        if (token && token.trim() !== "") {
            delete this.insensitive[token];
        }
    }
    /**
     * Gives a suggestion for a token, sorted by likelyhood with the first item
     * in the resulting array being the most likely.
     */
    suggest(input) {
        // If the input is blank or null, then we don't have a suggestion.
        if (!input || input.trim().length === 0) {
            return [];
        }
        // Gather up all the suggestions from the case-sensitive list.
        const weights = [];
        for (const token in this.sensitive) {
            const distance = natural.JaroWinklerDistance(input, token);
            if (distance >= this.maximumDistance) {
                weights.push({ distance, token });
            }
        }
        // Also gather up the weights from the insensitive list. When we go
        // through this one, we try to find the "best" approach which means if
        // the input is all uppercase, then we compare that. Otherwise, we try
        // initial capital, and finally we see if lowercase would work better.
        for (const token in this.insensitive) {
            // Figure out the best approah.
            let test = token;
            if (/[A-Z].*[A-Z]/.test(input)) {
                test = test.toUpperCase();
            }
            else if (/^[A-Z]/.test(input)) {
                test = test.charAt(0).toUpperCase() + test.slice(1);
            }
            // Figure out the distance as above.
            const distance = natural.JaroWinklerDistance(input, test);
            if (distance >= this.maximumDistance) {
                weights.push({ distance, token: test });
            }
        }
        // Sort the list based on the distances. This will have the first key
        // be the highest distance.
        const keys = Object.keys(weights).sort((key1, key2) => {
            const value1 = weights[key1];
            const value2 = weights[key2];
            if (value1.distance !== value2.distance) {
                return value1.distance - value2.distance;
            }
            return value1.token.localeCompare(value2.token);
        });
        // Go through the resulting items and pull out an ordered list.
        const results = [];
        for (const key of keys) {
            results.push(weights[key].token);
        }
        return results;
    }
}
exports.TokenSpellingManager = TokenSpellingManager;
//# sourceMappingURL=TokenSpellingManager.js.map