"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.sendConversationList = void 0;
const electron_better_ipc_1 = require("electron-better-ipc");
const elementReady = require("element-ready");
const selectors_1 = require("./selectors");
const browser_1 = require("../browser");
const icon = {
    read: 'data-caprine-icon',
    unread: 'data-caprine-icon-unread'
};
const padding = {
    top: 3,
    right: 0,
    bottom: 3,
    left: 0
};
function drawIcon(size, img) {
    const canvas = document.createElement('canvas');
    if (img) {
        canvas.width = size + padding.left + padding.right;
        canvas.height = size + padding.top + padding.bottom;
        const ctx = canvas.getContext('2d');
        ctx.beginPath();
        ctx.arc((size / 2) + padding.left, (size / 2) + padding.top, (size / 2), 0, Math.PI * 2, true);
        ctx.closePath();
        ctx.clip();
        ctx.drawImage(img, padding.left, padding.top, size, size);
    }
    else {
        canvas.width = 0;
        canvas.height = 0;
    }
    return canvas;
}
// Return canvas with rounded image
async function urlToCanvas(url, size) {
    return new Promise(resolve => {
        const img = new Image();
        img.setAttribute('crossorigin', 'anonymous');
        img.addEventListener('load', () => {
            resolve(drawIcon(size, img));
        });
        img.addEventListener('error', () => {
            console.error('Image not found', url);
            resolve(drawIcon(size));
        });
        img.src = url;
    });
}
async function createIcons(element, url) {
    const canvas = await urlToCanvas(url, 50);
    element.setAttribute(icon.read, canvas.toDataURL());
    const markerSize = 8;
    const ctx = canvas.getContext('2d');
    ctx.fillStyle = '#f42020';
    ctx.beginPath();
    ctx.ellipse(canvas.width - markerSize, markerSize, markerSize, markerSize, 0, 0, 2 * Math.PI);
    ctx.closePath();
    ctx.fill();
    element.setAttribute(icon.unread, canvas.toDataURL());
}
async function discoverIcons(isNewDesign, element) {
    if (isNewDesign) {
        if (element) {
            return createIcons(element, element.getAttribute('xlink:href'));
        }
    }
    else {
        const profilePicElement = element.querySelector('img:first-of-type');
        if (profilePicElement) {
            return createIcons(element, profilePicElement.src);
        }
        const groupPicElement = element.firstElementChild;
        if (groupPicElement) {
            const groupPicBackground = groupPicElement.style.backgroundImage;
            if (groupPicBackground) {
                return createIcons(element, groupPicBackground.replace(/^url\(["']?(.*?)["']?\)$/, '$1'));
            }
        }
    }
    console.warn('Could not discover profile picture. Falling back to default image.');
    // Fall back to messenger favicon
    const messengerIcon = document.querySelector('link[rel~="icon"]');
    if (messengerIcon) {
        return createIcons(element, messengerIcon.getAttribute('href'));
    }
    // Fall back to facebook favicon
    return createIcons(element, 'https://facebook.com/favicon.ico');
}
async function getIcon(isNewDesign, element, unread) {
    if (!element.getAttribute(icon.read)) {
        await discoverIcons(isNewDesign, element);
    }
    return element.getAttribute(unread ? icon.unread : icon.read);
}
async function createConversation(element) {
    const conversation = {};
    const muted = element.classList.contains('_569x');
    conversation.selected = element.classList.contains('_1ht2');
    conversation.unread = !muted && element.getAttribute('aria-live') !== null;
    const profileElement = element.querySelector('div[data-tooltip-content]');
    conversation.label = profileElement.getAttribute('data-tooltip-content');
    conversation.icon = await getIcon(false, profileElement, conversation.unread);
    return conversation;
}
async function getLabel(element) {
    var _a, _b, _c;
    const emojis = [...element.children];
    for (const emoji of emojis) {
        emoji.outerHTML = (_b = (_a = emoji.querySelector('img')) === null || _a === void 0 ? void 0 : _a.getAttribute('alt')) !== null && _b !== void 0 ? _b : '';
    }
    return (_c = element.textContent) !== null && _c !== void 0 ? _c : '';
}
async function createConversationNewDesign(element) {
    const conversation = {};
    // TODO: Exclude muted conversations
    /*
    const muted = Boolean(element.querySelector(selectors.muteIconNewDesign));
    */
    conversation.selected = Boolean(element.querySelector('[role=row] [role=link] > div:only-child'));
    conversation.unread = Boolean(element.querySelector('[aria-label="Mark as Read"]'));
    const unparsedLabel = element.querySelector('.j83agx80.cbu4d94t.ew0dbk1b.irj2b8pg span.a8c37x1j.ni8dbmo4.stjgntxs.l9j0dhe7.ltmttdrg.g0qnabr5 > span > span');
    conversation.label = await getLabel(unparsedLabel);
    const iconElement = element.querySelector('[role=row] [role=link] [role=img] image');
    conversation.icon = await getIcon(true, iconElement, conversation.unread);
    return conversation;
}
async function createConversationList(isNewDesign) {
    const conversationListSelector = isNewDesign ? selectors_1.default.conversationListNewDesign : selectors_1.default.conversationList;
    const list = await elementReady(conversationListSelector, {
        stopOnDomReady: false
    });
    if (!list) {
        console.error('Could not find conversation list', conversationListSelector);
        return [];
    }
    const elements = [...list.children];
    if (isNewDesign) {
        // Remove last element from childer list on new design
        elements.splice(-1, 1);
    }
    const conversations = await Promise.all(elements.map(async (element) => isNewDesign ? createConversationNewDesign(element) : createConversation(element)));
    return conversations;
}
async function sendConversationList(isNewDesign) {
    const conversationsToRender = await createConversationList(isNewDesign);
    electron_better_ipc_1.ipcRenderer.callMain('conversations', conversationsToRender);
}
exports.sendConversationList = sendConversationList;
window.addEventListener('load', async () => {
    const sidebar = await elementReady('[role=navigation]', { stopOnDomReady: false });
    const newDesign = await (0, browser_1.isNewDesign)();
    if (sidebar) {
        const conversationListObserver = new MutationObserver(async () => sendConversationList(newDesign));
        conversationListObserver.observe(sidebar, {
            subtree: true,
            childList: true,
            attributes: true,
            attributeFilter: ['class']
        });
    }
});
