"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const path = require("path");
const electron_1 = require("electron");
const electron_util_1 = require("electron-util");
const config_1 = require("./config");
const menu_bar_mode_1 = require("./menu-bar-mode");
let tray;
let previousMessageCount = 0;
let contextMenu;
exports.default = {
    create: (win) => {
        if (tray) {
            return;
        }
        function toggleWindow() {
            if (win.isVisible()) {
                win.hide();
            }
            else {
                if (config_1.default.get('lastWindowState').isMaximized) {
                    win.maximize();
                    win.focus();
                }
                else {
                    win.show();
                }
                // Workaround for https://github.com/electron/electron/issues/20858
                // `setAlwaysOnTop` stops working after hiding the window on KDE Plasma.
                const alwaysOnTopMenuItem = electron_1.Menu.getApplicationMenu().getMenuItemById('always-on-top');
                win.setAlwaysOnTop(alwaysOnTopMenuItem.checked);
            }
        }
        const macosMenuItems = electron_util_1.is.macos ?
            [
                {
                    label: 'Disable Menu Bar Mode',
                    click() {
                        config_1.default.set('menuBarMode', false);
                        (0, menu_bar_mode_1.toggleMenuBarMode)(win);
                    }
                },
                {
                    label: 'Show Dock Icon',
                    type: 'checkbox',
                    checked: config_1.default.get('showDockIcon'),
                    click(menuItem) {
                        config_1.default.set('showDockIcon', menuItem.checked);
                        if (menuItem.checked) {
                            electron_1.app.dock.show();
                        }
                        else {
                            electron_1.app.dock.hide();
                        }
                        const dockMenuItem = contextMenu.getMenuItemById('dockMenu');
                        dockMenuItem.visible = !menuItem.checked;
                    }
                },
                {
                    type: 'separator'
                },
                {
                    id: 'dockMenu',
                    label: 'Menu',
                    visible: !config_1.default.get('showDockIcon'),
                    submenu: electron_1.Menu.getApplicationMenu()
                }
            ] : [];
        contextMenu = electron_1.Menu.buildFromTemplate([
            {
                label: 'Toggle',
                visible: !electron_util_1.is.macos,
                click() {
                    toggleWindow();
                }
            },
            ...macosMenuItems,
            {
                type: 'separator'
            },
            {
                role: 'quit'
            }
        ]);
        tray = new electron_1.Tray(getIconPath(false));
        tray.setContextMenu(contextMenu);
        updateToolTip(0);
        const trayClickHandler = () => {
            if (!win.isFullScreen()) {
                toggleWindow();
            }
        };
        tray.on('click', trayClickHandler);
        tray.on('double-click', trayClickHandler);
        tray.on('right-click', () => {
            tray === null || tray === void 0 ? void 0 : tray.popUpContextMenu(contextMenu);
        });
    },
    destroy: () => {
        // Workaround for https://github.com/electron/electron/issues/14036
        setTimeout(() => {
            tray === null || tray === void 0 ? void 0 : tray.destroy();
            tray = undefined;
        }, 500);
    },
    update: (messageCount) => {
        if (!tray || previousMessageCount === messageCount) {
            return;
        }
        previousMessageCount = messageCount;
        tray.setImage(getIconPath(messageCount > 0));
        updateToolTip(messageCount);
    },
    setBadge: (shouldDisplayUnread) => {
        if (electron_util_1.is.macos || !tray) {
            return;
        }
        tray.setImage(getIconPath(shouldDisplayUnread));
    }
};
function updateToolTip(counter) {
    if (!tray) {
        return;
    }
    let tooltip = electron_1.app.name;
    if (counter > 0) {
        tooltip += `- ${counter} unread ${counter === 1 ? 'message' : 'messages'}`;
    }
    tray.setToolTip(tooltip);
}
function getIconPath(hasUnreadMessages) {
    const icon = electron_util_1.is.macos ?
        getMacOSIconName(hasUnreadMessages) :
        getNonMacOSIconName(hasUnreadMessages);
    return path.join(__dirname, '..', `static/${icon}`);
}
function getNonMacOSIconName(hasUnreadMessages) {
    return hasUnreadMessages ? 'IconTrayUnread.png' : 'IconTray.png';
}
function getMacOSIconName(hasUnreadMessages) {
    return hasUnreadMessages ? 'IconMenuBarUnreadTemplate.png' : 'IconMenuBarTemplate.png';
}
