/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.xpack.security.cli;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.PrintWriter;
import java.io.UncheckedIOException;
import java.io.Writer;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.PosixFileAttributeView;
import java.nio.file.attribute.PosixFilePermission;
import java.security.GeneralSecurityException;
import java.security.KeyPair;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.PrivateKey;
import java.security.cert.Certificate;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;
import java.time.Period;
import java.time.ZoneOffset;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeParseException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;
import javax.security.auth.x500.X500Principal;
import joptsimple.OptionParser;
import joptsimple.OptionSet;
import org.bouncycastle.asn1.DERIA5String;
import org.bouncycastle.asn1.x509.GeneralNames;
import org.bouncycastle.cert.CertIOException;
import org.bouncycastle.openssl.jcajce.JcaMiscPEMGenerator;
import org.bouncycastle.openssl.jcajce.JcaPEMWriter;
import org.bouncycastle.operator.OperatorCreationException;
import org.bouncycastle.operator.OperatorException;
import org.bouncycastle.pkcs.PKCS10CertificationRequest;
import org.bouncycastle.util.io.pem.PemObjectGenerator;
import org.elasticsearch.ElasticsearchException;
import org.elasticsearch.Version;
import org.elasticsearch.cli.EnvironmentAwareCommand;
import org.elasticsearch.cli.SuppressForbidden;
import org.elasticsearch.cli.Terminal;
import org.elasticsearch.cli.UserException;
import org.elasticsearch.common.Strings;
import org.elasticsearch.common.collect.MapBuilder;
import org.elasticsearch.common.io.PathUtils;
import org.elasticsearch.common.io.Streams;
import org.elasticsearch.common.network.InetAddresses;
import org.elasticsearch.common.util.set.Sets;
import org.elasticsearch.env.Environment;
import org.elasticsearch.xpack.core.ssl.CertParsingUtils;
import org.elasticsearch.xpack.core.ssl.PemUtils;
import org.elasticsearch.xpack.security.cli.CertGenUtils;
import org.elasticsearch.xpack.security.cli.CertificateTool;

class HttpCertificateCommand
extends EnvironmentAwareCommand {
    static final int DEFAULT_CERT_KEY_SIZE = 2048;
    static final Period DEFAULT_CERT_VALIDITY = Period.ofYears(5);
    static final X500Principal DEFAULT_CA_NAME = new X500Principal("CN=Elasticsearch HTTP CA");
    static final int DEFAULT_CA_KEY_SIZE = 2048;
    static final Period DEFAULT_CA_VALIDITY = DEFAULT_CERT_VALIDITY;
    private static final String ES_README_CSR = "es-readme-csr.txt";
    private static final String ES_YML_CSR = "es-sample-csr.yml";
    private static final String ES_README_P12 = "es-readme-p12.txt";
    private static final String ES_YML_P12 = "es-sample-p12.yml";
    private static final String CA_README_P12 = "ca-readme-p12.txt";
    private static final String KIBANA_README = "kibana-readme.txt";
    private static final String KIBANA_YML = "kibana-sample.yml";
    private static final byte[] MAGIC_BYTES1_PKCS12 = new byte[]{48, -126};
    private static final byte[] MAGIC_BYTES2_PKCS12 = new byte[]{48, 86};
    private static final byte[] MAGIC_BYTES_JKS = new byte[]{-2, -19};

    HttpCertificateCommand() {
        super("generate a new certificate (or certificate request) for the Elasticsearch HTTP interface");
    }

    protected void execute(Terminal terminal, OptionSet options, Environment env) throws Exception {
        char[] password;
        Period validity;
        CertificateTool.CAInfo caInfo;
        this.printHeader("Elasticsearch HTTP Certificate Utility", terminal);
        terminal.println("The 'http' command guides you through the process of generating certificates");
        terminal.println("for use on the HTTP (Rest) interface for Elasticsearch.");
        terminal.println("");
        terminal.println("This tool will ask you a number of questions in order to generate the right");
        terminal.println("set of files for your needs.");
        boolean csr = this.askCertSigningRequest(terminal);
        if (csr) {
            caInfo = null;
            validity = null;
        } else {
            boolean existingCa = this.askExistingCertificateAuthority(terminal);
            caInfo = existingCa ? this.findExistingCA(terminal, env) : this.createNewCA(terminal);
            terminal.println(Terminal.Verbosity.VERBOSE, "Using the following CA:");
            terminal.println(Terminal.Verbosity.VERBOSE, "\tSubject: " + caInfo.certAndKey.cert.getSubjectX500Principal());
            terminal.println(Terminal.Verbosity.VERBOSE, "\tIssuer: " + caInfo.certAndKey.cert.getIssuerX500Principal());
            terminal.println(Terminal.Verbosity.VERBOSE, "\tSerial: " + caInfo.certAndKey.cert.getSerialNumber());
            terminal.println(Terminal.Verbosity.VERBOSE, "\tExpiry: " + caInfo.certAndKey.cert.getNotAfter());
            terminal.println(Terminal.Verbosity.VERBOSE, "\tSignature Algorithm: " + caInfo.certAndKey.cert.getSigAlgName());
            validity = this.getCertificateValidityPeriod(terminal);
        }
        boolean multipleCertificates = this.askMultipleCertificates(terminal);
        ArrayList<CertOptions> certificates = new ArrayList<CertOptions>();
        String nodeDescription = multipleCertificates ? "node #1" : "your nodes";
        while (true) {
            CertOptions cert = this.getCertificateConfiguration(terminal, multipleCertificates, nodeDescription, validity, csr);
            terminal.println(Terminal.Verbosity.VERBOSE, "Generating the following " + (csr ? "CSR" : "Certificate") + ":");
            terminal.println(Terminal.Verbosity.VERBOSE, "\tName: " + cert.name);
            terminal.println(Terminal.Verbosity.VERBOSE, "\tSubject: " + cert.subject);
            terminal.println(Terminal.Verbosity.VERBOSE, "\tDNS Names: " + Strings.collectionToCommaDelimitedString(cert.dnsNames));
            terminal.println(Terminal.Verbosity.VERBOSE, "\tIP Names: " + Strings.collectionToCommaDelimitedString(cert.ipNames));
            terminal.println(Terminal.Verbosity.VERBOSE, "\tKey Size: " + cert.keySize);
            terminal.println(Terminal.Verbosity.VERBOSE, "\tValidity: " + HttpCertificateCommand.toString(cert.validity));
            certificates.add(cert);
            if (!multipleCertificates || !terminal.promptYesNo("Generate additional certificates?", true)) break;
            nodeDescription = "node #" + (certificates.size() + 1);
        }
        this.printHeader("What password do you want for your private key(s)?", terminal);
        if (csr) {
            terminal.println("Your private key(s) will be stored as a PEM formatted file.");
            terminal.println("We recommend that you protect your private keys with a password");
            terminal.println("");
            terminal.println("If you do not wish to use a password, simply press <enter> at the prompt below.");
            password = this.readPassword(terminal, "Provide a password for the private key: ", true);
        } else {
            terminal.println("Your private key(s) will be stored in a PKCS#12 keystore file named \"http.p12\".");
            terminal.println("This type of keystore is always password protected, but it is possible to use a");
            terminal.println("blank password.");
            terminal.println("");
            terminal.println("If you wish to use a blank password, simply press <enter> at the prompt below.");
            password = this.readPassword(terminal, "Provide a password for the \"http.p12\" file: ", true);
        }
        this.printHeader("Where should we save the generated files?", terminal);
        if (csr) {
            terminal.println("A number of files will be generated including your private key(s),");
            terminal.println("certificate request(s), and sample configuration options for Elastic Stack products.");
        } else {
            terminal.println("A number of files will be generated including your private key(s),");
            terminal.println("public certificate(s), and sample configuration options for Elastic Stack products.");
        }
        terminal.println("");
        terminal.println("These files will be included in a single zip archive.");
        terminal.println("");
        Path output = this.resolvePath("elasticsearch-ssl-http.zip");
        output = this.tryReadInput(terminal, "What filename should be used for the output zip file?", output, this::resolvePath);
        this.writeZip(output, password, caInfo, certificates, env);
        terminal.println("");
        terminal.println("Zip file written to " + output);
    }

    @SuppressForbidden(reason="CLI tool resolves files against working directory")
    protected Path resolvePath(String name) {
        return PathUtils.get((String)name, (String[])new String[0]).normalize().toAbsolutePath();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void writeZip(Path file, char[] password, CertificateTool.CAInfo caInfo, List<CertOptions> certificates, Environment env) throws UserException {
        if (Files.exists(file, new LinkOption[0])) {
            throw new UserException(74, "Output file '" + file + "' already exists");
        }
        boolean success = false;
        try {
            try (OutputStream fileStream = Files.newOutputStream(file, StandardOpenOption.CREATE_NEW);
                 ZipOutputStream zipStream = new ZipOutputStream(fileStream, StandardCharsets.UTF_8);){
                this.createZipDirectory(zipStream, "elasticsearch");
                if (certificates.size() == 1) {
                    this.writeCertificateAndKeyDetails(zipStream, "elasticsearch", certificates.get(0), caInfo, password, env);
                } else {
                    for (CertOptions cert : certificates) {
                        String dirName = "elasticsearch/" + cert.name;
                        this.createZipDirectory(zipStream, dirName);
                        this.writeCertificateAndKeyDetails(zipStream, dirName, cert, caInfo, password, env);
                    }
                }
                if (caInfo != null && caInfo.generated) {
                    this.createZipDirectory(zipStream, "ca");
                    this.writeCertificateAuthority(zipStream, "ca", caInfo, env);
                }
                this.createZipDirectory(zipStream, "kibana");
                this.writeKibanaInfo(zipStream, "kibana", caInfo, env);
                PosixFileAttributeView view = Files.getFileAttributeView(file, PosixFileAttributeView.class, new LinkOption[0]);
                if (view != null) {
                    view.setPermissions(Sets.newHashSet((Object[])new PosixFilePermission[]{PosixFilePermission.OWNER_READ, PosixFilePermission.OWNER_WRITE}));
                }
                success = true;
            }
            finally {
                if (!success) {
                    Files.deleteIfExists(file);
                }
            }
        }
        catch (IOException e) {
            throw new ElasticsearchException("Failed to write ZIP file '" + file + "'", (Throwable)e, new Object[0]);
        }
    }

    private void createZipDirectory(ZipOutputStream zip, String name) throws IOException {
        ZipEntry entry = new ZipEntry(name + "/");
        assert (entry.isDirectory());
        zip.putNextEntry(entry);
    }

    private void writeCertificateAndKeyDetails(ZipOutputStream zip, String dirName, CertOptions cert, CertificateTool.CAInfo ca, char[] password, Environment env) {
        try {
            boolean hasPassword;
            KeyPair keyPair = CertGenUtils.generateKeyPair(cert.keySize);
            GeneralNames sanList = CertificateTool.getSubjectAlternativeNamesValue(cert.ipNames, cert.dnsNames, Collections.emptyList());
            boolean bl = hasPassword = password != null && password.length > 0;
            if (ca == null) {
                PKCS10CertificationRequest csr = CertGenUtils.generateCSR(keyPair, cert.subject, sanList);
                String csrFile = "http-" + cert.name + ".csr";
                String keyFile = "http-" + cert.name + ".key";
                String certName = "http-" + cert.name + ".crt";
                String ymlFile = "sample-elasticsearch.yml";
                Map<String, String> substitutions = this.buildSubstitutions(env, MapBuilder.newMapBuilder().put((Object)"CSR", (Object)csrFile).put((Object)"KEY", (Object)keyFile).put((Object)"CERT", (Object)certName).put((Object)"YML", (Object)"sample-elasticsearch.yml").put((Object)"PASSWORD", (Object)(hasPassword ? "*" : "")).immutableMap());
                this.writeTextFile(zip, dirName + "/README.txt", ES_README_CSR, substitutions);
                this.writePemEntry(zip, dirName + "/" + csrFile, (PemObjectGenerator)new JcaMiscPEMGenerator((Object)csr));
                this.writePemEntry(zip, dirName + "/" + keyFile, (PemObjectGenerator)this.generator(keyPair.getPrivate(), password));
                this.writeTextFile(zip, dirName + "/" + "sample-elasticsearch.yml", ES_YML_CSR, substitutions);
            } else {
                ZonedDateTime notBefore = ZonedDateTime.now(ZoneOffset.UTC);
                ZonedDateTime notAfter = notBefore.plus(cert.validity);
                X509Certificate certificate = CertGenUtils.generateSignedCertificate(cert.subject, sanList, keyPair, ca.certAndKey.cert, ca.certAndKey.key, false, notBefore, notAfter, null);
                String p12Name = "http.p12";
                String ymlFile = "sample-elasticsearch.yml";
                Map<String, String> substitutions = this.buildSubstitutions(env, MapBuilder.newMapBuilder().put((Object)"P12", (Object)"http.p12").put((Object)"YML", (Object)"sample-elasticsearch.yml").put((Object)"PASSWORD", (Object)(hasPassword ? "*" : "")).immutableMap());
                this.writeTextFile(zip, dirName + "/README.txt", ES_README_P12, substitutions);
                this.writeKeyStore(zip, dirName + "/" + "http.p12", certificate, keyPair.getPrivate(), password, ca.certAndKey.cert);
                this.writeTextFile(zip, dirName + "/" + "sample-elasticsearch.yml", ES_YML_P12, substitutions);
            }
        }
        catch (IOException | GeneralSecurityException | OperatorException e) {
            throw new ElasticsearchException("Failed to write certificate to ZIP file", e, new Object[0]);
        }
    }

    private void writeCertificateAuthority(ZipOutputStream zip, String dirName, CertificateTool.CAInfo ca, Environment env) {
        assert (ca != null);
        assert (ca.generated);
        try {
            this.writeTextFile(zip, dirName + "/README.txt", CA_README_P12, this.buildSubstitutions(env, MapBuilder.newMapBuilder().put((Object)"P12", (Object)"ca.p12").put((Object)"DN", (Object)ca.certAndKey.cert.getSubjectX500Principal().getName()).put((Object)"PASSWORD", (Object)(ca.password == null || ca.password.length == 0 ? "" : "*")).immutableMap()));
            KeyStore pkcs12 = KeyStore.getInstance("PKCS12");
            pkcs12.load(null);
            pkcs12.setKeyEntry("ca", ca.certAndKey.key, ca.password, new Certificate[]{ca.certAndKey.cert});
            try (ZipEntryStream entry = new ZipEntryStream(zip, dirName + "/ca.p12");){
                pkcs12.store(entry, ca.password);
            }
        }
        catch (IOException | KeyStoreException | NoSuchAlgorithmException | CertificateException e) {
            throw new ElasticsearchException("Failed to write CA to ZIP file", (Throwable)e, new Object[0]);
        }
    }

    private void writeKibanaInfo(ZipOutputStream zip, String dirName, CertificateTool.CAInfo ca, Environment env) {
        String caCertName = "elasticsearch-ca.pem";
        String caCert = ca == null ? "" : "elasticsearch-ca.pem";
        String ymlFile = "sample-kibana.yml";
        Map<String, String> substitutions = this.buildSubstitutions(env, MapBuilder.newMapBuilder().put((Object)"CA_CERT_NAME", (Object)"elasticsearch-ca.pem").put((Object)"CA_CERT", (Object)caCert).put((Object)"YML", (Object)"sample-kibana.yml").immutableMap());
        try {
            this.writeTextFile(zip, dirName + "/README.txt", KIBANA_README, substitutions);
            if (ca != null) {
                this.writePemEntry(zip, dirName + "/" + caCert, (PemObjectGenerator)new JcaMiscPEMGenerator((Object)ca.certAndKey.cert));
            }
            this.writeTextFile(zip, dirName + "/" + "sample-kibana.yml", KIBANA_YML, substitutions);
        }
        catch (IOException e) {
            throw new ElasticsearchException("Failed to write Kibana details ZIP file", (Throwable)e, new Object[0]);
        }
    }

    private void writeTextFile(ZipOutputStream zip, String outputName, String resource, Map<String, String> substitutions) {
        try (InputStream stream = ((Object)((Object)this)).getClass().getResourceAsStream("certutil-http/" + resource);
             ZipEntryStream entry = new ZipEntryStream(zip, outputName);
             OutputStreamWriter osw = new OutputStreamWriter((OutputStream)entry, StandardCharsets.UTF_8);
             PrintWriter writer = new PrintWriter((Writer)osw, false);){
            if (stream == null) {
                throw new IllegalStateException("Cannot find internal resource " + resource);
            }
            HttpCertificateCommand.copyWithSubstitutions(stream, writer, substitutions);
            writer.flush();
        }
        catch (IOException e) {
            throw new UncheckedIOException("Cannot add resource " + resource + " to zip file", e);
        }
    }

    static void copyWithSubstitutions(InputStream stream, PrintWriter writer, Map<String, String> substitutions) throws IOException {
        boolean skip = false;
        for (String line : Streams.readAllLines((InputStream)stream)) {
            for (Map.Entry<String, String> subst : substitutions.entrySet()) {
                line = line.replace("${" + subst.getKey() + "}", subst.getValue());
            }
            if (line.startsWith("#if ")) {
                String key = line.substring(4).trim();
                skip = Strings.isNullOrEmpty((String)substitutions.get(key));
                continue;
            }
            if (line.equals("#else")) {
                skip = !skip;
                continue;
            }
            if (line.equals("#endif")) {
                skip = false;
                continue;
            }
            if (skip) continue;
            writer.println(line);
        }
    }

    private Map<String, String> buildSubstitutions(Environment env, Map<String, String> entries) {
        HashMap<String, String> map = new HashMap<String, String>(entries.size() + 4);
        ZonedDateTime now = ZonedDateTime.now().withNano(0);
        map.put("DATE", now.format(DateTimeFormatter.ISO_LOCAL_DATE));
        map.put("TIME", now.format(DateTimeFormatter.ISO_OFFSET_TIME));
        map.put("VERSION", Version.CURRENT.toString());
        map.put("CONF_DIR", env.configFile().toAbsolutePath().toString());
        map.putAll(entries);
        return map;
    }

    private void writeKeyStore(ZipOutputStream zip, String name, Certificate certificate, PrivateKey key, char[] password, X509Certificate caCert) throws IOException, GeneralSecurityException {
        KeyStore pkcs12 = KeyStore.getInstance("PKCS12");
        pkcs12.load(null);
        pkcs12.setKeyEntry("http", key, password, new Certificate[]{certificate});
        if (caCert != null) {
            pkcs12.setCertificateEntry("ca", caCert);
        }
        try (ZipEntryStream entry = new ZipEntryStream(zip, name);){
            pkcs12.store(entry, password);
        }
    }

    private void writePemEntry(ZipOutputStream zip, String name, PemObjectGenerator generator) throws IOException {
        try (ZipEntryStream entry = new ZipEntryStream(zip, name);
             JcaPEMWriter pem = new JcaPEMWriter((Writer)new OutputStreamWriter((OutputStream)entry, StandardCharsets.UTF_8));){
            pem.writeObject(generator);
            pem.flush();
        }
    }

    private JcaMiscPEMGenerator generator(PrivateKey privateKey, char[] password) throws IOException {
        if (password == null || password.length == 0) {
            return new JcaMiscPEMGenerator((Object)privateKey);
        }
        return new JcaMiscPEMGenerator((Object)privateKey, CertificateTool.getEncrypter(password));
    }

    private Period getCertificateValidityPeriod(Terminal terminal) {
        this.printHeader("How long should your certificates be valid?", terminal);
        terminal.println("Every certificate has an expiry date. When the expiry date is reached clients");
        terminal.println("will stop trusting your certificate and TLS connections will fail.");
        terminal.println("");
        terminal.println("Best practice suggests that you should either:");
        terminal.println("(a) set this to a short duration (90 - 120 days) and have automatic processes");
        terminal.println("to generate a new certificate before the old one expires, or");
        terminal.println("(b) set it to a longer duration (3 - 5 years) and then perform a manual update");
        terminal.println("a few months before it expires.");
        terminal.println("");
        terminal.println("You may enter the validity period in years (e.g. 3Y), months (e.g. 18M), or days (e.g. 90D)");
        terminal.println("");
        return this.readPeriodInput(terminal, "For how long should your certificate be valid?", DEFAULT_CERT_VALIDITY, 60);
    }

    private boolean askMultipleCertificates(Terminal terminal) {
        this.printHeader("Do you wish to generate one certificate per node?", terminal);
        terminal.println("If you have multiple nodes in your cluster, then you may choose to generate a");
        terminal.println("separate certificate for each of these nodes. Each certificate will have its");
        terminal.println("own private key, and will be issued for a specific hostname or IP address.");
        terminal.println("");
        terminal.println("Alternatively, you may wish to generate a single certificate that is valid");
        terminal.println("across all the hostnames or addresses in your cluster.");
        terminal.println("");
        terminal.println("If all of your nodes will be accessed through a single domain");
        terminal.println("(e.g. node01.es.example.com, node02.es.example.com, etc) then you may find it");
        terminal.println("simpler to generate one certificate with a wildcard hostname (*.es.example.com)");
        terminal.println("and use that across all of your nodes.");
        terminal.println("");
        terminal.println("However, if you do not have a common domain name, and you expect to add");
        terminal.println("additional nodes to your cluster in the future, then you should generate a");
        terminal.println("certificate per node so that you can more easily generate new certificates when");
        terminal.println("you provision new nodes.");
        terminal.println("");
        return terminal.promptYesNo("Generate a certificate per node?", false);
    }

    private CertOptions getCertificateConfiguration(Terminal terminal, boolean multipleCertificates, String nodeDescription, Period validity, boolean csr) {
        String certName = null;
        if (multipleCertificates) {
            this.printHeader("What is the name of " + nodeDescription + "?", terminal);
            terminal.println("This name will be used as part of the certificate file name, and as a");
            terminal.println("descriptive name within the certificate.");
            terminal.println("");
            terminal.println("You can use any descriptive name that you like, but we recommend using the name");
            terminal.println("of the Elasticsearch node.");
            terminal.println("");
            nodeDescription = certName = terminal.readText(nodeDescription + " name: ");
        }
        this.printHeader("Which hostnames will be used to connect to " + nodeDescription + "?", terminal);
        terminal.println("These hostnames will be added as \"DNS\" names in the \"Subject Alternative Name\"");
        terminal.println("(SAN) field in your certificate.");
        terminal.println("");
        terminal.println("You should list every hostname and variant that people will use to connect to");
        terminal.println("your cluster over http.");
        terminal.println("Do not list IP addresses here, you will be asked to enter them later.");
        terminal.println("");
        terminal.println("If you wish to use a wildcard certificate (for example *.es.example.com) you");
        terminal.println("can enter that here.");
        ArrayList<String> dnsNames = new ArrayList<String>();
        while (true) {
            terminal.println("");
            terminal.println("Enter all the hostnames that you need, one per line.");
            terminal.println("When you are done, press <ENTER> once more to move on to the next step.");
            terminal.println("");
            dnsNames.addAll(this.readMultiLineInput(terminal, this::validateHostname));
            if (dnsNames.isEmpty()) {
                terminal.println(Terminal.Verbosity.SILENT, "You did not enter any hostnames.");
                terminal.println("Clients are likely to encounter TLS hostname verification errors if they");
                terminal.println("connect to your cluster using a DNS name.");
            } else {
                terminal.println(Terminal.Verbosity.SILENT, "You entered the following hostnames.");
                terminal.println(Terminal.Verbosity.SILENT, "");
                dnsNames.forEach(s -> terminal.println(Terminal.Verbosity.SILENT, " - " + s));
            }
            terminal.println("");
            if (terminal.promptYesNo("Is this correct", true)) break;
            dnsNames.clear();
        }
        this.printHeader("Which IP addresses will be used to connect to " + nodeDescription + "?", terminal);
        terminal.println("If your clients will ever connect to your nodes by numeric IP address, then you");
        terminal.println("can list these as valid IP \"Subject Alternative Name\" (SAN) fields in your");
        terminal.println("certificate.");
        terminal.println("");
        terminal.println("If you do not have fixed IP addresses, or not wish to support direct IP access");
        terminal.println("to your cluster then you can just press <ENTER> to skip this step.");
        ArrayList<String> ipNames = new ArrayList<String>();
        while (true) {
            terminal.println("");
            terminal.println("Enter all the IP addresses that you need, one per line.");
            terminal.println("When you are done, press <ENTER> once more to move on to the next step.");
            terminal.println("");
            ipNames.addAll(this.readMultiLineInput(terminal, this::validateIpAddress));
            if (ipNames.isEmpty()) {
                terminal.println(Terminal.Verbosity.SILENT, "You did not enter any IP addresses.");
            } else {
                terminal.println(Terminal.Verbosity.SILENT, "You entered the following IP addresses.");
                terminal.println(Terminal.Verbosity.SILENT, "");
                ipNames.forEach(s -> terminal.println(Terminal.Verbosity.SILENT, " - " + s));
            }
            terminal.println("");
            if (terminal.promptYesNo("Is this correct", true)) break;
            ipNames.clear();
        }
        this.printHeader("Other certificate options", terminal);
        terminal.println("The generated certificate will have the following additional configuration");
        terminal.println("values. These values have been selected based on a combination of the");
        terminal.println("information you have provided above and secure defaults. You should not need to");
        terminal.println("change these values unless you have specific requirements.");
        terminal.println("");
        if (certName == null) {
            certName = dnsNames.stream().filter(n -> n.indexOf(42) == -1).findFirst().orElseGet(() -> dnsNames.stream().map(s -> s.replace("*.", "")).findFirst().orElse("elasticsearch"));
        }
        X500Principal dn = this.buildDistinguishedName(certName);
        int keySize = 2048;
        while (true) {
            terminal.println(Terminal.Verbosity.SILENT, "Key Name: " + certName);
            terminal.println(Terminal.Verbosity.SILENT, "Subject DN: " + dn);
            terminal.println(Terminal.Verbosity.SILENT, "Key Size: " + keySize);
            terminal.println(Terminal.Verbosity.SILENT, "");
            if (!terminal.promptYesNo("Do you wish to change any of these options?", false)) break;
            this.printHeader("What should your key be named?", terminal);
            if (csr) {
                terminal.println("This will be included in the name of the files that are generated");
            } else {
                terminal.println("This will be the entry name in the PKCS#12 keystore that is generated");
            }
            terminal.println("It is helpful to have a meaningful name for this key");
            terminal.println("");
            certName = this.tryReadInput(terminal, "Key Name", certName, Function.identity());
            this.printHeader("What subject DN should be used for your certificate?", terminal);
            terminal.println("This will be visible to clients.");
            terminal.println("It is helpful to have a meaningful name for each certificate");
            terminal.println("");
            dn = this.tryReadInput(terminal, "Subject DN", dn, name -> {
                try {
                    if (name.contains("=")) {
                        return new X500Principal((String)name);
                    }
                    return new X500Principal("CN=" + name);
                }
                catch (IllegalArgumentException e) {
                    terminal.println(Terminal.Verbosity.SILENT, "'" + name + "' is not a valid DN (" + e.getMessage() + ")");
                    return null;
                }
            });
            this.printHeader("What key size should your certificate have?", terminal);
            terminal.println("The RSA private key for your certificate has a fixed 'key size' (in bits).");
            terminal.println("Larger key sizes are generally more secure, but are also slower.");
            terminal.println("");
            terminal.println("We recommend that you use one of 2048, 3072 or 4096 bits for your key.");
            keySize = this.readKeySize(terminal, keySize);
            terminal.println("");
        }
        return new CertOptions(certName, dn, dnsNames, ipNames, keySize, validity);
    }

    private String validateHostname(String name) {
        if (DERIA5String.isIA5String((String)name)) {
            return null;
        }
        return name + " is not a valid DNS name";
    }

    private String validateIpAddress(String ip) {
        if (InetAddresses.isInetAddress((String)ip)) {
            return null;
        }
        return ip + " is not a valid IP address";
    }

    private X500Principal buildDistinguishedName(String name) {
        return new X500Principal("CN=" + name.replace(".", ",DC="));
    }

    private List<String> readMultiLineInput(Terminal terminal, Function<String, String> validator) {
        String input;
        ArrayList<String> lines = new ArrayList<String>();
        while (!Strings.isEmpty((CharSequence)(input = terminal.readText("")))) {
            String error = validator.apply(input);
            if (error == null) {
                lines.add(input);
                continue;
            }
            terminal.println("Error: " + error);
        }
        return lines;
    }

    private boolean askCertSigningRequest(Terminal terminal) {
        this.printHeader("Do you wish to generate a Certificate Signing Request (CSR)?", terminal);
        terminal.println("A CSR is used when you want your certificate to be created by an existing");
        terminal.println("Certificate Authority (CA) that you do not control (that is, you don't have");
        terminal.println("access to the keys for that CA). ");
        terminal.println("");
        terminal.println("If you are in a corporate environment with a central security team, then you");
        terminal.println("may have an existing Corporate CA that can generate your certificate for you.");
        terminal.println("Infrastructure within your organisation may already be configured to trust this");
        terminal.println("CA, so it may be easier for clients to connect to Elasticsearch if you use a");
        terminal.println("CSR and send that request to the team that controls your CA.");
        terminal.println("");
        terminal.println("If you choose not to generate a CSR, this tool will generate a new certificate");
        terminal.println("for you. That certificate will be signed by a CA under your control. This is a");
        terminal.println("quick and easy way to secure your cluster with TLS, but you will need to");
        terminal.println("configure all your clients to trust that custom CA.");
        terminal.println("");
        return terminal.promptYesNo("Generate a CSR?", false);
    }

    private CertificateTool.CAInfo findExistingCA(Terminal terminal, Environment env) throws UserException {
        this.printHeader("What is the path to your CA?", terminal);
        terminal.println("Please enter the full pathname to the Certificate Authority that you wish to");
        terminal.println("use for signing your new http certificate. This can be in PKCS#12 (.p12), JKS");
        terminal.println("(.jks) or PEM (.crt, .key, .pem) format.");
        Path caPath = this.requestPath("CA Path: ", terminal, env, true);
        FileType fileType = HttpCertificateCommand.guessFileType(caPath, terminal);
        switch (fileType) {
            case PKCS12: 
            case JKS: {
                terminal.println(Terminal.Verbosity.VERBOSE, "CA file " + caPath + " appears to be a " + (Object)((Object)fileType) + " keystore");
                return this.readKeystoreCA(caPath, fileType, terminal);
            }
            case PEM_KEY: {
                this.printHeader("What is the path to your CA certificate?", terminal);
                terminal.println(caPath + " appears to be a PEM formatted private key file.");
                terminal.println("In order to use it for signing we also need access to the certificate");
                terminal.println("that corresponds to that key.");
                terminal.println("");
                Path caCertPath = this.requestPath("CA Certificate: ", terminal, env, true);
                return this.readPemCA(caCertPath, caPath, terminal);
            }
            case PEM_CERT: {
                this.printHeader("What is the path to your CA key?", terminal);
                terminal.println(caPath + " appears to be a PEM formatted certificate file.");
                terminal.println("In order to use it for signing we also need access to the private key");
                terminal.println("that corresponds to that certificate.");
                terminal.println("");
                Path caKeyPath = this.requestPath("CA Key: ", terminal, env, true);
                return this.readPemCA(caPath, caKeyPath, terminal);
            }
            case PEM_CERT_CHAIN: {
                terminal.println(Terminal.Verbosity.SILENT, "The file at " + caPath + " contains multiple certificates.");
                terminal.println("That type of file typically represents a certificate-chain");
                terminal.println("This tool requires a single certificate for the CA");
                throw new UserException(65, caPath + ": Unsupported file type (certificate chain)");
            }
        }
        terminal.println(Terminal.Verbosity.SILENT, "The file at " + caPath + " isn't a file type that this tool recognises.");
        terminal.println("Please try again with a CA in PKCS#12, JKS or PEM format");
        throw new UserException(65, caPath + ": Unrecognized file type");
    }

    private CertificateTool.CAInfo createNewCA(Terminal terminal) {
        terminal.println("A new Certificate Authority will be generated for you");
        this.printHeader("CA Generation Options", terminal);
        terminal.println("The generated certificate authority will have the following configuration values.");
        terminal.println("These values have been selected based on secure defaults.");
        terminal.println("You should not need to change these values unless you have specific requirements.");
        terminal.println("");
        X500Principal dn = DEFAULT_CA_NAME;
        Period validity = DEFAULT_CA_VALIDITY;
        int keySize = 2048;
        while (true) {
            terminal.println(Terminal.Verbosity.SILENT, "Subject DN: " + dn);
            terminal.println(Terminal.Verbosity.SILENT, "Validity: " + HttpCertificateCommand.toString(validity));
            terminal.println(Terminal.Verbosity.SILENT, "Key Size: " + keySize);
            terminal.println(Terminal.Verbosity.SILENT, "");
            if (!terminal.promptYesNo("Do you wish to change any of these options?", false)) break;
            this.printHeader("What should your CA be named?", terminal);
            terminal.println("Every client that connects to your Elasticsearch cluster will need to trust");
            terminal.println("this custom Certificate Authority.");
            terminal.println("It is helpful to have a meaningful name for this CA");
            terminal.println("");
            dn = this.tryReadInput(terminal, "CA Name", dn, name -> {
                try {
                    if (name.contains("=")) {
                        return new X500Principal((String)name);
                    }
                    return new X500Principal("CN=" + name);
                }
                catch (IllegalArgumentException e) {
                    terminal.println(Terminal.Verbosity.SILENT, "'" + name + "' is not a valid CA name (" + e.getMessage() + ")");
                    return null;
                }
            });
            this.printHeader("How long should your CA be valid?", terminal);
            terminal.println("Every certificate has an expiry date. When the expiry date is reached, clients");
            terminal.println("will stop trusting your Certificate Authority and TLS connections will fail.");
            terminal.println("");
            terminal.println("We recommend that you set this to a long duration (3 - 5 years) and then perform a");
            terminal.println("manual update a few months before it expires.");
            terminal.println("You may enter the validity period in years (e.g. 3Y), months (e.g. 18M), or days (e.g. 90D)");
            validity = this.readPeriodInput(terminal, "CA Validity", validity, 90);
            this.printHeader("What key size should your CA have?", terminal);
            terminal.println("The RSA private key for your Certificate Authority has a fixed 'key size' (in bits).");
            terminal.println("Larger key sizes are generally more secure, but are also slower.");
            terminal.println("");
            terminal.println("We recommend that you use one of 2048, 3072 or 4096 bits for your key.");
            keySize = this.readKeySize(terminal, keySize);
            terminal.println("");
        }
        try {
            KeyPair keyPair = CertGenUtils.generateKeyPair(keySize);
            ZonedDateTime notBefore = ZonedDateTime.now(ZoneOffset.UTC);
            ZonedDateTime notAfter = notBefore.plus(validity);
            X509Certificate caCert = CertGenUtils.generateSignedCertificate(dn, null, keyPair, null, null, true, notBefore, notAfter, null);
            this.printHeader("CA password", terminal);
            terminal.println("We recommend that you protect your CA private key with a strong password.");
            terminal.println("If your key does not have a password (or the password can be easily guessed)");
            terminal.println("then anyone who gets a copy of the key file will be able to generate new certificates");
            terminal.println("and impersonate your Elasticsearch cluster.");
            terminal.println("");
            terminal.println("IT IS IMPORTANT THAT YOU REMEMBER THIS PASSWORD AND KEEP IT SECURE");
            terminal.println("");
            char[] password = this.readPassword(terminal, "CA password: ", true);
            return new CertificateTool.CAInfo(caCert, keyPair.getPrivate(), true, password);
        }
        catch (GeneralSecurityException | CertIOException | OperatorCreationException e) {
            throw new IllegalArgumentException("Cannot generate CA key pair", e);
        }
    }

    Period readPeriodInput(Terminal terminal, String prompt, Period defaultValue, int recommendedMinimumDays) {
        Period period = this.tryReadInput(terminal, prompt, defaultValue, input -> {
            String periodInput = input.replaceAll("[,\\s]", "");
            if (input.charAt(0) != 'P') {
                periodInput = "P" + periodInput;
            }
            try {
                Period parsed = Period.parse(periodInput);
                long approxDays = 30L * parsed.toTotalMonths() + (long)parsed.getDays();
                if (approxDays < (long)recommendedMinimumDays) {
                    terminal.println("The period '" + HttpCertificateCommand.toString(parsed) + "' is less than the recommended period");
                    if (!terminal.promptYesNo("Are you sure?", false)) {
                        return null;
                    }
                }
                return parsed;
            }
            catch (DateTimeParseException e) {
                terminal.println("Sorry, I do not understand '" + input + "' (" + e.getMessage() + ")");
                return null;
            }
        });
        return period;
    }

    private Integer readKeySize(Terminal terminal, int keySize) {
        return this.tryReadInput(terminal, "Key Size", keySize, input -> {
            try {
                int size = Integer.parseInt(input);
                if (size < 1024) {
                    terminal.println("Keys must be at least 1024 bits");
                    return null;
                }
                if (size > 8192) {
                    terminal.println("Keys cannot be larger than 8192 bits");
                    return null;
                }
                if (size % 1024 != 0) {
                    terminal.println("The key size should be a multiple of 1024 bits");
                    return null;
                }
                return size;
            }
            catch (NumberFormatException e) {
                terminal.println("The key size must be a positive integer");
                return null;
            }
        });
    }

    private char[] readPassword(Terminal terminal, String prompt, boolean confirm) {
        char[] password;
        while ((password = terminal.readSecret(prompt + " [<ENTER> for none]")).length != 0) {
            if (CertificateTool.isAscii(password)) {
                char[] again;
                if (confirm && !Arrays.equals(password, again = terminal.readSecret("Repeat password to confirm: "))) {
                    terminal.println("Passwords do not match");
                    continue;
                }
                return password;
            }
            terminal.println(Terminal.Verbosity.SILENT, "Passwords must be plain ASCII");
        }
        return password;
    }

    private CertificateTool.CAInfo readKeystoreCA(Path ksPath, FileType fileType, Terminal terminal) throws UserException {
        String storeType = fileType == FileType.PKCS12 ? "PKCS12" : "jks";
        terminal.println("Reading a " + storeType + " keystore requires a password.");
        terminal.println("It is possible for the keystore's password to be blank,");
        terminal.println("in which case you can simply press <ENTER> at the prompt");
        char[] password = terminal.readSecret("Password for " + ksPath.getFileName() + ":");
        try {
            Map keys = CertParsingUtils.readKeyPairsFromKeystore((Path)ksPath, (String)storeType, (char[])password, alias -> password);
            if (keys.size() != 1) {
                if (keys.isEmpty()) {
                    terminal.println(Terminal.Verbosity.SILENT, "The keystore at " + ksPath + " does not contain any keys ");
                } else {
                    terminal.println(Terminal.Verbosity.SILENT, "The keystore at " + ksPath + " contains " + keys.size() + " keys,");
                    terminal.println(Terminal.Verbosity.SILENT, "but this command requires a keystore with a single key");
                }
                terminal.println("Please try again with a keystore that contains exactly 1 private key entry");
                throw new UserException(65, "The CA keystore " + ksPath + " contains " + keys.size() + " keys");
            }
            Map.Entry pair = keys.entrySet().iterator().next();
            return new CertificateTool.CAInfo((X509Certificate)pair.getKey(), (PrivateKey)pair.getValue());
        }
        catch (IOException | GeneralSecurityException e) {
            throw new ElasticsearchException("Failed to read keystore " + ksPath, (Throwable)e, new Object[0]);
        }
    }

    private CertificateTool.CAInfo readPemCA(Path certPath, Path keyPath, Terminal terminal) throws UserException {
        X509Certificate cert = this.readCertificate(certPath, terminal);
        PrivateKey key = this.readPrivateKey(keyPath, terminal);
        return new CertificateTool.CAInfo(cert, key);
    }

    private X509Certificate readCertificate(Path path, Terminal terminal) throws UserException {
        try {
            X509Certificate[] certificates = CertParsingUtils.readX509Certificates(Collections.singletonList(path));
            switch (certificates.length) {
                case 0: {
                    terminal.errorPrintln("Could not read any certificates from " + path);
                    throw new UserException(65, path + ": No certificates found");
                }
                case 1: {
                    return certificates[0];
                }
            }
            terminal.errorPrintln("Read [" + certificates.length + "] certificates from " + path + " but expected 1");
            throw new UserException(65, path + ": Multiple certificates found");
        }
        catch (IOException | CertificateException e) {
            throw new ElasticsearchException("Failed to read certificates from " + path, (Throwable)e, new Object[0]);
        }
    }

    private PrivateKey readPrivateKey(Path path, Terminal terminal) {
        try {
            return PemUtils.readPrivateKey((Path)path, () -> {
                terminal.println("");
                terminal.println("The PEM key stored in " + path + " requires a password.");
                terminal.println("");
                return terminal.readSecret("Password for " + path.getFileName() + ":");
            });
        }
        catch (IOException e) {
            throw new ElasticsearchException("Failed to read private key from " + path, (Throwable)e, new Object[0]);
        }
    }

    private boolean askExistingCertificateAuthority(Terminal terminal) {
        this.printHeader("Do you have an existing Certificate Authority (CA) key-pair that you wish to use to sign your certificate?", terminal);
        terminal.println("If you have an existing CA certificate and key, then you can use that CA to");
        terminal.println("sign your new http certificate. This allows you to use the same CA across");
        terminal.println("multiple Elasticsearch clusters which can make it easier to configure clients,");
        terminal.println("and may be easier for you to manage.");
        terminal.println("");
        terminal.println("If you do not have an existing CA, one will be generated for you.");
        terminal.println("");
        return terminal.promptYesNo("Use an existing CA?", false);
    }

    private <T> T tryReadInput(Terminal terminal, String prompt, T defaultValue, Function<String, T> parser) {
        String input;
        T parsed;
        String defaultStr;
        String string = defaultStr = defaultValue instanceof Period ? HttpCertificateCommand.toString((Period)defaultValue) : String.valueOf(defaultValue);
        do {
            if (!Strings.isEmpty((CharSequence)(input = terminal.readText(prompt + " [" + defaultStr + "] ")))) continue;
            return defaultValue;
        } while ((parsed = parser.apply(input)) == null);
        return parsed;
    }

    static String toString(Period period) {
        if (period == null) {
            return "N/A";
        }
        if (period.isZero()) {
            return "0d";
        }
        ArrayList<String> parts = new ArrayList<String>(3);
        if (period.getYears() != 0) {
            parts.add(period.getYears() + "y");
        }
        if (period.getMonths() != 0) {
            parts.add(period.getMonths() + "m");
        }
        if (period.getDays() != 0) {
            parts.add(period.getDays() + "d");
        }
        return Strings.collectionToCommaDelimitedString(parts);
    }

    private Path requestPath(String prompt, Terminal terminal, Environment env, boolean requireExisting) {
        while (true) {
            String input = terminal.readText(prompt);
            Path path = env.configFile().resolve(input).toAbsolutePath();
            if (path.getFileName() == null) {
                terminal.println(Terminal.Verbosity.SILENT, input + " is not a valid file");
                continue;
            }
            if (!requireExisting || Files.isReadable(path)) {
                return path;
            }
            if (Files.notExists(path, new LinkOption[0])) {
                terminal.println(Terminal.Verbosity.SILENT, "The file " + path + " does not exist");
                continue;
            }
            terminal.println(Terminal.Verbosity.SILENT, "The file " + path + " cannot be read");
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    static FileType guessFileType(Path path, Terminal terminal) {
        String fileName = path == null ? "" : path.getFileName().toString().toLowerCase(Locale.ROOT);
        if (fileName.endsWith(".p12")) return FileType.PKCS12;
        if (fileName.endsWith(".pfx")) return FileType.PKCS12;
        if (fileName.endsWith(".pkcs12")) {
            return FileType.PKCS12;
        }
        if (fileName.endsWith(".jks")) {
            return FileType.JKS;
        }
        try (InputStream in = Files.newInputStream(path, new OpenOption[0]);){
            byte[] leadingBytes = new byte[2];
            int read = in.read(leadingBytes);
            if (read < leadingBytes.length) {
                FileType fileType = FileType.UNRECOGNIZED;
                return fileType;
            }
            if (Arrays.equals(leadingBytes, MAGIC_BYTES1_PKCS12) || Arrays.equals(leadingBytes, MAGIC_BYTES2_PKCS12)) {
                FileType fileType = FileType.PKCS12;
                return fileType;
            }
            if (Arrays.equals(leadingBytes, MAGIC_BYTES_JKS)) {
                FileType fileType = FileType.JKS;
                return fileType;
            }
        }
        catch (IOException e) {
            terminal.errorPrintln("Failed to read from file " + path);
            terminal.errorPrintln(e.toString());
            return FileType.UNRECOGNIZED;
        }
        try (Stream<String> lines = Files.lines(path, StandardCharsets.UTF_8);){
            List types = lines.filter(s -> s.startsWith("-----BEGIN")).map(s -> {
                if (s.contains("BEGIN CERTIFICATE")) {
                    return FileType.PEM_CERT;
                }
                if (s.contains("PRIVATE KEY")) {
                    return FileType.PEM_KEY;
                }
                return null;
            }).filter(ft -> ft != null).collect(Collectors.toList());
            switch (types.size()) {
                case 0: {
                    FileType fileType = FileType.UNRECOGNIZED;
                    return fileType;
                }
                case 1: {
                    FileType fileType = (FileType)((Object)types.get(0));
                    return fileType;
                }
            }
            if (types.contains((Object)FileType.PEM_KEY)) {
                terminal.errorPrintln("Cannot determine a type for the PEM file " + path + " because it contains: [" + Strings.collectionToCommaDelimitedString(types) + "]");
                return FileType.UNRECOGNIZED;
            }
            FileType fileType = FileType.PEM_CERT_CHAIN;
            return fileType;
        }
        catch (IOException | UncheckedIOException e) {
            terminal.errorPrintln("Cannot determine the file type for " + path);
            terminal.errorPrintln(e.toString());
            return FileType.UNRECOGNIZED;
        }
    }

    private void printHeader(String text, Terminal terminal) {
        terminal.println("");
        terminal.println(Terminal.Verbosity.SILENT, "## " + text);
        terminal.println("");
    }

    OptionParser getParser() {
        return this.parser;
    }

    private class ZipEntryStream
    extends OutputStream {
        private final ZipOutputStream zip;

        ZipEntryStream(ZipOutputStream zip, String name) throws IOException {
            this(zip, new ZipEntry(name));
        }

        ZipEntryStream(ZipOutputStream zip, ZipEntry entry) throws IOException {
            this.zip = zip;
            assert (!entry.isDirectory());
            zip.putNextEntry(entry);
        }

        @Override
        public void write(int b) throws IOException {
            this.zip.write(b);
        }

        @Override
        public void write(byte[] b) throws IOException {
            this.zip.write(b);
        }

        @Override
        public void write(byte[] b, int off, int len) throws IOException {
            this.zip.write(b, off, len);
        }

        @Override
        public void flush() throws IOException {
            this.zip.flush();
        }

        @Override
        public void close() throws IOException {
            this.zip.closeEntry();
        }
    }

    private class CertOptions {
        final String name;
        final X500Principal subject;
        final List<String> dnsNames;
        final List<String> ipNames;
        final int keySize;
        final Period validity;

        private CertOptions(String name, X500Principal subject, List<String> dnsNames, List<String> ipNames, int keySize, Period validity) {
            this.name = name;
            this.subject = subject;
            this.dnsNames = dnsNames;
            this.ipNames = ipNames;
            this.keySize = keySize;
            this.validity = validity;
        }
    }

    static enum FileType {
        PKCS12,
        JKS,
        PEM_CERT,
        PEM_KEY,
        PEM_CERT_CHAIN,
        UNRECOGNIZED;

    }
}

