/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.index.store.cache;

import java.io.IOException;
import java.io.UncheckedIOException;
import java.nio.channels.FileChannel;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.concurrent.Future;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.Consumer;
import org.apache.lucene.store.AlreadyClosedException;
import org.elasticsearch.action.ActionListener;
import org.elasticsearch.common.Nullable;
import org.elasticsearch.common.collect.Tuple;
import org.elasticsearch.common.lease.Releasable;
import org.elasticsearch.common.lease.Releasables;
import org.elasticsearch.common.util.concurrent.AbstractRefCounted;
import org.elasticsearch.common.util.concurrent.AbstractRunnable;
import org.elasticsearch.index.store.cache.SparseFileTracker;

public class CacheFile {
    private static final StandardOpenOption[] OPEN_OPTIONS = new StandardOpenOption[]{StandardOpenOption.READ, StandardOpenOption.WRITE, StandardOpenOption.CREATE, StandardOpenOption.SPARSE};
    private final AbstractRefCounted refCounter = new AbstractRefCounted("CacheFile"){

        protected void closeInternal() {
            assert (CacheFile.this.evicted.get());
            assert (CacheFile.this.assertNoPendingListeners());
            try {
                Files.deleteIfExists(CacheFile.this.file);
            }
            catch (IOException e) {
                throw new UncheckedIOException(e);
            }
        }
    };
    private final SparseFileTracker tracker;
    private final String description;
    private final Path file;
    private final Set<EvictionListener> listeners = new HashSet<EvictionListener>();
    private final AtomicBoolean evicted = new AtomicBoolean(false);
    @Nullable
    private volatile FileChannelReference channelRef;

    public CacheFile(String description, long length, Path file) {
        this.tracker = new SparseFileTracker(file.toString(), length);
        this.description = Objects.requireNonNull(description);
        this.file = Objects.requireNonNull(file);
        assert (this.invariant());
    }

    public long getLength() {
        return this.tracker.getLength();
    }

    public Path getFile() {
        return this.file;
    }

    @Nullable
    FileChannel getChannel() {
        FileChannelReference reference = this.channelRef;
        return reference == null ? null : reference.fileChannel;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean acquire(EvictionListener listener) throws IOException {
        assert (listener != null);
        this.ensureOpen();
        boolean success = false;
        if (this.refCounter.tryIncRef()) {
            try {
                Set<EvictionListener> set = this.listeners;
                synchronized (set) {
                    this.ensureOpen();
                    if (this.listeners.isEmpty()) {
                        assert (this.channelRef == null);
                        this.channelRef = new FileChannelReference();
                    }
                    boolean added = this.listeners.add(listener);
                    assert (added) : "listener already exists " + listener;
                }
                success = true;
            }
            finally {
                if (!success) {
                    this.decrementRefCount();
                }
            }
        }
        assert (this.invariant());
        return success;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean release(EvictionListener listener) {
        assert (listener != null);
        boolean success = false;
        try {
            Set<EvictionListener> set = this.listeners;
            synchronized (set) {
                boolean removed = this.listeners.remove(Objects.requireNonNull(listener));
                assert (removed) : "listener does not exist " + listener;
                if (!removed) {
                    throw new IllegalStateException("Cannot remove an unknown listener");
                }
                if (this.listeners.isEmpty()) {
                    this.channelRef.decRef();
                    this.channelRef = null;
                }
            }
            success = true;
        }
        finally {
            if (success) {
                this.decrementRefCount();
            }
        }
        assert (this.invariant());
        return success;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private boolean assertNoPendingListeners() {
        Set<EvictionListener> set = this.listeners;
        synchronized (set) {
            assert (this.listeners.isEmpty());
            assert (this.channelRef == null);
        }
        return true;
    }

    private void decrementRefCount() {
        boolean released = this.refCounter.decRef();
        assert (!released || this.evicted.get() && Files.notExists(this.file, new LinkOption[0]));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void startEviction() {
        if (this.evicted.compareAndSet(false, true)) {
            HashSet<EvictionListener> evictionListeners;
            Set<EvictionListener> set = this.listeners;
            synchronized (set) {
                evictionListeners = new HashSet<EvictionListener>(this.listeners);
            }
            this.decrementRefCount();
            evictionListeners.forEach(listener -> listener.onEviction(this));
        }
        assert (this.invariant());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private boolean invariant() {
        Set<EvictionListener> set = this.listeners;
        synchronized (set) {
            if (this.listeners.isEmpty()) {
                assert (this.channelRef == null);
            } else {
                assert (this.channelRef != null);
                assert (this.refCounter.refCount() > 0);
                assert (this.channelRef.refCount() > 0);
                assert (Files.exists(this.file, new LinkOption[0]));
            }
        }
        return true;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public String toString() {
        Set<EvictionListener> set = this.listeners;
        synchronized (set) {
            return "CacheFile{desc='" + this.description + "', file=" + this.file + ", length=" + this.tracker.getLength() + ", channel=" + (this.channelRef != null ? "yes" : "no") + ", listeners=" + this.listeners.size() + ", evicted=" + this.evicted + ", tracker=" + this.tracker + '}';
        }
    }

    private void ensureOpen() {
        if (this.evicted.get()) {
            throw new AlreadyClosedException("Cache file is evicted");
        }
    }

    Future<Integer> populateAndRead(Tuple<Long, Long> rangeToWrite, Tuple<Long, Long> rangeToRead, RangeAvailableHandler reader, final RangeMissingHandler writer, Executor executor) {
        CompletableFuture<Integer> future = new CompletableFuture<Integer>();
        Releasable decrementRef = null;
        try {
            final FileChannelReference reference = this.acquireFileChannelReference();
            decrementRef = Releasables.releaseOnce(() -> ((FileChannelReference)reference).decRef());
            final List<SparseFileTracker.Gap> gaps = this.tracker.waitForRange(rangeToWrite, rangeToRead, CacheFile.rangeListener(rangeToRead, reader, future, reference, decrementRef));
            if (!gaps.isEmpty()) {
                executor.execute((Runnable)new AbstractRunnable(){

                    protected void doRun() {
                        for (SparseFileTracker.Gap gap : gaps) {
                            try {
                                if (!reference.tryIncRef()) {
                                    assert (false) : "expected a non-closed channel reference";
                                    throw new AlreadyClosedException("Cache file channel has been released and closed");
                                }
                                try {
                                    CacheFile.this.ensureOpen();
                                    writer.fillCacheRange(reference.fileChannel, gap.start(), gap.end(), gap::onProgress);
                                }
                                finally {
                                    reference.decRef();
                                }
                                gap.onCompletion();
                            }
                            catch (Exception e) {
                                gap.onFailure(e);
                            }
                        }
                    }

                    public void onFailure(Exception e) {
                        gaps.forEach(gap -> gap.onFailure(e));
                    }
                });
            }
        }
        catch (Exception e) {
            CacheFile.releaseAndFail(future, decrementRef, e);
        }
        return future;
    }

    @Nullable
    Future<Integer> readIfAvailableOrPending(Tuple<Long, Long> rangeToRead, RangeAvailableHandler reader) {
        CompletableFuture<Integer> future = new CompletableFuture<Integer>();
        Releasable decrementRef = null;
        try {
            FileChannelReference reference = this.acquireFileChannelReference();
            decrementRef = Releasables.releaseOnce(() -> ((FileChannelReference)reference).decRef());
            if (this.tracker.waitForRangeIfPending(rangeToRead, CacheFile.rangeListener(rangeToRead, reader, future, reference, decrementRef))) {
                return future;
            }
            decrementRef.close();
            return null;
        }
        catch (Exception e) {
            CacheFile.releaseAndFail(future, decrementRef, e);
            return future;
        }
    }

    private static void releaseAndFail(CompletableFuture<Integer> future, Releasable decrementRef, Exception e) {
        try {
            Releasables.close((Releasable)decrementRef);
        }
        catch (Exception ex) {
            e.addSuppressed(ex);
        }
        future.completeExceptionally(e);
    }

    private static ActionListener<Void> rangeListener(Tuple<Long, Long> rangeToRead, RangeAvailableHandler reader, CompletableFuture<Integer> future, FileChannelReference reference, Releasable releasable) {
        return ActionListener.runAfter((ActionListener)ActionListener.wrap(success -> {
            int read = reader.onRangeAvailable(reference.fileChannel);
            assert ((long)read == (Long)rangeToRead.v2() - (Long)rangeToRead.v1()) : "partial read [" + read + "] does not match the range to read [" + rangeToRead.v2() + '-' + rangeToRead.v1() + ']';
            future.complete(read);
        }, future::completeExceptionally), () -> ((Releasable)releasable).close());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private FileChannelReference acquireFileChannelReference() {
        FileChannelReference reference;
        Set<EvictionListener> set = this.listeners;
        synchronized (set) {
            this.ensureOpen();
            reference = this.channelRef;
            assert (reference != null && reference.refCount() > 0) : "impossible to run into a fully released channel reference under the listeners mutex";
            assert (this.refCounter.refCount() > 0) : "file should not be fully released";
            reference.incRef();
        }
        return reference;
    }

    public Tuple<Long, Long> getAbsentRangeWithin(long start, long end) {
        this.ensureOpen();
        return this.tracker.getAbsentRangeWithin(start, end);
    }

    @FunctionalInterface
    static interface RangeMissingHandler {
        public void fillCacheRange(FileChannel var1, long var2, long var4, Consumer<Long> var6) throws IOException;
    }

    @FunctionalInterface
    static interface RangeAvailableHandler {
        public int onRangeAvailable(FileChannel var1) throws IOException;
    }

    private final class FileChannelReference
    extends AbstractRefCounted {
        private final FileChannel fileChannel;

        FileChannelReference() throws IOException {
            super("FileChannel[" + CacheFile.this.file + "]");
            this.fileChannel = FileChannel.open(CacheFile.this.file, OPEN_OPTIONS);
            CacheFile.this.refCounter.incRef();
        }

        protected void closeInternal() {
            try {
                this.fileChannel.close();
            }
            catch (IOException e) {
                throw new UncheckedIOException(e);
            }
            finally {
                CacheFile.this.decrementRefCount();
            }
        }
    }

    @FunctionalInterface
    public static interface EvictionListener {
        public void onEviction(CacheFile var1);
    }
}

