const installedApp = {};

const registerExtension = function(i, e, d) {
  installedApp.init = i;
  installedApp.enable = e;
  installedApp.disable = d;
};

function init() {
  if (installedApp.init) {
    installedApp.init();
  }
}
function enable() {
  if (installedApp.enable) {
    installedApp.enable();
  }
}

function disable() {
  if (installedApp.disable) {
    installedApp.disable();
  }
}
 /* Polyfill from Mozilla. */

  // https://tc39.github.io/ecma262/#sec-array.prototype.find
if (!Array.prototype.find && Object.defineProperty) {
  Object.defineProperty(Array.prototype, 'find', {
    value: function(predicate) {
     // 1. Let O be ? ToObject(this value).
      if (this == null) {
        throw new TypeError('"this" is null or not defined');
      }

      var o = Object(this);

      // 2. Let len be ? ToLength(? Get(O, "length")).
      var len = o.length >>> 0;

      // 3. If IsCallable(predicate) is false, throw a TypeError exception.
      if (typeof predicate !== 'function') {
        throw new TypeError('predicate must be a function');
      }

      // 4. If thisArg was supplied, let T be thisArg; else let T be undefined.
      var thisArg = arguments[1];

      // 5. Let k be 0.
      var k = 0;

      // 6. Repeat, while k < len
      while (k < len) {
        // a. Let Pk be ! ToString(k).
        // b. Let kValue be ? Get(O, Pk).
        // c. Let testResult be ToBoolean(? Call(predicate, T, « kValue, k, O »)).
        // d. If testResult is true, return kValue.
        var kValue = o[k];
        if (predicate.call(thisArg, kValue, k, o)) {
          return kValue;
        }
        // e. Increase k by 1.
        k++;
      }

      // 7. Return undefined.
      return undefined;
    }
  });
}

if (!Array.prototype.findIndex && Object.defineProperty) {
  Object.defineProperty(Array.prototype, 'findIndex', {
    value: function(predicate) {
     // 1. Let O be ? ToObject(this value).
      if (this == null) {
        throw new TypeError('"this" is null or not defined');
      }

      var o = Object(this);

      // 2. Let len be ? ToLength(? Get(O, "length")).
      var len = o.length >>> 0;

      // 3. If IsCallable(predicate) is false, throw a TypeError exception.
      if (typeof predicate !== 'function') {
        throw new TypeError('predicate must be a function');
      }

      // 4. If thisArg was supplied, let T be thisArg; else let T be undefined.
      var thisArg = arguments[1];

      // 5. Let k be 0.
      var k = 0;

      // 6. Repeat, while k < len
      while (k < len) {
        // a. Let Pk be ! ToString(k).
        // b. Let kValue be ? Get(O, Pk).
        // c. Let testResult be ToBoolean(? Call(predicate, T, « kValue, k, O »)).
        // d. If testResult is true, return k.
        var kValue = o[k];
        if (predicate.call(thisArg, kValue, k, o)) {
          return k;
        }
        // e. Increase k by 1.
        k++;
      }

      // 7. Return -1.
      return -1;
    }
  });
}
/* Logging
 * Written by Sergey
*/
let debug = false;
/**
 * If called with a false argument, log statements are suppressed.
 */
function setLoggingEnabled(enabled) {
    debug = enabled;
}
/**
 * Log logs the given message using the gnome shell logger (global.log) if the
 * debug variable is set to true.
 *
 * Debug messages may be viewed using the bash command `journalctl
 * /usr/bin/gnome-shell` and grepping the results for 'gTile'.
 */
function log(message) {
    if (debug) {
        global.log("gTile " + message);
    }
}

/* Determine if gnome-shell version newer than required
 * Written by Sergey
*/
function getConfig() {
    return imports.misc.config;
}
const VERSION_34 = { major: 3, minor: 34 };
const VERSION_36 = { major: 3, minor: 36 };
/**
 * ShellVersion is used to parse the version string
 */
class ShellVersion {
    constructor(version) {
        const parts = version.split('.').map((part) => Number(part));
        if (parts.length < 2) {
            throw new Error(`invalid version supplied: ${version}`);
        }
        this.major = parts[0];
        this.minor = parts[1];
        // Tolerate "40.alpha.1" for example. See https://github.com/gTile/gTile/issues/187.
        if (isNaN(this.minor)) {
            this.minor = 0;
        }
        if (isNaN(this.major)) {
            throw new Error(`invalid version supplied: ${JSON.stringify(version)}; got major = ${this.major}, minor = ${this.minor}`);
        }
        this.rawVersion = version;
    }
    static defaultVersion() {
        return ShellVersion.parse(getConfig().PACKAGE_VERSION);
    }
    static parse(version) {
        return new ShellVersion(version);
    }
    version_at_least_34() {
        return versionGreaterThanOrEqualTo(this, VERSION_34);
    }
    version_at_least_36() {
        return versionGreaterThanOrEqualTo(this, VERSION_36);
    }
    print_version() {
        log("Init gnome-shell version " + this.rawVersion + " major " + this.major + " minor " + this.minor);
    }
}
/**
 * Returns true if a is >= b.
 */
function versionGreaterThanOrEqualTo(a, b) {
    return a.major > b.major || (a.major === b.major && a.minor >= b.minor);
}

// Library imports
const Main = imports.ui.main;
const Meta = imports.gi.Meta;
const Shell = imports.gi.Shell;
// Extension imports
const Me = imports.misc.extensionUtils.getCurrentExtension();
const Settings = Me.imports.settings;
function bind(keyBindings) {
    log("Binding keys");
    let mySettings = Settings.get();
    keyBindings.forEach((callback, key) => {
        //const key = keyString as KeyBindingSettingName;
        if (Main.wm.addKeybinding && Shell.ActionMode) { // introduced in 3.16
            Main.wm.addKeybinding(key, mySettings, Meta.KeyBindingFlags.NONE, Shell.ActionMode.NORMAL, callback);
        }
        else if (Main.wm.addKeybinding && Shell.KeyBindingMode) { // introduced in 3.7.5
            Main.wm.addKeybinding(key, mySettings, Meta.KeyBindingFlags.NONE, Shell.KeyBindingMode.NORMAL | Shell.KeyBindingMode.MESSAGE_TRAY, callback);
        }
        else {
            global.display.add_keybinding(key, mySettings, Meta.KeyBindingFlags.NONE, callback);
        }
    });
}
function unbind(keyBindings) {
    log("Unbinding keys");
    for (let key of keyBindings.keys()) {
        if (Main.wm.removeKeybinding) { // introduced in 3.7.2
            Main.wm.removeKeybinding(key);
        }
        else {
            global.display.remove_keybinding(key);
        }
    }
}

/* Edited by Sergey after
https://github.com/tpyl/gssnaptoneighbors
 by Timo Pylvanainen <tpyl@iki.fi>
 */
const Meta$1 = imports.gi.Meta;
const WorkspaceManager = global.screen || global.workspace_manager;
const OVERLAP_TOLERANCE = 5;
const SCAN_BOX_SIZE = 50;
/**
 * Return all windows on the currently active workspace
 */
function getWindowsOnActiveWorkspace() {
    let windows = [];
    let windowActors = global.get_window_actors();
    let curWorkSpace = WorkspaceManager.get_active_workspace();
    for (let i = 0; i < windowActors.length; i++) {
        let win = windowActors[i].meta_window;
        if (win.located_on_workspace(curWorkSpace) &&
            !win.minimized &&
            win.get_frame_type() == 0) {
            windows.push(win);
        }
    }
    return windows;
}
/**
 * Find the maximum horzontal expansion from x and
 * returns minx, maxx. The initial maximum x is given
 * as argument, and the expansion is never larger than
 * that.
 *
 * The upper and lower limits define the y coordinate
 * range to check for overlapping windows.
 */
function expandHorizontally(x, upper, lower, minx, maxx, windows) {
    for (let i = 0; i < windows.length; i++) {
        let rect = windows[i].get_frame_rect();
        let wt = rect.y;
        let wb = rect.y + rect.height;
        let wl = rect.x;
        let wr = rect.x + rect.width;
        // Check only  if the window overlaps vertically
        if (wb > upper && wt < lower) {
            if (wr < x && minx < wr) {
                minx = wr;
            }
            if (wl > x && wl < maxx) {
                maxx = wl;
            }
        }
    }
    return { min: minx, max: maxx };
}
/**
 * Find the maximum vertical expansion from  y, and
 * returns miny, maxy. The initial maximum y is given
 * as argument, and the expansion is never larger than
 * that.
 *
 * The left and right limits define the x coordinate
 * range to check for overlapping windows.
 */
function expandVertically(y, left, right, miny, maxy, windows) {
    for (let i = 0; i < windows.length; i++) {
        let rect = windows[i].get_frame_rect();
        let wt = rect.y;
        let wb = rect.y + rect.height;
        let wl = rect.x;
        let wr = rect.x + rect.width;
        // Check only  if the window overlaps horizontally
        if (wr > left && wl < right) {
            if (wb < y && miny < wb) {
                miny = wb;
            }
            if (wt > y && wt < maxy) {
                maxy = wt;
            }
        }
    }
    return { min: miny, max: maxy };
}
/**
 * Resize & move the *window* so it touches adjacent windows or
 * screen edge top, bottom, left and right. The top-left corner
 * of the window defines the expansion point.
 *
 * There is an L-ambiguity where the window could be expanded
 * both vertically and horizontally. The expnasion that results
 * in closer to 1 aspect ratio is selected.
 */
function snapToNeighbors(window) {
    log("snapToNeighbors " + window.get_title());
    // Unmaximize first
    if (window.maximized_horizontally || window.maximized_vertically)
        window.unmaximize(Meta$1.MaximizeFlags.HORIZONTAL | Meta$1.MaximizeFlags.VERTICAL);
    let workArea = window.get_work_area_current_monitor();
    let myrect = window.get_frame_rect();
    let windows = getWindowsOnActiveWorkspace();
    // Scan for overlapping windows in a thin bar around the top of the 
    // window. The vertical height of the window will be adjusted later. 
    let maxHorizw = expandHorizontally(myrect.x + Math.min(SCAN_BOX_SIZE, myrect.width / 2), myrect.y + Math.min(SCAN_BOX_SIZE, myrect.height / 2), myrect.y + Math.min(SCAN_BOX_SIZE, myrect.height / 2) + SCAN_BOX_SIZE, workArea.x, workArea.x + workArea.width, windows);
    let maxHorizh = expandVertically(myrect.y + Math.min(SCAN_BOX_SIZE, myrect.height / 2), maxHorizw.min + OVERLAP_TOLERANCE, maxHorizw.max - OVERLAP_TOLERANCE, workArea.y, workArea.y + workArea.height, windows);
    let maxVerth = expandVertically(myrect.y + Math.min(SCAN_BOX_SIZE, myrect.height / 2), myrect.x + Math.min(SCAN_BOX_SIZE, myrect.width / 2), myrect.x + Math.min(SCAN_BOX_SIZE, myrect.width / 2) + SCAN_BOX_SIZE, workArea.y, workArea.y + workArea.height, windows);
    let maxVertw = expandHorizontally(myrect.x + Math.min(SCAN_BOX_SIZE, myrect.width / 2), maxVerth.min + OVERLAP_TOLERANCE, maxVerth.max - OVERLAP_TOLERANCE, workArea.x, workArea.x + workArea.width, windows);
    if ((maxHorizw.max - maxHorizw.min) * (maxHorizh.max - maxHorizh.min) >
        (maxVertw.max - maxVertw.min) * (maxVerth.max - maxVerth.min)) {
        window.move_resize_frame(true, maxHorizw.min, maxHorizh.min, maxHorizw.max - maxHorizw.min, maxHorizh.max - maxHorizh.min);
    }
    else {
        window.move_resize_frame(true, maxVertw.min, maxVerth.min, maxVertw.max - maxVertw.min, maxVerth.max - maxVerth.min);
    }
}

const MAX_TUPLE_MEMBER_VALUE = Number.MAX_SAFE_INTEGER;
/**
 * TileSpec represents a rectangular area on display by means of specifying a
 * number of evenly spaced tiles and two corners.
 */
class TileSpec {
    constructor(gridWidth, gridHeight, luc, rdc) {
        this.gridWidth = gridWidth;
        this.gridHeight = gridHeight;
        this.luc = luc;
        this.rdc = rdc;
    }
    toString() {
        return [[this.gridWidth, this.gridHeight].join('x'),
            [this.luc.x, this.luc.y].join(':'),
            [this.rdc.x, this.rdc.y].join(':')].join(' ');
    }
    toFrameRect(workArea) {
        const elemSize = new Size(Math.floor(workArea.size.width / this.gridWidth), Math.floor(workArea.size.height / this.gridHeight));
        return new Rect(new XY(workArea.origin.x + this.luc.x * elemSize.width, workArea.origin.y + this.luc.y * elemSize.height), new Size((this.rdc.x + 1 - this.luc.x) * elemSize.width, (this.rdc.y + 1 - this.luc.y) * elemSize.height));
    }
    get gridSize() {
        return new GridSize(this.gridWidth, this.gridHeight);
    }
    viewSize() {
        const sizeXY = this.rdc.minus(this.luc);
        return new GridSize(sizeXY.x + 1, sizeXY.y + 1);
    }
    isFullscreen() {
        return this.viewSize().equals(this.gridSize);
    }
}
class GridSize {
    constructor(
    // Number of columns.
    width, 
    // Number of rows.
    height) {
        this.width = width;
        this.height = height;
    }
    toString() {
        return `${this.width}x${this.height}`;
    }
    equals(other) {
        return this.width === other.width && this.height == other.height;
    }
}
function parseGridSizesIgnoringErrors(s) {
    return s.split(',').flatMap((part) => {
        const size = parseGridSizeIgnoringErrors(part.trim());
        return size ? [size] : [];
    });
}
function parseGridSizeIgnoringErrors(s) {
    const parts = s.split("x").map(Number);
    if (parts.length !== 2 || !parts.every(x => !isNaN(x))) {
        return null;
    }
    return new GridSize(parts[0], parts[1]);
}
class XY {
    constructor(x, y) {
        this.x = x;
        this.y = y;
    }
    clone() {
        return new XY(this.x, this.y);
    }
    toString() {
        return 'XY(' + [this.x, this.y].join(', ') + ')';
    }
    dot(b) {
        return this.x * b.x + this.y * b.y;
    }
    unit() {
        const norm = this.l2norm();
        return new XY(this.x / norm, this.y / norm);
    }
    l2norm() {
        return Math.sqrt(this.l2normSquared());
    }
    l2normSquared() {
        return this.dot(this);
    }
    scale(s) {
        return new XY(this.x * s, this.y * s);
    }
    project(b) {
        return b.scale(this.dot(b) / b.l2normSquared());
    }
    scalarProjection(b) {
        return this.dot(b.unit());
    }
    minus(b) {
        return new XY(this.x - b.x, this.y - b.y);
    }
    plus(b) {
        return new XY(this.x + b.x, this.y + b.y);
    }
}
const ADJOIN_DOT_PRODUCT_TOL = .02;
class LineSegment {
    constructor(a, b) {
        this.a = a;
        this.b = b;
    }
    static fromTwoPoints(a, b) {
        return new LineSegment(a, b);
    }
    direction() {
        return this.b.minus(this.a).unit();
    }
    adjoins(other, distTol) {
        return this.parallels(other) && this.lineDistance(other) < distTol;
    }
    parallels(other) {
        const unitDot = this.direction().dot(other.direction());
        return withinTol(Math.abs(unitDot), 1, ADJOIN_DOT_PRODUCT_TOL);
    }
    // The distance between the lines of two line segments. If lines are not
    // (close to) parallel, 0 is returned
    lineDistance(other) {
        return this.perpVectorBetweenLines(other).l2norm();
    }
    // The perpendicular vector between the lines of two line segments. If lines
    // are not (close to) parallel, [0, 0] is returned
    perpVectorBetweenLines(other) {
        const otherDir = other.direction();
        const unitDot = this.direction().dot(otherDir);
        if (!withinTol(Math.abs(unitDot), 1, ADJOIN_DOT_PRODUCT_TOL)) {
            return new XY(0, 0);
        }
        // Basically parallel. Now measure the perpendicular distance between
        // this.a->other.a and other.a->other.b.
        const d = other.a.minus(this.a);
        return d.minus(d.project(otherDir));
    }
}
class Size {
    constructor(width, height) {
        this.width = width;
        this.height = height;
    }
    clone() {
        return new Size(this.width, this.height);
    }
    toString() {
        return [this.width, this.height].join('x');
    }
    area() {
        return this.width * this.height;
    }
}
/**
 * A screen rectangle. A (0, 0) origin represents the top left of a display
 * area. Units are typically pixels.
 */
class Rect {
    constructor(origin, size) {
        this.origin = origin;
        this.size = size;
    }
    clone() {
        return new Rect(this.origin.clone(), this.size.clone());
    }
    toString() {
        return [this.origin, this.size].join(' ');
    }
    equal(r, tol) {
        const close = (a, b) => Math.abs(a - b) <= tol;
        return (close(this.origin.x, r.origin.x) &&
            close(this.origin.y, r.origin.y) &&
            close(this.size.width, r.size.width) &&
            close(this.size.height, r.size.height));
    }
    inset(s) {
        return new Rect(new XY(this.origin.x + s.width, this.origin.y + s.height), new Size(this.size.width - 2 * s.width, this.size.height - 2 * s.height));
    }
    edges() {
        const down = new XY(0, this.size.height);
        const right = new XY(this.size.width, 0);
        const seg = (a, b) => LineSegment.fromTwoPoints(a, b);
        // a---b
        // c---d
        const a = this.origin;
        const b = a.plus(right);
        const c = a.plus(down);
        const d = c.plus(right);
        const rv = new Edges({
            top: seg(a, b),
            right: seg(b, d),
            bottom: seg(c, d),
            left: seg(a, c)
        });
        return rv;
    }
    translate(vec) {
        return new Rect(this.origin.plus(vec), this.size);
    }
    // Increases or decreases the size of the rectangle by moving one of its
    // edges d units along the postive x or y axis, where positive x is right
    // and positive y is down.
    translateEdge(side, d) {
        const [w, h] = [this.size.width, this.size.height];
        switch (side) {
            case Side.Top:
                return new Rect(this.origin.plus(new XY(0, d)), new Size(w, h - d));
            case Side.Bottom:
                return new Rect(this.origin, new Size(w, h + d));
            case Side.Right:
                return new Rect(this.origin, new Size(w + d, h));
            case Side.Left:
                return new Rect(this.origin.plus(new XY(d, 0)), new Size(w - d, h));
            default:
                throw TypeError('bad side type ' + side);
        }
    }
    topLeft() {
        return this.origin;
    }
    topRight() {
        return this.origin.plus(new XY(this.size.width, 0));
    }
    bottomRight() {
        return this.origin.plus(new XY(this.size.width, this.size.height));
    }
    bottomLeft() {
        return this.origin.plus(new XY(0, this.size.height));
    }
    intersection(other) {
        // Not optimized, but that's not necessary.
        const origin = new XY(Math.max(this.topLeft().x, other.topLeft().x), Math.max(this.topLeft().y, other.topLeft().y));
        const br = new XY(Math.min(this.bottomRight().x, other.bottomRight().x), Math.min(this.bottomRight().y, other.bottomRight().y));
        const sizeXY = br.minus(origin);
        const size = new Size(sizeXY.x, sizeXY.y);
        if (size.width < 0 || size.height < 0) {
            return new Rect(new XY(0, 0), new Size(0, 0));
        }
        return new Rect(origin, size);
    }
    valid() {
        return this.size.width >= 0 && this.size.height >= 0;
    }
}
var Side;
(function (Side) {
    Side["Top"] = "TOP";
    Side["Right"] = "RIGHT";
    Side["Bottom"] = "BOTTOM";
    Side["Left"] = "LEFT";
})(Side || (Side = {}));
class Edges {
    constructor(obj) {
        this.top = obj.top;
        this.left = obj.left;
        this.bottom = obj.bottom;
        this.right = obj.right;
    }
    getSide(s) {
        switch (s) {
            case Side.Top: return this.top;
            case Side.Right: return this.right;
            case Side.Bottom: return this.bottom;
            case Side.Left: return this.left;
        }
    }
}
/**
 * parsePreset parses a sequence of TileSpec objects from a string like "8x8 0:0
 * 0:7, 8x10 0:0 2:7" or "8x8 0:0 0:7, 0:0 2:7"
 *
 * The 8x8 and 8x10 values above are the "grid size." The grid size may be
 * omitted in all but the first component of the preset.
 */
function parsePreset(preset) {
    const parts = preset.split(',').map(x => x.trim());
    let mostRecentSpec = null;
    return parts.map((part, index) => {
        if (hasImpliedGridSize(part)) {
            if (mostRecentSpec === null) {
                throw new Error(`preset component[${index}] ${part} of ${preset} is missing grid size (e.g., '3x3')`);
            }
            part = `${mostRecentSpec.gridWidth}x${mostRecentSpec.gridHeight} ${part}`;
        }
        const parsed = parseSinglePreset(part);
        mostRecentSpec = parsed;
        return parsed;
    });
}
function parseSinglePreset(preset) {
    const ps = preset.trim().split(" ");
    if (ps.length != 3) {
        throw new Error(`Bad preset: ${JSON.stringify(preset)}`);
    }
    const gridFormat = parseTuple(ps[0], "x");
    const luc = parseTuple(ps[1], ":");
    const rdc = parseTuple(ps[2], ":");
    if (gridFormat.x < 1 || luc.x < 0 || rdc.x < 0
        || gridFormat.y < 1 || luc.y < 0 || rdc.y < 0
        || gridFormat.x <= luc.x || gridFormat.x <= rdc.x
        || gridFormat.y <= luc.y || gridFormat.y <= rdc.y
        || luc.x > rdc.x || luc.y > rdc.y) {
        throw new Error(`Bad preset: ${JSON.stringify(preset)}`);
    }
    return new TileSpec(gridFormat.x, gridFormat.y, luc, rdc);
}
function hasImpliedGridSize(singlePreset) {
    return singlePreset.trim().split(" ").length === 2;
}
/**
 * Parses a value like like 6x4 or 1:2 into {X: 6, Y: 4} or {X: 1, Y: 2}.
 */
function parseTuple(unparsed, delim) {
    // parsing grid size in unparsed XdelimY, like 6x4 or 1:2
    const gssk = unparsed.split(delim);
    if (gssk.length !== 2) {
        throw new Error("Failed to split " + unparsed + " by delimiter " + delim + " into two numbers");
    }
    const numbers = gssk.map(Number);
    if (numbers.some(n => isNaN(n) || n < 0 || n > MAX_TUPLE_MEMBER_VALUE)) {
        throw new Error(`All elements of tuple must be intgers in [0, ${MAX_TUPLE_MEMBER_VALUE}]: ${JSON.stringify(unparsed)}`);
    }
    return new XY(numbers[0], numbers[1]);
}
function withinTol(a, b, tol) {
    return Math.abs(a - b) <= tol;
}

/**
 * @fileoverview This file contains incomplete typings for gnome shell types.
 *
 * Probably the best source of definitive API documentation is here:
 * https://gjs-docs.gnome.org/
 *
 * However, there are also some ways the GJS works that make the API docs above
 * slightly incomplete.
 * https://wiki.gnome.org/Projects/GnomeShell/Extensions/StepByStepTutorial
 * mentions that constructors can take a property map as an argument. This file
 * does not correctly type the constructors for these types.
 */
/**
 * From https://gjs-docs.gnome.org/meta4~4_api/meta.frametype.
 */
var FrameType;
(function (FrameType) {
    FrameType[FrameType["NORMAL"] = 0] = "NORMAL";
    FrameType[FrameType["DIALOG"] = 1] = "DIALOG";
    FrameType[FrameType["MODAL_DIALOG"] = 2] = "MODAL_DIALOG";
    FrameType[FrameType["UTILITY"] = 3] = "UTILITY";
    FrameType[FrameType["MENU"] = 4] = "MENU";
    FrameType[FrameType["BORDER"] = 5] = "BORDER";
    FrameType[FrameType["ATTACHED"] = 6] = "ATTACHED";
    FrameType[FrameType["LAST"] = 7] = "LAST";
})(FrameType || (FrameType = {}));
var WindowType;
(function (WindowType) {
    WindowType[WindowType["NORMAL"] = 0] = "NORMAL";
    WindowType[WindowType["DESKTOP"] = 1] = "DESKTOP";
    WindowType[WindowType["DOCK"] = 2] = "DOCK";
    WindowType[WindowType["DIALOG"] = 3] = "DIALOG";
    WindowType[WindowType["MODAL_DIALOG"] = 4] = "MODAL_DIALOG";
    WindowType[WindowType["TOOLBAR"] = 5] = "TOOLBAR";
    WindowType[WindowType["MENU"] = 6] = "MENU";
    WindowType[WindowType["UTILITY"] = 7] = "UTILITY";
    WindowType[WindowType["SPLASHSCREEN"] = 8] = "SPLASHSCREEN";
    WindowType[WindowType["DROPDOWN_MENU"] = 9] = "DROPDOWN_MENU";
    WindowType[WindowType["POPUP_MENU"] = 10] = "POPUP_MENU";
    WindowType[WindowType["TOOLTIP"] = 11] = "TOOLTIP";
    WindowType[WindowType["NOTIFICATION"] = 12] = "NOTIFICATION";
    WindowType[WindowType["COMBO"] = 13] = "COMBO";
    WindowType[WindowType["DND"] = 14] = "DND";
    WindowType[WindowType["OVERRIDE_OTHER"] = 15] = "OVERRIDE_OTHER";
})(WindowType || (WindowType = {}));
var MaximizeFlags;
(function (MaximizeFlags) {
    MaximizeFlags[MaximizeFlags["HORIZONTAL"] = 1] = "HORIZONTAL";
    MaximizeFlags[MaximizeFlags["VERTICAL"] = 2] = "VERTICAL";
    MaximizeFlags[MaximizeFlags["BOTH"] = 3] = "BOTH";
})(MaximizeFlags || (MaximizeFlags = {}));

/*****************************************************************

  This extension has been developed by vibou

  With the help of the gnome-shell community

  Edited by Kvis for gnome 3.8
  Edited by Lundal for gnome 3.18
  Edited by Sergey to add keyboard shortcuts and prefs dialog

 ******************************************************************/
/*****************************************************************
  CONST & VARS
 *****************************************************************/
// Library imports
const St = imports.gi.St;
const Main$1 = imports.ui.main;
const Shell$1 = imports.gi.Shell;
const Lang = imports.lang;
const PanelMenu = imports.ui.panelMenu;
const Meta$2 = imports.gi.Meta;
const Clutter = imports.gi.Clutter;
const Signals = imports.signals;
const Workspace = imports.ui.workspace;
// Getter for accesing "get_active_workspace" on GNOME <=2.28 and >= 2.30
const WorkspaceManager$1 = (global.screen || global.workspace_manager);
// Extension imports
const Me$1 = imports.misc.extensionUtils.getCurrentExtension();
const Settings$1 = Me$1.imports.settings;
// Globals
const SETTINGS_GRID_SIZES = 'grid-sizes';
const SETTINGS_AUTO_CLOSE = 'auto-close';
const SETTINGS_ANIMATION = 'animation';
const SETTINGS_SHOW_ICON = 'show-icon';
const SETTINGS_GLOBAL_PRESETS = 'global-presets';
const SETTINGS_MOVERESIZE_ENABLED = 'moveresize-enabled';
const SETTINGS_WINDOW_MARGIN = 'window-margin';
const SETTINGS_WINDOW_MARGIN_FULLSCREEN_ENABLED = 'window-margin-fullscreen-enabled';
const SETTINGS_MAX_TIMEOUT = 'max-timeout';
const SETTINGS_MAIN_WINDOW_SIZES = 'main-window-sizes';
const SETTINGS_INSETS_PRIMARY = 'insets-primary';
const SETTINGS_INSETS_PRIMARY_LEFT = 'insets-primary-left';
const SETTINGS_INSETS_PRIMARY_RIGHT = 'insets-primary-right';
const SETTINGS_INSETS_PRIMARY_TOP = 'insets-primary-top';
const SETTINGS_INSETS_PRIMARY_BOTTOM = 'insets-primary-bottom';
const SETTINGS_INSETS_SECONDARY = 'insets-secondary';
const SETTINGS_INSETS_SECONDARY_LEFT = 'insets-secondary-left';
const SETTINGS_INSETS_SECONDARY_RIGHT = 'insets-secondary-right';
const SETTINGS_INSETS_SECONDARY_TOP = 'insets-secondary-top';
const SETTINGS_INSETS_SECONDARY_BOTTOM = 'insets-secondary-bottom';
const SETTINGS_DEBUG = 'debug';
const SETTINGS_THEME = 'theme';
const gridSettings = {
    [SETTINGS_GRID_SIZES]: [],
    [SETTINGS_AUTO_CLOSE]: null,
    [SETTINGS_ANIMATION]: null,
    [SETTINGS_SHOW_ICON]: null,
    [SETTINGS_GLOBAL_PRESETS]: null,
    [SETTINGS_MOVERESIZE_ENABLED]: null,
    [SETTINGS_WINDOW_MARGIN]: 0,
    [SETTINGS_WINDOW_MARGIN_FULLSCREEN_ENABLED]: false,
    [SETTINGS_MAX_TIMEOUT]: null,
    [SETTINGS_MAIN_WINDOW_SIZES]: [],
    [SETTINGS_INSETS_PRIMARY]: 0,
    [SETTINGS_INSETS_PRIMARY_LEFT]: 0,
    [SETTINGS_INSETS_PRIMARY_RIGHT]: 0,
    [SETTINGS_INSETS_PRIMARY_TOP]: 0,
    [SETTINGS_INSETS_PRIMARY_BOTTOM]: 0,
    [SETTINGS_INSETS_SECONDARY]: 0,
    [SETTINGS_INSETS_SECONDARY_LEFT]: 0,
    [SETTINGS_INSETS_SECONDARY_RIGHT]: 0,
    [SETTINGS_INSETS_SECONDARY_TOP]: 0,
    [SETTINGS_INSETS_SECONDARY_BOTTOM]: 0,
    [SETTINGS_DEBUG]: null,
    [SETTINGS_THEME]: null,
};
let launcher;
let tracker;
let nbCols = 0;
let nbRows = 0;
let focusMetaWindow = null;
let focusConnect = false;
let settings = Settings$1.get();
settings.connect('changed', changed_settings);
let keyControlBound = false;
let enabled = false;
let mainWindowSizes = new Array();
let monitorsChangedConnect = false;
const SHELL_VERSION = ShellVersion.defaultVersion();
const lastResizeInfo = {
    variantIndex: 0,
    lastCallTime: new Date(),
    presetName: '',
    windowTitle: '',
};
let theme = getTheme();
// Hangouts workaround
let excludedApplications = new Array("Unknown");
const keyBindings = new Map([
    ['show-toggle-tiling', () => { globalApp.toggleTiling(); }],
    ['show-toggle-tiling-alt', () => { globalApp.toggleTiling(); }],
]);
const key_bindings_tiling = new Map([
    ['move-left', () => { keyMoveResizeEvent('move', 'left'); }],
    ['move-right', () => { keyMoveResizeEvent('move', 'right'); }],
    ['move-up', () => { keyMoveResizeEvent('move', 'up'); }],
    ['move-down', () => { keyMoveResizeEvent('move', 'down'); }],
    ['resize-left', () => { keyMoveResizeEvent('resize', 'left'); }],
    ['resize-right', () => { keyMoveResizeEvent('resize', 'right'); }],
    ['resize-up', () => { keyMoveResizeEvent('resize', 'up'); }],
    ['resize-down', () => { keyMoveResizeEvent('resize', 'down'); }],
    ['move-left-vi', () => { keyMoveResizeEvent('move', 'left'); }],
    ['move-right-vi', () => { keyMoveResizeEvent('move', 'right'); }],
    ['move-up-vi', () => { keyMoveResizeEvent('move', 'up'); }],
    ['move-down-vi', () => { keyMoveResizeEvent('move', 'down'); }],
    ['resize-left-vi', () => { keyMoveResizeEvent('resize', 'left'); }],
    ['resize-right-vi', () => { keyMoveResizeEvent('resize', 'right'); }],
    ['resize-up-vi', () => { keyMoveResizeEvent('resize', 'up'); }],
    ['resize-down-vi', () => { keyMoveResizeEvent('resize', 'down'); }],
    ['cancel-tiling', () => { keyCancelTiling(); }],
    ['set-tiling', () => { keySetTiling(); }],
    ['change-grid-size', () => { keyChangeTiling(); }],
    ['autotile-main', () => { AutoTileMain(); }],
    ['autotile-1', () => { autoTileNCols(1); }],
    ['autotile-2', () => { autoTileNCols(2); }],
    ['autotile-3', () => { autoTileNCols(3); }],
    ['autotile-4', () => { autoTileNCols(4); }],
    ['autotile-5', () => { autoTileNCols(5); }],
    ['autotile-6', () => { autoTileNCols(6); }],
    ['autotile-7', () => { autoTileNCols(7); }],
    ['autotile-8', () => { autoTileNCols(8); }],
    ['autotile-9', () => { autoTileNCols(9); }],
    ['autotile-10', () => { autoTileNCols(10); }],
    ['snap-to-neighbors', () => { snapToNeighborsBind(); }],
    ['snap-to-neighbors', () => { snapToNeighborsBind(); }],
]);
const key_bindings_presets = new Map([
    ['preset-resize-1', () => { presetResize(1, 'resize1'); }],
    ['preset-resize-2', () => { presetResize(2, 'resize2'); }],
    ['preset-resize-3', () => { presetResize(3, 'resize3'); }],
    ['preset-resize-4', () => { presetResize(4, 'resize4'); }],
    ['preset-resize-5', () => { presetResize(5, 'resize5'); }],
    ['preset-resize-6', () => { presetResize(6, 'resize6'); }],
    ['preset-resize-7', () => { presetResize(7, 'resize7'); }],
    ['preset-resize-8', () => { presetResize(8, 'resize8'); }],
    ['preset-resize-9', () => { presetResize(9, 'resize9'); }],
    ['preset-resize-10', () => { presetResize(10, 'resize10'); }],
    ['preset-resize-11', () => { presetResize(11, 'resize11'); }],
    ['preset-resize-12', () => { presetResize(12, 'resize12'); }],
    ['preset-resize-13', () => { presetResize(13, 'resize13'); }],
    ['preset-resize-14', () => { presetResize(14, 'resize14'); }],
    ['preset-resize-15', () => { presetResize(15, 'resize15'); }],
    ['preset-resize-16', () => { presetResize(16, 'resize16'); }],
    ['preset-resize-17', () => { presetResize(17, 'resize17'); }],
    ['preset-resize-18', () => { presetResize(18, 'resize18'); }],
    ['preset-resize-19', () => { presetResize(19, 'resize19'); }],
    ['preset-resize-20', () => { presetResize(20, 'resize20'); }],
    ['preset-resize-21', () => { presetResize(21, 'resize21'); }],
    ['preset-resize-22', () => { presetResize(22, 'resize22'); }],
    ['preset-resize-23', () => { presetResize(23, 'resize23'); }],
    ['preset-resize-24', () => { presetResize(24, 'resize24'); }],
    ['preset-resize-25', () => { presetResize(25, 'resize25'); }],
    ['preset-resize-26', () => { presetResize(26, 'resize26'); }],
    ['preset-resize-27', () => { presetResize(27, 'resize27'); }],
    ['preset-resize-28', () => { presetResize(28, 'resize28'); }],
    ['preset-resize-29', () => { presetResize(29, 'resize29'); }],
    ['preset-resize-30', () => { presetResize(30, 'resize30'); }],
]);
const keyBindingGlobalResizes = new Map([
    ['action-change-tiling', () => { keyChangeTiling(); }],
    ['action-contract-bottom', () => { keyMoveResizeEvent('contract', 'bottom', true); }],
    ['action-contract-left', () => { keyMoveResizeEvent('contract', 'left', true); }],
    ['action-contract-right', () => { keyMoveResizeEvent('contract', 'right', true); }],
    ['action-contract-top', () => { keyMoveResizeEvent('contract', 'top', true); }],
    ['action-expand-bottom', () => { keyMoveResizeEvent('expand', 'bottom', true); }],
    ['action-expand-left', () => { keyMoveResizeEvent('expand', 'left', true); }],
    ['action-expand-right', () => { keyMoveResizeEvent('expand', 'right', true); }],
    ['action-expand-top', () => { keyMoveResizeEvent('expand', 'top', true); }],
    ['action-move-down', () => { keyMoveResizeEvent('move', 'down', true); }],
    ['action-move-left', () => { keyMoveResizeEvent('move', 'left', true); }],
    ['action-move-right', () => { keyMoveResizeEvent('move', 'right', true); }],
    ['action-move-up', () => { keyMoveResizeEvent('move', 'up', true); }],
    ['action-move-next-monitor', () => { moveWindowToNextMonitor(); }],
]);
class App {
    constructor() {
        this.gridsByMonitorKey = {};
        this.gridShowing = false;
        this.gridWidget = null;
    }
    enable() {
        this.gridShowing = false;
        tracker = Shell$1.WindowTracker.get_default();
        initSettings();
        const gridWidget = (new St.BoxLayout({ style_class: `${theme}__preview` }));
        this.gridWidget = gridWidget;
        Main$1.uiGroup.add_actor(gridWidget);
        this.initGrids(gridWidget);
        log("Create Button on Panel");
        launcher = new GTileStatusButton(`${theme}__icon`);
        if (gridSettings[SETTINGS_SHOW_ICON]) {
            Main$1.panel.addToStatusArea("GTileStatusButton", launcher);
        }
        bind(keyBindings);
        if (gridSettings[SETTINGS_GLOBAL_PRESETS]) {
            bind(key_bindings_presets);
        }
        if (gridSettings[SETTINGS_MOVERESIZE_ENABLED]) {
            bind(keyBindingGlobalResizes);
        }
        if (monitorsChangedConnect) {
            Main$1.layoutManager.disconnect(monitorsChangedConnect);
        }
        log("Connecting monitors-changed");
        monitorsChangedConnect = Main$1.layoutManager.connect('monitors-changed', () => {
            log("Reinitializing grids on monitors-changed");
            this.destroyGrids();
            this.initGrids(gridWidget);
        });
        enabled = true;
        log("Extention enable completed");
    }
    getGrid(monitor) {
        return this.gridsByMonitorKey[getMonitorKey(monitor)];
    }
    initGrids(gridWidget) {
        log("initGrids");
        log("initGrids nobCols " + nbCols + " nbRows " + nbRows);
        const monitors = activeMonitors();
        for (let monitorIdx = 0; monitorIdx < monitors.length; monitorIdx++) {
            log("New Grid for monitor " + monitorIdx);
            let monitor = monitors[monitorIdx];
            let grid = new Grid(gridWidget, monitorIdx, monitor, "gTile", nbCols, nbRows);
            const key = getMonitorKey(monitor);
            this.gridsByMonitorKey[key] = grid;
            log("initGrids adding grid key " + key);
            // TODO: addChrome is poorly documented. I can't find any reference
            // to it in the gjs-docs site.
            Main$1.layoutManager.addChrome(grid.actor, { trackFullscreen: true });
            grid.actor.set_opacity(0);
            grid.hide(true);
            log("Connect hide-tiling for monitor " + monitorIdx);
            grid.connectHideTiling = grid.connect('hide-tiling', () => this.hideTiling());
        }
        log("Init grid done");
    }
    destroyGrids() {
        log("destroyGrids");
        for (let gridKey in this.gridsByMonitorKey) {
            const grid = this.gridsByMonitorKey[gridKey];
            grid.hide(true);
            Main$1.layoutManager.removeChrome(grid.actor);
            log("Disconnect hide-tiling for monitor " + grid.monitor_idx);
            grid.disconnect(grid.connectHideTiling);
            delete this.gridsByMonitorKey[gridKey];
        }
        log("destroyGrids done");
        for (let gridKey in this.gridsByMonitorKey) {
            log("ERROR: gridKey still found in this.gridsByMonitorKey: " + gridKey);
        }
    }
    refreshGrids() {
        log("refreshGrids");
        for (let gridIdx in this.gridsByMonitorKey) {
            const grid = this.gridsByMonitorKey[gridIdx];
            log("refreshGrids calling refresh on " + gridIdx);
            grid.refresh();
        }
        log("refreshGrids done");
    }
    setGridSize(gridSize) {
        nbCols = gridSize.width;
        nbRows = gridSize.height;
        this.refreshGrids();
    }
    moveGrids() {
        log("moveGrids");
        if (!this.gridShowing) {
            return;
        }
        let window = focusMetaWindow;
        if (window) {
            for (let gridKey in this.gridsByMonitorKey) {
                let grid = this.gridsByMonitorKey[gridKey];
                let pos_x;
                let pos_y;
                const monitor = grid.monitor;
                if (!monitor) {
                    return;
                }
                if (window.get_monitor() == grid.monitor_idx) {
                    pos_x = window.get_frame_rect().width / 2 + window.get_frame_rect().x;
                    pos_y = window.get_frame_rect().height / 2 + window.get_frame_rect().y;
                    let [mouse_x, mouse_y, mask] = global.get_pointer();
                    let act_x = pos_x - grid.actor.width / 2;
                    let act_y = pos_y - grid.actor.height / 2;
                    if (mouse_x >= act_x
                        && mouse_x <= act_x + grid.actor.width
                        && mouse_y >= act_y
                        && mouse_y <= act_y + grid.actor.height) {
                        log("Mouse x " + mouse_x + " y " + mouse_y +
                            " is inside grid actor rectangle, changing actor X from " + pos_x + " to " + (mouse_x + grid.actor.width / 2) +
                            ", Y from " + pos_y + " to " + (mouse_y + grid.actor.height / 2));
                        pos_x = mouse_x + grid.actor.width / 2;
                        pos_y = mouse_y + grid.actor.height / 2;
                    }
                }
                else {
                    pos_x = monitor.x + monitor.width / 2;
                    pos_y = monitor.y + monitor.height / 2;
                }
                pos_x = Math.floor(pos_x - grid.actor.width / 2);
                pos_y = Math.floor(pos_y - grid.actor.height / 2);
                if (window.get_monitor() == grid.monitor_idx) {
                    pos_x = (pos_x < monitor.x) ? monitor.x : pos_x;
                    pos_x = ((pos_x + grid.actor.width) > (monitor.width + monitor.x)) ? monitor.x + monitor.width - grid.actor.width : pos_x;
                    pos_y = (pos_y < monitor.y) ? monitor.y : pos_y;
                    pos_y = ((pos_y + grid.actor.height) > (monitor.height + monitor.y)) ? monitor.y + monitor.height - grid.actor.height : pos_y;
                }
                let time = (gridSettings[SETTINGS_ANIMATION]) ? 0.3 : 0.1;
                grid.actor.ease({
                    time: time,
                    x: pos_x,
                    y: pos_y,
                    transition: Clutter.AnimationMode.EASE_OUT_QUAD,
                });
            }
        }
    }
    updateRegions() {
        var _a;
        /*Main.layoutManager._chrome.updateRegions();*/
        log("updateRegions");
        this.refreshGrids();
        for (let idx in this.gridsByMonitorKey) {
            (_a = this.gridsByMonitorKey[idx].elementsDelegate) === null || _a === void 0 ? void 0 : _a.reset();
        }
    }
    logState() {
        let count = 0;
        let states = '';
        for (let gridKey in this.gridsByMonitorKey) {
            states += `; ${this.gridsByMonitorKey[gridKey].state()}`;
            count++;
        }
        log(`${count} grids; showing = ${this.gridShowing}: ${states}`);
    }
    showTiling() {
        // TODO(#168): See https://github.com/gTile/gTile/issues/168. Without
        // these two lines, the grid UI does not properly respond to mouseover
        // and other events except for the first time it is shown.
        log("showTiling with fix");
        this.hideTiling();
        this.destroyGrids();
        this.initGrids(this.gridWidget);
        this.logState();
        log("issue#168/showTiling with fix");
        const window = getFocusApp();
        // TODO: remove this global side effect
        focusMetaWindow = window;
        if (!this.gridWidget) {
            return;
        }
        const shouldShowTiling = (() => {
            if (!window) {
                // The tiling UI is for manipulating a particular window.
                return false;
            }
            const wmType = window.get_window_type();
            const layer = window.get_layer();
            return wmType !== WindowType.DESKTOP && layer > 0;
        })();
        if (!shouldShowTiling) {
            return;
        }
        this.gridWidget.visible = true;
        const monitors = activeMonitors();
        for (let monitorIdx = 0; monitorIdx < monitors.length; monitorIdx++) {
            const monitor = monitors[monitorIdx];
            const grid = this.getGrid(monitor);
            if (grid === null) {
                log(`issue#168/showTiling ERROR: did not find grid for monitor ${getMonitorKey(monitor)}`);
                continue;
            }
            let pos_x;
            let pos_y;
            if (window && window.get_monitor() === monitorIdx) {
                // Set pos_x, pos_y to center of the selected window initially.
                pos_x = window.get_frame_rect().width / 2 + window.get_frame_rect().x;
                pos_y = window.get_frame_rect().height / 2 + window.get_frame_rect().y;
                // Old logic with no justifying comments: If the mouse is active
                // and within the rectangle,  set pos_x and pos_y  at mouse
                // position + half the size of the Grid window.
                let [mouse_x, mouse_y, mask] = global.get_pointer();
                let act_x = pos_x - grid.actor.width / 2;
                let act_y = pos_y - grid.actor.height / 2;
                if (mouse_x >= act_x
                    && mouse_x <= act_x + grid.actor.width
                    && mouse_y >= act_y
                    && mouse_y <= act_y + grid.actor.height) {
                    log("Mouse x " + mouse_x + " y " + mouse_y +
                        " is inside grid actor rectangle, changing actor X from " + pos_x + " to " + (mouse_x + grid.actor.width / 2) +
                        ", Y from " + pos_y + " to " + (mouse_y + grid.actor.height / 2));
                    pos_x = mouse_x + grid.actor.width / 2;
                    pos_y = mouse_y + grid.actor.height / 2;
                }
            }
            else {
                pos_x = monitor.x + monitor.width / 2;
                pos_y = monitor.y + monitor.height / 2;
            }
            grid.set_position(Math.floor(pos_x - grid.actor.width / 2), Math.floor(pos_y - grid.actor.height / 2));
            grid.show();
        }
        this.gridShowing = true;
        this.onFocus();
        launcher === null || launcher === void 0 ? void 0 : launcher.activate();
        bindKeyControls();
        this.moveGrids();
    }
    disable() {
        log("Extension disable begin");
        enabled = false;
        if (monitorsChangedConnect) {
            log("Disconnecting monitors-changed");
            Main$1.layoutManager.disconnect(monitorsChangedConnect);
            monitorsChangedConnect = false;
        }
        unbind(keyBindings);
        unbind(key_bindings_presets);
        unbind(keyBindingGlobalResizes);
        if (keyControlBound) {
            unbind(key_bindings_tiling);
            keyControlBound = false;
        }
        launcher === null || launcher === void 0 ? void 0 : launcher.destroy();
        launcher = null;
        Main$1.uiGroup.remove_actor(this.gridWidget);
        this.destroyGrids();
        resetFocusMetaWindow();
        log("Extention disable completed");
    }
    hideTiling() {
        var _a;
        log("hideTiling");
        for (let key in this.gridsByMonitorKey) {
            const grid = this.gridsByMonitorKey[key];
            (_a = grid.elementsDelegate) === null || _a === void 0 ? void 0 : _a.reset();
            grid.hide(false);
        }
        if (this.gridWidget) {
            this.gridWidget.visible = false;
        }
        resetFocusMetaWindow();
        launcher === null || launcher === void 0 ? void 0 : launcher.deactivate();
        this.gridShowing = false;
        unbindKeyControls();
    }
    toggleTiling() {
        if (this.gridShowing) {
            this.hideTiling();
        }
        else {
            this.showTiling();
        }
        return this.gridShowing;
    }
    /**
     * onFocus is called when the global focus changes.
     */
    onFocus() {
        log("_onFocus ");
        resetFocusMetaWindow();
        const window = getFocusApp();
        if (window && this.gridShowing) {
            log("_onFocus " + window.get_title());
            focusMetaWindow = window;
            let app = tracker.get_window_app(focusMetaWindow);
            let title = focusMetaWindow.get_title();
            const monitors = activeMonitors();
            for (let monitorIdx = 0; monitorIdx < monitors.length; monitorIdx++) {
                let monitor = monitors[monitorIdx];
                let grid = this.getGrid(monitor);
                if (app) {
                    grid === null || grid === void 0 ? void 0 : grid.topbar.setApp(app, title);
                }
                else {
                    grid === null || grid === void 0 ? void 0 : grid.topbar.setTitle(title);
                }
            }
            this.moveGrids();
        }
        else {
            if (this.gridShowing) {
                log("No focus window, hide tiling");
                this.hideTiling();
            }
            else {
                log("tiling window not active");
            }
        }
    }
}
const globalApp = new App();
function changed_settings() {
    log("changed_settings");
    if (enabled) {
        disable();
        enable();
    }
    log("changed_settings complete");
}
const GTileStatusButton = new Lang.Class({
    Name: 'GTileStatusButton',
    Extends: PanelMenu.Button,
    _init: function (classname) {
        this.parent(0.0, "gTile", false);
        //Done by default in PanelMenuButton - Just need to override the method
        if (SHELL_VERSION.version_at_least_34()) {
            this.add_style_class_name(classname);
            this.connect('button-press-event', this._onButtonPress.bind(this));
        }
        else {
            this.actor.add_style_class_name(classname);
            this.actor.connect('button-press-event', this._onButtonPress.bind(this));
        }
        log("GTileStatusButton _init done");
    },
    reset: function () {
        this.activated = false;
        if (SHELL_VERSION.version_at_least_34()) {
            this.remove_style_pseudo_class('activate');
        }
        else {
            this.actor.remove_style_pseudo_class('activate');
        }
    },
    activate: function () {
        if (SHELL_VERSION.version_at_least_34()) {
            this.add_style_pseudo_class('activate');
        }
        else {
            this.actor.add_style_pseudo_class('activate');
        }
    },
    deactivate: function () {
        if (SHELL_VERSION.version_at_least_34()) {
            this.remove_style_pseudo_class('activate');
        }
        else {
            this.actor.remove_style_pseudo_class('activate');
        }
    },
    _onButtonPress: function (actor, event) {
        log(`_onButtonPress Click Toggle Status on system panel ${this}`);
        globalApp.toggleTiling();
    },
    _destroy: function () {
        this.activated = null;
    }
});
/*****************************************************************
  SETTINGS
 *****************************************************************/
function initGridSizes(configValue) {
    let gridSizes = parseGridSizesIgnoringErrors(configValue);
    if (gridSizes.length === 0) {
        gridSizes = [
            new GridSize(8, 6),
            new GridSize(6, 4),
            new GridSize(4, 4),
        ];
    }
    gridSettings[SETTINGS_GRID_SIZES] = gridSizes;
}
function getBoolSetting(settingName) {
    const value = settings.get_boolean(settingName);
    if (value === undefined) {
        log("Undefined settings " + settingName);
        gridSettings[settingName] = false;
        return false;
    }
    else {
        gridSettings[settingName] = value;
    }
    return value;
}
function getIntSetting(settingsValue) {
    let iss = settings.get_int(settingsValue);
    if (iss === undefined) {
        log("Undefined settings " + settingsValue);
        return 0;
    }
    else {
        return iss;
    }
}
function initSettings() {
    log("Init settings");
    theme = getTheme();
    const gridSizes = settings.get_string(SETTINGS_GRID_SIZES) || '';
    log(SETTINGS_GRID_SIZES + " set to " + gridSizes);
    initGridSizes(gridSizes);
    getBoolSetting(SETTINGS_AUTO_CLOSE);
    getBoolSetting(SETTINGS_ANIMATION);
    getBoolSetting(SETTINGS_SHOW_ICON);
    getBoolSetting(SETTINGS_GLOBAL_PRESETS);
    getBoolSetting(SETTINGS_MOVERESIZE_ENABLED);
    gridSettings[SETTINGS_WINDOW_MARGIN] = getIntSetting(SETTINGS_WINDOW_MARGIN);
    gridSettings[SETTINGS_WINDOW_MARGIN_FULLSCREEN_ENABLED] = getBoolSetting(SETTINGS_WINDOW_MARGIN_FULLSCREEN_ENABLED);
    gridSettings[SETTINGS_MAX_TIMEOUT] = getIntSetting(SETTINGS_MAX_TIMEOUT);
    // initialize these from settings, the first set of sizes
    if (nbCols == 0 || nbRows == 0) {
        nbCols = gridSettings[SETTINGS_GRID_SIZES][0].width;
        nbRows = gridSettings[SETTINGS_GRID_SIZES][0].height;
    }
    const mainWindowSizesString = settings.get_string(SETTINGS_MAIN_WINDOW_SIZES);
    log(SETTINGS_MAIN_WINDOW_SIZES + " settings found " + mainWindowSizesString);
    mainWindowSizes = [];
    let mainWindowSizesArray = mainWindowSizesString.split(",");
    for (var i in mainWindowSizesArray) {
        let size = mainWindowSizesArray[i];
        if (size.includes("/")) {
            let fraction = size.split("/");
            let ratio = parseFloat(fraction[0]) / parseFloat(fraction[1]);
            mainWindowSizes.push(ratio);
        }
        else {
            mainWindowSizes.push(parseFloat(size));
        }
    }
    log(SETTINGS_MAIN_WINDOW_SIZES + " set to " + mainWindowSizes);
    log("Init complete, nbCols " + nbCols + " nbRows " + nbRows);
}
function getMonitorTier(monitor) {
    return isPrimaryMonitor(monitor) ? 'primary' : 'secondary';
}
function getMonitorInsets(tier) {
    switch (tier) {
        case 'primary':
            return {
                top: getIntSetting(SETTINGS_INSETS_PRIMARY_TOP),
                bottom: getIntSetting(SETTINGS_INSETS_PRIMARY_BOTTOM),
                left: getIntSetting(SETTINGS_INSETS_PRIMARY_LEFT),
                right: getIntSetting(SETTINGS_INSETS_PRIMARY_RIGHT)
            }; // Insets on primary monitor
        case 'secondary':
            return {
                top: getIntSetting(SETTINGS_INSETS_SECONDARY_TOP),
                bottom: getIntSetting(SETTINGS_INSETS_SECONDARY_BOTTOM),
                left: getIntSetting(SETTINGS_INSETS_SECONDARY_LEFT),
                right: getIntSetting(SETTINGS_INSETS_SECONDARY_RIGHT)
            };
        default:
            throw new Error(`unknown monitor name ${JSON.stringify(tier)}`);
    }
}
function enable() {
    setLoggingEnabled(getBoolSetting(SETTINGS_DEBUG));
    log("Extension enable begin");
    SHELL_VERSION.print_version();
    globalApp.enable();
}
function disable() {
    globalApp.disable();
}
function resetFocusMetaWindow() {
    log("resetFocusMetaWindow");
    focusMetaWindow = null;
}
function reset_window(metaWindow) {
    metaWindow.unmaximize(Meta$2.MaximizeFlags.HORIZONTAL);
    metaWindow.unmaximize(Meta$2.MaximizeFlags.VERTICAL);
    metaWindow.unmaximize(Meta$2.MaximizeFlags.BOTH);
}
function _getInvisibleBorderPadding(metaWindow) {
    let outerRect = metaWindow.get_frame_rect();
    let inputRect = metaWindow.get_buffer_rect();
    let borderX = outerRect.x - inputRect.x;
    let borderY = outerRect.y - inputRect.y;
    return [borderX, borderY];
}
function _getVisibleBorderPadding(metaWindow) {
    let clientRect = metaWindow.get_frame_rect();
    let outerRect = metaWindow.get_frame_rect();
    let borderX = outerRect.width - clientRect.width;
    let borderY = outerRect.height - clientRect.height;
    return [borderX, borderY];
}
function move_maximize_window(metaWindow, x, y) {
    const [borderX, borderY] = _getInvisibleBorderPadding(metaWindow);
    x = x - borderX;
    y = y - borderY;
    metaWindow.move_frame(true, x, y);
    metaWindow.maximize(Meta$2.MaximizeFlags.HORIZONTAL | Meta$2.MaximizeFlags.VERTICAL);
}
/**
 * Resizes window considering margin settings
 * @param metaWindow
 * @param x
 * @param y
 * @param width
 * @param height
 */
function moveResizeWindowWithMargins(metaWindow, x, y, width, height) {
    let [borderX, borderY] = _getInvisibleBorderPadding(metaWindow);
    let [vBorderX, vBorderY] = _getVisibleBorderPadding(metaWindow);
    log("move_resize_window_with_margins " + metaWindow.get_title() + " " + x + ":" + y + " - " + width
        + ":" + height + " margin " + gridSettings[SETTINGS_WINDOW_MARGIN] + " borders invisible " +
        borderX + ":" + borderY + " visible " + vBorderX + ":" + vBorderY);
    x = x + gridSettings[SETTINGS_WINDOW_MARGIN];
    y = y + gridSettings[SETTINGS_WINDOW_MARGIN];
    width = width - gridSettings[SETTINGS_WINDOW_MARGIN] * 2;
    height = height - gridSettings[SETTINGS_WINDOW_MARGIN] * 2;
    x = x + vBorderX;
    y = y + vBorderY;
    width = width - 2 * vBorderX;
    height = height - 2 * vBorderY;
    log("After margins and visible border window is " + x + ":" + y + " - " + width + ":" + height);
    metaWindow.move_frame(true, x, y);
    metaWindow.move_resize_frame(true, x, y, width, height);
}
function getNotFocusedWindowsOfMonitor(monitor) {
    const monitors = activeMonitors();
    let windows = global.get_window_actors().filter(function (w) {
        let app = tracker.get_window_app(w.meta_window);
        if (app == null) {
            return false;
        }
        let appName = app.get_name();
        return !contains(excludedApplications, appName)
            && w.meta_window.get_window_type() == Meta$2.WindowType.NORMAL
            && w.meta_window.get_workspace() == WorkspaceManager$1.get_active_workspace()
            && w.meta_window.showing_on_its_workspace()
            && monitors[w.meta_window.get_monitor()] == monitor
            && focusMetaWindow != w.meta_window;
    });
    return windows;
}
function getMonitorKey(monitor) {
    return monitor.x + ":" + monitor.width + ":" + monitor.y + ":" + monitor.height;
}
function contains(a, obj) {
    var i = a.length;
    while (i--) {
        if (a[i] === obj) {
            return true;
        }
    }
    return false;
}
/**
 * Get focused window by iterating though the windows on the active workspace.
 * @returns {Object} The focussed window object. False if no focussed window was found.
 */
function getFocusApp() {
    if (tracker.focus_app == null) {
        return null;
    }
    let focusedAppName = tracker.focus_app.get_name();
    if (contains(excludedApplications, focusedAppName)) {
        return null;
    }
    return WorkspaceManager$1.get_active_workspace().list_windows().find((window) => window.has_focus()) || null;
}
function getFocusWindow() {
    const focus_app = tracker.focus_app;
    if (!focus_app || excludedApplications[focus_app.get_name()]) {
        return null;
    }
    return WorkspaceManager$1.get_active_workspace().list_windows()
        .find(w => w.has_focus());
}
function activeMonitors() {
    return Main$1.layoutManager.monitors;
}
/**
 * Determine if the given monitor is the primary monitor.
 * @param {Object} monitor The given monitor to evaluate.
 * @returns {boolean} True if the given monitor is the primary monitor.
 * */
function isPrimaryMonitor(monitor) {
    return Main$1.layoutManager.primaryMonitor.x == monitor.x && Main$1.layoutManager.primaryMonitor.y == monitor.y;
}
function getWorkAreaByMonitor(monitor) {
    const monitors = activeMonitors();
    for (let monitor_idx = 0; monitor_idx < monitors.length; monitor_idx++) {
        let mon = monitors[monitor_idx];
        if (mon.x == monitor.x && mon.y == monitor.y) {
            return getWorkArea(monitor, monitor_idx);
        }
    }
    return null;
}
/**
 * @deprecated Use {@link workAreaRectByMonitorIndex} instead.
 */
function getWorkAreaByMonitorIdx(monitor_idx) {
    const monitors = activeMonitors();
    let monitor = monitors[monitor_idx];
    return getWorkArea(monitor, monitor_idx);
}
function workAreaRectByMonitorIndex(monitorIndex) {
    const monitor = activeMonitors()[monitorIndex];
    if (!monitor) {
        return null;
    }
    const waLegacy = getWorkArea(monitor, monitorIndex);
    return (new Rect(new XY(waLegacy.x, waLegacy.y), new Size(waLegacy.width, waLegacy.height)));
}
/**
 * @deprecated Use {@link workAreaRectByMonitorIndex} instead.
 */
function getWorkArea(monitor, monitor_idx) {
    const wkspace = WorkspaceManager$1.get_active_workspace();
    const work_area = wkspace.get_work_area_for_monitor(monitor_idx);
    const insets = getMonitorInsets(getMonitorTier(monitor));
    return {
        x: work_area.x + insets.left,
        y: work_area.y + insets.top,
        width: work_area.width - insets.left - insets.right,
        height: work_area.height - insets.top - insets.bottom
    };
}
function bindKeyControls() {
    if (!keyControlBound) {
        bind(key_bindings_tiling);
        if (focusConnect) {
            global.display.disconnect(focusConnect);
        }
        focusConnect = global.display.connect('notify::focus-window', () => globalApp.onFocus());
        if (!gridSettings[SETTINGS_GLOBAL_PRESETS]) {
            bind(key_bindings_presets);
        }
        keyControlBound = true;
    }
}
function unbindKeyControls() {
    if (keyControlBound) {
        unbind(key_bindings_tiling);
        if (focusConnect) {
            log("Disconnect notify:focus-window");
            global.display.disconnect(focusConnect);
            focusConnect = false;
        }
        if (!gridSettings[SETTINGS_GLOBAL_PRESETS]) {
            unbind(key_bindings_presets);
        }
        if (!gridSettings[SETTINGS_MOVERESIZE_ENABLED]) {
            unbind(keyBindingGlobalResizes);
        }
        keyControlBound = false;
    }
}
function keyCancelTiling() {
    log("Cancel key event");
    globalApp.hideTiling();
}
function keySetTiling() {
    var _a, _b;
    log("keySetTiling");
    if (focusMetaWindow) {
        const monitors = activeMonitors();
        let mind = focusMetaWindow.get_monitor();
        let monitor = monitors[mind];
        let mkey = getMonitorKey(monitor);
        const grid = globalApp.getGrid(monitor);
        log("In grid " + grid);
        (_b = (_a = grid === null || grid === void 0 ? void 0 : grid.elementsDelegate) === null || _a === void 0 ? void 0 : _a.currentElement) === null || _b === void 0 ? void 0 : _b._onButtonPress();
    }
}
function keyChangeTiling() {
    log("keyChangeTiling. Current nbCols " + nbCols + " nbRos " + nbRows);
    const gridSizes = gridSettings[SETTINGS_GRID_SIZES];
    if (gridSizes.length === 0) {
        return;
    }
    const currentIndex = gridSizes.findIndex((size) => size.width === nbCols && size.height === nbRows);
    const newIndex = currentIndex === -1 ? 0 : (currentIndex + 1) % gridSizes.length;
    const newGridSize = gridSizes[newIndex];
    globalApp.setGridSize(newGridSize);
    setInitialSelection();
}
function setInitialSelection() {
    if (!focusMetaWindow) {
        return;
    }
    let mind = focusMetaWindow.get_monitor();
    const monitors = activeMonitors();
    let monitor = monitors[mind];
    let workArea = getWorkArea(monitor, mind);
    let wx = focusMetaWindow.get_frame_rect().x;
    let wy = focusMetaWindow.get_frame_rect().y;
    let wwidth = focusMetaWindow.get_frame_rect().width;
    let wheight = focusMetaWindow.get_frame_rect().height;
    const grid = globalApp.getGrid(monitor);
    if (!grid) {
        log("no grid widget available in setInitialSelection");
        return;
    }
    grid.setInitialSelection(focusMetaWindow);
}
function keyMoveResizeEvent(type, key, is_global = false) {
    var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k, _l, _m, _o, _p;
    if (is_global) {
        focusMetaWindow = getFocusApp();
    }
    log("Got key event " + type + " " + key);
    if (!focusMetaWindow) {
        return;
    }
    log("Going on..");
    let mind = focusMetaWindow.get_monitor();
    const monitors = activeMonitors();
    let monitor = monitors[mind];
    const grid = globalApp.getGrid(monitor);
    if (!grid) {
        return;
    }
    const delegate = grid.elementsDelegate;
    if (!(delegate === null || delegate === void 0 ? void 0 : delegate.currentElement)) {
        log("Key event while no mouse activation - set current and second element");
        setInitialSelection();
    }
    else {
        if (!delegate.first) {
            log("currentElement is there but no first yet");
            delegate.currentElement._onButtonPress();
        }
    }
    if (!delegate) {
        return;
    }
    if (!delegate.currentElement) {
        log("bug: keyMoveResizeEvent currentElement is not set!");
        return;
    }
    const cX = delegate.currentElement.coordx;
    const cY = delegate.currentElement.coordy;
    const fX = (_a = delegate.first) === null || _a === void 0 ? void 0 : _a.coordx;
    const fY = (_b = delegate.first) === null || _b === void 0 ? void 0 : _b.coordy;
    log("Before move/resize first fX " + fX + " fY " + fY + " current cX " + cX + " cY " + cY);
    log("Grid cols " + nbCols + " rows " + nbRows);
    if (type === 'move') {
        if (fX === undefined || fY === undefined) {
            log(`bug: tried to move window without a 'first' selection`);
            return;
        }
        switch (key) {
            case 'right':
                if (fX < nbCols - 1 && cX < nbCols - 1) {
                    delegate.first = grid.getElement(fY, fX + 1);
                    (_c = grid.getElement(cY, cX + 1)) === null || _c === void 0 ? void 0 : _c._onHoverChanged();
                }
                break;
            case 'left':
                if (fX > 0 && cX > 0) {
                    delegate.first = grid.getElement(fY, fX - 1);
                    (_d = grid.getElement(cY, cX - 1)) === null || _d === void 0 ? void 0 : _d._onHoverChanged();
                }
                break;
            case 'up':
                if (fY > 0 && cY > 0) {
                    delegate.first = grid.getElement(fY - 1, fX);
                    (_e = grid.getElement(cY - 1, cX)) === null || _e === void 0 ? void 0 : _e._onHoverChanged();
                }
                break;
            case 'down':
                if (fY < nbRows - 1 && cY < nbRows - 1) {
                    delegate.first = grid.getElement(fY + 1, fX);
                    (_f = grid.getElement(cY + 1, cX)) === null || _f === void 0 ? void 0 : _f._onHoverChanged();
                }
                break;
        }
    }
    else if (type == "resize") {
        switch (key) {
            case 'right':
                if (cX < nbCols - 1) {
                    (_g = grid.getElement(cY, cX + 1)) === null || _g === void 0 ? void 0 : _g._onHoverChanged();
                }
                break;
            case 'left':
                if (cX > 0) {
                    (_h = grid.getElement(cY, cX - 1)) === null || _h === void 0 ? void 0 : _h._onHoverChanged();
                }
                break;
            case 'up':
                if (cY > 0) {
                    (_j = grid.getElement(cY - 1, cX)) === null || _j === void 0 ? void 0 : _j._onHoverChanged();
                }
                break;
            case 'down':
                if (cY < nbRows - 1) {
                    (_k = grid.getElement(cY + 1, cX)) === null || _k === void 0 ? void 0 : _k._onHoverChanged();
                }
                break;
        }
    }
    else if (type == "contract") {
        if (fX === undefined || fY === undefined) {
            log(`bug: tried to contract window without a 'first' selection`);
            return;
        }
        switch (key) {
            case 'left':
                // Contract left edge of current window right one column
                if (cX > fX) {
                    delegate.first = grid.getElement(fY, fX + 1);
                }
                break;
            case 'right':
                // Contract right edge of current window left one column
                if (cX > fX) {
                    (_l = grid.getElement(cY, cX - 1)) === null || _l === void 0 ? void 0 : _l._onHoverChanged();
                }
                break;
            case 'top':
                // Contract top edge of current window down one row
                if (cY > fY) {
                    delegate.first = grid.getElement(fY + 1, fX);
                }
                break;
            case 'bottom':
                // Contract bottom edge of current window up one row
                if (cY > fY) {
                    (_m = grid.getElement(cY - 1, cX)) === null || _m === void 0 ? void 0 : _m._onHoverChanged();
                }
                break;
        }
    }
    else if (type == "expand") {
        if (fX === undefined || fY === undefined) {
            log(`bug: tried to expand window without a 'first' selection`);
            return;
        }
        switch (key) {
            case 'right':
                if (cX < nbCols) {
                    (_o = grid.getElement(cY, cX + 1)) === null || _o === void 0 ? void 0 : _o._onHoverChanged();
                }
                break;
            case 'left':
                if (fX > 0) {
                    delegate.first = grid.getElement(fY, fX - 1);
                }
                break;
            case 'top':
                if (fY > 0) {
                    delegate.first = grid.getElement(fY - 1, fX);
                }
                break;
            case 'bottom':
                if (cY < nbRows - 1) {
                    (_p = grid.getElement(cY + 1, cX)) === null || _p === void 0 ? void 0 : _p._onHoverChanged();
                }
                break;
        }
    }
    if (is_global) {
        keySetTiling();
    }
}
/**
 * Resize window to the given preset.
 * @param  {number}  Identifier of the resize preset (1 - 30)
 */
function presetResize(presetName, settingName) {
    // Check if there's a focusable window
    let window = getFocusApp();
    if (!window) {
        log("No focused window - ignoring keyboard shortcut");
        return;
    }
    // Lets assume window titles are always unique.
    // Note: window roles 'window.get_role()' would offer a unique identifier.
    // Unfortunately role is often set to NULL.
    log("presetResize window title: " + window.get_title());
    // Ensure that the window is not maximized
    reset_window(window);
    // Fetch, parse and validate the given preset.
    // Expected preset format is "XxY x1:y1 x2:y2[,x1:y1 x2:y2]":
    //  - XxY is grid size like 6x8
    //  - x1:y1 is left upper corner tile coordinates in grid tiles, starting from 0
    //  - x2:y2 is right down corner tile coordinates in grid tiles
    //  - a preset can define multiple variants (e.g. "3x2 0:0 0:1,0:0 1:1,0:0 2:1")
    //  - variants can be activated using the same shortcut consecutively
    const presetString = settings.get_string(settingName) || '';
    log("Preset resize " + presetName + "  is " + presetString);
    let tileSpecs = [];
    try {
        tileSpecs = parsePreset(presetString);
    }
    catch (err) {
        log(`Bad preset ${presetName} ${JSON.stringify(presetString)}: ${err}`);
        return;
    }
    if (tileSpecs.length === 0) {
        return;
    }
    // parse the preset string (grid size, left-upper-corner, right-down-corner)
    let gridSize = tileSpecs[0].gridSize;
    // handle preset variants (if there are any)
    let variantCount = tileSpecs.length;
    if (variantCount > 1) {
        if ((lastResizeInfo.lastCallTime.getTime() + gridSettings[SETTINGS_MAX_TIMEOUT]) > new Date().getTime() &&
            lastResizeInfo.presetName === presetName.toString() &&
            lastResizeInfo.windowTitle == window.get_title()) {
            // within timeout (default: 2s), same preset & same window:
            // increase variant counter, but consider upper boundary
            lastResizeInfo.variantIndex = (lastResizeInfo.variantIndex + 1) % variantCount;
        }
        else {
            // timeout, new preset or new window:
            // update presetState.last_preset and reset variant counter
            lastResizeInfo.variantIndex = 0;
        }
    }
    else {
        lastResizeInfo.variantIndex = 0;
    }
    // retrieve current preset variant
    const tileSpec = tileSpecs[lastResizeInfo.variantIndex];
    // do the maths to resize the window
    const workArea = workAreaRectByMonitorIndex(window.get_monitor());
    if (!workArea) {
        log(`Failed to get active work area for window while performing preset ${presetName} ${JSON.stringify(presetString)}`);
        return;
    }
    // The rectangle already incorporates insets, but it doesn't incorporate
    // window margin.
    const zeroMargins = tileSpec.isFullscreen() && !getBoolSetting(SETTINGS_WINDOW_MARGIN_FULLSCREEN_ENABLED);
    const marginSize = new Size(zeroMargins ? 0 : gridSettings[SETTINGS_WINDOW_MARGIN], zeroMargins ? 0 : gridSettings[SETTINGS_WINDOW_MARGIN]);
    const rect = tileSpec.toFrameRect(workArea).inset(marginSize);
    moveWindowToRect(window, rect);
    lastResizeInfo.presetName = presetName.toString();
    lastResizeInfo.windowTitle = window.get_title();
    lastResizeInfo.lastCallTime = new Date();
}
// Move the window to the next monitor.
function moveWindowToNextMonitor() {
    log("moveWindowToNextMonitor");
    let window = getFocusWindow();
    if (!window) {
        log("No focused window - ignoring keyboard shortcut to move window");
        return;
    }
    reset_window(window);
    const numMonitors = activeMonitors().length;
    if (numMonitors == 0) {
        return;
    }
    const ts = parsePreset("5x5 1:1 3:3")[0];
    const srcMonitorIndex = window.get_monitor();
    const dstMonitorIndex = (srcMonitorIndex + 1) % numMonitors;
    const workArea = workAreaRectByMonitorIndex(dstMonitorIndex);
    if (!workArea) {
        log(`Failed to get active work area for window while moving it to the next monitor.`);
        return;
    }
    const rect = ts.toFrameRect(workArea);
    moveWindowToRect(window, rect);
}
/**
 * Moves a window to the specified region. This may resize the window as well as
 * move its origin.
 */
function moveWindowToRect(window, rect) {
    // The move_frame line is a workaround for a bug affecting gnome terminal
    // windows. See https://github.com/gTile/gTile/issues/176#issue-751198339.
    window.move_frame(true, rect.origin.x, rect.origin.y);
    window.move_resize_frame(true, rect.origin.x, rect.origin.y, rect.size.width, rect.size.height);
}
/*****************************************************************
  PROTOTYPES
 *****************************************************************/
class TopBar {
    constructor(_title) {
        this._title = _title;
        this.actor = new St.BoxLayout({
            style_class: `${theme}__title-container`
        });
        this._stlabel = new St.Label({
            style_class: `${theme}__title`,
            text: this._title
        });
        this._closebutton = new St.Button({
            style_class: `${theme}__close`,
        });
        this._closebutton.add_style_class_name(`${theme}__close-container`);
        this._connect_id = this._closebutton.connect('button-press-event', Lang.bind(this, this._onButtonPress));
        this.actor.add_child(this._closebutton);
        this.actor.add_child(this._stlabel);
    }
    setTitle(title) {
        this._title = title;
        this._stlabel.text = this._title;
    }
    setApp(app, title) {
        this._title = app.get_name() + " - " + title;
        log("title: " + this._title);
        this._stlabel.text = this._title;
    }
    _onButtonPress() {
        globalApp.hideTiling();
    }
    destroy() {
        this._closebutton.disconnect(this._connect_id);
    }
}
class ToggleSettingsButtonListener {
    constructor() {
        this.actors = [];
    }
    addActor(actor) {
        log("ToggleSettingsButtonListener Connect update-toggle");
        actor.connect('update-toggle', this._updateToggle.bind(this));
        this.actors.push(actor);
    }
    _updateToggle() {
        log("ToggleSettingsButtonListener _updateToggle");
        for (let actor of this.actors) {
            actor._update();
        }
    }
}
function ToggleSettingsButton(text, property) {
    this._init(text, property);
}
ToggleSettingsButton.prototype = {
    _init: function (text, property) {
        this.text = text;
        this.actor = new St.Button({
            style_class: `${theme}__action-button`,
            reactive: true,
            can_focus: true,
            track_hover: true
        });
        this.label = new St.Label({
            style_class: `${theme}__action-label`,
            reactive: true,
            can_focus: true,
            track_hover: true,
            text: this.text
        });
        this.icon = new St.BoxLayout({
            style_class: `${theme}__action-button--${this.text}`,
            reactive: true,
            can_focus: true,
            track_hover: true
        });
        this.actor.add_actor(this.icon);
        this.property = property;
        this._update();
        log("ToggleSettingsButton Connect button-press-event");
        this.actor.connect('button-press-event', this._onButtonPress.bind(this));
        log("ToggleSettingsButton Connect update-toggle");
        this.connect('update-toggle', this._update.bind(this));
    },
    _update: function () {
        log("ToggleSettingsButton _update event " + this.property);
        if (gridSettings[this.property]) {
            this.actor.add_style_pseudo_class('activate');
        }
        else {
            this.actor.remove_style_pseudo_class('activate');
        }
    },
    _onButtonPress: function () {
        gridSettings[this.property] = !gridSettings[this.property];
        log("ToggleSettingsButton _onButtonPress " + this.property + ": " + gridSettings[this.property] + ", emitting signal update-toggle");
        this.emit('update-toggle');
    }
};
Signals.addSignalMethods(ToggleSettingsButton.prototype);
const ACTION_CLASSES = {
    BUTTON: "__action-button",
    MAIN_AND_LIST: "__action-button--main-and-list",
    TWO_LIST: "__action-button--two-list",
};
class ActionButton {
    constructor(grid, classname) {
        this.grid = grid;
        this.grid = grid;
        this.actor = new St.Button({
            style_class: `${getTheme()}${ACTION_CLASSES.BUTTON}`,
            reactive: true,
            can_focus: true,
            track_hover: true
        });
        this.icon = new St.BoxLayout({ style_class: classname, reactive: true, can_focus: true, track_hover: true });
        this.actor.add_actor(this.icon);
        log("ActionButton Connect button-press-event");
        this.actor.connect('button-press-event', this._onButtonPress.bind(this));
    }
    _onButtonPress() {
        log("ActionButton _onButtonPress Emitting signal button-press-event");
        this.emit('button-press-event');
    }
    /** Functions replaced by Signals.addSignalMethods. */
    connect(eventName, handler) { return 0; }
    disconnect(id) { }
    emit(name, ...args) { }
}
Signals.addSignalMethods(ActionButton.prototype);
class AutoTileMainAndList extends ActionButton {
    constructor(grid) {
        const theme = `${getTheme()}${ACTION_CLASSES.MAIN_AND_LIST}`;
        super(grid, theme);
        this.classname = theme;
        log("AutoTileMainAndList connect button-press-event");
        this.connect('button-press-event', () => this._onButtonPress());
    }
    _onButtonPress() {
        AutoTileMain();
        log("AutoTileMainAndList _onButtonPress Emitting signal resize-done");
        this.emit('resize-done');
    }
}
Signals.addSignalMethods(AutoTileMainAndList.prototype);
function AutoTileMain() {
    let preset = "AutoTileMain";
    log("AutoTileMain");
    let window = getFocusApp();
    if (!window) {
        log("No focused window - ignoring keyboard shortcut AutoTileMain");
        return;
    }
    reset_window(window);
    let mind = window.get_monitor();
    let work_area = getWorkAreaByMonitorIdx(mind);
    const monitors = activeMonitors();
    let monitor = monitors[mind];
    let workArea = getWorkAreaByMonitor(monitor);
    let notFocusedwindows = getNotFocusedWindowsOfMonitor(monitor);
    if (Object.keys(notFocusedwindows).length === 0) {
        moveResizeWindowWithMargins(focusMetaWindow, workArea.x, workArea.y, workArea.width, workArea.height);
        return;
    }
    log("SETTINGS_MAIN_WINDOW_SIZES:" + mainWindowSizes);
    let ps_variants = mainWindowSizes;
    log("Main window sizes: " + ps_variants);
    // handle preset variants (if there are any)
    let variantCount = ps_variants.length;
    if (variantCount > 1) {
        if ((lastResizeInfo.lastCallTime.getTime() + gridSettings[SETTINGS_MAX_TIMEOUT]) > new Date().getTime() &&
            lastResizeInfo.presetName === preset.toString() &&
            lastResizeInfo.windowTitle == window.get_title()) {
            // within timeout (default: 2s), same preset & same window:
            // increase variant counter, but consider upper boundary
            lastResizeInfo.variantIndex = (lastResizeInfo.variantIndex + 1) % variantCount;
        }
        else {
            // timeout, new preset or new window:
            // update presetState.last_preset and reset variant counter
            lastResizeInfo.variantIndex = 0;
        }
    }
    else {
        lastResizeInfo.variantIndex = 0;
    }
    let mainRatio = ps_variants[lastResizeInfo.variantIndex];
    let mainWidth = workArea.width * mainRatio;
    let minorWidth = workArea.width - mainWidth;
    moveResizeWindowWithMargins(focusMetaWindow, workArea.x, workArea.y, mainWidth, workArea.height);
    let winHeight = workArea.height / notFocusedwindows.length;
    let countWin = 0;
    log("AutoTileMain MonitorHeight: " + monitor.height + ":" + notFocusedwindows.length);
    for (let windowIdx in notFocusedwindows) {
        let metaWindow = notFocusedwindows[windowIdx].meta_window;
        let newOffset = workArea.y + (countWin * winHeight);
        reset_window(metaWindow);
        moveResizeWindowWithMargins(metaWindow, workArea.x + mainWidth, newOffset, minorWidth, winHeight);
        countWin++;
    }
    log("AutoTileMain done");
    lastResizeInfo.presetName = preset;
    lastResizeInfo.windowTitle = window.get_title();
    lastResizeInfo.lastCallTime = new Date();
    log("Resize preset last call: " + lastResizeInfo.lastCallTime);
}
class AutoTileTwoList extends ActionButton {
    constructor(grid) {
        const theme = `${getTheme()}${ACTION_CLASSES.TWO_LIST}`;
        super(grid, theme);
        this.classname = theme;
        log("AutoTileTwoList connect button-press-event");
        this.connect('button-press-event', () => this._onButtonPress());
    }
    _onButtonPress() {
        log("AutotileTwoList");
        autoTileNCols(2);
        log("AutoTileTwoList _onButtonPress Emitting signal resize-done");
        this.emit('resize-done');
        log("Autotile2 done");
    }
}
Signals.addSignalMethods(AutoTileTwoList.prototype);
function autoTileNCols(cols) {
    log("AutoTileNCols " + cols);
    let window = getFocusApp();
    if (!window) {
        log("No focused window - ignoring keyboard shortcut AutoTileNCols");
        return;
    }
    reset_window(window);
    let mind = window.get_monitor();
    let work_area = getWorkAreaByMonitorIdx(mind);
    const monitors = activeMonitors();
    let monitor = monitors[mind];
    let workArea = getWorkAreaByMonitor(monitor);
    if (!workArea) {
        return;
    }
    let windows = getNotFocusedWindowsOfMonitor(monitor);
    let nbWindowOnEachSide = Math.ceil((windows.length + 1) / cols);
    let winHeight = workArea.height / nbWindowOnEachSide;
    let countWin = 0;
    moveResizeWindowWithMargins(window, workArea.x + countWin % cols * workArea.width / cols, workArea.y + (Math.floor(countWin / cols) * winHeight), workArea.width / cols, winHeight);
    countWin++;
    // todo make function
    for (let windowIdx in windows) {
        let metaWindow = windows[windowIdx].meta_window;
        reset_window(metaWindow);
        moveResizeWindowWithMargins(metaWindow, workArea.x + countWin % cols * workArea.width / cols, workArea.y + (Math.floor(countWin / cols) * winHeight), workArea.width / cols, winHeight);
        countWin++;
    }
    log("AutoTileNCols done");
}
function snapToNeighborsBind() {
    log("SnapToNeighbors keybind invoked");
    let window = getFocusApp();
    if (!window) {
        log("No focused window - ignoring keyboard shortcut SnapToNeighbors");
        return;
    }
    snapToNeighbors(window);
}
/**
 * GridSettingsButton is a GUI button that displays text like "8x8" and changes
 * the main grid size used for GUI elements and some presets.
 */
class GridSettingsButton {
    constructor(gridSize) {
        this.text = gridSize.toString();
        this.cols = gridSize.width;
        this.rows = gridSize.height;
        this.actor = new St.Button({
            style_class: `${theme}__preset-button`,
            reactive: true,
            can_focus: true,
            track_hover: true,
            label: this.text,
        });
        this.actor.connect('button-press-event', () => this._onButtonPress());
    }
    _onButtonPress() {
        nbCols = this.cols;
        nbRows = this.rows;
        globalApp.refreshGrids();
    }
}
const GRID_WIDGET_BORDER_SIZE = 1;
class Grid {
    constructor(gridWidget, monitor_idx, monitor, title, cols, rows) {
        this.gridWidget = gridWidget;
        this.tableWidth = 320;
        this.borderwidth = 2;
        this.elements = [];
        let workArea = getWorkArea(monitor, monitor_idx);
        this.tableHeight = (this.tableWidth / workArea.width) * workArea.height;
        this.actor = new St.BoxLayout({
            vertical: true,
            style_class: theme,
            reactive: true,
            can_focus: true,
            track_hover: true
        });
        log(`created actor for monitor ${monitor_idx}: ${this.actor} with cols=${cols}, rows=${rows}`);
        log("Grid connect enter-event leave-envent BEGIN");
        // TODO: disconnect these events on teardown.
        this.actor.connect('enter-event', () => this._onMouseEnter());
        this.actor.connect('leave-event', () => this._onMouseLeave());
        log("Grid connect enter-event leave-envent FINISH");
        this.animation_time = gridSettings[SETTINGS_ANIMATION] ? 0.3 : 0;
        this.topbar = new TopBar(title);
        this.bottombarContainer = new St.Bin({
            style_class: `${theme}__preset-container`,
            reactive: true,
            can_focus: true,
            track_hover: true
        });
        this.bottomBarTableLayout = new Clutter.GridLayout();
        this.bottombar = new St.Widget({
            style_class: `${theme}__preset`,
            can_focus: true,
            track_hover: true,
            reactive: true,
            width: this.tableWidth - 20,
            height: 36,
            layout_manager: this.bottomBarTableLayout
        });
        this.bottomBarTableLayout.set_row_homogeneous(true);
        this.bottomBarTableLayout.set_column_homogeneous(true);
        this.bottombarContainer.add_actor(this.bottombar);
        this.veryBottomBarContainer = new St.Bin({
            style_class: `${theme}__action-container`,
            reactive: true,
            can_focus: true,
            track_hover: true
        });
        this.veryBottomBarTableLayout = new Clutter.GridLayout();
        this.veryBottomBar = new St.Widget({
            style_class: `${theme}__action`,
            can_focus: true,
            track_hover: true,
            reactive: true,
            width: this.tableWidth - 20,
            height: 36,
            layout_manager: this.veryBottomBarTableLayout,
        });
        this.bottomBarTableLayout.set_row_homogeneous(true);
        this.veryBottomBarTableLayout.set_column_homogeneous(true);
        this.veryBottomBarContainer.add_actor(this.veryBottomBar);
        let rowNum = 0;
        let colNum = 0;
        let maxPerRow = 4;
        for (let gridSize of gridSettings[SETTINGS_GRID_SIZES]) {
            if (colNum >= maxPerRow) {
                colNum = 0;
                rowNum += 1;
            }
            const button = new GridSettingsButton(gridSize);
            this.bottomBarTableLayout.attach(button.actor, colNum, rowNum, 1, 1);
            button.actor.connect('notify::hover', () => this._onSettingsButton());
            colNum++;
        }
        this.bottombar.height *= (rowNum + 1);
        this.tableContainer = new St.Bin({
            style_class: `${theme}__tile-container`,
            reactive: true,
            can_focus: true,
            track_hover: true
        });
        this.tableLayoutManager = new Clutter.GridLayout();
        this.table = new St.Widget({
            style_class: `${theme}__tile-table`,
            can_focus: true,
            track_hover: true,
            reactive: true,
            height: this.tableHeight,
            width: this.tableWidth - GRID_WIDGET_BORDER_SIZE * 2,
            layout_manager: this.tableLayoutManager,
        });
        this.tableLayoutManager.set_row_homogeneous(true);
        this.tableLayoutManager.set_column_homogeneous(true);
        this.tableContainer.add_actor(this.table);
        this.actor.add_child(this.topbar.actor);
        this.actor.add_child(this.tableContainer);
        this.actor.add_child(this.bottombarContainer);
        this.actor.add_child(this.veryBottomBarContainer);
        this.monitor = monitor;
        this.monitor_idx = monitor_idx;
        this.rows = rows;
        this.title = title;
        this.cols = cols;
        this.isEntered = false;
        const toggleSettingListener = new ToggleSettingsButtonListener();
        let toggle = new ToggleSettingsButton("animation", SETTINGS_ANIMATION);
        this.veryBottomBarTableLayout.attach(toggle.actor, 0, 0, 1, 1);
        toggleSettingListener.addActor(toggle);
        toggle = new ToggleSettingsButton("auto-close", SETTINGS_AUTO_CLOSE);
        this.veryBottomBarTableLayout.attach(toggle.actor, 1, 0, 1, 1);
        toggleSettingListener.addActor(toggle);
        let action = new AutoTileMainAndList(this);
        this.veryBottomBarTableLayout.attach(action.actor, 2, 0, 1, 1);
        action.connect('resize-done', this._onResize.bind(this));
        action = new AutoTileTwoList(this);
        this.veryBottomBarTableLayout.attach(action.actor, 3, 0, 1, 1);
        action.connect('resize-done', this._onResize.bind(this));
        this.x = 0;
        this.y = 0;
        this.interceptHide = false;
        this._displayElements();
        this.normalScaleY = this.actor.scale_y;
        this.normalScaleX = this.actor.scale_x;
    }
    state() {
        return `grid with actor ${this.actor}`;
    }
    _displayElements() {
        if (this.monitor === null) {
            return;
        }
        log("Grid _displayElements " + this.cols + ":" + this.rows);
        this.elements = new Array();
        let width = (this.tableWidth / this.cols); // - 2*this.borderwidth;
        let height = (this.tableHeight / this.rows); // - 2*this.borderwidth;
        const delegate = new GridElementDelegate(this, this.gridWidget);
        this.elementsDelegate = delegate;
        this.elementsDelegate.connect('resize-done', (actor, event) => this._onResize());
        for (let r = 0; r < this.rows; r++) {
            for (let c = 0; c < this.cols; c++) {
                if (c == 0) {
                    this.elements[r] = new Array();
                }
                const element = new GridElement(this.elementsDelegate, this.monitor, width, height, c, r);
                this.elements[r][c] = element;
                this.tableLayoutManager.attach(element.actor, c, r, 1, 1);
                element.show();
            }
        }
    }
    getElement(row, col) {
        return this.elements[row][col] || null;
    }
    forceGridElementDelegate(x, y, w, h) {
        var _a;
        (_a = this.elementsDelegate) === null || _a === void 0 ? void 0 : _a.forceArea(this.elements[y][x], this.elements[h][w]);
    }
    refresh() {
        log("Grid.refresh from " + this.cols + ":" + this.rows + " to " + nbCols + ":" + nbRows);
        this.elementsDelegate._resetGrid();
        for (let r = 0; r < this.rows; r++) {
            for (let c = 0; c < this.cols; c++) {
                this.elements[r][c]._disconnect();
            }
        }
        this.table.destroy_all_children();
        this.cols = nbCols;
        this.rows = nbRows;
        this._displayElements();
    }
    set_position(x, y) {
        this.x = x;
        this.y = y;
        this.actor.set_position(x, y);
    }
    show() {
        this.interceptHide = true;
        this.elementsDelegate.reset();
        let time = (gridSettings[SETTINGS_ANIMATION]) ? 0.3 : 0;
        Main$1.uiGroup.set_child_above_sibling(this.actor, null);
        Main$1.layoutManager.removeChrome(this.actor);
        Main$1.layoutManager.addChrome(this.actor);
        //this.actor.y = 0 ;
        if (time > 0) {
            this.actor.scale_y = 0;
            this.actor.scale_x = 0;
            this.actor.ease({
                time: this.animation_time,
                opacity: 255,
                visible: true,
                transition: Clutter.AnimationMode.EASE_OUT_QUAD,
                scale_x: this.normalScaleX,
                scale_y: this.normalScaleY,
                onComplete: this._onShowComplete
            });
        }
        else {
            this.actor.scale_x = this.normalScaleX;
            this.actor.scale_y = this.normalScaleY;
            this.actor.opacity = 255;
            this.actor.visible = true;
        }
        this.interceptHide = false;
    }
    hide(immediate) {
        var _a;
        log(`Grid hide immediate=${immediate}`);
        (_a = this.elementsDelegate) === null || _a === void 0 ? void 0 : _a.reset();
        let time = (gridSettings[SETTINGS_ANIMATION]) ? 0.3 : 0;
        if (!immediate && time > 0) {
            this.actor.ease({
                time: this.animation_time,
                opacity: 0,
                visible: false,
                scale_x: 0,
                scale_y: 0,
                transition: Clutter.AnimationMode.EASE_OUT_QUAD,
                onComplete: this._onHideComplete
            });
        }
        else {
            this.actor.opacity = 0;
            this.actor.visible = false;
            //this.actor.y = 0;
            this.actor.scale_x = 0;
            this.actor.scale_y = 0;
        }
    }
    setInitialSelection(focusMetaWindow) {
        var _a, _b, _c, _d;
        // The window being focused is part of the monitor that this Grid is
        // responsible for manipulating.
        if (focusMetaWindow.get_monitor() !== this.monitor_idx) {
            return;
        }
        if (!this.monitor) {
            return;
        }
        const monitor = this.monitor;
        const workArea = getWorkArea(this.monitor, this.monitor_idx);
        let wx = focusMetaWindow.get_frame_rect().x;
        let wy = focusMetaWindow.get_frame_rect().y;
        let wwidth = focusMetaWindow.get_frame_rect().width;
        let wheight = focusMetaWindow.get_frame_rect().height;
        const grid = this;
        const delegate = grid.elementsDelegate;
        log("Set initial selection");
        log("Focus window position x " + wx + " y " + wy + " width " + wwidth + " height " + wheight);
        log("Focus monitor position x " + monitor.x + " y " + monitor.y + " width " + monitor.width + " height " + monitor.height);
        log("Workarea position x " + workArea.x + " y " + workArea.y + " width " + workArea.width + " height " + workArea.height);
        let wax = Math.max(wx - workArea.x, 0);
        let way = Math.max(wy - workArea.y, 0);
        let grid_element_width = workArea.width / nbCols;
        let grid_element_height = workArea.height / nbRows;
        log("width " + grid_element_width + " height " + grid_element_height);
        let lux = Math.min(Math.max(Math.round(wax / grid_element_width), 0), nbCols - 1);
        log("wx " + (wx - workArea.x) + " el_width " + grid_element_width + " max " + (nbCols - 1) + " res " + lux);
        let luy = Math.min(Math.max(Math.round(way / grid_element_height), 0), grid.rows - 1);
        log("wy " + (wy - workArea.y) + " el_height " + grid_element_height + " max " + (nbRows - 1) + " res " + luy);
        let rdx = Math.min(Math.max(Math.round((wax + wwidth) / grid_element_width) - 1, lux), grid.cols - 1);
        log("wx + wwidth " + (wx + wwidth - workArea.x - 1) + " el_width " + grid_element_width + " max " + (nbCols - 1) + " res " + rdx);
        let rdy = Math.min(Math.max(Math.round((way + wheight) / grid_element_height) - 1, luy), grid.rows - 1);
        log("wy + wheight " + (wy + wheight - workArea.y - 1) + " el_height " + grid_element_height + " max " + (nbRows - 1) + " res " + rdy);
        log("Initial tile selection is " + lux + ":" + luy + " - " + rdx + ":" + rdy);
        grid.forceGridElementDelegate(lux, luy, rdx, rdy);
        grid.elements[luy][lux]._onButtonPress();
        grid.elements[rdy][rdx]._onHoverChanged();
        const cX = (_a = delegate === null || delegate === void 0 ? void 0 : delegate.currentElement) === null || _a === void 0 ? void 0 : _a.coordx;
        const cY = (_b = delegate === null || delegate === void 0 ? void 0 : delegate.currentElement) === null || _b === void 0 ? void 0 : _b.coordy;
        const fX = (_c = delegate === null || delegate === void 0 ? void 0 : delegate.first) === null || _c === void 0 ? void 0 : _c.coordx;
        const fY = (_d = delegate === null || delegate === void 0 ? void 0 : delegate.first) === null || _d === void 0 ? void 0 : _d.coordy;
        log("After initial selection first fX " + fX + " fY " + fY + " current cX " + cX + " cY " + cY);
    }
    _onHideComplete() {
    }
    _onShowComplete() {
    }
    _onResize() {
        log("resize-done");
        globalApp.updateRegions();
        if (gridSettings[SETTINGS_AUTO_CLOSE]) {
            log("Emitting hide-tiling");
            this.emit('hide-tiling');
        }
    }
    _onMouseEnter() {
        var _a;
        log("Grid: onMouseEnter");
        if (!this.isEntered) {
            (_a = this.elementsDelegate) === null || _a === void 0 ? void 0 : _a.reset();
            this.isEntered = true;
        }
    }
    _onMouseLeave() {
        var _a;
        log("Grid: onMouseLeave");
        let [x, y, mask] = global.get_pointer();
        if (this.elementsDelegate && (x <= this.actor.x || x >= (this.actor.x + this.actor.width)) || (y <= this.actor.y || y >= (this.actor.y + this.actor.height))) {
            this.isEntered = false;
            (_a = this.elementsDelegate) === null || _a === void 0 ? void 0 : _a.reset();
            globalApp.refreshGrids();
        }
    }
    _onSettingsButton() {
        var _a;
        log("_onSettingsButton");
        (_a = this.elementsDelegate) === null || _a === void 0 ? void 0 : _a.reset();
    }
    _destroy() {
        var _a;
        log("Grid _destroy");
        for (let r in this.elements) {
            for (let c in this.elements[r]) {
                this.elements[r][c]._destroy();
            }
        }
        (_a = this.elementsDelegate) === null || _a === void 0 ? void 0 : _a._destroy();
        this.topbar.destroy();
        this.monitor = null;
        this.rows = 0;
        this.title = "";
        this.cols = 0;
        log("Disconnect hide-tiling");
        this.disconnect(this.connectHideTiling);
    }
    // Methods replaced by Signals.addSignalMethods.
    connect(name, callback) { return -1; }
    disconnect(id) { }
    emit(name, ...args) { }
}
Signals.addSignalMethods(Grid.prototype);
class GridElementDelegate {
    constructor(grid, gridWidget) {
        this.grid = grid;
        this.gridWidget = gridWidget;
        // The first element clicked in the rectangular selection.
        this.first = null;
        // 
        this.currentElement = null;
        // Elements that are in a highlighted state in the UI. The highlighting
        // happens when the grid rectangle selcted includes a particular grid
        // element.
        this.activatedElements = [];
        this.activated = false;
    }
    allSelected() {
        return (this.activatedElements.length === (nbCols * nbRows));
    }
    onButtonPress(gridElement) {
        log("GridElementDelegate _onButtonPress " + gridElement.coordx + ":" + gridElement.coordy);
        //this._logActiveActors("GridElementDelegate _onButtonPress active actors");
        if (!this.currentElement) {
            this.currentElement = gridElement;
        }
        if (!this.activated) {
            log("GridElementDelegate first activation");
            this.activated = true;
            gridElement.active = true;
            this.activatedElements = [gridElement];
            this.first = gridElement;
            return;
        }
        log("GridElementDelegate resize");
        //Check this.activatedActors if equals to nbCols * nbRows
        //before doing anything with the window it must be unmaximized
        //if so move the window then maximize instead of change size
        //if not move the window and change size
        if (!focusMetaWindow) {
            return;
        }
        reset_window(focusMetaWindow);
        if (this.first) {
            const computedSize = this._computeAreaPositionSize(this.first, gridElement);
            if (!computedSize) {
                return;
            }
            const [areaX, areaY, areaWidth, areaHeight] = computedSize;
            if (this.allSelected() && gridSettings[SETTINGS_WINDOW_MARGIN_FULLSCREEN_ENABLED] === false) {
                move_maximize_window(focusMetaWindow, areaX, areaY);
            }
            else {
                moveResizeWindowWithMargins(focusMetaWindow, areaX, areaY, areaWidth, areaHeight);
            }
        }
        this._resizeDone();
    }
    _resizeDone() {
        log("resizeDone, emitting signal resize-done");
        this.emit('resize-done');
    }
    reset() {
        this._resetGrid();
        this.activated = false;
        this.first = null;
        this.currentElement = null;
    }
    _resetGrid() {
        this._hideArea();
        if (this.currentElement) {
            this.currentElement._deactivate();
        }
        for (let act of this.activatedElements) {
            act._deactivate();
        }
        this.activatedElements = new Array();
    }
    _getVarFromGridElement(fromGridElement, toGridElement) {
        let minX = Math.min(fromGridElement.coordx, toGridElement.coordx);
        let maxX = Math.max(fromGridElement.coordx, toGridElement.coordx);
        let minY = Math.min(fromGridElement.coordy, toGridElement.coordy);
        let maxY = Math.max(fromGridElement.coordy, toGridElement.coordy);
        return [minX, maxX, minY, maxY];
    }
    refreshGrid(fromGridElement, toGridElement) {
        this._resetGrid();
        let [minX, maxX, minY, maxY] = this._getVarFromGridElement(fromGridElement, toGridElement);
        if (!fromGridElement.monitor) {
            return;
        }
        for (let r = minY; r <= maxY; r++) {
            for (let c = minX; c <= maxX; c++) {
                const highlightedElement = this.grid.getElement(r, c);
                if (!highlightedElement) {
                    log(`bug: GridElementDelegate refreshGrid is out of sync with Grid: trying to highlight element (row=${r}, col=${c}) and getElement() returned null.`);
                    continue;
                }
                highlightedElement._activate();
                this.activatedElements.push(highlightedElement);
            }
        }
        this._displayArea(fromGridElement, toGridElement);
    }
    _computeAreaPositionSize(fromGridElement, toGridElement) {
        let [minX, maxX, minY, maxY] = this._getVarFromGridElement(fromGridElement, toGridElement);
        let monitor = fromGridElement.monitor;
        const workArea = getWorkAreaByMonitor(monitor);
        if (!workArea) {
            return null;
        }
        let areaWidth = Math.round((workArea.width / nbCols) * ((maxX - minX) + 1));
        let areaHeight = Math.round((workArea.height / nbRows) * ((maxY - minY) + 1));
        let areaX = workArea.x + Math.round((minX * (workArea.width / nbCols)));
        let areaY = workArea.y + Math.round((minY * (workArea.height / nbRows)));
        return [areaX, areaY, areaWidth, areaHeight];
    }
    forceArea(fromGridElement, toGridElement) {
        const computedSize = this._computeAreaPositionSize(fromGridElement, toGridElement);
        if (!computedSize) {
            return;
        }
        const [areaX, areaY, areaWidth, areaHeight] = computedSize;
        this.gridWidget.width = areaWidth;
        this.gridWidget.height = areaHeight;
        this.gridWidget.x = areaX;
        this.gridWidget.y = areaY;
    }
    _displayArea(fromGridElement, toGridElement) {
        const computedSize = this._computeAreaPositionSize(fromGridElement, toGridElement);
        if (!computedSize) {
            return;
        }
        const [areaX, areaY, areaWidth, areaHeight] = computedSize;
        this.gridWidget.add_style_pseudo_class('activate');
        if (gridSettings[SETTINGS_ANIMATION]) {
            this.gridWidget.ease({
                time: 0.2,
                x: areaX,
                y: areaY,
                width: areaWidth,
                height: areaHeight,
                transition: Clutter.AnimationMode.EASE_OUT_QUAD
            });
        }
        else {
            this.gridWidget.width = areaWidth;
            this.gridWidget.height = areaHeight;
            this.gridWidget.x = areaX;
            this.gridWidget.y = areaY;
        }
    }
    _hideArea() {
        this.gridWidget.remove_style_pseudo_class('activate');
    }
    _onHoverChanged(gridElement) {
        log("GridElementDelegate _onHoverChange " + gridElement.coordx + ":" + gridElement.coordy);
        if (this.activated) {
            log("GridElementDelegate _onHoverChange/not active: " + gridElement.coordx + ":" + gridElement.coordy);
            this.refreshGrid(this.first, gridElement);
            this.currentElement = gridElement;
        }
        else if (!this.currentElement || gridElement.id != this.currentElement.id) {
            log("GridElementDelegate _onHoverChange/active: " + gridElement.coordx + ":" + gridElement.coordy);
            if (this.currentElement) {
                this.currentElement._deactivate();
            }
            this.currentElement = gridElement;
            this.currentElement._activate();
            this._displayArea(gridElement, gridElement);
        }
        else {
            log("GridElementDelegate _onHoverChange/else: " + gridElement.coordx + ":" + gridElement.coordy);
        }
    }
    _destroy() {
        this.activated = false;
        this.first = null;
        this.currentElement = null;
        this.activatedElements = [];
    }
    // Methods replaced by Signals.addSignalMethods.
    connect(name, callback) { return -1; }
    disconnect(id) { }
    emit(name, ...args) { }
}
Signals.addSignalMethods(GridElementDelegate.prototype);
class GridElement {
    constructor(delegate, monitor, width, height, coordx, coordy) {
        this.delegate = delegate;
        this.monitor = monitor;
        this.width = width;
        this.height = height;
        this.coordx = coordx;
        this.coordy = coordy;
        this.actor = new St.Button({
            style_class: `${theme}__tile-table-item`,
            reactive: true,
            can_focus: true,
            track_hover: true
        });
        this.actor.visible = false;
        this.actor.opacity = 0;
        this.id = getMonitorKey(monitor) + "-" + coordx + ":" + coordy;
        this.actor.connect('button-press-event', this._onButtonPress.bind(this));
        this.hoverConnect = this.actor.connect('notify::hover', () => this._onHoverChanged());
        this.active = false;
    }
    state() {
        return `id: ${this.id}; visible: ${this.actor.visible}`;
    }
    show() {
        this.actor.opacity = 255;
        this.actor.visible = true;
    }
    hide() {
        this.actor.opacity = 0;
        this.actor.visible = false;
    }
    _onButtonPress() {
        this.delegate.onButtonPress(this);
    }
    _onHoverChanged() {
        this.delegate._onHoverChanged(this);
    }
    _activate() {
        if (!this.active) {
            this.actor.add_style_pseudo_class('activate');
            this.active = true;
        }
    }
    _deactivate() {
        if (this.active) {
            this.actor.remove_style_pseudo_class('activate');
            this.active = false;
        }
    }
    // This logic should probably go into disable().
    // _clean() {
    //     Main.uiGroup.remove_actor(this.gridWidget);
    // }
    _disconnect() {
        this.actor.disconnect(this.hoverConnect);
    }
    _destroy() {
        this.active = false;
    }
}
function getTheme() {
    let themeName = settings.get_string(SETTINGS_THEME) || 'Default';
    themeName = themeName.toLowerCase().replace(/[^A-Za-z0-9]/g, '-');
    const theme = `gtile-${themeName}`;
    log("Theme changed to the" + theme);
    return theme;
}

registerExtension(function () { }, enable, disable);
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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
