/*
 * Decompiled with CFR 0.152.
 */
package jadx.core.clsp;

import jadx.api.plugins.utils.ZipSecurity;
import jadx.core.clsp.ClspClass;
import jadx.core.clsp.ClspMethod;
import jadx.core.dex.info.AccessInfo;
import jadx.core.dex.info.ClassInfo;
import jadx.core.dex.info.MethodInfo;
import jadx.core.dex.instructions.args.ArgType;
import jadx.core.dex.nodes.ClassNode;
import jadx.core.dex.nodes.MethodNode;
import jadx.core.dex.nodes.RootNode;
import jadx.core.utils.Utils;
import jadx.core.utils.exceptions.DecodeException;
import jadx.core.utils.exceptions.JadxRuntimeException;
import jadx.core.utils.files.FileUtils;
import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Stream;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;
import java.util.zip.ZipOutputStream;
import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ClsSet {
    private static final Logger LOG = LoggerFactory.getLogger(ClsSet.class);
    private static final String CLST_EXTENSION = ".jcst";
    private static final String CLST_FILENAME = "core.jcst";
    private static final String CLST_PKG_PATH = ClsSet.class.getPackage().getName().replace('.', '/');
    private static final String JADX_CLS_SET_HEADER = "jadx-cst";
    private static final int VERSION = 3;
    private static final String STRING_CHARSET = "US-ASCII";
    private static final ArgType[] EMPTY_ARGTYPE_ARRAY = new ArgType[0];
    private final RootNode root;
    private ClspClass[] classes;

    public ClsSet(RootNode root) {
        this.root = root;
    }

    public void loadFromClstFile() throws IOException, DecodeException {
        long startTime = System.currentTimeMillis();
        try (InputStream input = this.getClass().getResourceAsStream(CLST_FILENAME);){
            if (input == null) {
                throw new JadxRuntimeException("Can't load classpath file: core.jcst");
            }
            this.load(input);
        }
        if (LOG.isDebugEnabled()) {
            long time = System.currentTimeMillis() - startTime;
            int methodsCount = Stream.of(this.classes).mapToInt(clspClass -> clspClass.getMethodsMap().size()).sum();
            LOG.debug("Clst file loaded in {}ms, classes: {}, methods: {}", new Object[]{time, this.classes.length, methodsCount});
        }
    }

    public void loadFrom(RootNode root) {
        List<ClassNode> list = root.getClasses(true);
        HashMap<String, ClspClass> names = new HashMap<String, ClspClass>(list.size());
        int k = 0;
        for (ClassNode cls : list) {
            ArgType clsType = cls.getClassInfo().getType();
            String clsRawName = clsType.getObject();
            cls.load();
            ClspClass nClass = new ClspClass(clsType, k);
            if (names.put(clsRawName, nClass) != null) {
                throw new JadxRuntimeException("Duplicate class: " + clsRawName);
            }
            ++k;
            nClass.setTypeParameters(cls.getGenericTypeParameters());
            nClass.setMethods(this.getMethodsDetails(cls));
        }
        this.classes = new ClspClass[k];
        k = 0;
        for (ClassNode cls : list) {
            ClspClass nClass = ClsSet.getCls(cls, names);
            if (nClass == null) {
                throw new JadxRuntimeException("Missing class: " + cls);
            }
            nClass.setParents(ClsSet.makeParentsArray(cls));
            this.classes[k] = nClass;
            ++k;
        }
    }

    private List<ClspMethod> getMethodsDetails(ClassNode cls) {
        List<MethodNode> methodsList = cls.getMethods();
        ArrayList<ClspMethod> methods = new ArrayList<ClspMethod>(methodsList.size());
        for (MethodNode mth : methodsList) {
            this.processMethodDetails(mth, methods);
        }
        return methods;
    }

    private void processMethodDetails(MethodNode mth, List<ClspMethod> methods) {
        AccessInfo accessFlags = mth.getAccessFlags();
        if (accessFlags.isPrivate()) {
            return;
        }
        ArgType genericRetType = mth.getReturnType();
        boolean varArgs = accessFlags.isVarArgs();
        List<ArgType> throwList = mth.getThrows();
        List<ArgType> typeParameters = mth.getTypeParameters();
        if (varArgs || Utils.notEmpty(throwList) || Utils.notEmpty(typeParameters) || genericRetType.containsGeneric() || mth.containsGenericArgs() || mth.isArgsOverloaded()) {
            ClspMethod clspMethod = new ClspMethod(mth.getMethodInfo(), mth.getArgTypes(), genericRetType, typeParameters, varArgs, throwList);
            methods.add(clspMethod);
        }
    }

    public static ArgType[] makeParentsArray(ClassNode cls) {
        ArgType superClass = cls.getSuperClass();
        if (superClass == null) {
            return EMPTY_ARGTYPE_ARRAY;
        }
        ArgType[] parents = new ArgType[1 + cls.getInterfaces().size()];
        parents[0] = superClass;
        int k = 1;
        Iterator<ArgType> iterator = cls.getInterfaces().iterator();
        while (iterator.hasNext()) {
            ArgType iface;
            parents[k] = iface = iterator.next();
            ++k;
        }
        return parents;
    }

    private static ClspClass getCls(ClassNode cls, Map<String, ClspClass> names) {
        return ClsSet.getCls(cls.getRawName(), names);
    }

    private static ClspClass getCls(ArgType clsType, Map<String, ClspClass> names) {
        return ClsSet.getCls(clsType.getObject(), names);
    }

    private static ClspClass getCls(String fullName, Map<String, ClspClass> names) {
        ClspClass cls = names.get(fullName);
        if (cls == null) {
            LOG.debug("Class not found: {}", (Object)fullName);
        }
        return cls;
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    public void save(Path path) throws IOException {
        FileUtils.makeDirsForFile(path);
        String outputName = path.getFileName().toString();
        if (outputName.endsWith(CLST_EXTENSION)) {
            try (BufferedOutputStream outputStream = new BufferedOutputStream(Files.newOutputStream(path, new OpenOption[0]));){
                this.save(outputStream);
                return;
            }
        }
        if (!outputName.endsWith(".jar")) throw new JadxRuntimeException("Unknown file format: " + outputName);
        Path temp = FileUtils.createTempFile(".zip");
        Files.copy(path, temp, StandardCopyOption.REPLACE_EXISTING);
        try (ZipOutputStream out = new ZipOutputStream(Files.newOutputStream(path, new OpenOption[0]));
             ZipInputStream in = new ZipInputStream(Files.newInputStream(temp, new OpenOption[0]));){
            String clst = CLST_PKG_PATH + '/' + CLST_FILENAME;
            boolean clstReplaced = false;
            ZipEntry entry = in.getNextEntry();
            while (entry != null) {
                String entryName = entry.getName();
                ZipEntry copyEntry = new ZipEntry(entryName);
                copyEntry.setLastModifiedTime(entry.getLastModifiedTime());
                out.putNextEntry(copyEntry);
                if (entryName.equals(clst)) {
                    this.save(out);
                    clstReplaced = true;
                } else {
                    FileUtils.copyStream(in, out);
                }
                entry = in.getNextEntry();
            }
            if (clstReplaced) return;
            out.putNextEntry(new ZipEntry(clst));
            this.save(out);
            return;
        }
    }

    private void save(OutputStream output) throws IOException {
        DataOutputStream out = new DataOutputStream(output);
        out.writeBytes(JADX_CLS_SET_HEADER);
        out.writeByte(3);
        HashMap<String, ClspClass> names = new HashMap<String, ClspClass>(this.classes.length);
        out.writeInt(this.classes.length);
        for (ClspClass cls : this.classes) {
            String clsName = cls.getName();
            ClsSet.writeString(out, clsName);
            names.put(clsName, cls);
        }
        for (ClspClass cls : this.classes) {
            ClsSet.writeArgTypesArray(out, cls.getParents(), names);
            ClsSet.writeArgTypesList(out, cls.getTypeParameters(), names);
            List<ClspMethod> methods = cls.getSortedMethodsList();
            out.writeShort(methods.size());
            for (ClspMethod method : methods) {
                ClsSet.writeMethod(out, method, names);
            }
        }
        int methodsCount = Stream.of(this.classes).mapToInt(c -> c.getMethodsMap().size()).sum();
        LOG.info("Classes: {}, methods: {}, file size: {}B", new Object[]{this.classes.length, methodsCount, out.size()});
    }

    private static void writeMethod(DataOutputStream out, ClspMethod method, Map<String, ClspClass> names) throws IOException {
        MethodInfo methodInfo = method.getMethodInfo();
        ClsSet.writeString(out, methodInfo.getName());
        ClsSet.writeArgTypesList(out, methodInfo.getArgumentsTypes(), names);
        ClsSet.writeArgType(out, methodInfo.getReturnType(), names);
        ClsSet.writeArgTypesList(out, method.containsGenericArgs() ? method.getArgTypes() : Collections.emptyList(), names);
        ClsSet.writeArgType(out, method.getReturnType(), names);
        ClsSet.writeArgTypesList(out, method.getTypeParameters(), names);
        out.writeBoolean(method.isVarArg());
        ClsSet.writeArgTypesList(out, method.getThrows(), names);
    }

    private static void writeArgTypesList(DataOutputStream out, List<ArgType> list, Map<String, ClspClass> names) throws IOException {
        int size = list.size();
        ClsSet.writeUnsignedByte(out, size);
        if (size != 0) {
            for (ArgType type : list) {
                ClsSet.writeArgType(out, type, names);
            }
        }
    }

    private static void writeArgTypesArray(DataOutputStream out, @Nullable ArgType[] arr, Map<String, ClspClass> names) throws IOException {
        if (arr == null) {
            out.writeByte(-1);
            return;
        }
        int size = arr.length;
        out.writeByte(size);
        if (size != 0) {
            for (ArgType type : arr) {
                ClsSet.writeArgType(out, type, names);
            }
        }
    }

    private static void writeArgType(DataOutputStream out, ArgType argType, Map<String, ClspClass> names) throws IOException {
        if (argType == null) {
            out.writeByte(-1);
            return;
        }
        if (argType.isPrimitive()) {
            out.writeByte(TypeEnum.PRIMITIVE.ordinal());
            out.writeByte(argType.getPrimitiveType().getShortName().charAt(0));
        } else if (argType.getOuterType() != null) {
            out.writeByte(TypeEnum.OUTER_GENERIC.ordinal());
            ClsSet.writeArgType(out, argType.getOuterType(), names);
            ClsSet.writeArgType(out, argType.getInnerType(), names);
        } else if (argType.getWildcardType() != null) {
            out.writeByte(TypeEnum.WILDCARD.ordinal());
            ArgType.WildcardBound bound = argType.getWildcardBound();
            out.writeByte(bound.getNum());
            if (bound != ArgType.WildcardBound.UNBOUND) {
                ClsSet.writeArgType(out, argType.getWildcardType(), names);
            }
        } else if (argType.isGeneric()) {
            out.writeByte(TypeEnum.GENERIC.ordinal());
            out.writeInt(ClsSet.getCls(argType, names).getId());
            ClsSet.writeArgTypesList(out, argType.getGenericTypes(), names);
        } else if (argType.isGenericType()) {
            out.writeByte(TypeEnum.GENERIC_TYPE_VARIABLE.ordinal());
            ClsSet.writeString(out, argType.getObject());
            ClsSet.writeArgTypesList(out, argType.getExtendTypes(), names);
        } else if (argType.isObject()) {
            out.writeByte(TypeEnum.OBJECT.ordinal());
            out.writeInt(ClsSet.getCls(argType, names).getId());
        } else if (argType.isArray()) {
            out.writeByte(TypeEnum.ARRAY.ordinal());
            ClsSet.writeArgType(out, argType.getArrayElement(), names);
        } else {
            throw new JadxRuntimeException("Cannot save type: " + argType);
        }
    }

    private void load(File input) throws IOException, DecodeException {
        String name = input.getName();
        if (name.endsWith(CLST_EXTENSION)) {
            try (FileInputStream inputStream = new FileInputStream(input);){
                this.load(inputStream);
            }
        } else if (name.endsWith(".jar")) {
            ZipSecurity.readZipEntries((File)input, (entry, in) -> {
                if (entry.getName().endsWith(CLST_EXTENSION)) {
                    try {
                        this.load((InputStream)in);
                    }
                    catch (Exception e) {
                        throw new JadxRuntimeException("Failed to load jadx class set");
                    }
                }
            });
        } else {
            throw new JadxRuntimeException("Unknown file format: " + name);
        }
    }

    private void load(InputStream input) throws IOException, DecodeException {
        try (DataInputStream in = new DataInputStream(new BufferedInputStream(input));){
            int i;
            byte[] header = new byte[JADX_CLS_SET_HEADER.length()];
            int readHeaderLength = in.read(header);
            byte version = in.readByte();
            if (readHeaderLength != JADX_CLS_SET_HEADER.length() || !JADX_CLS_SET_HEADER.equals(new String(header, STRING_CHARSET)) || version != 3) {
                throw new DecodeException("Wrong jadx class set header");
            }
            int clsCount = in.readInt();
            this.classes = new ClspClass[clsCount];
            for (i = 0; i < clsCount; ++i) {
                String name = ClsSet.readString(in);
                this.classes[i] = new ClspClass(ArgType.object(name), i);
            }
            for (i = 0; i < clsCount; ++i) {
                ClspClass nClass = this.classes[i];
                ClassInfo clsInfo = ClassInfo.fromType(this.root, nClass.getClsType());
                nClass.setParents(this.readArgTypesArray(in));
                nClass.setTypeParameters(this.readArgTypesList(in));
                nClass.setMethods(this.readClsMethods(in, clsInfo));
            }
        }
    }

    private List<ClspMethod> readClsMethods(DataInputStream in, ClassInfo clsInfo) throws IOException {
        int mCount = in.readShort();
        ArrayList<ClspMethod> methods = new ArrayList<ClspMethod>(mCount);
        for (int j = 0; j < mCount; ++j) {
            methods.add(this.readMethod(in, clsInfo));
        }
        return methods;
    }

    private ClspMethod readMethod(DataInputStream in, ClassInfo clsInfo) throws IOException {
        ArgType genericRetType;
        String name = ClsSet.readString(in);
        List<ArgType> argTypes = this.readArgTypesList(in);
        ArgType retType = this.readArgType(in);
        List<ArgType> genericArgTypes = this.readArgTypesList(in);
        if (genericArgTypes.isEmpty() || Objects.equals(genericArgTypes, argTypes)) {
            genericArgTypes = argTypes;
        }
        if (Objects.equals(genericRetType = this.readArgType(in), retType)) {
            genericRetType = retType;
        }
        List<ArgType> typeParameters = this.readArgTypesList(in);
        boolean varArgs = in.readBoolean();
        List<ArgType> throwList = this.readArgTypesList(in);
        MethodInfo methodInfo = MethodInfo.fromDetails(this.root, clsInfo, name, argTypes, retType);
        return new ClspMethod(methodInfo, genericArgTypes, genericRetType, typeParameters, varArgs, throwList);
    }

    private List<ArgType> readArgTypesList(DataInputStream in) throws IOException {
        int count = in.readByte();
        if (count == 0) {
            return Collections.emptyList();
        }
        ArrayList<ArgType> list = new ArrayList<ArgType>(count);
        for (int i = 0; i < count; ++i) {
            list.add(this.readArgType(in));
        }
        return list;
    }

    @Nullable
    private ArgType[] readArgTypesArray(DataInputStream in) throws IOException {
        int count = in.readByte();
        if (count == -1) {
            return null;
        }
        if (count == 0) {
            return EMPTY_ARGTYPE_ARRAY;
        }
        ArgType[] arr = new ArgType[count];
        for (int i = 0; i < count; ++i) {
            arr[i] = this.readArgType(in);
        }
        return arr;
    }

    private ArgType readArgType(DataInputStream in) throws IOException {
        byte ordinal = in.readByte();
        if (ordinal == -1) {
            return null;
        }
        if (ordinal >= TypeEnum.values().length) {
            throw new JadxRuntimeException("Incorrect ordinal for type enum: " + ordinal);
        }
        switch (TypeEnum.values()[ordinal]) {
            case WILDCARD: {
                ArgType.WildcardBound bound = ArgType.WildcardBound.getByNum(in.readByte());
                if (bound == ArgType.WildcardBound.UNBOUND) {
                    return ArgType.WILDCARD;
                }
                ArgType objType = this.readArgType(in);
                return ArgType.wildcard(objType, bound);
            }
            case OUTER_GENERIC: {
                ArgType outerType = this.readArgType(in);
                ArgType innerType = this.readArgType(in);
                return ArgType.outerGeneric(outerType, innerType);
            }
            case GENERIC: {
                ArgType clsType = this.classes[in.readInt()].getClsType();
                return ArgType.generic(clsType, this.readArgTypesList(in));
            }
            case GENERIC_TYPE_VARIABLE: {
                String typeVar = ClsSet.readString(in);
                List<ArgType> extendTypes = this.readArgTypesList(in);
                return ArgType.genericType(typeVar, extendTypes);
            }
            case OBJECT: {
                return this.classes[in.readInt()].getClsType();
            }
            case ARRAY: {
                return ArgType.array(this.readArgType(in));
            }
            case PRIMITIVE: {
                char shortName = (char)in.readByte();
                return ArgType.parse(shortName);
            }
        }
        throw new JadxRuntimeException("Unsupported Arg Type: " + ordinal);
    }

    private static void writeString(DataOutputStream out, String name) throws IOException {
        byte[] bytes = name.getBytes(STRING_CHARSET);
        int len = bytes.length;
        if (len >= 255) {
            throw new JadxRuntimeException("String is too long: " + name);
        }
        ClsSet.writeUnsignedByte(out, bytes.length);
        out.write(bytes);
    }

    private static String readString(DataInputStream in) throws IOException {
        int len = ClsSet.readUnsignedByte(in);
        return ClsSet.readString(in, len);
    }

    private static String readString(DataInputStream in, int len) throws IOException {
        int res;
        byte[] bytes = new byte[len];
        for (int count = in.read(bytes); count != len; count += res) {
            res = in.read(bytes, count, len - count);
            if (res != -1) continue;
            throw new IOException("String read error");
        }
        return new String(bytes, STRING_CHARSET);
    }

    private static void writeUnsignedByte(DataOutputStream out, int value) throws IOException {
        if (value < 0 || value >= 255) {
            throw new JadxRuntimeException("Unsigned byte value is too big: " + value);
        }
        out.writeByte(value);
    }

    private static int readUnsignedByte(DataInputStream in) throws IOException {
        return in.readByte() & 0xFF;
    }

    public int getClassesCount() {
        return this.classes.length;
    }

    public void addToMap(Map<String, ClspClass> nameMap) {
        for (ClspClass cls : this.classes) {
            nameMap.put(cls.getName(), cls);
        }
    }

    private static enum TypeEnum {
        WILDCARD,
        GENERIC,
        GENERIC_TYPE_VARIABLE,
        OUTER_GENERIC,
        OBJECT,
        ARRAY,
        PRIMITIVE;

    }
}

