// D import file generated from 'src/core/sync/semaphore.d'
module core.sync.semaphore;
public import core.sync.exception;
public import core.time;
version (OSX)
{
	version = Darwin;
}
else
{
	version (iOS)
	{
		version = Darwin;
	}
	else
	{
		version (TVOS)
		{
			version = Darwin;
		}
		else
		{
			version (WatchOS)
			{
				version = Darwin;
			}
		}
	}
}
version (Windows)
{
	import core.sys.windows.basetsd;
	import core.sys.windows.winbase;
	import core.sys.windows.windef;
	import core.sys.windows.winerror;
}
else
{
	version (Darwin)
	{
		import core.sync.config;
		import core.stdc.errno;
		import core.sys.posix.time;
		import core.sys.darwin.mach.semaphore;
	}
	else
	{
		version (Posix)
		{
			import core.sync.config;
			import core.stdc.errno;
			import core.sys.posix.pthread;
			import core.sys.posix.semaphore;
		}
		else
		{
			static assert(false, "Platform not supported");
		}
	}
}
class Semaphore
{
	this(uint count = 0)
	{
		version (Windows)
		{
			m_hndl = CreateSemaphoreA(null, count, (int).max, null);
			if (m_hndl == m_hndl.init)
				throw new SyncError("Unable to create semaphore");
		}
		else
		{
			version (Darwin)
			{
				auto rc = semaphore_create(mach_task_self(), &m_hndl, SYNC_POLICY_FIFO, count);
				if (rc)
					throw new SyncError("Unable to create semaphore");
			}
			else
			{
				version (Posix)
				{
					int rc = sem_init(&m_hndl, 0, count);
					if (rc)
						throw new SyncError("Unable to create semaphore");
				}

			}
		}
	}
	~this()
	{
		version (Windows)
		{
			BOOL rc = CloseHandle(m_hndl);
			assert(rc, "Unable to destroy semaphore");
		}
		else
		{
			version (Darwin)
			{
				auto rc = semaphore_destroy(mach_task_self(), m_hndl);
				assert(!rc, "Unable to destroy semaphore");
			}
			else
			{
				version (Posix)
				{
					int rc = sem_destroy(&m_hndl);
					assert(!rc, "Unable to destroy semaphore");
				}

			}
		}
	}
	void wait()
	{
		version (Windows)
		{
			DWORD rc = WaitForSingleObject(m_hndl, INFINITE);
			if (rc != WAIT_OBJECT_0)
				throw new SyncError("Unable to wait for semaphore");
		}
		else
		{
			version (Darwin)
			{
				while (true)
				{
					auto rc = semaphore_wait(m_hndl);
					if (!rc)
						return ;
					if (rc == KERN_ABORTED && (errno == EINTR))
						continue;
					throw new SyncError("Unable to wait for semaphore");
				}
			}
			else
			{
				version (Posix)
				{
					while (true)
					{
						if (!sem_wait(&m_hndl))
							return ;
						if (errno != EINTR)
							throw new SyncError("Unable to wait for semaphore");
					}
				}

			}
		}
	}
	bool wait(Duration period)
	in
	{
		assert(!period.isNegative);
	}
	do
	{
		version (Windows)
		{
			auto maxWaitMillis = dur!"msecs"((uint).max - 1);
			while (period > maxWaitMillis)
			{
				auto rc = WaitForSingleObject(m_hndl, cast(uint)maxWaitMillis.total!"msecs");
				switch (rc)
				{
					case WAIT_OBJECT_0:
					{
						return true;
					}
					case WAIT_TIMEOUT:
					{
						period -= maxWaitMillis;
						continue;
					}
					default:
					{
						throw new SyncError("Unable to wait for semaphore");
					}
				}
			}
			switch (WaitForSingleObject(m_hndl, cast(uint)period.total!"msecs"))
			{
				case WAIT_OBJECT_0:
				{
					return true;
				}
				case WAIT_TIMEOUT:
				{
					return false;
				}
				default:
				{
					throw new SyncError("Unable to wait for semaphore");
				}
			}
		}
		else
		{
			version (Darwin)
			{
				mach_timespec_t t = void;
				(cast(byte*)&t)[0..t.sizeof] = 0;
				if (period.total!"seconds" > t.tv_sec.max)
				{
					t.tv_sec = t.tv_sec.max;
					t.tv_nsec = cast(typeof(t.tv_nsec))period.split!("seconds", "nsecs")().nsecs;
				}
				else
					period.split!("seconds", "nsecs")(t.tv_sec, t.tv_nsec);
				while (true)
				{
					auto rc = semaphore_timedwait(m_hndl, t);
					if (!rc)
						return true;
					if (rc == KERN_OPERATION_TIMED_OUT)
						return false;
					if (rc != KERN_ABORTED || errno != EINTR)
						throw new SyncError("Unable to wait for semaphore");
				}
			}
			else
			{
				version (Posix)
				{
					import core.sys.posix.time : clock_gettime, CLOCK_REALTIME;
					timespec t = void;
					clock_gettime(CLOCK_REALTIME, &t);
					mvtspec(t, period);
					while (true)
					{
						if (!sem_timedwait(&m_hndl, &t))
							return true;
						if (errno == ETIMEDOUT)
							return false;
						if (errno != EINTR)
							throw new SyncError("Unable to wait for semaphore");
					}
				}

			}
		}
	}
	void notify()
	{
		version (Windows)
		{
			if (!ReleaseSemaphore(m_hndl, 1, null))
				throw new SyncError("Unable to notify semaphore");
		}
		else
		{
			version (Darwin)
			{
				auto rc = semaphore_signal(m_hndl);
				if (rc)
					throw new SyncError("Unable to notify semaphore");
			}
			else
			{
				version (Posix)
				{
					int rc = sem_post(&m_hndl);
					if (rc)
						throw new SyncError("Unable to notify semaphore");
				}

			}
		}
	}
	bool tryWait()
	{
		version (Windows)
		{
			switch (WaitForSingleObject(m_hndl, 0))
			{
				case WAIT_OBJECT_0:
				{
					return true;
				}
				case WAIT_TIMEOUT:
				{
					return false;
				}
				default:
				{
					throw new SyncError("Unable to wait for semaphore");
				}
			}
		}
		else
		{
			version (Darwin)
			{
				return wait(dur!"hnsecs"(0));
			}
			else
			{
				version (Posix)
				{
					while (true)
					{
						if (!sem_trywait(&m_hndl))
							return true;
						if (errno == EAGAIN)
							return false;
						if (errno != EINTR)
							throw new SyncError("Unable to wait for semaphore");
					}
				}

			}
		}
	}
	protected 
	{
		version (Windows)
		{
			alias Handle = HANDLE;
		}
		else
		{
			version (Darwin)
			{
				alias Handle = semaphore_t;
			}
			else
			{
				version (Posix)
				{
					alias Handle = sem_t;
				}
			}
		}
		Handle m_hndl;
	}
}
