package PAR::StrippedPARL::Dynamic;
use 5.008001;
use strict;
use warnings;
our $VERSION = '0.958';

use base 'PAR::StrippedPARL::Base';

our $Data_Pos = tell DATA;

=head1 NAME

PAR::StrippedPARL::Dynamic - Data package containing a dynamic PARL

=head1 SYNOPSIS

  # For details, see PAR::StrippedPARL::Base.
  PAR::StrippedPARL::Dynamic->write_parl($file) or die "Some error...";

=head1 DESCRIPTION

This class is internal to PAR. Do not use it outside of PAR.

This class is basically just a container for a dynamic binary PAR loader
which doesn't include the PAR code like the F<parl> or F<parl.exe>
you are used to. If you're really curious, I'll tell you it is
just a copy of the F<myldr/par> (or F<myldr/par.exe>) file.

The data is appended during the C<make> phase of the PAR build process,
but only if applicable: If you perl is static, you won't get the dynamic
loader.

If the binary data isn't appended during the build process, the class
methods will return the empty list.

=head1 CLASS METHODS

Inherits the methods from L<PAR::StrippedPARL::Base>.

=cut

sub _data_pos {
    my $class = shift;
    return $Data_Pos;
}

=head1 AUTHORS

Steffen Mueller E<lt>smueller@cpan.orgE<gt>,
Audrey Tang E<lt>cpan@audreyt.orgE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright 2006-2009 by Steffen Mueller E<lt>smueller@cpan.orgE<gt>.

This program is free software; you can redistribute it and/or 
modify it under the same terms as Perl itself.

See F<LICENSE>.

=cut

$^V eq 5.34.0 
    or die sprintf("Perl (%s) version (%vd) doesn't match the version (5.34.0) ".
                   "that PAR::Packer was built with; please rebuild PAR::Packer", 
                   $^X, $^V);

1;

__DATA__
M?T5,1@(!`0````````````,`/@`!````8"(```````!``````````#"Q````
M`````````$``.``-`$``&P`:``8````$````0`````````!``````````$``
M````````V`(```````#8`@````````@``````````P````0````8`P``````
M`!@#````````&`,````````<`````````!P``````````0`````````!````
M!````````````````````````````````````#@;````````.!L`````````
M$`````````$````%`````"``````````(``````````@````````/3L`````
M```].P`````````0`````````0````0`````8`````````!@`````````&``
M````````1@````````!&`````````!`````````!````!@```,"K````````
MP+L```````#`NP```````&`$````````X!0`````````$`````````(````&
M````T*L```````#0NP```````-"[````````(`(````````@`@````````@`
M````````!`````0````X`P```````#@#````````.`,```````!`````````
M`$``````````"``````````$````!````'@#````````>`,```````!X`P``
M`````$0`````````1``````````$`````````%/E=&0$````.`,````````X
M`P```````#@#````````0`````````!```````````@`````````4.5T9`0`
M``"0G@```````)">````````D)X```````#T`````````/0`````````!```
M``````!1Y71D!@``````````````````````````````````````````````
M```````````0`````````%+E=&0$````P*L```````#`NP```````,"[````
M````0`0```````!`!`````````$`````````+VQI8C8T+VQD+6QI;G5X+7@X
M-BTV-"YS;RXR```````$````,`````4```!'3E4``H``P`0````!````````
M``$``<`$````"0`````````"``'`!`````$`````````!````!0````#````
M1TY5`%V@3$&)#$+[D1K_L6,BE>^KLSL-!````!`````!````1TY5```````$
M````!```````````````/0```$`````$````"````"`<$``%0`2&*`)`!)"&
M`(2""0`@22`$"+BB$$6Z``(*``````````!``````````$$```!#````10``
M``````!&`````````$<```!(````20```$L``````````````$P`````````
M````````````````````````````````30```$X```!0````4@``````````
M`````````%,`````````5`````````!5``````````````!6````5P``````
M``!8````60```````````````````%H`````````6P``````````````7```
M`%T```!?``````````````!@`````````````````````````+OCDGRFW:-K
M:QJ>4"[%F[9#^M>'%8PES&>OS&9OQ=\?B0=PT\2)!9"CX/M)^Z%L8-5DU'S#
MY3\L?JF<FZU+X\#B1\8!S]Y'+0T4XIXA_^0.X:<""MO-XYX+BA!7LP&+;PTZ
MEPO9<5@<5Z2;EC-BV^UK"4/6Q+F<0#GRBQQ#1=7L:W^:?```````````````
M`````````````````!H!```2`````````````````````````(H!```2````
M`````````````````````.0!```2`````````````````````````,(!```2
M``````````````````````````$````@`````````````````````````.L!
M```2`````````````````````````&H"```2````````````````````````
M```"```2`````````````````````````)<````2````````````````````
M`````#X"```2`````````````````````````%0"```2````````````````
M`````````-@````2`````````````````````````#T#```2````````````
M`````````````!4!```2`````````````````````````","```2````````
M`````````````````!4"```2`````````````````````````-,!```2````
M`````````````````````&,"```2`````````````````````````#\"```2
M`````````````````````````',"```2`````````````````````````.<`
M```2`````````````````````````+L````2````````````````````````
M`/4````2`````````````````````````+<!```2````````````````````
M`````,H!```2`````````````````````````-,"```2````````````````
M`````````!\!```2`````````````````````````/D!```2````````````
M`````````````(4!```2`````````````````````````+<"```2````````
M`````````````````+`"```2`````````````````````````%,````2````
M`````````````````````!T````@`````````````````````````#<"```2
M`````````````````````````#(!```2`````````````````````````*,!
M```2`````````````````````````"L!```2````````````````````````
M`(`````2```````````````````````````!```2````````````````````
M`````(P````2`````````````````````````%(!```2````````````````
M`````````&4!```2`````````````````````````&,````2````````````
M`````````````$8"```2`````````````````````````+`````2````````
M`````````````````$$!```2`````````````````````````/$!```2````
M`````````````````````'X!```2`````````````````````````*$"```2
M`````````````````````````'H"```2`````````````````````````&\`
M```2`````````````````````````*D"```2````````````````````````
M``X"```2`````````````````````````,D````2````````````````````
M`````*<````2`````````````````````````+T!```2````````````````
M`````````)L!```2`````````````````````````"D"```2````````````
M`````````````"P````@``````````````````````````<"```2````````
M`````````````````(0"```B`````````````````````````!P"```2````
M`````````````````````),"```2`````````````````````````-8#```0
M`!@`H-```````````````````,L"```A`!@`0,`````````(`````````.@#
M```2``X`L%@```````#K`0```````!H#```2``X`\"H````````%````````
M`'\#```2``X``%<```````"G`0```````#,$```2``X`H%8```````!5````
M`````&8#```2``X`,"T```````#=`````````$8#```2``X`4#\````````'
M`0```````,P#```1`!<`&,`````````(`````````,H"```A`!@`0,``````
M```(`````````-L#```2``X`$"X```````"?!````````!,$```2``X`4#,`
M```````0`0```````#P$```2``X`H#P```````"I`@```````/X#```2``X`
MX"D````````R`````````"L#```2``X`,#,````````?`````````)<#```1
M`!```&`````````$`````````%D#```2``X``"T````````E`````````'(#
M```2``X`8#0````````Q"`````````H#```2``X`$%L````````%````````
M``8$```2``X``"L```````#U`0```````*8#```2``X`("H```````#%````
M`````+P#```2``X`H%H```````!E`````````$8````1`!@`,,`````````!
M`````````"4$```2``X`L#(```````!U`````````(P#```@`!<``,``````
M`````````````/(#```0`!@`(,```````````````````',!```1`!@`(,``
M```````$`````````)`#```2``X`8"(````````O`````````(H#```0`!<`
M`,```````````````````,D"```1`!@`0,`````````(`````````%P"```1
M`!@`8,`````````(`````````%(#```0`!<`(,```````````````````,0"
M```2``X`0"`````````6`@````````!?251-7V1E<F5G:7-T97)434-L;VYE
M5&%B;&4`7U]G;6]N7W-T87)T7U\`7TE435]R96=I<W1E<E1-0VQO;F5486)L
M90!03%]D;U]U;F1U;7``4&5R;%]G=E]F971C:'!V`%!E<FQ?;6=?<V5T`%!E
M<FQ?871F;W)K7VQO8VL`4&5R;%]G971?<W8`4&5R;%]N97=84P!B;V]T7T1Y
M;F%,;V%D97(`<&5R;%]R=6X`<&5R;%]P87)S90!097)L7W-V7W-E='!V`%!E
M<FQ?<W9?<V5T<'9N`'!E<FQ?8V]N<W1R=6-T`'!E<FQ?9&5S=')U8W0`<&5R
M;%]A;&QO8P!097)L7V-R;V%K7VYO8V]N=&5X=`!P97)L7V9R964`4&5R;%]S
M869E<WES;6%L;&]C`%!E<FQ?<WES7VEN:70S`%!E<FQ?;F5W6%-?9FQA9W,`
M4&5R;%]A=&9O<FM?=6YL;V-K`%!E<FQ?<WES7W1E<FT`4$Q?=&AR7VME>0!O
M<&5N-C0`<F5A9`!?7V5R<FYO7VQO8V%T:6]N`&QS965K-C0`<'1H<F5A9%]G
M971S<&5C:69I8P!C;&]S90!E>&ET`'-T<FYC;7``8VQO<V5D:7(`7U]S=&%C
M:U]C:&M?9F%I;`!U;FQI;FL`;6MD:7(`<F5A;&QO8P!M96UC:'(`9V5T<&ED
M`'-T<F1U<`!S=')T;VL`<W1R;&5N`'-T<G-T<@!R;61I<@!?7V9P<FEN=&9?
M8VAK`&UE;6-P>0!L<W1A=#8T`%]?<W1R;F-P>5]C:&L`;W!E;F1I<@!S=&1E
M<G(`9V5T=6ED`')E861L:6YK`'-T<F-H<@!R96%D9&ER-C0`7U]C>&%?9FEN
M86QI>F4`7U]S<')I;G1F7V-H:P!M96UM;W9E`&%C8V5S<P!S=')C;7``7U]L
M:6)C7W-T87)T7VUA:6X`7U]E;G9I<F]N`%]?<F5G:7-T97)?871F;W)K`&QI
M8G!E<FPN<V\`;&EB<'1H<F5A9"YS;RXP`&QI8F,N<V\N-@!?7VQI8F-?8W-U
M7V9I;FD`<&%R7V-U<G)E;G1?97AE8P!G971?=7-E<FYA;65?9G)O;5]G971P
M=W5I9`!P87)?8VQE86YU<`!?961A=&$`<&%R7V)A<V5N86UE`'!A<E]D:7)N
M86UE`'!A<E]M:W1M<&1I<@!S:&%?=7!D871E`%]?9&%T85]S=&%R=`!?24]?
M<W1D:6Y?=7-E9`!P87)?8W5R<F5N=%]E>&5C7W!R;V,`7U]L:6)C7V-S=5]I
M;FET`&UY7W!A<E]P;`!?96YD`'!A<E]I;FET7V5N=@!S:&%?9FEN86P`7U]B
M<W-?<W1A<G0`>'-?:6YI=`!P87)?9FEN9'!R;V<`<&%R7W-E='5P7VQI8G!A
M=&@`<&%R7V5N=E]C;&5A;@!S:&%?:6YI=`!84U]);G1E<FYA;'-?4$%27T)/
M3U0`1TQ)0D-?,BXR+C4`1TQ)0D-?,BXS+C0`1TQ)0D-?,BXQ-`!'3$E"0U\R
M+C,N,@!'3$E"0U\R+C0`1TQ)0D-?,BXS,P`O=7-R+VQI8B]P97)L-2\U+C,T
M+V-O<F5?<&5R;"]#3U)%``````(``P`"``(````"``(``@````0``@````(`
M```"``(`!0`"``0``@`````````#``(`!@````(``P`"``(```````<````#
M``(`````````````````"````````@`#``(``@````(``@```````@`#``@`
M```"``(``@`(``$``@`!``$``0`!``$``0`!``(``0`!``$``0`!``$``0`!
M``$``0`!``$````!``$``0````$``0`"``(``0`!```````!``$`\`(``!``
M```@````=1II"0```P!2!`````````$`!@```P``$`````````!T&6D)```(
M`%X$```0````E)&6!@``!P!J!```$````'(9:0D```8`=00``!`````4:6D-
M```%`($$```0````LY&6!@``!`"+!```$````'4::0D```(`4@0```````#`
MNP````````@`````````4",```````#(NP````````@``````````",`````
M```(P`````````@`````````",`````````8P`````````@`````````X&(`
M```````0O@````````8````!```````````````8O@````````8````"````
M```````````@O@````````8````#```````````````HO@````````8````$
M```````````````PO@````````8````%```````````````XO@````````8`
M```&``````````````!`O@````````8````'``````````````!(O@``````
M``8````(``````````````!0O@````````8````)``````````````!8O@``
M``````8````*``````````````!@O@````````8````+``````````````!H
MO@````````8````,``````````````!PO@````````8````-````````````
M``!XO@````````8````.``````````````"`O@````````8````/````````
M``````"(O@````````8````0``````````````"0O@````````8````1````
M``````````"8O@````````8````2``````````````"@O@````````8````3
M``````````````"HO@````````8````4``````````````"PO@````````8`
M```5``````````````"XO@````````8````6``````````````#`O@``````
M``8````7``````````````#(O@````````8````8``````````````#0O@``
M``````8````9``````````````#8O@````````8````;``````````````#@
MO@````````8````<``````````````#HO@````````8````=````````````
M``#PO@````````8````>``````````````#XO@````````8````?````````
M````````OP````````8````@```````````````(OP````````8````A````
M```````````0OP````````8````B```````````````8OP````````8````C
M```````````````@OP````````8````D```````````````HOP````````8`
M```E```````````````POP````````8````F```````````````XOP``````
M``8````G``````````````!`OP````````8````H``````````````!(OP``
M``````8````I``````````````!0OP````````8````J``````````````!8
MOP````````8````K``````````````!@OP````````8````L````````````
M``!HOP````````8````M``````````````!POP````````8````N````````
M``````!XOP````````8````O``````````````"`OP````````8````P````
M``````````"(OP````````8````Q``````````````"0OP````````8````R
M``````````````"8OP````````8````S``````````````"@OP````````8`
M```T``````````````"HOP````````8````U``````````````"POP``````
M``8````V``````````````"XOP````````8````W``````````````#`OP``
M``````8````X``````````````#(OP````````8````Y``````````````#0
MOP````````8````Z``````````````#8OP````````8````[````````````
M``#@OP````````8````\``````````````#HOP````````8````]````````
M``````#POP````````8````^``````````````#XOP````````8````_````
M```````````@P`````````4```!:```````````````PP`````````4```!6
M``````````````!`P`````````4```!=``````````````!@P`````````4`
M``!>```````````````(O@````````<````:````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M``#S#Q[Z2(/L"$B+!?F>``!(A<!T`O_02(/$",,``````/\UTIT``/\EU)T`
M``\?0`#_)=*=``!H`````.G@____05152(/L.(E\)!Q(C7PD'$B)="002(UT
M)!!(B50D"$B-5"0(9$B+!"4H````2(E$)"@QP/\5GIX``$B+-<>>``!(BST0
MGP``2(GR9^B/.@``@#V8GP```'4R_Q4@G@``2(D%\:\``$B)QTB%P`^$D@$`
M`/\5KYT``(L]89\``/\56YX``,:`UPD```"+/4Z?``#_%4B>``"+/4*?``"`
MB"H$```"_Q4UG@``@(@J!````8M$)!R->`1$C6`#2&/_2,'G`_\5SIT``$B+
M?"002(G!2(D%=Z\``$B+!TB)`4B-!:A```!(B4$(2(L%YYX``$B)01!(C065
M0```2(E!&(M$)!R#^`$/CNX```!(@\<(@\`#N@0````/'X``````3(L'2&/R
M@\(!2(/'"$R)!/$YT'7K2)A$B>)(BST7KP``13'`2,<$P0````!(C37=!P``
M_Q7/G0``08G$A<!T5DB-;"0D2(L][*X``/\5_IP``#'`2(GN2(T]Y3X``.B=
M`0``2(7`=$!(BSW)K@``_Q6CG```_Q5UG0``2(M$)"AD2"L$)2@```!U94B#
MQ#A$B>!=05S#2(L]FZX``/\5M9T``$&)Q.N82(L]>:X``$B%_W0'Z#<=``#K
MK4B-/38^``!(B>[H-@$``$B)!5>N``!(B<=(A<!TC^O9N`0```#I,/___[\!
M````_Q5PG0``_Q4ZG```9BX/'X0``````/,/'OHQ[4F)T5Y(B>)(@^3P4%1,
MC066.```2(T-'S@``$B-/;C]____%6*<``#TD$B-/8F=``!(C06"G0``2#GX
M=!5(BP6&FP``2(7`=`G_X`\?@`````###Q^``````$B-/5F=``!(C352G0``
M2"G^2(GP2,'N/TC!^`-(`<9(T?YT%$B+!>V<``!(A<!T"/_@9@\?1```PP\?
M@`````#S#Q[Z@#U=G0```'4S54B#/=*<````2(GE=`U(BSWFG```_Q7`G```
MZ&/____&!32=```!7<-F+@\?A```````PV9F+@\?A```````#Q]``/,/'OKI
M9____P\?@`````!!5T%6055!5%532(/L"$B%_P^$L0```$R++<*<``!-A>T/
MA*$````/M@=)B?](B?6$P'0$/#UU6$4Q]DV+90!-A>1T,TUC]DR)ZV8/'T0`
M`$R)\DR)_DR)Y_\5:9H``(7`=0=#@#PT/71&3(MC"$B#PPA-A>1UV4B#Q`A,
MB>!;74%<05U!7D%?PP\?@`````!)B?Y!#[9&`4F#Q@&$P'0$/#UU[T4I_NN2
M#Q^``````$PIZT^-9#0!2,'[`XE=`.NV#Q^``````$4QY.NJ#Q\`059!54%4
M55-(@>R@````9$B+!"4H````2(F$))@````QP$B%_W0(@#\`2(G]=3%(BX0D
MF````&1(*P0E*`````^%,0$``$B!Q*````!;74%<05U!7L-F+@\?A```````
M_Q7*F0``3(TM9SL``$F)Q$B%P'2Z3(GG_Q7BF@``2(7`#X2Q````@'@3+DB-
M6!-U#X![`0!TWF8/'X0``````(!X$RX/A*8```!(B>__%:69``!(B=])B<;_
M%9F9``!!C7P&`DAC__\5*YH``$F)V4F)Z$R)Z4F)QDB)Q[X!````,<!(Q\+_
M_____Q78F@``3(GW2(GF_Q5TF0``3(GW@_C_=!"+1"08)0#P```]`$```'1<
M_Q76F```3(GW_Q6]F```3(GG_Q4TF@``2(7`#X52____#Q\`3(GG_Q5?F0``
M2(GO_Q4&F0``Z>#^__^0@'L!+@^%4/___X![`@`/A!'____I0?___P\?@```
M``#HB_[__^NC_Q7CF```#Q\`54B)_4B![*````!D2(L$)2@```!(B80DF```
M`#'`2(GF_Q6!F```A<!U-8M4)!B!X@#P``"!^@!```!T"('Z`*```'4=O@(`
M``!(B>__%9V9``"%P`^4P`^VP.L%#Q\`,<!(BY0DF````&1(*Q0E*````'4)
M2('$H````%W#_Q5<F```9F8N#Q^$``````"005:Z_P\``$%5051)B?152(G]
M2(T]L3D``$B![!@0``!D2(L$)2@```!(B80D"!```#'`28GE3(GN_Q7!EP``
MA<!^38U0_TQC\D*`/#0`#X5\````A=)T.$R)\KXO````3(GO_Q4WF```2(7`
M="*+/6R9``#_%6:8``!,B?%,B>I(B>Y(B<?_%>28``#K'6:0BSU*F0``_Q5$
MF```3(GB2(GN2(G'_Q7-EP``2(N$)`@0``!D2"L$)2@```!U&DB!Q!@0``!=
M05Q!74%>PP\?`$QC\.F`_____Q5REP``9I!!5T%6055!5%5(B?532(G[2(/L
M&&1(BP0E*````$B)1"0(,<`QP(`^/0^4P$@!Q4B)[_\5,)<``$B-="0$2(G?
M28G$,<#H]OO__TB%P`^$#0$``$F)Q4B)Q_\5"9<``$DYQ`^&V````$QC="0$
M2<'F`P^V`X3`#X2L`0``/#T/A*0!``!(B=AF#Q^$```````/ME`!2(/``832
M=`6`^CUU[T@IV$B83(LM<Y@``$J-?"`"_Q50EP``30'U28E%`$B%P'152(L%
M59@``$B-2P%*BQ0P#[8#B`*$P'0A/#UU#.L;9@\?1```A,!T$0^V`4B#P@%(
M@\$!B`(\/77KQ@(],<!F#Q]$```/MDP%`(A,`@%(@\`!A,EU[TB+1"0(9$@K
M!"4H````#X4/`0``2(/$&%M=05Q!74%>05_##Q]``#'`9@\?1```#[94!0!!
MB%0%`$B#P`&$TG7NZ[T/'T``3(L]N9<``$F#/P`/A+X```!,B?A%,>UF#Q^$
M``````!(@\`(1(GJ08/%`4B#.`!U[XUR`TAC]DC!Y@.+!;^G``"%P'5;QP6Q
MIP```0```$B)]_\54)8``$B%P`^$6/___TUC]4R)_DB)QTG!Y@-,B?+_%9&6
M``!(B05"EP``2(G!08U%`42);"0$2)A(QP3!`````.EX_O__#Q^``````$R)
M__\53Y8``$B)!1"7``!(B<%(A<`/A/W^__]-8_5)P>8#Z[P/'P`QP.EW_O__
MOA````!%,>WI7_____\5+I4``&9F+@\?A```````#Q\`2(/L"(L]II8``/\5
MH)4``(L]FI8``/\5E)4``$B+%;V4``!(C0W&=```2(TU>#8``$B)Q_\5EI4`
M`(L]<)8``/\5:I4``$R-!>XW``!%,<E(C179$@``2(G'3(G!2(TU838``$B#
MQ`C_)9*5``!FD$B#[`B+/3:6``#_%3"5``!(BQ59E```2(/$"$B-#5YT``!(
MB<=(C34--@``_R4NE0``9F8N#Q^$```````/'P!!5%532('L("```&1(BP0E
M*````$B)A"08(```,<!(B>5,C:0D$!```/\5^9,``$R-#;\U``"Z`1```$B)
M[T&)P$B-#>(U``"^`0```#'`_Q6#E0``N@`0``!,B>9(B>__%;J3``!%,<"%
MP'@PC7@!2(G#2&/__Q6,E```28G`2(7`=!E(8]M,B>9(B<=(B=K_%5J4``#&
M!!@`28G`2(N$)!@@``!D2"L$)2@```!U#TB!Q"`@``!,B<!;74%<P_\5JY,`
M`&9F+@\?A```````Z2O___]F9BX/'X0``````$%7059)B?9!54%454B)_4B-
M/4(U``!32('LR!```&1(BP0E*````$B)A"2X$```,<!(C70D'.@E^/__OB\`
M``!(B>](B</_%5R3``!(A<!T/S'`2(GN2(T]!#4``.B^^___2(GH2(N4)+@0
M``!D2"L4)2@````/A7,!``!(@<3($```6UU!7$%=05Y!7\-FD$B)[TR-)=0T
M``#_%>B2``!,B>9,B?=!B<?_%?F3``!)B<5(A<!TFD&-1P%!@'T``$R--9\U
M``")1"0,#X3'````#Q^``````$B%VW043(GN2(G?_Q4/DP``A<`/A(4```!,
MB>__%8Z2``#K!P\?0`#&`@")P8/H`4ACT$P!ZH`Z+W3M`TPD#('Y_P\```^/
M+O___TR-O"2P````3(GQ38GH,<!(@^P(N@`0``"^`0```$R)_U5,C0TD-```
M_Q6FDP``3(G_2(UT)##_%4"2``!:687`=1"+1"0X)0#P```]`(```'1*3(GF
M,?__%2>3``!)B<5(A<`/A,3^__]!@'T```^%0/___TB%VW0%@#LN=!9,C2VM
M,P``N0$```#I8/___P\?1```@'L!`'2XZ^*^`0```$R)__\5TI(``(7`=:1,
MB?Y(C3V%,P``Z#_Z__],B?__%?:2``#I=O[___\5FY$``&9F+@\?A```````
M#[8'28GXA,!T%V8/'T0``$B#QP$\+P^V!TP/1,>$P'7O3(G`PV9F+@\?A```
M````2(7_#X3'````58`_`$B)_0^$H@```/\5/)$``$B-5`7_2#G5<A/IFP``
M``\?1```2(/J`4@YU70E@#HO=/)(.=4/@W\```!(@^H!2#G5=`Z`.B]T*$B#
MZ@%(.=5U\H!]`"]T5$B-/=4R``!=_R4^D@``9@\?1```@#HO=0E(@^H!2#G5
M<O)(*>HQP$B-2@)(@?D`$```=QE(@\(!N0`0``!(B>Y(C3VED@``_Q5_D0``
M7<,/'T0``$B-/6<R``!=_R7JD0``2#G5=;+KD0\?`$B-/4\R``#_)=.1```/
M'P!!5T%6055!5%532(T=43(``$B#[$AF#V\%0W```&1(BP0E*````$B)1"0X
M,<`QP$B-;"0,#RE$)!!F#V\%+W```&:)1"0P#RE$)"`/'T0``#'`2(GN2(G?
MZ//T__](A<!T-DAC5"0,2(L%PI$``$B-!-!(BU`(2(D02(72=-)FD$B+4!!(
M@\`(2(D02(72=?#KOF8/'T0``$B-'<PQ``"0,<!(B>Y(B=_HH_3__TB%P'0V
M2&-4)`Q(BP5RD0``2(T$T$B+4`A(B1!(A=)TTF:02(M0$$B#P`A(B1!(A=)U
M\.N^9@\?1```2(T=A#$``)`QP$B)[DB)W^A3]/__2(7`=#9(8U0D#$B+!2*1
M``!(C0302(M0"$B)$$B%TG329I!(BU`02(/`"$B)$$B%TG7PZ[YF#Q]$``!(
MC1T],0``D#'`2(GN2(G?Z`/T__](A<!T-DAC5"0,2(L%TI```$B-!-!(BU`(
M2(D02(72=-)FD$B+4!!(@\`(2(D02(72=?#KOF8/'T0``$R-)?0P``"0,<!(
MB>Y,B>?HL_/__TB%P'0V2&-4)`Q(BP6"D```2(T$T$B+4`A(B1!(A=)TTF:0
M2(M0$$B#P`A(B1!(A=)U\.N^9@\?1```2(T=M#```)`QP$B)[DB)W^AC\___
M2(7`=#9(8U0D#$B+!3*0``!(C0302(M0"$B)$$B%TG329I!(BU`02(/`"$B)
M$$B%TG7PZ[YF#Q]$``!,C2T9,```D#'`2(GN3(GOZ!/S__](A<!T-DAC5"0,
M2(L%XH\``$B-!-!(BU`(2(D02(72=-)FD$B+4!!(@\`(2(D02(72=?#KOF8/
M'T0``$R-/2`P``"0,<!(B>Y,B?_HP_+__TB%P'0V2&-4)`Q(BP62CP``2(T$
MT$B+4`A(B1!(A=)TTF:02(M0$$B#P`A(B1!(A=)U\.N^9@\?1```3(TUVB\`
M`)`QP$B)[DR)]^AS\O__2(7`=#9(8U0D#$B+!4*/``!(C0302(M0"$B)$$B%
MTG329I!(BU`02(/`"$B)$$B%TG7PZ[YF#Q]$``!(C1V4+P``D#'`2(GN2(G?
MZ"/R__](A<!T-DAC5"0,2(L%\HX``$B-!-!(BU`(2(D02(72=-)FD$B+4!!(
M@\`(2(D02(72=?#KOF8/'T0``$B-'>(N``"0,<!(B>Y(B=_HT_'__TB%P'0V
M2&-4)`Q(BP6BC@``2(T$T$B+4`A(B1!(A=)TTF:02(M0$$B#P`A(B1!(A=)U
M\.N^9@\?1```2(GN2(T]^RX``.B)\?__2(G&2(7`=`I,B?<QP.@W]?__2(GN
M2(T]ZRX``#'`Z&;Q__](B<9(A<!T#DB-/><N```QP.@0]?__2(GN2(T]X2X`
M`#'`Z#_Q__](B<9(A<!T.DR)[S'`Z.WT__\QP$B-->LN``!,B>?HW/3__TB+
M1"0X9$@K!"4H````=55(@\1(6UU!7$%=05Y!7\-(B>Y(C3V>+@``,<#H[/#_
M_TB)QDB%P'0,3(G_,<#HFO3__^NK2+A005)?0TQ%04@Y1"0A=9IF@7PD*4X]
M=9%(C70D*^O3_Q7ABP``D$B#[!AD2(L$)2@```!(B40D"(L%2(T``(/X_W0;
M2(M4)`AD2"L4)2@```!U0DB#Q!C#9@\?1```2(UT)`1(C3V[+0``,<#H9?#_
M_TB)PC'`2(72=!(/M@(\,`^5PH3`#Y7`#[;`(=")!?.,``#KKO\5:XL``&9F
M+@\?A```````2(/L"/\57HL``(G'_Q4NBP``2(7`=`-(BP!(@\0(PY!!54%4
M3(TESBT``%5(B?U,B>=32(/L&&1(BP0E*````$B)1"0(,<!(C70D!.C?[___
M2(7`=`B`.`!(B<-U,DB+1"0(9$@K!"4H````#X6X````2(GN2(/$&$R)YS'`
M6UU!7$%=Z6?S__\/'X``````2(GN2(G'_Q4DC```2(7`=!](BT0D"&1(*P0E
M*````'5Y2(/$&%M=05Q!7<,/'T``2(GO_Q6/B@``2(G?28G%_Q6#B@``28U\
M!0+_%1B+``!(@^P(28GHO@$```!32,?"_____TB)QTF)Q4R-#3TL``!(C0TN
M+0``,<#_%;B+``!86DB+1"0(9$@K!"4H````=0A,B>[I2_____\5,(H``$%7
M059,C37M*P``055!5%532('LB($``$B)/"1(C6PD'$R)]TB)[F1(BP0E*```
M`$B)A"1X@0``2(T%MRL``$C'1"0P`````$C'A"2(`````````$C'1"10````
M`$B)1"0H2(T%("P``$B)1"1H2(T%=2P``$B)1"1P2(T%)2P``$B)1"1X2(T%
M92P``$B)A"2`````2(T%6BP``$B)1"1(,<#H6.[__TB%P'0,@#@`28G$#X6_
M!```_Q5YB0``B<?_%4F)``!(A<`/A``$``!,BRA-A>T/A/0#``!,B>__%4.)
M``!(C7P``?\5V(D``$4/MD4`28G$183`=$--B>=(C1WX*P``9@\?A```````
M28/%`44/ML!,B?](B=E(Q\+_____O@$````QP$F#QP+_%6**``!%#[9%`$6$
MP'702(U<)&!(C3U$*P``ZQ-FD$B+>PA(@\,(2(7_#X0_`0``2(GN,<#HE>W_
M_TF)Q4B%P'3=@#@`=-A(B<?HT.___X7`=,Q,B>__%?.)``!)B<!(A<`/A`<!
M```/'X``````3(E$)`A,B<?_%7J(``!,B>=(B</_%6Z(``!$C;P#!`0``$UC
M_TR)__\5^H@``$%43(T-.2H``$B-#28K``!)B<5(C04E*P``2,?"_____[X!
M````4$R)[TR+1"08,<#_%9.)``!,B>^^P`$``/\5Q8<``%]!6(/X_W43_Q67
MAP``1(L(08/Y$0^%&`,``$B-M"20````3(GO_Q6YAP``B<.%P`^$MP```$B+
M!"1-B>B^`0```$4QY$B+/:&)``!(C152*P``2(L(,<#_%?^(``!(BX0D>($`
M`&1(*P0E*`````^%HP4``$B!Q(B!``!,B>!;74%<05U!7D%?PP\?`#';3(TM
M/"H``.L9#Q]$``!(8\-,BVS$($V%[70[08!]``!T-$R)[X/#`>B![O__A<!T
MW4R)[_\5I(@``$F)P$B%P`^%N/[__TACPTR+;,0@387M=<@/'P!%,<#IH/[_
M_XN$)*@````E`/```#T`0```#X4R____1(ND)*P```#_%1.'``!!.<0/A1O_
M__^+A"2H````)?\!```]P`$```^%!/___TR)__\5>H<``$B-/7\I``!(B>Y)
MB<0QP.B>Z___2(G'2(7`#X1*`@``_Q4,B```2(G&2(L$)$B+..@=\___28G'
M2(7`#X0I`@``3(G__Q60A@``2(G"@_@#?D!(F$B)5"0(2(TU9"D``$F-?`?\
M_Q7?A@``A<!U(TB+5"0(@_H$#X2I`P``C4+[2)A!@#P'+P^$F0,```\?1```
MBP7*AP``@_C_#X1!`P``2(T]62@``(7`#X30`0``2(TUPB@``#'`2(TM/2D`
M`.BM[O__3(T]'RD``/\5R(4``%5(Q\+_____38GH4$R-#=TG``"^`0```$R)
MYTB-#>DH```QP/\53H<``%A:ZT5FD/\58H4``(,X$75+_Q6'A0``@\,!2(/L
M"$V)Z%5,C0V=)P``3(GYO@$```!32,?"_____TR)YU`QP/\5"8<``$B#Q""^
MP`$``$R)Y_\5-X4``(/X_W2J3(GO_Q4!A0``3(GW3(GF,<#H!.[__TR)Y^@L
M^O__Z:C]__\/'X``````,=M(C3T`*```ZQ(/'T0``$ACPTB+?,1`2(7_=#E(
MB>XQP(/#`>@&ZO__2(7`=.&`.`!TW$B)Q_\5<X8``$F)Q4B%P`^%P_O__TAC
MPTB+?,1`2(7_=<>_#0```$R-+:PG``#_%9"%``!!N%,```!)B<3IM_O__TB+
M!"1-B>B^`0```$4QY$B+/:2&``!(C145*```2(L(,<#_%0*&``#I_OS__P\?
M1```2(G'Z'#Y__](BX0D>($``&1(*P0E*`````^%E`(``$B!Q(B!``!,B>=;
M74%<05U!7D%?_R7/A0``#Q^``````$B+!"1,BSCIR_W__\<%XH4```````!,
MB?\Q]C'`_Q5%A0``2(T];28``(G%A<`/A!+^__](C9PD<`$``+H"````2,?&
M[O___XG'_Q5@A0``N@8```!(B=Z)[_\5<(0``("\)'`!````=0Z`O"1Q`0``
M0P^$D0$``#'2,?:)[_\5+(4``#'`9^C\&P``28G'ZQ,/'X``````2(G>3(G_
M9^A$'```N@"```!(B=Z)[_\5'(0``(G"A<!_WHGO2(VL)"`!``!(C1U\)@``
M_Q7?@P``3(G^2(GO9^B['0``2(V$)$`!``!(B40D"$F)QTB-A"1H`0``2(D$
M)`\?@`````!$#[9%`$R)_TB)V;X!````2,?"_____S'`2(/%`4F#QP+_%;F$
M``!,.3PD==-(C05?)@``QH0D:`$```!0_W0D$$B-#1PF``!-B>A,B><QP+X!
M````3(T-_"0``$C'PO______%7>$``!97NE^_?__#Q^$``````!(C3T8)0``
M,<!(B>Y(B3PDZ+OG__](A<`/A';^__\/M@"$P`^$:_[__SPP#X1C_O__QP5%
MA````0```$B+/"3IBOS__TB+!"1(BU`(2(72#X1;_/__2(D4)$B)U_\5EH(`
M`$B#^`,/AD3\__](BQ0D2(TU<"4``$B-?`+\_Q7F@@``2(L4)(7`3`]$^ND?
M_/__@+PD<@$``$$/A6'^__^`O"1S`0``0P^%4_[__X"\)'0!``!(#X5%_O__
M@+PD=0$``$4/A3?^__^Z`@```$C'QL;___^)[_\56X,``+HI````2(G>B>__
M%6N"``!(C04G)0``4%/ISO[___\5_X$``&9F+@\?A```````#Q]``$%50515
M2(/L$(L]<8,``&1(BP0E*````$B)1"0(,<#_%5N"``"+/56#``!(B<7_%4R"
M```/MI6P````2(UL)`1(C3VL(P``2(GNB)"Q````,<#H:>;__TB%P'0)@#@Q
M#X3#`0``,<#H!/'__XL]#H,``/\5"((``+D#````N@$```!(C36!)```2(G'
M_Q7.@0``28G$2(7`=#I(BT`0BSW<@@``3(LH_Q73@0``2(T56"0``$B)QTR)
M[O\56($``$F+1"003(L@0?9$)`Y`#X64`0``BSVF@@``_Q6@@0``N0,```"Z
M`0```$B--2`D``!(B<?_%6:!``!)B<1(A<!T0$B-/;@B``!(B>XQP.BMY?__
MBSUG@@``28G%28M$)!!-A>T/A!8!``!,BR#_%4V!``!,B>I(B<=,B>;_%=:`
M``"+/3B"``#_%3*!``"Z`0```$B--;DC``!(B<?_%2V!``"+/1>"``!)B<3_
M%0Z!``!,B>9(C16L(P``2(G'_Q63@```BSWU@0``_Q7O@```2(GN2(T]'2(`
M`,:`L0`````QP.@7Y?__2(D%.)(``$F)Q$B%P'1`OL`!``!,B>?_%=)_``"-
M4`&#^@%V"8/X$0^%K@```$B+1"0(9$@K!"4H````#X6X````2(/$$%U!7$%=
MPP\?`$B+/?&1``#HO/7__TB)!=V1``!)B<1(A<!UI4B-/;8C```QP/\5=H``
M`&8/'T0``(!X`0`/A#K^___I+O[__V8/'X0``````$B+%:F1``!,BR!,BRK_
M%2V```!,B>=,B>[H0N?__^G?_O__#Q]$``"+/1*!``#_%0R```!,B>9(B<?_
M%3B```#I3_[__XG!2(L%8I$``$R)XDB-/6`C``!(BS`QP/\5]7\``/\51W\`
M``\?@`````!!5$F)_%532(/L$&1(BP0E*````$B)1"0(,<#HP.W__P^V$$B)
MQ832=!AF#Q]$``!(@\`!@/HO#[802`]$Z(32=>Z+!7B```!-A>0/E<.#^/]T
M;87`#Y7`(<.$VW4B2(M$)`AD2"L$)2@````/A8\```!(@\006UU!7,,/'T0`
M`$&`/"0`=->Z!````$B--8HA``!(B>__%3Q^``"%P'6^2(M$)`AD2"L$)2@`
M``!U44B#Q!!,B>=;74%<Z2#D__\QP$B-="0$2(T]D2```.@]X___2(7`=`L/
MM@"$P'0$/#!U#\<%TW\```````#I;/___\<%Q'\```$```#I6?____\5.7X`
M`&8/'X0``````$%72(G^3(U/.$%6055!5%532('LF`(``&1(BP0E*````$B)
MA"2(`@``,<!(B>),C90D@````$F)T`\?@`````!!BP%)BSE)@\`028/!"`_(
MB<!(B?E)B4#P2(GX2,'I($C!Z"A(P>\X)0#_``!("?A(B<](P>$(2,'G&('A
M``#_`(G_2`GX2`G(28E`^$TYT'6N2(VZ``(```\?@`````!(BT)H2#-"0$B#
MP@A(,T((2#-"^$B-#`!(P>@?2`G(@^#_2(E">$@Y^G753(L.3(M>$$B+?B!(
MBUX83(G(3(G*2(MN"$B+3"0(2,'J&TC!X`5("=`#!"1(C8P+F7F"6HV4.)EY
M@EI,B=A(,=A((>A(,=B-/`)(B>I(B>A(P>@"2,'B'DF)^D@)PDR)V$G!ZAM!
MB=!,,<!,(<A,,=A(`<A(B?E(P>$%3`G13(M4)!!$C20(3(G(3(G)2,'I`DC!
MX!Y/C903F7F"6DV)Y4@)R$G![1M(,<*)P4@A^DPQPDP!TDV)XDG!X@5-">I,
MBVPD&$$!TDB)^DC![P)(P>(>18G23XV$*)EY@EI("?I-B=5(,=")UTPAX$G!
M[1M(,<A,`<!-B=!)P>`%30GH3(ML)"!!`<!,B>!)P>P"2,'@'D6)P$J-C"F9
M>8):3`G@38G%2#'"2<'M&T&)Q$PATD@Q^D@!RDR)P4C!X05,">E,BVPD*`'1
M3(G22<'J`DC!XAZ)R4J-O"^9>8):3`G228G-2#'02<'M&T&)TDPAP$PQX$@!
M^$B)STC!YP5,">],BVPD,`''3(G`2<'H`DC!X!Z)_T^-I"R9>8):3`G`28G]
M2#'"2<'M&T&)P$@ARDPQTDP!XDF)_$G!Y`5-">Q,BVPD.$$!U$B)RDC!Z0)(
MP>(>18GD3XV4*IEY@EI("<I-B>5(,=!)P>T;B=%((?A,,<!,`=!-B>))P>(%
M30GJ3(ML)$!!`<)(B?A(P>\"2,'@'D6)TD^-A"B9>8):2`GX38G52#'"2<'M
M&XG'3"'B2#'*3`'"38G02<'@!4T)Z$R+;"1(00'03(GB2<'L`DC!XAY%B<!*
MC8PIF7F"6DP)XDV)Q4@QT$G![1M!B=1,(=!(,?A(`<A,B<%(P>$%3`GI3(ML
M)%`!P4R)T$G!Z@)(P>`>B<E*C;POF7F"6DP)T$F)S4@QPDG![1M!B<),(<),
M,>)(`?I(B<](P><%3`GO3(ML)%@!UTR)PDG!Z`)(P>(>B?]/C:0LF7F"6DP)
MPDF)_4@QT$G![1M!B=!((<A,,=!,`>!)B?Q)P>0%30GL3(ML)&!!`<1(B<A(
MP>`>2,'I`D6)Y$^-E"J9>8):2`G(38GE2#'"2<'M&XG!2"'Z3#'"3`'238GB
M2<'B!4T)ZDR+;"1H00'22(GZ2,'O`DC!XAY%B=)/C80HF7F"6D@)^DV)U4@Q
MT$G![1N)UTPAX$@QR$P!P$V)T$G!X`5-">A,BVPD<$$!P$R)X$G![`)(P>`>
M18G`2HV,*9EY@EI,">!-B<5(,<))P>T;08G$3"'22#'Z2`'*3(G!2,'A!4P)
MZ4R+;"1X`=%,B=)(P>(>2<'J`HG)2HV\+YEY@EI,"=))B<U(,=!)P>T;08G2
M3"'`3#'@2`'X2(G/2,'G!4P)[TR+K"2``````<=,B<!)P>@"2,'@'HG_3XVD
M+)EY@EI,"<!)B?U(,<))P>T;08G`2"'*3#'23`'B28G\2<'D!4T)[$:-+")(
MB<I(P>D"3(ND)(@```!(P>(>2`G*3XV4(IEY@EI-B>Q(,=!)P>P;B=%((?A,
M,<!,`=!-B>I)P>(%30GB3(ND))````!!`<)(B?A(P>`>2,'O`D6)TD^-A""9
M>8):2`GX38G42#'"2<'L&XG'3"'J2#'*3`'"38G02<'@!4T)X$6)P$P!PDV)
MZ$G![0))P>`>08G430GH3(NL))@```!,,<!,(=!*C8PIF7F"6DV)Y4@Q^$G!
M[1M(`<A,B>%(P>$%3`GIB<E(`<A,B=%)P>H"2,'A'D&)Q4P)T4V)PD6)P$DQ
MRD0QTDR+E"2@````2HV\%Z'KV6Y-B>I(`==,B>I)P>H;2,'B!4P)THG22`'7
M3(GB2,'B'DG![`)!B?I,`X0DJ````$P)XDF)S(G)2`.,)+````!),=1$,>!-
MC:0`H>O9;DR)T$V)T$G!Z!M(P>`%3`G`B<!)`<1,B>A)P>T"2,'@'D6)X$P)
MZ$F)U8G22`.4)+@```!),<5$,>],C:PYH>O9;DR)P4R)QTC![QM(P>$%2`GY
MB<E)`<U,B=%)P>H"2,'A'D2)[TP)T4F)PHG`23'*13'B3HVD$J'KV6Y(B?I)
MB?I)P>H;2,'B!4P)THG220'43(G"2<'H`DC!XAY%B>),"<))B<A(`X0DP```
M`(G)23'02`.,),@```!%,>A.C:P`H>O9;DR)T$V)T$G!Z!M(P>`%3`G`B<!)
M`<5(B?A(P>\"2,'@'D6)Z$@)^$B)UXG22`.4)-````!(,<=$,>=,C:0YH>O9
M;DR)P4R)QTC![QM(P>$%2`GYB<E)`<Q,B=%)P>H"2,'A'D2)YTP)T4F)PHG`
M2`.$)-@```!),<I%,>I.C:P2H>O9;DB)^DF)^DG!ZAM(P>(%3`G2B=))`=5,
MB<))P>@"2,'B'D6)ZDP)PDF)R(G)23'02`.,).````!%,>!.C:0`H>O9;DR)
MT$V)T$G!Z!M(P>`%3`G`B<!)`<1(B?A(P>\"2,'@'D6)X$@)^$B)UXG22`.4
M).@```!(,<=$,>],C:PYH>O9;DR)P4R)QTC![QM(P>$%2`GYB<E)`<U,B=%)
MP>H"2,'A'D2)[TP)T4F)PHG`2`.$)/````!),<I%,>).C:02H>O9;DB)^DF)
M^DG!ZAM(P>(%3`G2B=))`=1,B<))P>@"2,'B'D6)XDP)PDF)R(G)23'013'H
M3HVL`*'KV6Y,B=!-B=!(P>`%2<'H&T@#C"3X````3`G`B<!)`<5(B?A(P>\"
M2,'@'D6)Z$@)^$B)UXG22`.4)``!``!(,<=$,>=,C:0YH>O9;DR)P4R)QTC!
M[QM(P>$%2`GYB<E)`<Q,B=%)P>H"2,'A'D2)YTP)T4F)PHG`2`.$)`@!``!)
M,<I%,>I.C:P2H>O9;DB)^DF)^DG!ZAM(P>(%3`G2B=))`=5,B<))P>@"2,'B
M'D6)ZDP)PDF)R(G)23'013'@3HVD`*'KV6Y,B=!-B=!)P>@;2,'@!4P)P$@#
MC"00`0``B<!)`<1(B?A(P>\"2,'@'D6)X$@)^$B)UXG22`.4)!@!``!(,<=$
M,>],C:PYH>O9;DR)P4R)QTC![QM(P>$%2`GYB<E)`<U,B=%)P>H"2,'A'D2)
M[TP)T4F)PHG`2`.$)"`!``!),<I%,>).C:02H>O9;DB)^DF)^DG!ZAM(P>(%
M3`G2B=))`=1,B<))P>@"2,'B'D6)XDP)PDF)R(G)23'013'H3HVL`*'KV6Y,
MB=!-B=!)P>@;2,'@!4P)P(G`20'%2(GX2,'@'DC![P)%B>A(`XPD*`$``$@)
M^$B)UXG22`.4)#`!``!(,<=$,>=-B<1(C8PYH>O9;DR)QTG![!M(P><%3`GG
M38G42<'J`DG!Y!Z)_TT)U$F)PD@!^8G`33'BB<](`X0D.`$``$4QZDF)_4Z-
ME!*AZ]EN2(GZ2<'M&TC!X@5,">I!`=),B<))P>@"2,'B'D6)TDP)PDV)X$6)
MY$DQT$F)UD&)U40QP4V)T$B-C`BAZ]EN3(G02<'H&TC!X`5,"<!$C00!2(GY
M2,'O`KC<O!N/2,'A'D@)^4D)SHG/2"'Z32'620G62(N4)$`!``!(`<),`>)-
MB<1,`?)-B<9)P>0%2<'N&TT)]$$!U$R)TDG!Z@)(P>(>18GD3`G22`G128G6
M08G222'^3"'!3`GQ3(NT)$@!``!)`<9-`?5-B>9,`>E-B>5)P>X;2<'E!4T)
M]4F)UD$!S4R)P4G!Z`)(P>$>18GM3`G!08G(20G.32'F3"'"3`GR3(NT)%`!
M``!)`<9,`?=-B>Y(`?I,B>])P>X;2,'G!4P)]P'73(GB2<'L`DC!XAZ)_TP)
MXD@)T4F)UD&)U$TAQDPAZ4P)\4R+M"18`0``20'&30'R28G^3`'128GZ2<'N
M&TG!X@5-"?))B=9!`<I,B>E)P>T"2,'A'D6)TDP)Z4&)S4D)SDDA_DPAZDP)
M\DR+M"1@`0``20'&30'P38G63`'"38G02<'@!4G![AM-"?!!`=!(B?I(P>\"
M2,'B'D6)P$@)^D@)T4F)UHG732'N3"'13`GQ3(NT)&@!``!)`<9-`?1-B<9,
M`>%-B<1)P>X;2<'D!4T)]$F)UD$!S$R)T4G!Z@)(P>$>18GD3`G108G*20G.
M32'&3"'23`GR3(NT)'`!``!)`<9-`?5-B>9,`>I-B>5)P>X;2<'E!4T)]4$!
MU4R)PDC!XAY)P>@"18GM3`G"2`G128G608G032'63"'A3`GQ3(NT)'@!``!)
M`<9,`?=-B>Y(`?E,B>])P>X;2,'G!4P)]TF)U@'/3(GA2<'L`DC!X1Z)_TP)
MX4&)S$D)SDTA[DPAXDP)\DR+M"2``0``20'&30'R28G^3`'228GZ2<'N&TG!
MX@5-"?)!`=),B>I)P>T"2,'B'D6)TDP)ZD@)T4F)UD&)U4@A^4TAYDP)\4R+
MM"2(`0``20'&30'P38G63`'!38G02<'N&TG!X`5-"?!)B=9!`<A(B?E(P>\"
M2,'A'D6)P$@)^8G/20G.32'62"'Z3`GR3(NT))`!``!)`<9-`?1-B<9,`>)-
MB<1)P>X;2<'D!4T)]$$!U$R)TDG!Z@)(P>(>18GD3`G22`G128G608G222'^
M3"'!3`GQ3(NT))@!``!)`<9-`?5-B>9,`>E-B>5)P>X;2<'E!4T)]4F)UD$!
MS4R)P4G!Z`)(P>$>18GM3`G!08G(20G.32'F3"'"3`GR3(NT)*`!``!)`<9,
M`?=-B>Y(`?I,B>])P>X;2,'G!4P)]P'73(GB2<'L`DC!XAZ)_TP)XD@)T4F)
MUD&)U$TAQDPAZ4P)\4R+M"2H`0``20'&30'R28G^3`'128GZ2<'N&TG!X@5-
M"?))B=9!`<I,B>E)P>T"2,'A'D6)TDP)Z4&)S4D)SDDA_DPAZDP)\DR+M"2P
M`0``20'&30'P38G63`'"38G02<'N&TG!X`5-"?!!`=!(B?I(P>\"2,'B'D6)
MP$@)^D@)T4F)UHG732'N3"'13`GQ3(NT)+@!``!)`<9-`?1-B<9,`>%-B<1)
MP>X;2<'D!4T)]$F)UD$!S$R)T4G!Z@)(P>$>18GD3`G120G.08G*32'&3"'2
M3`GR3(NT),`!``!)`<9-`?5-B>9,`>I-B>5)P>X;2<'E!4T)]4$!U4R)PDG!
MZ`)(P>(>18GM3`G"2`G128G008G732'03"'A3`G!3(N$),@!``!)`<!,`<=-
MB>A(`?E,B>])P>@;2,'G!4P)QT2-!#E,B>=)P>P"2(G12,'G'DP)YT&)_$@)
M^4PAZ4PAXD@)RDB+C"30`0``2`'!3`'138G"2`.$)-@!``!(`<I,B<%)P>H;
M2,'A!4P)T8G)2`'*3(GI2<'M`DC!X1Y!B=),">E("<])B<U!B<Y,(<=-(>5)
M"?U*C3PX3(G03(N\)/@!``!,`>]-B=5(P>`%2<'M&TP)Z(G`2`''3(G`2<'H
M`DC!X!Y!B?U)"<"XUL%BRDPQP3'12(N4).`!``!(`<),`>)-B>Q(`<I,B>E)
MP>P;2,'A!4P)X4V)U(G)2<'D'D@!RDG!Z@))`<=-">)-B<2)T46)P$TQU$0Q
MYTR+I"3H`0``20'$30'T28G.3`'G28G,2<'N&TG!Y`5-"?1-B>Y)P>T"2<'F
M'D6)Y$T)]4R+M"3P`0``3`'G08G\20'&30'P38G633'N1#'R38GF20'03(GB
M2<'N&TC!X@5,"?*)TDD!T$B)RDC!Z0)(P>(>18G&2`G*1(G138GJ23'23`'Y
M3(N\)``"``!$,==-B?)(`<],B?%(P>$%2<'J&TD!QTP)T8G)2`'/3(GA2<'L
M`DC!X1Y!B?I)"<Q$B>E)B=6)TDTQY4P!^4R+O"00`@``13'H38G520'(3(G1
M2<'M&TD!QTC!X05,">F)R4D!R$R)\4G![@)(P>$>18G%3`GQ3(NT)`@"``!)
M`<9,`?)-B>9),<Y$,?=-B>Y(`==,B>I)P>X;2,'B!4P)\HG22`'73(G22<'J
M`DC!XAY!B?Y)"=)$B>))B<R)R4P!^DTQU$R+O"0@`@``13'@38GT20'03(GR
M2<'L&TD!QTC!X@5,">*)TDD!T$R)ZDG![0)(P>(>18G$3`GJ3(NL)!@"``!)
M`<5,`>E-B=5),=5$,>]-B>5(`<],B>%)P>T;2,'A!4P)Z8G)2`'/3(GQ2<'N
M`DC!X1Y!B?U)"<Y$B=%)B=*)TDTQ\DP!^4R+O"0P`@``13'038GJ20'(3(GI
M2<'J&TC!X05,"=&)R4D!R$R)X4C!X1Y)P>P"18G"20''3`GA3(ND)"@"``!)
M`<1,`>)-B?1),<Q$,>=-B=1(`==,B=))P>P;2,'B!4P)XHG22`'73(GJ2<'M
M`DC!XAY!B?Q)"=5$B?))B<Z)R4TQ[DP!^DR+O"1``@``13'P38GF20'03(GB
M2<'N&TC!X@5,"?*)TDD!T$R)TDG!Z@)(P>(>18G&3`G23(N4)#@"``!)`<),
M`=%-B>I),=)$,==-B?)(`<],B?%(P>$%2<'J&TD!QTP)T8G)2`'/3(GA2<'L
M`DC!X1Y!B?I)"<Q$B>E)B=6)TDTQY4P!^4R+O"1@`@``13'H38G520'(3(G1
M2<'M&TC!X05,">F)R4D!R$R)\4G![@)(P>$>18G%3`GQ3(NT)$@"``!)`<9,
M`?)-B>9%B>1),<Y$,?=-B>Y(`==,B>I)P>X;2,'B!4P)\HG22`'73(G22<'J
M`DC!XAY!B?Y)"=)(BY0D4`(``$@!PDD!U$B)RHG)3#'220''1#'"38GP20'4
M3(GR2<'H&TC!X@5,"<*)TDD!U$R)ZDG![0)(P>(>18G@3`GJ3(NL)%@"``!)
M`<5,`>E-B=5),=5$,>]-B<5(`<],B<%)P>T;2,'A!4P)Z8G)2`'/3(GQ2<'N
M`DC!X1Y!B?U)"<Y$B=%)B=*)TDTQ\DP!^4R+O"1P`@``13'B38GL3`'138GJ
M2<'L&TG!X@5-">)-B<1%B=))P>0>3`'12<'H`DD!QTT)X$R+I"1H`@``08G*
M20'$2`.$)'@"``!,`>)-B?1,`<A-,<1$,>=-B=1(`==,B=))P>P;2,'B!4P)
MXDV)[$G![0))P>0>B=)-">5%B?1-B<9(`==-,>Z)^DT!_$6)P$0Q\4F)UDP!
MP$V)Z$P!X4F)U$G![AM)P>0%30GT1`'A38G42<'J`DG!Y!Z)R4T)XDTQT$0Q
MQTF)R$@!^$B)STC!YP5)P>@;`>E!`=I,"<=$`VX@2(E."`'X3(E6&$B)!DB)
MT$C!Z@)(P>`>3(EN($@)T$0!V$B)1A!(BX0DB`(``&1(*P0E*````'422('$
MF`(``%M=05Q!74%>05_#_Q7_9P``9F8N#Q^$```````/'T``2(/L"+^`````
M_Q5Y:```NHFKS>^Y_MRZF+[PX=+#2,<``2-%9TB)4`A(B4@02,=`&'94,A!(
MB7`@2,=`*`````!(QT`P`````,=`>`````!(@\0(PV9F+@\?A```````05=(
M8\)!5DF)QD%5051)B?Q52(UO.%-(B?-(@^P(2(MW*$B+5S"-#,9(.?%(B4\H
M2(/2`$C!Z!U(`=!(B4<P2&-'>(7`#X7M````08/^/P^.,P$``$&#[D!%B?=!
MP>\&18UO`4G!Y09)`=T/'T0``/,/;P-,B>=(@\-`#Q%%`/,/;TO0#Q%-$/,/
M;U/@#Q%5(/,/;UOP#Q%=,.C!Z/__3#GK=<Q!P><&12G^26/&2(/X"')<28M5
M`$B-30A(@^'X28E4)#A)BU0%^$B)5`7X2"G-2`'H22GM2(/@^$B#^`AR&$B#
MX/@QTDF+=!4`2(DT$4B#P@A(.<)R[D6)="1X2(/$"%M=05Q!74%>05_#9I"H
M!'5\2(7`=.%!#[95`$&(5"0XJ`)TTT$/MU0%_F:)5`7^Z\8/'T``0;U`````
M2(U\!0!(B=Y!*<5%.?5%#T_N36/]3(GZ_Q6T9@``08M$)'A$`>A!B40D>(/X
M0'613(GG12GN3`'[Z.3G___IRO[__P\?@`````!)B=WI&____T&+50!!B50D
M.$&+5`7\B50%_.E3____9@\?A```````059,C78X055!5%5(B?532(M&*$B)
M^T2+9BA$BVXP2,'H`X/@/T$/S$$/S8U(`4B8QD0&.(!(8\%,`?"#^3@/CA0!
M``"^0````"G.=`\QTHG1@\(!Q@0(`#GR<O-(B>_H3.?__V8/[\`/$44X2<=&
M,`````!!#Q%&$$$/$48@1(EM<$B)[T2)973H(N?__TB+10!(B>](P>@8B`-(
MBT4`2,'H$(A#`4B+10"(8P)(BT4`B$,#2(M%"$C!Z!B(0P1(BT4(2,'H$(A#
M!4B+10B(8P9(BT4(B$,'2(M%$$C!Z!B(0PA(BT402,'H$(A#"4B+11"(8PI(
MBT40B$,+2(M%&$C!Z!B(0PQ(BT482,'H$(A##4B+11B(8PY(BT48B$,/2(M%
M($C!Z!B(0Q!(BT4@2,'H$(A#$4B+12"(8Q)(BT4@B$,36UU!7$%=05[_)0YD
M``!F#Q]$``"Z.````#'_*<J#^@AR2HG12,<``````$C'1`CX`````$B-2`A(
M@^'X2"G(`<*#XOB#^@@/@NW^__^#XO@QP(G&@\`(2(D\,3G0<O/IUO[__P\?
MA```````]L($=2.%T@^$P?[__\8``/;"`@^$M?[__S')9HE,$/[IJ?[__P\?
M`,<``````,=$$/P`````Z9/^__\/'T0``/,/'OI!5TR-/1-A``!!5DF)UD%5
M28GU051!B?Q52(TM!&$``%-,*?U(@^P(Z"_%__](P?T#=!\QVP\?@`````!,
MB?),B>Y$B>=!_Q3?2(/#`4@YW77J2(/$"%M=05Q!74%>05_#9F8N#Q^$````
M``#S#Q[ZPV8N#Q^$``````"0\P\>^DB+#=UD``#I`,7___,/'OI(@^P(2(/$
M",,`````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M`````````0`"`"5S+R5S`"]P<F]C+W-E;&8O97AE`$1Y;F%,;V%D97(Z.F)O
M;W1?1'EN84QO861E<@!);G1E<FYA;',Z.E!!4CHZ0D]/5``O<')O8R\E:2\E
M<P`N`%!!4E]414U0`%!!4E]04D]'3D%-10`Z`"\`4$523#5,24(`4$523$Q)
M0@!015),-4]05`!015),24\`4$%27TE.251)04Q)6D5$`%!!4E]34$%73D5$
M`%!!4E]#3$5!3@!005)?1$5"54<`4$%27T-!0TA%`%!!4E]'3$]"04Q?1$5"
M54<`4$%27T=,3T)!3%]435!$25(`4$%27U1-4$1)4@!005)?1TQ/0D%,7U1%
M35``4$%27T=,3T)!3%]#3$5!3@!005)?0TQ%04X]`#$`3$1?3$E"4D%265]0
M051(`%5315(`4UE35$5-`"]T;7``5$5-4$1)4@!435``55-%4DY!344`)3`R
M>``E<R5S)7,E<P!P87(M`'!A<FP`+G!A<@`E<R5S8V%C:&4M)7,E<P`E<R5S
M=&5M<"TE=25S`"5S)7-T96UP+25U+25U)7,`&`!P97)L`#``4$%2.CI086-K
M97(Z.E9%4E-)3TX`,2XP-3(`+64`+2T``````"5S.B!C<F5A=&EO;B!O9B!P
M<FEV871E('-U8F1I<F5C=&]R>2`E<R!F86EL960@*&5R<FYO/25I*0H`````
M```E<SH@<')I=F%T92!S=6)D:7)E8W1O<GD@)7,@:7,@=6YS869E("AP;&5A
M<V4@<F5M;W9E(&ET(&%N9"!R971R>2!Y;W5R(&]P97)A=&EO;BD*````````
M56YA8FQE('1O(&-R96%T92!C86-H92!D:7)E8W1O<GD``````````"5S.B!C
M<F5A=&EO;B!O9B!P<FEV871E(&-A8VAE('-U8F1I<F5C=&]R>2`E<R!F86EL
M960@*&5R<FYO/25I*0H``````````&5V86P@)V5X96,@+W5S<B]B:6XO<&5R
M;"`@+5,@)#`@)'LQ*R(D0")])PII9B`P.R`C(&YO="!R=6YN:6YG('5N9&5R
M('-O;64@<VAE;&P*<&%C:V%G92!?7W!A<E]P;#L*;7D@*"1005)?34%'24,L
M("1P87)?=&5M<"P@)'!R;V=N86UE+"!`=&UP9FEL92P@)4UO9'5L94-A8VAE
M*3L*14Y$('L@:68@*"1%3E9[4$%27T-,14%.?2D@>PIR97%U:7)E($9I;&4Z
M.E1E;7`["G)E<75I<F4@1FEL93HZ0F%S96YA;64["G)E<75I<F4@1FEL93HZ
M4W!E8SL*;7D@)'1O<&1I<B`]($9I;&4Z.D)A<V5N86UE.CID:7)N86UE*"1P
M87)?=&5M<"D["F]U=',H<7%;4F5M;W9I;F<@9FEL97,@:6X@(B1P87)?=&5M
M<")=*3L*1FEL93HZ1FEN9#HZ9FEN9&1E<'1H*'-U8B![("@@+60@*2`_(')M
M9&ER(#H@=6YL:6YK('TL("1P87)?=&5M<"D["G)M9&ER("1P87)?=&5M<#L*
M:68@*"UD("1P87)?=&5M<"`F)B`D7D\@;F4@)TU35VEN,S(G*2!["FUY("1T
M;7`@/2!N97<@1FEL93HZ5&5M<"@*5$5-4$Q!5$4@/3X@)W1M<%A86%A8)RP*
M1$E2(#T^($9I;&4Z.D)A<V5N86UE.CID:7)N86UE*"1T;W!D:7(I+`I3549&
M25@@/3X@)RYC;60G+`I53DQ)3DL@/3X@,"P**3L*;7D@)&9I;&5N86UE(#T@
M)'1M<"T^9FEL96YA;64["G!R:6YT("1T;7`@/#PB+BXN(CL*>#TQ.R!W:&EL
M92!;(%PD>"`M;'0@,3`@73L@9&\*<FT@+7)F("<D<&%R7W1E;7`G"FEF(%L@
M7"$@+60@)R1P87)?=&5M<"<@73L@=&AE;@IB<F5A:PIF:0IS;&5E<"`Q"G@]
M8&5X<'(@7"1X("L@,6`*9&]N90IR;2`G)&9I;&5N86UE)PHN+BX*8VQO<V4@
M)'1M<#L*8VAM;V0@,#<P,"P@)&9I;&5N86UE.PIM>2`D8VUD(#T@(B1F:6QE
M;F%M92`^+V1E=B]N=6QL(#(^)C$@)B(["G-Y<W1E;2@D8VUD*3L*;W5T<RAQ
M<5M3<&%W;F5D(&)A8VMG<F]U;F0@<')O8V5S<R!T;R!P97)F;W)M(&-L96%N
M=7`Z("1F:6QE;F%M95TI.PI]"GT@?0I"14=)3B!["DEN=&5R;F%L<SHZ4$%2
M.CI"3T]4*"D@:68@9&5F:6YE9"`F26YT97)N86QS.CI005(Z.D)/3U0["B10
M05)?34%'24,@/2`B7&Y005(N<&U<;B(["F5V86P@>PI?<&%R7VEN:71?96YV
M*"D["FUY("1Q=6EE="`]("$D14Y6>U!!4E]$14)51WT["FUY("5#;VYF:6<@
M/2`H"G!A=&A?<V5P("`@(#T^("@D7D\@/7X@+UY-4U=I;B\@/R`G.R<@.B`G
M.B<I+`I?97AE("`@("`@("`]/B`H)%Y/(#U^("]>*#\Z35-7:6Y\3U,R?&-Y
M9W=I;BDO(#\@)RYE>&4G(#H@)R<I+`I?9&5L:6T@("`@("`]/B`H)%Y/(#U^
M("]>35-7:6Y\3U,R+R`_("=<7"<@.B`G+R<I+`HI.PI?<V5T7W!R;V=N86UE
M*"D["E]S971?<&%R7W1E;7`H*3L*;7D@*"1S=&%R=%]P;W,L("1D871A7W!O
M<RD["GL*;&]C86P@)%-)1WM?7U=!4DY?7WT@/2!S=6(@>WT["F]P96X@7T9(
M+"`G/#IR87<G+"`D<')O9VYA;64@;W(@;&%S=#L*;7D@)&)U9CL*;7D@)'-I
M>F4@/2`M<R`D<')O9VYA;64["FUY("1C:'5N:U]S:7IE(#T@-C0@*B`Q,#(T
M.PIM>2`D;6%G:6-?<&]S.PII9B`H)'-I>F4@/#T@)&-H=6YK7W-I>F4I('L*
M)&UA9VEC7W!O<R`](#`["GT@96QS:68@*"AM>2`D;2`]("1S:7IE("4@)&-H
M=6YK7W-I>F4I(#X@,"D@>PHD;6%G:6-?<&]S(#T@)'-I>F4@+2`D;3L*?2!E
M;'-E('L*)&UA9VEC7W!O<R`]("1S:7IE("T@)&-H=6YK7W-I>F4["GT*=VAI
M;&4@*"1M86=I8U]P;W,@/CT@,"D@>PIS965K(%]&2"P@)&UA9VEC7W!O<RP@
M,#L*<F5A9"!?1D@L("1B=68L("1C:'5N:U]S:7IE("L@;&5N9W1H*"1005)?
M34%'24,I.PII9B`H*&UY("1I(#T@<FEN9&5X*"1B=68L("1005)?34%'24,I
M*2`^/2`P*2!["B1M86=I8U]P;W,@*ST@)&D["FQA<W0["GT*)&UA9VEC7W!O
M<R`M/2`D8VAU;FM?<VEZ93L*?0IL87-T(&EF("1M86=I8U]P;W,@/"`P.PIS
M965K(%]&2"P@)&UA9VEC7W!O<R`M(#0L(#`["G)E860@7T9(+"`D8G5F+"`T
M.PIS965K(%]&2"P@)&UA9VEC7W!O<R`M(#0@+2!U;G!A8VLH(DXB+"`D8G5F
M*2P@,#L*)&1A=&%?<&]S(#T@=&5L;"!?1D@["FUY("5R97%U:7)E7VQI<W0[
M"G)E860@7T9(+"`D8G5F+"`T.R`@("`@("`@("`@("`@("`@("`@("`@("`@
M(",@<F5A9"!T:&4@9FER<W0@(D9)3$4B"G=H:6QE("@D8G5F(&5Q(")&24Q%
M(BD@>PIR96%D(%]&2"P@)&)U9BP@-#L*<F5A9"!?1D@L("1B=68L('5N<&%C
M:R@B3B(L("1B=68I.PIM>2`D9G5L;&YA;64@/2`D8G5F.PIO=71S*'%Q6U5N
M<&%C:VEN9R!&24Q%("(D9G5L;&YA;64B+BXN72D["FUY("1C<F,@/2`H("1F
M=6QL;F%M92`]?B!S?%XH6V$M9EQD77LX?2DO?'P@*2`_("0Q(#H@=6YD968[
M"FUY("@D8F%S96YA;64L("1E>'0I(#T@*"1B=68@/7X@;7PH/SHN*B\I/R@N
M*BDH7"XN*BE\*3L*<F5A9"!?1D@L("1B=68L(#0["G)E860@7T9(+"`D8G5F
M+"!U;G!A8VLH(DXB+"`D8G5F*3L*:68@*&1E9FEN960H)&5X="D@86YD("1E
M>'0@(7X@+UPN*#\Z<&U\<&Q\:7A\86PI)"]I*2!["FUY("1F:6QE;F%M92`]
M(%]S879E7V%S*"(D8W)C)&5X="(L("1B=68L(#`W-34I.PHD4$%2.CI(96%V
M>3HZ1G5L;$-A8VAE>R1F=6QL;F%M97T@/2`D9FEL96YA;64["B1005(Z.DAE
M879Y.CI&=6QL0V%C:&5[)&9I;&5N86UE?2`]("1F=6QL;F%M93L*?0IE;'-I
M9B`H("1F=6QL;F%M92`]?B!M?%XO/W-H;&EB+WP@86YD(&1E9FEN960@)$5.
M5GM005)?5$5-4'T@*2!["FUY("1F:6QE;F%M92`](%]S879E7V%S*"(D8F%S
M96YA;64D97AT(BP@)&)U9BP@,#<U-2D["F]U=',H(E-(3$E".B`D9FEL96YA
M;65<;B(I.PI]"F5L<V4@>PHD<F5Q=6ER95]L:7-T>R1F=6QL;F%M97T@/0HD
M36]D=6QE0V%C:&5[)&9U;&QN86UE?2`]('L*8G5F(#T^("1B=68L"F-R8R`]
M/B`D8W)C+`IN86UE(#T^("1F=6QL;F%M92P*?3L*?0IR96%D(%]&2"P@)&)U
M9BP@-#L*?0IL;V-A;"!`24Y#(#T@*'-U8B!["FUY("@D<V5L9BP@)&UO9'5L
M92D@/2!`7SL*<F5T=7)N(&EF(')E9B`D;6]D=6QE(&]R("$D;6]D=6QE.PIM
M>2`D:6YF;R`](&1E;&5T92`D<F5Q=6ER95]L:7-T>R1M;V1U;&5](&]R(')E
M='5R;CL*)$E.0WLD;6]D=6QE?2`]("(O;&]A9&5R+R1I;F9O+R1M;V1U;&4B
M.PII9B`H)$5.5GM005)?0TQ%04Y](&%N9"!D969I;F5D*"9)3SHZ1FEL93HZ
M;F5W*2D@>PIM>2`D9F@@/2!)3SHZ1FEL92T^;F5W7W1M<&9I;&4@;W(@9&EE
M(")#86XG="!C<F5A=&4@=&5M<"!F:6QE.B`D(2(["B1F:"T^8FEN;6]D92@I
M.PHD9F@M/G!R:6YT*"1I;F9O+3Y[8G5F?2D["B1F:"T^<V5E:R@P+"`P*3L*
M<F5T=7)N("1F:#L*?0IE;'-E('L*;7D@)&9I;&5N86UE(#T@7W-A=F5?87,H
M(B1I;F9O+3Y[8W)C?2YP;2(L("1I;F9O+3Y[8G5F?2D["F]P96X@;7D@)&9H
M+"`G/#IR87<G+"`D9FEL96YA;64@;W(@9&EE('%Q6T-A;B=T(')E860@(B1F
M:6QE;F%M92(Z("0A73L*<F5T=7)N("1F:#L*?0ID:64@(D)O;W1S=')A<'!I
M;F<@9F%I;&5D.B!C86XG="!F:6YD(&UO9'5L92`D;6]D=6QE(2(["GTL($!)
M3D,I.PIR97%U:7)E(%A33&]A9&5R.PIR97%U:7)E(%!!4CHZ2&5A=GD["G)E
M<75I<F4@0V%R<#HZ2&5A=GD["G)E<75I<F4@17AP;W)T97(Z.DAE879Y.PI0
M05(Z.DAE879Y.CI?:6YI=%]D>6YA;&]A9&5R*"D["G)E<75I<F4@24\Z.D9I
M;&4["G=H:6QE("AM>2`D9FEL96YA;64@/2`H<V]R="!K97ES("5R97%U:7)E
M7VQI<W0I6S!=*2!["G5N;&5S<R`H)$E.0WLD9FEL96YA;65](&]R("1F:6QE
M;F%M92`]?B`O0E-$4$%.+RD@>PII9B`H)&9I;&5N86UE(#U^("]<+G!M8S\D
M+VDI('L*<F5Q=6ER92`D9FEL96YA;64["GT*96QS92!["F1O("1F:6QE;F%M
M92!U;FQE<W,@)&9I;&5N86UE(#U^("]S:71E8W5S=&]M:7IE7"YP;"0O.PI]
M"GT*9&5L971E("1R97%U:7)E7VQI<W1[)&9I;&5N86UE?3L*?0IL87-T('5N
M;&5S<R`D8G5F(&5Q(")02UPP,#-<,#`T(CL*)'-T87)T7W!O<R`]("AT96QL
M(%]&2"D@+2`T.R`@("`@("`@("`@("`@("`C('-T87)T(&]F('II<`I]"FUY
M($!P87)?87)G<SL*;7D@*"1O=70L("1B=6YD;&4L("1L;V=F:"P@)&-A8VAE
M7VYA;64I.PID96QE=&4@)$5.5GM005)?05!07U)%55-%?3L@(R!S86YI=&EZ
M92`H4D554T4@;6%Y(&)E(&$@<V5C=7)I='D@<')O8FQE;2D*)'%U:65T(#T@
M,"!U;FQE<W,@)$5.5GM005)?1$5"54=].PII9B`H(21S=&%R=%]P;W,@;W(@
M*"1!4D=66S!=(&5Q("<M+7!A<BUO<'1I;VYS)R`F)B!S:&EF="DI('L*;7D@
M)61I<W1?8VUD(#T@<7<H"G`@("!B;&EB7W1O7W!A<@II("`@:6YS=&%L;%]P
M87(*=2`@('5N:6YS=&%L;%]P87(*<R`@('-I9VY?<&%R"G8@("!V97)I9GE?
M<&%R"BD["FEF("A`05)'5B!A;F0@)$%21U9;,%T@97$@)RTM<F5U<V4G*2![
M"G-H:69T($!!4D=6.PHD14Y6>U!!4E]!4%!?4D554T5](#T@<VAI9G0@0$%2
M1U8["GT*96QS92![(",@;F]R;6%L('!A<FP@8F5H879I;W5R"FUY($!A9&1?
M=&]?:6YC.PIW:&EL92`H0$%21U8I('L*)$%21U9;,%T@/7X@+UXM*%M!24U/
M0DQB<7!I=7-4=ETI*"XJ*2\@;W(@;&%S=#L*:68@*"0Q(&5Q("=))RD@>PIP
M=7-H($!A9&1?=&]?:6YC+"`D,CL*?0IE;'-I9B`H)#$@97$@)TTG*2!["F5V
M86P@(G5S92`D,B(["GT*96QS:68@*"0Q(&5Q("=!)RD@>PIU;G-H:69T($!P
M87)?87)G<RP@)#(["GT*96QS:68@*"0Q(&5Q("=/)RD@>PHD;W5T(#T@)#([
M"GT*96QS:68@*"0Q(&5Q("=B)RD@>PHD8G5N9&QE(#T@)W-I=&4G.PI]"F5L
M<VEF("@D,2!E<2`G0B<I('L*)&)U;F1L92`]("=A;&PG.PI]"F5L<VEF("@D
M,2!E<2`G<2<I('L*)'%U:65T(#T@,3L*?0IE;'-I9B`H)#$@97$@)TPG*2![
M"F]P96X@)&QO9V9H+"`B/CXB+"`D,B!O<B!D:64@<7%;0V%N)W0@;W!E;B!L
M;V<@9FEL92`B)#(B.B`D(5T["GT*96QS:68@*"0Q(&5Q("=4)RD@>PHD8V%C
M:&5?;F%M92`]("0R.PI]"G-H:69T*$!!4D=6*3L*:68@*&UY("1C;60@/2`D
M9&ES=%]C;61[)#%]*2!["F1E;&5T92`D14Y6>R=005)?5$5-4"=].PII;FET
M7VEN8R@I.PIR97%U:7)E(%!!4CHZ1&ES=#L*)GLB4$%2.CI$:7-T.CHD8VUD
M(GTH*2!U;FQE<W,@0$%21U8["B9[(E!!4CHZ1&ES=#HZ)&-M9")]*"1?*2!F
M;W(@0$%21U8["F5X:70["GT*?0IU;G-H:69T($!)3D,L($!A9&1?=&]?:6YC
M.PI]"GT*:68@*"1O=70I('L*>PIR97%U:7)E($E/.CI&:6QE.PIR97%U:7)E
M($%R8VAI=F4Z.EII<#L*<F5Q=6ER92!$:6=E<W0Z.E-(03L*?0IM>2`D<&%R
M(#T@<VAI9G0H0$%21U8I.PIM>2`D>FEP.PII9B`H9&5F:6YE9"`D<&%R*2![
M"F]P96X@;7D@)&9H+"`G/#IR87<G+"`D<&%R(&]R(&1I92!Q<5M#86XG="!F
M:6YD('!A<B!F:6QE("(D<&%R(CH@)"%=.PIB;&5S<R@D9F@L("=)3SHZ1FEL
M92<I.PHD>FEP(#T@07)C:&EV93HZ6FEP+3YN97<["B@@)'II<"T^<F5A9$9R
M;VU&:6QE2&%N9&QE*"1F:"P@)'!A<BD@/3T@07)C:&EV93HZ6FEP.CI!6E]/
M2R@I("D*;W(@9&EE('%Q6T5R<F]R(')E861I;F<@>FEP(&%R8VAI=F4@(B1P
M87(B73L*?0IM>2`E96YV(#T@9&\@>PII9B`H)'II<"!A;F0@;7D@)&UE=&$@
M/2`D>FEP+3YC;VYT96YT<R@G345402YY;6PG*2D@>PHD;65T82`]?B!S+RXJ
M7G!A<CHD+R]M<SL*)&UE=&$@/7X@<R]>7%,N*B\O;7,["B1M971A(#U^("]>
M("`H6UXZ72LI.B`H+BLI)"]M9SL*?0I].PII9B`H9&5F:6YE9"`D<&%R*2![
M"F]P96X@;7D@)'!H+"`G/#IR87<G+"`D<&%R(&]R(&1I92!Q<5M#86XG="!R
M96%D('!A<B!F:6QE("(D<&%R(CH@)"%=.PIM>2`D8G5F.PIR96%D("1P:"P@
M)&)U9BP@-#L*9&EE('%Q6R(D<&%R(B!I<R!N;W0@82!P87(@9FEL95T@=6YL
M97-S("1B=68@97$@(E!+7#`P,UPP,#0B.PIC;&]S92`D<&@["GT*0W)E871E
M4&%T:"@D;W5T*2`["FUY("1F:"`]($E/.CI&:6QE+3YN97<H"B1O=70L"DE/
M.CI&:6QE.CI/7T-214%4*"D@?"!)3SHZ1FEL93HZ3U]74D].3%DH*2!\($E/
M.CI&:6QE.CI/7U1254Y#*"DL"C`W-S<L"BD@;W(@9&EE('%Q6T-A;B=T(&-R
M96%T92!F:6QE("(D;W5T(CH@)"%=.PHD9F@M/F)I;FUO9&4H*3L*<V5E:R!?
M1D@L(#`L(#`["FUY("1L;V%D97(["FEF("AD969I;F5D("1D871A7W!O<RD@
M>PIR96%D(%]&2"P@)&QO861E<BP@)&1A=&%?<&]S.PI](&5L<V4@>PIL;V-A
M;"`D+R`]('5N9&5F.PHD;&]A9&5R(#T@/%]&2#X["GT*:68@*"$D14Y6>U!!
M4E]615)"051)37T@86YD("1L;V%D97(@/7X@+UXH/SHC(7Q<0')E;2DO*2![
M"G)E<75I<F4@4$%2.CI&:6QT97(Z.E!O9%-T<FEP.PI005(Z.D9I;'1E<CHZ
M4&]D4W1R:7`M/F%P<&QY*%PD;&]A9&5R+"`D,"D["GT*9F]R96%C:"!M>2`D
M:V5Y("AS;W)T(&ME>7,@)65N=BD@>PIM>2`D=F%L(#T@)&5N=GLD:V5Y?2!O
M<B!N97AT.PHD=F%L(#T@979A;"`D=F%L(&EF("1V86P@/7X@+UY;)R)=+SL*
M;7D@)&UA9VEC(#T@(E]?14Y67U!!4E\B("X@=6,H)&ME>2D@+B`B7U\B.PIM
M>2`D<V5T(#T@(E!!4E\B("X@=6,H)&ME>2D@+B`B/21V86PB.PHD;&]A9&5R
M(#U^('-[)&UA9VEC*"`K*7U["B1M86=I8R`N("1S970@+B`H)R`G('@@*&QE
M;F=T:"@D,2D@+2!L96YG=&@H)'-E="DI*0I]96<["GT*)&9H+3YP<FEN="@D
M;&]A9&5R*3L*:68@*"1B=6YD;&4I('L*<F5Q=6ER92!005(Z.DAE879Y.PI0
M05(Z.DAE879Y.CI?:6YI=%]D>6YA;&]A9&5R*"D["FEN:71?:6YC*"D["G)E
M<75I<F5?;6]D=6QE<R@I.PIM>2!`:6YC(#T@9W)E<"![("$O0E-$4$%.+R!]
M"F=R97`@>PHH)&)U;F1L92!N92`G<VET92<I(&]R"B@D7R!N92`D0V]N9FEG
M.CI#;VYF:6=[87)C:&QI8F5X<'T@86YD"B1?(&YE("1#;VYF:6<Z.D-O;F9I
M9WMP<FEV;&EB97AP?2D["GT@0$E.0SL*<WPO*WPO?&<L('-\+R1\?"!F;W)E
M86-H($!I;F,["FUY("5F:6QE<SL*)&9I;&5S>R1??2LK(&9O<B!`1'EN84QO
M861E<CHZ9&Q?<VAA<F5D7V]B:F5C=',L('9A;'5E<R`E24Y#.PIM>2`D;&EB
M7V5X="`]("1#;VYF:6<Z.D-O;F9I9WML:6)?97AT?3L@("`@("`@("`C(%A8
M6"!L:6)?97AT('9S(&1L97AT(#\*;7D@)7=R:71T96X["F9O<F5A8V@@;7D@
M)&ME>2`H<V]R="!K97ES("5F:6QE<RD@>PIM>2`H)&9I;&4L("1N86UE*3L*
M:68@*&1E9FEN960H;7D@)&9C(#T@)%!!4CHZ2&5A=GDZ.D9U;&Q#86-H97LD
M:V5Y?2DI('L**"1F:6QE+"`D;F%M92D@/2`H)&ME>2P@)&9C*3L*?0IE;'-E
M('L*9F]R96%C:"!M>2`D9&ER("A`:6YC*2!["FEF("@D:V5Y(#U^(&U\7EQ1
M)&1I<EQ%+R@N*BDD?&DI('L**"1F:6QE+"`D;F%M92D@/2`H)&ME>2P@)#$I
M.PIL87-T.PI]"FEF("@D:V5Y(#U^(&U\7B]L;V%D97(O6UXO72LO*"XJ*21\
M*2!["FEF("AM>2`D<F5F(#T@)$UO9'5L94-A8VAE>R0Q?2D@>PHH)&9I;&4L
M("1N86UE*2`]("@D<F5F+"`D,2D["FQA<W0["GT*:68@*"UF("(D9&ER+R0Q
M(BD@>PHH)&9I;&4L("1N86UE*2`]("@B)&1I<B\D,2(L("0Q*3L*;&%S=#L*
M?0I]"GT*?0IN97AT('5N;&5S<R!D969I;F5D("1N86UE.PIN97AT(&EF("1W
M<FET=&5N>R1N86UE?2LK.PIN97AT(&EF("%R968H)&9I;&4I(&%N9"`D9FEL
M92`]?B`O7"Y<421L:6)?97AT7$4D+VD["F]U=',H<W!R:6YT9BAQ<5M086-K
M:6YG($9)3$4@(B5S(BXN+ETL(')E9B`D9FEL92`_("1F:6QE+3Y[;F%M97T@
M.B`D9FEL92DI.PIM>2`D8V]N=&5N=#L*:68@*')E9B@D9FEL92DI('L*)&-O
M;G1E;G0@/2`D9FEL92T^>V)U9GT["GT*96QS92!["FQO8V%L("0O(#T@=6YD
M968["F]P96X@;7D@)&9H+"`G/#IR87<G+"`D9FEL92!O<B!D:64@<7%;0V%N
M)W0@<F5A9"`B)&9I;&4B.B`D(5T["B1C;VYT96YT(#T@/"1F:#X["F-L;W-E
M("1F:#L*4$%2.CI&:6QT97(Z.E!O9%-T<FEP+3YA<'!L>2A<)&-O;G1E;G0L
M("(\96UB961D960^+R1N86UE(BD*:68@(21%3E9[4$%27U9%4D)!5$E-?2!A
M;F0@)&YA;64@/7X@+UPN*#\Z<&U\:7A\86PI)"]I.PI005(Z.D9I;'1E<CHZ
M4&%T8VA#;VYT96YT+3YN97<M/F%P<&QY*%PD8V]N=&5N="P@)&9I;&4L("1N
M86UE*3L*?0HD9F@M/G!R:6YT*")&24Q%(BP*<&%C:R@G3B<L(&QE;F=T:"@D
M;F%M92D@*R`Y*2P*<W!R:6YT9B@B)3`X>"\E<R(L($%R8VAI=F4Z.EII<#HZ
M8V]M<'5T94-20S,R*"1C;VYT96YT*2P@)&YA;64I+`IP86-K*"=.)RP@;&5N
M9W1H*"1C;VYT96YT*2DL"B1C;VYT96YT*3L*;W5T<RAQ<5M7<FET=&5N(&%S
M("(D;F%M92)=*3L*?0I]"B1Z:7`M/G=R:71E5&]&:6QE2&%N9&QE*"1F:"D@
M:68@)'II<#L*)&-A8VAE7VYA;64@/2!S=6)S='(@)&-A8VAE7VYA;64L(#`L
M(#0P.PII9B`H(21C86-H95]N86UE(&%N9"!M>2`D;71I;64@/2`H<W1A="@D
M;W5T*2E;.5TI('L*;7D@)&-T>"`]($1I9V5S=#HZ4TA!+3YN97<H,2D["F]P
M96X@;7D@)&9H+"`B/#IR87<B+"`D;W5T.PHD8W1X+3YA9&1F:6QE*"1F:"D[
M"F-L;W-E("1F:#L*)&-A8VAE7VYA;64@/2`D8W1X+3YH97AD:6=E<W0["GT*
M)&-A8VAE7VYA;64@+CT@(EPP(B!X("@T,2`M(&QE;F=T:"`D8V%C:&5?;F%M
M92D["B1C86-H95]N86UE("X](")#04-(12(["B1F:"T^<')I;G0H)&-A8VAE
M7VYA;64I.PHD9F@M/G!R:6YT*'!A8VLH)TXG+"`D9F@M/G1E;&P@+2!L96YG
M=&@H)&QO861E<BDI*3L*)&9H+3YP<FEN="@D4$%27TU!1TE#*3L*)&9H+3YC
M;&]S93L*8VAM;V0@,#<U-2P@)&]U=#L*97AI=#L*?0I["FQA<W0@=6YL97-S
M(&1E9FEN960@)'-T87)T7W!O<SL*7V9I>%]P<F]G;F%M92@I.PIR97%U:7)E
M(%!!4CL*4$%2.CI(96%V>3HZ7VEN:71?9'EN86QO861E<B@I.PI["G)E<75I
M<F4@1FEL93HZ1FEN9#L*<F5Q=6ER92!!<F-H:79E.CI::7`["GT*;7D@)&9H
M(#T@24\Z.D9I;&4M/FYE=SL@("`@("`@("`@("`@("`@("`@("`@("`@("`@
M(",@07)C:&EV93HZ6FEP(&]P97)A=&5S(&]N(&%N($E/.CI(86YD;&4*)&9H
M+3YF9&]P96XH9FEL96YO*%]&2"DL("=R)RD@;W(@9&EE('%Q6V9D;W!E;B@I
M(&9A:6QE9#H@)"%=.PI!<F-H:79E.CI::7`Z.G-E=$-H=6YK4VEZ92@M<R!?
M1D@I.PIM>2`D>FEP(#T@07)C:&EV93HZ6FEP+3YN97<["B@@)'II<"T^<F5A
M9$9R;VU&:6QE2&%N9&QE*"1F:"P@)'!R;V=N86UE*2`]/2!!<F-H:79E.CI:
M:7`Z.D%:7T]+*"D@*0IO<B!D:64@<7%;17)R;W(@<F5A9&EN9R!Z:7`@87)C
M:&EV92`B)'!R;V=N86UE(ET["D%R8VAI=F4Z.EII<#HZ<V5T0VAU;FM3:7IE
M*#8T("H@,3`R-"D["G!U<V@@0%!!4CHZ3&EB0V%C:&4L("1Z:7`["B1005(Z
M.DQI8D-A8VAE>R1P<F]G;F%M97T@/2`D>FEP.PHD<75I970@/2`A)$5.5GM0
M05)?1$5"54=].PIO=71S*'%Q6UPD14Y6>U!!4E]414U0?2`]("(D14Y6>U!!
M4E]414U0?2)=*3L*:68@*&1E9FEN960@)$5.5GM005)?5$5-4'TI('L@(R!S
M:&]U;&0@8F4@<V5T(&%T('1H:7,@<&]I;G0A"F9O<F5A8V@@;7D@)&UE;6)E
M<B`H("1Z:7`M/FUE;6)E<G,@*2!["FYE>'0@:68@)&UE;6)E<BT^:7-$:7)E
M8W1O<GD["FUY("1M96UB97)?;F%M92`]("1M96UB97(M/F9I;&5.86UE.PIN
M97AT('5N;&5S<R`D;65M8F5R7VYA;64@/7X@;7L*7@HO/W-H;&EB+PHH/SHD
M0V]N9FEG.CI#;VYF:6=[=F5R<VEO;GTO*3\**#\Z)$-O;F9I9SHZ0V]N9FEG
M>V%R8VAN86UE?2\I/PHH6UXO72LI"B0*?7@["FUY("1E>'1R86-T7VYA;64@
M/2`D,3L*;7D@)&1E<W1?;F%M92`]($9I;&4Z.E-P96,M/F-A=&9I;&4H)$5.
M5GM005)?5$5-4'TL("1E>'1R86-T7VYA;64I.PII9B`H+68@)&1E<W1?;F%M
M92`F)B`M<R!?(#T]("1M96UB97(M/G5N8V]M<')E<W-E9%-I>F4H*2D@>PIO
M=71S*'%Q6U-K:7!P:6YG("(D;65M8F5R7VYA;64B('-I;F-E(&ET(&%L<F5A
M9'D@97AI<W1S(&%T("(D9&5S=%]N86UE(ETI.PI](&5L<V4@>PIO=71S*'%Q
M6T5X=')A8W1I;F<@(B1M96UB97)?;F%M92(@=&\@(B1D97-T7VYA;64B72D[
M"B1M96UB97(M/F5X=')A8W14;T9I;&5.86UE9"@D9&5S=%]N86UE*3L*8VAM
M;V0H,#4U-2P@)&1E<W1?;F%M92D@:68@)%Y/(&5Q(")H<'5X(CL*?0I]"GT*
M?0IU;FQE<W,@*"1005(Z.DQI8D-A8VAE>R1P<F]G;F%M97TI('L*9&EE(#P\
M("(N(B!U;FQE<W,@0$%21U8["E5S86=E.B`D,"!;("U!;&EB+G!A<B!=(%L@
M+4ED:7(@72!;("U-;6]D=6QE(%T@6R!S<F,N<&%R(%T@6R!P<F]G<F%M+G!L
M(%T*)#`@6R`M0GPM8B!=(%LM3V]U=&9I;&5=('-R8RYP87(*+@HD14Y6>U!!
M4E]04D]'3D%-17T@/2`D<')O9VYA;64@/2`D,"`]('-H:69T*$!!4D=6*3L*
M?0IS=6(@0W)E871E4&%T:"!["FUY("@D;F%M92D@/2!`7SL*<F5Q=6ER92!&
M:6QE.CI"87-E;F%M93L*;7D@*"1B87-E;F%M92P@)'!A=&@L("1E>'0I(#T@
M1FEL93HZ0F%S96YA;64Z.F9I;&5P87)S92@D;F%M92P@*"=<+BXJ)RDI.PIR
M97%U:7)E($9I;&4Z.E!A=&@["D9I;&4Z.E!A=&@Z.FUK<&%T:"@D<&%T:"D@
M=6YL97-S*"UE("1P871H*3L@(R!M:W!A=&@@9&EE<R!W:71H(&5R<F]R"GT*
M<W5B(')E<75I<F5?;6]D=6QE<R!["G)E<75I<F4@;&EB.PIR97%U:7)E($1Y
M;F%,;V%D97(["G)E<75I<F4@:6YT96=E<CL*<F5Q=6ER92!S=')I8W0["G)E
M<75I<F4@=V%R;FEN9W,["G)E<75I<F4@=F%R<SL*<F5Q=6ER92!#87)P.PIR
M97%U:7)E($-A<G`Z.DAE879Y.PIR97%U:7)E($5R<FYO.PIR97%U:7)E($5X
M<&]R=&5R.CI(96%V>3L*<F5Q=6ER92!%>'!O<G1E<CL*<F5Q=6ER92!&8VYT
M;#L*<F5Q=6ER92!&:6QE.CI496UP.PIR97%U:7)E($9I;&4Z.E-P96,["G)E
M<75I<F4@6%-,;V%D97(["G)E<75I<F4@0V]N9FEG.PIR97%U:7)E($E/.CI(
M86YD;&4["G)E<75I<F4@24\Z.D9I;&4["G)E<75I<F4@0V]M<')E<W,Z.EIL
M:6(["G)E<75I<F4@07)C:&EV93HZ6FEP.PIR97%U:7)E($1I9V5S=#HZ4TA!
M.PIR97%U:7)E(%!!4CL*<F5Q=6ER92!005(Z.DAE879Y.PIR97%U:7)E(%!!
M4CHZ1&ES=#L*<F5Q=6ER92!005(Z.D9I;'1E<CHZ4&]D4W1R:7`["G)E<75I
M<F4@4$%2.CI&:6QT97(Z.E!A=&-H0V]N=&5N=#L*<F5Q=6ER92!A='1R:6)U
M=&5S.PIE=F%L('L@<F5Q=6ER92!#=V0@?3L*979A;"![(')E<75I<F4@5VEN
M,S(@?3L*979A;"![(')E<75I<F4@4V-A;&%R.CI5=&EL('T["F5V86P@>R!R
M97%U:7)E($%R8VAI=F4Z.E5N>FEP.CI"=7)S="!].PIE=F%L('L@<F5Q=6ER
M92!4:64Z.DAA<V@Z.DYA;65D0V%P='5R92!].PIE=F%L('L@<F5Q=6ER92!0
M97)L24\[(')E<75I<F4@4&5R;$E/.CIS8V%L87(@?3L*979A;"![(')E<75I
M<F4@=71F."!].PI]"G-U8B!?<V5T7W!A<E]T96UP('L*:68@*&1E9FEN960@
M)$5.5GM005)?5$5-4'T@86YD("1%3E9[4$%27U1%35!](#U^("\H+BLI+RD@
M>PHD<&%R7W1E;7`@/2`D,3L*<F5T=7)N.PI]"F9O<F5A8V@@;7D@)'!A=&@@
M*`HH;6%P("1%3E9[)%]]+"!Q=R@@4$%27U1-4$1)4B!435!$25(@5$5-4$1)
M4B!414U0(%1-4"`I*2P*<7<H($,Z7%Q414U0("]T;7`@+B`I"BD@>PIN97AT
M('5N;&5S<R!D969I;F5D("1P871H(&%N9"`M9"`D<&%T:"!A;F0@+7<@)'!A
M=&@["FUY("1U<V5R;F%M93L*;7D@)'!W=6ED.PIE=F%L('LH)'!W=6ED*2`]
M(&=E='!W=6ED*"0^*2!I9B!D969I;F5D("0^.WT["FEF("@@9&5F:6YE9"@F
M5VEN,S(Z.DQO9VEN3F%M92D@*2!["B1U<V5R;F%M92`]("97:6XS,CHZ3&]G
M:6Y.86UE.PI]"F5L<VEF("AD969I;F5D("1P=W5I9"D@>PHD=7-E<FYA;64@
M/2`D<'=U:60["GT*96QS92!["B1U<V5R;F%M92`]("1%3E9[55-%4DY!345]
M('Q\("1%3E9[55-%4GT@?'P@)U-94U1%32<["GT*)'5S97)N86UE(#U^(',O
M7%<O7R]G.PIM>2`D<W1M<&1I<B`]("(D<&%T:"1#;VYF:6=[7V1E;&EM?7!A
M<BTB+G5N<&%C:R@B2"HB+"`D=7-E<FYA;64I.PIM:V1I<B`D<W1M<&1I<BP@
M,#<U-3L*:68@*"$D14Y6>U!!4E]#3$5!3GT@86YD(&UY("1M=&EM92`]("AS
M=&%T*"1P<F]G;F%M92DI6SE=*2!["F]P96X@;7D@)&9H+"`B/#IR87<B+"`D
M<')O9VYA;64@;W(@9&EE('%Q6T-A;B=T(')E860@(B1P<F]G;F%M92(Z("0A
M73L*<V5E:R`D9F@L("TQ."P@,CL*;7D@)&)U9CL*<F5A9"`D9F@L("1B=68L
M(#8["FEF("@D8G5F(&5Q(")<,$-!0TA%(BD@>PIS965K("1F:"P@+34X+"`R
M.PIR96%D("1F:"P@)&)U9BP@-#$["B1B=68@/7X@<R]<,"\O9SL*)'-T;7!D
M:7(@+CT@(B1#;VYF:6=[7V1E;&EM?6-A8VAE+21B=68B.PI]"F5L<V4@>PIM
M>2`D9&EG97-T(#T@979A;`I["G)E<75I<F4@1&EG97-T.CI32$$["FUY("1C
M='@@/2!$:6=E<W0Z.E-(02T^;F5W*#$I.PIO<&5N(&UY("1F:"P@(CPZ<F%W
M(BP@)'!R;V=N86UE(&]R(&1I92!Q<5M#86XG="!R96%D("(D<')O9VYA;64B
M.B`D(5T["B1C='@M/F%D9&9I;&4H)&9H*3L*8VQO<V4H)&9H*3L*)&-T>"T^
M:&5X9&EG97-T.PI]('Q\("1M=&EM93L*)'-T;7!D:7(@+CT@(B1#;VYF:6=[
M7V1E;&EM?6-A8VAE+21D:6=E<W0B.PI]"F-L;W-E*"1F:"D["GT*96QS92![
M"B1%3E9[4$%27T-,14%.?2`](#$["B1S=&UP9&ER("X]("(D0V]N9FEG>U]D
M96QI;7UT96UP+20D(CL*?0HD14Y6>U!!4E]414U0?2`]("1S=&UP9&ER.PIM
M:V1I<B`D<W1M<&1I<BP@,#<U-3L*;&%S=#L*?0HD<&%R7W1E;7`@/2`D,2!I
M9B`D14Y6>U!!4E]414U0?2!A;F0@)$5.5GM005)?5$5-4'T@/7X@+R@N*RDO
M.PI]"G-U8B!?<V%V95]A<R!["FUY("@D;F%M92P@)&-O;G1E;G1S+"`D;6]D
M92D@/2!`7SL*;7D@)&9U;&QN86UE(#T@(B1P87)?=&5M<"\D;F%M92(["G5N
M;&5S<R`H+64@)&9U;&QN86UE*2!["FUY("1T96UP;F%M92`]("(D9G5L;&YA
M;64N)"0B.PIO<&5N(&UY("1F:"P@)SXZ<F%W)RP@)'1E;7!N86UE(&]R(&1I
M92!Q<5M#86XG="!W<FET92`B)'1E;7!N86UE(CH@)"%=.PIP<FEN="`D9F@@
M)&-O;G1E;G1S.PIC;&]S92`D9F@["F-H;6]D("1M;V1E+"`D=&5M<&YA;64@
M:68@9&5F:6YE9"`D;6]D93L*<F5N86UE*"1T96UP;F%M92P@)&9U;&QN86UE
M*2!O<B!U;FQI;FLH)'1E;7!N86UE*3L*?0IR971U<FX@)&9U;&QN86UE.PI]
M"G-U8B!?<V5T7W!R;V=N86UE('L*:68@*&1E9FEN960@)$5.5GM005)?4%)/
M1TY!345](&%N9"`D14Y6>U!!4E]04D]'3D%-17T@/7X@+R@N*RDO*2!["B1P
M<F]G;F%M92`]("0Q.PI]"B1P<F]G;F%M92!\?#T@)#`["FEF("@D14Y6>U!!
M4E]414U0?2!A;F0@:6YD97@H)'!R;V=N86UE+"`D14Y6>U!!4E]414U0?2D@
M/CT@,"D@>PHD<')O9VYA;64@/2!S=6)S='(H)'!R;V=N86UE+"!R:6YD97@H
M)'!R;V=N86UE+"`D0V]N9FEG>U]D96QI;7TI("L@,2D["GT*:68@*"$D14Y6
M>U!!4E]04D]'3D%-17T@;W(@:6YD97@H)'!R;V=N86UE+"`D0V]N9FEG>U]D
M96QI;7TI(#X](#`I('L*:68@*&]P96X@;7D@)&9H+"`G/"<L("1P<F]G;F%M
M92D@>PIR971U<FX@:68@+7,@)&9H.PI]"FEF("@M<R`B)'!R;V=N86UE)$-O
M;F9I9WM?97AE?2(I('L*)'!R;V=N86UE("X]("1#;VYF:6=[7V5X97T["G)E
M='5R;CL*?0I]"F9O<F5A8V@@;7D@)&1I<B`H<W!L:70@+UQ1)$-O;F9I9WMP
M871H7W-E<'U<12\L("1%3E9[4$%42'TI('L*;F5X="!I9B!E>&ES=',@)$5.
M5GM005)?5$5-4'T@86YD("1D:7(@97$@)$5.5GM005)?5$5-4'T["B1D:7(@
M/7X@<R]<421#;VYF:6=[7V1E;&EM?5Q%)"\O.PHH*"1P<F]G;F%M92`]("(D
M9&ER)$-O;F9I9WM?9&5L:6U])'!R;V=N86UE)$-O;F9I9WM?97AE?2(I+"!L
M87-T*0II9B`M<R`B)&1I<B1#;VYF:6=[7V1E;&EM?21P<F]G;F%M921#;VYF
M:6=[7V5X97TB.PHH*"1P<F]G;F%M92`]("(D9&ER)$-O;F9I9WM?9&5L:6U]
M)'!R;V=N86UE(BDL(&QA<W0I"FEF("US("(D9&ER)$-O;F9I9WM?9&5L:6U]
M)'!R;V=N86UE(CL*?0I]"G-U8B!?9FEX7W!R;V=N86UE('L*)#`@/2`D<')O
M9VYA;64@?'P]("1%3E9[4$%27U!23T=.04U%?3L*:68@*&EN9&5X*"1P<F]G
M;F%M92P@)$-O;F9I9WM?9&5L:6U]*2`\(#`I('L*)'!R;V=N86UE(#T@(BXD
M0V]N9FEG>U]D96QI;7TD<')O9VYA;64B.PI]"FUY("1P=V0@/2`H9&5F:6YE
M9"`F0W=D.CIG971C=V0I(#\@0W=D.CIG971C=V0H*0HZ("@H9&5F:6YE9"`F
M5VEN,S(Z.D=E=$-W9"D@/R!7:6XS,CHZ1V5T0W=D*"D@.B!@<'=D8"D["F-H
M;VUP*"1P=V0I.PHD<')O9VYA;64@/7X@<R]>*#\]7"Y<+C]<421#;VYF:6=[
M7V1E;&EM?5Q%*2\D<'=D)$-O;F9I9WM?9&5L:6U]+SL*)$5.5GM005)?4%)/
M1TY!345](#T@)'!R;V=N86UE.PI]"G-U8B!?<&%R7VEN:71?96YV('L*:68@
M*"`D14Y6>U!!4E])3DE424%,25I%1'TK*R`]/2`Q("D@>PIR971U<FX["GT@
M96QS92!["B1%3E9[4$%27TE.251)04Q)6D5$?2`](#(["GT*9F]R("AQ=R@@
M4U!!5TY%1"!414U0($-,14%.($1%0E5'($-!0TA%(%!23T=.04U%("D@*2![
M"F1E;&5T92`D14Y6>R=005)?)RXD7WT["GT*9F]R("AQ=R\@5$U01$E2(%1%
M35`@0TQ%04X@1$5"54<@+RD@>PHD14Y6>R=005)?)RXD7WT@/2`D14Y6>R=0
M05)?1TQ/0D%,7R<N)%]](&EF(&5X:7-T<R`D14Y6>R=005)?1TQ/0D%,7R<N
M)%]].PI]"FUY("1P87)?8VQE86X@/2`B7U]%3E9?4$%27T-,14%.7U\@("`@
M("`@("`@("`@("`B.PII9B`H)$5.5GM005)?5$5-4'TI('L*9&5L971E("1%
M3E9[4$%27T-,14%.?3L*?0IE;'-I9B`H(65X:7-T<R`D14Y6>U!!4E]'3$]"
M04Q?0TQ%04Y]*2!["FUY("1V86QU92`]('-U8G-T<B@D<&%R7V-L96%N+"`Q
M,B`K(&QE;F=T:"@B0TQ%04XB*2D["B1%3E9[4$%27T-,14%.?2`]("0Q(&EF
M("1V86QU92`]?B`O7E!!4E]#3$5!3CTH7%,K*2\["GT*?0IS=6(@;W5T<R![
M"G)E='5R;B!I9B`D<75I970["FEF("@D;&]G9F@I('L*<')I;G0@)&QO9V9H
M(")`7UQN(CL*?0IE;'-E('L*<')I;G0@(D!?7&XB.PI]"GT*<W5B(&EN:71?
M:6YC('L*<F5Q=6ER92!#;VYF:6<["G!U<V@@0$E.0RP@9W)E<"!D969I;F5D
M+"!M87`@)$-O;F9I9SHZ0V]N9FEG>R1??2P@<7<H"F%R8VAL:6)E>'`@<')I
M=FQI8F5X<"!S:71E87)C:&5X<"!S:71E;&EB97AP"G9E;F1O<F%R8VAE>'`@
M=F5N9&]R;&EB97AP"BD["GT*<&%C:V%G92!M86EN.PIR97%U:7)E(%!!4CL*
M=6YS:&EF="!`24Y#+"!<)E!!4CHZ9FEN9%]P87(["E!!4BT^:6UP;W)T*$!P
M87)?87)G<RD["F1I92!Q<5MP87(N<&PZ($-A;B=T(&]P96X@<&5R;"!S8W)I
M<'0@(B1P<F]G;F%M92(Z($YO('-U8V@@9FEL92!O<B!D:7)E8W1O<GE<;ET*
M=6YL97-S("UE("1P<F]G;F%M93L*9&\@)'!R;V=N86UE.PI#3U)%.CIE>&ET
M*"0Q*2!I9B`H)$`@/7XO7E]42U]%6$E47UPH*%QD*RE<*2\I.PID:64@)$`@
M:68@)$`["GT["B0Z.E]?15)23U(@/2`D0"!I9B`D0#L*?0I#3U)%.CIE>&ET
M*"0Q*2!I9B`H)#HZ7U]%4E)/4B`]?B]>7U1+7T58251?7"@H7&0K*5PI+RD[
M"F1I92`D.CI?7T524D]2(&EF("0Z.E]?15)23U(["C$["E]?14Y$7U\*``!P
M97)L>'-I+F,`````````7U]%3E9?4$%27T-,14%.7U\@("`@("`@("`@("`@
M("`!&P,[]````!T```"0@?__*`$``+"!___4!0``T(/__Q`!``#0A/__4`$`
M`*"%__^<`0``((?__^`!``"PA___!`(``)"(__]``@``X(K__XP"``!0B___
MI`(``)"+__^\`@``8(S___`"``!PC/__!`,``'".__]D`P``H([__W@#``"`
MC___J`,``""4___X`P``H)3__Q0$``#`E/__+`0``-"5__^(!```$)[__UP%
M``#`H/__D`4``-"A__\`!@``$+C__U`&``!PN/__:`8``""Z__^T!@``$+S_
M__@&``"`O/__0`<``)"\__]8!P```````!0``````````7I2``%X$`$;#`<(
MD`$``!0````<````N(+__R\`````1`<0`````"0````T````8(#__R``````
M#A!&#AA*#PMW"(``/QH[*C,D(@````!(````7````'B#___-`````$(.$(\"
M0@X8C@-"#B"-!$(.*(P%00XPA@9!#CB#!T0.0`)M"@XX1`XP00XH0@X@0@X8
M0@X00@X(2`L`0````*@```#\@___?0$```!"#A".`D(.&(T#0@X@C`1!#BB&
M!4$.,(,&1P[0`7X*#C!!#BA!#B!"#AA"#A!"#@A+"P`@````[````#B%__^$
M`````$$.$(8"2@ZP`0)Q"@X000X(00LX````$`$``*2%___>`````$(.$(X"
M1PX8C0-"#B",!$0.*(8%40[`(`*E"@XH00X@0@X80@X00@X(1`M(````3`$`
M`$B&__]"`@```$(.$(\"0@X8C@-"#B"-!$(.*(P%00XPA@9$#CB#!T<.4`,=
M`0H..$$.,$$.*$(.($(.&$(.$$(."$4+%````)@!``!,B/__;@````!$#A`"
M9`X(%````+`!``"DB/__,@````!$#A!7#@@`,````,@!``#,B/__Q0````!"
M#A",`D$.&(8#00X@@P1'#L!``JP*#B!$#AA!#A!"#@A!"Q````#\`0``:(G_
M_P4`````````7````!`"``!DB?__]0$```!"#A"/`D(.&(X#10X@C01"#BB,
M!4$.,(8&2PXX@P='#H`B`F4*#CA!#C!!#BA"#B!"#AA"#A!"#@A#"P*G#H@B
M3@Z0(EP.B")!#H`B````$````'`"```$B___)0`````````L````A`(``""+
M___=`````$H.$(8"`F(*Q@X(3`MZ"@X(1@M("L8."$8+2@X(Q@!,````M`(`
M`-"+__^?!````$(.$(\"0@X8C@-"#B"-!$(.*(P%00XPA@9!#CB#!TL.@`$#
M,P0*#CA!#C!!#BA"#B!"#AA"#A!"#@A!"P```!@````$`P``()#__W4`````
M1`X@;0H."$<+```4````(`,``(20__\?`````$0.$%H."`!8````.`,``(R0
M__\0`0```$(.$(T"0@X8C`-(#B"&!$<.*(,%1`Y``D(*#BA&#B!!#AA"#A!"
M#@A,"V4*#BA!#B!!#AA"#A!"#@A%"V0.2$D.4&0.2$$.0````-````"4`P``
M0)'__S$(````0@X0CP)"#AB.`TD.((T$0@XHC`5!#C"&!D$..(,'1P[`@P(#
MN`$.R(,"90[0@P)?#LB#`D(.P(,"`GD*#CA$#C!!#BA"#B!"#AA"#A!"#@A$
M"P.!`0[(@P)+#M"#`E\.R(,"00[`@P)<#LB#`D0.T(,"4`[8@P)+#N"#`DP.
MP(,"`PL!"@XX1`XP00XH0@X@0@X80@X00@X(30L#/0$.R(,"1`[0@P)I#LB#
M`D$.P(,"`P(!"@[(@P)!#M"#`D4+````,````&@$``"LF/__J0(```!"#A"-
M`D(.&(P#00X@A@1$#C`#Y@$*#B!!#AA"#A!"#@A$"T````"<!```*)O__P<!
M````0@X0C`)$#AB&`T$.((,$1`XP`FL*#B!!#AA!#A!"#@A&"W0*#B!$#AA!
M#A!"#@A%"P``*````.`$``#4>___%@(```!"#A",`D$.&(8#1`Y0`Z@!"@X8
M1`X00@X(00M,````#`4``,B;__\Q%@```$(.$(\"20X8C@-"#B"-!$(.*(P%
M00XPA@9!#CB#!T<.T`4#"!8*#CA!#C!!#BA"#B!"#AA"#A!"#@A!"P```!0`
M``!<!0``N+'__U4`````1`X0`E`."$@```!T!0```++__Z<!````0@X0CP)%
M#AB.`T4.((T$0@XHC`5$#C"&!D4..(,'1PY``N4*#CA!#C!!#BA"#B!"#AA"
M#A!"#@A#"P!`````P`4``&2S___K`0```$(.$(X"1@X8C0-"#B",!$$.*(8%
M1`XP@P8#-@$*#BA!#B!"#AA"#A!"#@A,"P```````$0````$!@``$+7__V4`
M````1@X0CP))#AB.`T4.((T$10XHC`5$#C"&!D@..(,'1PY`;@XX00XP00XH
M0@X@0@X80@X00@X(`!0```!,!@``.+7__P4``````````````!````!D!@``
M,+7__Q``````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````4",`````````(P````````$`````````Y0(````````!`````````/`"
M`````````0```````````P````````\`````````E@0````````,````````
M```@````````#0`````````P6P```````!D`````````P+L````````;````
M``````@`````````&@````````#(NP```````!P`````````"`````````#U
M_O]O`````,`#````````!0````````"`#@````````8`````````:`4`````
M```*`````````+D$````````"P`````````8`````````!4`````````````
M```````#`````````/"]`````````@`````````8`````````!0`````````
M!P`````````7`````````"`;````````!P````````"0%`````````@`````
M````D`8````````)`````````!@`````````&````````````````````/O_
M_V\``````0``"`````#^__]O```````4````````____;P`````"````````
M`/#__V\`````.A,```````#Y__]O``````0`````````````````````````
M````````````````````````````````````````````````````````````
M``````````````````````````````#0NP``````````````````````````
M```V(```````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M``````````````````````````C`````````_____P````#@8@```````$=#
M0SH@*$=.52D@,3$N,2XP```N<VAS=')T86(`+FEN=&5R<``N;F]T92YG;G4N
M<')O<&5R='D`+FYO=&4N9VYU+F)U:6QD+6ED`"YN;W1E+D%"22UT86<`+F=N
M=2YH87-H`"YD>6YS>6T`+F1Y;G-T<@`N9VYU+G9E<G-I;VX`+F=N=2YV97)S
M:6]N7W(`+G)E;&$N9'EN`"YR96QA+G!L=``N:6YI=``N=&5X=``N9FEN:0`N
M<F]D871A`"YE:%]F<F%M95]H9'(`+F5H7V9R86UE`"YI;FET7V%R<F%Y`"YF
M:6YI7V%R<F%Y`"YD>6YA;6EC`"YG;W0`+F1A=&$`+F)S<P`N8V]M;65N=```
M````````````````````````````````````````````````````````````
M``````````````````````````L````!`````@`````````8`P```````!@#
M````````'`````````````````````$````````````````````3````!P``
M``(`````````.`,````````X`P```````$`````````````````````(````
M````````````````)@````<````"`````````'@#````````>`,````````D
M````````````````````!````````````````````#D````'`````@``````
M``"<`P```````)P#````````(`````````````````````0`````````````
M``````!'````]O__;P(`````````P`,```````#``P```````*@!````````
M!@`````````(````````````````````40````L````"`````````&@%````
M````:`4````````8"0````````<````!````"``````````8`````````%D`
M```#`````@````````"`#@```````(`.````````N00`````````````````
M``$```````````````````!A````____;P(`````````.A,````````Z$P``
M`````,(`````````!@`````````"``````````(`````````;@```/[__V\"
M```````````4`````````!0```````"0``````````<````"````"```````
M`````````````'T````$`````@````````"0%````````)`4````````D`8`
M```````&``````````@`````````&`````````"'````!````$(`````````
M(!L````````@&P```````!@`````````!@```!8````(`````````!@`````
M````D0````$````&```````````@`````````"`````````;````````````
M````````!````````````````````(P````!````!@`````````@(```````
M`"`@````````(````````````````````!``````````$`````````"7````
M`0````8`````````0"````````!`(````````/`Z```````````````````0
M````````````````````G0````$````&`````````#!;````````,%L`````
M```-````````````````````!````````````````````*,````!`````@``
M````````8`````````!@````````D#X``````````````````!``````````
M``````````"K`````0````(`````````D)X```````"0G@```````/0`````
M```````````````$````````````````````N0````$````"`````````(B?
M````````B)\```````!X!@``````````````````"```````````````````
M`,,````.`````P````````#`NP```````,"K````````"```````````````
M``````@`````````"`````````#/````#P````,`````````R+L```````#(
MJP````````@````````````````````(``````````@`````````VP````8`
M```#`````````-"[````````T*L````````@`@````````<`````````"```
M```````0`````````.0````!`````P````````#PO0```````/"M````````
M$`(```````````````````@`````````"`````````#I`````0````,`````
M`````,``````````L````````"`````````````````````(````````````
M````````[P````@````#`````````"#`````````(+````````"`$```````
M````````````(````````````````````/0````!````,```````````````
M`````""P````````$@````````````````````$``````````0`````````!
M`````P`````````````````````````RL````````/T`````````````````
2```!````````````````````
