"""Check if a host is in the Google Chrome HSTS Preload list"""

import functools
import os
import typing

__version__ = "2021.11.1"
__checksum__ = "58a16db00e7ca9ac6144d0cb5e3eb96363913923ec653d382d4f2f474e47e626"
__all__ = ["in_hsts_preload"]

# fmt: off
_GTLD_INCLUDE_SUBDOMAINS = {b'android', b'app', b'bank', b'boo', b'chrome', b'dad', b'day', b'dev', b'eat', b'esq', b'fly', b'foo', b'gle', b'gmail', b'google', b'hangout', b'ing', b'insurance', b'meet', b'meme', b'new', b'nexus', b'page', b'phd', b'play', b'prof', b'rsvp', b'search', b'youtube'}  # noqa: E501
_JUMPTABLE = [[(0, 11), (11, 10), (21, 9), (30, 61), (91, 26), (117, 12), None, (129, 19), (148, 22), (170, 7), (177, 20), (197, 18), None, (215, 29), (244, 45), (289, 7), (296, 9), (305, 36), (341, 16), (357, 10), (367, 28), None, (395, 54), (449, 8), (457, 18), (475, 19), (494, 13), (507, 14), (521, 14), None, None, (535, 33), (568, 20), (588, 35), (623, 14), (637, 24), (661, 9), (670, 9), (679, 25), (704, 27), (731, 8), (739, 13), None, None, (752, 17), (769, 6), (775, 26), (801, 10), (811, 5), (816, 19), (835, 14), (849, 11), (860, 12), (872, 27), None, (899, 11), (910, 11), (921, 7), (928, 29), (957, 18), (975, 27), (1002, 46), (1048, 25), (1073, 16), (1089, 8), (1097, 5), (1102, 22), (1124, 18), None, (1142, 36), (1178, 15), (1193, 8), (1201, 11), None, (1212, 5), (1217, 16), (1233, 14), (1247, 18), None, (1265, 20), (1285, 18), (1303, 48), (1351, 19), (1370, 12), (1382, 59), (1441, 14), (1455, 14), (1469, 20), None, (1489, 10), (1499, 13), (1512, 20), (1532, 19), None, (1551, 13), (1564, 19), (1583, 11), (1594, 4), (1598, 22), (1620, 10), (1630, 7), (1637, 14), (1651, 21), (1672, 11), (1683, 21), (1704, 12), (1716, 32), None, (1748, 10), (1758, 14), (1772, 19), (1791, 45), (1836, 15), None, (1851, 11), (1862, 23), (1885, 21), (1906, 26), (1932, 6), (1938, 6), (1944, 7), (1951, 5), (1956, 20), (1976, 23), (1999, 24), (2023, 13), (2036, 15), (2051, 19), (2070, 12), (2082, 61), (2143, 55), (2198, 12), (2210, 23), (2233, 16), (2249, 38), (2287, 6), (2293, 12), (2305, 44), (2349, 6), (2355, 41), (2396, 13), (2409, 23), (2432, 36), (2468, 20), (2488, 8), (2496, 15), (2511, 12), (2523, 19), (2542, 25), (2567, 15), None, (2582, 35), (2617, 21), (2638, 17), (2655, 19), (2674, 26), (2700, 5), (2705, 37), (2742, 30), (2772, 16), (2788, 10), (2798, 17), (2815, 23), (2838, 14), (2852, 17), (2869, 8), (2877, 8), (2885, 7), (2892, 29), (2921, 6), (2927, 18), (2945, 32), (2977, 20), (2997, 17), (3014, 24), (3038, 12), (3050, 40), (3090, 40), (3130, 12), (3142, 48), (3190, 25), (3215, 17), None, (3232, 8), (3240, 25), (3265, 19), (3284, 6), (3290, 23), None, (3313, 30), (3343, 33), (3376, 14), (3390, 16), (3406, 27), None, (3433, 30), (3463, 41), (3504, 50), (3554, 15), (3569, 20), (3589, 25), (3614, 21), (3635, 32), (3667, 24), (3691, 20), (3711, 17), (3728, 60), (3788, 19), (3807, 9), (3816, 12), (3828, 12), (3840, 11), (3851, 10), (3861, 48), (3909, 42), None, (3951, 25), (3976, 28), None, (4004, 8), (4012, 8), (4020, 7), None, (4027, 25), (4052, 17), None, (4069, 21), (4090, 35), (4125, 21), (4146, 10), (4156, 41), (4197, 20), (4217, 31), (4248, 23), (4271, 19), (4290, 12), (4302, 5), (4307, 30), (4337, 29), (4366, 14), (4380, 14), (4394, 47), (4441, 52), None, None, (4493, 51), (4544, 42), None, (4586, 14), None, (4600, 15), (4615, 8), (4623, 21), (4644, 6), (4650, 16), (4666, 17)], [(4683, 8543), (13226, 9348), (22574, 9490), (32064, 8270), (40334, 8576), (48910, 8306), (57216, 9146), (66362, 8206), (74568, 9262), (83830, 8458), (92288, 9602), (101890, 8424), (110314, 8987), (119301, 10226), (129527, 8664), (138191, 9134), (147325, 9465), (156790, 8801), (165591, 8771), (174362, 8311), (182673, 9363), (192036, 8993), (201029, 9063), (210092, 8593), (218685, 9066), (227751, 8664), (236415, 9162), (245577, 9171), (254748, 8246), (262994, 8882), (271876, 9217), (281093, 8637), (289730, 8781), (298511, 8943), (307454, 7994), (315448, 8924), (324372, 8893), (333265, 9666), (342931, 9138), (352069, 9096), (361165, 9692), (370857, 8384), (379241, 8426), (387667, 8576), (396243, 8776), (405019, 8762), (413781, 8833), (422614, 9852), (432466, 8632), (441098, 8151), (449249, 8648), (457897, 9118), (467015, 9032), (476047, 9046), (485093, 9291), (494384, 8810), (503194, 9122), (512316, 8874), (521190, 9100), (530290, 7734), (538024, 8848), (546872, 9017), (555889, 8712), (564601, 9126), (573727, 9058), (582785, 9246), (592031, 8499), (600530, 9610), (610140, 9330), (619470, 8873), (628343, 8994), (637337, 8466), (645803, 7871), (653674, 9260), (662934, 8968), (671902, 9514), (681416, 8181), (689597, 9492), (699089, 9005), (708094, 8461), (716555, 9098), (725653, 7785), (733438, 8553), (741991, 9158), (751149, 8549), (759698, 9083), (768781, 9605), (778386, 8792), (787178, 9121), (796299, 9021), (805320, 9706), (815026, 8301), (823327, 8895), (832222, 8841), (841063, 8798), (849861, 9444), (859305, 9146), (868451, 8728), (877179, 8618), (885797, 8304), (894101, 8442), (902543, 9077), (911620, 8794), (920414, 8642), (929056, 8607), (937663, 9452), (947115, 9343), (956458, 9357), (965815, 9976), (975791, 9194), (984985, 9244), (994229, 9235), (1003464, 8857), (1012321, 8666), (1020987, 9141), (1030128, 8957), (1039085, 8784), (1047869, 8884), (1056753, 8647), (1065400, 9615), (1075015, 9482), (1084497, 9317), (1093814, 9035), (1102849, 9186), (1112035, 9898), (1121933, 8657), (1130590, 8299), (1138889, 9420), (1148309, 8892), (1157201, 10422), (1167623, 9496), (1177119, 8554), (1185673, 9093), (1194766, 8664), (1203430, 8695), (1212125, 9215), (1221340, 8318), (1229658, 9583), (1239241, 8561), (1247802, 8676), (1256478, 9145), (1265623, 9301), (1274924, 8214), (1283138, 8531), (1291669, 9387), (1301056, 8598), (1309654, 8668), (1318322, 8812), (1327134, 8450), (1335584, 9270), (1344854, 9097), (1353951, 9001), (1362952, 9182), (1372134, 8613), (1380747, 9047), (1389794, 9329), (1399123, 8515), (1407638, 9023), (1416661, 8584), (1425245, 8028), (1433273, 8003), (1441276, 9003), (1450279, 9597), (1459876, 8464), (1468340, 8650), (1476990, 9702), (1486692, 8760), (1495452, 8457), (1503909, 9388), (1513297, 8877), (1522174, 8049), (1530223, 8700), (1538923, 10381), (1549304, 8274), (1557578, 8412), (1565990, 9496), (1575486, 8795), (1584281, 9300), (1593581, 8789), (1602370, 8443), (1610813, 11165), (1621978, 9131), (1631109, 8871), (1639980, 9243), (1649223, 9973), (1659196, 9758), (1668954, 8115), (1677069, 9174), (1686243, 8478), (1694721, 8827), (1703548, 9605), (1713153, 8473), (1721626, 9165), (1730791, 9044), (1739835, 8849), (1748684, 8821), (1757505, 8580), (1766085, 8488), (1774573, 8929), (1783502, 8720), (1792222, 9095), (1801317, 8475), (1809792, 9743), (1819535, 8872), (1828407, 9615), (1838022, 9310), (1847332, 8016), (1855348, 9213), (1864561, 8714), (1873275, 9073), (1882348, 9095), (1891443, 9293), (1900736, 8906), (1909642, 9246), (1918888, 9123), (1928011, 8908), (1936919, 8729), (1945648, 8780), (1954428, 9006), (1963434, 9380), (1972814, 8941), (1981755, 8184), (1989939, 9869), (1999808, 8870), (2008678, 8802), (2017480, 8620), (2026100, 8725), (2034825, 8232), (2043057, 9424), (2052481, 9021), (2061502, 9762), (2071264, 8958), (2080222, 8436), (2088658, 9466), (2098124, 8721), (2106845, 9819), (2116664, 8540), (2125204, 8478), (2133682, 7848), (2141530, 9581), (2151111, 9165), (2160276, 9466), (2169742, 8596), (2178338, 9039), (2187377, 8799), (2196176, 9341), (2205517, 8614), (2214131, 8144), (2222275, 8810), (2231085, 8296), (2239381, 9128), (2248509, 9444), (2257953, 9316), (2267269, 8784), (2276053, 8834), (2284887, 8992)], [(2293879, 961), (2294840, 825), (2295665, 871), (2296536, 1089), (2297625, 734), (2298359, 877), (2299236, 690), (2299926, 1027), (2300953, 800), (2301753, 851), (2302604, 656), (2303260, 713), (2303973, 892), (2304865, 930), (2305795, 1090), (2306885, 1083), (2307968, 1327), (2309295, 762), (2310057, 1067), (2311124, 887), (2312011, 960), (2312971, 950), (2313921, 1063), (2314984, 863), (2315847, 902), (2316749, 766), (2317515, 1167), (2318682, 1384), (2320066, 842), (2320908, 956), (2321864, 1095), (2322959, 916), (2323875, 697), (2324572, 863), (2325435, 1037), (2326472, 962), (2327434, 874), (2328308, 911), (2329219, 866), (2330085, 1237), (2331322, 741), (2332063, 1090), (2333153, 929), (2334082, 852), (2334934, 865), (2335799, 594), (2336393, 1108), (2337501, 1122), (2338623, 919), (2339542, 675), (2340217, 939), (2341156, 778), (2341934, 936), (2342870, 1145), (2344015, 1164), (2345179, 639), (2345818, 784), (2346602, 726), (2347328, 711), (2348039, 913), (2348952, 976), (2349928, 920), (2350848, 1189), (2352037, 1114), (2353151, 815), (2353966, 885), (2354851, 874), (2355725, 564), (2356289, 754), (2357043, 735), (2357778, 867), (2358645, 1019), (2359664, 732), (2360396, 939), (2361335, 700), (2362035, 851), (2362886, 759), (2363645, 844), (2364489, 869), (2365358, 602), (2365960, 948), (2366908, 724), (2367632, 1041), (2368673, 796), (2369469, 887), (2370356, 640), (2370996, 876), (2371872, 923), (2372795, 984), (2373779, 924), (2374703, 1156), (2375859, 1312), (2377171, 959), (2378130, 919), (2379049, 915), (2379964, 601), (2380565, 1053), (2381618, 985), (2382603, 686), (2383289, 799), (2384088, 895), (2384983, 1078), (2386061, 1007), (2387068, 629), (2387697, 753), (2388450, 960), (2389410, 571), (2389981, 610), (2390591, 1145), (2391736, 1113), (2392849, 882), (2393731, 899), (2394630, 808), (2395438, 861), (2396299, 952), (2397251, 891), (2398142, 746), (2398888, 739), (2399627, 829), (2400456, 769), (2401225, 1170), (2402395, 849), (2403244, 953), (2404197, 570), (2404767, 868), (2405635, 1023), (2406658, 911), (2407569, 1090), (2408659, 818), (2409477, 1140), (2410617, 984), (2411601, 759), (2412360, 1011), (2413371, 824), (2414195, 1059), (2415254, 845), (2416099, 810), (2416909, 776), (2417685, 885), (2418570, 736), (2419306, 775), (2420081, 841), (2420922, 824), (2421746, 651), (2422397, 676), (2423073, 637), (2423710, 785), (2424495, 745), (2425240, 901), (2426141, 788), (2426929, 849), (2427778, 645), (2428423, 665), (2429088, 995), (2430083, 844), (2430927, 778), (2431705, 880), (2432585, 1088), (2433673, 961), (2434634, 743), (2435377, 1115), (2436492, 902), (2437394, 723), (2438117, 900), (2439017, 1155), (2440172, 771), (2440943, 795), (2441738, 784), (2442522, 809), (2443331, 823), (2444154, 938), (2445092, 731), (2445823, 1052), (2446875, 909), (2447784, 972), (2448756, 1017), (2449773, 837), (2450610, 683), (2451293, 826), (2452119, 802), (2452921, 2043), (2454964, 709), (2455673, 879), (2456552, 819), (2457371, 1189), (2458560, 871), (2459431, 905), (2460336, 703), (2461039, 704), (2461743, 1046), (2462789, 727), (2463516, 691), (2464207, 933), (2465140, 887), (2466027, 1122), (2467149, 887), (2468036, 867), (2468903, 781), (2469684, 940), (2470624, 809), (2471433, 939), (2472372, 862), (2473234, 968), (2474202, 797), (2474999, 834), (2475833, 714), (2476547, 995), (2477542, 991), (2478533, 813), (2479346, 1083), (2480429, 830), (2481259, 961), (2482220, 1039), (2483259, 1184), (2484443, 996), (2485439, 863), (2486302, 1028), (2487330, 836), (2488166, 633), (2488799, 518), (2489317, 934), (2490251, 935), (2491186, 677), (2491863, 1196), (2493059, 686), (2493745, 814), (2494559, 990), (2495549, 1007), (2496556, 998), (2497554, 817), (2498371, 1022), (2499393, 805), (2500198, 984), (2501182, 712), (2501894, 742), (2502636, 668), (2503304, 730), (2504034, 486), (2504520, 917), (2505437, 1105), (2506542, 918), (2507460, 762), (2508222, 762), (2508984, 727), (2509711, 1033), (2510744, 667), (2511411, 655), (2512066, 1007), (2513073, 536), (2513609, 1102), (2514711, 2522), (2517233, 769), (2518002, 855), (2518857, 1002), (2519859, 1170), (2521029, 516)], [(2521545, 48), None, (2521593, 35), (2521628, 42), None, None, None, None, None, None, None, None, None, None, None, None, None, (2521670, 42), None, (2521712, 25), (2521737, 44), (2521781, 22), (2521803, 18), None, None, None, None, (2521821, 26), None, None, None, None, (2521847, 21), (2521868, 25), None, None, (2521893, 26), None, None, None, None, (2521919, 71), (2521990, 21), (2522011, 23), None, None, None, None, (2522034, 48), None, None, None, None, None, (2522082, 31), None, None, None, None, (2522113, 42), None, (2522155, 22), None, (2522177, 21), None, (2522198, 26), (2522224, 42), None, None, (2522266, 77), (2522343, 27), None, None, None, None, (2522370, 21), (2522391, 21), None, None, (2522412, 34), (2522446, 42), None, None, None, (2522488, 25), None, None, (2522513, 21), None, None, None, None, None, (2522534, 24), (2522558, 21), None, None, (2522579, 26), None, (2522605, 18), None, (2522623, 54), None, None, None, None, None, None, (2522677, 26), None, None, None, (2522703, 20), None, None, (2522723, 64), (2522787, 42), (2522829, 17), (2522846, 17), (2522863, 26), None, (2522889, 26), None, None, None, (2522915, 26), (2522941, 20), (2522961, 26), None, (2522987, 42), (2523029, 63), None, None, None, (2523092, 40), (2523132, 48), None, None, None, (2523180, 47), None, None, None, None, None, None, None, (2523227, 42), None, (2523269, 80), None, (2523349, 9), None, (2523358, 21), (2523379, 42), None, None, (2523421, 65), (2523486, 82), (2523568, 21), None, (2523589, 72), None, None, (2523661, 24), (2523685, 21), None, None, None, None, None, (2523706, 42), (2523748, 21), (2523769, 21), None, (2523790, 42), (2523832, 25), None, (2523857, 38), (2523895, 21), (2523916, 56), None, None, (2523972, 21), (2523993, 19), (2524012, 26), None, (2524038, 16), None, (2524054, 21), None, None, (2524075, 38), None, (2524113, 22), (2524135, 21), (2524156, 21), (2524177, 21), None, (2524198, 63), None, (2524261, 21), (2524282, 42), None, (2524324, 17), None, None, None, None, (2524341, 21), (2524362, 21), None, None, (2524383, 21), None, None, (2524404, 21), None, (2524425, 26), None, (2524451, 50), (2524501, 22), None, None, (2524523, 50), (2524573, 26), (2524599, 21), (2524620, 21), (2524641, 19), None, (2524660, 35), (2524695, 26), (2524721, 23), (2524744, 39), (2524783, 42), None, None, None, None, None, None, (2524825, 21), None, None, None, (2524846, 21), None, None, (2524867, 90), None, (2524957, 239), (2525196, 38), None, None, None, None]]  # noqa: E501
_CRC8_TABLE = [
    0x00, 0x07, 0x0e, 0x09, 0x1c, 0x1b, 0x12, 0x15,
    0x38, 0x3f, 0x36, 0x31, 0x24, 0x23, 0x2a, 0x2d,
    0x70, 0x77, 0x7e, 0x79, 0x6c, 0x6b, 0x62, 0x65,
    0x48, 0x4f, 0x46, 0x41, 0x54, 0x53, 0x5a, 0x5d,
    0xe0, 0xe7, 0xee, 0xe9, 0xfc, 0xfb, 0xf2, 0xf5,
    0xd8, 0xdf, 0xd6, 0xd1, 0xc4, 0xc3, 0xca, 0xcd,
    0x90, 0x97, 0x9e, 0x99, 0x8c, 0x8b, 0x82, 0x85,
    0xa8, 0xaf, 0xa6, 0xa1, 0xb4, 0xb3, 0xba, 0xbd,
    0xc7, 0xc0, 0xc9, 0xce, 0xdb, 0xdc, 0xd5, 0xd2,
    0xff, 0xf8, 0xf1, 0xf6, 0xe3, 0xe4, 0xed, 0xea,
    0xb7, 0xb0, 0xb9, 0xbe, 0xab, 0xac, 0xa5, 0xa2,
    0x8f, 0x88, 0x81, 0x86, 0x93, 0x94, 0x9d, 0x9a,
    0x27, 0x20, 0x29, 0x2e, 0x3b, 0x3c, 0x35, 0x32,
    0x1f, 0x18, 0x11, 0x16, 0x03, 0x04, 0x0d, 0x0a,
    0x57, 0x50, 0x59, 0x5e, 0x4b, 0x4c, 0x45, 0x42,
    0x6f, 0x68, 0x61, 0x66, 0x73, 0x74, 0x7d, 0x7a,
    0x89, 0x8e, 0x87, 0x80, 0x95, 0x92, 0x9b, 0x9c,
    0xb1, 0xb6, 0xbf, 0xb8, 0xad, 0xaa, 0xa3, 0xa4,
    0xf9, 0xfe, 0xf7, 0xf0, 0xe5, 0xe2, 0xeb, 0xec,
    0xc1, 0xc6, 0xcf, 0xc8, 0xdd, 0xda, 0xd3, 0xd4,
    0x69, 0x6e, 0x67, 0x60, 0x75, 0x72, 0x7b, 0x7c,
    0x51, 0x56, 0x5f, 0x58, 0x4d, 0x4a, 0x43, 0x44,
    0x19, 0x1e, 0x17, 0x10, 0x05, 0x02, 0x0b, 0x0c,
    0x21, 0x26, 0x2f, 0x28, 0x3d, 0x3a, 0x33, 0x34,
    0x4e, 0x49, 0x40, 0x47, 0x52, 0x55, 0x5c, 0x5b,
    0x76, 0x71, 0x78, 0x7f, 0x6a, 0x6d, 0x64, 0x63,
    0x3e, 0x39, 0x30, 0x37, 0x22, 0x25, 0x2c, 0x2b,
    0x06, 0x01, 0x08, 0x0f, 0x1a, 0x1d, 0x14, 0x13,
    0xae, 0xa9, 0xa0, 0xa7, 0xb2, 0xb5, 0xbc, 0xbb,
    0x96, 0x91, 0x98, 0x9f, 0x8a, 0x8d, 0x84, 0x83,
    0xde, 0xd9, 0xd0, 0xd7, 0xc2, 0xc5, 0xcc, 0xcb,
    0xe6, 0xe1, 0xe8, 0xef, 0xfa, 0xfd, 0xf4, 0xf3
]
# fmt: on

_IS_LEAF = 0x80
_INCLUDE_SUBDOMAINS = 0x40


try:
    from importlib.resources import open_binary

    def open_pkg_binary(path: str) -> typing.BinaryIO:
        return open_binary("hstspreload", path)


except ImportError:

    def open_pkg_binary(path: str) -> typing.BinaryIO:
        return open(
            os.path.join(os.path.dirname(os.path.abspath(__file__)), path),
            "rb",
        )


@functools.lru_cache(maxsize=1024)
def in_hsts_preload(host: typing.AnyStr) -> bool:
    """Determines if an IDNA-encoded host is on the HSTS preload list"""

    if isinstance(host, str):
        host = host.encode("ascii")
    labels = host.lower().split(b".")

    # Fast-branch for gTLDs that are registered to preload all sub-domains.
    if labels[-1] in _GTLD_INCLUDE_SUBDOMAINS:
        return True

    with open_pkg_binary("hstspreload.bin") as f:
        for layer, label in enumerate(labels[::-1]):
            # None of our layers are greater than 4 deep.
            if layer > 3:
                return False

            # Read the jump table for the layer and label
            jump_info = _JUMPTABLE[layer][_crc8(label)]
            if jump_info is None:
                # No entry: host is not preloaded
                return False

            # Read the set of entries for that layer and label
            f.seek(jump_info[0])
            data = bytearray(jump_info[1])
            f.readinto(data)

            for is_leaf, include_subdomains, ent_label in _iter_entries(data):
                # We found a potential leaf
                if is_leaf:
                    if ent_label == host:
                        return True
                    if include_subdomains and host.endswith(b"." + ent_label):
                        return True

                # Continue traversing as we're not at a leaf.
                elif label == ent_label:
                    break
            else:
                return False
    return False


def _iter_entries(data: bytes) -> typing.Iterable[typing.Tuple[int, int, bytes]]:
    while data:
        flags = data[0]
        size = data[1]
        label = bytes(data[2 : 2 + size])
        yield (flags & _IS_LEAF, flags & _INCLUDE_SUBDOMAINS, label)
        data = data[2 + size :]


def _crc8(value: bytes) -> int:
    # CRC8 reference implementation: https://github.com/niccokunzmann/crc8
    checksum = 0x00
    for byte in value:
        checksum = _CRC8_TABLE[checksum ^ byte]
    return checksum
